#---------------------------------------------------------------------------
#
#	Define a selection box with a label, a listbox, a number of
#	buttons (help, load, copy, new, delete, dismiss), and an entry line.
#
# The list of actions is as follows (SEL denotes the text of the currently
# selected item, NAME denotes the value in $name.pair for double boxes):
#
# InitMethod, InitMethod2
#	Determines the interpretation for the 'Init' attribute of the action
#	list. Possible values are 'file' (Init specifies a file name),
#	and 'list' (Init specifies a list of strings).
# Init
#	Initialization data.
# Init2 or {Init . NAME}
#	Initialization for second box. In the first case, the string associated
#	with Init2 is used as a format.
# Help
#	The name of the help topic for this window.
# Select, Remove, Edit
#	A function that will be invoked on SEL (or SEL NAME).
# New
#	The new item is inserted into the listbox and the given function
#	is applied to its index and text: NEWSEL or NEWSEL NAME
# Copy, Rename
#	A function that is applied to SEL NEWSEL (or SEL NEWSEL NAME).
# Action
#	A function that is applied to the listbox and the list of items
#	in the selector box.
# Ok
#	A function that is applied to the list of items in the listbox.
#	After this function has been called, a Dismiss function is invoked
#	automatically. For double boxes, the function is called on an alist
#	of **ALL** box value lists.
#
# Only buttons corresponding to actions listed in the new(selector) call
# will be displayed. The InitMethod, Init, and Help actions must be
# listed in any case.
#
# If neither 'New' nor 'Copy' are given, no input item will be displayed.
#
# If an action returns {}, it is assumed to have succeeded. Otherwise, the
# corresponding listbox operation will not be carried out.
#

defwidget Selector

defmethod Selector new {name args} {
  global data

  args	initmethod initmethod2 init init2 inits \
	label actions title title1 title2 nodouble \
	{titlefont bold} wait layout {inputlabel {New Name}} \
	{textfont text} {sorted true}

  set stuff [app(get) $initmethod $init]
  set actions [$self buildActions $actions $name]

  Toplevel new $name -title ${title} \
	-resizable true \
	-buttons {Select Select_New Copy Rename New Remove Edit Action Ok} \
	-handler $name \
	-actions [concat $actions [list [list Help {} Library/selector]]]
  defsuper $name Selector

  Corner new $name.c1 -layout {left}
  Corner new $name.c2 -layout {right}

  set needname 0
  foreach i $actions {
    if { [position [lindex $i 0] {Select_New Copy Rename New}] >= 0 } {
      set needname 1
      break
    }
  }
  if { $needname && $inputlabel != "" } {
    append inputlabel ": "
    Inputline new $name.name -layout {bottom padx 20 pady 20} \
	-label $inputlabel -width 24 -fit true
  }
  $name slot _needname $needname

  # create a label pair if necessary
  #
  if { $initmethod2 != {} } {
    Pair new $name.pair -layout {top frame sw} -label $label
  }

  # take care of defaulting for mouse button actions
  #
  if { $nodouble == "true" } {
    set double {}
  } {
    set double [lrange [assoc Select $actions] 1 end]
    if { $double == {} } {
      set double [lrange [assoc Edit $actions] 1 end]
    }
    if { $double != {} } {
      set double [list $name _doublebutton $double]
    }
  }

  # initialize the first listbox
  $name slot _display {}
  $name slot _sorted [expr {$sorted == "true"}]
  if { [$name slot _sorted] } {
    set stuff [lsort $stuff]
  }

  if { $init2 == {} } {

    Listbox new $name.sel -layout {top expand fill} \
	-title $title1 -titlefont $titlefont -textfont $textfont \
	-action [list $name _button] \
	-double $double

    $name slot _box	$name.sel
    $name slot _list	$stuff
    $name slot _boxes	{}
    $name slot _value	{}
    $name slot _index	-1
    $name.sel set $stuff

  } {

    Listbox new $name.switch -layout {left expand filly} \
	-action [list $name _switch] \
	-title $title1 -titlefont $titlefont -textfont $textfont

    $name slot _boxes $stuff
    $name.switch set $stuff

    set stuff2 [app(get) $initmethod2 $init2]
    set ind 0
    foreach i $stuff {

      set i2 [assoc $i $stuff2]
      if { $i2 == {} } {
	set i2 [assoc "*" $init2]
      }
      if { $sorted == "true" } {
        set i2 [lsort $i2]
      }
      $name slot _list${ind} $i2

      # Create and fill the listbox
      Listbox new $name.sel$ind \
	-action [list $name _button] -double $double \
	-title $title2 -titlefont $titlefont -textfont $textfont
      $name.sel$ind set $i2
      incr ind
    }
    $name _display 0
  }

  set data($name) Dismiss
  $name layout $layout

  if { $wait == "true" } {
    tkwait window $name
    set result $data($name)
    unset data($name)
    return $result
  } {
    return $name
  }
}

#---------------------------------------------------------------------------

defmethod Selector _update {newlist {newitem {}}} {

  if { [$self slot _sorted] } {
    set newlist [lsort $newlist]
  }
  $self slot _list $newlist

  # Update the listbox
  set listbox [$self slot _box]
  $listbox set $newlist
  $self _button $listbox.list {} $newitem
}

defmethod Selector _button {listbox isel sel} {

  $self slot _value $sel
  $self slot _index $isel

  $listbox xview 0

  if { $sel != {} && [$self slot _needname] } {
    $self.name set $sel
  }
}

defmethod Selector _doublebutton {action listbox isel sel} {

  $self _button $listbox $isel $sel
  $self Select $action
}

#---------------------------------------------------------------------------

defmethod Selector _switch {listbox isel sel} {
  $self _display $isel
}

defmethod Selector _display {index} {

  set oldindex [$self slot _display]

  if { $oldindex == $index } {
    return
  }

  if { $oldindex != {} } {
    $self slot _list$oldindex [$self slot _list]
    if { $index != {} } {
      pack unpack $self.sel$oldindex
    }
  }

  $self slot _display $index

  if { $index != {} } {
    $self slot _box	$self.sel$index
    $self slot _list	[$self slot _list$index]
    $self slot _value	{}
    $self slot _index	-1

    $self.pair set [lindex [$self slot _boxes] $index]
    $self.sel$index layout {right expand fill}
  }
}

#---------------------------------------------------------------------------

# Construct a procedure call
defmethod Selector _call {action args} {

  if { $action == "" } {
    return
  }
  if { [$self slot _boxes] == {} } {
    return [concat $action $args]
  } {
    return [concat $action [list [$self.pair get]] $args]
  }
}

# Evaluate a procedure call
defmethod Selector _eval {form} {

  if { $form != {} } {
    set res [uplevel #0 $form]
    if { $res != {} } {
      Dialog new * -help [list Library/failed $res]
      return 1
    }
  }
  return 0
}

# Return selection and selection index
defmethod Selector _sel {button} {
  upvar sel sel

  # Check current selection - complain if none present
  set sel [$self slot _value]
  if { $sel == "" } {
    Dialog new * -help [list Library/selector-complain $button]
    return 1
  }
  return 0
}

# Return new input
defmethod Selector _input {} {
  upvar newsel newsel

  set newsel [$self.name get]
  if { $newsel == "" } {
    Dialog new * -help Library/selector-enter
    return 1
  }
  if { [lsearch [$self slot _list] $newsel] >= 0 } {
    Dialog new * -help Library/selector-duplicate
    return 1
  }
  return 0
}

#---------------------------------------------------------------------------

defmethod Selector Select {action} {
  global data

  # Check current selection - complain if none present
  if { [$self _sel Select] } {
    return
  }

  # Let the window swiftly and softly vanish away...
  wm withdraw $self
  update

  if { $action != "" } {
    if { [$self _eval [$self _call $action $sel]] } {
      return
    }
  }

  set data($self) [list Select $sel]
  $self Dismiss
}

defmethod Selector Select_New {action} {
  global data

  if { [$self _input] } {
    return
  }

  # Let the window swiftly and softly vanish away...
  wm withdraw $self
  update

  if { $action != "" } {
    if { [$self _eval [$self _call $action $newsel]] } {
      return
    }
  }

  set data($self) [list Select_New $newsel]
  $self Dismiss
}

defmethod Selector Ok {action} {
  global data

  if { $action != "" } {
    # Extract the list of items or the list of item lists
    set boxes [$self slot _boxes]
    if { $boxes == {} } {
      set items [$self slot _list]
    } {
      $self _display {}
      set items {}
      set i 0
      foreach item $boxes {
	lappend items [concat [list $item] [$self slot _list$i]]
	incr i
      }
    }
    # Invoke the action handler
    if { [$self _eval [concat $action [list $items]]] } {
      return
    }
  }

  set data($self) [list Ok $items]
  $self Dismiss
}

#---------------------------------------------------------------------------

defmethod Selector Rename {action} {

  if { [$self _sel Rename] } {
    return
  }
  if { [$self _input] } {
    return
  }
  if { [$self _eval [$self _call $action $sel $newsel]] } {
    return
  }

  # Rename the element in the list and resort it if necessary
  set stuff [$self slot _list]
  set isel [$self slot _index]
  if { $isel < 0 } {
    return
  }
  set stuff [lreplace $stuff $isel $isel $newsel]
  $self _update $stuff $newsel
}

#---------------------------------------------------------------------------

defmethod Selector Copy {action} {

  if { [$self _sel Copy] } {
    return
  }

  $self New [concat ${action} [list ${sel}]]
}

defmethod Selector New {action} {

  if { [$self _input] } {
    return
  }
  if { [$self _eval [$self _call $action $newsel]] } {
    return
  }

  set stuff [$self slot _list]
  lappend stuff $newsel
  $self _update $stuff $newsel
}

#---------------------------------------------------------------------------

defmethod Selector Edit {action} {

  if { [$self _sel Edit] } {
    return
  }
  if { [$self _eval [$self _call $action $sel]] } {
    return
  }
}

#---------------------------------------------------------------------------

defmethod Selector Remove {action} {

  if { [$self _sel Remove] } {
    return
  }
  if { [Yesno new * -help [list Library/selector-delete $sel]] != "Yes" } {
    return
  }
  if { [$self _eval [$self _call $action $sel]] } {
    return
  }

  set isel [$self slot _index]
  $self slot _list [lreplace [$self slot _list] $isel $isel]

  [$self slot _box].list delete $isel
  $self slot _value {}
  $self slot _index {}
  return
}

#---------------------------------------------------------------------------

defmethod Selector Action {action} {

  if { $action != {} } {
    eval [concat $action [list [$self slot _list]]]
  }
  return
}

Window addDemo Selector

defmethod Selector demo {} {

  Selector new * -layout center -initmethod list \
	-init {sdfsdi  sdofjoisjfosf soifjsoif osjfos fsjfsfjsojfs} \
	-title "Selector Box" \
	-actions {
	{Select Auswahl - showAction select}
	{Select_New Neuer_Wert - showAction select}
	{Ok Bestaetigen - showAction ok}
	{Rename {} - showAction rename}
	{Remove Entfernen - showAction remove}
	{Copy {} - showAction copy}
	{New {} - showAction new}
	{Edit {} - showAction edit}
	}

  Selector new * -layout center -initmethod2 list \
	-init2 {{* bar baz}
		{foo fsdfsdi fsdofjoisjfosf fsoifjsoif fosjfos fsjfsfjsojfs}
		{bar rsdfsdi rsdofjoisjfosf rsoifjsoif fosjfos fsjfsfjsojfs}
		{baz fsdfsdi fsdofjoisjfosf zsoifjsoif zosjfosd}
		} \
	-initmethod list \
	-init {foo bar baz} \
	-title "Dual Selector Box" \
	-title1 "Selector Items" \
	-title2 "Associated Items" \
	-label "Current Selector" \
	-actions {
	{Select Auswahl - showAction select}
	{Select_New Neuer_Wert - showAction select}
	{Ok Bestaetigen - showAction ok}
	{Rename {} - showAction rename}
	{Remove Entfernen - showAction remove}
	{Copy {} - showAction copy}
	{New {} - showAction new}
	{Edit {} - showAction edit}
	}
}
