#---------------------------------------------------------------------------
#
#	Generic window objects
#
#---------------------------------------------------------------------------

defobject Window

defmethod Window DEFAULT {method args} {
  eval [concat [list $self! $method] $args]
}

defmethod Window instantiate {name layout {slots {}}} {

  rename $name $name!
  $self clone $name $slots
  $name layout $layout
}

defmethod vanilla-object buildActions {actions {object {}}} {

  if { [streq $object {}] } {
    set object $self
  }

  set new {}
  foreach act $actions {
    if { [streq [lindex $act 2] "-"] } {
      lappend new [lreplace $act 2 2 $object]
    } {
      lappend new $act
    }
  }
  return $new
}

defmethod vanilla-object buildAction {action {object {}}} {

  if { [streq $object {}] } {
    set object $self
  }

  if { [streq [lindex $action 0] "-"] } {
    return [lreplace $action 0 0 $object]
  } {
    return $action
  }
}

defmethod Window Dismiss0 {{action {}}} {

  if { ![streq $action {}] } {
    uplevel #0 $action
  }
  $self reclaimall
  catch {destroy $self}
  return
}

defmethod Window DismissChildren {{action {}}} {

  if { ![streq $action {}] } {
    uplevel #0 $action
  }
  $self reclaimall true
  return
}

defmethod Window Dismiss {{action {}}} {

  $self Dismiss0
#  if { [streq $self "."] } {
    update idletasks
    update
#  }
  return
}

defmethod Window demos {} {

  Window slot _demos
}

defmethod Window addDemo {name} {

  set demos [Window slot _demos]
  if { [position $name $demos] < 0 } {
    Window slotappend _demos $name
  }
  return $name
}

defmethod Window Help {topic} {

  Help new * -help $topic
}

proc defwidget {name {super {}} {slots {}}} {

  if { [streq $super {}] } {
    set super Window
  }
  defobject $name $super $slots
  return $name
}

#---------------------------------------------------------------------------
#
#	Place a window on the screen
#
#---------------------------------------------------------------------------

# window layout {name layout}
#	Place a window on the screen. For toplevel windows, the argument
#	LAYOUT may be one of "" (no placement), "center" (center window
#	on screen), "choose" (let the use choose the position), a window
#	name (center relative to that window), or a geometry specification.
#	For subwindows, LAYOUT specifies packing instructions.
#	Note: For toplevel windows, this defines the position. For all
#	others, this defines packing, i.e. layout.
#

defmethod Window layout {layout} {

  # don't do anything
  if { [streq $layout {}] } then {
    return $self
  }

  # simple packing
  if { ![streq [winfo toplevel $self] $self] } then {

    pack append [winfo parent $self] $self $layout
    return $self

  }

  # handle toplevel windows

  wm withdraw $self
  update

  if { [streq $layout "center"] } then {

    set sx [winfo screenwidth .]
    set sy [winfo screenheight .]

    set wx [winfo reqwidth $self]
    set wy [winfo reqheight $self]

    set cx [expr ($sx-$wx)/2]
    set cy [expr ($sy-$wy)/2]

    set layout "+$cx+$cy"

  } elseif { [streq $layout "choose"] } {

    set layout {}

  } elseif { [streq [string index $layout 0] "."] } {

    set sx [winfo rootx $layout]
    set sy [winfo rooty $layout]
    set sw [winfo width $layout]
    set sh [winfo height $layout]

    set wx [winfo reqwidth $self]
    set wy [winfo reqheight $self]

    set layout "+[expr $sx+($sw-$wx)/2]+[expr $sy+($sh-$wy)/2]"
  }

  wm geometry $self $layout
  wm deiconify $self
  update
  wm positionfrom $self {}

  return $self
}

#---------------------------------------------------------------------------

defmethod Window matrixlayout {objects args} {

  args	{dir horizontal} {max 5} {sep 5}

  set n [llength $objects]

  set win_w [winfo reqwidth [lindex $objects 0]]
  set win_h [winfo reqheight [lindex $objects 0]]

  if { [streq $dir "horizontal"] } then {
    set n_x [expr {($n >= $max) ? $max : $n}]
    set n_y [expr ($n+$max-1)/$max]
  } else {
    set n_x [expr ($n+$max-1)/$max]
    set n_y [expr {($n >= $max) ? $max : $n}]
  }

  $self configure \
	-width [expr $sep+($win_w+$sep)*$n_x] \
	-height [expr $sep+($win_h+$sep)*$n_y]

  set x 0
  set y 0

  foreach object $objects {
    place $object -in $self -anchor center \
	-relx [expr (2.0*$x+1.0)/(2.0*$n_x)] \
	-rely [expr (2.0*$y+1.0)/(2.0*$n_y)]

    if { [streq $dir "horizontal"] } then {
      incr x
      if { [streq $x $max] } {
	set x 0
	incr y
      }
    } else {
      incr y
      if { [streq $y $max] } {
	set y 0
	incr x
      }
    }
  }

  return $self
}

#---------------------------------------------------------------------------

Window slot grabcount 0

defmethod Window grab {args} {

  args	local

  if { [streq $local "true"] } {
    catch "grab set $self"
  } {
    catch "grab set -global $self"
  }
}

defmethod Window timeoutGrab {args} {

  args	{time 20}

  update
  catch {grab set -global $self}
  if { [streq $time {}] } {
    set x [uplevel #0 [list incr [slot-variable Window grabcount]]]
    Window slot grabwin $self
    after [expr $time*1000] "if { \[winfo exists $self\] } {$self _ungrab $x}"
  }
  return $self
}

defmethod Window _ungrab {id} {

  if {	[streq [Window slot grabwin] $self] && 
	[streq [Window slot grabcount] $id] } {
    $self ungrab
    $self Dismiss
  }
}

defmethod Window ungrab {} {

  foreach win [grab current] {
    grab release $win
  }
}

defmethod Window demo {} {

  set t [Toplevel new * -info true]
  Message new *$t \
	-text "Hit any of the buttons below to get a demo window." \
	-layout top -textfont largebold

  Frame new $t.f -layout top -relief flat
  set demos {}
  foreach i [lsort [Window demos]] {
    lappend demos [Button new *$t.f -text "Test $i" -textfont bold \
			-width 20 -action [list $i demo]]
  }
  $t.f matrixlayout $demos -dir vertical -max 15

  Button new *$t -layout {padx 20 pady 30 bottom} \
	-textfont largebold -text "Dismiss" \
	-action {. Dismiss}

  $t layout +10+10
}

#---------------------------------------------------------------------------

proc . {args} {
  error ". does not handle $args"
}

Window instantiate . {}

defmethod . Dismiss {} {
  destroy .
  exit 0
}
