
/*
 * TkTree.cc - member routines for class TkTree,
 *             implementation of the TCL tree command
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include "TkTree.h"
#include "TkTreeNode.h"
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <strstream.h>
#include <string.h>


/* 
 * declare a table of tree subcommands and the methods that handle them.
 * format: name, min_args, max_args, method
 */
static class _EXPORT TkTreeSubCmds {
public:
    char* name;      // method name
    int (TkTree::*fptr)(int argc, char* argv[]); 
    int min_args;    // minimum number of args
    int max_args;    // maximum number of args
} subcmds_[] = { 
    {"addlink",      &TkTree::addlink_,         3, 99},
    {"ancestors",     &TkTree::ancestors_,      1,  1},
    {"getcanvas",     &TkTree::getcanvas_,      0,  0},
    {"canvas",        &TkTree::getcanvas_,      0,  0},
    {"child",         &TkTree::child_,          1,  1},
    {"draw",          &TkTree::draw_,           0,  0},
    {"isleaf",        &TkTree::isleaf_,         1,  1},
    {"isroot",        &TkTree::isroot_,         1,  1},
    {"movelink",      &TkTree::movelink_,       2,  2},
    {"nodeconfigure", &TkTree::nodeconfigure_,  1, 99},
    {"nodeconfig",    &TkTree::nodeconfigure_,  1, 99},
    {"prune",         &TkTree::prune_,          1,  1},
    {"parent",        &TkTree::parent_,         1,  1},
    {"root",          &TkTree::root_,           1,  1},
    {"rmlink",        &TkTree::rmlink_,         1,  1},
    {"sibling",       &TkTree::sibling_,        1,  1},
    {"subnodes",      &TkTree::subnodes_,       1,  1}
};


/* 
 * tree config options - used to process command line options and for the
 * widget "configure" subcommand.
 */
static Tk_ConfigSpec configSpecs_[] = {
    {TK_CONFIG_PIXELS, "-parentdistance", "parentDistance", "ParentDistance",
     "30", Tk_Offset(TkTreeOptions, parentDistance), 0},
    {TK_CONFIG_STRING, "-layout", "layout", "Layout",
     NULL, Tk_Offset(TkTreeOptions, layout), 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
     "10", Tk_Offset(TkTreeOptions, borderWidth), 0},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};



/*
 * Call the given method in this class with the given arguments
 */
int TkTree::call(const char* name, int argc, char* argv[])
{
    for(int i = 0; i < sizeof(subcmds_)/sizeof(*subcmds_); i++) {
	TkTreeSubCmds* t = &subcmds_[i];
	if (strcmp(t->name, name) == 0) {
	    if (check_args(name, argc, t->min_args, t->max_args) != TCL_OK)
		return TCL_ERROR;
	    return (this->*t->fptr)(argc, argv);
	}
    }
    return TkWidget::call(name, argc, argv);
}


/*
 * A call to this function is made from the tkAppInit file at startup
 * to install the tree command
 */
extern "C"
int Tree_Init(Tcl_Interp* interp)  
{
    Tcl_CreateCommand(interp, "tree", TkTree::treeCmd, NULL, NULL);
    return TCL_OK;
}


/*
 * This static method is called by tcl to create a tree widget
 * Syntax: see tree man page
 */
int TkTree::treeCmd(ClientData, Tcl_Interp *interp, int argc, char* argv[])
{
    if (argc < 2) {
	Tcl_AppendResult("wrong # args:  should be \"",
	     argv[0], " path ?options?\"", NULL);
	return TCL_ERROR;
    }

    TkTree *t = new TkTree(interp, argc, argv);
    return t->status();
}


/*
 * Constructor: initialize a new tree with the command line args
 * 
 * This constructor is called for each tree declared in tk. The destructor
 * is called when the tree widget is destroyed.
 */
TkTree::TkTree(Tcl_Interp *interp, int argc, char* argv[])
: TkWidget(interp, "Canvas", configSpecs_, options_, argc, argv),
  canvas_(pname_),
  rootNode_(new TkTreeNode(this, "", "")),
  options_(30, "horizontal", 10)
{
    // do first time widget configuration
    if (initWidget(argc, argv) != TCL_OK) 
	return;
}


/*
 * destructor - clean up tree nodes when widget cmd is destroyed
 */
TkTree::~TkTree()
{
    delete rootNode_;
    rootNode_ = NULL;
}


/*
 * fix the root tree node at left center (for horizontal trees)
 * or top center for vertical trees
 */
int TkTree::fixRootNode()
{
    int x1 = 0, y1 = 0, x2 = 0, y2 = 0;

    tcl_ << canvas_ << " bbox all" << eval;
    if (tcl_.status() == TCL_OK)
	sscanf(interp_->result, "%d %d %d %d", &x1, &y1, &x2, &y2);
	
    // center the root node at left center for horizontal trees
    // or top center for vertical trees
    if (horizontal())    
        rootNode_->pos(TreePoint(x1-parentDistance()-2, (y2-y1)/2));
    else
        rootNode_->pos(TreePoint((x2-x1)/2, y1-parentDistance()-2));
    
    return TCL_OK;
}


/*
 * This procedure is called to process an argv/argc list, plus
 * the Tk option database, in order to configure (or reconfigure) 
 * a widget (redefined here to fix position of tree root node
 * after a configure command)
 * */
int TkTree::configureWidget(int argc, char* argv[], int flags)
{
    if (TkWidget::configureWidget(argc, argv, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    // set the pos of the root node incase the layout changed
    return fixRootNode();
}


/*
 * find the named node and return a pointer to it or 0 if not found
 */
TkTreeNode* TkTree::findNode(const char* tag)
{
    TkTreeNode* node = (TkTreeNode*)rootNode_->find(tag);
    if (node == NULL) 
	Tcl_AppendResult(interp_, 
			 "can't find tree node: \"", 
			 tag, "\"", NULL);
    return node;
}


/*
 * Set up the node's size, position and options
 */
int TkTree::setupNode(TkTreeNode* node, int argc, char* argv[])
{
    tcl_ << canvas_ << " bbox {" << node->tag() << "}" << eval;
    if (tcl_.status() != TCL_OK)
        return TCL_ERROR;
    
    // get the size of the node
    int x1, y1, x2, y2;
    sscanf(interp_->result, "%d %d %d %d", &x1, &y1, &x2, &y2);	
    node->box(x1, y1, x2, y2);
    
    // get the other options
    int d;
    for (int i = 0; i < argc; i++) {
	if (strcmp(argv[i], "-border") == 0 && ++i < argc) {
	    if (sscanf(argv[i], "%d", &d) == 1)
	        node->border(d);
	}
	else if (strcmp(argv[i], "-remove") == 0 && ++i < argc)
	    node->removeCmd(argv[i]);
    }
    
    return TCL_OK;
}


/*
 * add a new child node to the parent node in the tree
 * usage: addlink parent child line ?options?
 */
int TkTree::addlink_(int argc, char* argv[])
{
    TkTreeNode* parent = findNode(argv[0]);
    if (parent == NULL) 
	return TCL_ERROR;
    
    TkTreeNode* child = new TkTreeNode(this, argv[1], argv[2], 0, 0, 0, 0, 0);
    if (setupNode(child, argc-3, argv+3) != TCL_OK)
        return TCL_ERROR;
        
    parent->addLink(child);
    
    return TCL_OK;
}


/*
 * unhook the child subtree and make it a subtree of the parent tree
 * usage: $tree movelink child parent
 */
int TkTree::movelink_(int, char* argv[])
{
    TkTreeNode* parent = findNode(argv[1]);
    if (parent == NULL) 
	return TCL_ERROR;
	
    TkTreeNode* child = findNode(argv[0]);
    if (child == NULL) 
	return TCL_ERROR;

    /* don't allow the child to be itself, it's parent or
     * one of the parent's ancestors (no time travel allowed :-) 
     */ 
    if (parent == child || parent == child->parent()) {
	return error("illegal move");
    }
    for (TreeNode* p = parent; p && *p->tag(); p = p->parent()) {
	if (p == child) {
	    error("illegal move");
	    return TCL_ERROR;
	}
    }   
	
    child->unLink ();
    parent->addLink(child);
    
    return TCL_OK;
}
    
  

/*
 * remove the named node and its subnodes from the tree
 * usage: $tree rmlink tag
 */
int TkTree::rmlink_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    node->rmLink();
    return TCL_OK;
}
    
    
/*
 * Remove and delete the nodes children
 */
int TkTree::prune_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    node->prune();
    return TCL_OK;
}

    
/*
 * This command recalculates the node's size and also allows the same options
 * as the "addlink" sub-command (see the tree man page for details).
 */
int TkTree::nodeconfigure_(int argc, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    if (setupNode((TkTreeNode*)node, --argc, ++argv) != TCL_OK)
        return TCL_ERROR;
        
    return TCL_OK;
}

    
/*
 * Make the named node the new root of the tree.
 */
int TkTree::root_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    if (node->parent()) {
        node->unLink ();
        rootNode_->prune();
        rootNode_->addLink(node);
    }
    
    return TCL_OK;
}
 
   
/*
 * return (in tcl) the name of the tree's canvas
 */
int TkTree::getcanvas_(int, char**)
{
    // return the name of the canvas widget as a tcl result
    Tcl_SetResult(interp_, canvas_, TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns true if the node is a leaf node
 */
int TkTree::isleaf_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, (node->child() ? "0" : "1"), TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns true if the node is a root node
 * (root_ is invisible, so its children are the root nodes seen)
 */
int TkTree::isroot_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, ((node->parent() == rootNode_) ? "1" : "0"), TCL_STATIC);
    return TCL_OK;
}

    
/*
 * draw the tree in the canvas
 */
int TkTree::draw_(int, char**)
{
    rootNode_->drawTree();
    return TCL_OK;
}
    

/*
 * this command returns the name of the node's child node
 * or the empty string if there is none
 */
int TkTree::child_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, (char*)(node->child() ? node->child()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this node's subnodes
 */
int TkTree::subnodes_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    for (TreeNode* p = node->child(); p; p = p->sibling()) 
        Tcl_AppendResult(interp_, "{", p->tag(), "} ", NULL);
    return TCL_OK;
}


/*
 * this command returns the name of the node's sibling node
 * or the empty string if there is none
 */
int TkTree::sibling_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, 
		     (char*)((node->sibling()) ? node->sibling()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns the name of the node's parent node
 * or the empty string if there is none
 */
int TkTree::parent_(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, 
		     (char*)((node->parent()) ? node->parent()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this nodes ancestors
 */
int TkTree::ancestors_(int argc, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    // count the ancestors
    int n = 0;
    const TreeNode* p;
    for (p = node->parent(); p && *p->tag(); p = p->parent()) {
	n++;
    }
    if (n == 0) 
	return TCL_OK;

    // allocate the array arg for the tcl list routines
    argc = n;
    argv = new char*[argc];
    for (p = node->parent(); p && *p->tag(); p = p->parent()) {
	argv[--n] = (char*)p->tag();
    }   

    // create the tcl list
    char* list = Tcl_Merge(argc, argv);
    if (list) 
	Tcl_AppendResult(interp_, list, NULL);
    else
	return TCL_ERROR;

    free(list);
    delete argv;
    return TCL_OK;
}

