
#ifndef _TkTree_h_
#define _TkTree_h_

/*
 * TkTree.h - class definitions for drawing trees in Tk/Tcl
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include "config.h"
#include "TkWidget.h"
#include <string.h>
#include <stdlib.h>

/* 
 * tree options (used for tree configuration)
 */
class TkTreeOptions : public TkWidgetOptions {
public:
    int parentDistance;		// distance between nodes and parent
    char* layout;		// tree layout: "horizontal" or "vertical"
    int borderWidth;		// width of border around entire tree
    TkTreeOptions(int p, char* l, int b)
	:parentDistance(p), layout(strdup(l)), borderWidth(b) {}
    ~TkTreeOptions() {free(layout);}
};


// forward reference to class used to represent tree nodes
class _EXPORT TkTreeNode;


/*
 * Class TkTree
 * 
 * This class implements the extended Tk command "tree" for displaying
 * dynamic trees in a canvas.
 */
class _EXPORT TkTree: public TkWidget {
private:
    char* canvas_;			// parent of tree should be a canvas
    TkTreeNode* rootNode_;		// root of the layout tree
    TkTreeOptions options_;             // holds tree config options

    // -- private member functions  --
     
    // fix the invisible root node of the tree at center left or top
    int fixRootNode();
    
    // find the named node and return a pointer to it or 0 if not found
    TkTreeNode* findNode(const char* tag);
    
    // reconfigure the node (size, options, ...)
    int setupNode(TkTreeNode*, int argc, char* argv[]);

    // copy constructor: not defined
    TkTree(const TkTree&);
    
protected:
    // redefined from parent class
    int configureWidget(int argc, char* argv[], int flags);

public:
    // initialize the tree with the command line args
    TkTree(Tcl_Interp*, int argc, char* argv[]);
    
    // destructor - clean up tree nodes when widget cmd is destroyed
    ~TkTree();
    
    // entry point from tcl to create a tree
    static int treeCmd(ClientData, Tcl_Interp *interp, int argc, char* argv[]);

    // call a member function by name
    virtual int call(const char* name, int argc, char* argv[]);

    // read-only member access
    const char* canvas() const {return canvas_;}

    int parentDistance() const { return options_.parentDistance;}
    int borderWidth() const { return options_.borderWidth;}
    
    // boolean functions to determine the layout quickly
    int horizontal() const {return (*options_.layout != 'v');}
    int vertical() const {return (*options_.layout == 'v');}


    // -- tree subcommand methods --
    
    // add a new child node to the parent node in the tree
    int addlink_(int argc, char* argv[]);
    
    // move the subtree to a new position in the tree 
    int movelink_(int argc, char* argv[]);
    
    // remove a node and its subtree 
    int rmlink_(int argc, char* argv[]);
    
    // recalculate the size and pos and set options
    int nodeconfigure_(int argc, char* argv[]);
    
    // command to remove and delete the nodes children
    int prune_(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a leaf node (has no children)
    int isleaf_(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a root node (displays no parents)
    int isroot_(int argc, char* argv[]);
    
    // command that sets a new root node
    int root_(int argc, char* argv[]);
    
    // command to draw the tree on the canvas
    int draw_(int argc, char* argv[]);
    
    // command to return the bounding box of the tree
    int bbox_(int argc, char* argv[]);
    
    // return the name of the tree's canvas
    int getcanvas_(int argc, char* argv[]);

    // command that returns the name of the child node
    int child_(int argc, char* argv[]);
    
    // command that returns a list of the child nodes
    int subnodes_(int argc, char* argv[]);
    
    // command that returns the name of the sibling node
    int sibling_(int argc, char* argv[]);
    
    // command that returns the name of the parent node
    int parent_(int argc, char* argv[]);

    // command that returns a list of ancestors of a tree node
    int ancestors_(int argc, char* argv[]);
};


#endif /* _TkTree_h_ */
