/*
 * TkWidget.C - base class definitions for Tk widgets implemented in C++
 *
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 *
 */


#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <iostream.h>
#include "TkWidget.h"


/*
 * Call the given method in this class with the given arguments
 * (In this case there is only one command defined: "configure"
 */
int TkWidget::call(const char* name, int argc, char* argv[])
{
    if (strncmp(name, "configure", strlen(name)) == 0) {
	return configure_(argc, argv);
    }
    return TclCommand::call(name, argc, argv);
}


/*
 * Constructor - create the widget and install the tcl command of the
 * same name.
 * 
 * pclass - is the expected class of the parent widget (Canvas, Frame, ...)
 * 
 * specs - is a pointer to the widget's config struct, which defines
 *         the config options for the widget
 * 
 * options - is a derived struct that holds the values defined in specs.
 *
 * argc, argv - are passed from the widget command (argv[0] is the widget command)
 */
TkWidget::TkWidget(Tcl_Interp* interp, const char* pclass, 
		   Tk_ConfigSpec* specs, TkWidgetOptions& options,
		   int /* argc */, char* argv[])
: TclCommand(interp, argv[0], argv[1]),
  tkwin_(NULL),
  pname_(strdup(instname_)),
  wclass_(strdup(cmdname_)),
  configSpecsPtr_(specs),
  optionsPtr_(&options)
{
    // get the name of the parent window from the widgets's path
    char* p = strrchr(pname_, '.');
    if (p != NULL) 
	*p = '\0';

    if (p == NULL
	|| Tcl_VarEval(interp, "winfo class ", pname_, NULL) != TCL_OK
	|| strcmp(interp->result, pclass) != 0) {
        status_ = TCL_ERROR;
        tcl_ << "bad path name for " << cmdname_ <<  ": \"" << instname_ 
             << "\" parent of " << cmdname_ << " should be a " << pclass << " widget" 
             << ::error;
 	return;
    }
    
    // create the widget window
    tkwin_ = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), instname_, NULL);
    if (tkwin_ == NULL) {
	status_ = TCL_ERROR;
	return;
    }

    // set the class to the widget type with the first letter capitalized
    *wclass_ = toupper(*wclass_);
    Tk_SetClass(tkwin_, wclass_);
}


/* 
 * Dxestructor - called when tcl command is deleted
 */
TkWidget::~TkWidget() 
{
    free(pname_);
    free(wclass_);
}


/*
 * This procedure needs to be called from the derived class constructor
 * to complete the widget initialization. This can't be done in the 
 * constructor here since the options_ struct wouldn't be initialized yet.
 */
int TkWidget::initWidget(int argc, char* argv[])
{
    // handle the config options
    if ((status_ = configureWidget(argc-2, argv+2, 0)) != TCL_OK) {
        Tk_DestroyWindow(tkwin_);
        tkwin_ = NULL;
	return TCL_ERROR;
    }

    // return the name of the widget as a tcl result
    Tcl_SetResult(interp_, instname_, TCL_STATIC);
    return TCL_OK;
}


/*
 * This procedure is called to process an argv/argc list, plus
 * the Tk option database, in order to configure (or reconfigure) 
 * a widget.
 */
int TkWidget::configureWidget(int argc, char* argv[], int flags)
{
    if (Tk_ConfigureWidget(interp_, tkwin_, configSpecsPtr_,
	    argc, argv, (char*)optionsPtr_, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 * this is called for the configure widget command
 */
int TkWidget::configure_(int argc, char* argv[])
{
    if (argc == 0) {
	return Tk_ConfigureInfo(interp_, tkwin_, configSpecsPtr_, 
				(char*)optionsPtr_, NULL, 0);
    } else if (argc == 1) {
	return Tk_ConfigureInfo(interp_, tkwin_, configSpecsPtr_, 
				(char*)optionsPtr_, argv[0], 0);
    } else {
	return configureWidget(argc, argv, TK_CONFIG_ARGV_ONLY);
    }
}

