########################################################################
## Copyright (c) 1994, Aaron Jackson
## All rights reserved
##
## Permission to use, copy, modify, and distribute this software and
## its documentation for any purpose and without fee is hereby granted,
## provided that the above copyright notice appear in all copies and
## that both that copyright notice and this permission notice appear in
## supporting documentation.
##
## In no event shall the author(s) be liable to any party for direct,
## indirect, special, incidental, or consequential damages arising out
## of the use of this software and its documentation, even if the
## authors(s) have been advised of the possibility of such damage.
##
## The authors(s) specifically disclaim any warranties, including but
## not limited to, the implied warranties of merchantability and
## fitness for a particular purpose.  The software provided hereunder
## is on an "as is" basis, and the author(s) have no obligation to
## provide maintenance, support, updates, enhancements, or modifications
########################################################################

######################################################################
## vwt_matrix_backspace pathname
##
## This is a simple procedure to add a backspace for focused
## windows
######################################################################

proc vwt_matrix_backspace {pathname} {
    set x [expr {[$pathname focus index insert] - 1}] ;
    if {$x != -1} {$pathname focus delete $x}
}

######################################################################
## vwt_matrix_seeCaret 
##
## This ensures that the caret is in view
######################################################################

proc vwt_matrix_seeCaret {pathname} {
    set c [$pathname focus index insert] ;
    set left [$pathname focus index @0] ;

    if {$left >= $c} {
	if {$c > 0} {
	    $pathname focus view [expr $c - 1] ;
	} else {
	    $pathname focus view $c ;
	}

	return ;
    }

    set info  [$pathname focus where] ;
    set width [lindex $info 4] ;
    set x [expr $width - [lindex [$pathname configure -cellborderwidth] 4] - 1] ;
    while {([$pathname focus index @@$x] < $c) && ($left < $c)} {
	$pathname focus view [incr left] ;
    }
}

######################################################################
## vwt_matrix_select_mark pathname x y
##
## This is for setting the anchor point of a region selection.
## It makes use of the following:
##     __vwt_matrix(<pathname>,selFg)
##     __vwt_matrix(<pathname>,selBg)
##     __vwt_matrix(<pathname>,selFont)
######################################################################

proc vwt_matrix_select_mark {pathname x y} {
    if [catch { set pts [$pathname where $x $y] }] {
	return ;
    }

    if {([set col [lindex $pts 0]] == -1) ||
	([set row [lindex $pts 1]] == -1)} {
	return ;
    }

    global __vwt_matrix ;
    switch -- $__vwt_matrix($pathname,mode) {
	"focus" {
	    # Do a quick icursor, no pop required

	    set indx [$pathname focus index @$x] ;
	    $pathname focus icursor $indx ;
	    $pathname focus select clear ;
	    $pathname focus select from $indx ;
	}

	"none" {
	    catch {
		$pathname pop color \
		    $__vwt_matrix($pathname,from,x) $__vwt_matrix($pathname,from,y) \
		    $__vwt_matrix($pathname,to,x)   $__vwt_matrix($pathname,to,y) ;
	    }

	    set __vwt_matrix($pathname,from,x) $col ;
	    set __vwt_matrix($pathname,from,y) $row ;
	    set __vwt_matrix($pathname,to,x)   $__vwt_matrix($pathname,from,x) ;
	    set __vwt_matrix($pathname,to,y)   $__vwt_matrix($pathname,from,y) ;

	    $pathname push color \
		$__vwt_matrix($pathname,from,x) $__vwt_matrix($pathname,from,y) \
		$__vwt_matrix($pathname,to,x)   $__vwt_matrix($pathname,to,y) \
		$__vwt_matrix($pathname,selFg) \
		$__vwt_matrix($pathname,selBg) ;

	    # Add cell focus policy
	    
	    $pathname focus set $__vwt_matrix($pathname,from,x) $__vwt_matrix($pathname,from,y) ;
	    focus $pathname ;
	}

	"bd-right" -
	"bd-right-left" {
	    set __vwt_matrix($pathname,resize,x)  $col ;
	    set __vwt_matrix($pathname,resize,y)  $row ;
	    set __vwt_matrix($pathname,resize,xp) [lindex $pts 2] ;
	}

	"bd-bottom" -
	"bd-bottom-top" {
	    set __vwt_matrix($pathname,resize,x)  $col ;
	    set __vwt_matrix($pathname,resize,y)  $row ;
	    set __vwt_matrix($pathname,resize,yp) [lindex $pts 3] ;
	}

	"bd-bottom-right" {
	    set __vwt_matrix($pathname,resize,x)  $col ;
	    set __vwt_matrix($pathname,resize,y)  $row ;
	    set __vwt_matrix($pathname,resize,xp) [lindex $pts 2] ;
	    set __vwt_matrix($pathname,resize,yp) [lindex $pts 3] ;
	}
    }
}

######################################################################
## vwt_matrix_select_drag pathname x y
##
## This is for setting the end point of a region selection.
## It makes use of the following:
##     __vwt_matrix(<pathname>,selFg)
##     __vwt_matrix(<pathname>,selBg)
##     __vwt_matrix(<pathname>,selFont)
######################################################################

proc vwt_matrix_select_drag {pathname x y} {
    if [catch { set pts [$pathname where $x $y] }] {
	return ;
    }

    global __vwt_matrix ;
    if {([set col [lindex $pts 0]] == -1) ||
	([set row [lindex $pts 1]] == -1)} {
	return ;
    }

    switch -- $__vwt_matrix($pathname,mode) {
	"focus" {
	    # Do a quick select move, no pop required

	    set indx [$pathname focus index @$x] ;
	    $pathname focus select to $indx ;
	}

	"none" {
	    # Reset the region shown

	    catch {
		$pathname pop color \
		    $__vwt_matrix($pathname,from,x) $__vwt_matrix($pathname,from,y) \
		    $__vwt_matrix($pathname,to,x)   $__vwt_matrix($pathname,to,y) ;
	    }

	    set __vwt_matrix($pathname,to,x)   $col ;
	    set __vwt_matrix($pathname,to,y)   $row ;

	    $pathname push color \
		$__vwt_matrix($pathname,from,x) $__vwt_matrix($pathname,from,y) \
		$__vwt_matrix($pathname,to,x)   $__vwt_matrix($pathname,to,y) \
		$__vwt_matrix($pathname,selFg) \
		$__vwt_matrix($pathname,selBg) ;
	}

	"bd-right" -
	"bd-right-left" {
	    $pathname size \
		$__vwt_matrix($pathname,resize,x) \
		$__vwt_matrix($pathname,resize,y) \
		[expr $x - $__vwt_matrix($pathname,resize,xp)] \
		@current ;
	}

	"bd-bottom" -
	"bd-bottom-top" {
	    $pathname size \
		$__vwt_matrix($pathname,resize,x) \
		$__vwt_matrix($pathname,resize,y) \
		@current \
		[expr $y - $__vwt_matrix($pathname,resize,yp)] ;
	}

	"bd-bottom-right" {
	    $pathname size \
		$__vwt_matrix($pathname,resize,x) \
		$__vwt_matrix($pathname,resize,y) \
		[expr $x - $__vwt_matrix($pathname,resize,xp)] \
		[expr $y - $__vwt_matrix($pathname,resize,yp)] ;
	}
    }

}

######################################################################
## vwt_matrix_scan_mark pathname x y
##
## This is doing scanning of a focussed cell.
## It makes use of the following:
##     __vwt_matrix(<pathname>,selFg)
##     __vwt_matrix(<pathname>,selBg)
##     __vwt_matrix(<pathname>,selFont)
######################################################################

proc vwt_matrix_scan_mark {pathname x y} {
    if [catch { set pts [$pathname where $x $y] }] {
	return ;
    }

    if {([set col [lindex $pts 0]] == -1) ||
	([set row [lindex $pts 1]] == -1)} {
	return ;
    }

    global __vwt_matrix ;
    if {(($__vwt_matrix($pathname,from,x) == $col) &&
	 ($__vwt_matrix($pathname,from,y) == $row) &&
	 ($__vwt_matrix($pathname,to,x) == $col) &&
	 ($__vwt_matrix($pathname,to,y) == $row))} {
	$pathname focus scan mark $x ;
    }
}

######################################################################
## vwt_matrix_scan_drag pathname x y
##
## This is doing scanning of a focussed cell.
## It makes use of the following:
##     __vwt_matrix(<pathname>,selFg)
##     __vwt_matrix(<pathname>,selBg)
##     __vwt_matrix(<pathname>,selFont)
######################################################################

proc vwt_matrix_scan_drag {pathname x y} {
    if [catch { set pts [$pathname where $x $y] }] {
	return ;
    }

    if {([set col [lindex $pts 0]] == -1) ||
	([set row [lindex $pts 1]] == -1)} {
	return ;
    }

    global __vwt_matrix ;
    if {(($__vwt_matrix($pathname,from,x) == $col) &&
	 ($__vwt_matrix($pathname,from,y) == $row) &&
	 ($__vwt_matrix($pathname,to,x) == $col) &&
	 ($__vwt_matrix($pathname,to,y) == $row))} {
	$pathname focus scan dragto $x ;
    }
}

######################################################################
## vwt_matrix_motion pathname x y
##
## Follows motion events
######################################################################

proc vwt_matrix_motion {pathname x y} {
    global __vwt_matrix ;
    set cursor [$pathname info -padx 5 -pady 5 $x $y] ;
    set __vwt_matrix($pathname,mode) $cursor ;
    $pathname configure -cursor $__vwt_matrix(cursors,$cursor) ;
}

######################################################################
## vwt_matrix_menu pathname x y
##
## Posts a menu
######################################################################

proc vwt_matrix_menu {pathname x y} {
    $pathname.m activate none ;
    $pathname.m post $x $y ;
#    grab -global $pathname.m ;
}

######################################################################
## vwt_matrix_configure pathname configopts
##
## Extracted configuration information
######################################################################

proc vwt_matrix_configure {pathname configopts} {
    set confArgs {} ;
    set confResult {} ;

    global __vwt_matrix ;

    set index  0 ;
    if ![set length [llength $configopts]] {
	set confResult \
	    [list \
		 [list -regionFont regionFont RegionFont \
		      $__vwt_matrix($pathname,selFont) \
		      $__vwt_matrix($pathname,selFont) ] \
		 [list -regionBg regionBackground RegionBackground \
		      $__vwt_matrix($pathname,selBg) \
		      $__vwt_matrix($pathname,selBg) ] \
		 [list -regionFg regionForeground RegionForeground \
		      $__vwt_matrix($pathname,selFg) \
		      $__vwt_matrix($pathname,selFg) ] \
		];

    }

    while {$index < $length} {
	switch -- [set arg [lindex $configopts $index]] {
	    "-regionFont" {
		if {$index == ($length - 1)} {
		    lappend confResult \
			[list -regionFont regionFont RegionFont \
			     $__vwt_matrix($pathname,selFont) \
			     $__vwt_matrix($pathname,selFont) ] ;
		} else {
		    set __vwt_matrix($pathname,selFont) [lindex $configopts [incr index]] ;
		}
	    }

	    "-regionBg" -
	    "-regionBackground" {
		if {$index == ($length - 1)} {
		    lappend confResult \
			[list -regionBg regionBackground RegionBackground \
			     $__vwt_matrix($pathname,selBg) \
			     $__vwt_matrix($pathname,selBg) ] ;
		} else {
		    set __vwt_matrix($pathname,selBg) [lindex $configopts [incr index]] ;
		}
	    }

	    "-regionFg" -
	    "-regionForeground" {
		if {$index == ($length - 1)} {
		    lappend confResult \
			[list -regionFg regionForeground RegionForeground \
			     $__vwt_matrix($pathname,selFg) \
			     $__vwt_matrix($pathname,selFg) ] ;
		} else {
		    set __vwt_matrix($pathname,selFg) [lindex $configopts [incr index]] ;
		}
	    }

	    default {
		set confArgs [concat $confArgs [lrange $configopts $index [incr index]]]
	    }
	}
	incr index ;
    }
    
    set confResult [concat $confResult [eval "$pathname configure $confArgs"]] ;
    return [lsort $confResult] ;
}

######################################################################
## Editting functions
######################################################################

proc vwt_matrix_copy {pathname} {
    global __vwt_matrix ;

    catch {
	set __vwt_matrix($pathname,copy,data) \
	    [$pathname get \
		 $__vwt_matrix($pathname,from,x) \
		 $__vwt_matrix($pathname,from,y) \
		 $__vwt_matrix($pathname,to,x) \
		 $__vwt_matrix($pathname,to,y)] ;

	set __vwt_matrix($pathname,copy,cols) \
	    [expr ($__vwt_matrix($pathname,to,x) - \
		   $__vwt_matrix($pathname,from,x) + 1)] ;

	set __vwt_matrix($pathname,copy,rows) \
	    [expr ($__vwt_matrix($pathname,to,y) - \
		   $__vwt_matrix($pathname,from,y) + 1)] ;

	if {$__vwt_matrix($pathname,copy,cols) < 0} {
	    set __vwt_matrix($pathname,copy,cols) -$__vwt_matrix($pathname,copy,cols) ;
	}

	if {$__vwt_matrix($pathname,copy,rows) < 0} {
	    set __vwt_matrix($pathname,copy,rows) -$__vwt_matrix($pathname,copy,rows) ;
	}
    }
}

proc vwt_matrix_paste {pathname} {
    global __vwt_matrix ;

    catch {
	set end_x [expr $__vwt_matrix($pathname,from,x) + $__vwt_matrix($pathname,copy,cols)] ;
	set end_y [expr $__vwt_matrix($pathname,from,y) + $__vwt_matrix($pathname,copy,rows)] ;

	$pathname set \
	    $__vwt_matrix($pathname,from,x) \
	    $__vwt_matrix($pathname,from,y) \
	    $end_x \
	    $end_y \
	    $__vwt_matrix($pathname,copy,data) ;
    }
}

######################################################################
## vwt_matrix_option pathname command ? args ... ?
## 
## Root processor for commands
######################################################################

proc vwt_matrix_option {pathname funcname option args} {
    switch -- $option {
	configure  { return [vwt_matrix_configure $pathname.matrix $args] ; }
	region     { return [vwt_matrix_region $pathname.matrix $args] ; }
	cut        { eval "vwt_matrix_copy $pathname.matrix ; vwt_matrix_delete $pathname.matrix" ; }
	paste      { eval "vwt_matrix_paste $pathname.matrix" ; }
	delete     { eval "vwt_matrix_delete $pathname.matrix" ; }
	copy       { eval "vwt_matrix_copy $pathname.matrix" ; }
	default    { eval "$pathname.matrix $option $args" ; }
    }
}

######################################################################
## vwt_matrix pathname ? args ... ?
##
## Creates a matrix widget w/ a horizontal and vertical scrollbar
######################################################################

proc vwt_matrix {pathname args} {
    global __vwt_matrix ;

    # Create <pathname> and then fool tcl into believing that
    # <pathname> is now __vwt_matrix_$pathname and create a
    # procedure for <pathname> which called vwt_matrix_option

    set funcname "__vwt_matrix_[string trimleft $pathname .]" ;

    frame  $pathname ;
    rename $pathname $funcname ;
    proc  $pathname {option args} "eval \"vwt_matrix_option $pathname $funcname \$option \$args\""

    # Create the rest of the widgets in the matrix

    vwtable   $pathname.matrix ;
    scrollbar $pathname.vscroll -orient vertical ;
    scrollbar $pathname.hscroll -orient horizontal ;
    menu      $pathname.matrix.m ;

    $pathname.vscroll configure -command "$pathname.matrix yview" ;
    $pathname.hscroll configure -command "$pathname.matrix xview" ;
    $pathname.matrix  configure -xscroll "$pathname.hscroll set" \
	                        -yscroll "$pathname.vscroll set" ;

    pack $pathname.hscroll -side bottom -fill x ;
    pack $pathname.vscroll -side right  -fill y ;
    pack $pathname.matrix  -side right  -fill both -expand yes ;

    $pathname.matrix.m add command -label "Edit" ;
    $pathname.matrix.m add separator ;
    $pathname.matrix.m add command -label "Copy"   -command "$pathname copy" ;
    $pathname.matrix.m add command -label "Cut"    -command "$pathname cut" ;
    $pathname.matrix.m add command -label "Delete" -command "$pathname delete" ;
    $pathname.matrix.m add command -label "Paste"  -command "$pathname paste" ;

    # Add some bindings that we will need for future clean up
    # and some bindings that are default to this widget

    bind $pathname <Destroy>          "rename $funcname {} ;"
    bind $pathname.matrix <1>         {vwt_matrix_select_mark %W %x %y}
    bind $pathname.matrix <B1-Motion> {vwt_matrix_select_drag %W %x %y}
    bind $pathname.matrix <2>         {vwt_matrix_scan_mark %W %x %y}
    bind $pathname.matrix <B2-Motion> {vwt_matrix_scan_drag %W %x %y}
    bind $pathname.matrix <3>         {vwt_matrix_menu %W %x %y}
    bind $pathname.matrix <Motion>    {vwt_matrix_motion %W %x %y}
    bind $pathname.matrix <Control-a> {%W focus icursor 0}
    bind $pathname.matrix <Control-b> {%W focus icursor [expr [%W focus index insert] - 1]}
    bind $pathname.matrix <Control-e> {%W focus icursor end}
    bind $pathname.matrix <Control-f> {%W focus icursor [expr [%W focus index insert] + 1]}
    bind $pathname.matrix <Control-k> {%W focus delete insert end}
    bind $pathname.matrix <Control-u> {%W focus delete 0 end}
    bind $pathname.matrix <Control-h> {vwt_matrix_backspace %W ; vwt_matrix_seeCaret %W}
    bind $pathname.matrix <BackSpace> {vwt_matrix_backspace %W ; vwt_matrix_seeCaret %W}
    bind $pathname.matrix <Delete>    {
	if {[%W focus index sel.last] != -1} {
	    %W focus delete sel.first sel.last ;
	    %W focus select clear ;
	} else {
	    vwt_matrix_backspace %W ;
	}
	vwt_matrix_seeCaret %W ;
    }
    bind $pathname.matrix <Any-KeyPress> {
	if {"%A" != ""} {
	    %W focus insert insert %A
	}
	vwt_matrix_seeCaret %W ;
    }

    # Extract some selection information which may prove handy

    set __vwt_matrix($pathname.matrix,selFg)   [option get $pathname regionForeground RegionForeground] ;
    set __vwt_matrix($pathname.matrix,selBg)   [option get $pathname regionBackground RegionBackground] ;
    set __vwt_matrix($pathname.matrix,selFont) [option get $pathname regionFont RegionFont] ;

    set __vwt_matrix($pathname.matrix,from,x) -1 ;
    set __vwt_matrix($pathname.matrix,from,y) -1 ;
    set __vwt_matrix($pathname.matrix,to,x)   $__vwt_matrix($pathname.matrix,from,x) ;
    set __vwt_matrix($pathname.matrix,to,y)   $__vwt_matrix($pathname.matrix,from,y) ;

    if {$__vwt_matrix($pathname.matrix,selFg) == ""} { set __vwt_matrix($pathname.matrix,selFg) white }
    if {$__vwt_matrix($pathname.matrix,selBg) == ""} { set __vwt_matrix($pathname.matrix,selBg) grey80 }

    if ![info exists __vwt_matrix(cursors)] {
	set __vwt_matrix(cursors)                          {} ;
	set __vwt_matrix(cursors,none)                     {} ;
	set __vwt_matrix(cursors,space)                    {} ;
	set __vwt_matrix(cursors,focus)                    xterm ;
	set __vwt_matrix(cursors,bd-left)                  {} ;
	set __vwt_matrix(cursors,bd-right)                 sb_h_double_arrow ;
	set __vwt_matrix(cursors,bd-right-left)            sb_h_double_arrow ;
	set __vwt_matrix(cursors,bd-top)                   {} ;
	set __vwt_matrix(cursors,bd-top-left)              {} ;
	set __vwt_matrix(cursors,bd-top-right)             {} ;
	set __vwt_matrix(cursors,bd-top-right-left)        {} ;
	set __vwt_matrix(cursors,bd-bottom)                sb_v_double_arrow ;
	set __vwt_matrix(cursors,bd-bottom-left)           {} ;
	set __vwt_matrix(cursors,bd-bottom-right)          bottom_right_corner ;
	set __vwt_matrix(cursors,bd-bottom-right-left)     {} ;
	set __vwt_matrix(cursors,bd-bottom-top)            sb_v_double_arrow ;
	set __vwt_matrix(cursors,bd-bottom-top-left)       {} ;
	set __vwt_matrix(cursors,bd-bottom-top-right)      {} ;
	set __vwt_matrix(cursors,bd-bottom-top-right-left) {} ;
    }


    if [llength $args] {
	eval "$pathname configure $args" ;
    }
}

