/************************************************************************
 ** Copyright (c) 1994, Aaron Jackson
 ** All rights reserved
 **
 ** Permission to use, copy, modify, and distribute this software and
 ** its documentation for any purpose and without fee is hereby granted,
 ** provided that the above copyright notice appear in all copies and
 ** that both that copyright notice and this permission notice appear in
 ** supporting documentation.
 **
 ** In no event shall the author(s) be liable to any party for direct,
 ** indirect, special, incidental, or consequential damages arising out
 ** of the use of this software and its documentation, even if the
 ** authors(s) have been advised of the possibility of such damage.
 **
 ** The authors(s) specifically disclaim any warranties, including but
 ** not limited to, the implied warranties of merchantability and
 ** fitness for a particular purpose.  The software provided hereunder
 ** is on an "as is" basis, and the author(s) have no obligation to
 ** provide maintenance, support, updates, enhancements, or modifications
 ************************************************************************/

static char rcsid[] = "$Header: /home/edsdev/cvstree/tcldev/vwtable/vwMatrix.c,v 1.3 1994/11/11 22:35:45 aajackso Exp $" ;

/**
 ** $Log: vwMatrix.c,v $
 * Revision 1.3  1994/11/11  22:35:45  aajackso
 * *** empty log message ***
 *
 * Revision 1.2  1994/11/11  21:36:33  aajackso
 * *** empty log message ***
 *
 * Revision 1.1.1.1  1994/10/21  13:35:21  aajackso
 * Table Widget
 *
 * Revision 1.1  1994/08/24  00:55:59  aaron
 * Initial revision
 *
 **/

#include <malloc.h>
#include <memory.h>
#include <vwUtil.h>
#include <vwMatrix.h>

#ifdef Min
#undef Min
#endif /** Min **/

#define Min(xx,yy) (((xx) > (yy)) ? (yy) : (xx))

#ifndef True
#define True (1)
#endif

#ifndef False
#define False (0)
#endif

#ifdef STRICT_DEBUG
#include <stdio.h>
#endif /** STRICT_DEBUG **/

/************************************************************************
 ** void
 ** VWMatrix_ShiftX( VWMatrix*     matrix,
 **                  int           offset,
 **                  int           shifts )
 **
 ************************************************************************/

#ifdef __STDC__
void
VWMatrix_ShiftX( VWMatrix*          matrix,
                 int                offset,
                 int                shifts )
#else /** __STDC__ **/
void
VWMatrix_ShiftX( matrix, offset, shifts )
    VWMatrix*          matrix ;
    int                offset ;
    int                shifts ;
#endif /** __STDC__ **/
{
  char  *defval ;
  char  *marker ;
  int    z ;
  int    y ;

  if ((! matrix) || (! shifts))
    return ;

  defval = (char *) malloc(matrix->size) ;
  memset(defval, 0x0, matrix->size) ;

  /**
   ** Make other people happy at my expense!
   **/

  marker = matrix->memory ;

  for ( z = 0 ; z < matrix->depth ; z++ )
    for ( y = 0 ; y < matrix->height ; y++ )
      {
	VWUtil_Shift(marker, 
		     matrix->width,
		     matrix->size,
		     offset, shifts,
		     defval) ;

	marker += matrix->width * matrix->size ;
      }

  free((char *) defval) ;
}

/************************************************************************
 ** void
 ** VWMatrix_ShiftY( VWMatrix*     matrix,
 **                  int           offset,
 **                  int           shifts )
 **
 ************************************************************************/

#ifdef __STDC__
void
VWMatrix_ShiftY( VWMatrix*          matrix,
                 int                offset,
                 int                shifts )
#else /** __STDC__ **/
void
VWMatrix_ShiftY( matrix, offset, shifts )
    VWMatrix*          matrix ;
    int                offset ;
    int                shifts ;
#endif /** __STDC__ **/
{
  char  *defval ;
  char  *marker ;
  int    z ;

  if ((! matrix) || (! shifts))
    return ;

  defval = (char *) malloc(matrix->y_block_size) ;
  memset(defval, 0x0, matrix->y_block_size) ;

  /**
   ** Make other people happy at my expense!
   **/

  marker = matrix->memory ;

  for ( z = 0 ; z < matrix->depth ; z++ )
    {
      VWUtil_Shift(marker, 
		   matrix->height,
		   matrix->y_block_size,
		   offset, shifts,
		   defval) ;

      marker += matrix->y_block_size ;
    }

  free((char *) defval) ;
}

/************************************************************************
 ** void
 ** VWMatrix_Free( VWMatrix* matrix )
 **
 ** Frees the matrix.  You are responsible for freeing the memory
 ** associated in that matrix BEFORE calling this
 ************************************************************************/

#ifdef __STDC__
void
VWMatrix_Free( VWMatrix*    matrix )
#else /** __STDC__ **/
void
VWMatrix_Free( matrix )
    VWMatrix*    matrix ;
#endif /** __STDC__ **/
{
  if (matrix->memory)
    free((char *) matrix->memory) ;
}

/************************************************************************
 ** int
 ** VWMatrix_Alloc( VWMatrix*      matrix,
 **                 unsigned int   width,
 **                 unsigned int   height,
 **                 unsigned int   depth )
 **
 ************************************************************************/

#ifdef __STDC__
int
VWMatrix_Alloc( VWMatrix*             matrix,
                unsigned int          width,
                unsigned int          height,
                unsigned int          depth )
#else /** __STDC__ **/
int
VWMatrix_Alloc( matrix, width, height, depth )
    VWMatrix*             matrix ;
    unsigned int          width ;
    unsigned int          height ;
    unsigned int          depth ;
#endif /** __STDC__ **/
{
  unsigned long allocation ;
  char*         dest ;
  char*         src ;
  char*         cache ;

  unsigned long zero_size ;

  unsigned long z_block_size ;
  unsigned long y_block_size ;
  unsigned long x_block_size ;

  unsigned long z_block_movement[2] ;
  unsigned long y_block_movement[2] ;

  unsigned char z_block_growing ;
  unsigned char y_block_growing ;

  unsigned int  min[3] ;
  unsigned int  z ;
  unsigned int  y ;

  if (( width == matrix->width ) &&
      ( height == matrix->height ) &&
      ( depth == matrix->depth ))
    return True ;

  allocation = width * height * depth * matrix->size ;
  if (allocation > matrix->allocated_memory)
    {
      char *ptr ;

      ptr = ((matrix->memory != ((char *) 0)) ?
	     (realloc(matrix->memory, allocation)) :
	     (calloc(1, allocation))) ;

      if (ptr == ((char *) 0))
	return False ;

      matrix->memory = ptr ;
      matrix->allocated_memory = allocation ;
    }

  min[0] = Min(matrix->width, width) ;
  min[1] = Min(matrix->height, height) ;
  min[2] = Min(matrix->depth, depth) ;

  /**
   ** Direction is determined by the size of the new block
   ** Linear allocation is done as such:
   **
   ** -- Blocks ordered by z
   ** -- Blocks ordered by y
   ** -- Blocks ordered by x
   **
   ** Therefore x is the smallest piece of linear memory
   ** and z is the largest.  Organization is at its most
   ** point ordered by z.
   **/

  x_block_size = matrix->size ;
  y_block_size = x_block_size * width ;
  z_block_size = y_block_size * height ;

  zero_size = x_block_size * (width - min[0]) ;
  min[0] *= x_block_size ;

  cache = (char *) malloc(min[0]) ;
  
  /**
   ** Linear memory map:
   **
   ** {                     Z-Block                         }
   ** {          Y-Block        }                           |
   ** { X-Block }               |                           |
   **           |               |                           |
   ** +---------+---------------+---------------------------+
   ** +         |               |                           +
   ** +-----------------------------------------------------+
   **/

  /**
   ** Is the z-block growing or shrinking?
   **
   ** If the z-block is growing we must reallocate memory from the
   ** back to the front so that we do not overwrite critical data.
   ** Likewise, if the z-block is shrinking we want to allocate
   ** from front to back so that the smaller block sizes dont
   ** overwrite previous data
   **/

  z_block_growing = ( z_block_size > matrix->z_block_size ) ;
  y_block_growing = ( y_block_size > matrix->y_block_size ) ;

  src  = matrix->memory ;
  dest = matrix->memory ;

#ifdef STRICT_DEBUG
  fprintf(stderr, "memlocations: %x -- %x\n", src, dest) ;
#endif /** STRICT_DEBUG **/

  if ( z_block_growing )
    {
      src  += matrix->z_block_size * (matrix->depth - 1) ;
      dest += z_block_size * (matrix->depth - 1) ;
      z_block_movement[0] = -matrix->z_block_size ;
      z_block_movement[1] = -z_block_size ;
    }
  else
    {
      z_block_movement[0] = matrix->z_block_size ;
      z_block_movement[1] = z_block_size ;
    }

#ifdef STRICT_DEBUG
  fprintf(stderr, "z_block_growing: %d\n", z_block_growing) ;
  fprintf(stderr, "y_block_growing: %d\n", y_block_growing) ;
  fprintf(stderr, "memadjusted: %x -- %x\n", src, dest) ;
#endif /** STRICT_DEBUG **/

  for ( z = 0 ; z < min[2] ; z++, src += z_block_movement[0], dest += z_block_movement[1] )
    {
      memset(dest + y_block_size * min[1], 0x0, y_block_size * (height - min[1])) ;
#ifdef STRICT_DEBUG
      fprintf(stderr, "yblocks zero'd from %x to %x -- %d bytes (%d blocks)\n",
	      dest + y_block_size * min[1],
	      dest + y_block_size * min[1] + y_block_size * (height - min[1]),
	      y_block_size * (height - min[1]),
	      height - min[1]) ;
#endif

      if ( y_block_growing )
	{
	  src  += matrix->y_block_size * (matrix->height - 1) ;
	  dest += y_block_size * (matrix->height - 1) ;
	  y_block_movement[0] = -matrix->y_block_size ;
	  y_block_movement[1] = -y_block_size ;
	}
      else
	{
	  y_block_movement[0] = matrix->y_block_size ;
	  y_block_movement[1] = y_block_size ;
	}

#ifdef STRICT_DEBUG
      fprintf(stderr, "yblock adjusted: %x -- %x\n", src, dest) ;
#endif /** STRICT_DEBUG **/

      for ( y = 0 ; y < min[1] ; y++, src += y_block_movement[0], dest += y_block_movement[1] )
	{
#ifdef STRICT_DEBUG
	  fprintf(stderr, "x mod block(%3d): %x -- %x\n", y, src, dest) ;
#endif /** STRICT_DEBUG **/

	  /** memcpy() does not guarantee protection for overlapping regions **/

	  memcpy(cache, src, min[0]) ;
	  memcpy(dest, cache, min[0]) ;
	  memset(dest + min[0], 0x0, zero_size) ;

#ifdef STRICT_DEBUG
	  fprintf(stderr, "copy: %08x --> %08x (%04x)\n", src, dest, min[0]) ;
	  fprintf(stderr, "zero: %08x --> %08x (%04x)\n", dest + min[0], dest + min[0] + zero_size, zero_size) ;
#endif /** STRICT_DEBUG **/
	}
    }

  free(cache) ;

  matrix->z_block_size = z_block_size ;
  matrix->y_block_size = y_block_size ;

  matrix->width = width ;
  matrix->height = height ;
  matrix->depth = depth ;

  return True ;
}

		  
