/************************************************************************
 ** Copyright (c) 1994, Aaron Jackson
 ** All rights reserved
 **
 ** Permission to use, copy, modify, and distribute this software and
 ** its documentation for any purpose and without fee is hereby granted,
 ** provided that the above copyright notice appear in all copies and
 ** that both that copyright notice and this permission notice appear in
 ** supporting documentation.
 **
 ** In no event shall the author(s) be liable to any party for direct,
 ** indirect, special, incidental, or consequential damages arising out
 ** of the use of this software and its documentation, even if the
 ** authors(s) have been advised of the possibility of such damage.
 **
 ** The authors(s) specifically disclaim any warranties, including but
 ** not limited to, the implied warranties of merchantability and
 ** fitness for a particular purpose.  The software provided hereunder
 ** is on an "as is" basis, and the author(s) have no obligation to
 ** provide maintenance, support, updates, enhancements, or modifications
 ************************************************************************/

static char rcsid[] = "$Header: /home/edsdev/cvstree/tcldev/vwtable/vwTableDisplay.c,v 1.3 1994/11/11 21:36:53 aajackso Exp $" ;

/**
 ** $Log: vwTableDisplay.c,v $
 * Revision 1.3  1994/11/11  21:36:53  aajackso
 * *** empty log message ***
 *
 * Revision 1.2  1994/10/25  21:10:03  aajackso
 * Oops, added __STDC__ code for most source code and changed the Makefile
 * so that it was consistent w/ the new files
 *
 * Revision 1.1.1.1  1994/10/21  13:35:27  aajackso
 * Table Widget
 *
 * Revision 1.1  1994/08/24  00:55:59  aaron
 * Added compatibility with push/pop color stacks
 * Added compatibility with tremor code (ie. pixmap shifting)
 * Fixed edge of screen bug
 *
 * Revision 1.0  1994/08/17  23:54:06  aaron
 * Initial revision
 *
 **/

#include <tk.h>
#include <tkInt.h>

#include <tcl.h>

#include <vwTable.h>
#include <vwTableCell.h>
#include <vwTableColor.h>
#include <vwTableFont.h>
#include <vwTableImage.h>

#ifdef Max
#undef Max
#endif /** Max **/

#define Max(xx,yy) (((xx) > (yy)) ? (xx) : (yy))

#ifdef Min
#undef Min
#endif /** Min **/

#define Min(xx,yy) (((xx) > (yy)) ? (yy) : (xx))

typedef struct StateInfo StateInfo ;
struct         StateInfo
{
  int               width ;
  int               height ;
  unsigned long     fg ;
  unsigned long     bg ;
  unsigned long     default_fg ;
  unsigned long     default_bg ;
  XFontStruct*      font ;
};

/**
 ** Revision @ v0.32:
 **   The image data needs to be reorganized.  The data has always
 **   been viewed as sequential. (i.e. columns going from left to
 **   right)  This concept needs to be thrown out because we now
 **   have this concept of posted and hidden "positions".  The
 **   display image structure will be restructured to correctly
 **   model this type of data.
 **/

/* #define STRICT_DEBUG */

/************************************************************************
 ** static int
 ** MeasureChars( XFontStruct    *font,
 **               char           *data,
 **               int             len )
 **
 ** Measures the string returns the width
 ************************************************************************/
 
#ifdef __STDC__
static int
MeasureChars( XFontStruct*          font,
              char*                 data,
              int                   len )
#else /** __STDC__ **/
static int
MeasureChars( font, data, len )
    XFontStruct*          font ;
    char*                 data ;
    int                   len ;
#endif /** __STDC__ **/
{
  int end ;
  
  if (data == NULL)
    return 0 ;

  TkMeasureChars(font, data, len, 0, 0xffff, TK_NEWLINES_NOT_SPECIAL, &end) ;
  return end ;
}

/************************************************************************
 ** static void
 ** DisplayChars( VWTable        *vwptr,
 **               StateInfo      *state,
 **               int             x,
 **               int             y,
 **               unsigned int    fg,
 **               unsigned int    bg,
 **               XFontStruct    *font,
 **               char           *data,
 **               int             len )
 **
 ** Uh,
 ************************************************************************/

#ifdef __STDC__
static void
DisplayChars( VWTable*                            vwptr,
              StateInfo*                          state,
              int                                 x,
              int                                 y,
              unsigned int                        fg,
              unsigned int                        bg,
              XFontStruct*                        font,
              char*                               data,
              int                                 len )
#else /** __STDC__ **/
static void
DisplayChars( vwptr, state, x, y, fg, bg, font, data, len )
    VWTable*                            vwptr ;
    StateInfo*                          state ;
    int                                 x ;
    int                                 y ;
    unsigned int                        fg ;
    unsigned int                        bg ;
    XFontStruct*                        font ;
    char*                               data ;
    int                                 len ;
#endif /** __STDC__ **/
{
  int        gcmask = 0 ;
  XGCValues  gcvalues ;

  if (! data)
    return ;

  /**
   ** Make appropriate changes to the GC and state variable
   **/

  if (state->font != font)
    {
      gcmask |= GCFont ;
      gcvalues.font = font->fid ;
      state->font = font ;
    }

  /**
   ** Adjust the foreground pixel values
   **/

  if (state->fg != fg)
    {
      gcmask |= GCForeground ;
      gcvalues.foreground = state->fg = fg ;
    }

  /**
   ** Adjust the background pixel values
   **/

  if (state->bg != bg)
    {
      gcmask |= GCBackground ;
      gcvalues.background = state->bg = bg ;
    }
  
  /**
   ** Change the graphics context
   **/
  
  if (gcmask)
    XChangeGC(vwptr->display, vwptr->gc, gcmask, &gcvalues) ;

  /**
   ** Display the characters and move on
   **/

  TkDisplayChars(vwptr->display,
		 vwptr->image.pixmap,
		 vwptr->gc,
		 font,
		 data, len,
		 x, y,
		 0L) ;
}

/************************************************************************
 ** int
 ** VWT_GetDisplayIndex( VWTable       *vwptr,
 **                      VWTableCell   *vwcell,
 **                      int            width,
 **                      CONST char    *data,
 **                      CONST int      len )
 **
 ** API users can call this to find out index will be displayed at width.
 ** This is useful for determining what index is at a specific pixel.
 **
 ** This function has no concept of error.  If vwcell, vwptr or data are
 ** NULL this function will return 0 believing that there is no
 ** information to display.
 ************************************************************************/

#ifdef __STDC__
int
VWT_GetDisplayIndex( VWTable*                 vwptr,
                     VWTableCell*             vwcell,
                     int                      width,
                     CONST char*              data,
                     CONST int                len )
#else /** __STDC__ **/
int
VWT_GetDisplayIndex( vwptr, vwcell, width, data, len )
    VWTable*                 vwptr ;
    VWTableCell*             vwcell ;
    int                      width ;
    CONST char*              data ;
    CONST int                len ;
#endif /** __STDC__ **/
{
  int             end ;
  int             chars ;
  VWTableFont*    vwfont ;
  XFontStruct*    localFont ;

  /**
   ** Check the NULL conditions
   **/

  if (( vwptr == NULL ) ||
      ( vwcell == NULL ) ||
      ( data == NULL ))
    return 0 ;

  /**
   ** Determine the appropriate font for this cell.
   ** If no font is present use the font of the parent
   **/

  vwfont = (vwcell == NULL) ? (NULL) : (VWTableFont *) VWStack_Peek(&vwcell->fonts) ;
  localFont = ((vwfont == NULL) ? vwptr->fontPtr : vwfont->font) ;

  chars = TkMeasureChars(localFont, 
			 data, len,
			 vwptr->borderWidth,
			 width,
			 TK_NEWLINES_NOT_SPECIAL,
			 &end);

  if (end > width)
    chars++ ;

  return chars ;
}

/************************************************************************
 ** static void
 ** DisplayBlink( StateInfo         *state,
 **               VWTable           *vwptr,
 **               VWTableCell       *vwcell,
 **               int                x,
 **               int                y )
 **
 ** Does a blink handling for a cell
 ************************************************************************/

#ifdef __STDC__
static void
DisplayBlink( StateInfo*                state,
              VWTable*                  vwptr,
              VWTableCell*              vwcell,
              int                       x,
              int                       y )
#else /** __STDC__ **/
static void
DisplayBlink( state, vwptr, vwcell, x, y )
    StateInfo*                state ;
    VWTable*                  vwptr ;
    VWTableCell*              vwcell ;
    int                       x ;
    int                       y ;
#endif /** __STDC__ **/
{
  int                end ;
  int                incr ;
  int                chars = 1 ;

  XFontStruct*       font ;
  int                fontHeight ;

  unsigned int       normFG ;
  unsigned int       normBG ;
  unsigned int       seleFG ;
  unsigned int       seleBG ;
    
  VWTableFont*       vwfont ;
  VWTableColorPair*  vwcolor ;
  XRectangle         clips ;

  int                len = vwptr->focus.len ;
  char*              data = vwptr->focus.data ;

#ifdef STRICT_DEBUG
  fprintf(stderr, "Blink\n") ;
#endif /** STRICT_DEBUG **/

  /**
   ** Array index 0 may not be the leftmost character.  Check what
   ** the focus information says
   **/

  data += vwptr->focus.vpos ;
  len -= vwptr->focus.vpos ;

  /**
   ** Determine the appropriate font for this cell.
   ** If no font is present use the font of the parent
   **/

  vwfont = (vwcell == NULL) ? (NULL) : (VWTableFont *) VWStack_Peek(&vwcell->fonts) ;
  if (vwfont == NULL)
    {
      font = vwptr->fontPtr ;
      fontHeight = vwptr->fontHeight ;
    }
  else
    {
      font = vwfont->font ;
      fontHeight = vwfont->height ;
    }

  /**
   ** Adjust the foreground pixel values
   ** Adjust the background pixel values
   **/

  vwcolor = (vwcell == NULL) ? (NULL) : ((VWTableColorPair *) VWStack_Peek(&vwcell->colors)) ;
  if (vwcolor)
    {
      normFG = vwcolor->fg ;
      normBG = vwcolor->bg ;
    }
  else
    {
      normFG = state->default_fg ;
      normBG = state->default_bg ;
    }

  seleFG = vwptr->focus.selFg->pixel ;
  seleBG = normBG ;

  /** In case the fonts been changed adjust **/

  vwptr->focus.avgWidth = font->per_char['0'].width ;

  /** Handle the anchoring **/

  incr = (state->height - fontHeight) / 2 + font->ascent ;
  if (incr >= 0)
    y += incr ;

  /**
   ** Initialize the clipping regions
   **/

  clips.x = x ;
  clips.y = y - font->ascent ;
  clips.width = state->width ;
  clips.height = state->height ;

  XSetClipRectangles(vwptr->display,
		     vwptr->gc,
		     0, 0,
		     &clips, 1,
		     Unsorted) ;

  if (data != NULL)
    {
      /**
       ** There is a 3 phase display here.  We do as follows:
       **    Draw non-selected front
       **    Draw non-selected back
       **    Draw selected middle
       **/

      int   last ;
      int   first ;
      int   incr ;

      int   llen ; char* lchars ;
      int   mlen ; char* mchars ;
      int   rlen ; char* rchars ;

      int   m[3] ;  /** X markers **/
      int   w[3] ;  /** Widths **/
      int   h ;

      /**
       ** Setup markers for the last and first
       ** points of selection
       **/

      if ((vwptr->focus.selTo == -1) || (vwptr->focus.selFrom == -1))
	{
	  last = first = -1 ;
	}
      else if (vwptr->focus.selTo > vwptr->focus.selFrom)
	{
	  last = vwptr->focus.selTo ;
	  first = vwptr->focus.selFrom ;
	}
      else
	{
	  first = vwptr->focus.selTo ;
	  last = vwptr->focus.selFrom ;
	}

      /**
       ** Display routines
       **/

      lchars = data ;
      llen = TkMeasureChars(font, 
			    data, len,
			    vwptr->borderWidth,
			    state->width,
			    TK_NEWLINES_NOT_SPECIAL,
			    &end) ;
      chars = llen ;
      if (end < state->width)
	chars++ ;

      if ((first == -1) ||
	  (last < vwptr->focus.vpos) ||
	  (first > (vwptr->focus.vpos + llen)))
	{
	  mchars = NULL ; mlen = 0 ;
	  rchars = NULL ; rlen = 0 ;
	}
      else
	{
	  if (first < vwptr->focus.vpos)
	    {
	      incr = Min(last - vwptr->focus.vpos, llen) ;
	      if (incr == llen)
		rchars = NULL, rlen = 0 ;
	      else
		rchars = data + incr, rlen = llen - incr ;
		    
	      mchars = data ; mlen = incr ;
	      lchars = NULL ; llen = 0 ;
	    }
	  else
	    {
	      incr = first - vwptr->focus.vpos ;
	      mchars = data + incr ; mlen = Min(last, vwptr->focus.vpos + llen) - first ;

	      rlen = llen - mlen - incr ;
	      rchars = ((rlen) ? mchars + mlen : NULL) ;

	      llen = incr ;
	    }
	}		

      /**
       ** It'd be damn nice if TkDisplayChars told you where the
       ** next position is.  Unfortunately that means we'll just
       ** have to do it ourselves
       **/
	
      w[0] = MeasureChars(font, lchars, llen) ;
      w[1] = MeasureChars(font, mchars, mlen) ;

      m[0] = x ;
      m[1] = m[0] + w[0] ;
      m[2] = m[1] + w[1] ;

      h = font->ascent + font->descent + vwptr->focus.selBorderWidth ;

      Tk_Fill3DRectangle(vwptr->display,
			 vwptr->image.pixmap,
			 vwptr->focus.selBorder,
			 m[1], y - font->ascent - vwptr->focus.selBorderWidth,
			 w[1], Min(h, state->height),
			 vwptr->focus.selBorderWidth,
			 TK_RELIEF_RAISED) ;

      DisplayChars(vwptr, state, m[0], y, normFG, normBG, font, lchars, llen) ;
      DisplayChars(vwptr, state, m[1], y, seleFG, seleBG, font, mchars, mlen) ;
      DisplayChars(vwptr, state, m[2], y, normFG, normBG, font, rchars, rlen) ;
    }

  if (( vwptr->flags & VWT_CursorOn ) &&
      ( vwptr->focus.curp >= vwptr->focus.vpos ) &&
      ( vwptr->focus.curp < vwptr->focus.vpos + chars ))
    {
      int pseudo_width = Min(state->width, vwptr->focus.width) ;
      int pseudo_height = Min(font->ascent + font->descent, state->height) ;
      int pseudo_end = 0 ;

      /**
       ** Unfortunate limitation of the Tk library is that will tell you
       ** how many characters fit into a region but now where a character
       ** is in a region.
	 **
	 ** If one operates under the premise that the cursor position is
	 ** visible (this may not be true ; see above conditional) then we
	 ** can measure only to the cursor position and TkMeasureChars will
	 ** tell us the x-position of the terminating character.
	 **/

      if (data)
	TkMeasureChars(font,
		       data,
		       vwptr->focus.curp - vwptr->focus.vpos,
		       vwptr->borderWidth,
		       state->width,
		       TK_NEWLINES_NOT_SPECIAL,
		       &pseudo_end);

      Tk_Fill3DRectangle(vwptr->display,
			 vwptr->image.pixmap,
			 vwptr->focus.background,
			 x + pseudo_end,
			 y - font->ascent,
			 pseudo_width,
			 pseudo_height,
			 vwptr->focus.borderWidth,
			 TK_RELIEF_RAISED);
    }

  /** Remove clipping regions **/

  XSetClipOrigin(vwptr->display, vwptr->gc, 0, 0) ;
  XSetClipMask(vwptr->display, vwptr->gc, None) ;
}

/************************************************************************
 ** static void
 ** Display_BitmapCell( StateInfo         *state,
 **                     VWTable           *vwptr,
 **                     VWTableCell       *vwcell,
 **                     int                x,
 **                     int                y )
 **
 ** Displays a bitmap in the cell.  This function handles all those funky
 ** display pieces such as anchoring and such.
 ************************************************************************/

#ifdef __STDC__
static void
Display_BitmapCell( StateInfo*                state,
                    VWTable*                  vwptr,
                    VWTableCell*              vwcell,
                    int                       x,
                    int                       y )
#else /** __STDC__ **/
static void
Display_BitmapCell( state, vwptr, vwcell, x, y )
    StateInfo*                state ;
    VWTable*                  vwptr ;
    VWTableCell*              vwcell ;
    int                       x ;
    int                       y ;
#endif /** __STDC__ **/
{
  if (( vwcell == NULL ) || ( vwcell->data.bitmap.data == VWT_NullPixmap ))
    return ;

  {
    VWTableBitmapCell* bitmap = &vwcell->data.bitmap ;
    VWTableColorPair*  vwcolor ;
    XGCValues          localGC ;
    unsigned long      localMask = 0 ;
    unsigned int       width ;
    unsigned int       height ;

    vwcolor = (VWTableColorPair *) VWStack_Peek(&vwcell->colors) ;

    /**
     ** Adjust the foreground pixel values
     **/

    localGC.foreground = (( vwcolor ) ? ( vwcolor->fg ) : ( state->default_fg )) ;
    if (state->fg != localGC.foreground)
      {
	localMask |= GCForeground ;
	state->fg = localGC.foreground ;
      }

    /**
     ** Adjust the background pixel values
     **/

    localGC.background = (( vwcolor ) ? ( vwcolor->bg ) : ( state->default_bg )) ;
    if (state->bg != localGC.background)
      {
	localMask |= GCBackground ;
	state->bg = localGC.background ;
      }
    
    /**
     ** Change the graphic context if required
     **/

    if (localMask)
      XChangeGC(vwptr->display, vwptr->gc, localMask, &localGC) ;

    width = Min(state->width, bitmap->width) ;
    height = Min(state->height, bitmap->height) ;

    switch(bitmap->anchor)
      {
      case TK_ANCHOR_NW:
      case TK_ANCHOR_W:
      case TK_ANCHOR_SW:
	break ;

      case TK_ANCHOR_N:
      case TK_ANCHOR_CENTER:
      case TK_ANCHOR_S:
	x += (state->width - width) / 2 ;
	break ;

      default:
	x += state->width - width ;
	break ;
      }

    /** Now handle the vertical spacing **/

    switch(bitmap->anchor)
      {
      case TK_ANCHOR_N:
      case TK_ANCHOR_NE:
      case TK_ANCHOR_NW:
	break ;

      case TK_ANCHOR_CENTER:
      case TK_ANCHOR_E:
      case TK_ANCHOR_W:
	y += (state->height - height) / 2 ;
	break ;

      default:
	y += state->height - height ;
	break ;
      }

    XCopyPlane( Tk_Display(vwptr->tkWin),
	        bitmap->data,
	        vwptr->image.pixmap,
	        vwptr->gc,
	        0, 0,
	        width, height,
	        x, y,
	        1 ) ;
  }
}

/************************************************************************
 ** static void
 ** Display_TextCell( StateInfo         *state,
 **                   VWTable           *vwptr,
 **                   VWTableCell       *vwcell,
 **                   int                x,
 **                   int                y )
 **
 ** This function takes the data for the cell in question and assumes it
 ** is a text cell.  It will then handle items such as font, foreground
 ** etc. as well as display on the pixmap
 ************************************************************************/

#ifdef __STDC__
static void
Display_TextCell( StateInfo*                state,
                  VWTable*                  vwptr,
                  VWTableCell*              vwcell,
                  int                       x,
                  int                       y )
#else /** __STDC__ **/
static void
Display_TextCell( state, vwptr, vwcell, x, y )
    StateInfo*                state ;
    VWTable*                  vwptr ;
    VWTableCell*              vwcell ;
    int                       x ;
    int                       y ;
#endif /** __STDC__ **/
{
  if (( vwcell == NULL ) || ( vwcell->data.text.data == NULL ))
    return ;

  {
    int                end ;
    int                chars ;
    int                localHeight ;
    unsigned long      localMask = 0 ;
    XGCValues          localGC ;
    XFontStruct*       localFont ;
    VWTableFont*       vwfont ;
    VWTableColorPair*  vwcolor ;
    VWTableTextCell*   text = &vwcell->data.text ;

    char*              ldata ;
    int                ldatalen ;

    int                set_x ;
    int                set_y ;

    /**
     ** Determine the appropriate font for this cell.
     ** If no font is present use the font of the parent
     **/

    vwfont = (VWTableFont *) VWStack_Peek(&vwcell->fonts) ;
    
    if (vwfont == NULL)
      {
	localFont = vwptr->fontPtr ;
	localHeight = vwptr->fontHeight ;
      }
    else
      {
	localFont = vwfont->font ;
	localHeight = vwfont->height ;
      }

    /**
     ** If the text can not fit in the window, then
     ** do not display it
     **/

    if ((localFont->ascent + localFont->descent) > state->height)
      return ;

    if (state->font != localFont)
      {
	state->font = localFont ;
	localGC.font = localFont->fid ;
	localMask |= GCFont ;
      }

    vwcolor = (VWTableColorPair *) VWStack_Peek(&vwcell->colors) ;

    /**
     ** Adjust the foreground pixel values
     **/

    localGC.foreground = (( vwcolor ) ? ( vwcolor->fg ) : ( state->default_fg )) ;
    if (state->fg != localGC.foreground)
      {
	localMask |= GCForeground ;
	state->fg = localGC.foreground ;
      }

    /**
     ** Adjust the background pixel values
     **/

    localGC.background = (( vwcolor ) ? ( vwcolor->bg ) : ( state->default_bg )) ;
    if (state->bg != localGC.background)
      {
	localMask |= GCBackground ;
	state->bg = localGC.background ;
      }
    
    if (localMask)
      XChangeGC(vwptr->display, vwptr->gc, localMask, &localGC) ;

    /**
     ** Use Tk to complete the rest of the display sequence.
     **/
    
    if (text->data[0] == '{')
      {
	ldata = text->data + 1 ;
	ldatalen = text->datalen - 2 ;
      }
    else
      {
	ldata = text->data ;
	ldatalen = text->datalen ;
      }

    /** Handle the anchoring **/

    /** First handle the horizontal spacing **/

    chars = TkMeasureChars(localFont,
			   ldata,
			   ldatalen,
			   vwptr->borderWidth,
			   state->width,
			   0,
			   &end);

    set_x = x ;
    set_y = y ;

    switch(text->anchor)
      {
      case TK_ANCHOR_NW:
      case TK_ANCHOR_W:
      case TK_ANCHOR_SW:
	break ;

      case TK_ANCHOR_N:
      case TK_ANCHOR_CENTER:
      case TK_ANCHOR_S:
	x += (state->width - end) / 2 ;
	break ;

      default:
	x += state->width - end ;
	break ;
      }

    /** Now handle the vertical spacing **/

    switch(text->anchor)
      {
      case TK_ANCHOR_N:
      case TK_ANCHOR_NE:
      case TK_ANCHOR_NW:
	y += localFont->ascent ;
	break ;

      case TK_ANCHOR_CENTER:
      case TK_ANCHOR_E:
      case TK_ANCHOR_W:
	y += (state->height + localFont->ascent - localFont->descent) / 2 ;
	break ;

      default:
	y += state->height - localFont->descent ;
	break ;
      }

    if (x < set_x)  x = set_x ;
    if (y < set_y)  y = set_y + localFont->ascent;

    TkDisplayChars(vwptr->display,
		   vwptr->image.pixmap,
		   vwptr->gc,
		   localFont,
		   ldata,
		   chars,
		   x,
		   y,
		   0L) ;
  }
}

/************************************************************************
 ** static void
 ** UpdateAndLocalize( VWTable          *vwptr,
 **                    unsigned int     *x1,
 **                    unsigned int     *y1,
 **                    unsigned int     *x2,
 **                    unsigned int     *y2 )
 **
 ** Copies the image and looks at before and after images to determine
 ** what needs to be displayed and what does not.  All local pixmap
 ** copies are handled by this function.  Rows/Columns which need update
 ** are marked by bit.
 ************************************************************************/

#ifdef __STDC__
static void
UpdateAndLocalize( VWTable*                  vwptr,
                   unsigned int*             x1,
                   unsigned int*             y1,
                   unsigned int*             x2,
                   unsigned int*             y2 )
#else /** __STDC__ **/
static void
UpdateAndLocalize( vwptr, x1, y1, x2, y2 )
    VWTable*                  vwptr ;
    unsigned int*             x1 ;
    unsigned int*             y1 ;
    unsigned int*             x2 ;
    unsigned int*             y2 ;
#endif /** __STDC__ **/
{
  VWTableImage old_image ;

  old_image.elements[0] = NULL ;
  old_image.elements[1] = NULL ;

  if (! Tk_IsMapped(vwptr->tkWin))
    {
      VWT_CalibrateImage(vwptr, &vwptr->image) ;
      return ;
    }

  VWT_CopyImage(&old_image, &vwptr->image) ;

#ifdef STRICT_DEBUG
#if 0
  VWT_DumpImageElement(old_image.elements[VWT_Horizontal]) ;
  VWT_DumpImageElement(vwptr->image.elements[VWT_Horizontal]) ;
#endif
#endif

  VWT_CalibrateImage(vwptr, &vwptr->image) ;

  /**
   ** Compare the before image with the after image.  There are some
   ** cases to be careful of.
   **
   ** If the size of the row/column has changed then a bit is set
   ** stating that the entire row/column must be redrawn.
   **
   ** If the pixel has moved left or right find out how much space
   ** we can copy, its possible we can also copy a neighbor if nothing
   ** has changed.  Once that's determined copy the whole thing. I
   ** figure this should give us the greatest performance boost when
   ** dealing with scrolling
   **/

  /**
   ** When comparing all comparisons are done from a reference frame
   ** of the new image
   **/

  {
    VWTableImageElement*   object[4] ;
    int                    dimension ;
    int                    height = Tk_Height(vwptr->tkWin) ;

#ifdef STRICT_DEBUG
    fprintf(stderr, "\nStart examination\n") ;
#endif

    for ( object[0]  = vwptr->image.elements[VWT_Horizontal] ;
	  object[0] != NULL ;
	  object[0]  = object[0]->next )
      {
	object[1] = VWT_FindElementByPosition(old_image.elements[VWT_Horizontal], object[0]->position) ;
	if (object[1] == NULL)
	  {
	    /** This element was not found in the old image **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "Can't find: %d\n", object[0]->position);
#endif

	    VWBit_Set(&vwptr->flux[VWT_Horizontal], object[0]->position) ;
	    continue ;
	  }

	/**
	 ** This element was found in the old image, but has it
	 ** changed much.  And if so in what way?
	 **/

	if (( object[0]->pixel == object[1]->pixel ) &&
	    ( object[0]->size == object[1]->size ))
	  {
	    /** No change whatsoever **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "No change occurred: %d\n", object[0]->position) ;
#endif
	    continue ;
	  }

	if ( object[0]->size != object[1]->size )
	  {
	    /** Size changed.  This elements needs redisplay **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "Bit set: %d\n", object[0]->position) ;
#endif

	    VWBit_Set(&vwptr->flux[VWT_Horizontal], object[0]->position) ;
	    continue ;
	  }

	/**
	 ** Pixel is the only thing different.  Just copy the item.
	 ** The only trick is to ensure that we copy as many elements
	 ** as possible in one snatch.
	 **/

#ifdef STRICT_DEBUG
	fprintf(stderr, "Start search from: %d\n", object[0]->position) ;
#endif
	
	object[2] = object[0] ;
	object[3] = object[1] ;
	dimension = object[0]->size ;

	while (( object[2]->next ) && 
	       ( object[3]->next ) &&
	       ( object[2]->next->position == object[3]->next->position ) &&
	       ( object[2]->next->size == object[3]->next->size ))
	  {
	    dimension += object[2]->next->size ;
	    object[2] = object[2]->next ;
	    object[3] = object[3]->next ;
	  }

#ifdef STRICT_DEBUG
	fprintf(stderr, "End search at: %d\n", object[2]->position) ;
#endif

	/**
	 ** Copy begins at object[0] and ends at object[2] with
	 ** both ends inclusive so that if object[0] == object[2]
	 ** then we still copy object[0]
	 **/

	XCopyArea( vwptr->display,
		   vwptr->image.pixmap,
		   vwptr->image.pixmap,
		   vwptr->gc,
		   object[1]->pixel, 0,
		   dimension, height,
		   object[0]->pixel, 0 );

	if ( object[0]->pixel < x1[0] )                x1[0] = object[0]->pixel ;
	if (( object[0]->pixel + dimension ) > x2[0])  x2[0] = object[0]->pixel + dimension ;

	y1[0] = 0 ;
	y2[0] = height ;

	object[0] = object[2] ;
      }
  }

  /** Vertical axis **/

  {
    VWTableImageElement*   object[4] ;
    int                    dimension ;
    int                    width = Tk_Width(vwptr->tkWin) ;

    for ( object[0]  = vwptr->image.elements[VWT_Vertical] ;
	  object[0] != NULL ;
	  object[0]  = object[0]->next )
      {
	object[1] = VWT_FindElementByPosition(old_image.elements[VWT_Vertical], object[0]->position) ;
	if (object[1] == NULL)
	  {
	    /** This element was not found in the old image **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "Can't find: %d\n", object[0]->position);
#endif

	    VWBit_Set(&vwptr->flux[VWT_Vertical], object[0]->position) ;
	    continue ;
	  }

	/**
	 ** This element was found in the old image, but has it
	 ** changed much.  And if so in what way?
	 **/

	if (( object[0]->pixel == object[1]->pixel ) &&
	    ( object[0]->size == object[1]->size ))
	  {
	    /** No change whatsoever **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "No change occurred: %d\n", object[0]->position) ;
#endif
	    continue ;
	  }

	if ( object[0]->size != object[1]->size )
	  {
	    /** Size changed.  This elements needs redisplay **/

#ifdef STRICT_DEBUG
	    fprintf(stderr, "Bit set: %d\n", object[0]->position) ;
#endif

	    VWBit_Set(&vwptr->flux[VWT_Vertical], object[0]->position) ;
	    continue ;
	  }

	/**
	 ** Pixel is the only thing different.  Just copy the item.
	 ** The only trick is to ensure that we copy as many elements
	 ** as possible in one snatch.
	 **/

#ifdef STRICT_DEBUG
	fprintf(stderr, "Start search from: %d\n", object[0]->position) ;
#endif
	
	object[2] = object[0] ;
	object[3] = object[1] ;
	dimension = object[0]->size ;

	while (( object[2]->next ) && 
	       ( object[3]->next ) &&
	       ( object[2]->next->position == object[3]->next->position ) &&
	       ( object[2]->next->size == object[3]->next->size ))
	  {
	    dimension += object[2]->next->size ;
	    object[2] = object[2]->next ;
	    object[3] = object[3]->next ;
	  }

#ifdef STRICT_DEBUG
	fprintf(stderr, "End search at: %d\n", object[2]->position) ;
#endif

	/**
	 ** Copy begins at object[0] and ends at object[2] with
	 ** both ends inclusive so that if object[0] == object[2]
	 ** then we still copy object[0]
	 **/

	XCopyArea( vwptr->display,
		   vwptr->image.pixmap,
		   vwptr->image.pixmap,
		   vwptr->gc,
		   0, object[1]->pixel,
		   width, dimension,
		   0, object[0]->pixel );

	if ( object[0]->pixel < y1[0] )                y1[0] = object[0]->pixel ;
	if (( object[0]->pixel + dimension ) > y2[0])  y2[0] = object[0]->pixel + dimension ;

	x1[0] = 0 ;
	x2[0] = width ;

	object[0] = object[2] ;
      }
  }

  VWT_FlushImage(&old_image) ;
}

/************************************************************************
 ** void
 ** ClearCell( VWTable          *vwptr,
 **            VWTableCell      *vwcell,
 **            unsigned int      x,
 **            unsigned int      y,
 **            unsigned int      width,
 **            unsigned int      height,
 **            unsigned int     *x1,
 **            unsigned int     *y1,
 **            unsigned int     *x2,
 **            unsigned int     *y2 )
 **
 ** Clears the contents of the cell
 ************************************************************************/

#ifdef __STDC__
void
ClearCell( VWTable*                                vwptr,
           VWTableCell*                            vwcell,
           int                                     xposition,
           int                                     yposition,
           unsigned int                            width,
           unsigned int                            height,
           unsigned int*                           x1,
           unsigned int*                           y1,
           unsigned int*                           x2,
           unsigned int*                           y2 )
#else /** __STDC__ **/
void
ClearCell( vwptr, vwcell, xposition, yposition, width, height, x1, y1, x2, y2 )
    VWTable*                                vwptr ;
    VWTableCell*                            vwcell ;
    int                                     xposition ;
    int                                     yposition ;
    unsigned int                            width ;
    unsigned int                            height ;
    unsigned int*                           x1 ;
    unsigned int*                           y1 ;
    unsigned int*                           x2 ;
    unsigned int*                           y2 ;
#endif /** __STDC__ **/
{
  VWTableColorPair *vwcolor ;
  Tk_3DBorder       tkColor ;
  
  if ( xposition < x1[0] ) x1[0] = xposition ;
  if ( yposition + height > y2[0] ) y2[0] = yposition + height ;
  if (( yposition ) < y1[0]) y1[0] = yposition ;
  if (( xposition + width ) > x2[0])  x2[0] = xposition + width ;

  vwcolor = (( vwcell != NULL ) ? 
	     ((VWTableColorPair *) VWStack_Peek(&vwcell->colors)) :
	     ( NULL )) ;

  tkColor = (( vwcolor == NULL ) ?
	     ( vwptr->bgBorder ) :
	     ( vwcolor->background )) ;

  Tk_Fill3DRectangle(vwptr->display, vwptr->image.pixmap, tkColor,
		     xposition, yposition,
		     width, height,
		     vwptr->cellBorderWidth,
		     vwptr->cellRelief);
}

/************************************************************************
 ** void VWT_ClearCell( VWTable      *vwptr,
 **                     int           x,
 **                     int           y )
 **
 ** External method to clear one cell
 ************************************************************************/

#ifdef __STDC__
void
VWT_ClearCell( VWTable*          vwptr,
               int               x,
               int               y )
#else /** __STDC__ **/
void
VWT_ClearCell( vwptr, x, y )
    VWTable*          vwptr ;
    int               x ;
    int               y ;
#endif /** __STDC__ **/
{
  VWTableImageElement* object_x ;
  VWTableImageElement* object_y ;
  VWTableCell**        vwcell ;
  int x1, y1 ;
  int x2, y2 ;

  x1 = Tk_Width(vwptr->tkWin)  ; x2 = 0 ;
  y1 = Tk_Height(vwptr->tkWin) ; y2 = 0 ;

  object_x = VWT_FindElementByPosition(vwptr->image.elements[VWT_Horizontal], x) ;
  object_y = VWT_FindElementByPosition(vwptr->image.elements[VWT_Vertical], y) ;

  if ((object_x == NULL) || (object_y == NULL))
    {
      /**
       ** This image is not currently drawn
       **/

      return ;
    }

  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, x, y, 0) ;

  ClearCell(vwptr, vwcell[0],
	    object_x->pixel, object_y->pixel, object_x->size, object_y->size,
	    &x1, &y1, &x2, &y2) ;

  /**
   ** Copy the pixmap to the forward window
   **/

  XCopyArea( vwptr->display,
	     vwptr->image.pixmap,
	     Tk_WindowId(vwptr->tkWin),
	     vwptr->gc,
	     x1, y1,
	     x2 - x1 + 1, y2 - y1 + 1,
	     x1, y1 );
}

/************************************************************************
 ** void VWT_Display( ClientData clientData )
 **
 ** Redisplay the table
 ************************************************************************/

#ifdef __STDC__
void
VWT_Display( ClientData    clientData )
#else /** __STDC__ **/
void
VWT_Display( clientData )
    ClientData    clientData ;
#endif /** __STDC__ **/
{
  register VWTable      *vwptr ;
  register VWTableCell **vwcell ;
  register VWTableCell **focus ;

  VWTableImage          *image ;
  VWTableImageElement   *object_x ;
  VWTableImageElement   *object_y ;

  unsigned int  x1, x2 ;
  unsigned int  y1, y2 ;

  int       width ;
  int       height ;

  int       cellSpace ;
  int       borderSpace ;
  StateInfo state ;

  int       x_flux = FALSE ;
  int       y_flux = FALSE ;

  int       draw_bits ;

  static void (*DataHandler[]) _ANSI_ARGS_((StateInfo*, VWTable*, VWTableCell*, int, int)) =
    {
      NULL,
      Display_TextCell,
      Display_BitmapCell
    };

  vwptr = (VWTable *) clientData ;
  image = &vwptr->image ;

  width = Tk_Width(vwptr->tkWin) ;
  height = Tk_Height(vwptr->tkWin) ;

  x1 = width  ; x2 = 0 ;
  y1 = height ; y2 = 0 ;

  borderSpace = 2 * vwptr->borderWidth ;
  cellSpace = 2 * vwptr->cellBorderWidth ;

  if (! Tk_IsMapped(vwptr->tkWin))
    return ;

  vwptr->updatePending = 0 ;

  /**
   ** Mark a cell for focus (if used), a bit of research showed
   ** that I'd end up doing just as much branch evaluation out of
   ** the main loop as I would inside of the main loop (in most
   ** cases).  Since the performance penalty is minimal we'll
   ** just do the evaluation inline
   **/

  focus = ((vwptr->focus.focusing) ?
	   (VWMatrixIndex(vwptr->matrix, VWTableCell*, vwptr->focus.col, vwptr->focus.row, 0)) :
	   (NULL)) ;

#ifdef STRICT_DEBUG
  fprintf(stderr, "Focus: %x\n", focus) ;
#endif /** STRICT_DEBUG **/

  /**
   ** Since we'd like a default background for the window we draw the
   ** "window" background allowing it to do its relief borders.
   **
   ** Regenerates all background visuals.  This will depend on data in
   ** the table.  If redrawBackground is TRUE then the background will
   ** be entirely redrawn, otherwise if redrawOutline is TRUE only the
   ** outline of the background will be redrawn
   **/

  if (vwptr->redrawBackground)
    {
      Tk_Fill3DRectangle( vwptr->display,
			  vwptr->image.pixmap,
			  vwptr->bgBorder,
			  0, 0,
			  width, height,
			  vwptr->borderWidth, vwptr->relief );

      x1 = 0 ; x2 = width ;
      y1 = 0 ; y2 = height ;
    }
  else if ((vwptr->redrawOutline) || (vwptr->scrollOccurred))
    {
      Tk_Draw3DRectangle( vwptr->display,
			  vwptr->image.pixmap,
			  vwptr->bgBorder,
			  0, 0,
			  width, height,
			  vwptr->borderWidth, vwptr->relief );
    }

  /**
   ** Setup any secondary variables and changes to the graphics
   ** context before doing a cell by cell redraw
   **/

  state.font = NULL ;
  state.default_fg = state.fg = Tk_3DBorderColor(vwptr->fgBorder)->pixel ;
  state.default_bg = state.bg = Tk_3DBorderColor(vwptr->bgBorder)->pixel ;

  XSetForeground(vwptr->display, vwptr->gc, state.fg);
  XSetBackground(vwptr->display, vwptr->gc, state.bg);

  width  -= vwptr->borderWidth ;
  height -= vwptr->borderWidth ;

  /**
   ** Update the image.  If changes have occurred which are localized
   ** handle them
   **/

  UpdateAndLocalize(vwptr, &x1, &y1, &x2, &y2) ;

  /**
   ** Draw the image
   **/

  draw_bits = ( vwptr->redrawBackground ) ;

  for ( object_y = image->elements[VWT_Vertical] ; object_y != NULL ; object_y = object_y->next )
    {
      state.height = object_y->size - cellSpace ;

      /**
       ** Check the flux bit for this row.  If the flux bit is marked then
       ** this entire row must be redrawn.  To avoid checking this bit in
       ** every branch evaluation (noted below), we set the 2 bit in the
       ** draw_bits
       **/

      if (VWBit_IsSet(&vwptr->flux[VWT_Vertical], object_y->position))
	draw_bits |= 2, y_flux = TRUE ;
      else
	draw_bits &= ~2 ;

      /**
       ** Cycle over the objects which represent columns
       **/

      for ( object_x = image->elements[VWT_Horizontal] ; object_x != NULL ; object_x = object_x->next )
	{
	  state.width = object_x->size - cellSpace ;
	  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, object_x->position, object_y->position, 0) ;

	  /**
	   ** This branch evaluation is messy, but explainable and clear.
	   ** The branch first checks if this cell has been changed.  To do this
	   **    it first sees that there is a valid (meaning allocated) cell
	   **    here.  If the cell is allocated it checks the changed bit.
	   ** If the change is FALSE it will then check to see if the column
	   **    needs to be updated.  This is done by checking the horizontal
	   **    flux bit.
	   ** If no flux, it checks the already evaluated conditions such as
	   **    row flux, window redraw, etc.
	   **/

	  if (VWBit_IsSet(&vwptr->flux[VWT_Horizontal], object_x->position))
	    x_flux = TRUE ;

#ifdef STRICT_DEBUG
	  if (vwcell == focus)
	    fprintf(stderr, "Found\n") ;
#endif /** STRICT_DEBUG **/

	  if ((( vwcell[0] != NULL ) && ( vwcell[0]->changed )) ||
	      (vwcell == focus) ||
	      (VWBit_IsSet(&vwptr->flux[VWT_Horizontal], object_x->position)) ||
	      (draw_bits))
	    {
	      /** Turn off the changed flag **/

	      if (vwcell[0]) vwcell[0]->changed = FALSE ;

	      /**
	       ** ClearCell() is used to erase the contents of what is on the pixmap.
	       ** It will also set the appropriate x1, y1, x2, y2 coordinates once
	       ** it returns.
	       **/

	      ClearCell( vwptr, vwcell[0],
			 object_x->pixel, object_y->pixel, object_x->size, object_y->size,
			 &x1, &y1, &x2, &y2 ) ;

	      /**
 	       ** If the cell is THE focus cell then call the blink handler not a normal
	       ** data handler, else ...
	       **
	       ** If a DataHandler() and allocated cell exist, the DataHandler will then
	       ** display its current state
	       **/

	      if (vwcell == focus)
		DisplayBlink(&state, vwptr, vwcell[0],
			     object_x->pixel + vwptr->cellBorderWidth,
			     object_y->pixel - vwptr->cellBorderWidth) ;
	      else if (( vwcell[0] != NULL ) && ( DataHandler[vwcell[0]->dataType] != NULL ))
		DataHandler[vwcell[0]->dataType](&state, vwptr, vwcell[0],
						 object_x->pixel + vwptr->cellBorderWidth,
						 object_y->pixel + vwptr->cellBorderWidth) ;
	    }
	}
    }

  /** Zero the flux bits **/

  VWBitset_Zero(&vwptr->flux[VWT_Vertical]) ;
  VWBitset_Zero(&vwptr->flux[VWT_Horizontal]) ;

  /**
   ** The "picture" often has excess (or open) space which has not been accounted for.  If it
   ** is not dealt with, the "picture" isn't very happy.  There are two sides of this excess
   ** which must be dealt with.
   **/

  {
    int max_x = 0 ;
    int max_y = 0 ;

    for ( object_x = image->elements[VWT_Horizontal] ;
	  object_x->next != NULL ;
	  object_x = object_x->next )
      ;

    max_x = (object_x ? object_x->pixel + object_x->size : 0) ;

    if (( max_x < width ) && ( vwptr->scrollOccurred || vwptr->redrawBackground || x_flux ))
      {
	Tk_Fill3DRectangle( vwptr->display,
			    vwptr->image.pixmap,
			    vwptr->bgBorder,
			    max_x, vwptr->borderWidth,
			    width - max_x - vwptr->borderWidth,
			    height - 2 * vwptr->borderWidth,
			    0, TK_RELIEF_FLAT ) ;

	if (x1 > max_x)                               x1 = max_x ;
	if (x2 < max_x + width - vwptr->borderWidth)  x2 = max_x + width - vwptr->borderWidth ;
	if (y1 > vwptr->borderWidth)                  y1 = vwptr->borderWidth ;
	if (y2 < height - 2 * vwptr->borderWidth)     y2 = height - vwptr->borderWidth ;
      }

    for ( object_y = image->elements[VWT_Vertical] ;
	  object_y->next != NULL ;
	  object_y = object_y->next )
      ;

    max_y = (object_y ? object_y->pixel + object_y->size : 0) ;

    if (( max_y < height ) && ( vwptr->scrollOccurred || vwptr->redrawBackground || y_flux ))
      {
	Tk_Fill3DRectangle( vwptr->display,
			    vwptr->image.pixmap,
			    vwptr->bgBorder,
			    vwptr->borderWidth, max_y,
			    width - 2 * vwptr->borderWidth,
			    height - max_y - vwptr->borderWidth,
			    0, TK_RELIEF_FLAT ) ;

	if (y1 > max_y)                                y1 = max_y ;
	if (y2 < max_y + height - vwptr->borderWidth)  y2 = max_y + height - vwptr->borderWidth ;
	if (x1 > vwptr->borderWidth)                   x1 = vwptr->borderWidth ;
	if (x2 < width - 2 * vwptr->borderWidth)       x2 = width - vwptr->borderWidth ;
      }
  }
  
  /**
   ** Copy the pixmap to the forward window
   **/

  XCopyArea( vwptr->display,
	     vwptr->image.pixmap,
	     Tk_WindowId(vwptr->tkWin),
	     vwptr->gc,
	     x1, y1,
	     x2 - x1 + 1, y2 - y1 + 1,
	     x1, y1 );

  vwptr->redrawOutline = FALSE ;
  vwptr->redrawBackground = FALSE ;
  vwptr->scrollOccurred = FALSE ;
}
