/************************************************************************
 ** Copyright (c) 1994, Aaron Jackson
 ** All rights reserved
 **
 ** Permission to use, copy, modify, and distribute this software and
 ** its documentation for any purpose and without fee is hereby granted,
 ** provided that the above copyright notice appear in all copies and
 ** that both that copyright notice and this permission notice appear in
 ** supporting documentation.
 **
 ** In no event shall the author(s) be liable to any party for direct,
 ** indirect, special, incidental, or consequential damages arising out
 ** of the use of this software and its documentation, even if the
 ** authors(s) have been advised of the possibility of such damage.
 **
 ** The authors(s) specifically disclaim any warranties, including but
 ** not limited to, the implied warranties of merchantability and
 ** fitness for a particular purpose.  The software provided hereunder
 ** is on an "as is" basis, and the author(s) have no obligation to
 ** provide maintenance, support, updates, enhancements, or modifications
 ************************************************************************/

static char rcsid[] = "$Header: /home/edsdev/cvstree/tcldev/vwtable/vwTableImage.c,v 1.4 1994/11/11 22:35:52 aajackso Exp $" ;

/**
 ** $Log: vwTableImage.c,v $
 * Revision 1.4  1994/11/11  22:35:52  aajackso
 * *** empty log message ***
 *
 * Revision 1.3  1994/11/11  21:37:01  aajackso
 * *** empty log message ***
 *
 * Revision 1.2  1994/10/25  21:10:06  aajackso
 * Oops, added __STDC__ code for most source code and changed the Makefile
 * so that it was consistent w/ the new files
 *
 **/

#include <stdio.h>
#include <stdlib.h>

#include <vwTable.h>
#include <vwTableImageElem.h>

/**
 ** What is this module?
 **
 ** One of the primary reasons for the break between v0.33 and v0.4
 ** was the addition of "post" and "hide" functionality.  They made
 ** it physically possible to place rows and/or columns in orders which
 ** were no longer sequential as they related to thier logical row/
 ** column ordering.  Rather their sequence became a function of their
 ** pixel position on screen.
 **
 ** With the "post" functionality a row/column has the ability to
 ** determine at what pixel location they would like to be drawn, hence
 ** altering the shape of other rows around it.
 **
 ** The "hide" functionality makes it possible for a row/column to not
 ** be drawn even though they are physically a proper row/column.
 **
 ** The result of all of this redesign has led to a new engine regarding
 ** the display of information.  The primary concern here is to mimic
 ** the true state of affairs, which is to say, that we now sequence our
 ** objects according to pixel.  An ImageElem or image element is a
 ** structure used to map an object (row or column) to a position.
 **/

#ifdef Max
#undef Max
#endif /** Max **/

#define Max(xx,yy) (((xx) > (yy)) ? (xx) : (yy))

#ifdef Min
#undef Min
#endif /** Min **/

#define Min(xx,yy) (((xx) > (yy)) ? (yy) : (xx))

/**
 ** Management of node memory is a prime concern.  Since the nodes will
 ** often be "freed" and "allocated" it would be smartest to keep a pool
 ** of memory.
 **/

static VWTableImageElement*     pool = NULL ;
static int                      pool_depth = 0 ;
static int                      pool_max_depth = 200 ;

/************************************************************************
 ** VWTableImageElement*
 ** VWT_AllocElement()
 **
 ** This function can be used to allocate elements outside of this
 ** module.  If you use this, please use VWT_FreeElement() to ensure
 ** that correct means are taken for freeing of allocated memory
 ************************************************************************/

#ifdef __STDC__
VWTableImageElement*
VWT_AllocElement( )
#else /** __STDC__ **/
VWTableImageElement*
VWT_AllocElement(  )
#endif /** __STDC__ **/
{
  VWTableImageElement* object ;

  if (pool != NULL)
    {
      object = pool ;
      pool = pool->next ;
      pool_depth-- ;
    }
  else
    {
      object = (VWTableImageElement *) malloc(sizeof(VWTableImageElement)) ;
    }

  object->size = 0 ;
  object->pixel = 0 ;
  object->position = 0 ;
  object->posted = 0 ;
  object->next = NULL ;

  return object ;
}

/************************************************************************
 ** void
 ** VWT_FreeElement( VWTableImageElement* object )
 **
 ** This function is used to clean memory associated with calls to
 ** VWT_AllocElement().  These functions are provided so that external
 ** modules have access to element allocation.
 ************************************************************************/

#ifdef __STDC__
void
VWT_FreeElement( VWTableImageElement*    object )
#else /** __STDC__ **/
void
VWT_FreeElement( object )
    VWTableImageElement*    object ;
#endif /** __STDC__ **/
{
  if (pool_depth < pool_max_depth)
    {
      object->next = pool ;
      pool = object ;
      pool_depth++ ;
    }
  else
    {
      free((char *) object) ;
    }
}

/************************************************************************
 ** void
 ** VWT_FreeElementList( VWTableImage* elementList )
 **
 ** Destroys and element list
 ************************************************************************/

#ifdef __STDC__
void
VWT_FreeElementList( VWTableImageElement*    elementList )
#else /** __STDC__ **/
void
VWT_FreeElementList( elementList )
    VWTableImageElement*    elementList ;
#endif /** __STDC__ **/
{
  register VWTableImageElement *elem ;
  register VWTableImageElement *next ;

  for ( elem = elementList ; elem != NULL ; elem = next )
    {
      next = elem->next ;
      VWT_FreeElement(elem) ;
    }
}

/************************************************************************
 ** static void
 ** RemoveUnposted( VWTableImageElement**   element )
 **
 ** Removes all unposted images.  Use of VWT_FreeElement() rather than
 ** free() should be used to minimize calls to malloc()
 ************************************************************************/

#ifdef __STDC__
static void
RemoveUnposted( VWTableImageElement**    element )
#else /** __STDC__ **/
static void
RemoveUnposted( element )
    VWTableImageElement**    element ;
#endif /** __STDC__ **/
{
  register VWTableImageElement* object ;
  register VWTableImageElement* object_next ;
  register VWTableImageElement* object_parent = NULL ;

  for ( object  = element[0] ; object != NULL ; object = object_next )
    {
      object_next = object->next ;
      if ( object->posted )
	{
	  object_parent = object ;
	  continue ;
	}

      if ( object_parent == NULL )
	element[0] = object_next ;
      else
	object_parent->next = object_next ;

      VWT_FreeElement(object) ;
    }
}

/************************************************************************
 ** static void
 ** Reset_ObjectSize( VWTableImageElement*  parent,
 **                   VWTableImageElement*  object,
 **                   int                   max_pxl )
 **
 ** Resets the operative size of an object w/ respect to its parent and
 ** children if any exist
 ************************************************************************/

#ifdef __STDC__
static void
Reset_ObjectSize( VWTableImageElement*          parent,
                  VWTableImageElement*          object,
                  int                           max_pxl )
#else /** __STDC__ **/
static void
Reset_ObjectSize( parent, object, max_pxl )
    VWTableImageElement*          parent ;
    VWTableImageElement*          object ;
    int                           max_pxl ;
#endif /** __STDC__ **/
{
  int dimension ;

  /**
   ** Conflict resolution may still have to occur:
   ** Types of conflicts handled:
   ** 
   ** [1] Parent (i.e. element to left) has a size which falls into
   **     this object's space.
   **
   **     >> Reset parent size so that it now reflects available
   **        size
   **
   ** [2] Element to right is posted and thus shortens our possible size
   ** [3] No element to right, but end of window
   **
   **     >> Set the appropriate rsize so that it may be dealt with
   **        properly in future situations and adjust the size to fit
   **        the window
   **
   ** [4] Element to right is not posted but starts before this element
   **     ends.
   **
   **     >> Call the element shift operator
   **/

  object->size = object->rsize ;

  /**
   ** Situation [1]
   **/

  if (parent)
    {
      if ((parent->pixel + parent->rsize) > (object->pixel))
	parent->size = object->pixel - parent->pixel ;
    }

  /**
   ** Situation [2][3]
   **/
  
  dimension = ((object->next && object->next->posted) ?
	       (object->next->pixel) :
	       (max_pxl)) ;

  if ((object->pixel + object->rsize) > dimension)
    object->size = dimension - object->pixel ;

  /**
   ** Situation [4]
   **/
}

/************************************************************************
 ** void
 ** VWT_DumpImageElement( VWTableImageElement* list )
 **
 ** Dumps the element components
 ************************************************************************/

#ifdef __STDC__
void
VWT_DumpImageElement( VWTableImageElement*    list )
#else /** __STDC__ **/
void
VWT_DumpImageElement( list )
    VWTableImageElement*    list ;
#endif /** __STDC__ **/
{
  VWTableImageElement*   object ;    

  for ( object  = list ;
        object != NULL ;
        object  = object->next )
    {
      printf("%8x %3u %3d %3d\n",
	     object,
	     object->position,
	     object->pixel,
	     object->size) ;
    }

  printf("\n") ;
}

/************************************************************************
 ** void
 ** VWT_FlushImage( VWTableImage*    image )
 **
 ** Flushes components used in a copy image
 ************************************************************************/

#ifdef __STDC__
void
VWT_FlushImage( VWTableImage*    image )
#else /** __STDC__ **/
void
VWT_FlushImage( image )
    VWTableImage*    image ;
#endif /** __STDC__ **/
{
  VWTableImageElement *object ;
  VWTableImageElement *object_next ;
  int                  i ;

  for ( i = 0 ; i < 2 ; i++ )
    {
      for ( object  = image->elements[i] ;
	    object != NULL ;
	    object = object_next )
	{
	  object_next = object->next ;
	  VWT_FreeElement(object) ;
	}
    }
}

/************************************************************************
 ** void
 ** VWT_CopyImage( VWTableImage*    dest_image,
 **                VWTableImage*    src_image )
 **
 ** Copies an relevant pieces of an image needed for later comparison
 ************************************************************************/

#ifdef __STDC__
void
VWT_CopyImage( VWTableImage*       dest_image,
               VWTableImage*       src_image )
#else /** __STDC__ **/
void
VWT_CopyImage( dest_image, src_image )
    VWTableImage*       dest_image ;
    VWTableImage*       src_image ;
#endif /** __STDC__ **/
{
  VWTableImageElement *object ;
  VWTableImageElement *object_new ;
  VWTableImageElement *object_parent ;
  int                  i ;

  for ( i = 0 ; i < 2 ; i++ )
    {
      object_parent = NULL ;
      for ( object = src_image->elements[i] ; object != NULL ; object = object->next )
	{
	  object_new = VWT_AllocElement() ;
	  object_new->size = object->size ;
	  object_new->pixel = object->pixel ;
	  object_new->position = object->position ;
	  object_new->next = NULL ;

	  if (! object_parent)
	    dest_image->elements[i] = object_new ;
	  else
	    object_parent->next = object_new ;

	  object_parent = object_new ;
	}
    }
}

/************************************************************************
 ** void
 ** VWT_CalibrateDimension( VWTableImage*   image )
 **
 ** Calibrates a single dimension
 ************************************************************************/

#ifdef __STDC__
void
VWT_CalibrateDimension( VWTableImageElement**                         elements,
                        VWBitset*                                     posted,
                        VWBitset*                                     hidden,
                        int                                           min_dim,
                        int                                           max_dim,
                        int*                                          sz_dim,
                        int                                           min_pxl,
                        int                                           max_pxl )
#else /** __STDC__ **/
void
VWT_CalibrateDimension( elements, posted, hidden, min_dim, max_dim, sz_dim, min_pxl, max_pxl )
    VWTableImageElement**                         elements ;
    VWBitset*                                     posted ;
    VWBitset*                                     hidden ;
    int                                           min_dim ;
    int                                           max_dim ;
    int*                                          sz_dim ;
    int                                           min_pxl ;
    int                                           max_pxl ;
#endif /** __STDC__ **/
{
  VWTableImageElement *object ;
  VWTableImageElement *object_new ;
  VWTableImageElement *object_parent ;
  int                  dim ;
  int                  pxl ;

  RemoveUnposted(elements) ;

  /**
   ** Cycle through all elements visible because of normal
   ** viewport activities.  Create elements for them keeping
   ** in mind that posted elements have precedent over
   ** non-posted elements
   **/

  object = elements[0] ;
  object_parent = NULL ;
  pxl = min_pxl ;
  
  for (dim = min_dim ; dim < max_dim ;)
    {
      if ((VWBit_IsSet(posted, dim)) ||
	  (VWBit_IsSet(hidden, dim)))
	{
	  dim++ ;
	  continue ;
	}

      if (pxl >= max_pxl)
	break ;

      if (( object == NULL ) || ( pxl < object->pixel ))
	{
	  int u_width = ((object == NULL) ? (max_pxl - pxl) : (object->pixel - pxl)) ;

	  object_new = VWT_AllocElement() ;
	  object_new->size = Min(sz_dim[dim], u_width) ;
	  object_new->rsize = sz_dim[dim] ;
	  object_new->pixel = pxl ;
	  object_new->position = dim ;
	  object_new->next = object ;

	  if (object_parent)
	    object_parent->next = object_new ;
	  else
	    elements[0] = object_new ;

	  object = object_new ;
	  dim++ ;
	}
      else if (object->size <= 0)
	{
	  Reset_ObjectSize(object_parent, object, max_pxl) ;
	}

      pxl += object->size ;
      object_parent = object ;
      object = object->next ;
    }
}
		        
/************************************************************************
 ** void
 ** VWT_CalibrateImage( VWTable*        vwptr,
 **                     VWTableImage*   image )
 **
 ** When the screen is redisplayed or a scroll occurs unposted images
 ** must be removed from the image and the new elements are inserted
 ** according to position
 ************************************************************************/

#ifdef __STDC__
void
VWT_CalibrateImage( VWTable*         vwptr,
                    VWTableImage*    image )
#else /** __STDC__ **/
void
VWT_CalibrateImage( vwptr, image )
    VWTable*         vwptr ;
    VWTableImage*    image ;
#endif /** __STDC__ **/
{
  VWT_CalibrateDimension( &image->elements[VWT_Horizontal],
			  &image->posts[VWT_Horizontal],
			  &vwptr->hide[VWT_Horizontal],
			  vwptr->vcolumn,
			  vwptr->columns,
			  vwptr->columnWidth,
			  vwptr->borderWidth,
			  Tk_Width(vwptr->tkWin) - vwptr->borderWidth );

  VWT_CalibrateDimension( &image->elements[VWT_Vertical],
			  &image->posts[VWT_Vertical],
			  &vwptr->hide[VWT_Vertical],
			  vwptr->vrow,
			  vwptr->rows,
			  vwptr->rowHeight,
			  vwptr->borderWidth,
			  Tk_Height(vwptr->tkWin) - vwptr->borderWidth );
}

/************************************************************************
 ** void
 ** VWT_InsertElement( VWTableImageElement**  list,
 **                    VWTableImageElement*   object,
 **                    int                    max_dimension )
 **
 ** Inserts an element into a list of elements.  This functions deals
 ** with collision problems which may occur when a posted element is
 ** inserted and causes a collision with an unposted element
 ************************************************************************/

#ifdef __STDC__
void
VWT_InsertElement( VWTableImageElement**          list,
                   VWTableImageElement*           object,
                   int                            max_dimension )
#else /** __STDC__ **/
void
VWT_InsertElement( list, object, max_dimension )
    VWTableImageElement**          list ;
    VWTableImageElement*           object ;
    int                            max_dimension ;
#endif /** __STDC__ **/
{
  register VWTableImageElement*    node = list[0] ;
  register VWTableImageElement*    parent = NULL ;

  for ( ; node != NULL ; node = node->next )
    {
      if ( node->pixel > object->pixel )
	break ;
      
      parent = node ;
    }

  if (! parent)
    {
      object->next = list[0] ;
      list[0] = object ;
    }
  else
    {
      object->next = parent->next ;
      parent->next = object ;
    }

  Reset_ObjectSize(parent, object, max_dimension) ;
}

/************************************************************************
 ** void
 ** VWT_RemoveElement( VWTableImageElement**  list,
 **                    VWTableImageElement*   object )
 **
 ** Removes an element from a list of elements.  If the list becomes
 ** empty or the head of the list needs to be corrected a pointer to
 ** the list is passed so this may be accomplished
 ************************************************************************/

#ifdef __STDC__
void
VWT_RemoveElement( VWTableImageElement**       list,
                   VWTableImageElement*        object )
#else /** __STDC__ **/
void
VWT_RemoveElement( list, object )
    VWTableImageElement**       list ;
    VWTableImageElement*        object ;
#endif /** __STDC__ **/
{
  register VWTableImageElement*    node ;
  register VWTableImageElement*    parent ;

  parent = NULL ;

  for ( node  = list[0] ;
        node != NULL ;
        node  = node->next )
    {
      if ( node == object )
	break ;
      
      parent = node ;
    }

  if (! node)
    return ;
  
  if (! parent)
    list[0] = node->next ;
  else
    parent->next = node->next ;
}

/************************************************************************
 ** VWTableImageElement*
 ** VWT_FindElementAtPixel( VWTableImageElement*   list,
 **                         int                    pixel )
 **
 ** Finds an element by pixel
 ************************************************************************/

#ifdef __STDC__
VWTableImageElement*
VWT_FindElementAtPixel( VWTableImageElement*       list,
                        int                        pixel )
#else /** __STDC__ **/
VWTableImageElement*
VWT_FindElementAtPixel( list, pixel )
    VWTableImageElement*       list ;
    int                        pixel ;
#endif /** __STDC__ **/
{
  register VWTableImageElement*    object ;

  for ( object  = list ;
        object != NULL ;
        object  = object->next )
    {
      if (( object->pixel + object->size ) > pixel )
	break ;
    }
  
  return object ;
}

/************************************************************************
 ** VWTableImageElement*
 ** VWT_FindElementByPixel( VWTableImageElement*   list,
 **                         int                    pixel )
 **
 ** Finds an element by pixel
 ************************************************************************/

#ifdef __STDC__
VWTableImageElement*
VWT_FindElementByPixel( VWTableImageElement*       list,
                        int                        pixel )
#else /** __STDC__ **/
VWTableImageElement*
VWT_FindElementByPixel( list, pixel )
    VWTableImageElement*       list ;
    int                        pixel ;
#endif /** __STDC__ **/
{
  register VWTableImageElement*    object ;

  for ( object  = list ;
        object != NULL ;
        object  = object->next )
    {
      if ( object->pixel == pixel )
	break ;
    }
  
  return object ;
}

/************************************************************************
 ** VWTableImageElement*
 ** VWT_FindElementByPosition( VWTableImageElement*   list,
 **                            int                    position )
 **
 ** Finds an element by position
 ************************************************************************/

#ifdef __STDC__
VWTableImageElement*
VWT_FindElementByPosition( VWTableImageElement*       list,
                           int                        position )
#else /** __STDC__ **/
VWTableImageElement*
VWT_FindElementByPosition( list, position )
    VWTableImageElement*       list ;
    int                        position ;
#endif /** __STDC__ **/
{
  register VWTableImageElement*    object ;

  for ( object  = list ;
        object != NULL ;
        object  = object->next )
    {
      if ( object->position == position )
	break ;
    }
  
  return object ;
}


