#ifdef DCF
/*
 *  $Header: /src/NTP/v3/xntp/xntpd/RCS/refclock_dcf77.c,v 2.4 1992/07/09 17:10:37 kardel XNTP-DCF-1 $
 *  
 *  $Id: refclock_dcf77.c,v 2.4 1992/07/09 17:10:37 kardel XNTP-DCF-1 $
 *
 *  reference clock driver for the Meinberg DCF77 clock
 *  driven by a streams module on top of the tty (SunOS4.x)
 *
 *  Copyright (c) 1989,1990,1991,1992
 *  Frank Kardel Friedrich-Alexander Universitaet Erlangen-Nuernberg
 *
 *  This code can be modified and used freely provided that the
 *  credits remain intact.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * $Log: refclock_dcf77.c,v $
 * Revision 2.4  1992/07/09  17:10:37  kardel
 * added fudge factors for ELV DCF7000
 *
 * Revision 2.3  1992/07/08  11:55:51  kardel
 * log bad format counts
 *
 * Revision 2.2  1992/07/06  14:21:28  kardel
 * format type display
 *
 * Revision 2.1  1992/07/05  21:38:02  kardel
 * Fix release version (2.1)
 *
 * Revision 1.17  1992/07/05  21:26:59  kardel
 * update for new STREAMS (multi protocol) module
 *
 * Revision 1.16  1992/06/28  14:02:47  kardel
 * POWERUP heuristic improved
 * only trust receiver iff the receiver
 * has been seen in the SYNC state *after* having
 * seen no data from the receiver
 *
 * Revision 1.15  1992/06/21  16:55:43  kardel
 * trust constants new (trust until expected worst error is around 2 ms)
 * initialization message with current parameters
 * don't trust a POWERUP state unless the receiver was at least
 * synchronized once and synchronization has been within the
 * last clockinfo[CL_TYPE(dcf->unit)].cl_maxunsync seconds
 *
 * Revision 1.14  1992/06/09  17:43:13  kardel
 * subtle bug in rectime ts fixed (cannot send a future reference time!)
 * limited POWERUP/NOPZF messages to 1 every ten minutes
 *
 * Revision 1.13  1992/06/09  16:36:58  kardel
 * only providing PPS information on a synchronised clock status
 *
 * Revision 1.12  1992/06/09  15:19:04  kardel
 * noresponse logging (statistics) for PPS mode
 *
 * Revision 1.11  1992/06/07  11:41:42  kardel
 * revised PPS support (add 128 to unitnumber for PPS signalling)
 * this reduces the Number of available receiver types to 8 (still ok)
 * defines for dcf PPS support are -DPPS -DDCFPPS
 *
 * Revision 1.10  1992/06/06  23:40:50  kardel
 * PPS support for stabilizing local clock
 *
 * Revision 1.9  1992/06/03  13:52:43  kardel
 * default is to let NTP do the clock setting
 *
 * Revision 1.8  1992/03/13  17:19:58  kardel
 * xntp3 Beta
 *
 * Revision 1.7  1992/02/07  11:03:48  kardel
 * new constants and PZF 535/OCXO support
 *
 * Revision 1.6  1992/01/27  17:17:51  kardel
 * moved event generation after status update in dcf structure (trap
 * consistency bug)
 *
 * Revision 1.5  1992/01/16  19:46:03  kardel
 * new parameters for PZF 535
 * improved event code
 *   MAPPING: POWERUP/NOPZF -> CEVNT_FAULT
 *            NOSYNC        -> CEVNT_PROP
 *            SYNC          -> CEVNT_NOMINAL
 *
 * Revision 1.4  1992/01/10  18:53:09  kardel
 * compatibility hack fixed
 *
 * Revision 1.3  1992/01/10  18:36:40  kardel
 * added backward compatibility macros for v2 xntp
 *
 * Revision 1.2  1992/01/10  13:36:01  kardel
 * bugs (syntactic) removed
 *
 * Revision 1.1  1991/12/16  13:24:34  kardel
 * Initial revision
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/termios.h>
#include <sys/stream.h>
#include <sys/stropts.h>
#include <sys/errno.h>

#include "ntp_syslog.h"
#include "ntp_fp.h"
#include "ntp.h"
#include "ntp_refclock.h"
#include "ntp_unixtime.h"
#include "dcf77.h"

#if defined(REFCLOCK) && defined(DCF)

#if !defined(NO_SCCSID) && !defined(lint)
static char rcsid[]="$Id: refclock_dcf77.c,v 2.4 1992/07/09 17:10:37 kardel XNTP-DCF-1 $ FAU";
#endif

extern char *malloc();
extern char *ctime();
extern char *rindex();

/*
 * This driver provides the support for a Meinberg DCF77 receiver DCF77 PZF 535
 * and DCF 77 U/A 31
 * connected via a 9600 baud serial line
 * This driver requires a DCF streams module to be pushed onto
 * the tty line to preprocess the data from the receiver
 * 
 * Some receiver do NOT support:
 *          - announcement of alternate sender usage
 *          - leap second indication
 * 
 * The streams module checks for consistency of the data and converts 
 * this format to unix time (seconds since beginning of 1970)
 * also different time code formats are detected and converted accordingly
 * It returns the dcf time the time of receive plus some status information
 * to the streams head to be received from this driver
 *
 * Receiver setup:
 *	output time code every second
 *	Baud rate 9600 7E2S
 */

/*
 * the unit field selects for one the prot to be used (lower 4 bits)
 * and for the other the clock type in case of different but similar
 * receivers (bits 4-6)
 * the most significat bit encodes PPS support
 * when the most significant bit is set the pps telegrams will be used
 * for controlling the local clock (ntp_loopfilter.c)
 * receiver specific configration data is kept in the clockinfo field.
 */

/*
 * Definitions
 */
#define	MAXUNITS	16	/* maximum number of DCF units permitted */
#define DCFDEVICE	"/dev/dcf77-%d" /* device to open %d is unit number */
/*
 * Speeds we usually run the clock ports at.
 */
#define	DCFSPEED	B9600	/* there is no other choice */

/*
 * receiver specific constants (i wish we could configure better via the interface)
 */
/*
 * Meinberg DCF U/A 31 (AM) receiver
 */
#define	DCFUA31_PRECISION	(-8)		/* accounts for receiver errors */
#define	DCFUA31_ROOTDELAY	0x00000D00         /* 50.78125ms */
#define	DCFUA31_BASEDELAY	0x000002C0	   /* 10.7421875ms: 10 ms (+/- 3 ms) */
#define	DCFUA31_DESCRIPTION	"Meinberg DCF77 U/A 31"
#define DCFUA31_MAXUNSYNC       60*30              /* only trust clock for 1/2 hour */
#define DCFUA31_CFLAG           DCFSPEED|CS7|CREAD|PARENB|CLOCAL|HUPCL

/*
 * Meinberg DCF PZF535/TCXO receiver
 */
#define	DCFPZF535_PRECISION	(-18)		/* 355kHz clock (f(DCF77)*4(PLL))*/
#define	DCFPZF535_ROOTDELAY	0x00000034         /* 800us */
#define	DCFPZF535_BASEDELAY	0x000009D3	   /* 38.38ms +- 60us (oscilloscope) */
#define	DCFPZF535_DESCRIPTION	"Meinberg DCF77 PZF 535/TCXO"
#define DCFPZF535_MAXUNSYNC     60*60*12           /* only trust clock for 12 hours
						    * @ 5e-8df/f we have accumulated
						    * at most 2.16 ms (thus we move to
						    * NTP synchronisation */
#define DCFPZF535_CFLAG         DCFSPEED|CS7|CREAD|PARENB|CLOCAL|HUPCL

/*
 * Meinberg DCF PZF535/OCXO receiver
 */
#define	DCFPZF535OCXO_PRECISION	(-18)		/* 355kHz clock (f(DCF77)*4(PLL))*/
#define	DCFPZF535OCXO_ROOTDELAY	0x00000034         /* 800us (max error * 10) */
#define	DCFPZF535OCXO_BASEDELAY	0x000009D3	   /* 38.38ms +- 60us (oscilloscope) */
#define	DCFPZF535OCXO_DESCRIPTION "Meinberg DCF77 PZF 535/OCXO"
#define DCFPZF535OCXO_MAXUNSYNC     60*60*96       /* only trust clock for 4 days
						    * @ 5e-9df/f we have accumulated
						    * at most an error of 1.73 ms
						    * (thus we move to NTP synchronisation) */
#define DCFPZF535OCXO_CFLAG     DCFSPEED|CS7|CREAD|PARENB|CLOCAL|HUPCL

/*
 * ELV DCF7000 Wallclock-Receiver/Switching Clock (Kit)
 *
 * This is really not the hottest clock - but before you have nothing ...
 */
#define DCF7000_PRECISION (-8)	/*
				 * I don't hav much faith in this - but I haven't
				 * evaluated it yet
				 */
#define DCF7000_ROOTDELAY	0x00000364 /* 13 ms */
#define DCF7000_BASEDELAY	0x000067AE /* 405 ms - slow blow */
#define DCF7000_DESCRIPTION	"ELV DCF7000"
#define DCF7000_MAXUNSYNC	(60*5) /* sorry - but it just was not build as a clock */
#define DCF7000_CFLAG           DCFSPEED|CS8|CREAD|PARENB|PARODD|CLOCAL|HUPCL

struct clockinfo
{
  u_fp   cl_rootdelay;		/* rootdelay */
  u_fp   cl_basedelay;		/* current offset */
  s_char cl_precision;		/* device precision */
  char * cl_description;	/* device name */
  u_long cl_maxunsync;		/* time to trust oscillator after loosing synch */
  u_long cl_cflag;              /* terminal io flags */
} clockinfo[] =
{				/*   0.  0.0.128 - base offset for PPS support */
  {				/* 127.127.8.<device> */
    DCFPZF535_ROOTDELAY,
    DCFPZF535_BASEDELAY,
    DCFPZF535_PRECISION,
    DCFPZF535_DESCRIPTION,
    DCFPZF535_MAXUNSYNC,
    DCFPZF535_CFLAG
  },
  {				/* 127.127.8.16+<device> */
    DCFPZF535OCXO_ROOTDELAY,
    DCFPZF535OCXO_BASEDELAY,
    DCFPZF535OCXO_PRECISION,
    DCFPZF535OCXO_DESCRIPTION,
    DCFPZF535OCXO_MAXUNSYNC,
    DCFPZF535OCXO_CFLAG
  },
  {				/* 127.127.8.32+<device> */
    DCFUA31_ROOTDELAY,
    DCFUA31_BASEDELAY,
    DCFUA31_PRECISION,
    DCFUA31_DESCRIPTION,
    DCFUA31_MAXUNSYNC,
    DCFUA31_CFLAG
  },
  {				/* 127.127.8.48+<device> */
    DCF7000_ROOTDELAY,
    DCF7000_BASEDELAY,
    DCF7000_PRECISION,
    DCF7000_DESCRIPTION,
    DCF7000_MAXUNSYNC,
    DCF7000_CFLAG
  }
};

int ncltypes = sizeof(clockinfo) / sizeof(struct clockinfo);

#define CL_TYPE(x)  (((((x) >> 4) & 0x7) > ncltypes) ? 0 : ((x) >> 4) & 0x7)
#define CL_PPS(x)   (x & 0x80)
#define CL_UNIT(x)  ((x) & 0xF)
  
/*
 * Other constant stuff
 */
#define	DCFREFID	"DCF\0"
#define	DCFHSREFID	0x7f7f08ff	/* 127.127.8.255 refid for hi strata */

#define DCFNOSYNCREPEAT (10*60)		/* mention uninitialized clocks all 10 minutes */


/* Fudge factor data */

static u_char stratumtouse[MAXUNITS];

struct dcfunit
{
  struct peer        *peer;		/* backlink to peer structure */
  int                 fd;		/* device file descriptor */
  struct refclockio   io;		/* io system structure (used in PPS mode) */
  u_char              pollonly;		/* 1 for polling only (no PPS mode) */
  u_char              pollneeddata; 	/* 1 for receive sample expected in PPS mode */
  u_char              status;		/* current status */
  u_char              lastevent; 	/* last not NORMAL status */
  u_char              unit;		/* encoded unit/type/PPS */
  u_long              polls;		/* polls from NTP protocol machine */
  u_long              noresponse; 	/* number of expected but not seen datagrams */
  u_long              badformat; 	/* bad format (failed format conversions) */
  u_long              baddata;		/* usually (bad receive length) */
  u_short	      lastformat;       /* last format used */
  u_long              lastsync;		/* time (xntp) when clock was last seen fully synchronized */
  u_long              timestarted; 	/* time (xntp) when peer clock was instantiated */
  u_long	      nosynctime; 	/* time (xntp) when last nosync message was posted */
  u_long              lastmissed;       /* time (xntp) when poll didn't get data (powerup heuristic) */
  char               *description; 	/* clock description string */
  dcftime_t           time;		/* last STREAMS (dcf module) data */
};

static struct dcfunit *dcfunits[MAXUNITS];
static u_char	      unitinuse[MAXUNITS];
static dcfctl_t	        dcf_ctl[MAXUNITS];

static void dcf_receive();
static void dcf_process();

extern int io_addclock();
extern void io_closeclock();

extern u_long current_time;
extern u_long ustotslo[];
extern u_long ustotsmid[];
extern u_long ustotshi[];

static char *dcfstate(state, buffer)
  unsigned long state;
  char *buffer;
{
  static struct bits
    {
      unsigned long bit;
      char         *name;
    } flagstrings[] =
    {
      { DCFB_ANNOUNCE, "DST SWITCH WARNING" },
      { DCFB_POWERUP,  "NOT SYNCHRONIZED" },
      { DCFB_NOSYNC,   "TIME CODE NOT CONFIRMED" },
      { DCFB_DST,      "DST" },
      { DCFB_UTC,      "UTC DISPLAY" },
      { DCFB_LEAP,     "LEAP WARNING" },
      { DCFB_ALTERNATE,"ALTERNATE ANTENNA" },
      { 0 }
    };
  int i;

  *buffer = '\0';
  
  i = 0;
  while (flagstrings[i].bit)
    {
      if (flagstrings[i].bit & state)
	{
	  if (buffer[0])
	    strcat(buffer, ", ");
	  strcat(buffer, flagstrings[i].name);
	}
      i++;
    }
  if (state & (DCFB_S_LEAP|DCFB_S_LOCATION))
    {
      if (buffer[0])
	strcat(buffer, ", ");
      strcat(buffer, "supported: ");
      if (DCF_S_LEAP(state)) strcat(buffer, "LEAP");
      if (buffer[0])
	strcat(buffer, ", ");
      if (DCF_S_LOCATION(state)) strcat(buffer, "LOCATION");
    }
  return buffer;
}

/*
 * dcf_init - initialize internal dcf driver data
 */
void
dcf_init()
{
  register int i;		/* loop variable */
  /* 
   * zero out static structures
   */
  bzero((char *)&unitinuse[0],sizeof(unitinuse));
  for (i = 0; i < MAXUNITS; i++)
    {
      dcf_ctl[i].dcfstatus.flags = DCF_STAT_FILTER; /* get an impression on quality */
      /* init fudge vals */
      stratumtouse[i] = STRATUM_REFCLOCK;
    }
}


/*
 * dcf_start - open the DCF devices and initialize data for processing
 */
int
dcf_start(sysunit, peer)
	u_int sysunit;
	struct peer *peer;
{
  u_int unit;
  int fd232, i;
  struct termios tm;
  struct strioctl strioc;
  struct dcfunit * dcf;
  char dcfdev[sizeof(DCFDEVICE)+20];
  u_int type;

  type = CL_TYPE(sysunit);
  unit = CL_UNIT(sysunit);

  if (unit >= MAXUNITS)
    {
      syslog(LOG_ERR, "dcf clock[%d]: unit number invalid (max %d)",
	     unit, MAXUNITS);
      return 0;
    }
  if (unitinuse[unit])
    {
      syslog(LOG_ERR, "dcf clock[%d]: unit in use", unit);
      return 0;
    }

  /*
   * Unit okay, attempt to open the device.
   */
  (void) sprintf(dcfdev, DCFDEVICE, unit);

  fd232 = open(dcfdev, O_RDWR, 0777);
  if (fd232 == -1)
    {
      syslog(LOG_ERR, "dcf clock[%d]: open of %s failed: %m", unit, dcfdev);
      return 0;
    }

  /* 
   * pop all possibly active streams moduls from the tty
   */
  while(ioctl(fd232, I_POP, 0) == 0)
    /* empty loop !! */;
	
  /* 
   * configure terminal line
   */
  if (ioctl(fd232, TCGETS, &tm) == -1)
    {
      syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, TCGETS, &tm): %m", unit, fd232);
      close(fd232);
      return 0;
    }
  else
    {
      tm.c_cc[VSTOP] = '\0';	/* ETX */
      tm.c_cc[VINTR] = '\0';	/* disable interrupt (usual '\3'!!!) */
      tm.c_cflag     = clockinfo[type].cl_cflag;
      DCFSPEED|CS7|CREAD|PARENB|CLOCAL|HUPCL;
	    
      if (ioctl(fd232, TCSETS, &tm) == -1)
	{
	  syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, TCSETS, &tm): %m", unit, fd232);
	  close(fd232);
	  return 0;
	}
    }

  /* 
   * now push the dcf streams module
   * it will ensure exclusive access to the device
   */
  if (ioctl(fd232, I_PUSH, "dcf") == -1)
    {
      syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_PUSH, \"dcf\"): %m",
	     unit, fd232);
      close(fd232);
      return 0;
    }
	
  /* 
   * now configure the streams module
   */
  strioc.ic_cmd     = DCFIOC_SETSTAT;
  strioc.ic_timout  = 0;
  strioc.ic_dp      = (char *)&dcf_ctl[unit];
  strioc.ic_len     = sizeof (dcf_ctl[0]);
	
  if (ioctl(fd232, I_STR, &strioc) == -1)
    {
      syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_STR, DCFIOC_SETSTAT): %m", unit, fd232);
      close(fd232);
      return 0;
    }

  /*
   * Looks like this might succeed.  Find memory for the structure.
   * Look to see if there are any unused ones, if not we malloc()
   * one.
   */
  if (dcfunits[unit] != 0)
    {
      dcf = dcfunits[unit];	/* The one we want is okay */
    }
  else
    {
      for (i = 0; i < MAXUNITS; i++)
	{
	  if (!unitinuse[i] && dcfunits[i] != 0)
	    break;
	}
      if (i < MAXUNITS)
	{
	  /*
	   * Reclaim this one
	   */
	  dcf = dcfunits[i];
	  dcfunits[i] = 0;
	}
      else
	{
	  dcf = (struct dcfunit *)
	    emalloc(sizeof(struct dcfunit));
	}
    }
  bzero((char *)dcf, sizeof(struct dcfunit));
  dcfunits[unit] = dcf;

  /*
   * Set up the structures
   */
  dcf->peer = peer;
  dcf->unit = (u_char)sysunit;
  dcf->timestarted = current_time;
  dcf->description = clockinfo[type].cl_description;
  dcf->pollneeddata = 0;
  dcf->pollonly = 1;		/* go for default polling mode */
  dcf->lastformat = ~0;		/* assume no format known */
  dcf->nosynctime = 0;		/* assume clock reasonable */
  dcf->lastmissed = 0;		/* assume got everything */

  /* We do not want to get signals on any packet from the streams module
   * It will keep one old packet and after this is read the next read
   * will return an accurate new packet to be processed.
   */
  dcf->fd = fd232;
	
  /*
   * All done.  Initialize a few random peer variables, then
   * return success.
   */
  peer->rootdelay  = clockinfo[type].cl_rootdelay;
  peer->precision = clockinfo[type].cl_precision;
  peer->stratum = stratumtouse[unit];
  if (stratumtouse[unit] <= 1)
    bcopy(DCFREFID, (char *)&peer->refid, 4);
  else
    peer->refid = htonl(DCFHSREFID);
  unitinuse[unit] = 1;
	
#if defined(PPS) && defined(DCFPPS)
  if (CL_PPS(dcf->unit))
    {
      /*
       * Insert in device list.
       */
      dcf->io.clock_recv = dcf_receive;
      dcf->io.srcclock = (caddr_t)dcf;
      dcf->io.datalen = 0;
      dcf->io.fd = dcf->fd;	/* replicated, but what the heck */
      if (!io_addclock(&dcf->io))
	{
	  syslog(LOG_ERR,
		 "dcf_io_addclock: addclock %s fails (switching to polling mode)", dcfdev);
	}
      else 
	{
	  dcf->pollonly = 0;	/*
				 * update at receipt of time_stamp - also
				 * supports PPS processing
				 */
	}
    }
#endif	
  syslog(LOG_INFO, "dcf clock[%d]: reference clock \"%s\" (device %s) added",
	 CL_UNIT(dcf->unit), 
	 dcf->description, dcfdev);
  
  syslog(LOG_INFO, "dcf clock[%d]:  Stratum %d, %sPPS support, trust time %d s, precision %d",
	 CL_UNIT(dcf->unit), 
	 dcf->peer->stratum, dcf->pollonly ? "no " : "",
	 clockinfo[CL_TYPE(dcf->unit)].cl_maxunsync, dcf->peer->precision);
  
  syslog(LOG_INFO, "dcf clock[%d]:  rootdelay %s s, phaseadjust %s s",
	 CL_UNIT(dcf->unit), 
	 ufptoa(clockinfo[CL_TYPE(dcf->unit)].cl_rootdelay, 6),
	 ufptoa(clockinfo[CL_TYPE(dcf->unit)].cl_basedelay, 6));
  
  return 1;
}

/*
 * dcf_shutdown - shut down a DCF clock
 */
void
dcf_shutdown(unit)
	int unit;
{
	register struct dcfunit *dcf;
	
	unit = CL_UNIT(unit);
	
	if (unit >= MAXUNITS) {
		syslog(LOG_ERR,
		  "dcf clock[%d]: INTERNAL ERROR, unit invalid (max %d)",
		    unit,MAXUNITS);
		return;
	}
	if (!unitinuse[unit]) {
		syslog(LOG_ERR,
		 "dcf clock[%d]: INTERNAL ERROR, unit not in use", unit);
		return;
	}

	dcf = dcfunits[unit];
	
	/*
	 * Tell the I/O module to turn us off.  We're history.
	 */
	if (!dcf->pollonly)
	  io_closeclock(&dcf->io);
	else
	  (void) close(dcf->fd);

	syslog(LOG_INFO, "dcf clock[%d]: reference clock \"%s\" removed",
	       CL_UNIT(dcf->unit), dcf->description);

	unitinuse[unit] = 0;
}



/*
 * event handling
 */

static void dcf_event(dcf, event)
  struct dcfunit *dcf;
  int event;
{
  if (dcf->status != (u_char) event)
    {
      dcf->status    = (u_char)event;
      if (event != CEVNT_NOMINAL)
        dcf->lastevent = dcf->status;
      report_event(EVNT_PEERCLOCK, dcf->peer);
    }
}

/*
 * dcf_receive - called by io handler
 *
 * this routine is called each time a time packed is
 * deliverd by the STREAMS module
 * if PPS processing is requested, pps_sample is called
 */
static void dcf_receive(rbufp)
  struct recvbuf *rbufp;
{
  struct dcfunit *dcf = (struct dcfunit *)rbufp->recv_srcclock;
  dcftime_t dcftime;
  
  if (rbufp->recv_length != sizeof(dcftime_t))
    {
      syslog(LOG_ERR,"dcf receive[%d]: bad size (got %d expected %d)",
	     CL_UNIT(dcf->unit), rbufp->recv_length, sizeof(dcftime_t));
      dcf->baddata++;
      return;
    }
  bcopy((caddr_t)&rbufp->recv_space, (caddr_t)&dcftime, sizeof(dcftime_t));

  dcf_process(dcf, &dcftime);
}

/*
 * dcf_poll - called by the transmit procedure
 */
void
dcf_poll(unit, peer)
	int unit;
	struct peer *peer;
{ 
  extern int errno;

  int fd, i, rtc;
  fd_set fdmask;
  struct timeval timeout;
  struct dcfunit *dcf;
  dcftime_t dcftime;

  unit = CL_UNIT(unit);
  
  if (unit >= MAXUNITS)
    {
      syslog(LOG_ERR, "dcf clock poll[%d]: INTERNAL: unit invalid",
	     unit);
      return;
    }
  if (!unitinuse[unit])
    {
      syslog(LOG_ERR, "dcf clock poll[%d]: INTERNAL: unit unused",
	     unit);
      return;
    }

  if (peer != dcfunits[unit]->peer)
    {
      syslog(LOG_ERR,
	     "dcf clock poll[%d]: INTERNAL: peer incorrect",
	     unit);
      return;
    }

  dcf = dcfunits[unit];

  /*
   * Update clock stat counters
   */
  dcf->polls++;

#if defined(PPS) && defined(DCFPPS)
  /*
   * in PPS mode we just mark that we want the next sample
   * for the clock filter
   */
  if (!dcf->pollonly)
    {
      if (dcf->pollneeddata)
	{
	  /*
	   * bad news - didn't get a response last time
	   */
	  dcf->noresponse++;
	  dcf->lastmissed = current_time;
          syslog(LOG_WARNING, "dcf clock[%d]: no data from device within poll interval", CL_UNIT(dcf->unit));
	}
      dcf->pollneeddata = 1;
      return;
    }
#endif
  
  /* 
   * now we do the following:
   *    - read the first packet from the dcf module  (OLD !!!)
   *    - read the second packet from the dcf module (fresh)
   *    - compute values for xntp
   */
	  
  FD_ZERO(&fdmask);
  fd = dcf->fd;
  FD_SET(fd, &fdmask);
  timeout.tv_sec = 0;
  timeout.tv_usec = 500000;	/* 0.5 sec */

  while ((rtc = select(fd + 1, &fdmask, 0, 0, &timeout)) != 1)
    {
      /* no data from the radio clock */

      if (rtc == -1)
	{
	  if (errno == EINTR)
	    {
	      continue;
	    }
	  else
	    {
              syslog(LOG_WARNING, "dcf clock[%d]: no data[old] from device (select() error: %m)", unit);
	    }
	}
      else
	{
          syslog(LOG_WARNING, "dcf clock[%d]: no data[old] from device", unit);
	}
      dcf->noresponse++;
      dcf->lastmissed = current_time;
      dcf_event(dcf, CEVNT_TIMEOUT);
      
      return;
    }

  while (((i = read(fd, &dcftime, sizeof(dcftime))) < sizeof(dcftime)))
    {
      /* bad packet */
      if ( i == -1)
	{
	  if (errno == EINTR)
	    {
	      continue;
	    }
	  else
	    {
              syslog(LOG_WARNING, "dcf clock[%d]: bad read[old] from streams module (read() error: %m)", unit, i, sizeof(dcftime));
	    }
	}
      else
	{
          syslog(LOG_WARNING, "dcf clock[%d]: bad read[old] from streams module (got %d bytes - expected %d bytes)", unit, i, sizeof(dcftime));
	}
      dcf->baddata++;
      dcf_event(dcf, CEVNT_BADREPLY);
      
      return;
    }

  timeout.tv_sec = 1;
  timeout.tv_usec = 500000;	/* 1.500 sec */
  FD_ZERO(&fdmask);
  FD_SET(fd, &fdmask);
  while ((rtc = select(fd + 1, &fdmask, 0, 0, &timeout)) != 1)
    {
      /* no data from the radio clock */

      if (rtc == -1)
	{
	  if (errno == EINTR)
	    {
	      continue;
	    }
	  else
	    {
              syslog(LOG_WARNING, "dcf clock[%d]: no data[new] from device (select() error: %m)", unit);
	    }
	}
      else
	{
          syslog(LOG_WARNING, "dcf clock[%d]: no data[new] from device", unit);
	}
	      
      dcf->noresponse++;
      dcf->lastmissed = current_time;
      dcf_event(dcf, CEVNT_TIMEOUT);
      
      return;
    }

  if (((i = read(fd, &dcftime, sizeof(dcftime))) < sizeof(dcftime)))
    {
      /* bad packet */
      if ( i== -1)
	{
          syslog(LOG_WARNING, "dcf clock[%d]: bad read[new] from streams module (read() error: %m)", unit, i, sizeof(dcftime));
	}
      else
	{
          syslog(LOG_WARNING, "dcf clock[%d]: bad read[new] from streams module (got %d bytes - expected %d bytes)", unit, i, sizeof(dcftime));
	}
      dcf->baddata++;
      dcf_event(dcf, CEVNT_BADREPLY);
      
      return;
    }
  dcf_process(dcf, &dcftime);
}

static void dcf_process(dcf, dcftime)
  struct dcfunit *dcf;
  dcftime_t      *dcftime;
{
  struct timeval usecdisp;
  l_fp off, rectime, reftime, dispersion;
  u_fp delay;
  unsigned long type = CL_TYPE(dcf->unit);
  
  /*
   * now, any changes ?
   */
  if (dcf->time.dcf_state != dcftime->dcf_state)
    {
      char tmp1[80];
      char tmp2[80];
      /*
       * something happend
       */
	    
      (void) dcfstate(dcftime->dcf_state, tmp1);
      (void) dcfstate(dcf->time.dcf_state, tmp2);
	    
      syslog(LOG_INFO,"dcf state change[%d]: OLD: %s, NEW: %s",
	     CL_UNIT(dcf->unit), tmp2, tmp1);
    }

  /*
   * check for format changes
   * (in case somebody has swapped clocks 8-)
   */
  if (dcf->lastformat != dcftime->dcf_format)
    {
      struct strioctl strioc;
      dcfctl_t tmpctl;

      strioc.ic_cmd     = DCFIOC_GETFMT;
      strioc.ic_timout  = 0;
      strioc.ic_dp      = (char *)&tmpctl;
      strioc.ic_len     = sizeof (tmpctl);
	
      tmpctl.dcfgetformat.dcf_format = dcftime->dcf_format;
      
      if (ioctl(dcf->fd, I_STR, &strioc) == -1)
	{
	  syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_STR, DCFIOC_GETFMT): %m", CL_UNIT(dcf->unit), dcf->fd);
	}
      else
	{
	  syslog(LOG_INFO, "dcf clock[%d]: new packet format \"%s\"",
		 CL_UNIT(dcf->unit), tmpctl.dcfgetformat.dcf_buffer);
	}
      dcf->lastformat = dcftime->dcf_format;
    }
  
  /*
   * remember for future
   */
  dcf->time = *dcftime;

  /*
   * check to see, whether the clock did a complete powerup or lost PZF signal
   * and post correct events for current condition
   */
  if (DCF_POWERUP(dcftime->dcf_state))
    {
      /*
       * this is bad, as we have completely lost synchronisation
       * well this is a problem with the receiver here
       * for DCF U/A 31 the lost synchronisation ist true
       * as it is the powerup state and the time is taken
       * from a crude real time clock chip
       * for the PZF series this is only partly true, as
       * DCF_POWERUP only means that the pseudo random
       * phase shift sequence cannot be found. this is only
       * bad, if we have never seen the clock in the SYNC
       * state, where the PHASE and EPOCH are correct.
       * for reporting events the above business does not
       * really matter, but we can use the time code
       * even in the POWERUP state after having seen
       * the clock in the synchronized state (PZF class
       * receivers) unless we have had a telegram disruption
       * after having seen the clock in the SYNC state. we
       * thus require having seen the clock in SYNC state
       * *after* having missed telegrams (noresponse) from
       * the clock. one problem remains: we might use erroneously
       * POWERUP data if the disruption is shorter than 1 polling
       * interval. fortunately powerdowns last usually longer than 64
       * seconds and the receiver is at least 2 minutes in the
       * POWERUP or NOSYNC state before switching to SYNC
       */
      dcf_event(dcf, CEVNT_FAULT);
      if (dcf->nosynctime)
	{
	  /*
	   * repeated POWERUP/NOSYNC state - look whether
	   * the message should be repeated
	   */
	   if (current_time - dcf->nosynctime > DCFNOSYNCREPEAT)
	     {
	       syslog(LOG_WARNING,"dcf clock[%d]: *STILL* NOT SYNCHRONIZED (POWERUP or no PZF signal)",
	              CL_UNIT(dcf->unit));
               dcf->nosynctime = current_time;
             }
	}
      else
	{
	  syslog(LOG_ERR,"dcf clock[%d]: NOT SYNCHRONIZED (POWERUP or no PZF signal)",
		 CL_UNIT(dcf->unit));
          dcf->nosynctime = current_time;
	}
    }
  else
    {
      /*
       * we have two states left
       *
       * SYNC:
       *  this state means that the EPOCH (timecode) and PHASE
       *  information has be read correctly (at least two
       *  successive DCF77 timecodes were received correctly)
       *  this is the best possible state - full trust
       *
       * NOSYNC:
       *  The clock should be on phase with respect to the second
       *  signal, but the timecode has not been received correctly within
       *  at least the last two minutes. this is a sort of half baked state
       *  for DCF U/A 31 this is bad news (clock running without timecode
       *  confirmation)
       *  PZF 535 has also no time confirmation, but the phase should be
       *  very precise as the PZF signal can be decoded
       */
      dcf->nosynctime = 0; /* current state is better than worst state */

      if (DCF_SYNC(dcftime->dcf_state))
	{
	  /*
	   * currently completely synchronized - best possible state
	   */
	  dcf->lastsync = current_time;
	  /*
	   * log OK status
	   */
	  dcf_event(dcf, CEVNT_NOMINAL);
	}
      else
	{
	  /*
	   * we have had some problems receiving the time code
	   */
	  dcf_event(dcf, CEVNT_PROP);
	}
    }

  /*
   * calculate time offset including systematic delays
   * off = DCF-timestamp + propagation delay - kernel time stamp
   */
  {
    l_fp ts;
    l_fp offset;
    
    FPTOLFP(clockinfo[type].cl_basedelay, &offset);
    
    if (!buftvtots(&dcftime->dcf_time, &off))
      return;

    reftime = off;
    
    L_ADD(&off, &offset);
    rectime = off; /* this makes org time and xmt time somewhat artificial */
    
    if (!buftvtots(&dcftime->dcf_stime, &ts))
      return;
    
    L_SUB(&off, &ts);
  }
  
#if defined(PPS) && defined(DCFPPS)
  if (!dcf->pollonly && DCF_SYNC(dcftime->dcf_state))
    {
      /*
       * only provide PPS information when clock
       * is in sync
       * thus PHASE and EPOCH are correct
       */
      pps_sample(&off);
    }

  /*
   * ready, unless the machine wants a sample
   */
  if (!dcf->pollonly && !dcf->pollneeddata)
    return;

  dcf->pollneeddata = 0;
#endif
  
  /*
   * convert usec dispersion into NTP TS world
   */
  
  usecdisp.tv_sec  = dcftime->dcf_usecdisp / 1000000;
  usecdisp.tv_usec = dcftime->dcf_usecdisp % 1000000;

  TVTOTS(&usecdisp, &dispersion);
  
  delay = 0;			/* no need for refclocks in V3 */
  
  /*
   * and now stick it into the clock machine
   * samples are only valid iff lastsync is not too old and
   * we have seen the clock in sync at least once
   * after the last time we did'nt see an expected data telegram
   * see the clock states section above for more reasoning
   */
  refclock_receive(dcf->peer, &off, delay, LFPTOFP(&dispersion), &reftime, &rectime,
		   (((current_time - dcf->lastsync) > clockinfo[type].cl_maxunsync) ||
		    (dcf->lastsync <= dcf->lastmissed)) ? LEAP_NOTINSYNC :
		    (DCF_LEAP(dcftime->dcf_state) ? LEAP_ADDSECOND : LEAP_NOWARNING));
				/*
				 * DCF only adds LEAP seconds - do they
				 * know more ?
				 */
}

/*
 * dcf_leap - called when a leap second occurs
 */

dcf_leap()
{
	/*
	 * DCF77 does encode a leap warning, if we only could use it with
	 * the current receivers ...
	 */
}


/*
 * dcf_control - set fudge factors, return statistics
 */
void
dcf_control(unit, in, out)
  u_int unit;
  struct refclockstat *in;
  struct refclockstat *out;
{
  register struct dcfunit *dcf;
  dcfctl_t tmpctl;
  struct strioctl strioc;
  
  unsigned long type;
  static char outstatus[129];	/* status output buffer / hope this will be mutex */
  
  type = CL_TYPE(unit);
  unit = CL_UNIT(unit);
  
  if (unit >= MAXUNITS)
    {
      syslog(LOG_ERR, "dcf clock[%d]: unit invalid (max %d)",
	     unit, MAXUNITS-1);
      return;
    }

  if (in != 0)
    {
      if (in->haveflags & CLK_HAVETIME1)
	clockinfo[type].cl_basedelay = LFPTOFP(&in->fudgetime1);

      if (in->haveflags & CLK_HAVETIME2)
	{
	  /* not USED */
	}
      
      if (in->haveflags & CLK_HAVEVAL1)
	{
	  stratumtouse[unit] = (u_char)(in->fudgeval1 & 0xf);
	  if (unitinuse[unit])
	    {
	      struct peer *peer;

	      /*
	       * Should actually reselect clock, but
	       * will wait for the next timecode
	       */
	      dcf = dcfunits[unit];
	      peer = dcf->peer;
	      peer->stratum = stratumtouse[unit];
	      if (stratumtouse[unit] <= 1)
		bcopy(DCFREFID, (char *)&peer->refid, 4);
	      else
		peer->refid = htonl(DCFHSREFID);
	    }

	}

      if (in->haveflags & CLK_HAVEVAL2)
	{
	  /* NOT USED */
	}
      
      if (in->haveflags & (CLK_HAVEFLAG1|CLK_HAVEFLAG2|CLK_HAVEFLAG3|CLK_HAVEFLAG4))
	{
	  dcf_ctl[unit].dcfstatus.flags = in->flags & (CLK_FLAG1|CLK_FLAG2|CLK_FLAG3|CLK_FLAG4);
	}

      if (in->haveflags & (CLK_HAVEVAL2|CLK_HAVETIME2|CLK_HAVEFLAG1|CLK_HAVEFLAG2|CLK_HAVEFLAG3|CLK_HAVEFLAG4))
	{
	  /* 
	   * now configure the streams module
	   */
	  strioc.ic_cmd     = DCFIOC_SETSTAT;
	  strioc.ic_timout  = 0;
	  strioc.ic_dp      = (char *)&dcf_ctl[unit];
	  strioc.ic_len     = sizeof (dcf_ctl[0]);
	
	if (ioctl(dcfunits[unit]->fd, I_STR, &strioc) == -1)
	  {
	    syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_STR, DCFIOC_SETSTAT): %m", unit, dcfunits[unit]->fd);
	  }
	}
    }
  
  if (out != 0)
    {
      out->type = REFCLK_DCF77;
      out->haveflags = CLK_HAVETIME1|CLK_HAVEVAL1|CLK_HAVEFLAG1;
      out->clockdesc = clockinfo[type].cl_description;

      FPTOLFP(clockinfo[type].cl_basedelay, &out->fudgetime1);
      
      out->fudgeval1 = (long)stratumtouse[unit];

      out->flags     = dcf_ctl[unit].dcfstatus.flags;
      if (unitinuse[unit])
	{
	  char *t;
	  
	  dcf = dcfunits[unit];

	  strcpy(outstatus, ctime(&dcf->time.dcf_time.tv_sec));
	  t = rindex(outstatus, '\n');
	  
	  strioc.ic_cmd     = DCFIOC_TIMECODE;
	  strioc.ic_timout  = 0;
	  strioc.ic_dp      = (char *)&tmpctl;
	  strioc.ic_len     = sizeof (tmpctl);
	
	  if (ioctl(dcfunits[unit]->fd, I_STR, &strioc) == -1)
	    {
	      syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_STR, DCFIOC_TIMECODE): %m", unit, dcfunits[unit]->fd);
	    }
	  else
	    {
	      if (t)
		{
		  *t = ' ';
		  (void) dcfstate(tmpctl.dcfgettc.dcf_state, t+1);
		}
	      else
		{
		  strcat(outstatus, " ");
		  (void) dcfstate(tmpctl.dcfgettc.dcf_state, outstatus + strlen(outstatus));
		}
	      strcat(outstatus," <");
	      strncat(outstatus, tmpctl.dcfgettc.dcf_buffer, tmpctl.dcfgettc.dcf_count);
	      strcat(outstatus,">");
	      dcf->badformat += tmpctl.dcfgettc.dcf_badformat;
	    }
	  
	  strioc.ic_cmd     = DCFIOC_GETFMT;

	  tmpctl.dcfgetformat.dcf_format = tmpctl.dcfgettc.dcf_format;
	  
	  if (ioctl(dcfunits[unit]->fd, I_STR, &strioc) == -1)
	    {
	      syslog(LOG_ERR, "dcf clock[%d]: ioctl(%d, I_STR, DCFIOC_GETFMT): %m", unit, dcfunits[unit]->fd);
	    }
	  else
	    {
	      strcat(outstatus," (");
	      strncat(outstatus, tmpctl.dcfgetformat.dcf_buffer, tmpctl.dcfgetformat.dcf_count);
	      strcat(outstatus,")");
	    }

	  out->lencode       = strlen(outstatus);
	  out->lastcode      = outstatus;
	  out->timereset     = dcf->timestarted;
	  out->polls         = dcf->polls;
	  out->noresponse    = dcf->noresponse;
	  out->badformat     = dcf->badformat;
	  out->baddata       = dcf->baddata;
	  out->lastevent     = dcf->lastevent;
	  out->currentstatus = dcf->status;
	}
      else
	{
	  out->lencode       = 0;
	  out->lastcode      = 0;
	  out->polls         = out->noresponse = 0;
	  out->badformat     = out->baddata    = 0;
	  out->timereset     = 0;
	  out->currentstatus = out->lastevent = CEVNT_NOMINAL;
	}
    }
}

#endif
#endif
