/*************************************************************************/
/* (c) Copyright Tai Jin, 1988.  All Rights Reserved.                    */
/*     Hewlett-Packard Laboratories.                                     */
/*                                                                       */
/* Permission is hereby granted for unlimited modification, use, and     */
/* distribution.  This software is made available with no warranty of    */
/* any kind, express or implied.  This copyright notice must remain      */
/* intact in all versions of this software.                              */
/*                                                                       */
/* The author would appreciate it if any bug fixes and enhancements were */
/* to be sent back to him for incorporation into future versions of this */
/* software.  Please send changes to tai@iag.hp.com or ken@sdd.hp.com.   */
/*************************************************************************/

#ifndef lint
static char RCSid[] = "@(#)$Header: adjtimed.c,v 1.23 90/02/07 15:34:24 src Exp $";
static char RCSid2[] = "@(#) NOTE: SDD/jks default rate is 15 now";
#endif
/*
 * 91/12/30 Changed to BEHAVE like BSD adjtime: fixed rate of change!
 */
/*$Log:	adjtimed.c,v $
 * Revision 1.23  90/02/07  15:34:24  15:34:24  src (Source Hacker)
 * CHANGED KEY !!!
 * 
 * Revision 1.22  89/02/09  12:26:37  12:26:37  tai (Tai Jin (Guest))
 * *** empty log message ***
 * 
 * Revision 1.22  89/02/09  12:26:37  12:26:37  tai (Tai Jin)
 * added comment
 * 
 * Revision 1.21  88/08/30  01:08:31  01:08:31  tai (Tai Jin)
 * fix copyright notice again
 * 
 * Revision 1.20  88/08/30  00:51:56  00:51:56  tai (Tai Jin)
 * fix copyright notice
 * 
 * Revision 1.19  88/08/26  14:36:15  14:36:15  tai (Tai Jin)
 * removed rtprio -- possible system hang problem
 * 
 * Revision 1.18  88/08/25  23:02:30  23:02:30  tai (Tai Jin)
 * use RTPRIO_MAX for rtprio
 * 
 * Revision 1.17  88/04/16  11:09:12  11:09:12  tai (Tai Jin)
 * accumulate leftover adjustments
 * 
 * Revision 1.16  88/04/03  17:51:16  17:51:16  tai (Tai Jin)
 * replace Timeval2Double with a macro tvtod
 * 
 * Revision 1.15  88/04/03  07:27:03  07:27:03  tai (Tai Jin)
 * add -p option for percent rate of change
 * 
 * Revision 1.14  88/04/03  05:10:56  05:10:56  tai (Tai Jin)
 * fix long to double conversion of some numbers
 * other things
 * 
 * Revision 1.13  88/04/02  23:30:23  23:30:23  tai (Tai Jin)
 * *** empty log message ***
 * 
 * Revision 1.12  88/04/02  18:11:53  18:11:53  tai (Tai Jin)
 * ifdef 800 and other minor details
 * 
 * Revision 1.11  88/04/02  17:34:09  17:34:09  tai (Tai Jin)
 * change correction message
 * 
 * Revision 1.10  88/04/02  17:24:47  17:24:47  tai (Tai Jin)
 * don't log new rate if same as default
 * 
 * Revision 1.9  88/04/02  16:29:53  16:29:53  tai (Tai Jin)
 * close std*
 * 
 * Revision 1.8  88/04/02  16:24:40  16:24:40  tai (Tai Jin)
 * *** empty log message ***
 * 
 * Revision 1.7  88/04/02  16:23:37  16:23:37  tai (Tai Jin)
 * *** empty log message ***
 * 
 * Revision 1.6  88/04/02  16:18:35  16:18:35  tai (Tai Jin)
 * *** empty log message ***
 * 
 * Revision 1.5  88/04/02  16:11:07  16:11:07  tai (Tai Jin)
 * add -k; check clock rate before resetting
 * 
 * Revision 1.4  88/04/02  15:48:32  15:48:32  tai (Tai Jin)
 * fix new correction log message
 * 
 * Revision 1.3  88/04/02  15:33:07  15:33:07  tai (Tai Jin)
 * fix fork
 * 
 * Revision 1.2  88/04/02  15:15:03  15:15:03  tai (Tai Jin)
 * fix syslog calls
 * 
 * Revision 1.1  88/04/02  14:57:01  14:57:01  tai (Tai Jin)
 * Initial revision
 * */

/*
 * Adjust time daemon.
 * This deamon adjusts the rate of the system clock a la BSD's adjtime().
 * The adjtime() routine uses SYSV messages to communicate with this daemon.
 *
 * Caveat: This emulation uses an undocumented kernel variable.  As such, it
 * cannot be guaranteed to work in future HP-UX releases.  Perhaps a real
 * adjtime(2) will be supported in the future.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <time.h>
#include <signal.h>
#include <nlist.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#ifdef SYSLOG
#include <syslog.h>
#endif
#include "adjtime.h"

double atof();
extern int optind;
extern char *optarg;

int InitClockRate();
int AdjustClockRate();
#ifdef notdef
long GetClockRate();
#endif
int SetClockRate();
void ResetClockRate();
void Cleanup();
void Exit();

#define MILLION		1000000L

#define tvtod(tv)	((double)(long)tv.tv_sec + \
			((double)tv.tv_usec / (double)MILLION))

int verbose = 0;
#ifdef SYSLOG
int sysdebug = 0;
#endif
static int mqid;
static double oldrate = 0.0;
static long RATE = 500;

main(argc, argv)
     int argc;
     char **argv;
{
  struct timeval remains;
  struct sigvec vec;
  MsgBuf msg;
  char ch;
  int nofork = 0;
  int fd;

#ifdef SYSLOG
  openlog("adjtimed", LOG_PID, LOG_LOCAL6);
#endif

  while ((ch = getopt(argc, argv, "hkrvdfp:")) != EOF) {
    switch (ch) {
    case 'k':
    case 'r':
      if ((mqid = msgget(KEY, 0)) != -1) {
	if (msgctl(mqid, IPC_RMID, (struct msqid_ds *)0) == -1) {
#ifdef SYSLOG
	  syslog(LOG_ERR, "remove old message queue: %m");
#endif
	  perror("adjtimed: remove old message queue");
	  exit(1);
        }
      }

      if (ch == 'k')
	exit(0);

      break;

    case 'v':
      ++verbose, nofork = 1;
      break;

    case 'd':
#ifdef SYSLOG
      ++sysdebug;
#endif
      break;

    case 'f':
      nofork = 1;
      break;

    case 'p':
      if ((RATE = 10000L*atof(optarg)) <= 0 || RATE >= MILLION/10) {
	fputs("adjtimed: percentage must be between 0.0 and 10.0\n", stderr);
	exit(1);
      }

      break;

    default:
      puts("usage: adjtimed -hkrvdf -p rate");
      puts("-h\thelp");
      puts("-k\tkill existing adjtimed, if any");
      puts("-r\trestart (kills existing adjtimed, if any)");
      puts("-v\tdebug output (repeat for more output)");
      puts("-d\tsyslog output (repeat for more output)");
      puts("-f\tno fork");
      puts("-p rate\tpercent rate of change");
#ifdef SYSLOG
      syslog(LOG_ERR, "usage error");
#endif
      exit(1);
    } /* switch */
  } /* while */

  if (!nofork) {
    switch (fork()) {
    case 0:
      close(fileno(stdin));
      close(fileno(stdout));
      close(fileno(stderr));

#ifdef TIOCNOTTY
      if ((fd = open("/dev/tty")) != -1) {
	ioctl(fd, TIOCNOTTY, 0);
	close(fd);
      }
#else
      setpgrp();
#endif
      break;

    case -1:
#ifdef SYSLOG
      syslog(LOG_ERR, "fork: %m");
#endif
      perror("adjtimed: fork");
      exit(1);

    default:
      exit(0);
    } /* switch */
  } /* if */

  if (nofork) {
    setvbuf(stdout, NULL, _IONBF, BUFSIZ);
    setvbuf(stderr, NULL, _IONBF, BUFSIZ);
  }

#ifdef SYSLOG
  syslog(LOG_INFO, "started (rate %.2f%%)", RATE / 10000.0);
#endif
  if (verbose) printf("adjtimed: started (rate %.2f%%)\n", RATE / 10000.0);

  if (InitClockRate() == -1)
    Exit(2);

  (void)signal(SIGHUP, SIG_IGN);
  (void)signal(SIGINT, SIG_IGN);
  (void)signal(SIGQUIT, SIG_IGN);
  (void)signal(SIGTERM, Cleanup);

  vec.sv_handler = ResetClockRate;
  vec.sv_flags = 0;
  vec.sv_mask = ~0;
  sigvector(SIGALRM, &vec, (struct sigvec *)0);

  if (msgget(KEY, IPC_CREAT|IPC_EXCL) == -1) {
    if (errno == EEXIST) {
#ifdef SYSLOG
      syslog(LOG_ERR, "message queue already exists, use -r to remove it");
#endif
      fputs("adjtimed: message queue already exists, use -r to remove it\n",
		stderr);
      Exit(1);
    }

#ifdef SYSLOG
    syslog(LOG_ERR, "create message queue: %m");
#endif
    perror("adjtimed: create message queue");
    Exit(1);
  }

  if ((mqid = msgget(KEY, 0)) == -1) {
#ifdef SYSLOG
    syslog(LOG_ERR, "get message queue id: %m");
#endif
    perror("adjtimed: get message queue id");
    Exit(1);
  }

  for (;;) {
    if (msgrcv(mqid, &msg.msgp, MSGSIZE, CLIENT, 0) == -1) {
      if (errno == EINTR) continue;
#ifdef SYSLOG
      syslog(LOG_ERR, "read message: %m");
#endif
      perror("adjtimed: read message");
      Cleanup();
    }

    switch (msg.msgb.code) {
    case DELTA1:
    case DELTA2:
      AdjustClockRate(&msg.msgb.tv, &remains);

      if (msg.msgb.code == DELTA2) {
	msg.msgb.tv = remains;
	msg.msgb.mtype = SERVER;

	while (msgsnd(mqid, &msg.msgp, MSGSIZE, 0) == -1) {
	  if (errno == EINTR) continue;
#ifdef SYSLOG
	  syslog(LOG_ERR, "send message: %m");
#endif
	  perror("adjtimed: send message");
	  Cleanup();
	}
      }

      if (remains.tv_sec + remains.tv_usec != 0L) {
	if (verbose) {
	  printf("adjtimed: previous correction remaining %.6fs\n",
			tvtod(remains));
	}
#ifdef SYSLOG
	if (sysdebug) {
	  syslog(LOG_INFO, "previous correction remaining %.6fs",
			tvtod(remains));
	}
#endif
      }
      break;

    default:
      fprintf(stderr, "adjtimed: unknown message code %d\n", msg.msgb.code);
#ifdef SYSLOG
      syslog(LOG_ERR, "unknown message code %d", msg.msgb.code);
#endif
    } /* switch */
  } /* loop */
} /* main */

/*
 * Default clock rate (old_tick).
 */
#define DEFAULT_RATE	(MILLION / HZ)
#define UNKNOWN_RATE	0L
#define SLEW_RATE	(MILLION / DEFAULT_RATE)
#define MIN_DELTA	SLEW_RATE
/*
#define RATE		0.005
#define PERIOD		(1.0 / RATE)
*/
static long default_rate = DEFAULT_RATE;
static long slew_rate = SLEW_RATE;

AdjustClockRate(delta, olddelta)
     register struct timeval *delta, *olddelta;
{
  register long rate, dt;
  struct itimerval period, remains;
  static long leftover = 0;
/*
 * rate of change
 */
  dt = (delta->tv_sec * MILLION) + delta->tv_usec + leftover;

  if (dt < MIN_DELTA && dt > -MIN_DELTA) {
    leftover += delta->tv_usec;

    if (olddelta) {
      getitimer(ITIMER_REAL, &remains);
      dt = ((remains.it_value.tv_sec * MILLION) + remains.it_value.tv_usec) *
		oldrate;
      olddelta->tv_sec = dt / MILLION;
      olddelta->tv_usec = dt - (olddelta->tv_sec * MILLION); 
    }

    if (verbose > 2) printf("adjtimed: delta is too small: %dus\n", dt);
#ifdef SYSLOG
    if (sysdebug > 2) syslog(LOG_INFO, "delta is too small: %dus", dt);
#endif
    return (1);
  }

  leftover = dt % MIN_DELTA;
  dt -= leftover;

  if (verbose)
    printf("adjtimed: new correction %.6fs\n", (double)dt / (double)MILLION);
#ifdef SYSLOG
  if (sysdebug)
    syslog(LOG_INFO, "new correction %.6fs", (double)dt / (double)MILLION);
#endif
  if (verbose > 2) printf("adjtimed: leftover %dus\n", leftover);
#ifdef SYSLOG
  if (sysdebug > 2) syslog(LOG_INFO, "leftover %dus", leftover);
#endif

  rate = (dt < 0L ? -RATE : RATE);
  if (dt >= MILLION || dt <= -MILLION)
    rate *= 10;
  dt /= (rate / slew_rate); /* ticks needed */
  period.it_value.tv_sec = dt / slew_rate;
  period.it_value.tv_usec = (dt % slew_rate) * (MILLION / slew_rate);

  if (verbose > 1)
    printf("adjtimed: will be complete in %.6fs\n",
	   period.it_value.tv_sec + (double)period.it_value.tv_usec/MILLION);
#ifdef SYSLOG
  if (sysdebug > 1)
    syslog(LOG_INFO, "will be complete in %.6fs",
	   period.it_value.tv_sec + (double)period.it_value.tv_usec/MILLION);
#endif
/*
 * adjust the clock rate
 */
  if (SetClockRate((rate / slew_rate) + default_rate) == -1) {
#ifdef SYSLOG
    syslog(LOG_ERR, "set clock rate: %m");
#endif
    perror("adjtimed: set clock rate");
  }
/*
 * start the timer
 * (do this after changing the rate because the period has been rounded down)
 */
  period.it_interval.tv_sec = period.it_interval.tv_usec = 0L;
  setitimer(ITIMER_REAL, &period, &remains);
/*
 * return old delta
 */
  if (olddelta) {
    dt = ((remains.it_value.tv_sec * MILLION) + remains.it_value.tv_usec) *
		oldrate;
    olddelta->tv_sec = dt / MILLION;
    olddelta->tv_usec = dt - (olddelta->tv_sec * MILLION); 
  }

  oldrate = (double)rate / (double)MILLION;
} /* AdjustClockRate */

static struct nlist nl[] = {
#ifdef hp9000s800
#ifdef PRE7_0
  { "tick" },
#else
  { "old_tick" },
#endif
#else
  { "_old_tick" },
#endif
  { "" }
};

static int kmem;

/*
 * The return value is the clock rate in old_tick units or -1 if error.
 */
long
GetClockRate()
{
  long rate, mask;

  if (lseek(kmem, (long)nl[0].n_value, 0) == -1L)
    return (-1L);

  mask = sigblock(sigmask(SIGALRM));

  if (read(kmem, (caddr_t)&rate, sizeof(rate)) != sizeof(rate))
    rate = UNKNOWN_RATE;

  sigsetmask(mask);
  return (rate);
} /* GetClockRate */

/*
 * The argument is the new rate in old_tick units.
 */
SetClockRate(rate)
     long rate;
{
  long mask;

  if (lseek(kmem, (long)nl[0].n_value, 0) == -1L)
    return (-1);

  mask = sigblock(sigmask(SIGALRM));

  if (write(kmem, (caddr_t)&rate, sizeof(rate)) != sizeof(rate)) {
    sigsetmask(mask);
    return (-1);
  }

  sigsetmask(mask);

  if (rate != default_rate) {
    if (verbose > 3) {
      printf("adjtimed: clock rate (%lu) %ldus/s\n", rate,
		(rate - default_rate) * slew_rate);
    }
#ifdef SYSLOG
    if (sysdebug > 3) {
      syslog(LOG_INFO, "clock rate (%lu) %ldus/s", rate,
		(rate - default_rate) * slew_rate);
    }
#endif
  }

  return (0);
} /* SetClockRate */

InitClockRate()
{
  if ((kmem = open("/dev/kmem", O_RDWR)) == -1) {
#ifdef SYSLOG
    syslog(LOG_ERR, "open(/dev/kmem): %m");
#endif
    perror("adjtimed: open(/dev/kmem)");
    return (-1);
  }

  nlist("/hp-ux", nl);

  if (nl[0].n_type == 0) {
    fputs("adjtimed: /hp-ux has no symbol table\n", stderr);
#ifdef SYSLOG
    syslog(LOG_ERR, "/hp-ux has no symbol table");
#endif
    return (-1);
  }
/*
 * Set the default to the system's original value
 */
  default_rate = GetClockRate();
  if (default_rate == UNKNOWN_RATE) default_rate = DEFAULT_RATE;
  slew_rate = (MILLION / default_rate);

  return (0);
} /* InitClockRate */

/*
 * Reset the clock rate to the default value.
 */
void
ResetClockRate()
{
  struct itimerval it;

  it.it_value.tv_sec = it.it_value.tv_usec = 0L;
  setitimer(ITIMER_REAL, &it, (struct itimerval *)0);

  if (verbose > 2) puts("adjtimed: resetting the clock");
#ifdef SYSLOG
  if (sysdebug > 2) syslog(LOG_INFO, "resetting the clock");
#endif

  if (GetClockRate() != default_rate) {
    if (SetClockRate(default_rate) == -1) {
#ifdef SYSLOG
      syslog(LOG_ERR, "set clock rate: %m");
#endif
      perror("adjtimed: set clock rate");
    }
  }

  oldrate = 0.0;
} /* ResetClockRate */

void
Cleanup()
{
  ResetClockRate();

  if (msgctl(mqid, IPC_RMID, (struct msqid_ds *)0) == -1) {
    if (errno != EINVAL) {
#ifdef SYSLOG
      syslog(LOG_ERR, "remove message queue: %m");
#endif
      perror("adjtimed: remove message queue");
    }
  }

  Exit(2);
} /* Cleanup */

void
Exit(status)
     int status;
{
#ifdef SYSLOG
  syslog(LOG_ERR, "terminated");
  closelog();
#endif
  if (kmem != -1) close(kmem);
  exit(status);
} /* Exit */
