/*
 * refclock_wwvbhp - clock driver for the Spectracom NETCLOCK/2 WWVB receiver
 *    (NOTE: Very HP-UX specific !!)
 */
#if defined(REFCLOCK) && defined(WWVBHP)
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <termio.h>

#include "ntp_syslog.h"
#include "ntp_fp.h"
#include "ntp.h"
#include "ntp_refclock.h"
#include "ntp_unixtime.h"

#ifdef SDDDEBUG
extern int debug;
#endif
/*
 * This should support the use of a Spectracom NETCLOCK/2 WWVB
 * clock under HP-UX.  The Spectracom clock has an RS232 time-of-year 
 * output (the code assumes you don't have the year support option) 
 * and a TTL-level PPS signal.
 *
 * The RS232 output is quite simple.  You send the clock an ASCII
 * letter `T' and it returns the time in the following format:
 *
 * <cr><lf>I<sp><sp>DDD<sp>HH:MM:SS<sp><sp>TZ=nn<cr><lf>
 *
 * The `I' is actually a <sp> when the clock is synchronized, and
 * a `?' when it isn't (it could also be a `*' if we set the time
 * manually, but we don't do this).  The start bit of the first <cr>
 * is supposed to be synchronized to the on-time second.
 *
 * Unfortunately, the CPU in the NETCLOCK/2 which generates the time code
 * is very sloppy, and hence the serial code is inaccurate.  To get
 * real good time out of the box you must synchronize to the PPS
 * signal.  This is a 100 ms positive-going TTL-level pulse with the
 * rising edge corresponding to the on-time second.  
 * 
 * On HP-UX, we use the ntp_gpio driver with a modified gpio card
 * to timestamp the leading edge of the PPS signal.  A read of this
 * device yeilds a struct timeval of the timestamp.  It supports
 * select and SIGIO so it can fit into the general xntpd structure.
 * Timestamps should be accurate to within 100 us and we check the
 * differences accordingly.
 *
 * We time stamp all PPS inputs, and attempt to receive 64 of them
 * to make up our time sample.  If the sloppy clock flag is one we
 * simply total these and compute an average.  Otherwise we pick the
 * sample we think is most representative.  After the 16th and 48th
 * sample we send a T and read the time code, this providing us with
 * the seconds-and-above part of the time.
 */

/*
 * Definitions
 */
#define	MAXUNITS	4	/* maximum number of WWVB units permitted */
#define	WWVBHP232	"/dev/wwvbhp%d"
#define	WWVBHPPPS	"/dev/wwvbhppps%d"

/*
 * When WWVB is operating optimally we want the primary clock distance
 * to come out at 100 ms.  Thus, peer.distance in the WWVBHP peer structure
 * is set to 90 ms and we compute delays which are at least 10 ms long.
 * The following are 90 ms and 10 ms expressed in u_fp format
 */
#define	WWVBHPDISTANCE	0x0000170a
#define	WWVBHPBASEDELAY	0x0000028f

/*
 * Other constant stuff
 */
#define	WWVBHPPRECISION	(-10)		/* what the heck */
#define	WWVBHPREFID	"WWVB"
#define	WWVBHPHSREFID	0x7f7f050a	/* 127.127.5.10 refid for hi strata */

/*
 * Description of clock
 */
#define	WWVBHPDESCRIPTION		"Spectracom NETCLOCK/2 WWVB Receiver"

/*
 * Speeds we run the clock ports at.
 */
#define	SPEED232	B9600	/* 9600 baud */

/*
 * Constants for the serial port correction.  This assumes the
 * RS232 code is received at 9600 bps and the PPS code at 19200.
 * Since your average uart interrupts sort of at the center of the
 * first stop bit, these are 9.5 bit times, respectively, as l_fp
 * fractions.
 */
#define	UARTDELAY232	0x0040da74	/* about 0.990 ms */
#define	UARTDELAYPPS	0x00000002	/* about 30 us ... Eat your heart out Dennis :-) */

/*
 * The number of PPS raw samples which we acquire to derive
 * a single estimate.
 */
#define	NPPS	64

/*
 * Samples after which we ask for a time code on the RS232 port, and
 * the sample numbers after which we start to worry that it won't
 * arrive.
 */
#define	NCODES		2
#define	FIRSTCODE	16
#define	FIRSTWORRY	19
#define	SECONDCODE	48
#define	SECONDWORRY	51


/*
 * Length of the serial time code, in characters.  The second length
 * is less the carriage returns and line feeds.
 */
#define	LENWWVBHPCODE	26
#define	LENWWVBHPPRT	22

/*
 * Code state.
 */
#define	WWVBHPCS_NOCODE		0	/* not expecting anything */
#define	WWVBHPCS_SENTT		1	/* have sent a T, received nothing */
#define	WWVBHPCS_GOTCR		2	/* got the first <cr> in the time code */
#define	WWVBHPCS_GOTSOME	3	/* have an incomplete time code buffered */

/*
 * Default fudge factor and character to receive
 */
#define	DEFFUDGETIME	0		/* we have no particular preferences */
#define	DEFMAGICCHAR	('\r' | 0x80)	/* we use this for VEOL */

/*
 * Leap hold time.  After a leap second the clock will no longer be
 * reliable until it resynchronizes.  Hope 40 minutes is enough.
 */
#define	WWVBHPLEAPHOLD	(40 * 60)

/*
 * Hack to avoid excercising the multiplier.  I have no pride.
 */
#define	MULBY10(x)	(((x)<<3) + ((x)<<1))

/*
 * WWVBHP unit control structure.
 */
struct wwvbhpunit {
	struct peer *peer;		/* associated peer structure */
	struct refclockio io;		/* given to the I/O handler */
	struct refclockio ppsio;	/* ibid. */
	u_long samples[NPPS];		/* the PPS time samples */
	l_fp lastsampletime;		/* time of last estimate */
	l_fp codeoffsets[NCODES];	/* the time of arrival of 232 codes */
	l_fp codetime;			/* temporary for arrival times */
	char lastcode[LENWWVBHPCODE+6];	/* last time code we received */
	u_long lasttime;		/* last time clock heard from */
	u_char lencode;			/* length of code in buffer */
	u_char nsamples;		/* number of samples we've collected */
	u_char codestate;		/* state of 232 code reception */
	u_char unit;			/* unit number for this guy */
	u_char unsynccount;		/* count down of unsynchronized data */
	u_char status;			/* clock status */
	u_char lastevent;		/* last clock event */
	u_char reason;			/* reason for last abort */
	u_char hour;			/* hour of day */
	u_char minute;			/* minute of hour */
	u_char second;			/* seconds of minute */
	u_char tz;			/* timezone from clock */
	u_short day;			/* day of year from last code */
	u_long yearstart;		/* start of current year */
	u_long leaphold;		/* time of leap hold expiry */
	u_long polls;			/* number of RS232 polls */
	u_long noresponse;		/* number of nonresponses */
	u_long badformat;		/* number of bad format responses */
	u_long baddata;			/* number of invalid time codes */
	u_long bigcodeoff;		/* # of rejects from big code offset */
	u_long toosloppy;		/* too few remain after slop removal */
	u_long timestarted;		/* time we started this */
};

#define	WWVBHPUNSYNCCNT	3		/* need to see 3 synchronized samples */

/*
 * We demand that consecutive PPS samples are more than 0.995 seconds
 * and less than 1.005 seconds apart.
 */
#define	PPSLODIFF_UI	0		/* 0.990 as an l_fp */
#define	PPSLODIFF_UF	0xfd70a3d6

#define	PPSHIDIFF_UI	1		/* 1.010 as an l_fp */
#define	PPSHIDIFF_UF	0x028f5c28

/*
 * We also require that the code offsets are no more that 0.125 seconds
 * apart, and that none of the PPS sample offsets are more than
 * 0.125 seconds from our averaged code offset.
 */
#define	CODEDIFF	0x20000000	/* 0.125 seconds as an l_fp fraction */
#define	PPSMARGINLO	0x20000000	/* 0.125 seconds as an l_fp fraction */
#define	PPSMARGINHI	0xe0000000	/* 0.875 seconds as an l_fp fraction */

/*
 * Some other l_fp fraction values used by the code
 */
#define	ZEROPT75	0xc0000000
#define	ZEROPT5		0x80000000
#define	ZEROPT25	0x40000000


/*
 * Limits on things.  If we have less than SAMPLELIMIT samples after
 * gross filtering, reject the whole thing.  Otherwise reduce the number
 * of samples to SAMPLEREDUCE by median elimination.  If we're running
 * with an accurate clock, chose the BESTSAMPLE as the estimated offset,
 * otherwise average the remainder.
 */
#define	SAMPLELIMIT	56
#define	SAMPLEREDUCE	32
#define	REDUCESHIFT	5	/* SAMPLEREDUCE root 2 */
#define	BESTSAMPLE	24	/* Towards the high end of half */

/*
 * reason codes
 */
#define	PPSREASON	20
#define	CODEREASON	40
#define	PROCREASON	60

/*
 * Data space for the unit structures.  Note that we allocate these on
 * the fly, but never give them back.
 */
static struct wwvbhpunit *wwvbhpunits[MAXUNITS];
static u_char unitinuse[MAXUNITS];

/*
 * Keep the fudge factors separately so they can be set even
 * when no clock is configured.
 */
static l_fp fudgefactor[MAXUNITS];
static u_char stratumtouse[MAXUNITS];
static u_char magicchar[MAXUNITS];
static u_char sloppyclockflag[MAXUNITS];

/*
 * Imported from the timer module
 */
extern u_long current_time;
extern struct event timerqueue[];

/*
 * wwvbhp_reset - reset the count back to zero
 */
#define	wwvbhp_reset(wwvbhp) \
	do { \
		(wwvbhp)->nsamples = 0; \
		(wwvbhp)->codestate = WWVBHPCS_NOCODE; \
	} while (0)

/*
 * wwvbhp_event - record and report an event
 */
#define	wwvbhp_event(wwvbhp, evcode) \
	do { \
		if ((wwvbhp)->status != (u_char)(evcode)) \
			wwvbhp_report_event((wwvbhp), (evcode)); \
	} while (0)


/*
 * Time conversion tables imported from the library
 */
extern u_long ustotslo[];
extern u_long ustotsmid[];
extern u_long ustotshi[];


/*
 * wwvbhp_init - initialize internal wwvbhp driver data
 */
void
wwvbhp_init()
{
	register int i;
	/*
	 * Just zero the data arrays
	 */
	bzero((char *)wwvbhpunits, sizeof wwvbhpunits);
	bzero((char *)unitinuse, sizeof unitinuse);

	/*
	 * Initialize fudge factors to default.
	 */
	for (i = 0; i < MAXUNITS; i++) {
		fudgefactor[i].l_ui = 0;
		fudgefactor[i].l_uf = DEFFUDGETIME;
		magicchar[i] = DEFMAGICCHAR;
		stratumtouse[i] = 0;
		sloppyclockflag[i] = 0;
	}
}


/*
 * wwvbhp_start - open the WWVBHP devices and initialize data for processing
 */
int
wwvbhp_start(unit, peer)
	u_int unit;
	struct peer *peer;
{
	register struct wwvbhpunit *wwvbhp;
	register int i;
	int fd232;
	int fdpps;
	int ldisc;
	char wwvbhpdev[20];
	char wwvbhpppsdev[20];
	struct termio ttyb;
	void wwvbhp_receive();
	void wwvbhp_pps_receive();
	extern int io_addclock();
	extern void io_closeclock();
	extern char *emalloc();

	if (unit >= MAXUNITS) {
		syslog(LOG_ERR, "wwvbhp clock: unit number %d invalid (max 3)",
		    unit);
		return 0;
	}
	if (unitinuse[unit]) {
		syslog(LOG_ERR, "wwvbhp clock: unit number %d in use", unit);
		return 0;
	}

	/*
	 * Unit okay, attempt to open the devices.  We do them both at
	 * once to make sure we can
	 */
	(void) sprintf(wwvbhpdev, WWVBHP232, unit);
	(void) sprintf(wwvbhpppsdev, WWVBHPPPS, unit);

        /* NOTE: Open should be exclusive but what can I do ? */
	fd232 = open(wwvbhpdev, O_RDWR);
	if (fd232 == -1) {
		syslog(LOG_ERR, "wwvbhp clock: open of %s failed: %m", wwvbhpdev);
		return 0;
	}
        /* Using ntp_gpio driver, open is exclusive only */
	fdpps = open(wwvbhpppsdev, O_RDONLY);
	if (fdpps == -1) {
		syslog(LOG_ERR, "wwvbhp clock: open of %s failed: %m",
		    wwvbhpppsdev);
		(void) close(fd232);
		return 0;
	}

	/*
	 * Have to run the clock port cooked on HP-UX ...
	 */
	if (ioctl(fd232, TCGETA, (char *)&ttyb) < 0) {
                syslog(LOG_ERR, "wwvbhp clock: ioctl(%s, TCGETA): %m", wwvbhpdev);
                return 0;
        }
        ttyb.c_cflag = (SPEED232|CS8|CLOCAL|CREAD|CSTOPB);
        ttyb.c_iflag = IGNBRK;
        ttyb.c_oflag = 0;
        ttyb.c_lflag = 0;
        ttyb.c_cc[VMIN] = 2;
        ttyb.c_cc[VTIME] = 1;
        if (ioctl(fd232, TCSETA, (char *)&ttyb) < 0) {
                syslog(LOG_ERR, "wwvbhp clock: ioctl(%s, TCSETA): %m", wwvbhpdev);
                return 0;
        }

	/*
	 * Looks like this might succeed.  Find memory for the structure.
	 * Look to see if there are any unused ones, if not we malloc()
	 * one.
	 */
	if (wwvbhpunits[unit] != 0) {
		wwvbhp = wwvbhpunits[unit];	/* The one we want is okay */
	} else {
		for (i = 0; i < MAXUNITS; i++) {
			if (!unitinuse[i] && wwvbhpunits[i] != 0)
				break;
		}
		if (i < MAXUNITS) {
			/*
			 * Reclaim this one
			 */
			wwvbhp = wwvbhpunits[i];
			wwvbhpunits[i] = 0;
		} else {
			wwvbhp = (struct wwvbhpunit *)
			    emalloc(sizeof(struct wwvbhpunit));
		}
	}
	bzero((char *)wwvbhp, sizeof(struct wwvbhpunit));
	wwvbhpunits[unit] = wwvbhp;

	/*
	 * Set up the structures
	 */
	wwvbhp->peer = peer;
	wwvbhp->unit = (u_char)unit;
	wwvbhp->timestarted = current_time;

	wwvbhp->io.clock_recv = wwvbhp_receive;
	wwvbhp->io.srcclock = (caddr_t)wwvbhp;
	wwvbhp->io.datalen = 0;
	wwvbhp->io.fd = fd232;

	wwvbhp->ppsio.clock_recv = wwvbhp_pps_receive;
	wwvbhp->ppsio.srcclock = (caddr_t)wwvbhp;
	wwvbhp->ppsio.datalen = 0;
	wwvbhp->ppsio.fd = fdpps;

	/*
	 * Okay.  Flush the input buffer and give it to the I/O code to 
	 * start receiving stuff.  Nothing to do for pps so use special
	 * call to give it to the I/O.
	 */
	/* flush everything */
	if (ioctl(fd232, TCFLSH, 2) < 0) {
		syslog(LOG_ERR, "wwvbhp clock: ioctl(%s, TCFLSH): %m",
		    wwvbhpdev);
		goto screwed;
	}
	if (!io_addclock(&wwvbhp->io)) {
		/*
		 * Oh shit.  Just close and return.
		 */
		goto screwed;
	}
	if (!io_addclock_simple(&wwvbhp->ppsio)) {
		/*
		 * Remove the other one and return
		 */
		io_closeclock(&wwvbhp->io);
		(void) close(fdpps);
		return 0;
	}

	/*
	 * All done.  Initialize a few random peer variables, then
	 * return success.
	 */
	peer->distance = WWVBHPDISTANCE;
	peer->precision = WWVBHPPRECISION;
	peer->stratum = stratumtouse[unit];
	if (stratumtouse[unit] <= 1)
		bcopy(WWVBHPREFID, (char *)&peer->refid, 4);
	else
		peer->refid = htonl(WWVBHPHSREFID);
	unitinuse[unit] = 1;
	return 1;

screwed:
	(void) close(fd232);
	(void) close(fdpps);
	return 0;
}


/*
 * wwvbhp_shutdown - shut down a WWVBHP clock
 */
void
wwvbhp_shutdown(unit)
	int unit;
{
	register struct wwvbhpunit *wwvbhp;
	extern void io_closeclock();

	if (unit >= MAXUNITS) {
		syslog(LOG_ERR,
		  "wwvbhp clock: INTERNAL ERROR, unit number %d invalid (max 4)",
		    unit);
		return;
	}
	if (!unitinuse[unit]) {
		syslog(LOG_ERR,
		 "wwvbhp clock: INTERNAL ERROR, unit number %d not in use", unit);
		return;
	}

	/*
	 * Tell the I/O module to turn us off.  We're history.
	 */
	wwvbhp = wwvbhpunits[unit];
	io_closeclock(&wwvbhp->ppsio);
	io_closeclock(&wwvbhp->io);
	unitinuse[unit] = 0;
}


/*
 * wwvbhp_report_event - note the occurance of an event
 */
wwvbhp_report_event(wwvbhp, code)
	struct wwvbhpunit *wwvbhp;
	int code;
{
	if (wwvbhp->status != (u_char)code) {
		wwvbhp->status = (u_char)code;
		if (code != CEVNT_NOMINAL)
			wwvbhp->lastevent = (u_char)code;
		/*
		 * Should report event to trap handler in here.
		 * Soon...
		 */
	}
}


/*
 * wwvbhp_receive - receive data from the serial interface on a Spectracom clock
 */
void
wwvbhp_receive(rbufp)
	struct recvbuf *rbufp;
{
	register int i;
	register struct wwvbhpunit *wwvbhp;
	register u_char *dpt;
	register char *cp;
	register u_char *dpend;

#ifdef SDDDEBUG
	if (debug)
	    printf("wwvbhp_receive: ENTER\n");
#endif
	/*
	 * Get the clock this applies to and a pointer to the data
	 */
	wwvbhp = (struct wwvbhpunit *)rbufp->recv_srcclock;
	dpt = (u_char *)&rbufp->recv_space;
	dpend = dpt + rbufp->recv_length;

	/*
	 * Check out our state and process appropriately
	 */
	switch (wwvbhp->codestate) {
	case WWVBHPCS_NOCODE:
		/*
		 * Not expecting anything.  Just return.
		 */
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: NOCODE\n");
#endif
		return;

	case WWVBHPCS_SENTT:
		/*
		 * Here we space past anything up to the first nl.
		 * Anything past this is considered to possibly be
		 * part of the time code itself.  The timestamp is
		 * that recorded by the I/O module when the data was
		 * input.
		 */
		while (dpt < dpend && (*dpt & 0x7f) != '\n')
			dpt++;
		if (dpt == dpend) {
			/*
			 * What the heck?  Forget about this one.
			 */
			syslog(LOG_ERR, "wwvbhp clock: data returned without a trailing newline");
			wwvbhp->reason = CODEREASON + 3;
			wwvbhp_event(wwvbhp, CEVNT_FAULT);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_receive: SENTT missing newline\n");
#endif
		}
		dpt++;

		wwvbhp->codetime = rbufp->recv_time;

		/*
		 * Here we have our timestamp.  Change the state to
		 * reflect this.
		 */
		wwvbhp->codestate = WWVBHPCS_GOTCR;
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: SENTT -> GOTCR\n");
#endif
		if (dpt == dpend)
			return;
		
		/*FALLSTHROUGH*/
	case WWVBHPCS_GOTCR:
		/*
		 * Here we have the timestamp, but have yet to
		 * receive any of the time code.  Delete leading
		 * \n's.
		 */
		while (dpt < dpend && (*dpt & 0x7f) == '\n')
			dpt++;
		if (dpt == dpend)
			return;		/* no state transition */
		if (!isprint(*dpt & 0x7f)) {
			wwvbhp->badformat++;
			wwvbhp->reason = CODEREASON + 4;
			wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_receive: GOTCR bad format\n");
#endif
			return;
		}

		wwvbhp->codestate = WWVBHPCS_GOTSOME;
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: GOTCR -> GOTSOME\n");
#endif
		wwvbhp->lencode = 0;

		/*FALLSTHROUGH*/
	case WWVBHPCS_GOTSOME:
		cp = &(wwvbhp->lastcode[wwvbhp->lencode]);
		while (dpt < dpend && isprint(*dpt & 0x7f)) {
			*cp++ = (char)(*dpt++ & 0x7f);
			wwvbhp->lencode++;
			if (wwvbhp->lencode > LENWWVBHPCODE) {
				wwvbhp->badformat++;
				wwvbhp->reason = CODEREASON + 5;
				wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
				wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
				if (debug)
				    printf("wwvbhp_receive: GOTSOME bad format\n");
#endif
				return;
			}
		}
		if (dpt == dpend) {
			/*
			 * Incomplete.  Wait for more.
			 */
			return;
		}
		if ((*dpt & 0x7f) != '\r') {
			wwvbhp->badformat++;
			wwvbhp->reason = CODEREASON + 6;
			wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_receive: GOTSOME missing newline (%o)\n",
				   (*dpt & 0x7f));
#endif
			return;
		}

		/*
		 * Finally, got a complete buffer.  Mainline code will
		 * continue on.
		 */
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BINGO !!!\n");
#endif
		break;
	
	default:
		syslog(LOG_ERR, "wwvbhp clock: INTERNAL ERROR: state %d",
		    wwvbhp->codestate);
		wwvbhp->reason = CODEREASON + 7;
		wwvbhp_event(wwvbhp, CEVNT_FAULT);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BAD STATE\n");
#endif
		return;
	}

	/*
	 * Boy!  After all that crap, the lastcode buffer now contains
	 * something we hope will be a valid time code.  Do length
	 * checks and sanity checks on constant data.
	 */
	wwvbhp->codestate = WWVBHPCS_NOCODE;
	if (wwvbhp->lencode != LENWWVBHPPRT) {
		wwvbhp->badformat++;
		wwvbhp->reason = CODEREASON + 8;
		wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BAD lencode (%d != %d)\n",
			   wwvbhp->lencode, LENWWVBHPPRT);
#endif
		return;
	}

	cp = wwvbhp->lastcode;
	if (cp[ 1] != ' ' || cp[ 2] != ' ' ||
	    cp[ 6] != ' ' || cp[ 9] != ':' ||
	    cp[12] != ':' || cp[15] != ' ' ||
	    cp[16] != ' ' || cp[17] != 'T' ||
	    cp[18] != 'Z' || cp[19] != '=') {
		wwvbhp->badformat++;
		wwvbhp->reason = CODEREASON + 9;
		wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BAD1 format in buf [%s]\n", cp);
#endif
		return;
	}

	/*
	 * Check sync character and numeric data
	 */
	if ((*cp != ' ' && *cp != '*' && *cp != '?') ||
	    !isdigit(cp[ 3]) || !isdigit(cp[ 4]) ||
	    !isdigit(cp[ 5]) || !isdigit(cp[ 7]) ||
	    !isdigit(cp[ 8]) || !isdigit(cp[10]) ||
	    !isdigit(cp[11]) || !isdigit(cp[13]) ||
	    !isdigit(cp[14]) || !isdigit(cp[20]) ||
	    !isdigit(cp[21])) {
		wwvbhp->badformat++;
		wwvbhp->reason = CODEREASON + 10;
		wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BAD2 format in buf [%s]\n", cp);
#endif
		return;
	}

	/*
	 * Do some finer range checks on the raw data
	 */
	if (cp[ 3] > '3' || cp[ 7] > '2' ||
	    cp[10] > '5' || cp[13] > '5') {
		wwvbhp->badformat++;
		wwvbhp->reason = CODEREASON + 11;
		wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: BAD3 format in buf [%s]\n", cp);
#endif
		return;
	}


#ifdef SDDDEBUG
	if (debug)
	    printf("wwvbhp_receive: Buffer [%s]\n", cp);
#endif
	/*
	 * So far, so good.  Compute days, hours, minutes, seconds,
	 * time zone.  Do range checks on these.
	 */
	wwvbhp->day = cp[3] - '0';
	wwvbhp->day = MULBY10(wwvbhp->day) + cp[4] - '0';
	wwvbhp->day = MULBY10(wwvbhp->day) + cp[5] - '0';

	wwvbhp->hour = MULBY10(cp[7] - '0') + cp[8] - '0';
	wwvbhp->minute = MULBY10(cp[10] - '0') + cp[11] -  '0';
	wwvbhp->second = MULBY10(cp[13] - '0') + cp[14] - '0';
	wwvbhp->tz = MULBY10(cp[20] - '0') + cp[21] - '0';

	if (wwvbhp->day > 366 || wwvbhp->tz > 12) {
		wwvbhp->baddata++;
		wwvbhp->reason = CODEREASON + 12;
		wwvbhp_event(wwvbhp, CEVNT_BADDATE);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: Bad date 1\n");
#endif
		return;
	}

	if (wwvbhp->hour > 23 || wwvbhp->minute > 59 || wwvbhp->second > 59) {
		wwvbhp->baddata++;
		wwvbhp->reason = CODEREASON + 13;
		wwvbhp_event(wwvbhp, CEVNT_BADDATE);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: Bad date 2\n");
#endif
		return;
	}

	/*
	 * Figure out whether we're computing the early time code or
	 * the late one.
	 */
	if (wwvbhp->nsamples < SECONDCODE)
		i = 0;
	else
		i = 1;

	/*
	 * Now, compute the reference time value.
	 */
	if (!clocktime(wwvbhp->day, wwvbhp->hour, wwvbhp->minute, wwvbhp->second,
	    wwvbhp->tz, rbufp->recv_time.l_ui, &wwvbhp->yearstart,
	    &wwvbhp->codeoffsets[i].l_ui)) {
		wwvbhp->baddata++;
		wwvbhp->reason = CODEREASON + 14;
		wwvbhp_event(wwvbhp, CEVNT_BADDATE);
		wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_receive: Bad date 3 (%d/%d/%d/%d/%d)\n",
			   wwvbhp->day, wwvbhp->hour, wwvbhp->minute,
			   wwvbhp->second, wwvbhp->tz);
#endif
		return;
	}

	/*
	 * Check for synchronization
	 */
	if (*cp == ' ') {
		if (wwvbhp->unsynccount > 0)
			wwvbhp->unsynccount--;
	} else {
		wwvbhp->unsynccount = WWVBHPUNSYNCCNT;
	}


	/*
	 * Compute the offset.  For the fractional part of the
	 * offset we use the expected delay for a character at the
	 * baud rate.
	 */
	wwvbhp->codeoffsets[i].l_uf = UARTDELAY232;
	L_SUB(&wwvbhp->codeoffsets[i], &wwvbhp->codetime);

	/*
	 * Done!
	 */
#ifdef SDDDEBUG
	if (debug)
	    printf("wwvbhp_receive: WOW !!!\n");
#endif
}


/*
 * wwvbhp_pps_receive - receive pps data
 */
void
wwvbhp_pps_receive(rbufp)
	struct recvbuf *rbufp;
{
	register struct wwvbhpunit *wwvbhp;
	register u_char *dpt;
	register u_long tmp_ui;
	register u_long tmp_uf;
	l_fp ts;
	void wwvbhp_process();
#ifdef SDDDEBUG
	extern char *lfptoa();
#endif

	/*
	 * Get the clock this applies to and a pointer to the data
	 */
	wwvbhp = (struct wwvbhpunit *)rbufp->recv_srcclock;
	dpt = (u_char *)&rbufp->recv_space;

#ifdef SDDDEBUG
	if (debug)
	    printf("wwvbhp_pps_receive: nsamples = %d\n", wwvbhp->nsamples);
#endif

	/*
	 * Record the time of this event
	 */
	wwvbhp->lasttime = current_time;

	/*
	 * Check the length of the buffer.  Start over if it isn't as
	 * expected.
	 */
	if (rbufp->recv_length != sizeof(struct timeval)) {
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_pps_receive: length wrong %d != %d\n",
			   rbufp->recv_length, sizeof(struct timeval));
#endif
		wwvbhp->reason = PPSREASON + 1;
		wwvbhp_event(wwvbhp, CEVNT_FAULT);
		wwvbhp_reset(wwvbhp);
		return;
	}

	if (!buftvtots(dpt, &ts)) {
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_pps_receive: buftvtots failed\n");
#endif
		wwvbhp->reason = PPSREASON + 2;
		wwvbhp_event(wwvbhp, CEVNT_FAULT);
		wwvbhp_reset(wwvbhp);
		return;
	}
	tmp_ui = ts.l_ui;
	tmp_uf = ts.l_uf;

	/*
	 * Got a timestamp.  Subtract the character delay offset
	 * from it.
	 */
	M_SUBUF(tmp_ui, tmp_uf, UARTDELAYPPS);

	/*
	 * Make sure this character arrived approximately 1 second
	 * after the previous one.  If not, start over.
	 */
	L_NEG(&wwvbhp->lastsampletime);
	M_ADD(wwvbhp->lastsampletime.l_ui, wwvbhp->lastsampletime.l_uf,
	    tmp_ui, tmp_uf);
	if ( wwvbhp->lastsampletime.l_ui > PPSHIDIFF_UI ||
	    (wwvbhp->lastsampletime.l_ui == PPSHIDIFF_UI &&
	     wwvbhp->lastsampletime.l_uf > PPSHIDIFF_UF) ||
	    (wwvbhp->lastsampletime.l_ui == PPSLODIFF_UI &&
	     wwvbhp->lastsampletime.l_uf < PPSLODIFF_UF)) {
#ifdef SDDDEBUG
		if (debug)
		    printf("wwvbhp_pps_receive: not 1 second apart (%s)\n",
			   lfptoa(&(wwvbhp->lastsampletime), 9));
#endif
		wwvbhp->reason = PPSREASON + 3;
		wwvbhp_event(wwvbhp, CEVNT_TIMEOUT);
		wwvbhp_reset(wwvbhp);
	}

	/*
	 * Save the current time.
	 */
	wwvbhp->lastsampletime.l_ui = tmp_ui;
	wwvbhp->lastsampletime.l_uf = tmp_uf;

	/*
	 * Also save the subsecond part of the time in the samples
	 * array, to be diddled with later.  Save the negative of
	 * this, since this is the offset.
	 */
	wwvbhp->samples[wwvbhp->nsamples++] = -tmp_uf;

	/*
	 * We time the acquisition of RS232 time code samples, and
	 * the detection of a failure, against the sample number.
	 * Also, if we've hit NPPS samples we process the lot.
	 */
	switch (wwvbhp->nsamples) {
	case FIRSTCODE:
	case SECONDCODE:
		/*
		 * Time to request a time code.  The state of the
		 * code receiver should be NOCODE.
		 */
		if (wwvbhp->codestate != WWVBHPCS_NOCODE) {
			syslog(LOG_ERR,
			    "wwvbhp clock: code state %d in wwvbhp_pps_receive()",
			    wwvbhp->codestate);
			wwvbhp->reason = PPSREASON + 4;
			wwvbhp_event(wwvbhp, CEVNT_FAULT);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_pps_receive: code state wrong (1)\n");
#endif
		} else if (write(wwvbhp->io.fd, "T", 1) != 1) {
			syslog(LOG_ERR,
			    "wwvbhp clock: write fails to unit %d: %m",
			    wwvbhp->unit);
			wwvbhp->reason = PPSREASON + 5;
			wwvbhp_event(wwvbhp, CEVNT_FAULT);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_pps_receive: T write failed\n");
#endif
		} else {
			syslog(LOG_ERR, "wwvbhp clock: clock polled");
			wwvbhp->codestate = WWVBHPCS_SENTT;
			wwvbhp->polls++;
		}
		break;

	case FIRSTWORRY:
	case SECONDWORRY:
		/*
		 * By now the code should have been entirely received
		 * and the state set back to NOCODE.  If this is not
		 * not the case, count it as a non-response and start
		 * over.
		 */
		if (wwvbhp->codestate != WWVBHPCS_NOCODE) {
			wwvbhp->noresponse++;
			wwvbhp->reason = PPSREASON + 6;
			wwvbhp_event(wwvbhp, CEVNT_TIMEOUT);
			wwvbhp_reset(wwvbhp);
#ifdef SDDDEBUG
			if (debug)
			    printf("wwvbhp_pps_receive: code state wrong (2)\n");
#endif
		}
		break;
	
	case NPPS:
		/*
		 * Here we've managed to complete an entire 64 second
		 * cycle without major mishap.  Process what has been
		 * received.
		 */
		wwvbhp_process(wwvbhp);
		break;
	}
}



/*
 * wwvbhp_process - process a pile of samples from the clock
 */
void
wwvbhp_process(wwvbhp)
	struct wwvbhpunit *wwvbhp;
{
	register int i, j;
	register u_long tmp_ui, tmp_uf;
	register int noff;
	l_fp offset;
	u_fp delay;
	u_long off[NPPS];
	int addpt5;
	int isinsync;
	extern void refclock_receive();

	/*
	 * Reset things to zero so we don't have to worry later
	 */
	wwvbhp_reset(wwvbhp);

	/*
	 * First deal with the two offsets received via the
	 * RS232 code.  If they aren't within 100 ms of each
	 * other, all bets are off.
	 */
	tmp_ui = wwvbhp->codeoffsets[0].l_ui;
	tmp_uf = wwvbhp->codeoffsets[0].l_uf;
	M_SUB(tmp_ui, tmp_uf, wwvbhp->codeoffsets[1].l_ui,
	    wwvbhp->codeoffsets[1].l_uf);
	i = 1;
	if (M_ISNEG(tmp_ui, tmp_uf)) {
		i = 0;
		M_NEG(tmp_ui, tmp_uf);
	}
	if (tmp_ui != 0 || tmp_uf > CODEDIFF) {
		wwvbhp->bigcodeoff++;
		wwvbhp->reason = PROCREASON + 1;
		wwvbhp_event(wwvbhp, CEVNT_BADREPLY);
		return;
	}

	/*
	 * When you weren't looking, the above code also computed the
	 * minimum of the two (in i), which is what we use from here on.
	 * Note that we only saved the subsecond part of the PPS
	 * data, which will be combined with the seconds part of
	 * the code offsets.  To avoid ambiguity when the subsecond
	 * part of the offset is close to zero, we add 0.5 to everything
	 * when the code offset subseconds are > 0.75 or < 0.25, and
	 * subtract it off at the end.
	 */
	offset.l_ui = wwvbhp->codeoffsets[i].l_ui;
	tmp_uf = wwvbhp->codeoffsets[i].l_uf;
	addpt5 = 0;
	if (tmp_uf > ZEROPT75 || tmp_uf < ZEROPT25) {
		addpt5 = 1;
		M_ADDUF(offset.l_ui, tmp_uf, ZEROPT5);
	}

	/*
	 * Now run through the PPS samples, adding 0.5 if necessary,
	 * and rejecting those that differ from the code time stamp
	 * by more than 0.125.  The rest get stuffed into the offset
	 * array, sorted in ascending order.  The sort is really gross,
	 * I should do this better.
	 */
	noff = 0;
	for (i = 0; i < NPPS; i++) {
		tmp_ui = wwvbhp->samples[i];
		if (addpt5)
			tmp_ui += ZEROPT5;
		if ((tmp_ui - tmp_uf) > PPSMARGINLO &&
		    (tmp_ui - tmp_uf) < PPSMARGINHI) {
			/*
			 * belch!  This one no good.
			 */
			continue;
		}

		if (noff == 0 || off[noff-1] < tmp_ui) {
			/*
			 * Easy, goes in at the end.
			 */
			off[noff++] = tmp_ui;
		} else {
			/*
			 * Start at the end, shifting everything up
			 * until we find the proper slot.
			 */
			j = noff;
			do {
				j--;
				off[j+1] = off[j];
			} while (j > 0 && off[j-1] > tmp_ui);
			off[j] = tmp_ui;
			noff++;
		}
	}

	/*
	 * If we have less than the prescribed limit, forget this.
	 */
	if (noff < SAMPLELIMIT) {
		wwvbhp->toosloppy++;
		wwvbhp->reason = PROCREASON + 2;
		wwvbhp_event(wwvbhp, CEVNT_BADDATE);
		return;
	}

	/*
	 * Now reject the end of the list furthest from the median
	 * until we get to SAMPLEREDUCE samples remaining.
	 */
	noff = NPPS;
	i = 0;
	while ((noff - i) > SAMPLEREDUCE) {
		if ((off[noff-1] - off[(noff + i)/2]) >
		    (off[(noff + i)/2] - off[i])) {
			noff--;
		} else {
			i++;
		}
	}

	/*
	 * What we do next depends on the setting of the
	 * sloppy clock flag.  If it is on, average the remainder
	 * to derive our estimate.  Otherwise, just pick a
	 * representative value from the remaining stuff.
	 */
	if (sloppyclockflag[wwvbhp->unit]) {
		tmp_ui = tmp_uf = 0;
		for (j = i; j < noff; j++)
			M_ADDUF(tmp_ui, tmp_uf, off[j]);
		for (j = REDUCESHIFT; j > 0; j--)
			M_RSHIFTU(tmp_ui, tmp_uf);
	} else {
		tmp_uf = off[i+BESTSAMPLE];
	}
	
	/*
	 * The subsecond part of the offset is now in tmp_uf.
	 * Subtract the 0.5 seconds if we added it, add in
	 * the fudge factor, then store this away.
	 */
	if (addpt5)
		M_SUBUF(offset.l_ui, tmp_uf, ZEROPT5);
	M_ADD(offset.l_ui, tmp_uf, fudgefactor[wwvbhp->unit].l_ui,
	    fudgefactor[wwvbhp->unit].l_uf);
	offset.l_uf = tmp_uf;

	/*
	 * Compute the delay as the difference between the
	 * lowest and highest offsets that remain in the
	 * consideration list.
	 */
	delay = MFPTOFP(0, off[noff-1] - off[i]) + WWVBHPBASEDELAY;

	/*
	 * Determine synchronization status.  Can be unsync'd either
	 * by a report from the clock or by a leap hold.
	 */
	if (wwvbhp->unsynccount != 0 || wwvbhp->leaphold > current_time)
		isinsync = 0;
	else
		isinsync = 1;

	/*
	 * Done.  Use codetime as the reference time and lastsampletime
	 * as the receive time.
	 */
	refclock_receive(wwvbhp->peer, offset, delay, &wwvbhp->codetime,
	    &wwvbhp->lastsampletime, isinsync);

	if (wwvbhp->unsynccount != 0)
		wwvbhp_event(wwvbhp, CEVNT_PROP);
	else
		wwvbhp_event(wwvbhp, CEVNT_NOMINAL);
}


/*
 * wwvbhp_poll - called by the transmit procedure
 */
void
wwvbhp_poll(unit, peer)
	int unit;
	char *peer;
{
	if (unit >= MAXUNITS) {
		syslog(LOG_ERR, "wwvbhp clock poll: INTERNAL: unit %d invalid",
		    unit);
		return;
	}
	if (!unitinuse[unit]) {
		syslog(LOG_ERR, "wwvbhp clock poll: INTERNAL: unit %d unused",
		    unit);
		return;
	}

	if ((current_time - wwvbhpunits[unit]->lasttime) > 150)
		wwvbhp_event(wwvbhpunits[unit], CEVNT_FAULT);
}


/*
 * wwvbhp_leap - called when a leap second occurs
 */
void
wwvbhp_leap()
{
	register int i;

	/*
	 * This routine should be entered a few seconds after
	 * midnight UTC when a leap second occurs.  To ensure we
	 * don't believe foolish time from the clock(s) we set a
	 * 40 minute hold on them.  It shouldn't take anywhere
	 * near this amount of time to adjust if the clock is getting
	 * data, but doing anything else is complicated.
	 */
	for (i = 0; i < MAXUNITS; i++) {
		if (unitinuse[i])
			wwvbhpunits[i]->leaphold = current_time + WWVBHPLEAPHOLD;
	}
}


/*
 * wwvbhp_control - set fudge factors, return statistics
 */
void
wwvbhp_control(unit, in, out)
	u_int unit;
	struct refclockstat *in;
	struct refclockstat *out;
{
	register struct wwvbhpunit *wwvbhp;

	if (unit >= MAXUNITS) {
		syslog(LOG_ERR, "wwvbhp clock: unit %d invalid (max %d)",
		    unit, MAXUNITS-1);
		return;
	}

	if (in != 0) {
		if (in->haveflags & CLK_HAVETIME1)
			fudgefactor[unit] = in->fudgetime1;
		if (in->haveflags & CLK_HAVEVAL1) {
			stratumtouse[unit] = (u_char)(in->fudgeval1 & 0xf);
			if (unitinuse[unit]) {
				struct peer *peer;

				/*
				 * Should actually reselect clock, but
				 * will wait for the next timecode
				 */
				wwvbhp = wwvbhpunits[unit];
				peer = wwvbhp->peer;
				peer->stratum = stratumtouse[unit];
				if (stratumtouse[unit] <= 1)
					bcopy(WWVBHPREFID, (char *)&peer->refid,
					    4);
				else
					peer->refid = htonl(WWVBHPHSREFID);
			}
		}
		if (in->haveflags & CLK_HAVEFLAG1) {
			sloppyclockflag[unit] = in->flags & CLK_FLAG1;
		}
		if (in->haveflags & CLK_HAVEFLAG2) {
			if (in->flags & CLK_FLAG2 && unitinuse[unit])
				wwvbhpunits[unit]->leaphold = 0;
		}
		if (in->haveflags & CLK_HAVEVAL2 && unitinuse[unit]) {
			magicchar[unit] = (u_char)(in->fudgeval2 & 0xff);
		}
	}

	if (out != 0) {
		out->type = REFCLK_WWVB_SPECTRACOM_HP;
		out->haveflags
		    = CLK_HAVETIME1|CLK_HAVEVAL1|CLK_HAVEVAL2|CLK_HAVEFLAG1;
		out->clockdesc = WWVBHPDESCRIPTION;
		out->fudgetime1 = fudgefactor[unit];
		out->fudgetime2.l_ui = 0;
		out->fudgetime2.l_uf = 0;
		out->fudgeval1 = (long)stratumtouse[unit];
		out->fudgeval2 = (long)magicchar[unit];
		out->flags = sloppyclockflag[unit];
		if (unitinuse[unit]) {
			wwvbhp = wwvbhpunits[unit];
			out->lencode = wwvbhp->lencode;
			out->lastcode = wwvbhp->lastcode;
			out->timereset = current_time - wwvbhp->timestarted;
			out->polls = wwvbhp->polls;
			out->noresponse = wwvbhp->noresponse;
			out->badformat = wwvbhp->badformat;
			out->baddata = wwvbhp->baddata;
			out->lastevent = wwvbhp->lastevent;
			out->currentstatus = wwvbhp->status;
		} else {
			out->lencode = 0;
			out->lastcode = "";
			out->polls = out->noresponse = 0;
			out->badformat = out->baddata = 0;
			out->timereset = 0;
			out->currentstatus = out->lastevent = CEVNT_NOMINAL;
		}
	}
}


/*
 * wwvbhp_buginfo - return clock dependent debugging info
 */
void
wwvbhp_buginfo(unit, bug)
	int unit;
	register struct refclockbug *bug;
{
	register struct wwvbhpunit *wwvbhp;
	register int i;
	register int n;

	bug->nvalues = bug->ntimes = 0;

	if (unit >= MAXUNITS) {
		syslog(LOG_ERR, "wwvbhp clock: unit %d invalid (max %d)",
		    unit, MAXUNITS-1);
		return;
	}

	if (!unitinuse[unit])
		return;
	wwvbhp = wwvbhpunits[unit];

	bug->nvalues = 14;
	bug->svalues = 0;
	if (wwvbhp->lasttime != 0)
		bug->values[0] = current_time - wwvbhp->lasttime;
	else
		bug->values[0] = 0;
	bug->values[1] = (u_long)wwvbhp->reason;
	bug->values[2] = (u_long)wwvbhp->nsamples;
	bug->values[3] = (u_long)wwvbhp->codestate;
	bug->values[4] = (u_long)wwvbhp->unsynccount;
	bug->values[5] = (u_long)wwvbhp->day;
	bug->values[6] = (u_long)wwvbhp->hour;
	bug->values[7] = (u_long)wwvbhp->minute;
	bug->values[8] = (u_long)wwvbhp->second;
	bug->values[9] = (u_long)wwvbhp->tz;
	bug->values[10] = wwvbhp->bigcodeoff;
	bug->values[11] = wwvbhp->toosloppy;
	bug->values[12] = wwvbhp->yearstart;
	if (wwvbhp->leaphold > current_time)
		bug->values[13] = wwvbhp->leaphold - current_time;
	else
		bug->values[13] = 0;

	bug->stimes = 0xc;
	bug->times[0] = wwvbhp->codetime;
	bug->times[1] = wwvbhp->lastsampletime;
	bug->times[2] = wwvbhp->codeoffsets[0];
	bug->times[3] = wwvbhp->codeoffsets[1];
	n = wwvbhp->nsamples - (NCLKBUGTIMES-4);
	if (n < 0)
		n = 0;
	i = 4;
	bug->ntimes += wwvbhp->nsamples - n + 4;
	while (n < wwvbhp->nsamples) {
		bug->stimes |= (1<<i);
		if (wwvbhp->samples[n] & 0x80000000)
			bug->times[i].l_i = -1;
		else
			bug->times[i].l_i = 0;
		bug->times[i++].l_uf = wwvbhp->samples[n++];
	}
}
#endif
