##############################################################################
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
#  Jabber
#  Copyright (C) 1998-1999 The Jabber Team http://jabber.org/
#
##############################################################################

package Jarl::Profile;

use strict;
use vars qw($VERSION);

use Jarl::Profile::Connection;
use Jarl::Profile::FavGroup;

$VERSION = "0.1";

sub new {
  my $proto = shift;
  my $self = { };

  $self->{PROFILE} = {};

  bless($self, $proto);

  if ($#_ > -1) {
    my ($file) = @_;

    $self->ParseCfg($file);
  }

  return $self;
}


sub ConnectionObj {
  my $self = shift;
  my ($name,$connection) = @_;
  $self->{PROFILE}->{connections}->{$name} = $connection if defined($connection);
  $self->Save();
  return $self->{PROFILE}->{connections}->{$name} if defined($name);
  return keys(%{$self->{PROFILE}->{connections}});
}


sub ConnectionObjClear {
  my $self = shift;
  $self->{PROFILE}->{connections} = { };
}


sub FavGroupObj {
  my $self = shift;
  my ($name,$favgroup) = @_;
  $self->{PROFILE}->{favgroups}->{$name} = $favgroup if defined($favgroup);
  return $self->{PROFILE}->{favgroups}->{$name} if defined($name);
  return keys(%{$self->{PROFILE}->{favgroups}});
}


sub FavGroupObjClear {
  my $self = shift;
  $self->{PROFILE}->{favgroups} = { };
}


sub Port {
  my $self = shift;
  my ($port) = @_;
  $self->{PROFILE}->{port} = $port if defined($port);
  return $self->{PROFILE}->{port};
}

sub Type {
  my $self = shift;
  my ($type) = @_;
  $self->{PROFILE}->{type} = $type if defined($type);
  return $self->{PROFILE}->{type};
}

sub Name {
  my $self = shift;
  my ($name) = @_;
  $self->{PROFILE}->{name} = $name if defined($name);
  return $self->{PROFILE}->{name};
}

sub Username {
  my $self = shift;
  my ($username) = @_;
  $self->{PROFILE}->{username} = $username if defined($username);
  return $self->{PROFILE}->{username};
}

sub Password {
  my $self = shift;
  my ($password) = @_;
  $self->{PROFILE}->{password} = $password if defined($password);
  return $self->{PROFILE}->{password};
}

sub Resource {
  my $self = shift;
  my ($resource) = @_;
  $self->{PROFILE}->{resource} = $resource if defined($resource);
  return $self->{PROFILE}->{resource};
}

sub Server {
  my $self = shift;
  my ($server) = @_;
  $self->{PROFILE}->{server} = $server if defined($server);
  return $self->{PROFILE}->{server};
}

sub ConnectionProfile {
  my $self = shift;
  my ($connectionprofile) = @_;
  $self->{PROFILE}->{connectionprofile} = $connectionprofile 
    if defined($connectionprofile);
  return $self->{PROFILE}->{connectionprofile};
}

sub FontFamily {
  my $self = shift;
  my ($family) = @_;
  $self->{PROFILE}->{font}->{family} = $family if defined($family);
  return "helvetica" if (($self->{PROFILE}->{font}->{family}) eq "");
  return $self->{PROFILE}->{font}->{family};
}

sub FontSize {
  my $self = shift;
  my ($size) = @_;
  $self->{PROFILE}->{font}->{size} = $size if defined($size);
  return 14 if (($self->{PROFILE}->{font}->{size}) eq "");
  return $self->{PROFILE}->{font}->{size};
}

sub WebBrowserCommand {
  my $self = shift;
  my ($command) = @_;
  $self->{PROFILE}->{webbrowser}->{command} = $command if defined($command);
  return "netscape %URL%" 
    if (($self->{PROFILE}->{webbrowser}->{command}) eq "");
  return $self->{PROFILE}->{webbrowser}->{command};
}

sub OnlineOnly {
  my $self = shift;
  my ($onlineonly) = @_;
  $self->{PROFILE}->{onlineonly} = $onlineonly if defined($onlineonly);
  return 0 if (($self->{PROFILE}->{onlineonly}) eq "");
  return $self->{PROFILE}->{onlineonly};
}

sub UnifiedPresence {
  my $self = shift;
  my ($unifiedpresence) = @_;
  $self->{PROFILE}->{unifiedpresence} = $unifiedpresence 
    if defined($unifiedpresence);
  return 1 if (($self->{PROFILE}->{unifiedpresence}) eq "");
  return $self->{PROFILE}->{unifiedpresence};
}

sub ShowDebug {
  my $self = shift;
  my ($showdebug) = @_;
  $self->{PROFILE}->{showdebug} = $showdebug if defined($showdebug);
  return 0 if (($self->{PROFILE}->{showdebug}) eq "");
  return $self->{PROFILE}->{showdebug};
}

sub GPGActive {
  my $self = shift;
  my ($gpgactive) = @_;
  $self->{PROFILE}->{gpgactive} = $gpgactive if defined($gpgactive);
  return 0 if (($self->{PROFILE}->{gpgactive}) eq "");
  return $self->{PROFILE}->{gpgactive};
}

sub GPGChat {
  my $self = shift;
  my ($gpgchat) = @_;
  $self->{PROFILE}->{gpgchat} = $gpgchat if defined($gpgchat);
  return 1 if (($self->{PROFILE}->{gpgchat}) eq "");
  return $self->{PROFILE}->{gpgchat};
}

sub GPGNormal {
  my $self = shift;
  my ($gpgnormal) = @_;
  $self->{PROFILE}->{gpgnormal} = $gpgnormal if defined($gpgnormal);
  return 1 if (($self->{PROFILE}->{gpgnormal}) eq "");
  return $self->{PROFILE}->{gpgnormal};
}

sub GPGAutoFetchTrust {
  my $self = shift;
  my ($gpgautofetchtrust) = @_;
  $self->{PROFILE}->{gpgautofetchtrust} = $gpgautofetchtrust if defined($gpgautofetchtrust);
  return 1 if (($self->{PROFILE}->{gpgautofetchtrust}) eq "");
  return $self->{PROFILE}->{gpgautofetchtrust};
}

sub GPGTrust {
  my $self = shift;
  my ($gpgtrust) = @_;
  $self->{PROFILE}->{gpgtrust} = $gpgtrust if defined($gpgtrust);
  return 3 if (($self->{PROFILE}->{gpgtrust}) eq "");
  return $self->{PROFILE}->{gpgtrust};
}

sub GPGKeyServer {
  my $self = shift;
  my ($gpgkeyserver) = @_;
  $self->{PROFILE}->{gpgkeyserver} = $gpgkeyserver if defined($gpgkeyserver);
  return "certserver.pgp.com" if (($self->{PROFILE}->{gpgkeyserver}) eq "");
  return $self->{PROFILE}->{gpgkeyserver};
}

sub RosterDoubleClick {
  my $self = shift;
  my ($rosterdoubleclick) = @_;
  $self->{PROFILE}->{rosterdoubleclick} = $rosterdoubleclick if defined($rosterdoubleclick);
  return "normal" if (($self->{PROFILE}->{rosterdoubleclick}) eq "");
  return $self->{PROFILE}->{rosterdoubleclick};
}

sub Subscriptions {
  my $self = shift;
  my ($subscriptions) = @_;
  $self->{PROFILE}->{subscriptions} = $subscriptions if defined($subscriptions);
  return "prompt" if (($self->{PROFILE}->{subscriptions}) eq "");
  return $self->{PROFILE}->{subscriptions};
}

sub SubscriptionDeny {
  my $self = shift;
  my ($subscriptiondeny) = @_;
  $self->{PROFILE}->{subscriptiondeny} = $subscriptiondeny if defined($subscriptiondeny);
  return 0 if (($self->{PROFILE}->{subscriptiondeny}) eq "");
  return $self->{PROFILE}->{subscriptiondeny};
}

sub SubscriptionDenyMessage {
  my $self = shift;
  my ($subscriptiondenymessage) = @_;
  $self->{PROFILE}->{subscriptiondenymessage} = $subscriptiondenymessage if defined($subscriptiondenymessage);
  return "Your subscription has been denied." if (($self->{PROFILE}->{subscriptiondenymessage}) eq "");
  return $self->{PROFILE}->{subscriptiondenymessage};
}

sub Subscribe {
  my $self = shift;
  my ($subscribe) = @_;
  $self->{PROFILE}->{subscribe} = $subscribe if defined($subscribe);
  return 0 if (($self->{PROFILE}->{subscribe}) eq "");
  return $self->{PROFILE}->{subscribe};
}


sub XML {
  my $self = shift;
  my $string;
  $string .= "<profile version='0.2'>";
  $string .= "<name>".&XML::Stream::EscapeXML($self->{PROFILE}->{name})."</name>";
  $string .= "<account>";
  $string .= "<server>".&XML::Stream::EscapeXML($self->Server())."</server>";
  $string .= "<username>".&XML::Stream::EscapeXML($self->Username())."</username>";
  $string .= "<password>".&XML::Stream::EscapeXML($self->Password())."</password>";
  $string .= "<resource>".&XML::Stream::EscapeXML($self->Resource())."</resource>";
  $string .= "<connectionprofile>".&XML::Stream::EscapeXML($self->ConnectionProfile())."</connectionprofile>";
  $string .= "<connections>";
  foreach my $name (keys(%{$self->{PROFILE}->{connections}})) {
    $string .= $self->{PROFILE}->{connections}->{$name}->XML();
  }
  $string .= "</connections>";
  $string .= "</account>";
  $string .= "<config>";

  $string .= "<font>";
  $string .= "<family>".&XML::Stream::EscapeXML($self->FontFamily())."</family>";
  $string .= "<size>".&XML::Stream::EscapeXML($self->FontSize())."</size>";
  $string .= "</font>";

  $string .= "<webbrowser>";
  $string .= "<command>".&XML::Stream::EscapeXML($self->WebBrowserCommand())."</command>";
  $string .= "</webbrowser>";
  $string .= "<onlineonly>".&XML::Stream::EscapeXML($self->OnlineOnly())."</onlineonly>";
  $string .= "<unifiedpresence>".&XML::Stream::EscapeXML($self->UnifiedPresence())."</unifiedpresence>";
  $string .= "<showdebug>".&XML::Stream::EscapeXML($self->ShowDebug())."</showdebug>";
  $string .= "<gpgactive>".&XML::Stream::EscapeXML($self->GPGActive())."</gpgactive>";
  $string .= "<gpgchat>".&XML::Stream::EscapeXML($self->GPGChat())."</gpgchat>";
  $string .= "<gpgnormal>".&XML::Stream::EscapeXML($self->GPGNormal())."</gpgnormal>";
  $string .= "<gpgautofetchtrust>".&XML::Stream::EscapeXML($self->GPGAutoFetchTrust())."</gpgautofetchtrust>";
  $string .= "<gpgtrust>".&XML::Stream::EscapeXML($self->GPGTrust())."</gpgtrust>";
  $string .= "<gpgkeyserver>".&XML::Stream::EscapeXML($self->GPGKeyServer())."</gpgkeyserver>";
  $string .= "<rosterdoubleclick>".&XML::Stream::EscapeXML($self->RosterDoubleClick())."</rosterdoubleclick>";
  $string .= "<subscriptions>".&XML::Stream::EscapeXML($self->Subscriptions())."</subscriptions>";
  $string .= "<subscriptiondeny>".&XML::Stream::EscapeXML($self->SubscriptionDeny())."</subscriptiondeny>";
  $string .= "<subscriptiondenymessage>".&XML::Stream::EscapeXML($self->SubscriptionDenyMessage())."</subscriptiondenymessage>";
  $string .= "<subscribe>".&XML::Stream::EscapeXML($self->Subscribe())."</subscribe>";
  $string .= "<favgroups>";
  foreach my $name (keys(%{$self->{PROFILE}->{favgroups}})) {
    $string .= $self->{PROFILE}->{favgroups}->{$name}->XML();
  }
  $string .= "</favgroups>";
  $string .= "</config>";
  $string .= "</profile>";
  return $string;
}


sub ParseCfg {
  my $self = shift;
  my ($file) = @_;

  my $parser = new XML::Stream::Parser();
  my @tree = $parser->parsefile("$ENV{HOME}/.jarl/$file");

  my $configVersion = &XML::Stream::GetXMLData("value",$tree[0],"","version","");

  $configVersion =~ s/\./\_/g;

  eval("\$self->ParseCfg_${configVersion}(\@tree);");
}


sub ParseCfg_0_2 {
  my $self = shift;
  my (@tree) = @_;

  $self->Name(&XML::Stream::GetXMLData("value",$tree[0],"name","",""));

  my @accountTree = &XML::Stream::GetXMLData("tree",$tree[0],"account","","");

  $self->Username(&XML::Stream::GetXMLData("value",\@accountTree,"username","",""));
  $self->Password(&XML::Stream::GetXMLData("value",\@accountTree,"password","",""));
  $self->Resource(&XML::Stream::GetXMLData("value",\@accountTree,"resource","",""));
  $self->Server(&XML::Stream::GetXMLData("value",\@accountTree,"server","",""));
  $self->ConnectionProfile(&XML::Stream::GetXMLData("value",\@accountTree,"connectionprofile","",""));

  my @connections = &XML::Stream::GetXMLData("tree",\@accountTree,"connections","","");
  my @connectionProfiles = &XML::Stream::GetXMLData("tree array",\@connections,"connection","","");
  foreach my $connectionProfile (@connectionProfiles) {
    my $name = &XML::Stream::GetXMLData("value",$connectionProfile,"","name","");
    $self->{PROFILE}->{connections}->{$name} = new Jarl::Profile::Connection;
    $self->{PROFILE}->{connections}->{$name}->Name($name);
    $self->{PROFILE}->{connections}->{$name}->Type(&XML::Stream::GetXMLData("value",$connectionProfile,"","type",""));
    $self->{PROFILE}->{connections}->{$name}->Port(&XML::Stream::GetXMLData("value",$connectionProfile,"","port",""));
  }

  my @configTree = &XML::Stream::GetXMLData("tree",$tree[0],"config","","");
  my @webBrowserTree = &XML::Stream::GetXMLData("tree",\@configTree,"webbrowser","","");
  $self->WebBrowserCommand(&XML::Stream::GetXMLData("value",\@webBrowserTree,"command","",""));

  my @fontTree = &XML::Stream::GetXMLData("tree",\@configTree,"font","","");
  $self->FontFamily(&XML::Stream::GetXMLData("value",\@fontTree,"family","",""));
  $self->FontSize(&XML::Stream::GetXMLData("value",\@fontTree,"size","",""));

  $self->OnlineOnly(&XML::Stream::GetXMLData("value",\@configTree,"onlineonly","",""));

  $self->UnifiedPresence(&XML::Stream::GetXMLData("value",\@configTree,"unifiedpresence","",""));

  $self->ShowDebug(&XML::Stream::GetXMLData("value",\@configTree,"showdebug","",""));

  $self->GPGActive(&XML::Stream::GetXMLData("value",\@configTree,"gpgactive","",""));
  $self->GPGChat(&XML::Stream::GetXMLData("value",\@configTree,"gpgchat","",""));
  $self->GPGNormal(&XML::Stream::GetXMLData("value",\@configTree,"gpgnormal","",""));
  $self->GPGAutoFetchTrust(&XML::Stream::GetXMLData("value",\@configTree,"gpgautofetchtrust","",""));
  $self->GPGTrust(&XML::Stream::GetXMLData("value",\@configTree,"gpgtrust","",""));
  $self->GPGKeyServer(&XML::Stream::GetXMLData("value",\@configTree,"gpgkeyserver","",""));

  $self->RosterDoubleClick(&XML::Stream::GetXMLData("value",\@configTree,"rosterdoubleclick","",""));

  $self->Subscriptions(&XML::Stream::GetXMLData("value",\@configTree,"subscriptions","",""));
  $self->SubscriptionDeny(&XML::Stream::GetXMLData("value",\@configTree,"subscriptiondeny","",""));
  $self->SubscriptionDenyMessage(&XML::Stream::GetXMLData("value",\@configTree,"subscriptiondenyMessage","",""));

  $self->Subscribe(&XML::Stream::GetXMLData("value",\@configTree,"subscribe","",""));

  my @favgroups = &XML::Stream::GetXMLData("tree",\@configTree,"favgroups","","");
  my @favgroupProfiles = &XML::Stream::GetXMLData("tree array",\@favgroups,"favgroup","","");
  foreach my $favgroupProfile (@favgroupProfiles) {
    my $name = &XML::Stream::GetXMLData("value",$favgroupProfile,"","name","");
    $self->{PROFILE}->{favgroups}->{$name} = new Jarl::Profile::FavGroup;
    $self->{PROFILE}->{favgroups}->{$name}->Name($name);
    $self->{PROFILE}->{favgroups}->{$name}->Channel(&XML::Stream::GetXMLData("value",$favgroupProfile,"","channel",""));
    $self->{PROFILE}->{favgroups}->{$name}->Server(&XML::Stream::GetXMLData("value",$favgroupProfile,"","server",""));
    $self->{PROFILE}->{favgroups}->{$name}->Nick1(&XML::Stream::GetXMLData("value",$favgroupProfile,"","nick1",""));
    $self->{PROFILE}->{favgroups}->{$name}->Nick2(&XML::Stream::GetXMLData("value",$favgroupProfile,"","nick2",""));
  }

}


sub Filename {
  my $self = shift;
  return $self->{PROFILE}->{username}."\@".$self->{PROFILE}->{server}."@".$self->{PROFILE}->{name};
}

sub Save {
  my $self = shift;
  my $filename = $self->Filename();
  system("mkdir $ENV{HOME}/.jarl") if !(-e "$ENV{HOME}/.jarl");
  open(CFG,">$ENV{HOME}/.jarl/$filename");
  print CFG $self->XML(),"\n";
  close(CFG);
}


sub Erase {
  my $self = shift;
  my $filename = $self->Filename();
  unlink("$ENV{HOME}/.jarl/$filename");
}


1;
