#ifndef lint
static char rcsid[] = "$Header: $";
#endif

/*
 * $Log: $
 */

#include "common.h"
#include "errors.h"
#if SYSLOG
#include <syslog.h>
#endif

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <sgtty.h>

/*
 *  Parse the config files into fields.  For a better description
 *  of what we're parsing, check out the input file as it is
 *  well-commented (at least, it should be).
 *
 * NEXTFIELD()	- moves `bcp' to start of next field & `cp' to
 *		  position after null-terminated end of field.
 * SUBFIELD()	- position `bcp' at next "," in current field (++bcp).
 * ISPAREN(P)	- ensures that P is a left paren (++bcp).
 * NOTNULL(P)	- ensures that P isnt end of string.
 * GETINT(F)	- atoi's current string to field F.
 * GETSTR(F,L)	- strcpy's L chars of string to field F & null-term.
 * RESERVED(I)	- make sure x co-ord is not on reserved lines.
 * MAKESTR(fld)	- alloc mem for string in bcp and point fld at it.
 * CHKALLOC(p)	- check if p got allocated memory or not.
 */

#define NEXTFIELD()		while (isspace(*cp)) cp++; bcp=cp; \
						while (!isspace(*cp)) cp++; *cp++='\0';
#define SUBFIELD()		if ((bcp=index(bcp,',')) == NULL) { \
							cantparse++; DisplayErr(lineno,PE_SUBFLD); continue;} \
						else bcp++;
#define ISPAREN(ch)		if (ch != '(') { \
							cantparse++; DisplayErr(lineno,PE_PAREN); continue; } \
						else bcp++;
#define NOTNULL(ch)		if (ch == '\0') { \
							cantparse++; DisplayErr(lineno,PE_EOL); continue; }
#define GETINT(fld)		if (isdigit(*bcp)) fld = atoi(bcp); \
						else { cantparse++; DisplayErr(lineno,PE_DIGIT); continue; }
#define GETSTR(fld,len)	(void) strncpy(fld, bcp, len); fld[len]='\0';
#define RESERVED(indx)	if (indx > reserved && \
							X_Doing != X_BAD && X_Doing != X_BRM && X_Doing != X_DAEMON) { \
							cantparse++; DisplayErr(lineno,PE_RESERV); continue; }
#define MAKESTR(fld)	if ((fld=(char *)NewStr(bcp)) == (char *)NULL) \
							{ sprintf(errmsg,"out of memory."); return(-1); }
#define CHKALLOC(p,t)	if ((p) == (t)NULL) \
							{ sprintf(errmsg,"out of memory."); return(-1); }

static char *cp;
static char *bcp;

/*
 * this takes a user-defined field type and returns the array index it
 * maps to.  it compares the size of "str" characters to determine a
 * match.  if more than one match is found, or no match at all, -1 is
 * returned.
 */
int GetFieldIndex(str)
	char *str;
{
	static int i, index, size, count;
	static field_t *mp;

	count = 0;
	size = strlen(str);
	for (i=0,mp=firstfield; mp && mp->str; i++,mp=mp->next)
		if (STRNCASEEQ(str, mp->str, size)) {
			count++;
			index=i;
		}
	if (count != 1) return(-1);
	else return(index);
}

/*
 * this takes a user-defined key type and returns the array index it
 * maps to.  it compares the size of "str" characters to determine a
 * match.  if more than one match is found, or no match at all, -2 is
 * returned.  I_GENFIELD is returned if the str matches S_GENFIELD.
 */
int GetKeyIndex(str)
	char *str;
{
	static int i, index, count, size;
	static mykey_t *kp;

	count = 0;
	size = strlen(str);
	if (strncasecmp(str, S_GENFIELD, size)==0)
		return(I_GENFIELD);
	for (i=0,kp=firstkey; kp && kp->str; i++,kp=kp->next) {
		if (STRNCASEEQ(str, kp->str, size)) {
			count++;
			index=i;
		}
	}
	if (count != 1) return(-2);
	else return(index);
}

/*
 * this reads the configuration files that define how the database is
 * structured.  if init equals 1, then we read the DBKeysFile and
 * DBFieldsFile first, and if no errors are encountered write out to the
 * DotDBKeysFile and DotDBFieldsFile.  this is an attempt to prevent the
 * database manager of making a mistake and crippling the database by
 * making a typo in one of the config files and preventing anyone from
 * using it.  if an error occurs in either file, neither Dot file is
 * written out.  if init equals 0, then we read the Dot files in as the
 * config files.  the formats of DBKeysFile and DotDBKeysFile are the
 * same. the format of DBFieldsFile and DotDBFieldsFile is the same, so we
 * can use the same routine to do the work.  the Dot files are readable,
 * but should not be edited directly for obvious reasons.
 *
 * if parsing errors are encountered, DisplayErr() is called.  there are 2
 * DisplayErr() routines.  one linked with the interactive program, and
 * one with the daemon.  a -1 is returned at the end of parsing a file if
 * parsing errors one or more parsing errors occurred.  a -1 is returned
 * immediately on a serious error.  if all goes well, a 0 is returned.
 */
int DBConfig(init)
	int init;
{
	FILE *fp;
	int cantparse=0, lineno, foundbang;
	int i;
	char *filename, *allow;

#ifdef REMOTE
	fp = REMOTEOUT;
	fprintf(REMOTEIN,"%s %s\n",C_CAT,S_DBKEYSFILE);
	fflush(REMOTEIN);
#else
	filename = init ? DBKeysFile : DotDBKeysFile;
	if ((fp=fopen(filename, "r")) == NULL) {
		SetSysErr();
		sprintf(errmsg,"%s: %s: %s", ProgName, filename, syserr);
		return(-1);
	}
#endif

	maxfieldlen = maxkeylen = nkeys = lineno = 0;
	pkeylen = -1;
	while (fgets(linein, LINEMAX, fp)) {
#ifdef REMOTE
		if (linein[3] != '-')
			break;
		bcp = cp = linein + 4;
#else
		bcp = cp = linein;
#endif

		lineno++;
		if (*cp == '#' || *cp == '\n')	/* comment, ignore it */
			continue;

		if (!firstkey) {
			CHKALLOC(curkey=firstkey=
				(mykey_t *)malloc(sizeof(mykey_t)),mykey_t*)
		} else {
			CHKALLOC(curkey->next=
				(mykey_t *)malloc(sizeof(mykey_t)),mykey_t*)
			curkey=curkey->next;
		}
		curkey->next = (mykey_t *)NULL;
		curkey->str = (char *)NULL;

		NEXTFIELD();		/* get KEY */
		NOTNULL(*cp);
		if ((i=GetKeyIndex(bcp)) >= 0) {
			DisplayErr(lineno,PE_DUP);
			cantparse++;
			continue;
		}
		MAKESTR(curkey->str)

		NEXTFIELD();		/* get key length */
		NOTNULL(*bcp);
		GETINT(curkey->len);
		if (maxkeylen < curkey->len) maxkeylen = curkey->len;

		CHKALLOC(curkey->filename =
			(char *)malloc(strlen(DBDir)+strlen(curkey->str)+2),char*)
		sprintf(curkey->filename,"%s/%s",DBDir,curkey->str);

		nkeys++;
	}
#ifndef REMOTE
	(void) fclose(fp);
#endif

	if (cantparse > 0) {
		sprintf(errmsg,"%s: %s: %d parsing error%s",
			ProgName, filename, cantparse, (cantparse==1) ? "" : "s");
		return(-1);
	}

	if (nkeys == 0) {
		sprintf(errmsg,"%s: no keys found, exiting...",ProgName);
		return(-1);
	}

	CHKALLOC(keys =
		(mykey_t **) malloc((sizeof(mykey_t *)*(nkeys))),mykey_t**)
	for (i=0,curkey=firstkey; i<nkeys; i++,curkey=curkey->next)
		keys[i] = curkey;

	i_pkey = i_access = i_creator = i_modifier = i_createdate = i_moddate = -1;

#ifdef REMOTE
	fprintf(REMOTEIN,"%s %s\n",C_CAT,S_DBFIELDSFILE);
	fflush(REMOTEIN);
#else
	filename = init ? DBFieldsFile : DotDBFieldsFile;
	if ((fp=fopen(filename, "r")) == NULL) {
		SetSysErr();
		sprintf(errmsg,"%s: %s: %s", ProgName, filename, syserr);
		return(-1);
	}
#endif

	nfields = 0;
	lineno = 0;
	recordlen = 0;
	while (fgets(linein, LINEMAX, fp)) {
#ifdef REMOTE
		if (linein[3] != '-')
			break;
		bcp = cp = linein + 4;
#else
		bcp = cp = linein;
#endif

		lineno++;
		if (*cp == '#' || *cp == '\n')	/* comment, ignore it */
			continue;

		if (!firstfield) {
			CHKALLOC(curfield=firstfield=
				(field_t *)malloc(sizeof(field_t)),field_t*)
		} else {
			CHKALLOC(curfield->next=
				(field_t *)malloc(sizeof(field_t)),field_t*)
			curfield=curfield->next;
		}
		curfield->next = (field_t *)NULL;
		curfield->str = (char *)NULL;

		if (nfields == 0 && cantparse) i_pkey = -1;

		NEXTFIELD();		/* get key */
		NOTNULL(*bcp);
		i = strlen(S_PRIMARY);
		if (strncasecmp(bcp,S_PRIMARY,i)==0 && *(bcp+i) == '=') {
			bcp += i + 1;
			if (i_pkey >= 0) {
				DisplayErr(lineno,PE_PKEYDUP);
				cantparse++;
				continue;
			}
			i_pkey = nfields;
		}

		curfield->keyindex = -1;
		if (STRCASEEQ(bcp,S_CREATOR)) {
			i_creator = nfields;
			curfield->exp = E_CREATOR;
			curfield->len = L_CREATOR;
		} else if (STRCASEEQ(bcp,S_MODIFIER)) {
			i_modifier = nfields;
			curfield->exp = E_MODIFIER;
			curfield->len = L_MODIFIER;
		} else if (STRCASEEQ(bcp,S_CREATEDATE)) {
			i_createdate = nfields;
			curfield->exp = E_CREATEDATE;
			curfield->len = L_CREATEDATE;
		} else if (STRCASEEQ(bcp,S_MODDATE)) {
			i_moddate = nfields;
			curfield->exp = E_MODDATE;
			curfield->len = L_MODDATE;
		} else if (STRCASEEQ(bcp,S_ACCESS))
			i_access = nfields;
		else if ((i=GetKeyIndex(bcp)) >= 0)
			curfield->keyindex = i;
		else if (i != I_GENFIELD) {
			DisplayErr(lineno,PE_BADKEY);
			cantparse++;
			continue;
		}

		if (nfields == i_pkey) {
			if (curfield->keyindex >= 0)
				pkeylen = curfield->len = keys[curfield->keyindex]->len;
			else {
				DisplayErr(lineno,PE_PRIMARY);
				cantparse++;
				continue;
			}
		}

		NEXTFIELD();		/* get field label */
		if ((i=GetFieldIndex(bcp)) > 0) {
			DisplayErr(lineno,PE_DUP);
			cantparse++;
			continue;
		}

		MAKESTR(curfield->str)

		if (nfields != i_modifier && nfields != i_moddate && nfields != i_creator && nfields != i_createdate) {
			NOTNULL(*cp)
			NEXTFIELD()		/* get field length */
			/* the key field length is as set in the DBKeysFile */
			if (nfields == i_pkey) curfield->len = pkeylen;
			else GETINT(curfield->len)
			if (curfield->len > maxfieldlen)
				maxfieldlen = curfield->len;

			NOTNULL(*cp)
			NEXTFIELD()		/* get choice->maxchosen */
			if (*bcp == '-')
				curfield->choice = (choice_t *)NULL;
			else {
				CHKALLOC(curfield->choice = 
					(choice_t *)malloc(sizeof(choice_t)),choice_t*)
				GETINT(curfield->choice->maxchosen)
			}

			NOTNULL(*cp)
			NEXTFIELD()		/* get properties */
			curfield->props = 0;
			CHKALLOC(curfield->propstr = (char *) NewStr(bcp),char*)

			i = cantparse;
			for (; *bcp; bcp++) {
				switch(*bcp) {
					case '-':       /* ignore */
						break;
					case 'M':       /* mandatory */
						curfield->props |= M_PROP;
						break;
					case 'R':       /* reserved */
						curfield->props |= R_PROP;
						break;
					default:        /* what? */
						DisplayErr(lineno,PE_PROP);
						cantparse++;
						break;
				}
				if (i != cantparse) break;
			}
			if (i != cantparse) continue;

			allow = curfield->allow;
			/* disallow all chars to start with */
			bzero(allow,ASCIICHARS);
			NEXTFIELD()
			NOTNULL(*cp)
			CHKALLOC(curfield->allowstr = (char *) NewStr(bcp),char*)
			for (; *bcp; bcp++) {
				if (*bcp == '!' && *(bcp+1)) {
					bcp++;
					foundbang = 1;
				} else
					foundbang = 0;
				switch(*bcp) {	/* determine what to accept */
					case 'A':		/* any printable characters */
						for (i=040; i < 0177; i++)
							allow[i] = foundbang ? 0 : 1;
						break;
					case 'C':		/* Any letters [A-Z,a-z] */
						for (i='a'; i <= 'z'; i++)
							allow[i] = (foundbang ? 0 : 1);
						for (i='A'; i <= 'Z'; i++)
							allow[i] = (foundbang ? 0 : 1);
						break;
					case 'L':		/* Any lower case letters [a-z] */
						for (i='a'; i <= 'z'; i++)
							allow[i] = (foundbang ? 0 : 1);
						break;
					case 'U':		/* Any upper case letters [A-Z] */
						for (i='A'; i <= 'Z'; i++)
							allow[i] = (foundbang ? 0 : 1);
						break;
					case 'D':		/* decimal digits [0-9] */
						for (i='0'; i <= '9'; i++)
							allow[i] = (foundbang ? 0 : 1);
						break;
					case 'X':		/* hex digits, [A-F,a-f,0-9]] */
						for (i='A'; i <= 'F'; i++)
							allow[i] = (foundbang ? 0 : 1);
						for (i='a'; i <= 'f'; i++)
							allow[i] = (foundbang ? 0 : 1);
						for (i='0'; i <= '9'; i++)
							allow[i] = (foundbang ? 0 : 1);
						break;
					case 'O':		/* octal digits, [0-7] */
						for (i='0'; i <= '7'; i++)
							allow[i] = (foundbang ? 0 : 1);
					case 'S':		/* spaces */
						i = (int) ' ';
						allow[i] = (foundbang ? 0 : 1);
						break;
					case '\\':		/* literal escape */
						if (*(++bcp) == '\0') {
							--bcp;	/* opps; nothing after the "\", ignore it */
							break;
						}
						/* fall thru and accept as actual character */
					default:		/* accept this if its is printable */
						i = (int) *bcp;
						if (isprint(i))
							allow[i] = (foundbang ? 0 : 1);
						break;
				}
			}
			/* we disallow spaces in the primary key field */
			if (nfields == i_pkey) {
				i = (int) ' ';
				allow[i] = 0;
			}

			/* take rest of line as the regular expression */

			while (isspace(*cp)) cp++; bcp=cp;
			/* nuke the newline on the end */
			if (*bcp != '\0')
				*(bcp+strlen(bcp)-1) = '\0';
			NOTNULL(*bcp);
			MAKESTR(curfield->exp)
			if ((i=mkchoicelist(curfield)) > 0) {
				DisplayErr(lineno,i);
				cantparse++;
				continue;
			}
		}

		/* +1 to account for null terminator */
		recordlen = recordlen + curfield->len + 1;

		nfields++;
	}
#ifndef REMOTE
	(void) fclose(fp);
#endif

	if (cantparse > 0) {
		sprintf(errmsg,"%s: %s: %d parsing error%s",
			ProgName, filename, cantparse, (cantparse==1) ? "" : "s");
		return(-1);
	}

	if (nfields == 0) {
		sprintf(errmsg,"%s: no fields found, exiting...",ProgName);
		return(-1);
	}

	CHKALLOC(fields =
		(field_t **) malloc((sizeof(field_t *)*(nfields))),field_t**)
	for (i=0,curfield=firstfield; i<nfields; i++,curfield=curfield->next)
		fields[i] = curfield;


	/* primary key and access fields are mandatory */
	if (i_pkey >= 0) fields[i_pkey]->props |= M_PROP;
	if (i_access >= 0) {
		fields[i_access]->props |= M_PROP;
		if (hasaccessall)
			mkaccesslist(accesschoices,naccess,1);
		else
			mkaccesslist(validchoices,nvalid,1);
	}

	if (init) {
		if ((fp=fopen(DotDBKeysFile, "w")) == (FILE *)NULL) {
			SetSysErr();
			sprintf(errmsg,"%s: fopen %s \"w\": %s",ProgName,DotDBKeysFile,syserr);
			return(-1);
		}
		fprintf(fp, "#\n# YOU ALMOST CERTAINLY SHOULD NOT EDIT THIS FILE BY HAND\n#\n");
		for (i=0; i < nkeys; i++)
			fprintf(fp, "%s %d\n", keys[i]->str, keys[i]->len);
		(void) fclose(fp);

		if ((fp=fopen(DotDBFieldsFile, "w")) == (FILE *)NULL) {
			SetSysErr();
			sprintf(errmsg, "%s: fopen %s \"w\": %s",ProgName,DotDBFieldsFile,syserr);
			return(-1);
		}
		fprintf(fp, "#\n# YOU ALMOST CERTAINLY SHOULD NOT EDIT THIS FILE BY HAND\n#\n");
		for (i=0; i < nfields; i++) {
			if (i == i_pkey) {
				fprintf(fp,"%s=%s %s - - - %s %s\n",S_PRIMARY,keys[fields[i]->keyindex]->str,fields[i]->str,fields[i]->allowstr,fields[i]->exp);
				continue;
			} else if (i == i_access)
				fprintf(fp,"%s ",S_ACCESS);
			else if (i == i_creator) {
				fprintf(fp,"%s %s - - - - %s\n",S_CREATOR,fields[i]->str,fields[i]->exp);
				continue;
			} else if (i == i_modifier) {
				fprintf(fp,"%s %s - - - - %s\n",S_MODIFIER,fields[i]->str,fields[i]->exp);
				continue;
			} else if (i == i_createdate) {
				fprintf(fp,"%s %s - - - - %s\n",S_CREATEDATE,fields[i]->str,fields[i]->exp);
				continue;
			} else if (i == i_moddate) {
				fprintf(fp,"%s %s - - - - %s\n",S_MODDATE,fields[i]->str,fields[i]->exp);
				continue;
			} else if (fields[i]->keyindex >= 0)
				fprintf(fp,"%s ",keys[fields[i]->keyindex]->str);
			else
				fprintf(fp,"%s ",S_GENFIELD);
			fprintf(fp, "%s %d ", fields[i]->str, fields[i]->len);
			if (!fields[i]->choice || i == i_access)
				fprintf(fp,"- ");
			else
				fprintf(fp,"%d ",fields[i]->choice->maxchosen);
			fprintf(fp,"%s %s ",fields[i]->propstr,fields[i]->allowstr);
			fprintf(fp, "%s\n",fields[i]->exp);
		}
		(void) fclose(fp);
#if SYSLOG || LOGTOFILE
		sprintf(msg,"addhoser %s initialized database config files.\n",InvokerAcct);
#if SYSLOG
		syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
		LogToFile(msg);
#endif
#endif
	}
	return(0);
}

/*
 * this routines behaves the same as DBConfig above (so read those
 * comments) except that this parses a single file, either SConfFile or
 * DotSConfFile.
 */
int ScreenConfig(init)
	int init;
{
	FILE *fp;
	register scrfield_t *fptr=NULL;
	register char *cp, *bcp;
	register int i;
	char *linetype, *filename;
	int cantparse=0, lineno=0;
	comment_t *curcomment=NULL,*firstcomment=NULL;
	specialscrfield_t *curspecialscrfield=NULL,*firstspecialscrfield=NULL;
	int offset, reserved;

#ifdef REMOTE
	fp = REMOTEOUT;
	fprintf(REMOTEIN,"%s %s\n",C_CAT,S_SCREENFILE);
	fflush(REMOTEIN);
#else
	filename = init ? SConfFile : DotSConfFile;
	if ((fp=fopen(filename, "r")) == NULL) {
		SetSysErr();
		sprintf(errmsg,"%s: %s: %s", ProgName, filename, syserr);
		return(-1);
	}
#endif

	/* nothing can be configured past this line */
	reserved = LastLine - 2;

	while (fgets(linein, LINEMAX, fp) != NULL) {
#ifdef REMOTE
		if (linein[3] != '-')
			break;
		bcp = cp = linein + 4;
#else
		bcp = cp = linein;
#endif

		lineno++;
		if (*cp == '#' || *cp == '\n')	/* comment, ignore it */
			continue;

		NEXTFIELD()			/* get prompt message */
		linetype = bcp;			/* for readability */
		if ((i=STREQ(S_ADDCOMM, linetype)) || (STREQ(S_RMCOMM, linetype))) {
			if (firstcomment == NULL) {
				firstcomment = (comment_t *) malloc(sizeof(comment_t));
				curcomment = firstcomment;
			} else {
				curcomment->next = (comment_t *) malloc(sizeof(comment_t));
				curcomment = curcomment->next;
			}
			CHKALLOC(curcomment,comment_t*)

			if (i)
				curcomment->type = X_ADD;
			else
				curcomment->type = X_RM;

			NEXTFIELD()			/* get prompt message */
			NOTNULL(*cp)
			MAKESTR(curcomment->text)
			/* convert dots to spaces */
			for (bcp=curcomment->text; *bcp; bcp++) 
				if (*bcp == '.')
			*bcp = ' ';
		
			NEXTFIELD()			/* read (x,y) for comment position */
			NOTNULL(*bcp)
			ISPAREN(*bcp)
			GETINT(curcomment->x_pos)
			SUBFIELD()
			GETINT(curcomment->y_pos)
			RESERVED(curcomment->y_pos)
			continue;
		}
		if (STREQ(S_SCREEN, linetype)) {
			if ((X_Doing == X_RM) && (curscreen != NULL))
				continue;

			if (curscreen==NULL) {
				firstscreen = curscreen = (screen_t *) malloc(sizeof(screen_t));
				CHKALLOC(firstscreen,screen_t*)
			} else {
				if (curscreen->begin==NULL) {
					DisplayErr(lineno,PE_EMPTY);
					cantparse++;
					continue;
				}
				if (curcomment != (comment_t *)NULL)
					curcomment->next = (comment_t *)NULL;
				if (curspecialscrfield != (specialscrfield_t *)NULL)
					curspecialscrfield->next = (specialscrfield_t *)NULL;
				curscreen->comment = firstcomment;
				curcomment = firstcomment = NULL;
				curscreen->specialscrfield = firstspecialscrfield;
				curspecialscrfield = firstspecialscrfield = NULL;
				CHKALLOC(curscreen->next =
					(screen_t *) malloc(sizeof(screen_t)),screen_t*)
				curscreen->next->prev = curscreen;
				curscreen->end = fptr;
				curscreen = curscreen->next;
			}
			if (X_Doing != X_DAEMON) {
				if ((curscreen->win=newwin(LastLine+1,LastCol+1,0,0)) == ERR) {
					sprintf(errmsg,"out of memory.");
					return(-1);
				}
				leaveok(curscreen->win, 0); /* after update, move to cur x/y */
				scrollok(curscreen->win, 0);/* like, dont *ever* scroll */
			}
			curscreen->begin = NULL;
			continue;
		}

		NOTNULL(*cp)
		if ((i=GetFieldIndex(linetype)) < 0) {
			DisplayErr(lineno,PE_NOTYPE);
			cantparse++;
			continue;
		}
		if ((X_Doing != X_RM) && (i==i_moddate || i==i_createdate || i==i_creator || i==i_modifier)) {
			if (firstspecialscrfield == NULL) {
				firstspecialscrfield = (specialscrfield_t *) malloc(sizeof(specialscrfield_t));
				curspecialscrfield = firstspecialscrfield;
			} else {
				curspecialscrfield->next = (specialscrfield_t *) malloc(sizeof(specialscrfield_t));
				curspecialscrfield = curspecialscrfield->next;
			}
			CHKALLOC(curspecialscrfield,specialscrfield_t*)
			curspecialscrfield->index = i;

			NEXTFIELD()			/* get prompt message */
			NOTNULL(*cp)
			MAKESTR(curspecialscrfield->label)
			/* convert dots to spaces */
			for (bcp=curspecialscrfield->label; *bcp; bcp++) 
				if (*bcp == '.')
			*bcp = ' ';
		
			NEXTFIELD()			/* read (x,y) for label position */
			NOTNULL(*cp)
			ISPAREN(*bcp)
			GETINT(curspecialscrfield->x_label)
			SUBFIELD()
			GETINT(curspecialscrfield->y_label)
			RESERVED(curspecialscrfield->y_label)

			NEXTFIELD()			/* read (x,y) for date string */
			NOTNULL(*bcp)
			ISPAREN(*bcp)
			GETINT(curspecialscrfield->x_pos)
			SUBFIELD()
			GETINT(curspecialscrfield->y_pos)
			RESERVED(curspecialscrfield->y_pos)
			continue;
		}

		if ((X_Doing == X_RM) && (i != i_pkey)) {
			continue;
		}

		if (fptr==NULL) {
			CHKALLOC(firstscrfield = fptr =
				(scrfield_t *) malloc(sizeof(scrfield_t)),scrfield_t*)
		} else {
			CHKALLOC(fptr->next =
				(scrfield_t *) malloc(sizeof(scrfield_t)),scrfield_t*)
			fptr->next->prev = fptr;
			fptr = fptr->next;
		}

		if ((curscreen!=NULL) && (curscreen->begin==NULL)){
			curscreen->begin = fptr;
		}

		fptr->index = i;

		NEXTFIELD()			/* get prompt message */
		NOTNULL(*cp)
		MAKESTR(fptr->label)
		for (bcp=fptr->label; *bcp; bcp++)	/* convert dots to spaces */
			if (*bcp == '.')
		*bcp = ' ';

		NEXTFIELD()			/* read (x,y) for prompt message */
		NOTNULL(*cp)
		ISPAREN(*bcp)
		GETINT(fptr->x_label)
		SUBFIELD()
		GETINT(fptr->y_label)
		RESERVED(fptr->y_label)

		NEXTFIELD()			/* read (x,y) for input field */
		NOTNULL(*cp)
		ISPAREN(*bcp)
		GETINT(fptr->x_input)
		SUBFIELD()
		GETINT(fptr->y_input)
		RESERVED(fptr->y_input)

		NEXTFIELD()			/* read (x,y) for error message */
		ISPAREN(*bcp)
		GETINT(fptr->x_error)
		SUBFIELD()
		GETINT(fptr->y_error)
	}
#ifndef REMOTE
	(void) fclose(fp);
#endif

	if (curcomment != (comment_t *)NULL)
		curcomment->next = (comment_t *)NULL;
	if (curspecialscrfield != (specialscrfield_t *)NULL)
		curspecialscrfield->next = (specialscrfield_t *)NULL;
	if (curscreen == (screen_t *)NULL) {
		if (fptr == NULL) {
			DisplayErr(lineno,PE_EMPTY);
			cantparse++;
		} else {
			CHKALLOC(curscreen =
				(screen_t *) malloc(sizeof(screen_t)),screen_t*)
			firstscreen = curscreen;
			if (X_Doing != X_DAEMON) {
				if ((curscreen->win = newwin(LastLine+1,LastCol+1,0,0)) == ERR) {
					sprintf(stderr, "out of memory.");
					return(-1);
				}
				leaveok(curscreen->win, 0); /* after update, move to cur x/y */
				scrollok(curscreen->win, 0);/* like, dont *ever* scroll */
			}
			curscreen->comment = firstcomment;
			curscreen->specialscrfield = firstspecialscrfield;
			curscreen->begin = firstscrfield;
			curscreen->end = fptr;
			curscreen->next = curscreen;
			curscreen->prev = curscreen;
		}
	} else {
		if (curscreen->begin==NULL) {
			DisplayErr(lineno,PE_EMPTY);
			cantparse++;
		} else {
			curscreen->comment = firstcomment;
			curscreen->specialscrfield = firstspecialscrfield;
			curscreen->next = firstscreen;
			firstscreen->prev = curscreen;
			curscreen->end = fptr;
			curscreen = firstscreen;
		}
	}
	if (cantparse > 0) {
		sprintf(errmsg,"%s: %s: %d parsing error%s",
			ProgName,filename,cantparse, (cantparse==1) ? "" : "s");
		return(-1);
	}

	fptr->next = firstscrfield;
	firstscrfield->prev = fptr;

	if (init) {
		if ((fp=fopen(DotSConfFile,"w")) == (FILE *)NULL) {
			SetSysErr();
			sprintf(errmsg,"%s: fopen %s \"w\": %s",ProgName,DotSConfFile,syserr);
			return(-1);
		}
        fprintf(fp,"#\n# YOU ALMOST CERTAINLY SHOULD NOT EDIT THIS FILE BY HAND\n#\n");
		fptr = firstscreen->begin;
		curscreen = firstscreen;
		do {
			if (fptr == curscreen->begin) {
				fprintf(fp,"%s\n",S_SCREEN);
				for (curcomment=curscreen->comment; curcomment; curcomment = curcomment->next) {
					if (curcomment->type == X_ADD)
						fprintf(fp,"%s ",S_ADDCOMM);
					else
						fprintf(fp,"%s ",S_RMCOMM);
					for (cp=curcomment->text; *cp; cp++) {
						if (*cp == ' ')
							fputc('.',fp);
						else
							fputc(*cp,fp);
					}
					fprintf(fp," (%d,%d)\n",curcomment->x_pos,curcomment->y_pos);
				}
				for (curspecialscrfield=curscreen->specialscrfield; curspecialscrfield; curspecialscrfield = curspecialscrfield->next) {
					fprintf(fp,"%s ",fields[curspecialscrfield->index]->str);
					for (cp=curspecialscrfield->label; *cp; cp++) {
						if (*cp == ' ')
							fputc('.',fp);
						else
							fputc(*cp,fp);
					}
					fprintf(fp," (%d,%d) (%d,%d)\n",curspecialscrfield->x_label,curspecialscrfield->y_label,
						curspecialscrfield->x_pos,curspecialscrfield->y_pos);
				}
			}
			fprintf(fp,"%s ",fields[fptr->index]->str);
			for (cp=fptr->label; *cp; cp++) {
				if (*cp == ' ')
					fputc('.',fp);
				else
					fputc(*cp,fp);
			}
			fprintf(fp," (%d,%d) (%d,%d) (%d,%d)\n",fptr->x_label,fptr->y_label,
				fptr->x_input,fptr->y_input,fptr->x_error,fptr->y_error);
			if (fptr == curscreen->end) {
				if (curscreen != curscreen->next) {
					curscreen = curscreen->next;
					fptr = curscreen->begin;
				} else
					fptr = curscreen->begin;
			} else
				fptr = fptr->next;
		} while (fptr != firstscreen->begin);
        (void) fclose(fp);
	}

	inputctrl();
	return(0);
}

/*
 * this sets up the vars in the choice_t structure that change from record
 * to record (or as the user changes selections), for each field that has
 * choices associated with it.
 */
SetUpChosen() {
	int i, j, n, *lens, nchosen;
	char *chosen, **list, *data, *cp;

	for (i=0; i<nfields; i++) {
		if (!fields[i]->choice)
			continue;
		if (fields[i]->choice->n != 0) {
			n = fields[i]->choice->n;
			chosen = fields[i]->choice->chosen;
			list = fields[i]->choice->list;
			data = fields[i]->data;
			lens = fields[i]->choice->lens;
			bzero(chosen,n);
			nchosen = 0;
			for (j=0; j<n; j++)
				if (cp=StrCaseStr(data,list[j])) {
					if ((cp == data || *(cp-1) == ' ') &&
					    (*(cp+lens[j]) == '\0' || *(cp+lens[j]) == ' ')) {
						chosen[j] = (char)1;
						nchosen++;
					}
				}
			fields[i]->choice->nchosen = nchosen;
		}
	}
}

/*
 * this is called for each field to see if the expression is such as to
 * warrant making it a choice field.  this would be an expression
 * beginning with '{', ending with '}', and with no other special
 * characters in between (only a comma separated list of choices).  if
 * it's a choice field, then the choice_t structure is set up for this
 * field.
 */
static mkchoicelist(mp)
	field_t *mp;
{
	char *cp, *cp2, *cp3, **list, *exp, *chosen;
	int wildcards=0, n=0, longest=0, len, len2, i, previ, *lens;
	
	if (!mp->choice)
		return(0);
	exp = mp->exp;
	mp->choice->n = 0;
	len = strlen(exp);
	cp = exp;
	while (*cp) {
		switch (*cp) {
			case ',':
				n++;
				break;
			case '\\':
				cp++;
				break;
			case '*': case '[': case '?': case '(':
				wildcards = 1;
				break;
			default:
				break;
		}
		if (*cp) cp++;
	}
/*{*/if (wildcards || len == 0 || *(exp+len-1) != '}')
		return(PE_BADEXP);
	n++;

	CHKALLOC(cp=(char *)NewStr(exp),char*)
	CHKALLOC(list=(char **)malloc(sizeof(char *)*n),char**)
	CHKALLOC(chosen=(char *)malloc(sizeof(char)*n),char*)
	CHKALLOC(lens=(int *)malloc(sizeof(int)*n),int*)
	cp2 = cp;
	cp3 = ++cp2;
	for (i=0; i<n; i++) {
/*{*/	while (*cp3 != '}' && *cp3 != ',') cp3++;
		*cp3++ = '\0';
		list[i] = cp2;
		lens[i] = strlen(list[i]);
		if ((len2=strlen(cp2)) > longest) longest = len2;
		cp2 = cp3;
		if (len2 > mp->len)
			return(PE_LONGEXP);
	}
	mp->choice->n = n;
	mp->choice->list = list;
	mp->choice->longest = longest;
	mp->choice->chosen = chosen;
	mp->choice->lens = lens;
	return(0);
}

/*
 * there is no expression for the access field by definition.  the only
 * valid choices are the ones found in AccessUsersFile, so this routine
 * fills in the choice_t structure for the access field.  the list var
 * that's passed to this routine is assume to have a char ** list of
 * choices, and n is the number of choices in list.
 */
static mkaccesslist(list,n)
	char **list;
	int n;
{
	char *cp, *cp2, *cp3, *chosen;
	int longest=0, len, i, *lens;
	
	CHKALLOC(fields[i_access]->choice =
		(choice_t *)malloc(sizeof(choice_t)),choice_t*)
	fields[i_access]->choice->n = n;

	CHKALLOC(chosen=(char *)malloc(sizeof(char)*n),char*)
	CHKALLOC(lens=(int *)malloc(sizeof(int)*n),int*)

	for (i=0; i<n; i++) {
		lens[i] = strlen(list[i]);
		if (lens[i] > longest)
			longest = lens[i];
	}

	fields[i_access]->choice->n = n;
	fields[i_access]->choice->list = list;
	fields[i_access]->choice->longest = longest;
	fields[i_access]->choice->chosen = chosen;
	fields[i_access]->choice->maxchosen = 1;
	fields[i_access]->choice->lens = lens;
	return(0);
}

/*
 *  inputctrl() grabs the users favorite erase, line and kill chars.
 */
static inputctrl() {
	struct sgttyb tty;
	struct ltchars ltc;

	(void) ioctl(0, (u_long) TIOCGETP, (char *) &tty);
	(void) ioctl(0, (u_long) TIOCGLTC, (char *) &ltc);
	c_erase = tty.sg_erase;
	c_kill = tty.sg_kill;
	c_werase = ltc.t_werasc;
}
