#ifndef lint
static char rcsid[] = "$Header: $";
#endif

/*
 * $Log:$
 */

#include <sys/file.h>
#include <stdio.h>
#include <ndbm.h>
#include <string.h>
#include <strings.h>
#include <sys/dir.h>
#include "common.h"
#if SYSLOG
#include <syslog.h>
#endif
#include <sys/stat.h>

static char path[MAXPATHLEN+1];
static FILE *DATA = (FILE *)NULL;
static DBM *KEYDB = (DBM *)NULL;
static datum key = {0,0};
static datum content = {0,0};
static datum retdatum;
static char *prevkeyfield = (char *)NULL;
static char *keyfield = (char *)NULL;
static char *lockfile1=(char *)NULL, *lockfile2=(char *)NULL;
static int lockmindex1, lockmindex2;
static datalocked = 0;
static char *keyfilename; /* used by FirstKey(), NextKey(), and FindKey() */

typedef struct datumlist {
	char *dptr;
	char type;
	struct datumlist *next;
} datumlist_t;

/*
 * this searches the string s1 for the string contained in s2 and returns
 * the pointer to the first occurence in s1 of s2, or NULL if s2 does not
 * occur in s1.  the routine is case insensitive.
 */
char *StrCaseStr(s1, s2)
	char *s1, *s2;
{
	static char *cp1, *cp2;
	static char c;

	cp1 = s1;
	cp2 = s2;
	if (!*cp1 || !*cp2) return((char *)NULL);
	c = TOLOWER(*cp2);

	while (1) {
		while (*cp1 && (TOLOWER(*cp1) != c)) cp1++;
		if (!*cp1) return((char *)NULL);
		s1 = cp1;
		do {
			cp1++;
			cp2++;
		} while ((*cp1 && *cp2) && (TOLOWER(*cp1) == TOLOWER(*cp2)));
		if (!*cp1 || !*cp2) break;
		cp1 = ++s1;
		cp2 = s2;
	}
	
	if (*cp2) return((char *)NULL);
	else return(s1);
}

/*
 * Allocate space and initialize variables needed for database routines.
 */
InitDB() {
	int i, offset;

	/* create space for the keyfield if not already done */
	if (!keyfield)
		if ((keyfield=(char *)malloc(maxfieldlen+1)) == (char *)NULL)
			FatalError("out of memory");
	if (!prevkeyfield)
		if ((prevkeyfield=(char *)malloc(maxfieldlen+1)) == (char *)NULL)
			FatalError("out of memory");

	/* create space for the key and content datum if not already done */
	if (!key.dptr)
		if ((key.dptr=(char *)malloc(maxkeylen+1)) == (char *)NULL)
			FatalError("out of memory");
	if (!content.dptr) 
		if ((content.dptr=(char *)malloc(sizeof(long))) == (char *)NULL)
			FatalError("out of memory");

	content.dsize = sizeof(long);

	/* used to blank out old creator or modifier on display */
	for (i=0; i<L_CREATOR; i++)
		unamewhitespace[i] = ' ';
	unamewhitespace[L_CREATOR] = '\0';

	data = (char *) malloc(recordlen);
	newdata = (char *) malloc(recordlen);
	prevdata = (char *) malloc(recordlen);
	origdata = (char *) malloc(recordlen);
	moddatestr = (char *) malloc(26);
	createdatestr = (char *) malloc(26);
	tmpfieldspace = (char *) malloc(maxfieldlen+1);

	if (!data || !newdata || !prevdata || !origdata ||
	    !moddatestr || !createdatestr || !tmpfieldspace) {
		(void) fprintf(stderr, "Insufficient memory, try later\n");
		CleanUp();
		exit(1);
	}

	bzero(data, recordlen);
	bzero(newdata, recordlen);
	bzero(prevdata, recordlen);
	bzero(origdata, recordlen);

	*moddatestr = *createdatestr = '\0';

	offset = 0;
	for (i=0; i<nfields; i++) {
		fields[i]->data = data + offset;
		fields[i]->newdata = newdata + offset;
		fields[i]->prevdata = prevdata + offset;
		fields[i]->origdata = origdata + offset;
		offset += fields[i]->len + 1;
	}
}

/*
 * if writeflag isn't 0 then we lock DataFile before writing to it so that
 * 2 processes aren't trying to write to it at the same time.  this
 * assumes that CloseData() is called as soon as the write is done in order
 * for the lock file DataLockFile to get removed in a timely fashion so
 * that others can do their writes.  OpenData() will wait up to 5 seconds
 * for the lock to be removed.  if this time period passes then it removes
 * the lock, in effect "forcing the lock".  errmsg is set and -1 returned
 * if an error occurred, 0 otherwise.
 */
int OpenData(writeflag)
	int writeflag;
{
	int fd = -1;
	int count=0;

	if (writeflag) {
		if (lockdata() < 0)
			return(-1);
		DATA = fopen(DataFile, "a+");
	} else
		DATA = fopen(DataFile, "r");
	if (DATA == (FILE *)NULL) {
		if (errno != ENOENT || (DATA=fopen(DataFile, "w+"))==(FILE *)0) {
			SetSysErr();
			sprintf(errmsg,"fopen %s: %s",DataFile,syserr);
			return(-1); 
		}
	}
	return(0);
}

CloseData() {
	if (DATA) fclose(DATA);
	DATA = (FILE *)NULL;
	if (datalocked) {
		unlink(DataLockFile);
		datalocked = 0;
	}
}

/*
 * after OpenData() is called read only, this routine is used to
 * sequentially read records from the database.  records that have null
 * primary key data are skipped.  -1 is returned on error, newdata will
 * contain the next record and a 0 is returned if there is another record
 * to be read, or the primary key in newdata will be set to null if no
 * more records are to be found.
 */
ReadNextRec() {
	static int retval;

	while (1) {
		if (!(retval=fread(newdata,recordlen,1,DATA)) && ferror(DATA)) {
			SetSysErr();
			sprintf(errmsg,"fread %s: %s",DataFile,syserr);
			return(-1);
		}
		if (retval == 0) {
			*fields[i_pkey]->newdata = '\0';
			return(0);
		}
		if (*fields[i_pkey]->newdata) return(0);
	}
}

/*
 * This is used to verify that keystr is a valid key in the key file
 * associated with keyindex.  If the key exists, then the offset is
 * returned.  If the key does not exist, then a -2 is returned.  If an
 * error occurred, then errmsg is set and -1 is returned.
 */
long FindKey(keyindex,keystr,retkeystr)
	int keyindex;
	char *keystr;
	char **retkeystr;
{
	long offset;
	char *cp;

	keyfilename = keys[keyindex]->filename;

	if ((KEYDB=dbm_open(keyfilename,O_RDONLY))==(DBM *)0) {
		if (errno != ENOENT) {
			SetSysErr();
			sprintf(errmsg,"dbm_open %s: %s",keyfilename,syserr);
			return(-1L);
		}
	}
	key.dsize = keys[keyindex]->len + 1;
	bzero(key.dptr,key.dsize);
	strcpy(key.dptr,keystr);
	MAKELO(key.dptr)
	*retkeystr = key.dptr;
	content = dbm_fetch(KEYDB,key);
	if (dbm_error(KEYDB)) {
		SetSysErr();
		dbm_close(KEYDB);
		sprintf(errmsg,"dbm_fetch %s: %s",keyfilename,syserr);
		return(-1L);
	}
	if (!content.dptr)
		return(-2L);
	bcopy(content.dptr,&offset,sizeof(long));
	if (KEYDB) dbm_close(KEYDB);
	if (offset < 0)
		return(-2L);
	else
		return(offset);
}

/*
 * This is used to get the first key from the key file associated with
 * keyindex.  If no keys exist for the given keyindex, then a -2 is
 * returned.  If the first key is found, then *keystr is pointed to the
 * key data, and the offset is returned.  This data may need to be copied
 * somewhere else since the space for the key is reused when NextKey() is
 * called.  If an error occurred, then errmsg is set and -1 is returned.
 */
long FirstKey(keyindex,keystr)
	int keyindex;
	char **keystr;
{
	static long offset;

	keyfilename = keys[keyindex]->filename;

	if ((KEYDB=dbm_open(keyfilename,O_RDONLY))==(DBM *)0) {
		if (errno != ENOENT) {
			SetSysErr();
			sprintf(errmsg,"dbm_open %s: %s",keyfilename,syserr);
			return(-1L);
		} else
			return(-2L);
	}
	key = dbm_firstkey(KEYDB);
	if (dbm_error(KEYDB)) {
		SetSysErr();
		dbm_close(KEYDB);
		sprintf(errmsg,"dbm_firstkey %s: %s",keyfilename,syserr);
		return(-1L);
	}
	if (key.dptr) {
		*keystr = key.dptr;
		content = dbm_fetch(KEYDB,key);
		if (dbm_error(KEYDB)) {
			SetSysErr();
			dbm_close(KEYDB);
			sprintf(errmsg,"dbm_fetch %s: %s",keyfilename,syserr);
			return(-1L);
		}
		bcopy(content.dptr,&offset,sizeof(long));
		if (offset < 0)
			return(NextKey(keystr));
		else
			return(offset);
	} else {
		if (KEYDB) dbm_close(KEYDB);
		return(-2L);
	}
}

/*
 * This should be called after FirstKey() to get the next key from the key
 * file associated with keyindex.  If no more keys exist for the given
 * keyindex, then a 1 is returned.  If another key is found, then *keystr
 * is pointed to the key data, and a 0 is returned.  This data may need to
 * be copied somewhere else since the space for the key is reused when
 * NextKey() is called again.  If an error occurred, then errmsg is set
 * and -1 is returned.
 */
long NextKey(keystr)
	char **keystr;
{
	static long offset;

	if (!KEYDB)
		return(-2L);
	while (1) {
		key = dbm_nextkey(KEYDB);
		if (dbm_error(KEYDB)) {
			SetSysErr();
			dbm_close(KEYDB);
			sprintf(errmsg,"dbm_nextkey %s: %s",keyfilename,syserr);
			return(-1L);
		}
		if (key.dptr) {
			*keystr = key.dptr;
			content = dbm_fetch(KEYDB,key);
			if (dbm_error(KEYDB)) {
				SetSysErr();
				dbm_close(KEYDB);
				sprintf(errmsg,"dbm_fetch %s: %s",keyfilename,syserr);
				return(-1L);
			}
			bcopy(content.dptr,&offset,sizeof(long));
			if (offset >= 0)
				return(offset);
		} else {
			if (KEYDB) dbm_close(KEYDB);
			return(-2L);
		}
	}
}

/*
 * if doing=0, then list reserved keys (keys with a -1 offset), and list
 * the locks in the RecLockDir and DBLockDir directories.  if doing=1,
 * then in addition list erroneous keys (keys that point to a null primary
 * key in the data file).  if doing=2 then remove reserved keys and locks.
 */
CheckDB(doing)
	int doing;
{
	int i, j, fd, retval, count=0;
	char *lockdir;
	struct direct **namelist;
	int first;
	struct stat statb;
	datumlist_t *topdatum, *newdatum, *datump;

	if (doing == 2) {
		/*
		 * had to create this datumlist because dbm_nextkey didn't
		 * return the next key once dbm_delete was called.
		 */
		if ((topdatum = (datumlist_t *)malloc(sizeof(datumlist_t))) ==
				(datumlist_t *)NULL)
			FatalError("out of memory");
		topdatum->next = (datumlist_t *)NULL;
#if SYSLOG || LOGTOFILE
		sprintf(msg,"addhoser %s cleaned out reserved keys and locks.\n",InvokerAcct);
#if SYSLOG
		syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
		LogToFile(msg);
#endif
#endif
	}
	for (i=0; i<nkeys; i++) {
		if (doing == 2)
			newdatum = topdatum;
		else
			first = 1;
		if ((KEYDB=dbm_open(keys[i]->filename,O_RDWR))==(DBM *)0) {
			/* if key file doesn't exist, we don't have to clean it */
			if (errno == ENOENT) continue;
			SetSysErr();
			sprintf(errmsg,"dbm_open %s: %s",keys[i]->filename,syserr);
			FatalError(errmsg);
		}
		for (retdatum=dbm_firstkey(KEYDB); retdatum.dptr; retdatum=dbm_nextkey(KEYDB)) {
			content = dbm_fetch(KEYDB,retdatum);
			if (dbm_error(KEYDB)) {
				SetSysErr();
				dbm_close(KEYDB);
				sprintf(errmsg,"dbm_fetch %s: %s",keys[i]->filename,syserr);
				FatalError(errmsg);
			}
			bcopy(content.dptr,&recordoffset,sizeof(long));
			if (recordoffset < 0) {
				if (doing == 2) {
					if ((newdatum->dptr=NewStr(retdatum.dptr)) == (char *)NULL)
						FatalError("out of memory");
					newdatum->type = 'r';
					if ((newdatum->next = (datumlist_t *)malloc(sizeof(datumlist_t))) == (datumlist_t *)NULL)
						FatalError(stderr,"out of memory");
					newdatum = newdatum->next;
					newdatum->next = (datumlist_t *)NULL;
				} else {
					if (first) {
						printf("keyfile %s\n",keys[i]->filename);
						first = 0;
					}
					printf("  %-32s (reserved key)\n",retdatum.dptr);
					count++;
				}
			} else if (doing == 1) {
				if (!DATA && OpenData(0) < 0)
					FatalError(errmsg);
				if (fseek(DATA,recordoffset,0) < 0) {
					SetSysErr();
					sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
					FatalError(errmsg);
				}
				if (!fread(newdata,recordlen,1,DATA)) {
					SetSysErr();
					sprintf(errmsg,"fread %s: %s",DataFile,syserr);
					FatalError(errmsg);
				}
				if (*fields[i_pkey]->newdata != '\0')
					continue;
				if (first) {
					printf("keyfile %s\n",keys[i]->filename);
					first = 0;
				}
				printf("  %-32s (erroneous key)\n",retdatum.dptr);
				count++;
			}
		}
		if (doing == 2) {
			j = 0;
			key.dsize = keys[i]->len + 1;
			DisableInterrupts();
			for (datump=topdatum; datump != newdatum; datump = datump->next) {
				count++;
				if (!j++) printf("keyfile %s\n",keys[i]->filename);
				bzero(key.dptr,key.dsize);
				strcpy(key.dptr,datump->dptr);
				if (datump->type == 'r')
					printf("  removing reserved key  '%s'.\n",datump->dptr);
				else
					printf("  removing erroneous key '%s'.\n",datump->dptr);
				if (dbm_delete(KEYDB,key) < 0) {
					SetSysErr();
					dbm_close(KEYDB);
					sprintf(errmsg,"dbm_delete %s: %s",
						keys[i]->filename,syserr);
					FatalError(errmsg);
				}
				free(datump->dptr);
			}
		}
		dbm_close(KEYDB);
		EnableInterrupts();
	}
	CloseData();

	sprintf(path,"%d",pid);
	lockdir = RecLockDir;
	while (lockdir) {
		first = 1;
		if ((i=scandir(lockdir, &namelist, 0, 0)) < 0) {
			SetSysErr();
			sprintf(errmsg,"scandir %s: %s",lockdir,syserr);
			FatalError(errmsg);
		}
		if (i > 2) {
			if (chdir(lockdir) < 0) {
				SetSysErr();
				sprintf(errmsg,"chdir %s: %s",lockdir,syserr);
				FatalError(errmsg);
			}
		}
		for (j=0; j<i; j++) {
			if (STREQ(".",namelist[j]->d_name) ||
			    STREQ("..",namelist[j]->d_name) ||
				(lockdir == DBLockDir && STREQ(path,namelist[j]->d_name))
			   ) continue;
			if ((fd=open(namelist[j]->d_name, O_RDONLY)) < 0) {
				SetSysErr();
				sprintf(errmsg,"open %s: %s",namelist[j]->d_name,syserr);
				FatalError(errmsg);
			}
			if ((retval=read(fd,LockAcct,IACCTLEN)) < 0) {
				SetSysErr();
				close(fd);
				sprintf(errmsg,"read %s: %s",namelist[j]->d_name,syserr);
				FatalError(errmsg);
			}
			LockAcct[retval] = '\0';
			close(fd);

			if (doing == 2) {
				if (first) {
					printf("lockdir %s\n",lockdir);
					first = 0;
				}
				printf("  removing %s lock left by %s.\n",
					namelist[j]->d_name,LockAcct);
				unlink(namelist[j]->d_name);
			} else {
				if (first) {
					printf("lockdir %s\n",lockdir);
					first = 0;
				}
				printf("  %-32s %s\n",LockAcct,namelist[j]->d_name);
			}
			*LockAcct = '\0';
			count++;
		}
		if (lockdir == RecLockDir)
			lockdir = DBLockDir;
		else
			lockdir = (char *)NULL;
	}
	if (stat(DataLockFile,&statb) < 0) {
		if (errno != ENOENT) {
			SetSysErr();
			sprintf(errmsg,"stat %s: %s",DataFile,syserr);
			FatalError(errmsg);
		}
	} else {
		if (doing == 2) {
			printf("Removing data lock file.\n");
			unlink(DataLockFile);
		} else
			printf("Data lock file exists.\n");
		count++;
	}
	if (count == 0) {
		if (doing == 1)
			printf("Found no reserved or erroneous keys and no locks.\n");
		else
			printf("Found no reserved keys and no locks.\n");
	}
}
	
/*
 * rid data file of empty slots (created when a record is removed) and
 * reserved and erroneous keys by recreating all the database files from
 * the old data file.
 */
CleanAll() {
	FILE *OLDDATA;
	int reccount=0, emptycount=0, i;
	struct stat statb;
	char *OldDataFile;

	sprintf(path,"%s.old",DBDir);
	if (stat(path,&statb) >= 0 || errno != ENOENT) {
		sprintf(errmsg,"%s must be removed by hand first.",path);
		FatalError(errmsg);
	}

	if (rm(NewDBDir) < 0 && errno != ENOENT) {
		sprintf(errmsg,"Couldn't remove %s",NewDBDir);
		FatalError(errmsg);
	}
	if (mkdir(NewDBDir,DBDIRMODE) < 0) {
		SetSysErr();
		sprintf(errmsg,"mkdir %s: %s",NewDBDir,syserr);
		FatalError(errmsg);
	}
	chmod(NewDBDir,DBDIRMODE);
	if (OpenData(0) < 0)
		FatalError(errmsg);
	OLDDATA = DATA;
	DATA = (FILE *)NULL;
	OldDataFile = DataFile;
	DataFile = NewDataFile;
	for (i=0; i < nkeys; i++) {
		keys[i]->filename = NewStr(sprintf(path,"%s/%s",
			NewDBDir,keys[i]->str));
		if (keys[i]->filename == (char *)NULL)
			FatalError("out of memory");
	}
	if (OpenData(1) < 0)
		FatalError(errmsg);
	bzero(prevdata,recordlen);
	bzero(newdata,recordlen);
	while (fread(data,recordlen,1,OLDDATA) == 1) {
		if (*fields[i_pkey]->data == '\0')
			emptycount++;
		else {
			reccount++;
			recordoffset = -1L;
			bcopy(data,origdata,recordlen);
			WriteData();
		}
	}
	if (ferror(OLDDATA)) {
		SetSysErr();
		sprintf(errmsg,"fread %s: %s",OldDataFile,syserr);
		rm(NewDBDir);
		FatalError(errmsg);
	}
	fclose(OLDDATA);
	CloseData();
	sprintf(path,"%s.old",DBDir);
	/* don't want any interrupts when moving the database files around */
	DisableInterrupts();
	if (rename(DBDir,path) < 0) {
		SetSysErr();
		sprintf(errmsg,"rename %s to %s: %s",DBDir,path,syserr);
		FatalError(errmsg);
	}
	if (rename(NewDBDir,DBDir) < 0) {
		/* try and back out of changes, then report error */
		SetSysErr();
		sprintf(errmsg,"rename %s to %s: %s",NewDBDir,DBDir,syserr);
		rename(path,DBDir);
		rm(NewDBDir);
		FatalError(errmsg);
	}
	rm(path);
	printf("Database cleaned, %d records found, %d empty records removed.\n",
		reccount,emptycount);
#if SYSLOG || LOGTOFILE
	sprintf(msg,"addhoser %s did a 'cleanall', %d records found, %d empty records removed.\n",InvokerAcct,reccount,emptycount);
#if SYSLOG
	syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
	LogToFile(msg);
#endif
#endif
	return(0);
}

/*
 * this tries to read a record from the database using the key found in
 * the field identified with mindex.  this assumes that the field contains
 * only a single key (no spaces).  this is used by the daemon process to
 * retrieve a record from the database if a key field is given with no
 * wildcards (to match multiple records).  ReserveKeys() actually reads
 * data from the database for the batch and screen oriented interfaces.
 * errmsg is set and -1 returned if an error is encountered.  1 is
 * returned if no record was found with that key.  0 is returned and the
 * record left in newdata if a record was found.
 */
int ReadData(mindex)
	int mindex;
{
	static int keyindex, returncode;
	static char *cp;

	keyindex = fields[mindex]->keyindex;

	if ((KEYDB=dbm_open(keys[keyindex]->filename,O_CREAT|O_RDWR,DBMMODE))==(DBM *)0) {
		SetSysErr();
		sprintf(errmsg,"dbm_open %s: %s",keys[fields[mindex]->keyindex]->filename,syserr);
		return(-1);
	}

	key.dsize = keys[keyindex]->len + 1;
	bzero(key.dptr,key.dsize);
	strcpy(key.dptr,fields[mindex]->data);
	MAKELO(key.dptr)

	retdatum = dbm_fetch(KEYDB,key);
	if (dbm_error(KEYDB)) {
		SetSysErr();
		sprintf(errmsg,"dbm_fetch %s: %s",keys[keyindex]->filename,syserr);
		return(-1);
	} else {
		newrecordoffset = -1L;
		if (retdatum.dptr)
			bcopy(retdatum.dptr, &newrecordoffset, sizeof(long));
		if (newrecordoffset < 0 || !retdatum.dptr) {
			returncode = 1;
		} else {
			if ((X_Doing != X_BRM || lockall == 0) && (DATA = fopen(DataFile,"r"))==(FILE *)NULL) {
				SetSysErr();
				sprintf(errmsg,"fopen %s: %s",DataFile,syserr);
				if (KEYDB) dbm_close(KEYDB);
				return(-1); 
			}
			if (fseek(DATA,newrecordoffset,0) < 0) {
				SetSysErr();
				sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
				returncode = -1;
			} else if (!fread(newdata,recordlen,1,DATA)) {
				SetSysErr();
				sprintf(errmsg,"fread %s: %s",DataFile,syserr);
				returncode = -1;
			} else
				returncode = 0;
		}
		if (DATA && ((X_Doing != X_BRM) || lockall == 0))
			fclose(DATA);
		if (KEYDB) dbm_close(KEYDB);
		return(returncode);
	}
}

/*
 * this opens DataFile if not already open, then reads the record located
 * at offset into newdata.
 */
int ReadDataByOffset(offset)
	long offset;
{
	if (!DATA && (DATA = fopen(DataFile,"r"))==(FILE *)NULL) {
		SetSysErr();
		sprintf(errmsg,"fopen %s: %s",DataFile,syserr);
		if (KEYDB) dbm_close(KEYDB);
		return(-1); 
	}
	if (fseek(DATA,offset,0) < 0) {
		SetSysErr();
		sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
		return(-1);
	} else if (!fread(newdata,recordlen,1,DATA)) {
		SetSysErr();
		sprintf(errmsg,"fread %s: %s",DataFile,syserr);
		return(-1);
	} else
		return(0);
}

/*
 * this writes out the record contained in data to DATA at the offset
 * defined by the global variable recordoffset.  if the offset is less
 * than zero then the offset for the end of the file is gotten, then the
 * record is written at this offset.  once the record is written, all the
 * keys associated with it are written to their respective key files with
 * the offset just used.  keys are assumed to be separated by spaces, so a
 * key field can have more than one key in it.  keys are always written in
 * lowercase to insure case insensitivity.  if all goes well, 0 is
 * returned.  on error, errmsg is set and a -1 is returned.
 */
int WriteData() {
	static int i, j, retval;
	static char *cp, *cp2, *cp3;

	if (recordoffset < 0) {
		if (fseek(DATA,0,2) < 0) {
			SetSysErr();
			sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
			fclose(DATA);
			return(-1);
		}
		recordoffset = ftell(DATA);
	} else {
		if (fseek(DATA,recordoffset,0) < 0) {
			SetSysErr();
			sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
			fclose(DATA);
			return(-1);
		}
	}

	/* write data to the data file at current location */
	if (!fwrite(data,recordlen,1,DATA)) {
		SetSysErr();
		sprintf(errmsg,"fwrite %s: %s",DataFile,syserr);
		fclose(DATA);
		return(-1);
	}

	/* write each key associated with this data to ndbm database */
	for (i=0; i<nkeys; i++) {
		KEYDB = 0;	/* set so we know if KEYDB has been opened yet or not */
		key.dsize = keys[i]->len + 1;
		for (j=0; j<nfields; j++) {
			if (fields[j]->keyindex != i) continue;
			cp = fields[j]->data;
			cp3 = fields[j]->origdata;
			if (fields[j]->keyindex == i && *cp != '\0') {
				if (!KEYDB) {
					if ((KEYDB=dbm_open(keys[i]->filename,O_CREAT|O_RDWR,DBMMODE))==(DBM *)0) {
						SetSysErr();
						sprintf(errmsg,"dbm_open %s: %s",keys[i]->filename,syserr);
						return(-1);
					}
					bcopy(&recordoffset,content.dptr,sizeof(long));
				}
				while (*cp) {
					cp2 = key.dptr;
					while (*cp && *cp == ' ') cp++;
					while (*cp && *cp != ' ') {
						*cp2++ = TOLOWER(*cp);
						cp++;
					}
					bzero(cp2,key.dsize-(cp2-key.dptr));
					if ((retval=dbm_store(KEYDB,key,content,DBM_REPLACE)) < 0) {
						SetSysErr();
						sprintf(errmsg,"dbm_store %s: %s",keys[i]->filename,syserr);
						dbm_close(KEYDB);
						return(-1);
					}
					while (*cp == ' ') cp++;
				}
				/* assume that no keys in data conflict with other records */
			}
		}
		if (KEYDB) dbm_close(KEYDB);
	}
	bcopy(origdata,prevdata,recordlen);
	bcopy(data,origdata,recordlen);
	FreeAllKeys();
	bcopy(data,prevdata,recordlen);
	return(0);
}

/*
 * this removes the record found in data from DataFile by zero'ing out
 * that space.  FreeAllKeys() is called to free all the keys associated
 * with the record.  RmRecLock() is called to free the lock on this (now
 * deleted) record.  errmsg is set and -1 returned if an error occurred, 0
 * otherwise.
 */
int RemoveData() {
	char *cp;

	key.dsize = fields[i_pkey]->len + 1;
	bzero(key.dptr,key.dsize);
	strcpy(key.dptr,fields[i_pkey]->data);
	MAKELO(key.dptr)

	if ((X_Doing != X_BRM || lockall == 0) && OpenData(1) < 0)
		return(-1);

	if (fseek(DATA, recordoffset, 0) < 0) {
		SetSysErr();
		sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
		fclose(DATA);
		return(-1);
	}
	bzero(origdata,recordlen);
	/* write null'd out record to mark space as free */
	if (!fwrite(origdata,recordlen,1,DATA)) {
		SetSysErr();
		sprintf(errmsg,"fwrite %s: %s",DataFile,syserr);
		fclose(DATA);
		return(-1);
	}
	if (X_Doing != X_BRM || lockall == 0)
		CloseData();
	FreeAllKeys();
	RmRecLock(i_pkey,fields[i_pkey]->data);
	bzero(data,recordlen);
	bzero(prevdata,recordlen);
	return(0);
}

/*
 * frees the keys in the string "before" that don't occur in the string
 * "after" and that don't occur in any like key fields in origdata
 * (figured out from keyindex).  origdata should have the same data as
 * what is currently written on disk in DataFile.  this is done because
 * the user may quit the program without actually saving the data they've
 * been modifying.  keys are considered used by a record until the record
 * is saved at which time keys can be freed if no longer in the record.
 * if no problems arise, then 0 is returned.  if an error occurs, errmsg
 * is set and -1 is returned.
 */
FreeKeys(keyindex,before,after)
	int keyindex;
	char *before;
	char *after;
{
	static int foundorig, len, j, retval;
	static char *cp, *cp2, *cp3;

	KEYDB = (DBM *)NULL;
	cp = before;
	while (*cp) {
		/* skip spaces */
		while (*cp && *cp == ' ') cp++;
		key.dsize = keys[keyindex]->len + 1;

		/* copy the key into key.dptr */
		cp2 = key.dptr;
		while (*cp && *cp != ' ') {
			*cp2++ = TOLOWER(*cp);
			cp++;
		}

		/* make sure that all trailing space is zero'd out */
		bzero(cp2,key.dsize-(cp2-key.dptr));

		len = strlen(key.dptr);
		/* check if current data has key, if so, then don't free it */
		if (cp3=StrCaseStr(after,key.dptr)) {
			/* make sure that our "key" is a key, and not an embedded string */
			if ((cp3 == after || *(cp3-1) == ' ') &&
			    (*(cp3+len) == '\0' || *(cp3+len) == ' '))
				continue;
		}

		foundorig = 0;
		for (j=0; j<nfields; j++) {
			if (fields[j]->keyindex != keyindex) continue;
			/* check if origdata has key */
			if (cp3=StrCaseStr(fields[j]->origdata,key.dptr)) {
				/* make sure that our "key" is a key, not an embedded string */
				if ((cp3 == fields[j]->origdata || *(cp3-1) == ' ') &&
				    (*(cp3+len) == '\0' || *(cp3+len) == ' ')) {
					foundorig++;
					break;
				}
			}
			if (foundorig) break;
		}
		if (foundorig) continue;

		if (!KEYDB) {
			if ((KEYDB=dbm_open(keys[keyindex]->filename,O_CREAT|O_RDWR,DBMMODE))==(DBM *)0) {
				SetSysErr();
				sprintf(errmsg,"dbm_open %s: %s",keys[keyindex]->filename,syserr);
				return(-1);
			}
		}

		if ((retval=dbm_delete(KEYDB,key)) < 0) {
			SetSysErr();
			sprintf(errmsg,"dbm_delete %s: %s",keys[keyindex]->filename,syserr);
			dbm_close(KEYDB);
			return(-1);
		}
	}
	if (KEYDB) dbm_close(KEYDB);
	return(0);
}

/*
 * frees keys in prevdata that don't occur in origdata.  this is used to
 * remove all the keys associated with a record after RemoveData() is
 * called.  this is also used by WriteData() to free up any keys that were
 * previously used in a record, but no longer are.  this is also in other
 * locations when data is being entered for a key field, and keys have
 * been reserved by writing the keys with a -1 offset attached to them,
 * but then the record doesn't get saved.  in this case, these reserved
 * keys with -1 offsets need to be freed.
 */
FreeAllKeys() {
	static int i, j, keyindex, len, retval;
	static char *cp, *cp2, *cp3;

	if (!prevdata) return;
	for (i=0; i<nkeys; i++) {
		KEYDB = (DBM *)NULL;
		key.dsize = keys[i]->len + 1;
		for (j=0; j<nfields; j++) {
			if ((keyindex=fields[j]->keyindex) != i) continue;
			cp = fields[j]->prevdata;
			while (*cp) {
				/* skip spaces */
				while (*cp && *cp == ' ') cp++;
				key.dsize = keys[keyindex]->len + 1;
		
				/* copy the key into key.dptr */
				cp2 = key.dptr;
				while (*cp && *cp != ' ') {
					*cp2++ = TOLOWER(*cp);
					cp++;
				}
		
				/* make sure that all trailing space is zero'd out */
				bzero(cp2,key.dsize-(cp2-key.dptr));
		
				/* check if origdata has key, if so, then don't free it */
				if (cp3=StrCaseStr(fields[j]->origdata,key.dptr)) {
					len = strlen(key.dptr);
					/* make sure that our "key" is a key, and not an embedded string */
					if ((cp3 == fields[j]->origdata || *(cp3-1) == ' ') &&
					    (*(cp3+len) == '\0' || *(cp3+len) == ' '))
						continue;
				}
		
				if (!KEYDB) {
					if ((KEYDB=dbm_open(keys[keyindex]->filename,O_CREAT|O_RDWR,DBMMODE))==(DBM *)0) {
						SetSysErr();
						sprintf(errmsg,"dbm_open %s: %s",keys[keyindex]->filename,syserr);
						return(-1);
					}
				}
		
				if ((retval=dbm_delete(KEYDB,key)) < 0) {
					SetSysErr();
					sprintf(errmsg,"dbm_delete %s: %s",keys[keyindex]->filename,syserr);
					dbm_close(KEYDB);
					return(-1);
				}
			}
		}
		if (KEYDB) dbm_close(KEYDB);
	}
	return(0);
}

/*
 * this reserves keys for the field associated with mindex.  multiple keys
 * can exist in the field if separated by spaces.  a check is first made
 * to insure that the keys are not already used by attempting to write
 * each key to the appropriate key file.  if a key is already there, then
 * we put a null in fields[mindex]->data at the point where the used key
 * was found so FreeKeys() will not accidently free keys that we didn't
 * reserve.  errmsg is set and -1 is returned on error.  0 is returned if
 * all key(s) in the field got reserved.  1 is returned if we found a key
 * that has a record associated with it or is reserved (has an offset of
 * -1 with it).  if this happens then the record is read into newdata (but
 * a lock on that record is not attempted at this time).  2 is returned if
 * a key is used twice in this record.
 */
ReserveKeys(mindex, keystr)
	int mindex;
	char **keystr;
{
	static int returncode, foundorig, foundprev, found, i, j, retval, len;
	char *limit, *cp, *cp2, *cp3, *cp4, *keybegin;

	returncode = 0;
	newrecordoffset = -1L;
	/* keys with -1 offsets are being reserved */
	bcopy(&newrecordoffset,content.dptr,sizeof(long));
	KEYDB = (DBM *)0;
	cp = fields[mindex]->data;
	while (*cp) {
		/* skip spaces */
		while (*cp && *cp == ' ') cp++;
		if (!*cp) break;
		key.dsize = keys[fields[mindex]->keyindex]->len + 1;

		keybegin = cp;

		/* copy the key into key.dptr */
		cp2 = key.dptr;
		while (*cp && *cp != ' ') {
			*cp2++ = TOLOWER(*cp);
			cp++;
		}

		/* make sure that all trailing bytes are zero'd out */
		bzero(cp2,key.dsize-(cp2-key.dptr));

		/* if key is in origdata, then it's reserved for us */
		len = strlen(key.dptr);
		foundorig = 0;
		i = fields[mindex]->keyindex;
		for (j=0; j<nfields; j++) {
			if (fields[j]->keyindex != i) continue;
			/* check if origdata has key */
			if (cp3=StrCaseStr(fields[j]->origdata,key.dptr)) {
				/* make sure that our "key" is a key, not an embedded string */
				if ((cp3 == fields[j]->origdata || *(cp3-1) == ' ') &&
				    (*(cp3+len) == '\0' || *(cp3+len) == ' ')) {
					foundorig++;
					break;
				}
			}
			if (foundorig) break;
		}
		foundprev = 0;
		/* check if prevdata has key */
		if (cp3=StrCaseStr(fields[mindex]->prevdata,key.dptr)) {
			/* make sure that our "key" is a key, and not an embedded string */
			if ((cp3 == fields[mindex]->prevdata || *(cp3-1) == ' ') &&
			    (*(cp3+len) == '\0' || *(cp3+len) == ' ')) {
				foundprev++;
			}
		}
		/* make sure that key is not used twice in this record */
		found = 0;
		i = fields[mindex]->keyindex;
		for (j=0; j<nfields; j++) {
			if (fields[j]->keyindex != i) continue;
			cp3 = fields[j]->data;
			limit = fields[j]->data + fields[j]->len;
			while (cp3+len <= limit && (cp4=StrCaseStr(cp3,key.dptr))) {
				if ((cp3 == cp4 || *(cp4-1) == ' ') &&
				    (*(cp4+len) == '\0' || *(cp4+len) == ' ')) {
					found++;
				}
				cp3 = cp4+len;
			}
		}
		/* if foundorig then this key is attached to the existing record */
		/* if foundprev then this key is reserved with a -1 offset */
		/* found must equal 1 or there is a duplicate key in this record */
		if (found == 1 && (foundorig || foundprev)) continue;
		if (found > 1) {
			*keystr = strcpy(tmpfieldspace,key.dptr);
			/* make sure we don't try to free keys we haven't reserved */
			*keybegin = '\0';
			return(2);
		}

		/* check if we've opened the database or not */
		if (!KEYDB) {
			if ((KEYDB=dbm_open(keys[fields[mindex]->keyindex]->filename,O_CREAT|O_RDWR,DBMMODE))==(DBM *)0) {
				SetSysErr();
				sprintf(errmsg,"dbm_open %s: %s",keys[fields[mindex]->keyindex]->filename,syserr);
				returncode = -1;
				break;
			}
		}

		/* try and store our key */
		if ((retval=dbm_store(KEYDB,key,content,DBM_INSERT)) < 0) {
			SetSysErr();
			sprintf(errmsg,"dbm_store %s: %s",keys[fields[mindex]->keyindex]->filename,syserr);
			returncode = -1;
			break;
		} else if (retval == 1) {
			/* key is already reserved, find out what it's attached to */
			retdatum = dbm_fetch(KEYDB,key);
			if (dbm_error(KEYDB)) {
				SetSysErr();
				sprintf(errmsg,"dbm_fetch %s: %s",keys[fields[mindex]->keyindex]->filename,syserr);
				returncode = -1;
			} else {
				bcopy(retdatum.dptr, &newrecordoffset, sizeof(long));
				*keystr = strcpy(tmpfieldspace,key.dptr);
				if (newrecordoffset < 0)
					returncode = 1;
				else {
					if (!DATA && OpenData(0) < 0)
						returncode = -1;
					else {
						if (fseek(DATA,newrecordoffset,0)) {
							SetSysErr();
							sprintf(errmsg,"fseek %s: %s",DataFile,syserr);
							returncode = -1;
						} else if (!fread(newdata,recordlen,1,DATA)) {
							SetSysErr();
							sprintf(errmsg,"fread %s: %s",DataFile,syserr);
							returncode = -1;
						} else
							returncode = 1;
					}
				}
			}
			/* make sure we don't try to free keys we haven't reserved */
			*keybegin = '\0';
			break;
		} /* else we have successfully reserved a key */
	}

	if (KEYDB)
		dbm_close(KEYDB);
	/* if running batch, we don't CloseData() */
	if (DATA && ((X_Doing != X_BAD && X_Doing != X_BRM) || lockall == 0))
		CloseData();
	return(returncode);
}

off_t NumberOfRecs() {
	struct stat statb;

	if (stat(DataFile,&statb) < 0) {
		if (errno == ENOENT)
			statb.st_size = 0;
		else {
			SetSysErr();
			sprintf(errmsg,"stat %s: %s",DataFile,syserr);
			return(-1);
		}
	}
	return(statb.st_size/recordlen);
}

SendSignal(sig)
	int sig;
{
	int fd, retval, i,j,k;
	FILE *MSG;
	struct direct **namelist;

	if ((i=scandir(DBLockDir, &namelist, 0, 0)) < 0) {
		SetSysErr();
		sprintf(errmsg,"scandir %s: %s",DBLockDir,syserr);
		FatalError(errmsg);
	}
	if (i < 3) {
		if (sig == SIGHUP)
			fprintf(stderr, "No processes to send message to.\n");
		else
			fprintf(stderr, "No processes to kill.\n");
		exit(1);
	}
	if (chdir(DBLockDir) < 0) {
		SetSysErr();
		sprintf(errmsg,"chdir %s: %s",DBLockDir,syserr);
		FatalError(errmsg);
	}

	if (sig == SIGHUP) {
		if ((MSG=fopen(HUPMesgFile, "w+"))==(FILE *)0) {
			SetSysErr();
			sprintf(errmsg,"fopen %s: %s",HUPMesgFile,syserr);
			FatalError(errmsg);
		}
		fprintf(MSG,"Message from %s.\n\n",InvokerAcct);
		printf("enter a message ended by ^D.\n");
		while (fgets(linein, LINEMAX, stdin))
			fputs(linein, MSG);
		fclose(MSG);
		printf("sending message to following processes:\n");
	} else
		printf("killing following processes:\n");
	for (j=0; j<i; j++) {
		if (*namelist[j]->d_name == '.' || STRCASEEQ(namelist[j]->d_name,S_ALL))
		   continue;
		if (!isdigit(*namelist[j]->d_name)) {
			fprintf(stderr,"  \"%s\" skipped, not a pid.\n",namelist[j]->d_name);
			continue;
		}
		k = atol(namelist[j]->d_name);
		if (kill(k,sig) == 0) {
			if (sig == SIGHUP)
				printf("  pid %d was sent message.\n",k);
			else {
				printf("  pid %d was sent TERM signal.\n",k);
#if SYSLOG || LOGTOFILE
				sprintf(msg,"addhoser %s killed pid %d.\n",InvokerAcct,k);
#if SYSLOG
				syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
				LogToFile(msg);
#endif
#endif
			}
		} else {
			switch (errno) {
				case EINVAL:
					fprintf(stderr,"  pid %d, invalid signal number %d (should not happen)\n",k,sig);
					break;
				case EPERM:
					fprintf(stderr,"  pid %d isn't ours (removing that lock).\n",k);
					unlink(namelist[j]->d_name);
					break;
				case ESRCH:
					fprintf(stderr,"  pid %d not found (removing that lock).\n",k);
					unlink(namelist[j]->d_name);
					break;
				default:
					fprintf(stderr,"  pid %d, unknown error.\n",k);
					break;
			}
		}
	}
	exit(0);
}

/*
 * if i==i_moddate then set current date for moddate and current user for
 * modifier. if i==i_createdate then set current date for both moddate and
 * createdate and current user for both modifier and creator.
 */
StampRec(i)
	int i;
{
	time_t clock;
	char *c;

	clock = time(0);
	if (i_moddate >= 0) {
		strcpy(moddatestr,ctime(&clock));
		moddatestr[24]='\0';
		bzero(fields[i_moddate]->data,L_MODDATE+1);
		sprintf(fields[i_moddate]->data,"%d",clock);
	}
	if (i_modifier >= 0) {
		bzero(fields[i_modifier]->data,L_MODIFIER+1);
		strcpy(fields[i_modifier]->data,InvokerAcct);
	}
	if (i == i_createdate) {
		if (i_createdate >= 0) {
			strcpy(createdatestr,ctime(&clock));
			createdatestr[24]='\0';
			bzero(fields[i_createdate]->data,L_CREATEDATE+1);
			sprintf(fields[i_createdate]->data,"%d",clock);
		}
		if (i_creator >= 0) {
			bzero(fields[i_creator]->data,L_CREATOR+1);
			strcpy(fields[i_creator]->data,InvokerAcct);
		}
	}
}

/*
 * converts the numeric form of date from the data to ascii using ctime
 * for both moddate and createdate.
 */
ConvertDate(flag)
	int flag;
{
	time_t clock;
	char *c;

	if (flag == 0) {
		*moddatestr = '\0';
		*createdatestr = '\0';
		return;
	}

	if (i_moddate >= 0) {
		clock = (time_t) atoi(fields[i_moddate]->data);
		strcpy(moddatestr,ctime(&clock));
		moddatestr[24]='\0';
	}

	if (i_createdate >= 0) {
		clock = (time_t) atoi(fields[i_createdate]->data);
		strcpy(createdatestr,ctime(&clock));
		createdatestr[24]='\0';
	}
}

/*****************************************************************************/
/* Lock routines                                                             */
/*****************************************************************************/

/*
 * this locks the DataFile by creating DataLockFile for the brief amount
 * of time it takes for an update to DataFile.  this is to insure no other
 * writes are taking place at the same time.  this routine will wait up to
 * 5 seconds for the DataLockFile to disappear since it's assumed that the
 * lock file should only exist for brief periods of time.  if the 5 second
 * time limit expires, then the lock is assume to be erroneous and the
 * routine returns as if it had successfully created the lock.  on error,
 * errmsg is set and -1 is returned.  on success, 0 is returned.
 */
lockdata() {
	static int count, fd;

	count = 0;
	while (count++ < DATALOCKTIME) {
		if ((fd=open(DataLockFile, O_WRONLY|O_CREAT|O_EXCL, 0644)) < 0) {
			if (errno == EEXIST) {
				sleep(1);
				continue;
			} else {
				SetSysErr();
				sprintf(errmsg,"open %s: %s",DataLockFile,syserr);
				return(-1);
			}
		} else if (write(fd,InvokerAcct,strlen(InvokerAcct)) < 0) {
			if (fd >= 0) (void) close(fd);
			SetSysErr();
			sprintf(errmsg,"write %s: %s",DataLockFile,syserr);
			return(-1);
		}
		(void) close(fd);
		break;
	}
	if (fd < 0) {
#if SYSLOG || LOGTOFILE
		sprintf(msg,"data lock still there after %d seconds, ignoring lock.\n",DATALOCKTIME);
#if SYSLOG
		syslog(LOG_INFO, msg);
#endif
#if LOGTOFILE
		LogToFile(msg);
#endif
#endif
	}
	datalocked = 1;
	return(0);
}

/*
 * will create the LockAllFile lock file if it doesn't already exist.  if
 * the file already exists, or the file already exists and the contents
 * don't match InvokerAcct, then we exit the program, and print out an
 * error.  otherwise return normally to caller.
 */
LockOutAll(ask)
	int ask;
{
	int fd, retval, i;
	FILE *LOCKMSG;
	struct direct **namelist;

	if ((fd=open(LockAllFile, O_WRONLY|O_CREAT|O_EXCL, 0644)) < 0)	{
		/* if error isn't EEXIST, we have a problem */
		if (errno == EEXIST) {
			/* if open fails with read-only, we have a problem */
			if ((fd=open(LockAllFile,O_RDONLY)) >= 0) {
				/* if read fails, we have a problem */
				if ((retval=read(fd,LockAcct,IACCTLEN)) >= 0) {
					if (retval >= 0) LockAcct[retval] = '\0';
					close(fd);
					/* file exists, print name of who locked it */
					if (X_Doing == X_BAD || X_Doing == X_BRM)
						printf("210-");
					if (STREQ(LockAcct,InvokerAcct)) {
						printf("You already have the database locked.\n");
						lockall = 2;
					} else
						printf("The database is already locked by %s.\n",LockAcct);
					return;
				}
			}
		}
		/* if we fell through to here, we need to get the error message */
		SetSysErr();
		printf("open %s: %s\n",LockAllFile,syserr);
		if (fd >= 0) (void) close(fd);
		return;
	}

	if (write(fd,InvokerAcct,strlen(InvokerAcct)) < 0) {
		SetSysErr();
		printf("write %s: %s\n",LockAllFile,syserr);
		if (fd >= 0) (void) close(fd);
		unlink(LockAllFile);
		return;
	}
	(void) close(fd);

	if ((i=scandir(DBLockDir, &namelist, 0, 0)) < 0) {
		SetSysErr();
		sprintf(errmsg,"scandir %s: %s",DBLockDir,syserr);
		unlink(LockAllFile);
		FatalError(errmsg);
	}
	if (i > 3) {
		if (ask)
			fprintf(stderr, "WARNING! It appears there are users accessing the database.\n");
		else if ((X_Doing == X_BAD || X_Doing == X_BRM) && i != 4) {
			unlink(LockAllFile);
			FatalError("Could not lock the database, it appears there are users accessing the database.");
		}
	}

	if ((LOCKMSG=fopen(LockMsgFile, "w+"))==(FILE *)0) {
		SetSysErr();
		sprintf(errmsg,"fopen %s: %s",LockMsgFile,syserr);
		unlink(LockAllFile);
		FatalError(errmsg);
	}
	if (ask) {
		printf("Database is now locked, enter a rejection message ended by ^D.\n");
		while (fgets(linein, LINEMAX, stdin))
			fputs(linein, LOCKMSG);
	} else {
		if (X_Doing == X_BAD || X_Doing == X_BRM)
			printf("210-");
		printf("Database is now locked.\n");
		fputs("Database administrator is working on the database.\n",LOCKMSG);
	}
	fclose(LOCKMSG);
#if SYSLOG || LOGTOFILE
	sprintf(msg,"addhoser %s locked database.",InvokerAcct);
#if SYSLOG
	syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
	LogToFile(msg);
#endif
#endif
	lockall = 1;
	return;
}

/*
 * return 0 if database got unlocked, 1 if database was unlocked already.
 */
UnLockOutAll() {
	if (unlink(LockAllFile) < 0) {
		if (errno == ENOENT) {
			if (X_Doing == X_BAD || X_Doing == X_BRM)
				printf("211-Database is not locked.\n");
			else
				fprintf(stderr,"Database is not locked.\n");
		} else {
			SetSysErr();
			if (X_Doing == X_BAD || X_Doing == X_BRM)
				printf("450-unlink %s: %s\n",LockMsgFile,syserr);
			else
				fprintf(stderr,"unlink %s: %s\n",LockMsgFile,syserr);
		}
		return(1);
	}
	unlink(LockMsgFile);
	lockall = 0;
	if (X_Doing == X_BAD || X_Doing == X_BRM)
		printf("211-");
	printf("Database is now unlocked.\n");
#if SYSLOG || LOGTOFILE
	sprintf(msg,"addhoser %s unlocked database.",InvokerAcct);
#if SYSLOG
	syslog(LOG_INFO,msg);
#endif
#if LOGTOFILE
	LogToFile(msg);
#endif
#endif
	return(0);
}

/*
 * create LockAllFile and return to caller if LockAllFile doesn't exist,
 * or the file exists and the contents of the file match InvokerAcct.
 * otherwise, print out an error and exit program.
 */
CheckLockAll() {
	int fd, retval;
	FILE *LOCKMSG;

	if ((fd=open(LockAllFile,O_RDONLY)) >= 0) {
		/* if read fails, we have a problem */
		if ((retval=read(fd,LockAcct,IACCTLEN)) >= 0) {
			if (retval >= 0) LockAcct[retval] = '\0';
			close(fd);
			/* file exists, print name of who locked it */
			if (STREQ(LockAcct,InvokerAcct))
				return;
			else {
				if (X_Doing == X_DAEMON)
					printf("500-Sorry, the database is locked by %s.\n",LockAcct);
				else
					fprintf(stderr,"Sorry, the database is locked by %s.\n",LockAcct);
				if ((LOCKMSG=fopen(LockMsgFile, "r"))==(FILE *)0) {
					if (errno == ENOENT) {
						CleanUp();
						exit(1);
					}
					SetSysErr();
					if (X_Doing == X_DAEMON)
						printf("500-fopen %s: %s",LockMsgFile,syserr);
					else
						fprintf(stderr,"fopen %s: %s",LockMsgFile,syserr);
					CleanUp();
					exit(1);
				}
				while (fgets(linein, LINEMAX, LOCKMSG)) {
					if (X_Doing == X_DAEMON)
						printf("500-%s",linein);
					else
						fputs(linein, stderr);
				}
				fclose(LOCKMSG);
				CleanUp();
				exit(1);
			}
		}
	} else if (errno == ENOENT)
		return;

	/* if we fell through to here, we need to get the error message */
	if (fd >= 0) (void) close(fd);
	SetSysErr();
	if (X_Doing == X_DAEMON)
		printf("500-open %s: %s\n",LockAllFile,syserr);
	else
		fprintf(stderr,"open %s: %s\n",LockAllFile,syserr);
	CleanUp();
	exit(1);
}

/*
 * RecLock() checks to see if file exists.  if it does then char pointer
 * is returned with username of who is locking the record.  if it doesn't
 * exist, then a null string is returned.  if an error occurs, then a null
 * pointer is returned and errmsg is set to the error.
 */
char *RecLock(mindex,file)
	int mindex;
	char *file;
{
	static int fd, retval;
	static char *cp, *cp2;

	if (*file == '\0') {
		/* this happens when a key points to a null block of data */
		sprintf(errmsg,"That key points to null data, please inform database administrator.");
		return((char *)NULL);
	}
	if ((cp=(char *)NewStr(file)) == (char *)NULL)
		FatalError("out of memory");
	for (cp2=cp; *cp2; cp2++)
		*cp2 = TOLOWER(*cp2);
	
	sprintf(path,"%s/%s.%s",RecLockDir,keys[fields[mindex]->keyindex]->str,cp);
	if ((fd=open(path, O_WRONLY|O_CREAT|O_EXCL, 0644)) < 0)	{
		/* if error isn't EEXIST, we have a problem */
		if (errno == EEXIST) {
			/* if open fails with read-only, we have a problem */
			if ((fd=open(path,O_RDONLY)) >= 0) {
				/* if read fails, we have a problem */
				if ((retval=read(fd,LockAcct,IACCTLEN)) >= 0) {
					if (retval >= 0) LockAcct[retval] = '\0';
					close(fd);
					/* file exists, return name of who locked it */
					return(LockAcct);
				}
			}
		}
		/* if we fell through to here, we need to get the error message */
		if (fd >= 0) (void) close(fd);
		SetSysErr();
		sprintf(errmsg,"open %s: %s",path,syserr);
		return((char *)NULL);
	}

	if (write(fd,InvokerAcct,strlen(InvokerAcct)) < 0) {
		if (fd >= 0) (void) close(fd);
		SetSysErr();
		sprintf(errmsg,"write %s: %s",path,syserr);
		unlink(path);
		return((char *)NULL);
	}
	(void) close(fd);

	if (!lockfile1) {
		lockfile1 = cp;
		lockmindex1 = mindex;
	} else {
		lockfile2 = cp;
		lockmindex2 = mindex;
	}
	return("");
}

/*
 * removes the lock file in RecLockDir by the name of the incoming file
 * variable.  file is converted to lowercase to insure case insensitivity
 * for keys.  lockfile1 or lockfile2 variable is cleared appropriately.
 */
RmRecLock(mindex,file)
	int mindex;
	char *file;
{
	static char *cp, *cp2;

	if (file[0] != '\0') {
		sprintf(path, "%s/%s.", RecLockDir, keys[fields[mindex]->keyindex]->str);
		for (cp=path; *cp; cp++);	/* position cp at null terminator */
		for (cp2=file; *cp2; cp++,cp2++)
			*cp = TOLOWER(*cp2);	/* copy file to end of path */
		*cp = '\0';					/* terminate path */
		(void) unlink(path);
		if (lockfile1) {
			if (STRCASEEQ(lockfile1,file)) {
				free(lockfile1);
				lockfile1 = (char *)NULL;
			}
		} else if (lockfile2) {
			if (STRCASEEQ(lockfile2,file)) {
				free(lockfile2);
				lockfile2 = (char *)NULL;
			}
		}
	}
}

/*
 * remove record locks (up to 2 could have been created by this program).
 */
RmAllRecLocks() {
	if (lockfile1) {
		RmRecLock(lockmindex1,lockfile1);
		free(lockfile1);
		lockfile1 = (char *)NULL;
	}
	if (lockfile2) {
		RmRecLock(lockmindex2,lockfile2);
		free(lockfile2);
		lockfile2 = (char *)NULL;
	}
}

/*
 * DBLock() creates a file in DBLockDir with a filename according the pid
 * of the current process, and writes the username of the user that
 * started this process into the file.  if an error occurs, it exits by
 * calling FatalError(), otherwise it returns normally to caller.
 */
DBLock() {
	int fd;
	int retval;

	sprintf(path, "%s/%d", DBLockDir, pid);
	if ((fd=open(path, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0)	{
		SetSysErr();
		sprintf(errmsg,"open %s: %s",path,syserr);
		FatalError(errmsg);
	}

	if (write(fd,InvokerAcct,strlen(InvokerAcct)) < 0) {
		if (fd >= 0) (void) close(fd);
		SetSysErr();
		sprintf(errmsg,"write %s: %s",path,syserr);
		FatalError(errmsg);
	}
	(void) close(fd);
}

/*
 * removes lock created by DBLock().
 */
RmDBLock() {
	sprintf(path,"%s/%d",DBLockDir,pid);
	unlink(path);
}

/*
 * recursively remove the file given in "f".  if an error occurs, put
 * message in errmsg and return -1, on success return 0.
 */
rm(f)
	char *f;
{
	char *p, pathbuf[MAXPATHLEN+1];
	struct stat statb;
	struct direct *dp;
	DIR *dfd;
	short fmode;

	if (stat(f,&statb) < 0)
		return(-1);
	else if ((statb.st_mode & S_IFMT) != S_IFDIR)
		return(unlink(f));

	if (!getcwd(pathbuf,MAXPATHLEN))
		return(-1);
	else if (chdir(f) < 0)
		return(-1);
	else if ((dfd = opendir(".")) == NULL)
		return(-1);

	for (dp = readdir(dfd); dp; dp = readdir(dfd)) {
		if (lstat((p=dp->d_name),&statb) < 0)
			continue;
		fmode = (statb.st_mode & 07777);
		switch (statb.st_mode & S_IFMT) {
			case S_IFDIR:
				if (*p == '.')
					if (!*(p+1) || (*(p+1) == '.' && !*(p+2)))
						break;
				if (rm(p) < 0)
					return(-1);
				break;
			default:
				if (unlink(p) < 0)
					return(-1);
				break;
		}
	}
	(void) closedir(dfd);
	if (chdir(pathbuf) < 0)
		return(-1);
	return(rmdir(f) < 0);
}
