#include "common.h"
#include "errors.h"
#include <string.h>

/*
 * displays a message on the last line of the screen and waits for 'y' or
 * 'n' input.  1 is returned for "y", 0 for "n".
 */
int PromptMsg(msg)
	char *msg;	/* prompt message to display */
{
	WINDOW *pwin;			/* cache current screen */
	char yesno;	/* user input (we only want the first character) */
	int x_old, y_old;

	pwin = curscreen->win;
	getyx(curscr, y_old, x_old);	/* save old cursor location */
	wmove(pwin,LastLine, 0);
	wclrtoeol(pwin);
	waddstr(pwin,msg);
	wrefresh(pwin);
	do {			/* read input until [YyNn] is entered */
		yesno = wgetch(pwin);
		if (index("yYnN",yesno) != NULL)
			break;
		(void) putc('\007', stdout);
	} while (1);

	wmove(pwin,LastLine, 0);
	wclrtoeol(pwin);
	wmove(pwin,y_old, x_old);
	wrefresh(pwin);

	return((yesno == 'y') || (yesno == 'Y'));
}

/*
 * show msg, and prompt user to hit any key.
 */
GetAnyKey(msg)
	char *msg;	/* prompt message to display */
{
	WINDOW *pwin;			/* cache current screen */
	char c;	/* user input (we only want the first character) */
	int x_old, y_old;

	pwin = curscreen->win;
	getyx(curscr, y_old, x_old);	/* save old cursor location */
	wmove(pwin,LastLine, 0);
	wclrtoeol(pwin);
	waddstr(pwin,msg);
	waddstr(pwin,", press any key...");
	wrefresh(pwin);
	c = wgetch(pwin); /* we don't really need to store the character in 'c' */

	wmove(pwin,LastLine, 0);
	wclrtoeol(pwin);
	wmove(pwin,y_old, x_old);
	wrefresh(pwin);

	return;
}

/*
 * beep, and print message to (y,x)=(LastLine,40).
 */
BeepErr(msg,restorepos)
	char *msg;
	int restorepos;
{
	int y_old, x_old;
	WINDOW *pwin;
	pwin = curscreen->win;

	if (restorepos) getyx(curscr,y_old,x_old);
	(void) putc('\007', stdout);
	wmove(pwin,LastLine,40);
	wstandout(pwin);
	waddstr(pwin,msg);
	wstandend(pwin);
	if (restorepos) wmove(pwin,y_old,x_old);
	wrefresh(pwin);
	wasbeeperr++;
}

/*
 * clear the message that was output with BeepErr.
 */
ClearBeepErr(restorepos)
	int restorepos;
{
	int y_old, x_old;
	WINDOW *pwin;

	wasbeeperr = 0;
	if (restorepos) getyx(curscr,y_old,x_old);
	pwin = curscreen->win;
	wmove(pwin,LastLine,40);
	wclrtoeol(pwin);
	if (restorepos) wmove(pwin,y_old,x_old);
	wrefresh(pwin);
}

/*
 *  DispMsg() displays a message on the screen at the given coordinates.
 *  If `revvideo' equals STANDOUT, the message is displayed in standout
 *  mode, otherwise it is displayed normally.
 */
DispMsg(y_row, x_col, revvideo, str)
	int y_row, x_col, revvideo;	/* coordinates & mode */
	char *str;			/* message to display */
{
	WINDOW *pwin;			/* cache current screen */
	int y_old, x_old;

	getyx(curscr, y_old, x_old);	/* save old cursor location */

	pwin = curscreen->win;

	wmove(pwin,y_row, x_col);		/* position cursor */
	wclrtoeol(pwin);			/* clear to EOL */
	if (revvideo == STANDOUT)
		wstandout(pwin);
	waddstr(pwin,str);	/* put string on current screen */
	if (revvideo == STANDOUT)
		wstandend(pwin);

	wmove(pwin,y_old, x_old);/* restore old cursor location */
	wrefresh(pwin);	/* show what we've done */
}

/*
 * RedrawScreen() redraws each line of input with the current information
 * for the field.
 */
RedrawScreen() {
	register scrfield_t *fptr;		/* cached pointer for fields[N] */
	register WINDOW *pwin;		/* cache current screen */
	register int i, j;			/* indices */
	screen_t *tempscreen;
	comment_t *cptr;
	specialscrfield_t *sfptr;

	tempscreen = firstscreen;
	do {
		pwin = tempscreen->win;
		fptr = tempscreen->begin;
		do {
			mvwaddstr(pwin,fptr->y_label,fptr->x_label,fptr->label);
			mvwaddstr(pwin,fptr->y_input,fptr->x_input,fields[fptr->index]->data);
			for(j=(fields[fptr->index]->len-strlen(fields[fptr->index]->data)); j > 0; j--)
				waddch(pwin,' ');
			cptr=tempscreen->comment;
			while (cptr != NULL) {
				if (cptr->type == X_Doing)
					mvwaddstr(pwin,cptr->y_pos,cptr->x_pos,cptr->text);
				cptr = cptr->next;
			}
			sfptr=tempscreen->specialscrfield;
			while (sfptr != NULL) {
				mvwaddstr(pwin,sfptr->y_label,sfptr->x_label,sfptr->label);
				if (sfptr->index == i_moddate) {
					if (moddatestr != (char *)NULL)
						mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,moddatestr);
				} else if (sfptr->index == i_createdate) {
					if (createdatestr != (char *)NULL)
						mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,createdatestr);
				} else if (sfptr->index == i_creator) {
					mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,unamewhitespace);
					if (fields[i_creator]->data != (char *)NULL)
						mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,fields[i_creator]->data);
				} else if (sfptr->index == i_modifier) {
					mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,unamewhitespace);
					if (fields[i_modifier]->data != (char *)NULL)
						mvwaddstr(pwin,sfptr->y_pos,sfptr->x_pos,fields[i_modifier]->data);
				}
				sfptr = sfptr->next;
			}
			fptr = fptr->next;
		} while (fptr != tempscreen->end->next);
		tempscreen = tempscreen->next;
	} while (tempscreen != firstscreen);

	wrefresh(curscreen->win);		/* show them current screen */
}

/*
 * redraw the data for one field.
 */
RedrawField(fptr)
	scrfield_t *fptr;
{
	register WINDOW *pwin;
	register int j;

	pwin = curscreen->win;
	mvwaddstr(pwin,fptr->y_input,fptr->x_input,fields[fptr->index]->data);
	for (j=(fields[fptr->index]->len-strlen(fields[fptr->index]->data)); j > 0; j--)
		waddch(pwin,' ');
}

/*
 * show status on SecLastLine on each screen about the current record.
 * show if it's new, existing, view-only, and the current primary key.
 */
RedrawStatus() {
	register WINDOW *pwin;		/* cache current screen */
	screen_t *tempscreen;
	int y_old,x_old;

	if (X_Doing == X_RM) return;
	getyx(curscr,y_old,x_old);	/* save old cursor location */
	if (recflags&REC_NEW) {
		if (*fields[i_pkey]->data)
			sprintf(msg,"New Record (%s).",fields[i_pkey]->data);
		else
			sprintf(msg,"New Record (?).");
	} else if (recflags&REC_EXISTS) {
		if (recflags&REC_VALID) {
			if (STREQ(fields[i_pkey]->origdata,fields[i_pkey]->data))
				sprintf(msg,"Existing Record (%s).",fields[i_pkey]->data);
			else
				sprintf(msg,"Existing Record (%s -> %s).",fields[i_pkey]->origdata,fields[i_pkey]->data);
		} else {
			if (*fields[i_pkey]->data)
				sprintf(msg,"Existing Record - View Only (%s).",fields[i_pkey]->data);
			else
				sprintf(msg,"View Only (?).");
		}
	} else
		*msg = '\0';

	tempscreen = firstscreen;
	do {
		pwin = tempscreen->win;
		tempscreen = tempscreen->next;
		wmove(pwin,SecLastLine,0);		/* position cursor */
		wclrtoeol(pwin);			/* clear to EOL */
		waddstr(pwin,msg);	/* put string on current screen */
	} while (tempscreen != firstscreen);
	wmove(pwin,y_old,x_old);/* restore old cursor location */
}

/*
 * Move cursor to bottom of screen and exit curses package.
 */
ResetScreen() {
	WINDOW *pwin;			/* cache current screen */

	pwin = curscreen->win;
	wmove(pwin,LastLine,0);
	wclrtoeol(pwin);
	wrefresh(pwin);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
}

/*
 * show a message to user, and wait for space bar to be hit.
 */
ShowMessage(filename,ask)
	char *filename;
	int ask;
{
	FILE *fp;
	int row;
	char ichar, *cp;

#ifdef REMOTE
	fp = REMOTEOUT;
	fprintf(REMOTEIN,"%s %s\n",C_CAT,S_MOTDFILE);
	fflush(REMOTEIN);
#else
	if ((fp=fopen(filename, "r"))==(FILE *)0)
		return; /* if we can't open it, then who cares? */
#endif

	row = 0;			/* start at top of screen */

	(void) wclear(curscr);
	(void) wstandend(curscr);

	while (fgets(linein, LINEMAX, fp) != NULL) {
#ifdef REMOTE
		if (linein[3] != '-') {
			if (linein[0] == '2')
				break;
			else
				return;
		}
		cp = linein + 4;
#else
		cp = linein;
#endif
		linein[LastCol] = '\0';			/* truncate line */
		if (row == 0)
			(void) wstandout(curscr);
		mvwaddstr(curscr, row, 0, cp);
		if (row == 0)
			(void) wstandend(curscr);
		if (++row >= LastLine)
			break;
	}
	if (ask) {
		wmove(curscreen,LastLine,40);
		wstandout(curscr);
		mvwaddstr(curscr, LastLine, 40, "<Hit space key when finished reading>");
		wstandend(curscr);
		(void) wrefresh(curscr);
		while (1) {
			ichar = wgetch(curscr);
			if (ichar == ' ')
				break;
			else
				putc('\007', stdout);
		}
		touchwin(curscreen->win);
		wrefresh(curscreen->win);
	} else
		HitAnyKey();
}

/*
 * prompt user to hit any key.
 */
HitAnyKey() {
	(void) wstandout(curscr);
	mvwaddstr(curscr, LastLine, 40, "<Hit any key to continue>");
	(void) wstandend(curscr);
	(void) wrefresh(curscr);
	(void) wgetch(curscr);
	touchwin(curscreen->win);
	wrefresh(curscreen->win);
}

/*
 * bring up choice box for user to choose from.  return char pointer to
 * string that was chosen, or NULL if <ESC> was pressed.
 */
char *ChoiceBox(fptr)
	scrfield_t *fptr;
{
	WINDOW *win;
	int y_old, x_old, y, x;
	int y_origin, x_origin;
	char **list, *chosen;
	int n, longest, *lens, i, j, k, previ, width, topi, prevtopi, height;
	int fieldlen, choices, maxchosen, nchosen;
	char c;

	n = fields[fptr->index]->choice->n;
	list = fields[fptr->index]->choice->list;
	maxchosen = fields[fptr->index]->choice->maxchosen;

	if (n == 0)
		return((char *)NULL);
	if (n == 1) {
		fields[fptr->index]->choice->nchosen = 1;
		return(strcpy(tmpfieldspace,list[0]));
	}

	nchosen = fields[fptr->index]->choice->nchosen;
	chosen = fields[fptr->index]->choice->chosen;
	longest = fields[fptr->index]->choice->longest;
	lens = fields[fptr->index]->choice->lens;
	fieldlen = strlen(fields[fptr->index]->data);
	width = (longest < 10) ? 10 : longest+2;
	y_origin = fptr->y_input;
	if (y_origin+n+3 >= SecLastLine) y_origin = SecLastLine-n-3;
	if (y_origin<0) {
		y_origin = 0;
		height = SecLastLine-2;
	} else
		height = n;
	x_origin = fptr->x_input+fields[fptr->index]->len;
	if (x_origin+width+2 >= LastCol) x_origin = LastCol-width-2;
	/* save old cursor location */
	getyx(curscr, y_old, x_old);

	for (i=0; i<width; i++)
		tmpfieldspace[i] = ' ';
	tmpfieldspace[width] = '\0';

	if ((win=newwin(height+2,width+2,y_origin,x_origin))==ERR)
		FatalError("out of memory");
	leaveok(win, 0);
	scrollok(win, 0);

	wstandout(win);
	mvwaddch(win,0,0,'+');
	for (i=0; i<width; i++)
		waddch(win,'-');
	waddch(win,'+');
	for (i=1; i<=height; i++) {
		mvwaddch(win,i,0,'|');
		mvwaddch(win,i,width+1,'|');
	}
	mvwaddch(win,height+1,0,'+');
	for (i=0; i<width; i++)
		waddch(win,'-');
	waddch(win,'+');
	wstandend(win);

	prevtopi = -1;
	i = previ = topi = 0;
	while (1) {
		if (topi != prevtopi) {
			/* j is the temporary index of choices */
			/* y is the y coordinate of the window */
			j = topi;
			for (y=1; y<=height; y++,j++) {
				if (i==j) continue;
				if (chosen[j])
					mvwaddch(win,y,1,'>');
				else
					mvwaddch(win,y,1,' ');
				waddstr(win,list[j]);
				k = width-lens[j]-2;
				if (k > 0) {
					tmpfieldspace[k] = '\0';
					mvwaddstr(win,y,width-k,tmpfieldspace);
					tmpfieldspace[k] = ' ';
				}
			}
			k = width-lens[i]-2;
			if (k > 0) {
				tmpfieldspace[k] = '\0';
				mvwaddstr(win,i-topi+1,width-k,tmpfieldspace);
				tmpfieldspace[k] = ' ';
			}
			if (chosen[i])
				mvwaddch(win,i-topi+1,1,'>');
			else
				mvwaddch(win,i-topi+1,1,' ');
			wstandout(win);
			waddstr(win,list[i]);
			wstandend(win);
			wrefresh(win);
			prevtopi = topi;
		} else if (i != previ) {
			mvwaddstr(win,previ-topi+1,2,list[previ]);
			wstandout(win);
			mvwaddstr(win,i-topi+1,2,list[i]);
			wstandend(win);
			previ = i;
			wrefresh(win);
		}
		c = wgetch(win);
		if (wasbeeperr) ClearBeepErr(0);
		switch (c) {
			case 'k': case '\013': /* move highlight bar up */
				if (i == 0)
					BeepErr("Can't move any further up.",0);
				else if (i == topi)
					topi = previ = --i;
				else
					i--;
				break;
			case 'j': case '\012': /* move highlight bar down */
				if (i == n-1)
					BeepErr("Can't move any further down.",0);
				else if (i == topi+height-1) {
					previ = ++i;
					topi++;
				} else
					i++;
				break;
			case '\002':           /* ^B, backward a screen */
				if (topi == 0) {
					/* if highlight is not at top, then put it there */
					if (i != topi)
						i = topi;
					else
						BeepErr("Can't move any further back.",0);
				} else {
					topi -= height;
					if (topi < 0)
						topi = 0;
					i -= prevtopi - topi;
					previ = i;
				}
				break;
			case '\006':           /* ^F, forward a screen */
				if (topi+height == n) {
					/* if highlight is not at bottom, then put it there */
					if (i != n-1)
						i = n-1;
					else
						BeepErr("Can't move any further forward.",0);
				} else {
					topi += height;
					if (topi > n-height)
						topi = n-height;
					i += topi - prevtopi;
					previ = i;
				}
				break;
			case '\033':           /* <ESC>, quit loop with no choice */
				i = previ = n;
				break;
			case '\r':             /* <RET>, choose, unchoose or quit */
				previ = n;
				break;
			case ' ':             /* <SPACE>, choose, unchoose or quit */
				if (maxchosen == 1 && !chosen[i]) {
					bzero(chosen,n);
					fieldlen = lens[i];
					nchosen = 1;
					chosen[i] = (char)1;
					previ = n;
					break;
				}
				if (chosen[i]) {
					fieldlen -= lens[i];
					if (fieldlen > 0) fieldlen--;
					nchosen--;
					chosen[i] = (char)0;
				} else {
					if (maxchosen > 0 && nchosen >= maxchosen) {
						BeepErr("No more choices allowed in this field.",0);
						break;
					}
					if (fieldlen != 0) fieldlen++;
					fieldlen += lens[i];
					if (fieldlen > fields[fptr->index]->len) {
						fieldlen -= lens[i];
						if (fieldlen == 1) fieldlen = 0;
						BeepErr("That is too big to fit in this field.",0);
						break;
					}
					nchosen++;
					chosen[i] = (char)1;
				}
				if (i == topi+height-1 && i != n-1) {
					previ = ++i;
					topi++;
				} else {
					getyx(curscr,y,x);
					if (chosen[i])
						mvwaddch(win,i-topi+1,1,'>');
					else
						mvwaddch(win,i-topi+1,1,' ');
					if (i<n-1)
						i++;
					else
						wmove(win,y-y_origin,x-x_origin);
					wrefresh(win);
				}
				break;
			case '?':
				(void) HelpThem(I_GENHELP,0);
				touchwin(win);
				wrefresh(win);
				break;
			case '\011':             /* <TAB>, help */
				(void) HelpThem(I_CHOICEHELP,0);
				touchwin(win);
				wrefresh(win);
				break;
			case '\014':             /* ^L -- redraw the screen */
				wrefresh(curscr);
				break;
			default:               /* error, beep */
				BeepErr("Invalid key pressed.",0);
				break;
		}
		if (previ == n) break;
	}
	delwin(win);

	/* restore screen appearance below where the choice box was */
	wmove(curscreen->win,y_old,x_old);
	touchwin(curscreen->win);
	wrefresh(curscreen->win);

	if (c == '\033')
		return((char *)NULL);
	fieldlen = 0;
	for (i=0; i<n; i++)
		if (chosen[i]) {
			if (fieldlen != 0)
				tmpfieldspace[fieldlen++] = ' ';
			bcopy(list[i],&(tmpfieldspace[fieldlen]),lens[i]);
			fieldlen += lens[i];
		}
	tmpfieldspace[fieldlen] = '\0';
	fields[fptr->index]->choice->nchosen = nchosen;

	return(tmpfieldspace);
}
