/*	This file contains the source of the 'mt' program intended for
	Linux systems. The program supports the basic mt commands found
	in most Unix-like systems. In addition to this the program
	supports several commands designed for use with the Linux SCSI
	tape drive.

	Maintained by Kai Mkisara (email Kai.Makisara@metla.fi)
	Copyright by Kai Mkisara, 1998. The program may be distributed
	according to the GNU Public License

	Last Modified: Sun Aug 16 09:51:17 1998 by makisara@home
*/

#ifndef lint
static char rcsid[] = "$Id: /usr2/users/makisara/src/sys/mt-st-0.5b/mt.c at Sun Aug 16 09:51:17 1998 by makisara@kai.makisara.fi$";
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>

#define VERSION "0.5b"

/* Set to one to include the datcompression command (not necessary in
   Linux 2.x) */
#define DATCOMPRESSION 0

typedef int (* cmdfunc)(/* int, struct cmdef_tr *, int, char ** */);

typedef struct cmdef_tr {
    char *cmd_name;
    int cmd_code;
    cmdfunc cmd_function;
    int cmd_count_bits;
    unsigned char cmd_fdtype;
    unsigned char arg_cnt;
} cmdef_tr;

#define NO_FD      0
#define FD_RDONLY  1
#define FD_RDWR    2

#define NO_ARGS       0
#define ONE_ARG       1
#define TWO_ARGS      2
#define MULTIPLE_ARGS 255

#define DO_BOOLEANS    1002
#define SET_BOOLEANS   1003
#define CLEAR_BOOLEANS 1004

static void usage(int);
static int do_standard(int, cmdef_tr *, int, char **);
static int do_drvbuffer(int, cmdef_tr *, int, char **);
static int do_options(int, cmdef_tr *, int, char **);
static int do_tell(int, cmdef_tr *, int, char **);
static int do_partseek(int, cmdef_tr *, int, char **);
static int do_status(int, cmdef_tr *, int, char **);
static int print_densities(int, cmdef_tr *, int, char **);
static int do_asf(int, cmdef_tr *, int, char **);
#if DATCOMPRESSION
static int do_dat_compression(int, cmdef_tr *, int, char **);
#endif


static cmdef_tr cmds[] = {
    { "weof",		MTWEOF,	  do_standard, 0, FD_RDWR,   ONE_ARG },
    { "wset",		MTWSM,	  do_standard, 0, FD_RDWR,   ONE_ARG },
    { "eof",		MTWEOF,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "fsf",		MTFSF,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "fsfm",		MTFSFM,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "bsf",		MTBSF,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "bsfm",		MTBSFM,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "fsr",		MTFSR,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "bsr",		MTBSR,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "fss",		MTFSS,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "bss",		MTBSS,	  do_standard, 0, FD_RDONLY, ONE_ARG },
    { "rewind",		MTREW,	  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "offline",	MTOFFL,	  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "rewoffl",	MTOFFL,	  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "retension",	MTRETEN,  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "eod",		MTEOM,	  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "seod",		MTEOM,	  do_standard, 0, FD_RDONLY, NO_ARGS },
    { "seek",		MTSEEK,   do_standard, 0, FD_RDONLY, ONE_ARG },
    { "tell",		MTTELL,	  do_tell,     0, FD_RDONLY, NO_ARGS },
    { "status",		MTNOP,	  do_status,   0, FD_RDONLY, NO_ARGS },
    { "erase",		MTERASE,  do_standard, 0, FD_RDWR,   NO_ARGS },
    { "setblk",		MTSETBLK, do_standard, 0, FD_RDONLY, ONE_ARG },
#ifdef MTLOCK
    { "lock",		MTLOCK,   do_standard, 0, FD_RDONLY, NO_ARGS },
    { "unlock", 	MTUNLOCK, do_standard, 0, FD_RDONLY, NO_ARGS },
#endif
#ifdef MTLOAD
    { "load",		MTLOAD,	  do_standard, 0, FD_RDONLY, NO_ARGS },
#endif
#ifdef MTCOMPRESSION
    { "compression",	MTCOMPRESSION,  do_standard,  0, FD_RDONLY, ONE_ARG},
#endif
    { "setdensity",	MTSETDENSITY,   do_standard,  0, FD_RDONLY, ONE_ARG},
    { "drvbuffer",	MTSETDRVBUFFER, do_drvbuffer, 0, FD_RDONLY, ONE_ARG},
    { "stwrthreshold",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_WRITE_THRESHOLD,
      FD_RDONLY, ONE_ARG},
    { "stoptions",	DO_BOOLEANS,    do_options,   0, FD_RDONLY,
      MULTIPLE_ARGS},
#ifdef MT_ST_SETBOOLEANS
    { "stsetoptions",   SET_BOOLEANS,   do_options,   0, FD_RDONLY,
      MULTIPLE_ARGS},
    { "stclearoptions", CLEAR_BOOLEANS, do_options,   0, FD_RDONLY,
      MULTIPLE_ARGS},
    { "defblksize",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_DEF_BLKSIZE,
      FD_RDONLY, ONE_ARG},
    { "defdensity",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_DEF_DENSITY,
      FD_RDONLY, ONE_ARG},
    { "defdrvbuffer",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_DEF_DRVBUFFER,
      FD_RDONLY, ONE_ARG},
    { "defcompression", MTSETDRVBUFFER, do_drvbuffer, MT_ST_DEF_COMPRESSION,
      FD_RDONLY, ONE_ARG},
#endif
#ifdef MT_ST_SET_TIMEOUT
    { "sttimeout",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_SET_TIMEOUT,
      FD_RDONLY, ONE_ARG},
    { "stlongtimeout",	MTSETDRVBUFFER, do_drvbuffer, MT_ST_SET_LONG_TIMEOUT,
      FD_RDONLY, ONE_ARG},
#endif
    { "densities",	0, print_densities,     0, NO_FD,     NO_ARGS},
#if DATCOMPRESSION
    { "datcompression", 0, do_dat_compression,  0, FD_RDWR,   ONE_ARG},
#endif
#ifdef MTSETPART
    { "setpartition",	MTSETPART, do_standard, 0, FD_RDONLY, ONE_ARG},
    { "mkpartition",	MTMKPART,  do_standard, 0, FD_RDWR,   ONE_ARG},
    { "partseek",	0,         do_partseek, 0, FD_RDONLY, TWO_ARGS},
#endif
    { "asf",		0,         do_asf, MTREW,  FD_RDONLY, ONE_ARG},
    { NULL, 0, 0, 0 }
};


static struct densities {
    int code;
    char *name;
} density_tbl[] = {
    {0x00, "default"},
    {0x01, "NRZI (800 bpi)"},
    {0x02, "PE (1600 bpi)"},
    {0x03, "GCR (6250 bpi)"},
    {0x04, "QIC-11"},
    {0x05, "QIC-45/60 (GCR, 8000 bpi)"},
    {0x06, "PE (3200 bpi)"},
    {0x07, "IMFM (6400 bpi)"},
    {0x08, "GCR (8000 bpi)"},
    {0x09, "GCR /37871 bpi)"},
    {0x0a, "MFM (6667 bpi)"},
    {0x0b, "PE (1600 bpi)"},
    {0x0c, "GCR (12960 bpi)"},
    {0x0d, "GCR (25380 bpi)"},
    {0x0f, "QIC-120 (GCR 10000 bpi)"},
    {0x10, "QIC-150/250 (GCR 10000 bpi)"},
    {0x11, "QIC-320/525 (GCR 16000 bpi)"},
    {0x12, "QIC-1350 (RLL 51667 bpi)"},
    {0x13, "DDS (61000 bpi)"},
    {0x14, "EXB-8200 (RLL 43245 bpi)"},
    {0x15, "EXB-8500 or QIC-1000"},
    {0x16, "MFM 10000 bpi"},
    {0x17, "MFM 42500 bpi"},
    {0x22, "QIC-2GB"},
    {0x24, "DDS-2"},
    {0x25, "DDS-3"},
    {0x45, "QIC-3095-MC (TR-4)"},
    {140, "EXB-8505 compressed"},
    {144, "EXB-8205 compressed"},
    {-1, NULL}};


static struct booleans {
    char *name;
    unsigned long bitmask;
    char *expl;
} boolean_tbl[] = {
    {"buffer-writes", MT_ST_BUFFER_WRITES, "buffered writes"},
    {"async-writes",  MT_ST_ASYNC_WRITES,  "asynchronous writes"},
    {"read-ahead",    MT_ST_READ_AHEAD,    "read-ahead for fixed block size"},
    {"debug",         MT_ST_DEBUGGING,     "debugging (if compiled into driver)"},
    {"two-fms",       MT_ST_TWO_FM,        "write two filemarks when file closed"},
    {"fast-eod",      MT_ST_FAST_MTEOM, "space directly to eod (and lose file number)"},
    {"auto-lock",     MT_ST_AUTO_LOCK,     "automatically lock/unlock drive door"},
    {"def-writes",    MT_ST_DEF_WRITES,    "the block size and density are for writes"},
    {"can-bsr",       MT_ST_CAN_BSR,       "drive can space backwards well"},
    {"no-blklimits",  MT_ST_NO_BLKLIMS,    "drive doesn't support read block limits"},
    {"can-partitions",MT_ST_CAN_PARTITIONS,"drive can handle partitioned tapes"},
    {"scsi2logical",  MT_ST_SCSI2LOGICAL,  "logical block addresses used with SCSI-2"},
#ifdef MT_ST_SYSV
    {"sysv",	      MT_ST_SYSV,	   "enable the SystemV semantics"},
#endif
    {NULL, 0}};

static char *tape_name;   /* The tape name for messages */


	int
main(int argc, char **argv)
{
    int mtfd, cmd_code, i, argn, len;
    char *cmdstr;
    cmdef_tr *comp, *comp2;

    for (argn=1; argn < argc; argn++)
	if (*argv[argn] == '-')
	    switch (*(argv[argn] + 1)) {
	    case 'f':
	    case 't':
		argn += 1;
		if (argn >= argc) {
		    usage(0);
		    exit(1);
		}
		tape_name = argv[argn];
		break;
	    case 'h':
		usage(1);
		exit(0);
		break;
	    case 'v':
		printf("mt-st v. %s\n", VERSION);
		exit(0);
		break;
	    default:
		usage(0);
		exit(1);
	}
	else
	    break;

    if (tape_name == NULL && (tape_name = getenv("TAPE")) == NULL)
	tape_name = DEFTAPE;
       
    if (argn >= argc ) {
	usage(0);
	exit(1);
    }
    cmdstr = argv[argn++];

    len = strlen(cmdstr);
    for (comp = cmds; comp->cmd_name != NULL; comp++)
	if (strncmp(cmdstr, comp->cmd_name, len) == 0)
	    break;
    if (comp->cmd_name == NULL) {
	fprintf(stderr, "mt: unknown command \"%s\"\n", cmdstr);
	usage(1);
	exit(1);
    }
    if (len != strlen(comp->cmd_name)) {
	for (comp2 = comp + 1; comp2->cmd_name != NULL; comp2++)
	    if (strncmp(cmdstr, comp2->cmd_name, len) == 0)
		break;
	if (comp2->cmd_name != NULL) {
	    fprintf(stderr, "mt: ambiguous command \"%s\"\n", cmdstr);
	    usage(1);
	    exit(1);
	}
    }
    if (comp->arg_cnt != MULTIPLE_ARGS && comp->arg_cnt < argc - argn) {
	fprintf(stderr, "mt: too many arguments for the command '%s'.\n",
		comp->cmd_name);
	exit(1);
    }
    cmd_code = comp->cmd_code;

    if (comp->cmd_fdtype != NO_FD) {
	if ((mtfd = open(tape_name,
			 comp->cmd_fdtype == FD_RDONLY ? O_RDONLY : O_RDWR)) < 0) {
	    perror(tape_name);
	    exit(1);
	}
    }
    else
	mtfd = (-1);

    if (comp->cmd_function != NULL)
	i = comp->cmd_function(mtfd, comp, argc - argn,
			       (argc - argn > 0 ? argv + argn : NULL));
    else {
	fprintf(stderr, "mt: Internal error: command without function.\n");
	i = 1;
    }

    if (mtfd >= 0)
	close(mtfd);
    return i;
}


	static void
usage(int explain)
{
    int ind;
    char line[100];

    fprintf(stderr, "usage: mt [-v] [-h] [ -f device ] command [ count ]\n");
    if (explain) {
	for (ind=0; cmds[ind].cmd_name != NULL; ) {
	    if (ind == 0)
		strcpy(line, "commands: ");
	    else
		strcpy(line, "          ");
	    for ( ; cmds[ind].cmd_name != NULL; ind++) {
		strcat(line, cmds[ind].cmd_name);
		if (cmds[ind+1].cmd_name != NULL)
		    strcat(line, ", ");
		else
		    strcat(line, ".");
		if (strlen(line) >= 70 || cmds[ind+1].cmd_name == NULL) {
		    fprintf(stderr, "%s\n", line);
		    ind++;
		    break;
		}
	    }
	}
    }
}



/* Do a command that simply feeds an argument to the MTIOCTOP ioctl */
	static int
do_standard(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtop mt_com;

    mt_com.mt_op = cmd->cmd_code;
    mt_com.mt_count = (argc > 0 ? strtol(*argv, NULL, 0) : 1);
    mt_com.mt_count |= cmd->cmd_count_bits;
    if (mt_com.mt_count < 0) {
	fprintf(stderr, "mt: negative repeat count\n");
	return 1;
    }
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    return 0;
}


/* The the drive buffering and other things with this (highly overloaded)
   ioctl function. (See also do_options below.) */
	static int
do_drvbuffer(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtop mt_com;

    mt_com.mt_op = MTSETDRVBUFFER;
    mt_com.mt_count = (argc > 0 ? strtol(*argv, NULL, 0) : 1);
    if ((cmd->cmd_count_bits & MT_ST_OPTIONS) == MT_ST_DEF_OPTIONS)
	mt_com.mt_count &= 0xfffff;
#ifdef MT_ST_TIMEOUTS
    else if ((cmd->cmd_count_bits & MT_ST_OPTIONS) == MT_ST_TIMEOUTS)
	mt_com.mt_count &= 0x7ffffff;
#endif
    else
	mt_com.mt_count &= 0xfffffff;
    mt_com.mt_count |= cmd->cmd_count_bits;
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    return 0;
}


/* Set the tape driver options */
	static int
do_options(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    int i, an, len;
    struct mtop mt_com;

    mt_com.mt_op = MTSETDRVBUFFER;
    if (argc == 0)
	mt_com.mt_count = 0;
    else if (isdigit(**argv))
	mt_com.mt_count = strtol(*argv, NULL, 0) & ~MT_ST_OPTIONS;
    else
	for (an = 0, mt_com.mt_count = 0; an < argc; an++) {
	    len = strlen(argv[an]);
	    for (i=0; boolean_tbl[i].name != NULL; i++)
		if (!strncmp(boolean_tbl[i].name, argv[an], len)) {
		    mt_com.mt_count |= boolean_tbl[i].bitmask;
		    break;
		}
	    if (boolean_tbl[i].name == NULL) {
		fprintf(stderr, "Illegal property name '%s'.\n", argv[an]);
		fprintf(stderr, "The implemented property names are:\n");
		for (i=0; boolean_tbl[i].name != NULL; i++)
		    fprintf(stderr, "  %9s -> %s\n", boolean_tbl[i].name,
			    boolean_tbl[i].expl);
		return 1;
	    }
	    if (len != strlen(boolean_tbl[i].name))
		for (i++ ; boolean_tbl[i].name != NULL; i++)
		    if (!strncmp(boolean_tbl[i].name, argv[an], len)) {
			fprintf(stderr, "Property name '%s' ambiguous.\n",
				argv[an]);
			return 1;
		    }
	}

    switch (cmd->cmd_code) {
    case DO_BOOLEANS:
	mt_com.mt_count |= MT_ST_BOOLEANS;
	break;
    case SET_BOOLEANS:
	mt_com.mt_count |= MT_ST_SETBOOLEANS;
	break;
    case CLEAR_BOOLEANS:
	mt_com.mt_count |= MT_ST_CLEARBOOLEANS;
	break;
    }
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    return 0;
}


/* Tell where the tape is */
	static int
do_tell(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtpos mt_pos;

    if (ioctl(mtfd, MTIOCPOS, (char *)&mt_pos) < 0) {
	perror(tape_name);
	return 2;
    }
    printf("At block %ld.\n", mt_pos.mt_blkno);
    return 0;
}


/* Position the tape to a specific location within a specified partition */
	static int
do_partseek(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtop mt_com;

    mt_com.mt_op = MTSETPART;
    mt_com.mt_count = (argc > 0 ? strtol(*argv, NULL, 0) : 0);
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    mt_com.mt_op = MTSEEK;
    mt_com.mt_count = (argc > 1 ? strtol(argv[1], NULL, 0) : 0);
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    return 0;
}


/* Position to start of file n. This might be implemented more intelligently
   some day. */
	static int
do_asf(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtop mt_com;

    mt_com.mt_op = MTREW;
    mt_com.mt_count = 1;
    if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	perror(tape_name);
	return 2;
    }
    mt_com.mt_count = (argc > 0 ? strtol(*argv, NULL, 0) : 0);
    if (mt_com.mt_count > 0) {
	mt_com.mt_op = MTFSF;
	if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
	    perror(tape_name);
	    return 2;
	}
    }
    return 0;
}


/*** Decipher the status ***/

	static int
do_status(int mtfd, cmdef_tr *cmd, int argc, char **argv)
{
    struct mtget status;
    int dens, i;
    char *type, *density;

    if (ioctl(mtfd, MTIOCGET, (char *)&status) < 0) {
	perror(tape_name);
	return 2;
    }

    if (status.mt_type == MT_ISSCSI1)
	type = "SCSI 1";
    else if (status.mt_type == MT_ISSCSI2)
	type = "SCSI 2";
    else
	type = NULL;
    if (type == NULL) {
	if (status.mt_type & 0x800000)
	    printf ("qic-117 drive type = 0x%05lx\n", status.mt_type & 0x1ffff);
	else
	    printf("Unknown tape drive type (type code %ld)\n", status.mt_type);
	printf("File number=%d, block number=%d.\n",
	       status.mt_fileno, status.mt_blkno);
	printf("mt_resid: %ld, mt_erreg: 0x%lx\n",
	       status.mt_resid, status.mt_erreg);
	printf("mt_dsreg: 0x%lx, mt_gstat: 0x%lx\n",
	       status.mt_dsreg, status.mt_gstat);
    }
    else {
	printf("%s tape drive:\n", type);
	if (status.mt_type == MT_ISSCSI2)
	    printf("File number=%d, block number=%d, partition=%ld.\n",
		   status.mt_fileno, status.mt_blkno, (status.mt_resid & 0xff));
	else
	    printf("File number=%d, block number=%d.\n",
		   status.mt_fileno, status.mt_blkno);
	if (status.mt_type == MT_ISSCSI1 ||
	    status.mt_type == MT_ISSCSI2) {
	    dens = (status.mt_dsreg & MT_ST_DENSITY_MASK) >> MT_ST_DENSITY_SHIFT;
	    density = "unknown to this mt";
	    for (i=0; density_tbl[i].code >= 0; i++)
		if (density_tbl[i].code == dens) {
		    density = density_tbl[i].name;
		    break;
		}
	    printf("Tape block size %ld bytes. Density code 0x%x (%s).\n",
		   ((status.mt_dsreg & MT_ST_BLKSIZE_MASK) >> MT_ST_BLKSIZE_SHIFT),
		   dens, density);

	    printf("Soft error count since last status=%ld\n",
		   (status.mt_erreg & MT_ST_SOFTERR_MASK) >> MT_ST_SOFTERR_SHIFT);
	}
    }

    printf("General status bits on (%lx):\n", status.mt_gstat);
    if (GMT_EOF(status.mt_gstat))
	printf(" EOF");
    if (GMT_BOT(status.mt_gstat))
	printf(" BOT");
    if (GMT_EOT(status.mt_gstat))
	printf(" EOT");
    if (GMT_SM(status.mt_gstat))
	printf(" SM");
    if (GMT_EOD(status.mt_gstat))
	printf(" EOD");
    if (GMT_WR_PROT(status.mt_gstat))
	printf(" WR_PROT");
    if (GMT_ONLINE(status.mt_gstat))
	printf(" ONLINE");
    if (GMT_D_6250(status.mt_gstat))
	printf(" D_6250");
    if (GMT_D_1600(status.mt_gstat))
	printf(" D_1600");
    if (GMT_D_800(status.mt_gstat))
	printf(" D_800");
    if (GMT_DR_OPEN(status.mt_gstat))
	printf(" DR_OPEN");	  
    if (GMT_IM_REP_EN(status.mt_gstat))
	printf(" IM_REP_EN");
    printf("\n");
    return 0;
}


/* Print a list of possible density codes */
	static int
print_densities(int fd, cmdef_tr *cmd, int argc, char **argv)
{
    int i;

    printf("Some SCSI tape density codes:\ncode   explanation\n");
    for (i=0; density_tbl[i].code >= 0; i++)
	printf("0x%02x   %s\n", density_tbl[i].code, density_tbl[i].name);
    return 0;
}



#if DATCOMPRESSION

/*** Get and set the DAT compression (Mode Page 15) ***/

#if 0
#include <linux/scsi.h>
#else
/* The file scsi.h is not visible to the users after 1.3.98. The necessary
   SCSI constants are defined here (not so bad because these are constants
   defined in the SCSI standard. */
#define MODE_SELECT           0x15
#define MODE_SENSE            0x1a
#endif

#ifndef SCSI_IOCTL_SEND_COMMAND
/* This has not been defined in a public include file */
#define SCSI_IOCTL_SEND_COMMAND 1
#endif

#define COMPRESSION_PAGE        0x0f
#define COMPRESSION_PAGE_LENGTH 16

#define IOCTL_HEADER_LENGTH 8
#define CMD_6_LENGTH        6
#define MODE_HEADER_OFFSET  (IOCTL_HEADER_LENGTH + CMD_6_LENGTH)
#define MODE_HEADER_LENGTH  4

#define DCE_MASK  0x80
#define DCC_MASK  0x40
#define RED_MASK  0x60


static int read_mode_page(int fn, int page, int length, unsigned char *buffer,
			  int do_mask)
{
    int result, *ip;
    unsigned char tmpbuffer[30], *cmd;

    memset(tmpbuffer, 0, IOCTL_HEADER_LENGTH + CMD_6_LENGTH);
    ip = (int *)&(tmpbuffer[0]);
    *ip = 0;
    *(ip+1) = length + MODE_HEADER_LENGTH;

    cmd = &(tmpbuffer[8]);
    cmd[0] = MODE_SENSE;
    cmd[1] = 8;
    cmd[2] = page;
    if (do_mask)
	cmd[2] |= 0x40;  /* Get changeable parameter mask */
    cmd[4] = length + MODE_HEADER_LENGTH;

    result = ioctl(fn, SCSI_IOCTL_SEND_COMMAND, tmpbuffer);
    if (result) {
	if (getuid() != 0)
	    fprintf(stderr, "%s: This operation is only allowed for root.\n",
		    tape_name);
	else
	    fprintf(stderr, "%s: Can't read SCSI mode page.\n", tape_name);
	return 0;
    }
    memcpy(buffer, tmpbuffer + IOCTL_HEADER_LENGTH, length + MODE_HEADER_LENGTH);
    return 1;
}


static int write_mode_page(int fn, int page, int length, unsigned char *buffer)
{
    int result, *ip;
    unsigned char tmpbuffer[40], *cmd;

    memset(tmpbuffer, 0, IOCTL_HEADER_LENGTH + CMD_6_LENGTH);
    ip = (int *)&(tmpbuffer[0]);
    *ip = length + MODE_HEADER_LENGTH;
    *(ip+1) = 0;

    cmd = &(tmpbuffer[IOCTL_HEADER_LENGTH]);
    cmd[0] = MODE_SELECT;
    cmd[1] = 0x10;
    cmd[4] = length + MODE_HEADER_LENGTH;

    memcpy(tmpbuffer + MODE_HEADER_OFFSET, buffer, length + MODE_HEADER_LENGTH);
    tmpbuffer[MODE_HEADER_OFFSET] = 0;      /* reserved data length */
    tmpbuffer[MODE_HEADER_OFFSET + 1] = 0;  /* reserved meadia type */
    tmpbuffer[MODE_HEADER_OFFSET + MODE_HEADER_LENGTH] &= 0x3f; 
                                 /* reserved bits in page code byte */

    result = ioctl(fn, SCSI_IOCTL_SEND_COMMAND, tmpbuffer);
    if (result) {
	fprintf(stderr, "%s: Can't write mode page.\n", tape_name);
	return 0;
    }
    return 1;
}


	static int
do_dat_compression(int fn, cmdef_tr *cmd, int argc, char **new)
{
    int i, alter = (argc > 0);
    unsigned char buffer[30], mask[30];
    struct mtpos mt_pos;

    if (!read_mode_page(fn, COMPRESSION_PAGE, COMPRESSION_PAGE_LENGTH,
			buffer, 0)) {
	fprintf(stderr, "%s: Can't read compression mode page.\n", tape_name);
	return 2;
    }

    if (alter) {
	if (!read_mode_page(fn, COMPRESSION_PAGE, COMPRESSION_PAGE_LENGTH,
			    mask, 1)) {
	    fprintf(stderr, "%s: Can't read changeable parameter page.\n", tape_name);
	    return 2;
	}

	if ((buffer[MODE_HEADER_LENGTH + 2] & DCC_MASK) == 0) {
	    fprintf(stderr, "%s: Drive does not support compression.\n", tape_name);
	    return 2;
	}
	if ((mask[MODE_HEADER_LENGTH + 2] & DCE_MASK) == 0) {
	    fprintf(stderr, "%s: Compression not controllable with software.\n",
		    tape_name);
	    return 2;
	}

	if (((buffer[MODE_HEADER_LENGTH + 3] & RED_MASK) >> 5) == 2) {
	    if (ioctl(fn, MTIOCPOS, (char *)&mt_pos) >= 0
		&& mt_pos.mt_blkno != 0) {
		fprintf(stderr, "%s: Can't change compression here. Please rewind.\n",
			tape_name);
		return 2;
	    }
	}

	if (**new == '0' || !strncmp(*new, "off", 3))
	    buffer[MODE_HEADER_LENGTH + 2] &= ~DCE_MASK;
	else
	    buffer[MODE_HEADER_LENGTH + 2] |= DCE_MASK;

	if (!write_mode_page(fn, COMPRESSION_PAGE, COMPRESSION_PAGE_LENGTH,
			     buffer)) {
	    for (i=2; i < 16; i++)
		buffer[MODE_HEADER_LENGTH + i] &= mask[MODE_HEADER_LENGTH + i];
	    if (!write_mode_page(fn, COMPRESSION_PAGE, COMPRESSION_PAGE_LENGTH,
				 buffer)) {
		fprintf(stderr, "%s: Writing mode SCSI mode page failed.\n", tape_name);
		fprintf(stderr, "%s: Compression mode not changed.\n", tape_name);
		return 2;
	    }
	}
	if (!read_mode_page(fn, COMPRESSION_PAGE, COMPRESSION_PAGE_LENGTH,
			    buffer, 0)) {  /* Re-read to check */
	    fprintf(stderr, "%s: Re-read of the compression page failed.\n", tape_name);
	    return 2;
	}
    }

    if (buffer[4+2] & 0x80)
	printf("Compression on.\n");
    else
	printf("Compression off.\n");

    close(fn);
    return 0;
}

#endif
