/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/*
 *
 * Public entry points in this file:
 *
 * check_request
 * get_deny_message
 *
 */

static char     rcsid[] = "$Id: authenticate.c,v 1.10 1998/07/06 17:43:07 web Exp $";

#include	<sys/types.h>
#include	<sys/param.h>

#ifdef sys5
#include	<sys/sysmacros.h>
#endif	/* sys5 */

#ifdef OSF
#include	<sys/security.h>
#include	<prot.h>
#endif	/* OSF */

#ifdef SIA
#include	<sia.h>
#include	<siad.h>
#endif	/* SIA */

#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/wait.h>
#include	<net/if.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<errno.h>
#include	<signal.h>
#include	<memory.h>
#include	<syslog.h>

#ifndef NOSHADOW
#include	<shadow.h>
#endif	/* NOSHADOW */

#ifdef __hpuxtrust
#include	<hpsecurity.h>
#endif	/* __hpuxtrust */

#include	"radius.h"

extern char     ourhostname[MAXHOSTNAMELEN];
extern char     default_radius_server[];
extern int      debug_flag;
extern int      token_caching_auth_type[];
extern AATV    *authtype_tv[];

/*************************************************************************
 *
 *	Function: check_request
 *
 *	Purpose: Process check and deny items on a request.
 *
 *	Returns: EV_ACK, EV_NAK, etc.
 *
 *************************************************************************/

int
check_request (authreq, list, sws)

AUTH_REQ       *authreq;
VALUE_PAIR     *list;		/* Check against this list (if present) */
int             sws;		/* 0 == do accept check, 1 == do deny check */

{
	int             result = EV_ACK;	/* Assume success. */
	VALUE_PAIR     *each;
	VALUE_PAIR     *vp;
	char           *why = (char *) NULL;
	char            buffer[MAXPATHLEN];	/* For optional denial msg. */
	static char    *func = "check_request";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (authreq == (AUTH_REQ *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: NULL authreq!", func);
		return EV_NAK;
	}

	/* If no list, find one. */
	if (list == NULL_VP)
	{
		if ((sws & CHK_DENY) == CHK_DENY)
		{
			list = authreq->user_deny;
		}
		else
		{
			list = authreq->user_check;
		}
	}

	/* Short circut test. */
	if (list == NULL_VP)
	{
		return EV_ACK;	/* Nothing to check or deny, let it through */
	}

	/* Check each check item in the list... */
	for (each = list;
		(each != NULL_VP) && (result == EV_ACK);
		each = each->next)
	{
	 	/* Ignore (config) items. */
		if ((each->ap->flags & ATTR_CONFIG) == ATTR_CONFIG)
		{
			continue;
		}

		/*
		 * Just look for the matching attribute in the request.
		 */

		if ((vp = get_vp_vend (authreq->cur_request, each->attribute,
					each->ap->vendor_id)) != NULL_VP)
		{
			switch (each->ap->type)
			{
			    case PW_TYPE_OCTETS:
			    case PW_TYPE_TAG_STR:
			    case PW_TYPE_STRING:
				if ((vp->lvalue == each->lvalue) && 
					(memcmp (each->strvalue, vp->strvalue,
						each->lvalue) == 0))
				{
					if ((sws & CHK_DENY) == CHK_DENY)
					{
						result = EV_NAK;
						why = buffer;
						sprintf (why,
						    "Access not allowed, %s=%s",
							vp->ap->name,
							avpair_vtoa (vp, 0));
					}
				}
				else
				{
					if ((sws & CHK_DENY) == 0)
					{
						result = EV_NAK;
						why = buffer;
						sprintf (why,
						    "Access not allowed, %s=%s",
							vp->ap->name,
							avpair_vtoa (vp, 0));
					}
				}
				break;

			    case PW_TYPE_IPADDR:
				if ((each->strvalue != NULL) && 
					(*each->strvalue != '\0'))
				{
					/* Resolve all DNS names */
					if (find_host_by_name (&each->lvalue,
							each->strvalue) == 1)
					{
						why = buffer;
						sprintf (why,
					"Access temporarily not allowed, %s=%s",
							each->ap->name,
							each->strvalue);

						if ((sws & CHK_DENY) == CHK_DENY)
						{
							result = EV_NAK;
						}
						else
						{
							logit (LOG_AUTH,
								LOG_WARNING,
		    "%s: request ignored - unresolved host '%s' in check item",
								func,
								each->strvalue);
							result = EV_ABORT;
						}
						break; /* switch() statement */
					}
				}
				/***FALLTHROUGH****/

			    case PW_TYPE_SHORT:
			    case PW_TYPE_OCTET:
			    case PW_TYPE_TAG_INT:
			    case PW_TYPE_INTEGER:
				if (each->lvalue == vp->lvalue)
				{
					if ((sws & CHK_DENY) == CHK_DENY)
					{
						result = EV_NAK;
						why = buffer;
						sprintf (why,
						   "Access not allowed, %s=%s",
							vp->ap->name,
							avpair_vtoa (vp, 0));
					}
				}
				else
				{
					if ((sws & CHK_DENY) == 0)
					{
						result = EV_NAK;
						why = buffer;
						sprintf (why,
						   "Access not allowed, %s=%s",
							vp->ap->name,
							avpair_vtoa (vp, 0));
					}
				}
				break;

			    default:
				why = buffer;
				sprintf (why,
			      "Configuration error, access not allowed, %s=%s",
					vp->ap->name,
					avpair_vtoa (vp, 0));
				logit (LOG_DAEMON, LOG_ERR,
				     "%s: Configuration error, %s bad type %d",
					func, each->ap->name, each->ap->type);
				result = EV_NAK;
				break;
			}
	        }

		/* If a check item is not present, deny the request. */
		else
		{
			if ((sws & CHK_DENY) == 0)
			{
				result = EV_NAK;
				why = buffer;
				sprintf (why, "Access not allowed, no %s",
					each->ap->name);
			}
		}

		/* If we are to check only one, do only one. */
		if ((sws & CHK_ONCE) == CHK_ONCE)
	        {
			break;
		}
	} /* end of "Check each check-item" */

	/*
	 * Explain why denial occured, allowing explanation to be configured.
	 */
	if ((result != EV_ACK) && (sws & CHK_MESSAGE) == CHK_MESSAGE)
	{
		reply_sprintf (0, authreq,
				get_deny_message (authreq, list, why));
	}

	return result;
} /* end of check_request () */

static int      chk_pass PROTO((AUTH_REQ *, int, char *));

static AATV     realm_aatv = DEF_AATV_DIRECT_TYPE("REALM", AA_REALM, chk_pass);

AATVPTR         rad_realm_aatv = & realm_aatv;

/*************************************************************************
 *
 *	Function: chk_pass
 *
 *	Purpose: Check the users password against any server
 *
 *************************************************************************/

static int
chk_pass (authreq, value, af_param)

AUTH_REQ       *authreq;
int             value;
char           *af_param;

{
	int             authprot;
	int             type;
	VALUE_PAIR     *vp;
	char           *agent;
	char           *realm;
	char           *filter;
	char            user_realm[AUTH_ID_LEN + 1];
	static char    *func = "chk_pass";

	dprintf(1, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (get_vp (authreq->cur_request, PW_CHAP_PASSWORD))
	{
		authprot = PW_PROTTYPE_CHAP;
	}
	else
	{
		authprot = PW_PROTTYPE_PW;
	}

	if ((vp = parse_realm (authreq)) == NULL_VP)
	{
		reply_sprintf (0, authreq,
				"Improper 'userid@realm' specification");
		return EV_NAK;
	}
	strcpy (user_realm, vp->strvalue);

	if ((vp = get_vp_vend (authreq->cur_request, PW_USER_ID, VC_MERIT))
	    != NULL_VP)
	{
		if (strchr (vp->strvalue, '.') != (char *) NULL)
		{
			authprot = PW_PROTTYPE_HASDOT;
		}
	}

	if (user_realm[0] == '\0')		/* user_realm is never NULL */
	{
		if (find_auth_type ("", authprot,
					(char *) authreq->client->file_pfx,
					&type, &agent, &realm, &filter) != 0)
		{
			reply_sprintf (0, authreq,
				"Improper 'userid@realm' specification");
			return EV_NAK;
		}
	}
	else			/* Check for protected keyword realm value */
	{
		if (strcasecmp (user_realm, DEFAULT_REALM) == 0 ||
			strcasecmp (user_realm, NULL_REALM) == 0)
		{
			reply_sprintf (0, authreq,
					"Invalid authentication realm: '%s'",
					user_realm);
			return EV_NAK;
		}

		/* Try to match realm name in AUTHFILE */
		if (find_auth_type (user_realm, authprot,
		    	     (char *) authreq->client->file_pfx, &type, &agent,
				    &realm, &filter) != 0)
		{
			/* If no match, try for DEFAULT entry */
			if (find_auth_type (DEFAULT_REALM, authprot,
				     (char *) authreq->client->file_pfx, &type,
					    &agent, &realm, &filter) != 0)
			{		/* No DEFAULT configured so give up */
				reply_sprintf (0, authreq,
					"Invalid authentication realm: '%s'",
					user_realm);
				return EV_NAK;
			}
		}
	}

	authreq->realm_filter = filter;

	if (type == AA_REALM) 	/* Watch for misconfiguration */
	{
		/* Can't have REALM auth type in authfile! */
		logit (LOG_AUTH, LOG_ERR, "%s: Invalid Type '%d' in authfile",
			 func, type);
		return EV_NAK;
	}

#ifdef USR_CCA 
	if (type == AA_LOCAL_VPN)
	{
		/* This is an error since all users belonging to a Local VPN 
		   must be defined in the local users file, and so this function
		   must not be called in the first place
		*/

		vp = get_vp (authreq->cur_request, PW_USER_NAME);
		logit (LOG_AUTH, LOG_ERR, "%s: User %s of realm %s not defined",
				func, vp->strvalue, user_realm);
		return EV_NAK;
	}
#endif	/* USR_CCA */

	return call_action (authtype_tv[type], authreq, 0, agent);
} /* end of chk_pass () */

/*************************************************************************
 *
 *	Function: get_deny_message
 *
 *	Purpose: Get the configured denial message, or a canned message.
 *
 *************************************************************************/

char *
get_deny_message (authreq, list, why)

AUTH_REQ       *authreq;
VALUE_PAIR     *list;	/* Suggested list to try. */
char           *why;	/* Suggested message, if CI_DENY_MESSAGE is "*" */

{
	VALUE_PAIR     *vp;
	char           *reason = "Access not allowed";
	static char    *func = "get_deny_message";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/*
	 *	Find the deny message, first in the list...
	 *	then in the authreq somewhere...
	 */
	if ((vp = get_vp_ci (list, CI_DENY_MESSAGE, 0)) == NULL_VP)
	{
		if (authreq != (AUTH_REQ *) NULL)
		{
			if ((vp = get_vp_ci (authreq->user_check,
						CI_DENY_MESSAGE, 0)) == NULL_VP)
			{
				vp = get_vp_ci (authreq->user_deny,
						CI_DENY_MESSAGE, 0);
			}
		}
	}

	if (vp != NULL_VP)
	{
		reason = vp->strvalue;
	}

	/*
	 *	Special case: deny-message = "*"
	 *
	 *	Allows message "access not allowed, xxx = value"
	 */
	if (strcmp (reason, "*") == 0)
	{
		if (why != (char *) NULL)
		{
			reason = why;
	        }
		else
		{
			reason = "Access not allowed";
		}
	}

	return reason;
} /* end of get_deny_message () */

static int      rad_authenticate PROTO((AUTH_REQ *, int, char *));

static AATV     authen_aatv = DEF_AATV_DIRECT("AUTHENTICATE", rad_authenticate);

AATVPTR		rad_authen_aatv = & authen_aatv;

/*************************************************************************
 *
 *	Function: rad_authenticate
 *
 *	Purpose: Process and reply to an authentication request
 *
 *************************************************************************/

static int
rad_authenticate (authreq, value, realm)

AUTH_REQ       *authreq;
int             value;
char           *realm;

{

#ifdef MERIT_LAS
	int          cache_token;
#endif	/* MERIT_LAS */

	int          found_pw = 0;
	int          i;
	int          protocol;
	int          result;
	int          retval;
	VALUE_PAIR  *auth_item;
	VALUE_PAIR  *check_item;
	VALUE_PAIR  *namepair;
	VALUE_PAIR  *protpair;
	VALUE_PAIR  *user_reply;
	VALUE_PAIR **prev_ptr;
	char        *server_name = "";
	char         pwmsg[AUTH_PASS_LEN + 1];

#ifdef USR_CCA
        /****
	char         res_assigned = FALSE;
	UINT4        nas = 0;
	UINT4        framed_ip = 0;
	UINT4        nas_port = 0;
	VALUE_PAIR  *framed_ip_vp;
	VALUE_PAIR  *nas_port_vp;
	VALUE_PAIR  *nas_vp;
	****/
	VALUE_PAIR  *term_act;
#endif	/* USR_CCA */

	char        *func = "rad_authenticate";

	authreq->startlas = time (0); /* record time starting authentication */

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Get the username from the request */
	if (((namepair =
		get_vp (authreq->cur_request, PW_USER_NAME)) == NULL_VP) ||
		(strlen (namepair->strvalue) <= 0))
	{
		logit (LOG_AUTH, LOG_ERR, "%s: from %s - No User Name",
			 func, ip_hostname (authreq->ipaddr));
		return EV_NAK;
	}

	protpair = get_vp (authreq->cur_request, PW_FRAMED_PROTOCOL);
	protocol = (protpair == NULL_VP) ? 0 : protpair->lvalue;

	if (authreq->user_check == NULL_VP)
	{
		/* Look in the database only if both lists are NULL. */
		if ((retval = user_find ((char *) authreq->client->file_pfx,
					namepair->strvalue, protocol,
					&authreq->user_check,
					&authreq->user_deny,
					&user_reply, 0)) < 0)
		{
			logit (LOG_AUTH, LOG_ERR,
				"%s: from %s - Problem in user_find",
			 	func, ip_hostname (authreq->ipaddr));
			return EV_NAK;
		}
		if (retval > 0)		/* If no match, use DEFAULT entry */
		{
			if (user_find ((char *) authreq->client->file_pfx, 
				"DEFAULT", protocol, &authreq->user_check,
				&authreq->user_deny, &user_reply, 0) != 0)
			{
				logit (LOG_AUTH, LOG_ERR,
			      "%s: from %s - Invalid/missing 'DEFAULT' entry",
			   		func, ip_hostname (authreq->ipaddr));
				return EV_NAK;
			}
		}
		list_cat (&authreq->cur_request, user_reply);
	}
	else
	{
		if (authreq->user_deny == NULL_VP)
		{
			/* Look in the database only if both lists are NULL. */
			if ((retval =
				user_find ((char *) authreq->client->file_pfx,
						namepair->strvalue, protocol,
						(VALUE_PAIR **) NULL,
						&authreq->user_deny,
						(VALUE_PAIR **) NULL, 0)) < 0)
			{
				logit (LOG_AUTH, LOG_ERR,
					"%s: from %s - Problem in user_find()",
					func, ip_hostname (authreq->ipaddr));
			}
			else
			{
				if (retval > 0)
				{
					if (user_find ((char *)
						authreq->client->file_pfx,
						"DEFAULT", protocol,
						(VALUE_PAIR **) NULL,
						&authreq->user_deny,
						(VALUE_PAIR **) NULL, 0) != 0)
					{
						logit (LOG_AUTH, LOG_ERR,
				"%s: from %s - Invalid/missing 'DEFAULT' entry",
							func,
							ip_hostname (authreq->
								     ipaddr));
					}
				}
			}
		}
	}

	result = EV_ACK; /* assume good until proven otherwise */

	/* No null names allowed */

	if (*namepair->strvalue == '\0')
	{
		result = EV_NAK;
	}

/* #ifdef USR_CCA */
	/* RESOURCE MANAGEMENT */

/*	if (authreq->code == PW_ACCESS_REQUEST && result != EV_NAK &&
 *		(term_act = get_vp (authreq->cur_request,
 *				PW_TERMINATION_ACTION)) != NULL_VP)
 *	{
 *		if (term_act->lvalue == PW_MANAGE_RESOURCES)
 *		{
 *			result = resource_mgmt (authreq);
 *			if (result == EV_ACK)
 *			{
 *				res_assigned = TRUE;
 *			}
 *		}
 *	} 
 */
/* #endif	* USR_CCA */

	/*
	 *	See if this is strictly an authentication check.
	 *	Throw out all but pw and pw expiration items if it is.
	 */
	if ((auth_item =
		get_vp (authreq->cur_request, PW_SERVICE_TYPE)) != NULL_VP &&
		(auth_item->lvalue == PW_AUTHENTICATE_ONLY))
	{
		for (prev_ptr = &authreq->user_check, check_item = *prev_ptr;
			check_item != NULL_VP;
			check_item = *prev_ptr)
		{
			if (check_item->ap->flags & ATTR_CONFIG)
			{
				switch (check_item->attribute)
				{
				    case CI_PROHIBIT:
					if (check_item->lvalue == PW_AUTH_ONLY)
					{
						result = EV_NAK;
					}
					/* Leave in user_check list */
					prev_ptr = &check_item->next;
					continue;

				    case CI_ENCRYPTED_PASSWORD:
				    case CI_USER_PASSWORD:
				    case CI_AUTHENTICATION_TYPE:
					found_pw = 1;
					/***FALLTHROUGH***/

				    case CI_EXPIRATION:
				    case CI_SERVER_NAME:
					/* Leave in user_check list */
					prev_ptr = &check_item->next;
					continue;

				    default:
					break;
				}
			}
			/* Remove all others from list */
			*prev_ptr = check_item->next;
			avpair_free (check_item);
		}

		if (found_pw == 0) /* Authenticate-Only requires a password! */
		{
			result = EV_NAK;
		}
	}


	/* Check those check items */

	dprintf(4, (LOG_AUTH, LOG_DEBUG,
		"%s: about to check user_check items", func));

	for (check_item = authreq->user_check;
		((result == EV_ACK) && (check_item != NULL_VP));
		check_item = check_item->next)
	{
		if (check_item->ap->flags & ATTR_CONFIG)
		{
			switch (check_item->attribute)
			{
			    /*
			     * Check expiration date if we are doing password
			     * aging.
			     */
			    case CI_EXPIRATION:

				/* Has this user's password expired */
				retval = pw_expired (check_item->lvalue,
							user_reply);

				if (retval < 0)
				{
					reply_sprintf (0, authreq,
							"Password Has Expired");

#ifdef ASCEND
					result = EV_PW_EXPIRED;
				}
				else if (retval > 0)
				{
					reply_sprintf (0, authreq,
					      "Password will expire in %d days",
						retval);
#else	/* ASCEND */
					result = EV_NAK;
#endif	/* ASCEND */

				}

				if (retval > 0)
				{
					reply_sprintf (RS_IF_ACK, authreq,
					      "Password Will Expire in %d Days",
					      retval);
				}
				break;

			    case CI_SERVER_NAME:
				server_name = check_item->strvalue;
				break;

			    case CI_AUTHENTICATION_TYPE:
			    case CI_SIMULTANEOUS_USE:
			    case CI_COMMENT:
			    case CI_PROHIBIT:
			    case CI_SERVICE_CLASS:
				break;

			    case CI_ENCRYPTED_PASSWORD:
				retval = get_passwd (authreq, pwmsg,
					    check_item->strvalue, 
						check_item->strvalue);
				if (retval != 0)
				{
					result = EV_NAK;
				}
				memset ((char *) pwmsg, '\0', sizeof (pwmsg));
				break;

			    case CI_USER_PASSWORD:

			    /*
			     *	Pass password to get_passwd() so it will
			     *	do CHAP and PASSWORD check for us.
			     */

#ifdef ASCEND
				/*
				 *	If this is Ascend ARADES authentication,
				 *	this is the wrong place to check this.
				 *	This will fail since the password is
				 *	not sent in the Access-Request.
				 */
				if ((auth_item =
					get_vp_vend (authreq->cur_request,
							PW_ASCEND_ARADES,
							VC_ASCEND)) != NULL_VP)
				{
					break;
				}
#endif	/* ASCEND */

				retval = get_passwd (authreq, pwmsg,
					   check_item->strvalue, (char *) NULL);

				if (retval != 0)
				{
					dprintf(2, (LOG_AUTH, LOG_DEBUG,
					   "%s: invalid password, retval = %d",
						func, retval));
					result = EV_NAK;
				}

				/* Test Code for Challenge */
				if (strcmp (pwmsg, "challenge") == 0 &&
								retval == 1)
				{
				  memset ((char *) pwmsg, 0, sizeof (pwmsg));
				  reply_sprintf (RS_IF_ACK, authreq,
			       "You want me to challenge you??\r\nOkay I will");
				  avpair_add (&authreq->cur_request, PW_STATE,
						"1", -1);
				  result = EV_ACC_CHAL;
				}
				memset ((char *) pwmsg, 0, sizeof (pwmsg));
				break;

			    default:
				break;
			}
			continue;
		}

		/*
		 * Process check items that didn't require special processing
		 * above.  Just look for the matching attribute in the
		 * request.
		 */

		result = check_request (authreq, check_item,
					CHK_ACCEPT | CHK_ONCE);

	} /* end of "Check those check items" */

	/* Check those deny items */

	if (result == EV_ACK)
	{
		result = check_request (authreq, authreq->user_deny,
					CHK_DENY | CHK_MESSAGE);
	} /* end of "Check those deny items" */

	if ((result == EV_ACK) &&
	     (check_item = get_vp_ci (authreq->user_check,
					CI_AUTHENTICATION_TYPE, 0)) != NULL_VP)
	{
		if (authtype_tv[check_item->lvalue] == (AATV *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
			 "%s: FATAL: Unsupported authentication type %d for %s",
				func, check_item->lvalue, namepair->strvalue);

			logit (LOG_DAEMON, LOG_ERR,
				"%s: CHECK THE MAKEFILE BUILD!", func);

			result = EV_FATAL;
		}
		else
		{

#ifdef MERIT_LAS
			/*
			 *	See if it's one of the authentication types
			 *	which allow caching.  If so, call the AATV
			 *	which checks the cache before calling the
			 *	authentication AATV.
			 */
			cache_token = 0;
			for (i = 0; token_caching_auth_type[i]; i++)
			{
				if (check_item->lvalue ==
						token_caching_auth_type[i])
				{
					cache_token = 1;
				}
			}

			if (cache_token > 0)
			{
				result = call_action (rad_cache_chk_aatv,
						      authreq, 0, server_name);
			}
			else
#endif	/* MERIT_LAS */

			{
				result =
				   call_action (authtype_tv[check_item->lvalue],
						authreq, 0, server_name);
			}
		}
	}

/* #ifdef USR_CCA
	if (result != EV_ACK && result != EV_WAIT &&
		res_assigned == TRUE)
	{
		if ((framed_ip_vp = get_vp (authreq->cur_request, 
					    PW_FRAMED_IP_ADDRESS)) != NULL_VP)
		{
			framed_ip = framed_ip_vp->lvalue;
		}
		if ((nas_vp = get_vp (authreq->cur_request, PW_NAS_IP_ADDRESS))
				     != NULL_VP)
		{
			nas = nas_vp->lvalue;
		}
		
		if ((nas_port_vp = get_vp (authreq->cur_request, 
					   PW_NAS_PORT)) != NULL_VP)
		{
			nas_port = nas_port_vp->lvalue;
		}
		
		free_resources (namepair->strvalue, framed_ip, nas, nas_port);
	}
*/
/* #endif * USR_CCA */

#ifdef USR_CCA
	/* RESOURCE MANAGEMENT */

	/* For Local-VPN type, CI_AUTHENTICATON_TYPE will be NULL */
	if (result != EV_NAK && check_item == NULL_VP)
	{
		if (authreq->code == PW_ACCESS_REQUEST &&
			(term_act = get_vp (authreq->cur_request,
					    PW_TERMINATION_ACTION)) != NULL_VP)
		{
			if (term_act->lvalue == PW_MANAGE_RESOURCES)
			{
				result = resource_mgmt (authreq);
				/* if (result == EV_ACK)
				{
					res_assigned = TRUE;
				} */
			}
		}
	}

#endif	/* USR_CCA */

	return result;
} /* end of rad_authenticate () */

static void     rad_2rad_init PROTO((AATV *));
static int      radius_pass PROTO((AUTH_REQ *, int, char *));

static AATV     rad2rad_aatv = DEF_AATV_SOCKET_TYPE("RAD2RAD", AA_RAD,
							rad_2rad_init,
							radius_pass,
							rad_2rad_recv);
AATVPTR		rad_2rad_aatv = & rad2rad_aatv;

/*************************************************************************
 *
 *	Function: rad_2rad_init
 *
 *	Purpose: Perform RADIUS to RADIUS initialization.
 *
 *************************************************************************/

static void
rad_2rad_init (aatv)

AATV   *aatv;

{
	struct sockaddr_in lclsin;
	static char    *func = "rad_2rad_init";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (aatv->sockfd == -1)
	{
		aatv->sockfd = setupsock (&lclsin, 0);
	}
	return;
} /* end of rad_2rad_init () */

/*************************************************************************
 *
 *	Function: radius_pass
 *
 *	Purpose: Have remote RADIUS system handle authentication for this
 *		 RADIUS to RADIUS request.
 *
 *	Returns: EV_ACK if valid userid and pw,
 *		 EV_NAK if invalid,
 *		 EV_WAIT if request issued.
 *
 *************************************************************************/

static int
radius_pass (authreq, value, realm)

AUTH_REQ       *authreq;
int             value;
char           *realm;

{
	VALUE_PAIR     *vp;
	char            id[AUTH_ID_LEN + 1];
        int             code;
	static char    *func = "radius_pass";

	code = authreq->code;

#ifdef USR_CCA
	if (code != PW_NAS_REB_REQ &&
		(vp = get_vp_vend (authreq->cur_request, PW_USER_ID, VC_MERIT))
								== NULL_VP)
#else	/* USR_CCA */
	if ((vp = get_vp_vend (authreq->cur_request, PW_USER_ID, VC_MERIT))
								== NULL_VP)
#endif	/* USR_CCA */

	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: Improper userid specification", func);
		reply_message (authreq, EC_INTERNAL, func);
		return EV_NAK;
	}
	strcpy (id, vp->strvalue);

	/*
	 *	If no server system DNS name is provided, a default is used.
	 *	Set #define DEFAULT_RADIUS_SERVER (or set it in authfile)
	 *	to the name of the default server system to use for
	 *	RADIUS authentication.
	 */

	if ((realm == NULL) || (realm[0] == '\0'))
	{
		realm = default_radius_server;
	}

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: name = %s  realm = %s",
		func, id, realm));

	/* Check to see if we are the server for this realm */

#ifdef USR_CCA
        if (code != PW_NAS_REB_REQ &&
	    strcasecmp (ourhostname, realm) == 0)
#else	/* USR_CCA */
	if (strcasecmp (ourhostname, realm) == 0)
#endif	/* USR_CCA */

	{
		dprintf(2, (LOG_AUTH, LOG_DEBUG,
			"%s: handle locally", func));
		/* Treat this one just like Unix-PW authentication */
		return call_action (authtype_tv[AA_UNIX], authreq, 0, realm);
	}

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: handle remotely", func));

        if (radius_send ((char *) authreq->fsm_aatv->id, code, 0,
                        realm, authreq, rad2rad_aatv.sockfd) == -1)
	{
		return EV_NAK;
	}

	return EV_WAIT;	/* RC says to expect reply later */
} /* end of radius_pass () */

static int      slow_pass PROTO((AUTH_REQ *, int, char *));

static AATV     slow_aatv = DEF_AATV_FREPLY_TYPE("SLOW", AA_SLOW, NULL,
								slow_pass, 2);

AATVPTR         rad_slow_aatv = & slow_aatv;

/*************************************************************************
 *
 *	Function: slow_pass
 *
 *	Purpose: Enable flexible debugging of realm type autthentication.
 *
 *************************************************************************/

static int
slow_pass (authreq, value, af_param)

AUTH_REQ       *authreq;
int             value;
char           *af_param;

{
	int             authprot;
	int             type;
	char           *agent;
	char           *filter;
	char           *pfx = (char *) NULL;
	char           *realm;
	VALUE_PAIR     *rlm_vp = NULL_VP;
	VALUE_PAIR     *vp;
	static char    *func = "slow_pass";

	dprintf(1, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* If there is a prefix assocated with this authreq, get it. */
	if (authreq != (AUTH_REQ *) NULL)
	{
		if (authreq->client != (CLIENT_ENTRY *) NULL)
		{
			pfx = authreq->client->file_pfx;
		}
	}

	if (af_param == (char *) NULL)
	{
		reply_sprintf (RS_LOG, authreq, "%s: Missing sub-realm", func);
		return EV_NAK;
	}

	/* Get realm for messages, etc. */
	rlm_vp = get_vp_vend (authreq->cur_request, PW_USER_REALM, VC_MERIT);

	/* Determine authentication protocol type */
	if (get_vp (authreq->cur_request, PW_CHAP_PASSWORD))
	{
		authprot = PW_PROTTYPE_CHAP;
	}
	else
	{
		authprot = PW_PROTTYPE_PW;
	}

	/* Try to match the realm name in the authfile. */
	if (find_auth_type (af_param, authprot, pfx, &type, &agent,
				&realm, &filter) != 0)
	{
		reply_sprintf (RS_LOG, authreq,
				"%s: Invalid authentication sub-realm: '%s'",
				func, af_param);
		return EV_NAK;
	}

	/* Check the various different types... */
	switch (type)
	{
	    case AA_SLOW:
		reply_sprintf (RS_LOG, authreq,
				"%s: Slow won't stack realm (%s)",
				func, af_param);
		return EV_NAK;

	    case AA_REALM:
		reply_sprintf (RS_LOG, authreq,
				"%s: Invalid type '%d' in authfile",
				func, type);
		return EV_NAK;

#ifdef USR_CCA
	    case AA_LOCAL_VPN:
		/*
		 *	This is an error since all users belonging to a
		 *	Local VPN must be defined in the local users file,
		 *	so this function must not be called in the first
		 *	place.
		 */
		vp = get_vp (authreq->cur_request, PW_USER_NAME);
		logit (LOG_AUTH, LOG_ERR, "%s: User %s of realm %s not defined",
			func, vp->strvalue, af_param);
		return EV_NAK;
#endif /* USR_CCA */

	} /* end of switch */

	if (authtype_tv[type] == (AATV *) NULL)
	{
		reply_sprintf (RS_LOG, authreq,
			"%s: Unsupported authentication type %d", func, type);
		return EV_NAK;
	}

	rad_sleep (10, "%s (xxx, %d, '%s'), sleeping", func, value, agent);

	return call_action (authtype_tv[type], authreq, value, agent);
} /* end of slow_pass () */



static int     unix_pass PROTO((AUTH_REQ *, int, char *));

static AATV    unix_aatv = DEF_AATV_FORK_TYPE("UNIX-PW", AA_UNIX, unix_pass, 0);

AATVPTR        rad_unix_aatv = & unix_aatv;

/*************************************************************************
 *
 *	Function: unix_pass
 *
 *	Purpose: Check the users password against the standard UNIX
 *		 password table.
 *
 *************************************************************************/

static int
unix_pass (authreq, value, af_param)

AUTH_REQ       *authreq;
int             value;
char           *af_param;

{
	int             errcode;
	int             has_etc_shells;
	int             len;

#ifdef NISPLUS
	int             uid;
#endif	/* NISPLUS */

	VALUE_PAIR     *vp;
	struct passwd  *pwd;
	char           *encpw;
	char           *encrypted_pass;
	char           *crypt ();

#ifdef	CHK_SHELLS
	char           *getusershell ();
#endif	/* CHK_SHELLS */

	char           *pshell;
	FILE           *fp;
	char            name[AUTH_ID_LEN + 1];
	char            passwd[AUTH_PASS_LEN + 1];
	char            buffer[MAXPATHLEN];
	static char    *func = "unix_pass";

#ifdef OSF
	struct pr_passwd *getprpwnam ();
	struct pr_passwd *osf_pw_info;
#endif /* OSF */

#ifdef SIA
	int           (*sia_collect) () = sia_collect_trm;
	char           *info[2];
	char           *progname = "radius";
	SIAENTITY      *ent = NULL;
#endif	/* SIA */

#if !defined(NOSHADOW)

#if defined(M_UNIX)
	struct passwd  *spwd;
#else	/* M_UNIX */
	struct spwd    *spwd;
#endif	/* M_UNIX */

#endif	/* !NOSHADOW */

	if ((vp = get_vp_vend (authreq->cur_request, PW_USER_ID, VC_MERIT))
								== NULL_VP)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: Improper userid specification", func);
		reply_message (authreq, EC_INTERNAL, func);
		return EV_NAK;
	}
	strcpy (name, vp->strvalue);

	dprintf(1, (LOG_AUTH, LOG_DEBUG, "%s: ID = '%s'", func, name));

	/* Decrypt pw - NULL 3rd arg. says we don't handle CHAP */
	if ((errcode = get_passwd (authreq, passwd, (char *) NULL,
							(char *) NULL)) != 0)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: Error %d decrypting password", func, errcode));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

	/* Get encrypted password from UNIX password file */

#ifdef NISPLUS
	uid = geteuid ();	/* Remember who we are (for later). */
	pwd = getpwnam (name);	/* Probe for this user's password file entry. */

	if (pwd == (struct passwd *) NULL) /* Then he is not on this machine. */
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: getpwnam() returned NULL", func));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

	/* Become this user to access his NIS+ entry. */
	if ((errcode = seteuid (pwd->pw_uid)) != 0)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: seteuid(%d) returned %d (errno=%d)",
			func, pwd->pw_uid, errcode, errno));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}
#endif	/* NISPLUS */

	pwd = getpwnam (name);	/* Get the user's password file entry. */

#ifdef NISPLUS
	/* Go back to whoever we were before. */
	if ((errcode = seteuid (uid)) != 0)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: seteuid(%d) returned %d (errno=%d)",
			func, uid, errcode, errno));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}
#endif	/* NISPLUS */

	if (pwd == (struct passwd *) NULL)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: getpwnam() returned NULL for user", func));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

	encrypted_pass = pwd->pw_passwd;

#ifdef OLDER_OSF
	/* Get encrypted password from security files */
	if ((osf_pw_info = getprpwnam (name)) == NULL)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: getprpwnam() returned NULL", func));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}
	strcpy (pwd->pw_passwd, osf_pw_info->ufld.fd_encrypt);
	pwd->pw_uid = osf_pw_info->ufld.fd_uid;
#endif /* OLDER_OSF */

#ifdef ULTRIX_ENHANCED		/* 4/3/94 jeff@oakland.edu */
	if ((errcode = authenticate_user (pwd, passwd, "/dev/ttypXX")) < 0)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: authenticate_user() returned %d", func, errcode));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

#elif  SIA  /* DEC OSF/1 SIA , 6/19/95 minnebo@oakland.edu */

	info[0] = progname;
	info[1] = NULL;
	if ((errcode = sia_ses_init (&ent, (1), info, NULL, pwd->pw_name,
					NULL, TRUE, NULL)) != SIASUCCESS)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: sia_ses_init() returned %d", func, errcode));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

	if ((errcode = sia_ses_authent (sia_collect, passwd, ent)) != SIASUCCESS)
	{
		(void) sia_ses_release (&ent);
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: sia_ses_authent() returned %d", func, errcode));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

	if ((errcode = sia_ses_release (&ent)) != SIASUCCESS)
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: sia_ses_release() returned %d", func, errcode));
		memset ((char *) passwd, '\0', sizeof (passwd));
		return EV_NAK;
	}

#else	/* ULTRIX_ENHANCED */

#if !defined(NOSHADOW) || (defined(__sun__) && defined(__svr4__))
	if ((strcmp (pwd->pw_passwd, "x") == 0) ||
		(strcmp (pwd->pw_passwd, "*") == 0) ||
		(strncmp (pwd->pw_passwd, "##",2) == 0)) /* C2 support */
	{
		if ((spwd = getspnam (name)) == NULL)
		{
			dprintf(1, (LOG_AUTH, LOG_DEBUG,
				"%s: getspnam() returned NULL", func));
			memset ((char *) passwd, '\0', sizeof (passwd));
			return EV_NAK;
		}

#if defined(M_UNIX)
	encrypted_pass = spwd->pw_passwd;
#else	/* M_UNIX */
	encrypted_pass = spwd->sp_pwdp;
#endif	/* M_UNIX */

	}
#endif  /* !NOSHADOW */

	/* Run encryption algorithm */

#ifdef __hpuxtrust
	encpw = bigcrypt (passwd, encrypted_pass);
#else	/* __hpuxtrust */
	encpw = crypt (passwd, encrypted_pass);
#endif  /* __hpuxtrust */

	memset ((char *) passwd, '\0', sizeof (passwd));

	/* Check it */
	if (strcmp (encpw, encrypted_pass))
	{
		dprintf(1, (LOG_AUTH, LOG_DEBUG,
			"%s: encrypted passwords do not match", func));
		return EV_NAK;
	}
#endif	/* ULTRIX_ENHANCED */

	/* Don't allow authentication with id "root" */
	if (pwd->pw_uid == 0)
	{
		logit (LOG_AUTH, LOG_WARNING,
			"%s: Attempt to authenticate using 'root'", func);
		return EV_NAK;	/* Don't do it */
	}

#ifdef	CHK_SHELLS
	/* Also make sure id uses a standard shell */
	if ((fp = fopen ("/etc/shells", "r")) == (FILE *) NULL)
	{
		has_etc_shells = 0;
	}
	else
	{
		has_etc_shells = 1;
	}

	while (has_etc_shells &&
		(fgets (buffer, sizeof (buffer), fp) != (char *) NULL))
	{
		len = strlen (buffer);

		if (buffer[len - 1] == '\n')
		{
			buffer[len - 1] = '\0';
		}

		if (strncmp (buffer, pwd->pw_shell, MAXPATHLEN) == 0)
		{
			fclose (fp);
			return EV_ACK;
		}
	}

	if (has_etc_shells)
	{
		fclose (fp);
	}

#if !(defined _AIX || defined ultrix || defined SCO || defined __sgi)
	else
	{
		setusershell ();
		while ((pshell = getusershell ()) != (char *) NULL)
		{
			if (strncmp (pshell, pwd->pw_shell,
				     strlen (pwd->pw_shell)) == 0)
			{
				endusershell ();
				return EV_ACK;
			}
		}
		endusershell ();
	}
#endif	/* !(defined _AIX || defined ultrix || defined SCO) */

	logit (LOG_AUTH, LOG_WARNING,
		"%s: Attempt to authenticate with funny id '%s' - shell = %s",
		 func, name, pwd->pw_shell);
	return EV_NAK;
#else	/* CHK_SHELLS */
	return EV_ACK;
#endif	/* CHK_SHELLS */

} /* end of unix_pass () */
