/*
 *
 *	RADIUS
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 Livingston Enterprises, Inc.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 *      Copyright (c) 1996 Ascend Communications, Inc.
 *      All rights reserved.
 *
 *      Permission to copy, display, distribute and make derivative works
 *      from this material in whole or in part for any purpose is granted
 *      provided that the above copyright notice and this paragraph are
 *      duplicated in all copies.  THIS SOFTWARE IS PROVIDED "AS IS" AND
 *      WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES INCLUDING, WITHOUT
 *      LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *      FOR A PARTICULAR PURPOSE.
 *
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/*
 *
 * Public entry points in this file:
 *
 * attr_check
 * attr_check_by_dict
 * attr_check_by_name
 * attr_check_clear
 * dict_attrfind
 * dict_attrget
 * dict_find_value
 * dict_init
 * dict_valfind
 * dict_valget
 * free_vendor_list
 * parse_for_vendor_list
 * vendor_list_toa
 *
 */

static char     sccsid[] =
		"@(#)dict.c 1.2 Copyright 1992 Livingston Enterprises Inc";

static char     rcsid[] = "$Id: dict.c,v 1.8 1998/07/01 21:51:48 rsc Exp $";

#include	<sys/types.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>

#include	<ctype.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<syslog.h>

#include	"radius.h"

extern int      debug_flag;
extern char    *radius_dir;
extern MF_ENT   vendor_mf;
extern MF_ENT   vendor_list_mf;

extern int      dumpcore;
static int      parse_flags PROTO((int *));
static char    *parse_for_vendor PROTO((char *, VENDOR **));
static int      vend_init PROTO((void));
static int      vend_verify PROTO((void));

static DICT_ATTR *dictionary_attributes = (DICT_ATTR *) NULL;
static DICT_VALUE *dictionary_values = (DICT_VALUE *) NULL;
static          have_vendors = 0;
static VENDOR  *vendors = (VENDOR *) NULL;

char           *dict_id = (char *) NULL;
char           *vend_id = (char *) NULL;

/*************************************************************************
 *
 *	Function: attr_check
 *
 *	Purpose: Used to specify what kind of attribute/value pair
 *		 to check.
 *
 *	Returns: success or failure code (0 == success, -1 == failure)
 *
 *************************************************************************/

int
attr_check (p_list, attr, vend_id)

DICT_ATTR_LIST **p_list;
int	         attr;
UINT4	         vend_id;

{
	DICT_ATTR *ap = dict_attrget (attr, vend_id);
	static char *func = "attr_check";

	if (ap == (DICT_ATTR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: No such attribute:vendor (%d:%d)", func,
		       attr, vend_id);
		return (-1);
	}

	return attr_check_by_dict (p_list, ap);
} /* end of attr_check () */

/*************************************************************************
 *
 *	Function: attr_check_by_dict
 *
 *	Purpose: Used to specify what kind of attribute/value pair
 *		 to check.
 *
 *	Returns: success or failure code (0 == success, -1 == failure)
 *
 *************************************************************************/

int
attr_check_by_dict (p_list, ap)

DICT_ATTR_LIST **p_list;
DICT_ATTR       *ap;

{
  	DICT_ATTR_LIST *new;
	static char *func = "attr_check_by_dict";

	if (ap == (DICT_ATTR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: No dictionary entry", func);
		return (-1);
	}

	if (p_list == (DICT_ATTR_LIST **) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: No list to add dictionary entry '%s' to", func,
		       ap->name);
		return (-1);
	}


	/*
	 * Check to see if it's in the check-list already. If so, 
	 * then don't add it again (but treat this as successful.)
	 */
	for (new = *p_list; new != (DICT_ATTR_LIST *) NULL; new = new->next)
	{
		if (new->ap == ap)
		{
			return (0);	/* Already found. */
		}
	}

	if ((new = (DICT_ATTR_LIST *) calloc(1, sizeof(DICT_ATTR_LIST)))
					    == (DICT_ATTR_LIST *) NULL)
	{
		logit(LOG_DAEMON, LOG_ERR, "%s: FATAL -- out of memory",
		      func);
		dumpcore = 0;
		abort ();
	}

	new->ap = ap;
	new->next = *p_list;
	*p_list = new;
	return (0);

} /* end of attr_check_by_dict () */

/*************************************************************************
 *
 *	Function: attr_check_by_name
 *
 *	Purpose: Used to specify what kind of attribute/value pair
 *		 to check.
 *
 *	Returns: success or failure code (0 == success, -1 == failure)
 *
 *************************************************************************/

int
attr_check_by_name (p_list, name)

DICT_ATTR_LIST **p_list;
char            *name;

{
	DICT_ATTR *ap = dict_attrfind (name);
	static char *func = "attr_check_by_name";

	if (ap == (DICT_ATTR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, 
		       "%s: No such dictionary entry, '%s'", func, name);
		return (-1);
	}

	return attr_check_by_dict (p_list, ap);
} /* end of attr_check_by_name () */

/*************************************************************************
 *
 *	Function: attr_check_clear
 *
 *	Purpose: Used to specify what kind of attribute/value pair
 *		 to check.
 *
 *	Returns: success or failure code (0 == success, -1 == failure)
 *
 *************************************************************************/

int
attr_check_clear (p_list)

DICT_ATTR_LIST **p_list;

{
  	int             count = 0;
	DICT_ATTR_LIST *link;
	static char    *func = "attr_check_clear";
	
	if (p_list == (DICT_ATTR_LIST **) NULL)
	{
		return (-1);
	}

	while (*p_list != (DICT_ATTR_LIST *) NULL)
	{
		link = *p_list;
		*p_list =link->next;
		free (link);
		count++;
	}

	return (count);
} /* end of attr_check_clear () */

/*************************************************************************
 *
 *	Function: dict_init
 *
 *	Purpose: Initialize the dictionary.  Read all ATTRIBUTES into
 *		 the dictionary_attributes list.  Read all VALUES into
 *		 the dictionary_values list.
 *
 *************************************************************************/

int
dict_init ()

{
        int             attorval;
	int             flags;
        int             j;
	int             line_no;
	int             type;
	UINT4           value;
	FILE           *dictfd;
	char           *attrstr;
	char           *dummy;
	char           *namestr;
	char           *ptr;
	DICT_ATTR      *attr;
	DICT_ATTR     **attr_last;
	DICT_VALUE     *dval;
	DICT_VALUE    **dval_last;
	VENDOR         *attr_vep;
	VENDOR         *vep;
	char            buffer[256];
	static char    *func = "dict_init";

#define	ATTRIBUTE                1	/* Used by dict_init() */
#define	VALUE                    2	/* Used by dict_init() */

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (vend_init () != 0)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: vend_init() failed", func);
		return (-1);
	}

	sprintf (buffer, "%s/%s", radius_dir, RADIUS_DICTIONARY);
	if ((dictfd = fopen (buffer, "r")) == (FILE *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: Couldn't open dictionary: %s",
			 func, buffer);
		return (-1);
	}

	attr_last = &dictionary_attributes;
	dval_last = &dictionary_values;
	line_no = 0;
	while (fgets (buffer, sizeof (buffer), dictfd) != (char *) NULL)
	{
		line_no++;

		/* Skip empty space */
		if (*buffer == COMMENT || *buffer == '\0' || *buffer == '\n')
		{
			continue;
		}

		/* Check for DICT_ID, etc. */
		if (*buffer == '%')
		{
			if ((ptr = strtok (buffer, " \t\n\r")) != (char *) NULL)
			{
				if (strcasecmp (ptr, "%DICTID") == 0)
				{
					if ((ptr = strtok (NULL, "\n\r"))
							!= (char *) NULL)
					{
						if (dict_id != (char *) NULL)
						{
							free (dict_id);
						}

						j = strlen (ptr);

						if (j > MAX_DICTID_LEN)
						{
						    ptr[MAX_DICTID_LEN] = '\0';
						}
						dict_id = rad_strdup (ptr);
					}
				}
			}
			continue;	/* treat as a comment. */
		}

		/* Parse the regular dictionary */
		attorval = 0;  /* assume the worst */
		if ((ptr = strtok (buffer, " \t\n\r")) != (char *) NULL)
		{
			for (vep = vendors; vep; vep = vep->next)
			{
				if (strcasecmp (vep->attr_name, buffer) == 0)
				{
					attorval = ATTRIBUTE;
					break;
				}

				if (strcasecmp (vep->value_name, buffer) == 0)
				{
					attorval = VALUE;
					break;
				}
			}
		}
		else /* Must have been a "blank" line (?) */
		{
			continue;
		}

		switch (attorval)
		{
		    case ATTRIBUTE:
			/* Read the ATTRIBUTE name */
			if ((attrstr = strtok (NULL, " \t\n\r"))
							== (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
			       "%s: Invalid attribute on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			ptr = parse_for_vendor (attrstr, &attr_vep);

			if (ptr == (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				  "%s: Missing vendor on line %d of dictionary",
					func, line_no);
				if (have_vendors == 0)
				{
					logit (LOG_DAEMON, LOG_ALERT,
			     "%s: Vendors file is required for this dictionary",
						func);
				}
				return (-1);
			}

			if (ptr != attrstr)
			{
				vep = attr_vep;
				attrstr = ptr;
			}

			if ((ptr = strtok (NULL, " \t\n\r")) == (char *) NULL ||
								!isdigit(*ptr))
			{
				logit (LOG_DAEMON, LOG_ALERT,
				 "%s: Invalid value on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			/* Attribute index value */
			value = strtol (ptr, &dummy, 0);

			if ((ptr = strtok (NULL, " \t\n\r")) == (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				 "%s: Invalid type on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			if (strcmp (ptr, "string") == 0)
			{
				type = PW_TYPE_STRING;
			}

#ifdef PW_TYPE_FILTER_BINARY
			else if (strcmp (ptr, "abinary") == 0)
			{
				type = PW_TYPE_FILTER_BINARY;
			}
#endif	/* PW_TYPE_FILTER_BINARY */

			else if (strcmp (ptr, "integer") == 0)
			{
				type = PW_TYPE_INTEGER;
			}
			else if (strcmp (ptr, "short") == 0)
			{
				type = PW_TYPE_SHORT;
			}
			else if (strcmp (ptr, "octet") == 0)
			{
				type = PW_TYPE_OCTET;
			}
			else if (strcmp (ptr, "ipaddr") == 0)
			{
				type = PW_TYPE_IPADDR;
			}
			else if (strcmp (ptr, "date") == 0)
			{
				type = PW_TYPE_DATE;
			}
			else if (strcmp (ptr, "octets") == 0)
			{
				type = PW_TYPE_OCTETS;
			}
			else if (strcmp (ptr, "vendor") == 0)
			{
				type = PW_TYPE_VENDOR;
			}
			else if (strcmp (ptr, "tag-int") == 0)
			{
				type = PW_TYPE_TAG_INT;
			}
			else if (strcmp (ptr, "tag-str") == 0)
			{
				type = PW_TYPE_TAG_STR;
			}
			else
			{
				logit (LOG_DAEMON, LOG_ALERT,
				  "%s: Invalid type on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			if (parse_flags (&flags) == -1)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				   "%s: Invalid flags on line %d of dictionary",
					func, line_no);
				return (-1);
			}

			/* Create a new attribute for the list */
			if ((attr =
				(DICT_ATTR *) calloc (1, sizeof (DICT_ATTR)))
							== (DICT_ATTR *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: FATAL out of memory", func);
				abort ();
			}
			attr->name = add_string (attrstr, ASIS);
			attr->value = value;
			attr->type = type;
			attr->vendor_id = vep->id;
			attr->vendor_ptr = vep;
			attr->flags = flags;
			attr->next = (DICT_ATTR *) NULL;
			attr->vnext = (DICT_ATTR *) NULL;

			/* Insert it at end of list */
			*attr_last = attr;
			attr_last = &attr->next;

			dprintf(3, (LOG_DAEMON, LOG_DEBUG,
"%s: dict line %d, attr %s, type = %d, val %d [0x%08X], vend %d, flags 0x%08X",
					func, line_no, attrstr, type, value,
					value, vep->id, flags));

			break;	/* from case attorval */

		    case VALUE:
			/* Read the ATTRIBUTE name */
			if ((attrstr = strtok (NULL, " \t\n\r"))
							== (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				   "%s: Invalid VALUE on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			if ((namestr = strtok (NULL, " \t\n\r"))
							== (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				   "%s: Invalid VALUE on line %d of dictionary",
					 func, line_no);
				return (-1);
			}

			if ((ptr = parse_for_vendor (attrstr, &attr_vep))
							!= attrstr)
			{
				vep = attr_vep;
				attrstr = ptr;
			}

			if ((ptr = strtok (NULL, " \t\n\r")) == (char *) NULL ||
						(!isdigit(*ptr) && *ptr != '-'))
			{
				logit (LOG_DAEMON, LOG_ALERT,
				   "%s: Invalid value on line %d of dictionary",
					func, line_no);
				return (-1);
			}

			if (((attr = dict_attrfind (attrstr))
						!= (DICT_ATTR *) NULL) &&
				(attr->type == PW_TYPE_IPADDR))
			{
				if (good_ipaddr (ptr) == 0)	/* IP address */
				{
					value = ntohl(inet_addr (ptr));
				}
				else /* was not dotted quad notation */
				{
					value = strtol (ptr, &dummy, 0);
				}
			}
			else /* normal integer value */
			{
				/* Value as integer */
				value = strtol (ptr, &dummy, 0);
			}

			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
	       "%s: dict line %d, value %s = %lu [0x%08X], attr %s, type = %d",
				func, line_no, namestr, value, value,
				attrstr, attr ? attr->type : -1));

			/* Create a new VALUE entry for the list */
			if ((dval =
				(DICT_VALUE *) calloc (1, sizeof (DICT_VALUE)))
							== (DICT_VALUE *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: FATAL out of memory", func);
				abort ();
			}
			dval->attrname = add_string (attrstr, ASIS);
			dval->name = add_string (namestr, ASIS);
			dval->dv_value = value;
			dval->vendor_ptr = vep;
			dval->next = (DICT_VALUE *) NULL;
			dval->vnext = (DICT_VALUE *) NULL;

			/* Insert it into the list */
			*dval_last = dval;
			dval_last = &dval->next;

 			break; /* from case attorval */

		    default:
			logit (LOG_DAEMON, LOG_ALERT,
                                "%s: Invalid line %d in dictionary",
                                func, line_no);
                        return (-1);
		} /* end of switch */
	} /* end of while */

	fclose (dictfd);
	return vend_verify ();  /* Set up and verify vendor info */
} /* end of dict_init () */

/*************************************************************************
 *
 *	Function: dict_attrget
 *
 *	Purpose: Return the full attribute structure based on the
 *		 attribute id number.
 *
 *************************************************************************/

DICT_ATTR *
dict_attrget (attribute, vid)

int             attribute;
UINT4           vid;

{
	VENDOR         *vep;
	DICT_ATTR      *attr;

	if ((vep = find_vendor_by_id (vid)) == (VENDOR *) NULL)
	{
		return (DICT_ATTR *) NULL;
	}

	for (attr = vep->attrs; attr; attr = attr->vnext)
	{
		if (attr->value == attribute)
		{
			break;	
		}
	}

	return attr;
} /* end of dict_attrget () */

/*************************************************************************
 *
 *	Function: dict_attrfind
 *
 *	Purpose: Return the full attribute structure based on the
 *		 attribute name.
 *
 *************************************************************************/

DICT_ATTR *
dict_attrfind (attrname)

char           *attrname;

{
	char	       *ptr;
	DICT_ATTR      *attr;
	VENDOR 	       *vep;
	static char    *func = "dict_attrfind";

	if ((ptr = parse_for_vendor (attrname, &vep)) == (char *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: invalid vendor name in attribute name '%s'",
			func, attrname);
		return (DICT_ATTR *) NULL;
	}

	/* Look on global list if vendor not known */
	if (ptr == attrname)
	{
		for (attr = dictionary_attributes; attr; attr = attr->next)
		{
			if (strcasecmp (attr->name, ptr) == 0)
			{
				break;
			}
		}
		return attr;
	}

	for (attr = vep->attrs; attr; attr = attr->vnext)
	{	
		if (strcasecmp (attr->name, ptr) == 0)
		{
			break;
		}
	}

	return attr;
} /* end of dict_attrfind () */

/*************************************************************************
 *
 *	Function: dict_find_value
 *
 *	Purpose: Return the full value structure based on the
 *		 actual value and the associated attribute.
 *
 *	Returns: Pointer to DICT_VALUE object found,
 *		 or, NULL, if failed.
 *
 *************************************************************************/

DICT_VALUE *
dict_find_value (value, attr)

UINT4           value;
DICT_ATTR      *attr;

{
        DICT_VALUE     *val;
	static char    *func = "dict_find_value";

        for (val = attr->vendor_ptr->values; val; val = val->vnext)
        {
		if (val->dv_value == value && val->attrnum == attr->value)
		{
			break;
		}
        }

        return val;
} /* end of dict_find_value () */

/*************************************************************************
 *
 *	Function: dict_valfind
 *
 *	Purpose: Return the full value structure based on the
 *		 value name.
 *
 *************************************************************************/

DICT_VALUE *
dict_valfind (value, attrname)

char           *value;
char           *attrname;

{
        DICT_VALUE     *val;
	DICT_ATTR      *attr;
	static char    *func = "dict_valfind";

	if (attrname == (char *) NULL)
	{
		/* No attrname given; Find first match on global list */
        	for (val = dictionary_values; val; val = val->next)
		{
			if (strcasecmp (val->name, value) == 0)
			{
				break;
			}
		}
		return val;
        }

	/* Search vendor list of values for match */
	if ((attr = dict_attrfind (attrname)) == (DICT_ATTR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: invalid attribute name '%s'",
			func, attrname);
		return (DICT_VALUE *) NULL;
	}

        for (val = attr->vendor_ptr->values; val; val = val->vnext)
        {
		if (strcasecmp (val->name, value) == 0 &&
			val->attrnum == attr->value)
		{
			break;
		}
        }
        return val;
} /* end of dict_valfind () */

/*************************************************************************
 *
 *	Function: dict_valget
 *
 *	Purpose: Return the full value structure based on the
 *		 actual value and the associated attribute name.
 *
 *************************************************************************/

DICT_VALUE *
dict_valget (value, attrname)

UINT4           value;
char           *attrname;

{
        DICT_ATTR      *attr;
	static char    *func = "dict_valget";

	if ((attr = dict_attrfind (attrname)) == (DICT_ATTR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: invalid attribute name '%s'", func, attrname);
		return (DICT_VALUE *) NULL;
	}

	return dict_find_value (value, attr);
} /* end of dict_valget () */

/*****************************************************************************
 *
 *	Function: find_vendor_by_id
 *
 *	Purpose:  Returns VENDOR data structure for vendor with given id
 *
 *****************************************************************************/

VENDOR *
find_vendor_by_id (vendor_id)

UINT4        vendor_id;

{
	VENDOR  *vend_ptr;

	for (vend_ptr = vendors; vend_ptr; vend_ptr = vend_ptr->next)
	{
		if (vend_ptr->id == vendor_id)
		{
			break;
		}
	}
	return vend_ptr;
} /* end of find_vendor_by_id () */

/*****************************************************************************
 *
 *	Function: find_vendor_by_name
 *
 *	Purpose:  Returns VENDOR data structure for vendor with given name
 *
 *****************************************************************************/

VENDOR *
find_vendor_by_name (vendor_name)

char              *vendor_name;

{
	VENDOR  *vend_ptr;

	for (vend_ptr = vendors; vend_ptr; vend_ptr = vend_ptr->next)
	{
		if (strcasecmp (vend_ptr->name, vendor_name) == 0)
		{
			break;
		}
	}
	return vend_ptr;
}  /* end of find_vendor_by_name () */

/*************************************************************************
 *
 *	Function: free_vendor_list
 *
 *	Purpose: Free all VENDOR_LIST entries and count results.
 *
 *************************************************************************/

int
free_vendor_list (veps)

VENDOR_LIST    *veps;

{
	int             count = 0;
	VENDOR_LIST    *next;
	static char    *func = "free_vendor_list";

	while (veps != (VENDOR_LIST *) NULL)
	{
		next = veps->next;
		free (veps);
		veps = next;
		count++;
		vendor_list_mf.f++;
	}

	return count;
} /* end free_vendor_list () */

/*************************************************************************
 *
 *	Function: parse_flags
 *
 *	Purpose: Return the attribute flags value specified in dictionary.
 *
 *	Remarks: parses the following syntax:
 *
 *		"(" <tail>
 *		<tail> ::=  "config"
 *			    "1," <nak_tail>
 *			    "*," <nak_tail>
 *			    "0," <nak_tail>
 *			    ","  <nak_tail>
 *
 *		<nak_tail> ::= "1" [ "," <etc_tail> ]
 *			       "0" [ "," <etc_tail> ]
 *			       "*" [ "," <etc_tail> ]
 *			       [ "," <etc_tail> ]
 *
 *		<etc_tail> ::=  "must" | "may" | "nolog" | "encaps" | "noencaps"
 *
 *************************************************************************/

static int
parse_flags (flags)

int            *flags;

{
	enum
	{
		FIND_LP,	/* Look for leading "(" */
		FIND_AF,
		FIND_CO1,
		FIND_NF,
		FIND_CO2,
		FIND_ETC,
		FIND_END
	} mode;

	char           *ptr;
	static char    *func = "parse_flags";

	/* Set up default values, in case nothing was specified. */
	*flags = ATTR_ACK_NONE | ATTR_NAK_NONE | ATTR_ENCAPS;

	if ((ptr = strtok (NULL, ")")) == (char *) NULL)
	{
		return 0;
	}

	mode = FIND_LP;
	while (*ptr != '\0')
	{
		if (isspace(*ptr))
		{
			ptr++;
			continue;
		}

		switch (mode)
		{
		    case FIND_LP:
			if (*ptr++ != '(')
			{
				return (0);
			}

			mode = FIND_AF;
			break;

		    case FIND_AF:
			if (strcasecmp (ptr, "config") == 0)
			{
				*flags = ATTR_CONFIG;
				mode = FIND_END;
				ptr += 6;
				return (0);
			}

			mode = FIND_CO1;
			switch (*ptr++)
			{
			    case '1':
				*flags |= ATTR_ACK_ONE;
				/****FALLTHROUGH****/

			    case '*':
				*flags &= ~ATTR_ACK_NONE;
				break;

			    case '0':
				break;

			    case ',':	/* Nothing - Assume default */
				mode = FIND_NF;
				break;

			    default:
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Junk in col one, '%s'",
					func, ptr);
				return (-1);
			}

			break;

		    case FIND_CO1:
			if (*ptr++ != ',')
			{
				return (-1);
			}

			mode = FIND_NF;
			break;

		    case FIND_NF:
			mode = FIND_CO2;

			switch (*ptr++)
			{
			    case '1':
				*flags |= ATTR_NAK_ONE;
				/****FALLTHROUGH****/

			    case '*':
				*flags &= ~ATTR_NAK_NONE;
				break;

			    case '0':
				break;

			    case ',':	/* Nothing - use default */
				mode = FIND_ETC;
				break;

			    default:
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Junk after column two, '%s'",
					func, ptr);
				return (-1);
			}

			break;

		    case FIND_CO2:
			if (*ptr++ != ',')
			{
				return (-1);
			}

			mode = FIND_ETC;
			break;

		    case FIND_ETC:
			/* Look for Version 2 MAY/MUST indicator */
			if (strncasecmp (ptr, "MUST", 4) == 0)
			{
				*flags |= ATTR_MUST;
				ptr += 4;
			}
			else if (strncasecmp (ptr, "MAY", 3) == 0)
			{
				if (*flags & ATTR_MUST)
				{
					return (-1);
				}
				ptr += 3;
			}
			else if (strncasecmp (ptr, "NOLOG", 5) == 0)
			{
				*flags |= ATTR_NOLOG;
				ptr += 5;
			}
			else if (strncasecmp (ptr, "ENCAPSULATE", 11) == 0)
			{
				*flags |= ATTR_ENCAPS;
				ptr += 11;
			}
			else if (strncasecmp (ptr, "ENCAPS", 6) == 0)
			{
				*flags |= ATTR_ENCAPS;
				ptr += 6;
			}
			else if (strncasecmp (ptr, "NOENCAPS", 6) == 0)
			{
				*flags &= ~ATTR_ENCAPS;
				ptr += 8;
			}
			else
			{
				logit (LOG_DAEMON, LOG_ERR,
				  "%s: Invalid flags text in dictionary, '%s'",
					func, ptr);
				return (-1);
			}

			while (isspace(*ptr))
			{
				ptr++;
			}

			if (*ptr == '\0')
			{
				return 0;
			}

			if (*ptr != ',')
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Invalid trailing text '%s'",
					func, ptr);
				return (-1);
			}

			ptr++;
			break;

		    case FIND_END:
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Shouldn't reach here with '%s'",
				func, ptr);
			return (-1);	/* What is this stuff at end? */
		} /* end of switch */
	} /* end of while */

	return 0;

} /* end of parse_flags */

/*****************************************************************************
 *
 *	Function: parse_for_vendor
 *
 *	Purpose: Checks for <vendor>:<attrname> and returns vendor ID
 *
 *****************************************************************************/

static char *
parse_for_vendor (name, vendor)

char              *name;
VENDOR           **vendor;

{
	char         *ptr;
	static char  *func = "parse_for_vendor";

	if ((ptr = strchr (name, ':')) == (char *) NULL)
	{
		if (DEFAULT_VENDOR_ID == 0)
		{
			*vendor = (VENDOR *) NULL;
		}
		else
		{
			*vendor = find_vendor_by_id (DEFAULT_VENDOR_ID);
		}
		return name;
	}

	*ptr = '\0';

	/* Check to see if there is a "none" vendor. */
	if (strcasecmp (name, "none") == 0)
	{
		*vendor = (VENDOR *) NULL;
	}
	else
	{
		if ((*vendor = find_vendor_by_name (name)) == (VENDOR *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: Cannot find vendor named '%s'",
				func, name);
			*ptr = ':';
			return (char *) NULL;
		}
	}

	*ptr++ = ':';
	return ptr;
} /* end of parse_for_vendor () */

/*****************************************************************************
 *
 *	Function: parse_for_vendor_list
 *
 *	Purpose: Checks for <vendor>[+<vendor>...]:<attrname> and 
 *		 returns list of vendor ids.
 *
 *****************************************************************************/

char *
parse_for_vendor_list (name, vep_list)

char              *name;
VENDOR_LIST      **vep_list;

{
	char         *each;
	char         *next;
	char         *ptr;
	VENDOR       *vendor;
	VENDOR_LIST  *new;
	static char  *func = "parse_for_vendor_list";

	(void) free_vendor_list (*vep_list);
	*vep_list = (VENDOR_LIST *) NULL;

	if ((ptr = strchr (name, ':')) == (char *) NULL)
	{
		if (DEFAULT_VENDOR_ID != 0)
		{
			*vep_list = (VENDOR_LIST *) calloc (1,
							sizeof (VENDOR_LIST));
			if (*vep_list == (VENDOR_LIST *) NULL)
			{
				logit (LOG_DAEMON, LOG_CRIT,
				    "%s: FATAL, calloc(1, %d) for VENDOR_LIST",
					func, sizeof (VENDOR_LIST));
				abort ();
			}

			vendor_list_mf.m++;
			(*vep_list)->vep =
					find_vendor_by_id (DEFAULT_VENDOR_ID);
			(*vep_list)->next = (VENDOR_LIST *) NULL;
		}
		return name;
	}

	*ptr = '\0';

	/* Check to see if there is a "none" vendor. */
	if (strcasecmp (name, "none") != 0)
	{
		for (each = name, next = strchr (each, '+');
			each != NULL;
			each = next, next = strchr (each, '+'))
		{
			/* If not end of list, mark off this vendor... */
			if (next != NULL)
			{
				*next = '\0';	/* Terminate vendor name */
				next++;		/* Ready for next... */
			}

			if ((vendor = find_vendor_by_name (each))
							== (VENDOR *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: Cannot find vendor named '%s'",
					func, each);
				*ptr = ':';
				return (char *) NULL;
			}
			
			new = (VENDOR_LIST *) calloc (1, sizeof (VENDOR_LIST));
			if (new == (VENDOR_LIST *) NULL)
			{
				logit (LOG_DAEMON, LOG_CRIT,
				    "%s: FATAL, calloc(1, %d) for VENDOR_LIST",
					func, sizeof (VENDOR_LIST));
				abort ();
			}

			vendor_list_mf.m++;

			new->vep = vendor;
			new->next = (VENDOR_LIST *) NULL;
			*vep_list = new;
			vep_list = &(new->next);

			/* Short circut test above. */
			if (next == NULL)
			{
				break;
			}
		}
	} /* end of if "none" */

	*ptr++ = ':';

	return ptr;
} /* end of parse_for_vendor_list () */

#define	PARSE_VENDOR  1
#define	PARSE_MAP     2

/******************************************************************************
 *
 *	Function: vend_init
 *
 *	Purpose:  Initialize the vendors list
 *
 *****************************************************************************/

static int
vend_init ()

{
	int             j;
	int             line_no;
	int             mode = PARSE_VENDOR;
	UINT4           s_attr;
	UINT4           s_attr2;
	UINT4           v_attr;
	UINT4           vendor_id;
	char           *attrstr;
	char           *dummy;
	char           *valstr;
	char           *ptr;
	char           *vend_name;
	FILE           *vendor_fd;
	VENDOR         *vend_tmp;
	VENDOR         *vend_last;
	VENDOR_MAP     *map_ent;
	char            buffer[256];
	static char    *func = "vend_init";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Create dummy vendor entry for RADIUS */
	if ((vend_tmp = (VENDOR *) calloc (1, sizeof (VENDOR)))
							== (VENDOR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: FATAL out of memory", func);
		abort ();
	}

	vendor_mf.m++;
	vend_tmp->attr_name = add_string ("ATTRIBUTE", ASIS);
	vend_tmp->value_name = add_string ("VALUE", ASIS);
	vend_tmp->name = add_string ("RADIUS", ASIS);
	vend_tmp->id = 0;
	vend_tmp->map = (VENDOR_MAP *) NULL;
	vend_tmp->attrs = (DICT_ATTR *) NULL;
	vend_tmp->values = (DICT_VALUE *) NULL;
	vend_tmp->next = (VENDOR *) NULL;

	vendors = vend_tmp;	/* Put it on the list. */
	vend_last = vend_tmp;

	sprintf (buffer, "%s/%s", radius_dir, RADIUS_VENDORS);
	if ((vendor_fd = fopen (buffer, "r")) == (FILE *) NULL)
	{
		dprintf(1, (LOG_DAEMON, LOG_ALERT,
			"%s: Assuming no Vendor Specific attributes defined",
			func));
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: No vendors file found", func);
		return 0;
	}

	have_vendors = 1;
	line_no = 0;
	while (fgets (buffer, sizeof (buffer), vendor_fd) != (char *) NULL)
	{
		line_no++;

		/* Skip empty space */
		if (*buffer == COMMENT || *buffer == '\0' || *buffer == '\n')
		{
			continue;
		}

		/* Check for VEND_ID, etc. */
		if (*buffer == '%')
		{
			if ((ptr = strtok (buffer, " \t\n\r")) != (char *) NULL)
			{
				if (strcasecmp (ptr, "%VENDORSID") == 0)
				{
					if ((ptr = strtok (NULL, "\n\r"))
							!= (char *) NULL)
					{
						if (vend_id != (char *) NULL)
						{
							free (vend_id);
						}

						j = strlen (ptr);

						if (j > MAX_VENDID_LEN)
						{
						    ptr[MAX_VENDID_LEN] = '\0';
						}
						vend_id = rad_strdup (ptr);
					}
				}
			}
			continue;	/* treat as a comment. */
		}

		/* See if we're reading attribute mapping table */
		if (mode == PARSE_MAP)
		{
			/* Check if we're to return to parsing vendor info */
			if (*buffer == ')')
			{
				mode = PARSE_VENDOR;
				continue;
			}

			if ((map_ent = vend_tmp->map) == (VENDOR_MAP *) NULL)
			{
				if ((map_ent = (VENDOR_MAP *) 
					calloc (1, sizeof (VENDOR_MAP)))
							== (VENDOR_MAP *) NULL)
				{
					logit (LOG_DAEMON, LOG_ALERT,
						"%s: FATAL out of memory",
						func);
					abort ();
				}
				map_ent->vid = vend_tmp->id;
				vend_tmp->map = map_ent;
			}

			/* Assume bad values */
			s_attr = -1;
			s_attr2 = -1;
			v_attr = -1;

			if ((ptr = strtok (buffer, " \t\n\r")) != (char *) NULL)
			{
				if (isdigit(*ptr))
				{
					s_attr = strtol (ptr, &dummy, 0);
					s_attr2 = strtol (ptr, &dummy, 0);
				}
			}

			if ((ptr = strtok (NULL, " \t\n\r")) != (char *) NULL)
			{
				if (isdigit(*ptr))
				{
					v_attr = strtol (ptr, &dummy, 0);
				}
			}

			if ((ptr = strtok (NULL, " \t\n\r")) != (char *) NULL)
			{
				if (isdigit(*ptr))
				{
					s_attr2 = v_attr;
					v_attr = strtol (ptr, &dummy, 0);
				}
			}

			if (s_attr > 255 ||
				s_attr2 > 255 ||
				v_attr > 255 ||
				s_attr > s_attr2)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: Invalid line %d in vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}

			while (s_attr <= s_attr2)
			{
				/* Request map */
				map_ent->s_attr[s_attr] = v_attr;
				/* Reply map */
				map_ent->v_attr[v_attr] = s_attr;
				s_attr++;
				v_attr++;
			}

			continue;
		}

		/* Check if attribute mapping table supplied */
		if (*buffer == '(' && vend_tmp)
		{
			mode = PARSE_MAP;
			continue;
		}

		valstr = (char *) NULL;

		/* Can have vendor number _OR_ attribute string here */
		if ((attrstr = strtok (buffer, " \t\n\r")) == (char *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: Invalid line %d in vendors",
				func, line_no);
			fclose (vendor_fd);
			return (-1);
		}

		if (!isdigit(*attrstr)) /* was ASCII attribute string */
		{
			if ((valstr = strtok (NULL, " \t\n\r"))
							== (char *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
		  "%s: Missing/invalid Vendor Value name on line %d of vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}

			if ((ptr = strtok (NULL, " \t\n\r")) == (char *) NULL ||
								!isdigit(*ptr))
			{
				logit (LOG_DAEMON, LOG_ALERT,
			      "%s: Missing Vendor number on line %d of vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}
		}
		else /* was numeric vendor code */
		{
			ptr = attrstr;
			attrstr = (char *) NULL;
		}

		/* Obtain numeric vendor code. */
		vendor_id = atol (ptr);
		vend_name = strtok (NULL, " \t\n\r");

		for (vend_tmp = vendors;
			vend_tmp != (VENDOR *) NULL;
			vend_tmp = vend_tmp->next)
		{
			if ((attrstr != (char *) NULL) &&
			    (strcasecmp ("ATTRIBUTE", attrstr) != 0) &&
			     (strcasecmp (vend_tmp->attr_name, attrstr) == 0 ||
			       strcasecmp (vend_tmp->value_name, attrstr) == 0))
			{
				logit (LOG_DAEMON, LOG_ALERT,
	      "%s: Duplicate Vendor Attribute Identifier on line %d of vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}

			if ((valstr != (char *) NULL) &&
			    (strcasecmp ("VALUE", valstr) != 0) &&
			     (strcasecmp (vend_tmp->value_name, valstr) == 0 ||
			       strcasecmp (vend_tmp->attr_name, valstr) == 0))
			{
				logit (LOG_DAEMON, LOG_ALERT,
   		"%s: Duplicate Vendor Value Identifier on line %d of vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}

			if (vendor_id == vend_tmp->id)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				"%s: Duplicate Vendor ID on line %d of vendors",
					func, line_no);
				fclose (vendor_fd);
				return (-1);
			}

			if (vend_name != (char *) NULL && *vend_name)
			{
				if (strcasecmp (vend_name, vend_tmp->name) == 0)
				{
					logit (LOG_DAEMON, LOG_ALERT,
			      "%s: Duplicate Vendor name on line %d of vendors",
 						func, line_no);
					fclose (vendor_fd);
					return (-1);
				}
			}
		}

		if ((vend_tmp = (VENDOR *) calloc (1, sizeof (VENDOR)))
							== (VENDOR *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: FATAL out of memory", func);
			abort ();
		}

		vendor_mf.m++;
		vend_tmp->attr_name = add_string (attrstr, ASIS);
		vend_tmp->value_name = add_string (valstr, ASIS);
		vend_tmp->name = add_string (vend_name, ASIS);
		vend_tmp->id = vendor_id;
		vend_tmp->map = (VENDOR_MAP *) NULL;
		vend_tmp->attrs = (DICT_ATTR *) NULL;
		vend_tmp->values = (DICT_VALUE *) NULL;

		vend_tmp->next = (VENDOR *) NULL;
		vend_last->next = vend_tmp;
		vend_last = vend_tmp;
	} /* end of while loop */

	fclose (vendor_fd);

	return 0;
} /* end of vend_init () */

/******************************************************************************
 *
 *	Function: vend_verify
 *
 *	Purpose: Verify any attribute mapping tables that we read in.
 *
 *****************************************************************************/

static int
vend_verify ()

{
	int             i;
	int             result = 0;
	VENDOR	       *vep;
	VENDOR_MAP     *vmp;
	DICT_ATTR      *attr;
	DICT_ATTR      *da;
	DICT_ATTR     **da_prev;
	DICT_VALUE     *value;
	DICT_VALUE     *dv;
	DICT_VALUE    **dv_prev;
	static char    *func = "vend_verify";

	/* 
	 * Check all attribute and values read in for duplicates or
	 * inconsistencies.  Build up lists of attrs. and values rooted in
	 * the appropriate vendor structures.  This is to speed up access.
         * Note that there is a vendor structure for OID zero, the standard
         * RADIUS attributes and values.
	 */

	for (attr = dictionary_attributes; attr; attr = attr->next)
	{
		for (da_prev = &attr->vendor_ptr->attrs; 
			(da = *da_prev) != (DICT_ATTR *) NULL; 
			da_prev = &da->vnext)
		{
			if (strcasecmp (da->name, attr->name) == 0)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: %s Attr. %s is multiply defined",
					func, attr->vendor_ptr->name,
					attr->name);
				result = -1;
				continue;
			}

			if (da->value == attr->value && da->type != attr->type)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				    "%s: %s Attrs. %s and %s have type confict",
					func, attr->vendor_ptr->name, 
					attr->name, da->name);
				result = -1;
				continue;
			}
		}
		attr->vnext = (DICT_ATTR *) NULL;	
		*da_prev = attr;
	}	

	for (value = dictionary_values; value; value = value->next)
	{
		for (attr = value->vendor_ptr->attrs; attr; attr = attr->vnext)
		{
			if (strcasecmp (attr->name, value->attrname) == 0)
			{
				break;
			}
		}

		if (attr != (DICT_ATTR *) NULL)
		{
			value->attrnum = attr->value;
		}
		else if (strcasecmp (value->attrname, "Server-Config") == 0)
		{
			/* Special name for configuration attribute */
			value->attrnum = -1;
		}
		else
		{
			logit (LOG_DAEMON, LOG_ALERT,
			       "%s: %s Value %s refers to non-existent attr %s",
				func, value->vendor_ptr->name, value->name,
				value->attrname);
			result = -1;
			continue;
		}

		for (dv_prev = &value->vendor_ptr->values; 
			(dv = *dv_prev) != (DICT_VALUE *) NULL; 
			dv_prev = &dv->vnext)
		{
			if (strcasecmp (dv->name, value->name) == 0 &&
				dv->attrnum == value->attrnum)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: %s Value %s is multiply defined",
					func, value->vendor_ptr->name, 
					value->name);
				result = -1;
				continue;
			}
		}
		value->vnext = (DICT_VALUE *) NULL;	
		*dv_prev = value;
	}	

	/* Now check any vendor attribute maps */
	for (vep = vendors; vep; vep = vep->next)
	{
		if ((vmp = vep->map) == (VENDOR_MAP *) NULL)
		{
			continue;
		}

		for (i = 0; i < 256; i++)
		{
			if (vmp->s_attr[i] == 0)
			{
				continue;
			}

			if (dict_attrget (vmp->s_attr[i], vep->id) 
							== (DICT_ATTR *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
				  "%s: Vendor Attr. %d in %s map is undefined ",
				       func, vmp->s_attr[i], vep->name);
/* Just log for now		result = -1; */
			}
		}
	}
	return result;
} /* end of vend_verify () */

/*************************************************************************
 *
 *	Function: vendor_list_toa
 *
 *	Purpose: Produce an ASCII list of vendor names.
 *
 *************************************************************************/

char *
vendor_list_toa (veps)

VENDOR_LIST    *veps;

{
	static char     buffers[20][40];
	static int      pos = 0;
	char           *buff = buffers[pos];
	static char    *func = "vendor_list_toa";

	if (veps == (VENDOR_LIST *) NULL)
	{
		return "none";
	}
	else
	{
		if (veps->next == (VENDOR_LIST *) NULL)
		{
			return veps->vep->name;
		}
	}

	/* Print out concatenated list. */
	*buff = '\0';

	for ( ; veps != (VENDOR_LIST *) NULL ; veps = veps->next)
	{
		strcat (buff, veps->vep->name);
		if (veps->next)
		{
			strcat (buff, "+");
		}
	}

	pos++;
	if (pos >= 20)
	{
		pos = 0;
	}

	return buff;
} /* end of vendor_list_toa () */
