/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation
 * for any purpose and without fee is hereby granted, provided:
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof,
 *
 * 2) that both the copyright notice and this permission and disclaimer notice
 *    appear in all supporting documentation, and
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/*
 *
 * dnscheck -- Check to see if the host's hostname/DNS configuration works.
 *
 * No public entry points in this file.  This is a standalone program.
 *
 */

static char     rcsid[] =
		"$Id: dnscheck.c,v 1.4 1998/07/06 16:34:21 web Exp $";

#include	<sys/param.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>

#include	<netdb.h>
#include	<stdio.h>
#include	<memory.h>
#include	<string.h>

#include	"radius.h"

#if defined(ultrix) || defined(__hpux) || defined(__bsdi__) || defined(linux) || defined(SCO)
extern int      h_errno;
#endif	/* ultrix */

/* Some private globals */

static char    *progname = "?";
static int      debug = 0;	/* To help the developer out. */
static int      terse = 0;	/* Turn this on to make it quieter. */
static int      check = 1;	/* Turn this on for RADIUS checking. */
static int      notice = 1;	/* Show our copyright. */

typedef struct named_list_struct
{
	struct named_list_struct *next;
	struct my_hostent_struct *mp;
} named_list_t;

static named_list_t *cached_name = NULL;
static named_list_t *cached_addr = NULL;

typedef struct my_hostent_struct
{
	char           *h_name;		/* just like struct hostent */
	char          **h_aliases;	/* just like struct hostent */
	int             h_addrtype;	/* just like struct hostent */
	int             h_length;	/* just like struct hostent */
	char          **h_addr_list;	/* just like struct hostent */

	/* Our special stuff. */

	struct my_hostent_struct  *m_name; /* For gethostbyaddr() thingy. */
	struct my_hostent_struct **m_addr_list;	/* Quick reference to other. */
} my_hostent_t;

/*
 * internal procedures
 */

static my_hostent_t *already_addrd PROTO((char *));
static my_hostent_t *already_named PROTO((char *));
static my_hostent_t *cache_hostent_addr PROTO((my_hostent_t *));
static my_hostent_t *cache_hostent_name PROTO((my_hostent_t *));
static void          copyright PROTO((void));
static void          dns_usage PROTO((void));
static void          free_my_hostent PROTO((my_hostent_t *));
static my_hostent_t *make_my_hostent PROTO((struct hostent *));
static char         *my_strdup PROTO((CONST char *));
static int           report PROTO((char *, char *, my_hostent_t *));
static my_hostent_t *report_addr PROTO((int, int, char *));
static my_hostent_t *report_name PROTO((int, int, char *));

/*****************************************************************************
 *
 *	Function: already_addrd
 *
 *	Purpose:  Find a cached address.
 *
 ****************************************************************************/

static my_hostent_t *
already_addrd (addr)

char           *addr;

{
	named_list_t   *each;
	char          **p;

	for (each = cached_addr; each; each = each->next)
	{
		for (p = each->mp->h_addr_list; *p; p++)
		{
			if (memcmp (addr, *p, 4) == 0)
			{
				return each->mp;
			}
		}
	}
	return NULL;
} /* end of already_addrd () */

/*****************************************************************************
 *
 *	Function: already_named
 *
 *	Purpose: Find a cached dns name
 *
 ****************************************************************************/

static my_hostent_t *
already_named (name)

char           *name;

{
	named_list_t   *each;
	char          **p;

	for (each = cached_name; each; each = each->next)
	{
		if (strcasecmp (name, each->mp->h_name) == 0)
		{
			return each->mp;
		}

		for (p = each->mp->h_aliases; *p; p++)
		{
			if (strcasecmp (name, *p) == 0)
			{
				return each->mp;
			}
		}
	}
	return NULL;
} /* end of already_named () */

/*****************************************************************************
 *
 *	Function: cache_hostent_addr
 *
 *	Purpose: Cache an address (if it isn't cached already.)
 *
 ****************************************************************************/

static my_hostent_t *
cache_hostent_addr (mp)

my_hostent_t   *mp;

{
	named_list_t   *each;
	char          **p;
	char          **q;

	for (each = cached_addr; each; each = each->next)
	{
		for (p = each->mp->h_addr_list; *p; p++)
		{
			for (q = mp->h_addr_list; *q; q++)
			{
				if (memcmp (*p, *q, 4) == 0)
				{
					return each->mp;
				}
			}
		}
	}

	each = (named_list_t *) calloc (1, sizeof (named_list_t));
	each->mp = mp;
	each->next = cached_addr;
	cached_addr = each;
	return each->mp;
} /* end of cache_hostent_addr () */

/*****************************************************************************
 *
 *	Function: cache_hostent_name
 *
 *	Purpose: Cache a dns entry by name (if it isn't cached already)
 *
 ****************************************************************************/

static my_hostent_t *
cache_hostent_name (mp)

my_hostent_t   *mp;

{
	named_list_t   *each;
	char          **p;
	char          **q;

	for (each = cached_name; each; each = each->next)
	{
		if (strcasecmp (mp->h_name, each->mp->h_name) == 0)
		{
			return each->mp;
		}

		for (p = each->mp->h_aliases; *p; p++)
		{
			if (strcasecmp (*p, mp->h_name) == 0)
			{
				return each->mp;
			}

			for (q = mp->h_aliases; *q; q++)
			{
				if (strcasecmp (*p, *q) == 0)
				{
					return each->mp;
				}
			}
		}
	}

	each = (named_list_t *) calloc (1, sizeof (named_list_t));
	each->mp = mp;
	each->next = cached_name;
	cached_name = each;
	return each->mp;
} /* end of cache_hostent_name () */

/*****************************************************************************
 *
 *	Function: copyright
 *
 *	Purpose: Display copyright info
 *
 ****************************************************************************/

static void
copyright ()

{
	static int      done = 0;

	/*
	 * Display copyright information only once.
	 */
	if ((!done) && (notice))
	{
		printf ("Merit AAA server, licensed software\n");
		printf ( "COPYRIGHT 1992, 1993, 1994, 1995, 1996, 1997, 1998\n");
		printf ( "THE REGENTS OF THE UNIVERSITY OF MICHIGAN\n");
		printf ( "ALL RIGHTS RESERVED\n");
		printf ("\n");

#ifdef BASIC_SERVER
		printf (
"PERMISSION IS GRANTED TO USE, COPY AND REDISTRIBUTE THIS VERSION OF THE MERIT\n");
		printf (
"BASIC AAA SERVER, SO LONG AS NO FEE IS CHARGED FOR THIS SOFTWARE, AND SO LONG\n");
		printf (
"AS THE COPYRIGHT NOTICE ABOVE, THIS GRANT OF PERMISSION, AND THE DISCLAIMER\n");
		printf (
"BELOW APPEAR IN ALL COPIES MADE; AND SO LONG AS THE NAME OF THE UNIVERSITY OF\n");
		printf (
"MICHIGAN OR MERIT NETWORK IS NOT USED IN ANY ADVERTISING OR PUBLICITY\n");
		printf (
"PERTAINING TO THE USE OR DISTRIBUTION OF THIS SOFTWARE WITHOUT SPECIFIC,\n");
		printf (
"WRITTEN PRIOR AUTHORIZATION.\n");
		printf ("\n");
		printf (
"NO RIGHTS ARE GRANTED HEREUNDER FOR ANY RECIPIENT TO MODIFY, DISASSEMBLE,\n");
		printf (
"DECOMPILE, REVERSE ENGINEER OR OTHERWISE CREATE DERIVATIVE WORKS OF THIS\n");
		printf (
"SOFTWARE.\n");
		printf ("\n");
		printf (
"THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION FROM THE UNIVERSITY\n");
		printf (
"OF MICHIGAN AS TO ITS FITNESS FOR ANY PURPOSE, AND WITHOUT WARRANTY BY THE\n");
		printf (
"UNIVERSITY OF MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING\n");
		printf (
"WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR\n");
		printf (
"A PARTICULAR PURPOSE.  THE REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE\n");
		printf (
"LIABLE FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR\n");
		printf (
"CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING OUT OF OR IN\n");
		printf (
"CONNECTION WITH THE USE OF THE SOFTWARE, EVEN IF IT HAS BEEN OR IS HEREAFTER\n");
		printf (
"ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.\n");
		printf ("\n");
		printf (
"FOR FURTHER INFORMATION ABOUT THE ENHANCED MERIT AAA SERVER, SEND EMAIL TO:\n");
		printf (
"aaa.license@merit OR, VISIT THE WWW SITE:  www.merit.edu/aaa/\n");
		printf ("\n");
#endif	/* BASIC_SERVER */

		done = 1;
	}

} /* end of copyright () */

/*****************************************************************************
 *
 *	Function: dns_usage
 *
 *	Purpose: Show how to use the program and what are its options.
 *
 ****************************************************************************/

static void
dns_usage ()

{
	fprintf (stderr, "%s [-ctvxh0] [<dns-name> ...]\n", progname);
	fprintf (stderr, "  where:\n");
	fprintf (stderr, "  -c toggles checking mode (defaults on)\n");
	fprintf (stderr, "  -t turns on TERSE mode (defaults off)\n");
	fprintf (stderr, "  -v shows the RCSID version information\n");
	fprintf (stderr, "  -x ups the debugging level (defaults 0)\n");
	fprintf (stderr, "  -h shows this message\n");
	fprintf (stderr, "  -0 suppresses the copyright notice\n");
	fprintf (stderr, "and\n");
	fprintf (stderr, "  <dns-name> is a name to check.\n");
	fprintf (stderr,
		"     (if no <dns-name> is given, then the system hostname\n");
	fprintf (stderr, "      is used.)\n");
} /* end of dns_usage () */

/*****************************************************************************
 *
 *	Function: free_my_hostent
 *
 *	Purpose: De-allocate the parallel structure (to netdb.h's
 *		 struct hostent).
 *
 ****************************************************************************/

static void
free_my_hostent (mp)

my_hostent_t   *mp;

{
	char          **p;

	if (mp)
	{
		if (mp->h_name)
		{
			free (mp->h_name);
		}

		for (p = mp->h_aliases; *p; p++)
		{
			free (*p);
		}
		free (mp->h_aliases);

		for (p = mp->h_addr_list; *p; p++)
		{
			free (*p);
		}
		free (mp->h_addr_list);

		/*
		 * Assume others are still NULL
		 */
		free (mp->m_addr_list);

		free (mp);
	}
	return;
} /* end of free_my_hostent () */

/*****************************************************************************
 *
 *	Function: make_my_hostent
 *
 *	Purpose: Copy a netdb.h struct hostent into a newly allocated
 *		 parallel data structure.
 *
 ****************************************************************************/

static my_hostent_t *
make_my_hostent (hp)

struct hostent *hp;

{
	my_hostent_t   *new;
	char          **tab;
	int             i;
	static char    *func = "make_my_hostent";

	if (!hp)
	{
		return NULL;
	}

	if (!(new = (my_hostent_t *) calloc (1, sizeof (my_hostent_t))))
	{
		fprintf (stderr,
			"%s: calloc (1, sizeof (my_hostent_t)) failed\n",
			progname);
		abort ();
	}

	new->h_name = my_strdup (hp->h_name);
	new->h_addrtype = hp->h_addrtype;
	new->h_length = hp->h_length;

	i = 1;
	for (tab = hp->h_aliases; *tab; tab++)
	{
		i++;
	}

	if (debug > 1)
	{
		fprintf (stderr, "%s-debug: %d aliases for \'%s\'\n",
			progname, i, new->h_name);
	}

	new->h_aliases = (char **) calloc (i, sizeof (char *));
	for (i = 0; hp->h_aliases[i]; i++)
	{
		new->h_aliases[i] = my_strdup (hp->h_aliases[i]);
		if (debug > 1)
		{
			fprintf (stderr, "%s-debug: alias[%d] is \'%s\'\n",
				progname, i, hp->h_aliases[i]);
		}
	}

	i = 1;
	for (tab = hp->h_addr_list; *tab; tab++)
	{
		i++;
	}

	new->h_addr_list = (char **) calloc (i, sizeof (char *));
	new->m_addr_list = (my_hostent_t **) calloc (i,
						     sizeof (my_hostent_t *));

	for (i = 0; hp->h_addr_list[i]; i++)
	{
		new->h_addr_list[i] = (char *) calloc (1, new->h_length);
		memcpy (new->h_addr_list[i], hp->h_addr_list[i], new->h_length);
	}

	return new;
} /* end of make_my_hostent () */

/*****************************************************************************
 *
 *	Function: my_strdup
 *
 *	Purpose: Because strdup (), despite being so useful, isn't
 *		 available everywhere.
 *
 ****************************************************************************/

static char *
my_strdup (src)

CONST char     *src;

{
	int             len;
	char           *new = NULL;

	if (src)
	{
		len = strlen (src) + 1;
		if ((new = (char *) malloc (len)))
		{
			strcpy (new, src);
		}
	}

	return new;
} /* end of my_strdup () */

/*****************************************************************************
 *
 *	Function: report
 *
 *	Purpose: Make a report
 *
 *	Returns: Number of problems found.
 *
 ****************************************************************************/

static int
report (prefix, hostname, root)

char           *prefix;
char           *hostname;
my_hostent_t   *root;

{
	int             problems = 0;
	int             c;
	my_hostent_t  **subs;
	char          **tabs;

	if (root == (my_hostent_t *) NULL)
	{
		return (1);	/* Nothing there, 1 problem. */
	}

	/* Analyze this. */
	if (strcmp (hostname, root->h_name) != 0)
	{
		printf ("%s '%s' isn't the CANONICAL DNS name '%s'\n",
			prefix, hostname, root->h_name);
		problems++;
	}

	for (subs = root->m_addr_list, c = 0, tabs = root->h_addr_list;
		tabs[c];
		c++)
	{
		if (!subs[c])
		{
			printf ("REVERSE MAP of [%s] back to '%s' not found\n",
				inet_ntoa (*(struct in_addr *) tabs[c]),
				root->h_name);
			problems++;
		}
		else if (subs[c]->m_name != root)
		{
			printf ("REVERSE MAP of [%s] back to '%s' incorrect\n",
				inet_ntoa (*(struct in_addr *) tabs[c]),
				root->h_name);

			if (subs[c]->m_name)
			{
				printf ("\treverse map points to \'%s\'\n",
					subs[c]->m_name->h_name);
			}
			else
			{
				printf ("\treverse map doesn't exist!\n");
			}
			problems++;
		}
	}

	if (terse == 0)
	{
		if (problems == 0)
		{
			printf ("No problems found with '%s'\n", hostname);
		}
		else
		{
			printf ("%d problems found with '%s'\n",
				problems, hostname);
		}
	}

	return problems;
} /* end of report () */

/*****************************************************************************
 *
 *	Function: report_addr
 *
 *	Purpose: Look up an entry by address and display it.
 *
 ****************************************************************************/

static my_hostent_t *
report_addr (lvl, sib, hostaddr)

int             lvl;
int             sib;
char           *hostaddr;

{
	my_hostent_t   *np;
	my_hostent_t   *mp = NULL;
	char          **p;
	int             i;

	if (terse == 0)
	{
		if (lvl > 0)
		{
			printf ("#%d.%d", lvl, sib);
		}

		for (i = 0; i < lvl; i++)
		{
			putchar ('\t');
		}
	}
	sib = 0;

	if ((mp = already_addrd (hostaddr)))
	{
		if (terse == 0)
		{
			printf ("([%s])\t\t# Shown earlier (by addr)\n",
				inet_ntoa (*(struct in_addr *) hostaddr));
		}

		return mp;
	}

	if ((mp = make_my_hostent (gethostbyaddr (hostaddr, 4, AF_INET))))
	{
		if ((np = cache_hostent_addr (mp)) != mp)
		{
			free_my_hostent (mp);
			mp = np;
		}

		if (terse == 0)
		{
			printf ("[%s] => %s",
				inet_ntoa (*(struct in_addr *) mp->h_addr),
				mp->h_name);

			printf (" [");
			for (p = mp->h_addr_list; *p; p++)
			{
				if (memcmp (*p, hostaddr, 4) != 0)
				{
					printf ("%s",
					  inet_ntoa (*(struct in_addr *) * p));
				}
				else
				{
					printf ("==");
				}

				if (p[1])
				{
					printf (", ");
				}
			}

			printf ("]\n");
		}

		np = report_name (lvl + 1, sib++, mp->h_name);
		if (np)
		{
			mp->m_name = np;
		}

		for (p = mp->h_aliases, i = 0; p[i]; i++)
		{
			np = report_name (lvl + 1, sib++, p[i]);
		}

		for (p = mp->h_addr_list, i = 0; p[i]; i++)
		{
			np = report_addr (lvl + 1, sib++, p[i]);
		}
	}
	else /* make_my_hostent() found errors */
	{
		printf ("%s: [%s]\t",
			progname, inet_ntoa (*(struct in_addr *) hostaddr));

		switch (h_errno)
		{
		    case HOST_NOT_FOUND:
			printf ("*** Not in DNS!\n");
			break;

		    case TRY_AGAIN:
			printf ("Try again later.\n");
			break;

		    case NO_DATA:
			printf ("Not a host addr.\n");
			break;

		    case NO_RECOVERY:
			printf ("*** DNS server error\n");
			break;

		    default:
			printf ("*** Unknown h_errno=%d\n", h_errno);
			break;
		}
	}
	return mp;
} /* end of report_addr () */

/*****************************************************************************
 *
 *	Function: report_name
 *
 *	Purpose: Wrapper for gethostbyname() and the internal cache of names.
 *
 ****************************************************************************/

static my_hostent_t *
report_name (lvl, sib, hostname)

int             lvl;
int             sib;
char           *hostname;

{
	int             i;
	struct hostent *hp;
	my_hostent_t   *np;
	my_hostent_t   *mp = NULL;
	char          **p;
	char            etc[200];

	if (debug > 0)
	{
		fprintf (stderr, "%s-debug: report_name(%d, %d, \'%s\')\n",
			progname, lvl, sib, hostname);
	}

	if (terse == 0)
	{
		if (lvl > 0)
		{
			printf ("#%d.%d", lvl, sib);
		}

		for (i = 0; i < lvl; i++)
		{
			putchar ('\t');
		}
	}

	sib = 1;

	if ((mp = already_named (hostname)))
	{
		if (terse == 0)
		{
			printf ("(%s)\t\t# Shown earlier (by name)\n",
				hostname);
		}

		return mp;
	}

	hp = gethostbyname (hostname);

	if ((mp = make_my_hostent (hp)))
	{
		if ((np = cache_hostent_name (mp)) != mp)
		{
			free_my_hostent (mp);
			mp = np;
		}

		if (terse == 0)
		{
			printf ("%s\t", mp->h_name);

			etc[0] = '\0';
			if (strcasecmp (hostname, mp->h_name) != 0)
			{
				strcat (etc, "\t <--- Warning, not canonical");
				printf ("( != %s)", hostname);
			}

			printf (" => [");
			for (p = mp->h_addr_list; *p; p++)
			{
				printf ("%s",
					inet_ntoa (*(struct in_addr *) * p));

				if (p[1])
				{
					printf (", ");
				}
			}

			printf ("]%s\n", etc);
		}

		for (p = mp->h_aliases, i = 0; p[i]; i++)
		{
			np = report_name (lvl + 1, sib++, p[i]);
		}

		for (p = mp->h_addr_list, i = 0; p[i]; i++)
		{
			np = report_addr (lvl + 1, sib++, p[i]);
			if (np)
			{
				mp->m_addr_list[i] = np;
			}
		}
	}
	else /* make_my_hostent() found errors */
	{
		printf ("%s: %s\t", progname, hostname);
		switch (h_errno)
		{
		    case HOST_NOT_FOUND:
			printf ("*** Not in DNS!\n");
			break;

		    case TRY_AGAIN:
			printf ("Try again later.\n");
			break;

		    case NO_DATA:
			printf ("Not a host name.\n");
			break;

		    case NO_RECOVERY:
			printf ("*** DNS server error\n");
			break;

		    default:
			printf ("*** Unknown h_errno=%d\n", h_errno);
			break;
		}
	} /* if ((mp == make_my_hostent (gethostbyname (hostname)))) */

	return mp;

} /* end of report_name () */

/*****************************************************************************
 *
 *	Function: main
 *
 *	Purpose: Start the game
 *
 ****************************************************************************/

int
main (argc, argv)

int             argc;
char          **argv;

{
	int             c;
	int             problems;
	char           *p;
	my_hostent_t   *root;
	char            our_hostname[MAXHOSTNAMELEN + 1];
	extern char    *optarg;
	extern int      optind;

	memset (our_hostname, 0, sizeof (our_hostname));
	gethostname (our_hostname, sizeof (our_hostname) - 1);
	our_hostname[sizeof (our_hostname) - 1] = '\0';	/* Safety. */

	progname = *argv;
	for (p = progname; *p; p++)
	{
		if (*p == '/')
		{
			progname = p + 1;
		}
	}

	while ((c = getopt (argc, argv, "ctvxh0")) != -1)
	{
		switch (c)
		{
		    case 'c':
			check = (check + 1) % 2;
			break;

		    case 'h':	/* HELP ME OUT */
			copyright ();
			dns_usage ();
			exit (1);

		    case 't':
			terse++;
			break;

		    case 'v':
			copyright ();
			fprintf (stderr, "version %s\n", rcsid);
			exit (0);

		    case 'x':
			debug++;
			break;

		    case '0':
			notice = (notice == 0) ? 1 : 0;
			if (notice)
			{
				copyright ();
			}
			break;

		    default:
			copyright ();
			fprintf (stderr, "unsupported option\n");
			dns_usage ();
			exit (20);
		}
	}

	copyright ();

	if (debug > 0)
	{
		fprintf (stderr, "%s-debug : ourhostname = \'%s\'\n",
			progname, our_hostname);
	}

	if ((terse > 0) && (check == 0))
	{
		fprintf (stderr,
	  "%s: use of -c without -t (or vice-versa) may result in no output\n",
			progname);
	}

	if (optind < argc)
	{
		for (; optind < argc; optind++)
		{
			problems = 0;
			root = report_name (0, 0, argv[optind]);
			if (check > 0)
			{
				problems = report ("The name ", argv[optind],
									root);
			}

			if ((optind < argc - 1) && (terse == 0))
			{
				printf ("\n---------------------------\n");
			}
		}
	}
	else
	{
		problems = 0;
		if (!(root = report_name (0, 0, our_hostname)))
		{
			printf ("SYSTEM'S HOSTNAME ('%s') IS NOT IN THE DNS!\n",
				our_hostname);
			exit (1);
		}
		else if (check > 0)
		{
			problems = report ("WARNING, your system's hostname",
					   our_hostname, root);
		}
	}

	exit (0);

} /* end of main () */
