#ifdef BINARY_FILTERS

/*
 *      Copyright (c) 1994 Ascend Communications, Inc.
 *      All rights reserved.
 *
 *	Permission to copy all or part of this material for any purpose is
 *	granted provided that the above copyright notice and this paragraph
 *	are duplicated in all copies.  THIS SOFTWARE IS PROVIDED ``AS IS''
 *	AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
 *	LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE.
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

static char     rcsid[] = "$Id: filters.c,v 1.2 1998/06/11 18:40:32 web Exp $";

#include	<sys/types.h>
#include	<stdio.h>
#include	<string.h>
#include	<ctype.h>
#include	<errno.h>
#include	<syslog.h>
#include	<netinet/in.h>
#include	<stdlib.h>
#include	<sys/time.h>		/* gettimeofday() */
#include	<memory.h>

#include	"radius.h"

extern int       debug_flag;

/*
 * Filter Port Type
 * 
 *	ASCII names of some well known TCP/UDP services.
 *	Used for filtering on a port type.
 * 
 */

static keywordstruct filterporttype[] =
{
	{ "ftp-data",	20 },
	{ "ftp",	21 },
	{ "telnet",	23 },
	{ "smtp",	25 },
	{ "nameserver",	42 },
	{ "domain",	53 },
	{ "tftp",	69 },
	{ "gopher",	70 },
	{ "finger",	79 },
	{ "www",	80 },
	{ "kerberos",	88 },
	{ "hostname",	101 },
	{ "nntp",	119 },
	{ "ntp",	123 },
	{ "exec",	512 },
	{ "login",	513 },
	{ "cmd",	514 },
	{ "talk",	517 },
	{ NULL,		NO_TOKEN },
};

static keywordstruct filterkeywords[] =
{
	{ "ip",		FILTER_IP_TYPE },
	{ "generic",	FILTER_GENERIC_TYPE },
	{ "in",		FILTER_IN },
	{ "out",	FILTER_OUT },
	{ "forward",	FILTER_FORWARD },
	{ "drop",	FILTER_DROP },
	{ "dstip",	FILTER_IP_DST },
	{ "srcip",	FILTER_IP_SRC },
	{ "dstport",	FILTER_IP_DST_PORT },
	{ "srcport",	FILTER_IP_SRC_PORT },
	{ "est",	FILTER_EST },
	{ "more",	FILTER_MORE },
	{ "!=",		FILTER_GENERIC_COMPNEQ },
	{ "==",		FILTER_GENERIC_COMPEQ },
	{ NULL,		NO_TOKEN },
};

#define FILTER_DIRECTION 	0
#define FILTER_DISPOSITION	1
#define IP_FILTER_COMPLETE  	0x3	/* bits shifted by FILTER_DIRECTION, */
					/* and by FILTER_DISPOSITION */
#define GENERIC_FILTER_COMPLETE 0x1c3	/* bits shifted by FILTER_DIRECTION, */
					/* and by FILTER_DISPOSITION, and by */
					/* FILTER_GENERIC_OFFSET, and by */
					/* FILTER_GENERIC_MASK, and by */
					/* FILTER_GENERIC_VALUE */

/*
 * Filter Protocol Name - ASCII names of protocols used for filtering.
 */
static keywordstruct filterprotoname[] =
{
	{ "tcp",	6 },
	{ "udp",	17 },
	{ "ospf",	89 },
	{ "icmp",	1 },
	{ NULL,		NO_TOKEN },
};

keywordstruct filtercompare[] =
{
	{ ">",	RAD_COMPARE_GREATER },
	{ "=",	RAD_COMPARE_EQUAL },
	{ "<",	RAD_COMPARE_LESS },
	{ "!=",	RAD_COMPARE_NOT_EQUAL },
	{ NULL,	NO_TOKEN },
};

static char     curstring[AUTH_STRING2_LEN];
static char     ip_address_digits[] = "1234567890./";

static int      find_key PROTO((char *, keywordstruct *));
static int      is_all_digit PROTO((char *));
static short    a2octet PROTO((char *, char *));
static char     default_netmask PROTO((UINT4));
static int      ipaddr_str2val PROTO((char *, UINT4 *, char *));
static int      parse_ip_filter PROTO((radfilter *));
static int      parse_gen_filter PROTO((radfilter *));

/*************************************************************************
 * 
 *	Function: find_key
 * 
 *	Purpose: Given a table of keywords, try to match the given string
 *		 to an entry.  If it matches, return that keyword value,
 *		 else NO_TOKEN is returned.  A sanity check is made for
 *		 upper-case characters.
 * 
 *	Returns: Keyword value on a match, or NO_TOKEN.
 * 
 *************************************************************************/

static int
find_key (string, list)

char           *string;		/* Pointer to the token to match. */
keywordstruct  *list;		/* Point to the list of keywords. */

{
	short           len;
	keywordstruct  *entry;
	char           *ptr;
	char            buf[80];
	static char    *func = "find_key";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	len = strlen ((char *) string);
	for (ptr = buf; len; len--, string++)
	{
		if (isupper(*string))
		{
			*ptr++ = tolower(*string);
		}
		else
		{
			*ptr++ = *string;
		}
	}
	*ptr = 0;

	entry = list;
	while (entry->name)
	{
		if (strcmp (entry->name, buf) == 0)
		{
			break;
		}
		entry++;
	}

	return (entry->value);
} /* end of find_key () */

/*************************************************************************
 * 
 *	Function: is_all_digit
 * 
 *	Purpose: Checks a string to make sure all values are digits.
 * 
 *	Returns: TRUE if all digits, or FALSE.
 * 
 *************************************************************************/

static int
is_all_digit (token)

char           *token;		/* Pointer to sting to check. */

{
	int             i;
	static char    *func = "is_all_digit";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	i = strlen ((char *) token);

	while (i--)
	{
		if (isdigit(*token))
		{
			token++;
		}
		else
		{
			break;
		}
	}

	if (i > 0)
	{
		return (FALSE);
	}

	return (TRUE);
} /* end of is_all_digit () */

/*************************************************************************
 *
 *	Function: a2octet
 * 
 *	Purpose: Converts the ASCII mask and value for generic filters
 *		 into octets.  Also does a sanity check to see if the
 *		 string is greater than MAX_FILTER_LEN.
 *		 Assumes the sting is hexadecimal with NO leading "0x".
 * 
 *	Returns: Number of octets, or -1 for error.
 * 
 *************************************************************************/

static short
a2octet (tok, retBuf)

char           *tok;		/* Pointer to the string. */
char           *retBuf;		/* Pointer to place the octets. */

{
	short           i;
	short           len;
	short           rc;
	short           retlen;
	short           val;
	char           *octet;
	char            buf[RAD_MAX_FILTER_LEN * 2];
	static char    *func = "a2octet";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	octet = buf;
	rc = -1;
	retlen = 0;

	if ((len = strlen ((char *) tok)) <= (RAD_MAX_FILTER_LEN * 2))
	{
		retlen = len / 2;
		if (len % 2)
		{
			retlen++;
		}

		memset (buf, '\0', RAD_MAX_FILTER_LEN * 2);

		for ( ; len ; len--)
		{
			if (*tok <= '9' && *tok >= '0')
			{
				val = '0';
				*octet++ = *tok++ - val;
			}
			else if (isxdigit(*tok))
			{
				if (*tok > 'Z')
				{
					val = 'a';
				}
				else
				{
					val = 'A';
				}
				*octet++ = (*tok++ - val) + 10;
			}
			else
			{
				break;
			}
		}

		if (!len)
		{
			/* merge the values */
			for (i = 0; i < RAD_MAX_FILTER_LEN * 2; i += 2)
			{
				*retBuf++ = (buf[i] << 4) | buf[i + 1];
			}
		}
	}

	if (len)
	{
		rc = -1;
	}
	else
	{
		rc = retlen;
	}
	return (rc);
} /* end of a2octet () */

/*************************************************************************
 *
 *	Function: default_netmask
 * 
 *	Purpose: Given an IP address, calculate a default netmask.
 * 
 *	Returns: Number of bits for the netmask.
 * 
 *************************************************************************/

static char
default_netmask (address)

UINT4           address;	/* The given IP address. */

{
	char            netmask;
	static char    *func = "default_netmask";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (!address)
	{
		netmask = 0;
	}
	else if ((address & htonl(0x80000000)) == 0)
	{
		netmask = 8;
	}
	else if ((address & htonl(0xc0000000)) == htonl(0x80000000))
	{
		netmask = 16;
	}
	else if ((address & htonl(0xe0000000)) == htonl(0xc0000000))
	{
		netmask = 24;
	}
	else
	{
		netmask = 32;
	}
	return netmask;
} /* end of default_netmask () */

/*************************************************************************
 *
 *	Function: ipaddr_str2val
 *
 *	Purpose: Attempts to convert an IP address (in ASCII dotted quad
 *		 notation with an optional netmask part) to a pair of
 *		 IP addresses.  Note: An IP address is always stored
 *		 in network byte order.
 * 
 * 
 *	Returns: TRUE if valid conversion, FALSE otherwise.
 * 
 *************************************************************************/

static int
ipaddr_str2val (string, ipaddress, netmask)

char           *string;		/* IP address string: xxx.xxx.xxx.xxx[/nn] */
UINT4          *ipaddress;	/* Resultant IP address to return to caller. */
char           *netmask;	/* Resultant netmask to return to caller. */

{
	int             i;
	int             numdots;
	long            value;
	u_char         *dst;
	char           *cp;
	static char    *func = "ipaddr_str2val";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (!string)
	{
		return (FALSE);
	}

	/*
	 *	Allow an IP address to be blanked instead of forcing
	 *	entry of 0.0.0.0 -- the user will like it.
	 */

	if (*string == 0)
	{
		*ipaddress = 0;
		*netmask = 0;
		return TRUE;
	}

	/*
	 *	First just count the number of dots in the address.
	 *	If there are more or less than three the address is invalid.
	 */

	cp = string;
	numdots = 0;

	while (*cp)
	{
		if (!strchr (ip_address_digits, *cp))
		{
			return (FALSE);
		}

		if (*cp == '.')
		{
			numdots++;
		}
		cp++;
	}

	if (numdots != 3)
	{
		return (FALSE);
	}

	dst = (u_char *) ipaddress;
	cp = string;

	for (i = 0; i < sizeof (*ipaddress); i++)
	{
		value = strtol (cp, (char **) &cp, 10);

		if ((value < 0) || (value > 255))
		{
			return (FALSE);
		}
		*dst++ = (u_char) value;

		if (*cp == '.')
		{
			cp += 1;
		}
	}

	/*
	 * If there is a netmask part, parse it, otherwise figure out the
	 * default netmask for this class of address.
	 */

	if (*cp == '/')
	{
		value = strtol (cp + 1, (char **) &cp, 10);
		if ((*cp != 0) || (value < 0) || (value > 32))
		{
			return (FALSE);
		}
		*netmask = (char) value;
	}
	else
	{
		*netmask = default_netmask (*ipaddress);
	}
	return (TRUE);
} /* end of ipaddr_str2val () */

/*************************************************************************
 *
 *	Function: parse_ip_filter
 * 
 *	Purpose: Parses an IP filter string from a RADIUS reply.
 *		 The format of the string is:
 * 
 *	    ip dir action [ dstip n.n.n.n/nn ] [ srcip n.n.n.n/nn ]
 *		[ proto [ dstport cmp value ] [ srcport cmd value ] [ est ] ]
 * 
 *	    Fields in [...] are optional.  Where:
 * 
 *		ip:	Keyword to designate an IP filter.
 * 
 *		dir:	Filter direction.  "IN" or "OUT"
 * 
 *		action:	Filter action.  "FORWARD" or "DROP"
 * 
 *		dstip:	Keyword for destination IP address.
 *			n.n.n.n = IP address.
 *			/nn = netmask.
 * 
 *		srcip:	Keyword for source IP address.
 *			n.n.n.n = IP address.
 *			/nn = netmask.
 * 
 *		proto:	Optional protocol field.  Either a name or number.
 *			Known names are in filterprotoname[].
 * 
 *		dstport: Keyword for destination port.
 *			Only valid for TCP or UDP.
 *			'cmp' are in radfiltercomp enumeration type.
 *			'value' entries are in filterporttype[].
 *			'value' may be a name or number.
 * 
 *		srcport: Keyword for source port.  Only valid for TCP or UDP.
 *			'cmp' are in radfiltercomp enumeration type.
 *			'value' entries are in filterporttype[].
 *			'value' may be a name or number.
 * 
 *		est:	Keyword for TCP established.  Valid only for TCP.
 * 
 *	Returns: -1 for error, or 0 otherwise.
 * 
 *************************************************************************/

static int
parse_ip_filter (curentry)

radfilter      *curentry;	/* Pointer to return the filter structure. */

{

	UINT4           elements = (UINT4) 0;
	int             tok;
	char           *token;
	radipfilter    *ip;
	static char    *func = "parse_ip_filter";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	token = (char *) strtok (NULL, " ");

	memset (curentry, '\0', sizeof (radfilter));
	curentry->type = RAD_FILTER_IP;
	ip = &curentry->u.ip;
	ip->established = FALSE;

	while (token)
	{
		dprintf(2, (LOG_DAEMON, LOG_DEBUG,
			"%s: token %s ", func, token));

		tok = find_key (token, filterkeywords);
		dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: tok %d ", func, tok));

		switch (tok)
		{
		    case FILTER_IN:
		    case FILTER_OUT:
			curentry->input = (tok == FILTER_IN) ? TRUE : FALSE;
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s ", func,
				tok == FILTER_IN ? "FILTER_IN" : "FILTER_OUT"));
			elements |= (1 << FILTER_DIRECTION);
			break;

		    case FILTER_FORWARD:
		    case FILTER_DROP:
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s ", func,
				tok == FILTER_DROP ?
					"FILTER_DROP" : "FILTER_FORWARD"));
			elements |= (1 << FILTER_DISPOSITION);

			if (tok == FILTER_FORWARD)
			{
				curentry->forward = TRUE;
			}
			else
			{
				curentry->forward = FALSE;
			}
			break;

		    case FILTER_IP_DST:
		    case FILTER_IP_SRC:
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s ", func,
				tok == FILTER_IP_DST ?
					"FILTER_IP_DST" : "FILTER_IP_SRC"));
			token = (char *) strtok (NULL, " ");

			if (token)
			{
				if (tok == FILTER_IP_DST)
				{

					if (ipaddr_str2val ((char *) token,
							&ip->dstip,
							(char *) &ip->dstmask))
					{
						dprintf(2, (LOG_DAEMON,
							LOG_DEBUG,
						       "%s: ip %lx netmask %lx",
							func, ip->dstip,
							ip->dstmask));
						break;
					}
				}
				else
				{
					if (ipaddr_str2val ((char *) token,
							&ip->srcip,
							(char *) &ip->srcmask))
					{
						dprintf(2, (LOG_DAEMON,
							LOG_DEBUG,
						       "%s: ip %lx netmask %lx",
							ip->srcip,
							ip->srcmask));
						break;
					}
				}
			}

			dprintf(2, (LOG_DAEMON, LOG_DEBUG,
				"%s: IP filter syntax error %s", func,
				(token == (char *) NULL) ? "?" : token ));
			logit (LOG_DAEMON, LOG_ERR,
			"%s: IP filter syntax error: do not recognize %s in %s",
				 func, token, curstring);
			return (-1);

		    case FILTER_IP_DST_PORT:
		    case FILTER_IP_SRC_PORT:
			{
				radfiltercomp   cmp;
				short           port;

				dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s",
					func, tok == FILTER_IP_DST_PORT
						? "FILTER_IP_DST_PORT"
						: "FILTER_IP_SRC_PORT"));
				token = (char *) strtok (NULL, " ");

				if (token)
				{
				    cmp = find_key (token, filtercompare);
				    dprintf(2, (LOG_DAEMON, LOG_DEBUG,
						"%s: cmp value = %d",
						func, cmp));

				    if (cmp != NO_TOKEN)
				    {
					token = (char *) strtok (NULL, " ");
					if (token)
					{
					    if (is_all_digit (token))
					    {
						port = atoi ((char *) token);
					    }
					    else
					    {
						port = find_key (token,
								filterporttype);
					    }

					    if (port != (short) NO_TOKEN)
					    {
						dprintf(2, (LOG_DAEMON,
							LOG_DEBUG,
							"%s: port = %d",
							func, port));

						if (tok == FILTER_IP_DST_PORT)
						{
						    ip->dstportcomp = cmp;
						    ip->dstport = htons(port);
						}
						else
						{
						    ip->srcportcomp = cmp;
						    ip->srcport = htons(port);
						}
						break;
					    }
					}
				    }
				}
				logit (LOG_DAEMON, LOG_ERR,
			"%s: IP filter syntax error: do not recognize %s in %s",
					func, token, curstring);
				dprintf(2, (LOG_DAEMON, LOG_DEBUG,
				      "%s: IP Filter syntax error %s", func,
				      (token == (char *) NULL) ? "?" : token ));
				return (-1);
				break;
			}

		    case FILTER_EST:
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got est %s",
				func,
				(token == (char *) NULL) ? "?" : token ));
			ip->established = TRUE;
			break;

		    default:
			/* no keyword match but may match a protocol list */
			if (is_all_digit (token))
			{
				tok = atoi ((char *) token);
			}
			else
			{
				tok = find_key (token, filterprotoname);

				if (tok == NO_TOKEN)
				{
					dprintf(2, (LOG_DAEMON, LOG_DEBUG,
						"%s: IP protocol error %s",
						func,
						(token == (char *) NULL)
							? "?" : token ));
					logit (LOG_DAEMON, LOG_ERR,
			"%s: IP filter syntax error: do not recognize %s in %s",
						func, token, curstring);
					return (-1);
				}
			}
			ip->proto = tok;
			dprintf(2, (LOG_DAEMON, LOG_DEBUG,
				"%s: IP protocol cmd = %d", func, tok));
		} /* end of switch */
		token = (char *) strtok (NULL, " ");
	}

	if (elements == IP_FILTER_COMPLETE)
	{
		return (0);
	}

	logit (LOG_DAEMON, LOG_ERR, "%s: Non-Fatal internal error!", func);

	return (-1);

} /* end of parse_ip_filter () */

/*************************************************************************
 *
 *	Function: parse_gen_filter:
 * 
 *	Purpose: Parses a Generic filter string from a RADIUS reply.
 *		 The format of the string is:
 * 
 *	    GENERIC dir action offset mask value [== or != ] [more]
 * 
 *	    Fields in [...] are optional.  Where:
 * 
 *		generic:	Keyword to indicate a generic filter.
 * 
 *		dir:		Filter direction.  "IN" or "OUT"
 * 
 *		action:		Filter action. "FORWARD" or "DROP"
 * 
 *		offset:		A Number.  Specifies an offset into
 *				a frame to start comparing.
 * 
 *		mask:		A hexadecimal mask of bits to compare.
 * 
 *		value:		A value to compare with the masked data.
 * 
 *		compneq:	Defines type of comparison.  ( "==" or "!=")
 *				Default is "==".
 * 
 *		more:		Optional keyword MORE, to represent the
 *				attachment to the next entry.
 * 
 *	Returns: -1 for error, or 0 otherwise.
 * 
 *************************************************************************/

static int
parse_gen_filter (curentry)

radfilter      *curentry;	/* Pointer to return the filter structure. */

{
	short           masklen;
	short           vallen;
	int             gstate = FILTER_GENERIC_OFFSET;
	int             tok;
	UINT4           elements = (UINT4) 0;
	char           *token;
	radgenfilter   *gen;
	static char    *func = "parse_gen_filter";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	token = (char *) strtok (NULL, " ");
	masklen = 0;
	memset ((char *) curentry, '\0', sizeof (radfilter));
	curentry->type = RAD_FILTER_GENERIC;
	gen = &curentry->u.generic;
	gen->more = FALSE;
	gen->compneq = FALSE;

	while (token)
	{
		dprintf(2, (LOG_DAEMON, LOG_DEBUG,
			"%s: token %s", func, token));

		tok = find_key (token, filterkeywords);
		dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: tok %d ", func, tok));

		switch (tok)
		{
		    case FILTER_IN:
		    case FILTER_OUT:
			curentry->input = (tok == FILTER_IN) ? TRUE : FALSE;
			elements |= (1 << FILTER_DIRECTION);
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s", func,
				tok == FILTER_IN ? "FILTER_IN" : "FILTER_OUT"));
			break;

		    case FILTER_FORWARD:
		    case FILTER_DROP:
			elements |= (1 << FILTER_DISPOSITION);
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got %s", func,
				tok == FILTER_DROP ? "FILTER_DROP"
							: "FILTER_FORWARD"));
			if (tok == FILTER_FORWARD)
			{
				curentry->forward = TRUE;
			}
			else
			{
				curentry->forward = FALSE;
			}
			break;

		    case FILTER_GENERIC_COMPNEQ:
			gen->compneq = TRUE;
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got comp != %s",
				func, token));
			break;

		    case FILTER_GENERIC_COMPEQ:
			gen->compneq = FALSE;
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got comp = %s",
				func, token));
			break;

		    case FILTER_MORE:
			gen->more = htons(TRUE);
			dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: got more %s",
				func, token));
			break;

		    default:
			elements |= (1 << gstate);
			switch (gstate)
			{
			    case FILTER_GENERIC_OFFSET:
				gstate = FILTER_GENERIC_MASK;
				gen->offset = htons(atoi ((char *) token));
				break;

			    case FILTER_GENERIC_MASK:
				gstate = FILTER_GENERIC_VALUE;
				masklen = a2octet (token, (char *) gen->mask);
				if (masklen == (short) -1)
				{
					logit (LOG_DAEMON, LOG_ERR,
						"%s: filter mask error: %s",
						func, curstring);
					return (-1);
				}

				dprintf(2, (LOG_DAEMON, LOG_DEBUG,
					"%s: octet retlen = %d ",
					func, masklen));
				for (tok = 0; tok < masklen; tok++)
				{
					dprintf(2, (LOG_DAEMON, LOG_DEBUG,
						"%2x", gen->mask[tok]));
				}
				break;

			    case FILTER_GENERIC_VALUE:
				gstate++;
				vallen = a2octet (token, (char *) gen->value);
				if (vallen != masklen)
				{
					logit (LOG_DAEMON, LOG_ERR,
			    "%s: filter value size not same as filter mask: %s",
						func, curstring);
					return (-1);
				}
				gen->len = htons(vallen);

				dprintf(2, (LOG_DAEMON, LOG_DEBUG,
					"%s: octet retlen = %d ",
					func, masklen));
				for (tok = 0; tok < masklen; tok++)
				{
					dprintf(2, (LOG_DAEMON, LOG_DEBUG,
						"%2x", gen->value[tok]));
				}
				break;

			    default:
				logit (LOG_DAEMON, LOG_ERR,
					"%s: do not know %s in %s",
					func, token, curstring);
				dprintf(2, (LOG_DAEMON, LOG_DEBUG,
					"%s: Filter syntax error %s",
					func, token));
				return (-1);
			} /* end of switch */
		} /* end of switch */
		token = (char *) strtok (NULL, " ");
	} /* end of while */

	if (elements == GENERIC_FILTER_COMPLETE)
	{
		return (0);
	}

	logit (LOG_DAEMON, LOG_ERR, "%s: Non-Fatal internal error!", func);

	return (-1);
} /* end of parse_gen_filter () */

/*************************************************************************
 *
 *	Function: filter_binary
 * 
 *	Purpose: Parse entries from an ASCII format to a
 *		 binary format recognized by Ascend boxes.
 * 
 *	Returns: the binary filter length,
 *		 or -1 otherwise.
 *
 *************************************************************************/

int
filter_binary (filter, valstr)

char           *filter;		/* Pointer to place to return parsed filter. */
char           *valstr;		/* The string to parse. */

{

	int             rc = -1;
	int             tok;
	char           *token;
/*	radfilter      *filt; See CODE REMOVED below. */
/*	radgenfilter   *gen; See CODE REMOVED below. */
	radfilter       radfil;
	static char    *func = "filter_binary";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	filter[0] = '\0';
	if (valstr != (char *) NULL)
	{
		strcpy (curstring, valstr);
	}
	else
	{
		return (rc);
	}

	token = (char *) strtok ((char *) valstr, " ");
	tok = find_key (token, filterkeywords);

	switch (tok)
	{
	    case FILTER_IP_TYPE:
		rc = parse_ip_filter (&radfil);
		break;

	    case FILTER_GENERIC_TYPE:
		rc = parse_gen_filter (&radfil);
		break;
	}

	/*************code below moved to avpair_add_vend()*******************
	 *
	 *	If "more" is set, then this new entry must exist, be of
	 *	FILTER_GENERIC_TYPE, the direction and disposition must
	 *	match for the previous "more" to be valid.  If any should
	 *	fail, then TURN OFF previous more.

VALUE_PAIR     *prevradpair = 0;
	if (prevradpair)
	{
		filt = (radfilter *) prevradpair->strvalue;
		if ((tok != FILTER_GENERIC_TYPE) ||
			(rc == -1) ||
			(prevradpair->attribute != pair->attribute) ||
			(filt->input != radfil.input) ||
			(filt->forward != radfil.forward))
		{
			gen = &filt->u.generic;
			gen->more = FALSE;
			logit (LOG_DAEMON, LOG_ERR,
			      "%s: 'more' for previous entry doesn't match: %s",
				func, curstring);
		}
	}

	prevradpair = (VALUE_PAIR *) NULL;
	if (rc != -1 && tok == FILTER_GENERIC_TYPE)
	{
		if (radfil.u.generic.more)
		{
			prevradpair = pair;
		}
	}
	**************above code moved to avpair_add_vend()*************/

	if (rc != -1)
	{
		memcpy (filter, (char *) &radfil, sizeof (radfilter));
		rc = sizeof (radfilter);
	}

	return (rc);
} /* end of filter_binary () */

#endif	/* BINARY_FILTERS */
