/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/* 
 *
 * Public entry points in this file:
 *
 * init_fsm
 *
 */

static char     rcsid[] = "$Id: fsm.c,v 1.3 1998/06/26 19:42:16 weiwang Exp $";

#include	<sys/types.h>
#include	<sys/param.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<syslog.h>
#include	<ctype.h>

#include	"radius.h"

extern int      debug_flag;    /* values > 0 allow various debug output */
extern FILE    *ddt;
extern char    *radius_dir;
extern char    *fsm_id;

static FSM_ENT **fsm;          /* pointer to FSM table */
static FSM_ENT **default_fsm;  /* pointer to default action table */

static int      add_state PROTO((STATELIST *, char *, u_int, int));
static int      enum_event PROTO((char *));
static int      extra_stuff PROTO((char *, int *, char **));
static NAME_LIST *find_state PROTO((STATELIST *, char *));
static void     rad_fsmdelete PROTO((int));
static int      rad_fsminit PROTO((char*, FSM_ENT ***, char *));

/*
 *	Default Action Table
 */

static char    *default_actions =		/* MUST NOT HAVE FSMID */
{
"Start:\n\
	*.*.DUP			REDO		Same\n\
	*.*.FATAL		FATAL_LOG	End\n\
	*.*.TIMEOUT		TIMEOUT		End\n\
	*.*.ABORT		NULL		End\n\
	*.*.RETRY_LIMIT		FAIL		ReplyHold\n\
\n\
ReplyHold:\n\
	*.*.ACK			REPLY		Unlimit\n\
	*.*.NAK			REPLY		Unlimit\n\
\n\
Unlimit:\n\
	*.*.ACK			RETRY_LIMIT	Hold		XVALUE=0\n\
	*.*.NAK			RETRY_LIMIT	Hold		XVALUE=0\n\
\n\
Hold:\n\
	*.*.TIMEOUT		NULL		End\n\
\n\
End:"
};

#ifdef	MERIT_LAS
/*
 *	Standard LAS State Machine Table when there is no FSM configuration file
 */

#ifdef USR_CCA
static char    *standard_fsm =
{
"%FSMID USR_STD_LAS\n\
Start:\n\
        *.+AUTHENTICATE.ACK     AUTHENTICATE    REPLYhold\n\
        *.+AUTHEN.ACK           AUTHENTICATE    REPLYhold\n\
        *.+AUTH_ONLY.ACK        AUTHENTICATE    REPLYhold\n\
        *.+ACCT.ACK             ACCT            REPLYhold\n\
        *.+MGT_POLL.ACK         SRV_STATUS      REPLYhold\n\
        *.+IPALLOC.ACK          IPPOOL          REPLYhold\n\
        *.+IPRELEASE.ACK        IPPOOL          REPLYhold\n\
        *.+RF_REQ.ACK           RES_FREE_REQ    End\n\
        *.+RF_RESP.ACK          RES_FREE_RESP   End\n\
        *.+RQ_REQ.ACK           FWD_TO_NAS      End\n\
        *.+RQ_RESP.ACK          RES_QRY_RESP    RQwait\n\
        *.+NAS_REB_REQ.ACK      NAS_REB         End\n\
        *.+NAS_REB_RESP.ACK     NULL            End\n\
        *.+EVENT_REQUEST.ACK    NULL            End\n\
        *.+TERM_SESSION.ACK     NULL            End\n\
        *.*.NAK                 REPLY           End\n\
\n\
REPLYhold:\n\
        *.*.ACK                 REPLY           Hold\n\
        *.*.NAK                 REPLY           Hold\n\
        *.*.ACC_CHAL            REPLY           Hold\n\
\n\
RQwait:\n\
        Start.RES_QRY_RESP.ACK  NULL            Start\n\
        Start.RES_QRY_RESP.DONE NULL            End\n\
        Start.RES_QRY_RESP.NAK  NULL            End\n\
\n\
Hold:\n\
        *.*.TIMEOUT             NULL            End\n\
        *.*.TIMER               NULL            End\n\
\n\
End:"
};

#else  /* USR_CCA */

/*
 *	Standard LAS State Machine Table when there is no FSM configuration file
 */

static char    *standard_fsm =
{
"%FSMID STD_LAS\n\
Start:\n\
        *.+AUTHEN.ACK           AUTHENTICATE    REPLYhold\n\
        *.+AUTH_ONLY.ACK        AUTHENTICATE    REPLYhold\n\
        *.+AUTHENTICATE.ACK     AUTHENTICATE    REPLYhold\n\
        *.+ACCT.ACK             ACCT            REPLYhold\n\
        *.+IPALLOC.ACK          IPPOOL          REPLYhold\n\
        *.+IPRELEASE.ACK        IPPOOL          REPLYhold\n\
        *.+MGT_POLL.ACK         SRV_STATUS      REPLYhold\n\
        *.+EVENT_REQUEST.ACK    NULL            End\n\
        *.+TERM_SESSION.ACK     NULL            End\n\
        *.*.NAK                 REPLY           End\n\
\n\
REPLYhold:\n\
        *.*.ACK                 REPLY           Hold\n\
        *.*.NAK                 REPLY           Hold\n\
        *.*.ACC_CHAL            REPLY           Hold\n\
\n\
Hold:\n\
        *.*.TIMEOUT             NULL            End\n\
\n\
End:"
};

#endif  /* USR_CCA */

#else	/* MERIT_LAS */

/*
 *	Standard State Machine Table when there is no FSM configuration file
 */

#ifdef USR_CCA
static char    *standard_fsm =
{
"%FSMID USR_STD\n\
Start:\n\
        *.+AUTHEN.ACK           AUTHENTICATE    REPLYhold\n\
        *.+AUTH_ONLY.ACK        AUTHENTICATE    REPLYhold\n\
        *.+ACCT.ACK             ACCT            REPLYhold\n\
        *.+MGT_POLL.ACK         SRV_STATUS      REPLYhold\n\
        *.+IPALLOC.ACK          IPPOOL          REPLYhold\n\
        *.+IPRELEASE.ACK        IPPOOL          REPLYhold\n\
        *.+RF_REQ.ACK           RES_FREE_REQ    End\n\
        *.+RF_RESP.ACK          RES_FREE_RESP   End\n\
        *.+RQ_REQ.ACK           FWD_TO_NAS      End\n\
        *.+RQ_RESP.ACK          RES_QRY_RESP    RQwait\n\
        *.+NAS_REB_REQ.ACK      NAS_REB         End\n\
        *.+NAS_REB_RESP.ACK     NULL            End\n\
        *.+EVENT_REQUEST.ACK    NULL            End\n\
        *.+TERM_SESSION.ACK     NULL            End\n\
        *.*.NAK                 REPLY           End\n\
\n\
REPLYhold:\n\
        *.*.ACK                 REPLY           Hold\n\
        *.*.NAK                 REPLY           Hold\n\
        *.*.ACC_CHAL            REPLY           Hold\n\
\n\
RQwait:\n\
        Start.RES_QRY_RESP.ACK  NULL            Start\n\
        Start.RES_QRY_RESP.DONE NULL            End\n\
        Start.RES_QRY_RESP.NAK  NULL            End\n\
\n\
Hold:\n\
        *.*.TIMEOUT             NULL            End\n\
        *.*.TIMER               NULL            End\n\
\n\
End:"
};

#else  /* USR_CCA */

/*
 *	Standard State Machine Table when there is no FSM configuration file
 */

static char    *standard_fsm =
{
"%FSMID STD\n\
Start:\n\
        *.+AUTHEN.ACK           AUTHENTICATE    REPLYhold\n\
        *.+AUTH_ONLY.ACK        AUTHENTICATE    REPLYhold\n\
        *.+ACCT.ACK             ACCT            REPLYhold\n\
        *.+MGT_POLL.ACK         SRV_STATUS      REPLYhold\n\
        *.+IPALLOC.ACK          IPPOOL          REPLYhold\n\
        *.+IPRELEASE.ACK        IPPOOL          REPLYhold\n\
        *.+EVENT_REQUEST.ACK    NULL            End\n\
        *.+TERM_SESSION.ACK     NULL            End\n\
        *.*.NAK                 REPLY           End\n\
\n\
REPLYhold:\n\
        *.*.ACK                 REPLY           Hold\n\
        *.*.NAK                 REPLY           Hold\n\
        *.*.ACC_CHAL            REPLY           Hold\n\
\n\
Hold:\n\
        *.*.TIMEOUT             NULL            End\n\
\n\
End:"
};

#endif  /* USR_CCA */

#endif	/* MERIT_LAS */

/*************************************************************************
 *
 *	Function: add_state
 *
 *	Purpose: Add a state name and flag to the list.
 *		 Assign the next available state number to it.
 *
 *	Returns: The (new) next available state number to use.
 *
 *************************************************************************/

static int
add_state (list, name, flag, num)

STATELIST      *list;		/* list of states names */
char           *name;		/* name of the new state */
u_int           flag;		/* flag (seen/defined) for the new state */
int             num;		/* next available state number to use */

{
	NAME_LIST      *node;
	static char    *func = "add_state";

	dprintf(4, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (num >= ST_RESERVED)
	{
		logit (LOG_DAEMON, LOG_DEBUG, "%s: FSM size (%d) too large\n",
			func, num);
		exit (-14);
	}

	/* make sure we'll have enough space in the FSM array */
	if (list->nst >= list->maxst)
	{
		dprintf(1, (LOG_DAEMON, LOG_DEBUG,
			"%s: growing FSM size by 8", func));

		list->maxst += 8;
		fsm = (FSM_ENT **) realloc (fsm,
					    (list->maxst) * sizeof (FSM_ENT *));
	}

	/* append new state to states list */
	if ((node = (NAME_LIST *) malloc (sizeof (NAME_LIST)))
							== (NAME_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: FATAL out of memory", func);
		abort ();
	}

	node->name = add_string (name, ASIS);
	node->num = num;
	node->flag = (u_char) flag;
	node->next = list->states;
	list->states = node;
	list->nst++;
	dprintf(4, (LOG_DAEMON, LOG_DEBUG,
		"%s: added '%s' as state %d with flag %s", func,
		(name == (char *) NULL) ? "?" : name,
		num, ((u_char) flag == ST_SEEN) ? "ST_SEEN" : "ST_DEFINED"));
	num++;
	return num;
} /* end of add_state () */

/*************************************************************************
 *
 *	Function: enum_event
 *
 *	Purpose: Enumerate a event (from string to number).
 *		 Note: These numbers are taken from #defines in radius.h
 *
 *	Returns: <number> of the event,
 *		 -2 if the event is not valid.
 *
 *************************************************************************/

static int
enum_event (pevent)

char           *pevent;

{
	typedef struct
	{
		char *ev_name;
		int   ev_value;
	} event_map_t;

	static event_map_t events[] =
	{
		{ EN_NAK,		EV_NAK			},
		{ EN_ACK,		EV_ACK			},
		{ EN_ERROR,		EV_ERROR		},
		{ EN_WAIT,		EV_WAIT			},
		{ EN_FATAL,		EV_FATAL		},
		{ EN_DUP_REQ,		EV_DUP_REQ		},
		{ EN_TIMER,		EV_TIMER		},
		{ EN_TIMEOUT,		EV_TIMEOUT		},
		{ EN_ABORT,		EV_ABORT		},
		{ EN_NEW_AUTHEN,	EV_NEW_AUTHEN		},
		{ EN_NEW_ACCT,		EV_NEW_ACCT		},
		{ EN_NEW_PASSWD,	EV_NEW_PASSWD		},
		{ EN_RE_ACCESS,		EV_RE_ACCESS		},
		{ EN_ACC_CHAL,		EV_ACC_CHAL		},
		{ EN_MGT_POLL,		EV_MGT_POLL		},
		{ EN_AUTH_ONLY,		EV_AUTH_ONLY		},
		{ EN_ACCT_START,	EV_ACCT_START		},
		{ EN_ACCT_STOP,		EV_ACCT_STOP		},
		{ EN_ACCT_ALIVE,	EV_ACCT_ALIVE		},
		{ EN_ACCT_MODEM_START,	EV_ACCT_MODEM_START	},
		{ EN_ACCT_MODEM_STOP,	EV_ACCT_MODEM_STOP	},
		{ EN_ACCT_CANCEL,	EV_ACCT_CANCEL		},
		{ EN_ACCT_ON,		EV_ACCT_ON		},
		{ EN_ACCT_OFF,		EV_ACCT_OFF		},
		{ EN_ACCT_DUP,		EV_ACCT_DUP		},
		{ EN_RC1,		EV_RC1			},
		{ EN_RC2,		EV_RC2			},
		{ EN_RC3,		EV_RC3			},
		{ EN_RC4,		EV_RC4			},
		{ EN_RC5,		EV_RC5			},
		{ EN_RC6,		EV_RC6			},
		{ EN_RC7,		EV_RC7			},
		{ EN_RC8,		EV_RC8			},
		{ EN_RC9,		EV_RC9			},
		{ EN_RC10,		EV_RC10			},
		{ EN_RC11,		EV_RC11			},
		{ EN_RC12,		EV_RC12			},
		{ EN_RC13,		EV_RC13			},
		{ EN_RC14,		EV_RC14			},
		{ EN_RC15,		EV_RC15			},
		{ EN_RC16,		EV_RC16			},
		{ EN_RC17,		EV_RC17			},
		{ EN_RC18,		EV_RC18			},
		{ EN_RC19,		EV_RC19			},
		{ EN_RC20,		EV_RC20			},
		{ EN_RC21,		EV_RC21			},

		{ EN_PW_EXPIRED,	EV_PW_EXPIRED		},
		{ EN_RETRY_LIMIT,	EV_RETRY_LIMIT		},

#ifdef USR_CCA
		{ EN_RF_REQ,		EV_RF_REQ		},
		{ EN_RQ_REQ,		EV_RQ_REQ		},
		{ EN_RQ_RESP,		EV_RQ_RESP		},
		{ EN_DONE,		EV_DONE			},
		{ EN_NAS_REB_REQ,	EV_NAS_REB_REQ		},
		{ EN_NAS_REB_RESP,	EV_NAS_REB_RESP		},
		{ EN_RF_RESP,		EV_RF_RESP		},
#endif	/* USR_CCA */

		{ NULL,			-2			} /* EOL */
	};

	int             i;
	static char    *func = "enum_event";

	dprintf(4, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	for (i = 0; events[i].ev_name != NULL; i++)
	{
		if (strcmp (events[i].ev_name, pevent) == 0)
		{
			return events[i].ev_value;
		}
	}

	return (-2);

} /* end of enum_event () */

/*************************************************************************
 *
 *	Function: extra_stuff
 *
 *	Purpose: Evaluate an extra value pair (or two) for this FSM entry.
 *
 *	Returns: 0, if successful,
 *		 -1, if there were errors.
 *
 *************************************************************************/

static int
extra_stuff (buf, value, string)

char           *buf;
int            *value;
char          **string;

{
	int             result;
	VALUE_PAIR     *list;
	VALUE_PAIR     *vp;
	static char    *func = "extra_stuff";

	dprintf(4, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	result = 0;
	*value = 0;
	*string = '\0';
	vp = (VALUE_PAIR *) NULL;
	list = (VALUE_PAIR *) NULL;
	if (pair_parse (buf, &list, (VALUE_PAIR **) NULL) != 0)
	{
		dprintf(4, (LOG_DAEMON, LOG_DEBUG, "%s: error parsing '%s'",
			func, (buf == (char *) NULL) ? "?" : buf ));
		result = -1;
	}
	else
	{
		vp = list;
		while (vp != (VALUE_PAIR *) NULL)
		{
			switch (vp->ap->type)
			{
			    case PW_TYPE_INTEGER:
				*value = vp->lvalue;
				break;

			    case PW_TYPE_STRING:
				if (strlen (vp->strvalue) < AUTH_ID_LEN)
				{
				    *string = add_string (vp->strvalue, ASIS);
				}
				break;

			    default:
				dprintf(4, (LOG_DAEMON, LOG_DEBUG,
					"%s: unsupported type %d", func,
					vp->ap->type));
				result = -1;
				break;
			}
			vp = vp->next;
		}
	}
	list_free (list);

	return result;
} /* end of extra_stuff () */

/*************************************************************************
 *
 *	Function: find_state
 *
 *	Purpose: Find the numer of the given state in the list of states.
 *
 *	Returns: pointer to the NAME_LIST node for this state,
 *		 (NAME_LIST *) NULL, if the name is not found in the list.
 *
 *************************************************************************/

static NAME_LIST *
find_state (list, name)

STATELIST      *list;		/* list of states names already seen */
char           *name;		/* the state name to search for */

{
	NAME_LIST      *state;
	static char    *func = "find_state";

	dprintf(4, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	for (state = list->states ;
		state != (NAME_LIST *) NULL ;
		state = state->next)
	{
		if (strcmp (state->name, name) == 0)
		{
			break;
		}
	}

	return state;
} /* end of find_state () */

/*************************************************************************
 *
 *	Function: init_fsm
 *
 *	Purpose: Does all the initialization for FSM.
 *
 *	Returns: <number of states>, if the (new) FSM is initialized properly,
 *		 -1, if any errors are detected.
 *
 *************************************************************************/

int
init_fsm (n, fsm_file, main_fsm, def_fsm)

int             n;                /* INPUT: current number of FSM states */
char           *fsm_file;         /* INPUT: pointer to FSM config file name */
FSM_ENT      ***main_fsm;         /* OUTPUT: pointer to FSM table */
FSM_ENT      ***def_fsm;          /* OUTPUT: pointer to default action table */

{
	int             len;
	int             nfsm;
	FILE           *fp;
	char           *fsmfile;
	struct stat     statbuf;
	char            buf[MAXPATHLEN];
	static char    *func = "init_fsm";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (n > 0)
	{
		rad_fsmdelete (n);
	}

	sprintf (buf, "%s/%s", radius_dir, fsm_file);
	if ((fp = fopen (buf, "r")) == (FILE *) NULL)
	{
		dprintf(2, (LOG_DAEMON, LOG_DEBUG,
			"%s: using built-in standard FSM table", func));
		fsmfile = standard_fsm;
		strcpy (buf, "built-in standard FSM table");
		fsm_file = "Built-in";
	}
	else
	{
		stat (buf, &statbuf);
		len = (long) statbuf.st_size;
		if ((fsmfile = (char *) malloc (len)) == (char *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: FATAL out of memory", func);
			abort ();
		}
		fread (fsmfile, len, 1, fp);
		fclose (fp);	/* Cleanup... */
	}

	nfsm = rad_fsminit (fsmfile, &fsm, fsm_file);

	if (fsmfile != standard_fsm)
	{
		free (fsmfile);
	}

	dprintf(1, (LOG_DAEMON, LOG_DEBUG,
		"%s: FSM defined with %d states from %s", func, nfsm, buf));

	if (nfsm == -2)
	{
		return (-1);
	}

	/* ... and the default action table */

	if (default_fsm == (FSM_ENT **) NULL)
	{
		len = rad_fsminit (default_actions, &default_fsm, "DEFAULT");

		if (len == -2)
		{
			dprintf(1, (LOG_DAEMON, LOG_DEBUG,
				"%s: return code %d from rad_fsminit()",
				func, len));
			return (-1);
		}
	}

	*main_fsm = fsm;
	*def_fsm = default_fsm;
	return nfsm;
} /* end of init_fsm () */

/*************************************************************************
 *
 *	Function: rad_fsmdelete
 *
 *	Purpose: Destroy the finite state machine (FSM) data structure.
 *
 *************************************************************************/

static void
rad_fsmdelete (nfsm)

int             nfsm;

{
	FSM_ENT        *pe;
	FSM_ENT        *pen;
	static char    *func = "rad_fsmdelete";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	while (nfsm > 0) /* free the individual FSM state entries */
	{
		nfsm--;
		for (pe = fsm[nfsm]; pe != (FSM_ENT *) NULL; pe = pen)
		{
			pen = pe->next;
			free (pe);
		}
	}

	free (fsm); /* free the FSM array itself */

	return;
} /* end of rad_fsmdelete () */

/*************************************************************************
 *
 *	Function: rad_fsminit
 *
 *	Purpose: Initialize the FSM table in memory from the given file.
 *
 *	Returns: number of states parsed, if successful,
 *		 -2, if errors occurred.
 *
 *************************************************************************/

static int
rad_fsminit (fsmfile, fsm, fsm_name)

char           *fsmfile;
FSM_ENT      ***fsm;
char           *fsm_name;

{
	int             i;		/* loop variable */
	int             more;		/* while loop control variable */
	int             nfsm;		/* state counter */
	int             line_cnt;	/* config file line count */
	int             err_cnt;	/* used in FSM verification */
	int             ref_cnt;	/* used in FSM verification */
	int             cst;		/* current state number */
	int             xvalue;
	FILE           *debugout = stdout;
	char           *ptr;		/* pointer into given file */
	char           *newline;	/* pointer into given file */
	char           *p;		/* pointer into line buffer */
	char           *s;		/* pointer into line buffer */
	char           *b;		/* pointer into line buffer */
	char           *e;		/* pointer into line buffer */
	char           *a;		/* pointer into line buffer */
	char           *n;		/* pointer into line buffer */
	char           *xtra;		/* pointer into line buffer */
	FSM_ENT        *pe;		/* pointer to an entry */
	FSM_ENT        *pn;		/* temporary pointer */
	FSM_ENT       **pfsm;		/* new FSM */
	NAME_LIST      *node;		/* pointer to a state NAME_LIST node */
	NAME_LIST      *node_state;	/* pointer to current FSM state */
	NAME_LIST     **prev;		/* used to free "sts" states below */
	STATELIST       sts;		/* state list data structure */
	char           *xstring;
	char            buf[256];	/* line buffer for file reading */
	static char    *func = "rad_fsminit";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	/* allocate new FSM */
	sts.nst = 0;
	sts.maxst = NUMSTATES;
	sts.states = (NAME_LIST *) NULL;
	if ((pfsm = (FSM_ENT **) calloc (sts.maxst, sizeof (FSM_ENT *)))
							== (FSM_ENT **) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: FATAL out of memory for FSM",
			func);
		abort ();
	}

	line_cnt = 0;		/* line number for debugging */
	nfsm = 0;		/* have seen no state yet */
	more = 1;
	ptr = fsmfile;
	node_state = (NAME_LIST *) NULL;	/* Saftey */

	while (more)
	{
		if ((newline = strchr (ptr, '\n')) == (char *) NULL)
		{
			more = 0;
			break;
		}
		else
		{
			i = newline - ptr;
			strncpy (buf, ptr, i);
			buf[i] = '\0';
			ptr = ++newline;
		}

		line_cnt++;

		if (*buf == COMMENT || *buf == '\n' || *buf == '\0')
		{
			continue;
		}
		else
		{
			if (*buf == '%')
			{
				if ((p = strtok (buf, " \t")) != (char *) NULL)
				{
					/* Have found a keyword */
					if (strcasecmp (p, "%FSMID") == 0)
					{
						/* Find end of line. */
						if ((p = strtok (NULL, "\n\r"))
							       != (char *) NULL)
						{
							int j;

							if (fsm_id
							       != (char *) NULL)
							{
								free (fsm_id);
							}

							j = strlen (p);
							if (j > MAX_FSMID_LEN)
							{
							   /* Truncate. */
							   p[MAX_FSMID_LEN]
									= '\0';
							}
							fsm_id = rad_strdup (p);
							fsm_name = fsm_id;
						}
					}
				}
				/* Just treat FSMID as a comment. */
				continue;
			}
		}

		while (i > 0) /* capitalize all tokens on line */
		{
			i--;
			buf[i] = toupper (buf[i]);
		}

		xvalue = 0;
		xstring = (char *) NULL;

		if (isalpha(*buf)) /* states start in column one */
		{
			if ((p = strtok (buf, ":")) != NULL) /* have state */
			{
				if ((s = strtok (NULL, " \t\r\n")) != NULL)
				{
					if ((*s != '#') && (*s != '\0'))
					{
						logit (LOG_DAEMON, LOG_ERR,
				"%s: Warning, extra junk on end of state label",
							func);
						logit (LOG_DAEMON, LOG_ERR,
					  "%s: at line %d, for state %s, '%s'",
							func, line_cnt, p, s);
					}
				}

				if (strcmp (p, "END") == 0)
				{
					dprintf(4, (LOG_DAEMON, LOG_DEBUG,
						"%s state detected", p));
					more = 0;
				}
				else		/* insert new state */
				{
					dprintf(4, (LOG_DAEMON, LOG_DEBUG,
					       "about to insert new state %s",
						p));
					if ((node = find_state (&sts, p))
							== (NAME_LIST *) NULL)
					{
						cst = nfsm;
						nfsm = add_state (&sts, p,
						       (u_int) ST_DEFINED, cst);
						node_state = find_state (&sts,
									 p);
					}
					else /* have seen this state already */
					{
						node_state = node;
						if (node->flag == ST_SEEN)
						{
							cst = node->num;
							node->flag = ST_DEFINED;
						}
						else /* oops, was defined */
						{
							logit (LOG_DAEMON,
								LOG_ERR,
						"%s: duplicate state: line %3d",
								func, line_cnt);
							logit (LOG_DAEMON,
								LOG_ERR,
								"'%s'", buf);
							return (-2);
						}
					}
				}
			}
			else
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: bad state: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}
		}
		else /* grab (state.action.event, action, nextstate) 3-tuple */
		{
			if (nfsm == 0)	/* do we have some/any state(s)? */
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: no states: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((s = strtok (buf, " \t.")) == NULL) /* state ? */
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: error parse state: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((b = strtok (NULL, " \t.")) == NULL) /* action ? */
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: error parse old action: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((e = strtok (NULL, " \t\n\r")) == NULL) /* event? */
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: error parse event: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((a = strtok (NULL, " \t\n\r")) == NULL) /* action?*/
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: error parse action: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((n = strtok (NULL, " \t\n\r")) == NULL) /* next ? */
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: error parse next state: line %3d",
					func, line_cnt);
				logit (LOG_DAEMON, LOG_ERR,
					"'%s'", buf); /* Log the parse error. */
				return (-2);
			}

			if ((xtra = strtok (NULL, " \t\n\r")) != NULL) /*extra*/
			{
				if (extra_stuff (xtra, &xvalue, &xstring) < 0)
				{
					logit (LOG_DAEMON, LOG_ERR,
					 "%s: error with extra stuff: line %3d",
						func, line_cnt);
					logit (LOG_DAEMON, LOG_ERR,
						"'%s'", buf); /* Log stuff. */
					return (-2);
				}
			}

			if ((pe = (FSM_ENT *) malloc (sizeof (FSM_ENT)))
							== (FSM_ENT *) NULL)
			{
				logit (LOG_DAEMON, LOG_ALERT,
					"%s: FATAL out of memory for FSM",
					func);
				abort ();
			}

			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"have 3-tuple '%s.%s.%s'", s, b, e));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"   action '%s', next state '%s'", a, n));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"   value %d and string '%s'", xvalue,
				(xstring == (char *) NULL) ? "?" : xstring));

			pe->xvalue = xvalue;
			pe->xstring = xstring;
			pe->fsm_name = fsm_name;
			pe->state_name = node_state->name;

			/* handle state part of (state.action.event) 3-tuple */
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"finding event state '%s' with number:", s));
			if (strcmp (s, "*") == 0)
			{
				pe->event.state = ST_ANY;
				dprintf(4, (LOG_DAEMON, LOG_DEBUG,
					"     %u", ST_ANY));
			}
			else
			{
				if ((node = find_state (&sts, s))
							== (NAME_LIST *) NULL)
				{
					i = nfsm; /* pick next state */
					nfsm = add_state (&sts, s,
							  (u_int) ST_SEEN, i);
					pe->event.state = (u_char) i;
					dprintf(4, (LOG_DAEMON, LOG_DEBUG,
						"     %u and flag %s", i,
						"ST_SEEN"));
				}
				else /* have seen already */
				{
					pe->event.state = (u_char) node->num;
					dprintf(4, (LOG_DAEMON, LOG_DEBUG,
						"     %u and flag %s",
						node->num,
						(node->flag == ST_SEEN)
						? "ST_SEEN" : "ST_DEFINED"));
				}
			}

			/* handle action part of (state.action.event) 3-tuple */
			if (b[0] == '+')
			{
				pe->event.a.proxy = add_string (&b[1], ASIS);
				pe->event.isproxy = 1;
			}
			else
			{
				pe->event.isproxy = 0;
				if ((pe->event.a.aatv = find_aatv (b))
							       == (AATV *) NULL)
				{
					/* Log invalid previous action. */
					logit (LOG_DAEMON, LOG_ERR,
				     "%s: invalid old AATV name: '%s' line %3d",
						func, b, line_cnt);
					return (-2);
				}
			}

			/* handle event part of (state.action.event) 3-tuple */
			if ((pe->event.value = enum_event (e)) == -2)
			{
				/* Log invalid event. */
				logit (LOG_DAEMON, LOG_ERR,
					"%s: invalid event name: '%s' line %3d",
					func, e, line_cnt);
				return (-2);
			}

			if ((pe->action = find_aatv (a)) == (AATV *) NULL)
			{
				/* Log invalid action. */
				logit (LOG_DAEMON, LOG_ERR,
				       "%s: invalid action name: '%s' line %3d",
					func, a, line_cnt);
				return (-2);
			}

			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"looking for next_state %s with number:", n));
			if (strcmp (n, "END") == 0)
			{
				pe->next_state = ST_END;
				dprintf(4, (LOG_DAEMON, LOG_DEBUG,
					"     %d", ST_END));
			}
			else
			{
				if (strcmp (n, "SAME") == 0)
				{
					pe->next_state = ST_SAME;
					dprintf(4, (LOG_DAEMON, LOG_DEBUG,
						"     %d", ST_SAME));
				}
				else
				{
					if ((node = find_state (&sts, n))
							== (NAME_LIST *) NULL)
					{
						i = nfsm; /* pick next state */
						nfsm = add_state (&sts, n,
							    (u_int) ST_SEEN, i);
						pe->next_state = (u_char) i;
						dprintf(4, (LOG_DAEMON,
							LOG_DEBUG,
							"     %u and flag %s",
							i, "ST_SEEN"));
					}
					else /* have seen already */
					{
						pe->next_state =
							(u_char) node->num;
						dprintf(4, (LOG_DAEMON,
							LOG_DEBUG,
							"     %u and flag %s",
							node->num,
							(node->flag == ST_SEEN)
							? "ST_SEEN"
							: "ST_DEFINED"));
					}
				}
			}

			pe->next = (FSM_ENT *) NULL;

			/* link *pe to the FSM table */
			if ((pn = pfsm[cst]) == (FSM_ENT *) NULL)
			{
				pfsm[cst] = pe;
			}
			else
			{
				while (pn->next)
				{
					pn = pn->next;
				}
				pn->next = pe;
			}
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"entry added to state %d:", cst));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"     event [%d,%s,%d]", pe->event.state,
				(pe->event.isproxy == 1) ? pe->event.a.proxy :
					(char *) pe->event.a.aatv->id,
				pe->event.value));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"     action %s", pe->action->id));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"     next_state %d", pe->next_state));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"     value %d", pe->xvalue));
			dprintf(4, (LOG_DAEMON, LOG_DEBUG,
				"     string '%s'",
				(pe->xstring != (char *) NULL)
						? pe->xstring : "?"));
		} /* end of else not isalpha() */
	} /* end of while (more) */

	/*
	 * now check if the state machine is valid
	 * 	-	See if there is any state (except Start)
	 *		not-reachable from any other states;
	 * 	-	See if there is any state which is
	 *		not defined.
	 *	-	Print out the whole state machine in debug
	 *		mode for inspection.
	 */
	err_cnt = 0; /* indicate assumption about clean FSM table */
	for (i = 1; i < nfsm; i++) /* for each state */
	{
		ref_cnt = 0; /* reference counter for state "i" */
		for (cst = 0; cst < nfsm; cst++)
		{
			for (pe = pfsm[cst];
				pe != (FSM_ENT *) NULL;
				pe = pe->next)
			{
				if (i == pe->next_state)
				{
					ref_cnt++;
				}
			}
		}

		/* error if no FSM entry refers to state "i" */
		if (ref_cnt == 0)
		{
			pe = pfsm[i];
			logit (LOG_DAEMON, LOG_ERR,
				"%s: non-reachable state %s::%s [%d]",
				func, pe->fsm_name, pe->state_name, i);
			err_cnt = i;
		}
	} /* end of for each state */

	if (debug_flag >= 2)
	{
		if (ddt)
		{
			debugout = ddt;
		}

		fprintf (debugout, "\nState  Event                 Action        Next State Value String\n");
		fprintf (debugout, "-----  --------------------  ------------  ---------- ----- ------\n\n");
		fprintf (debugout, "\n");
	}

	for (i = 0; i < nfsm; i++) /* for each state (again) */
	{
		for (pe = pfsm[i]; pe != (FSM_ENT *) NULL; pe = pe->next)
		{
			if ((pe->next_state >= nfsm) &&
				(pe->next_state < ST_RESERVED))
			{
				logit (LOG_DAEMON, LOG_ERR,
				    "%s: non-existent state %d in %s::%s [%d]",
					func, pe->next_state, pe->fsm_name,
					pe->state_name, i);
				err_cnt++;
			}

			if (debug_flag >= 2)
			{
				fprintf (debugout,
			  "%3d   [%3d,%-13s,%2d] %-12s  %5d       %3d  '%s'\n",
					i, pe->event.state,
					(pe->event.isproxy == 1) ?
						pe->event.a.proxy :
						(char *) pe->event.a.aatv->id,
					pe->event.value, pe->action->id,
					pe->next_state, pe->xvalue,
					pe->xstring ? pe->xstring : "");
			}
		}

		if (debug_flag >= 2)
		{
			fprintf (debugout, "\n");
		}
	} /* end of for each state (again) */

	if (debug_flag >= 2)
	{
		fprintf (debugout, "\n");
	}

	*fsm = pfsm;

	/* free the states list */
	if (debug_flag >= 2)
	{
		fprintf (debugout, "State  State Name     Number Flag\n");
		fprintf (debugout, "-----  -------------- ------ ----\n\n");
	}

	i = 0;
	while (sts.states != (NAME_LIST *) NULL)
	{
		prev = &sts.states;
		for (node = sts.states ;
			node != (NAME_LIST *) NULL ;
			node = node->next)
		{
			if (node->next == (NAME_LIST *) NULL)
			{
				break;
			}
			prev = &(node->next);
		}

		if (debug_flag >= 2)
		{
			fprintf (debugout, " %2d    %-14s %3d    %s", i, node->name,
			    node->num,
			    (node->flag == ST_SEEN) ? "ST_SEEN" : "ST_DEFINED");
		}

		if (node->flag == ST_SEEN)
		{
			if (debug_flag >= 2)
			{
				fprintf (debugout, " <--- this is an error\n");
			}
			logit (LOG_DAEMON, LOG_ERR,
				"%s: state %s seen but not defined",
				func, node->name);
			nfsm = -2;
		}
		else
		{
			if (debug_flag >= 2)
			{
				fprintf (debugout, "\n");
			}
		}

		free (node);
		*prev = (NAME_LIST *) NULL;
		i++;
	}

	if (err_cnt > 0)
	{
		return (-2);
	}

	if (debug_flag >= 2)
	{
		fprintf (debugout, "\n");
	}

	return nfsm;
} /* end of rad_fsminit () */
