/*
 *
 *      Copyright (c) 1996 Ascend Communications, Inc.
 *      All rights reserved.
 *
 *      Permission to copy, display, distribute and make derivative works
 *      from this material in whole or in part for any purpose is granted
 *      provided that the above copyright notice and this paragraph are
 *      duplicated in all copies.  THIS SOFTWARE IS PROVIDED "AS IS" AND
 *      WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES INCLUDING, WITHOUT
 *      LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *      FOR A PARTICULAR PURPOSE.
 *
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/*
 *
 * Public entry points in this file:
 *
 */

static char     rcsid[] = "$Id: pooladdr.c,v 1.3 1998/07/28 18:18:57 web Exp $";

#include	<sys/types.h>
#include	<sys/param.h>
#include	<sys/time.h>
#include	<sys/errno.h>
#include	<sys/file.h>
#include	<sys/wait.h>
#include	<sys/socket.h>
#include	<net/if.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<errno.h>
#include	<signal.h>
#include	<memory.h>
#include	<syslog.h>

#include	"radius.h"

#ifndef IP_ADDR_POOL
static AATV     ip_alloc_aatv = DEF_AATV_DIRECT("IPPOOL", fatal_action);

AATVPTR         rad_ip_pool_aatv = & ip_alloc_aatv;
#else	/* IP_ADDR_POOL */

extern char     recv_buffer[RAD_RECV_BUFFER_SIZE];
extern char     send_buffer[RAD_SEND_BUFFER_SIZE];
extern int      debug_flag;
extern int      spawn_flag;     /* 0 => no spawning, 1 => spawning allowed */
extern char    *radius_dir;

static int      sockfd = -1;
static u_char   timeout;
static char     white_space[] = " \t\r\n";

static int      ip_address_recv PROTO((UINT4, AUTH_REQ *));
static int      ip_address_send PROTO((int, void *, UINT4, ipaddr_t,
					ADDRESS_CHUNK *, int));
static int      parse_global PROTO((char *, ADDRESS_CHUNK *));
static UINT4    parse_hosts PROTO((void));
static int      parse_pool PROTO((char *, ADDRESS_CHUNK **));
static void     pool_time_out PROTO((int));
RADIPA_PACKET  *reorder_integers PROTO((char*));
static int      try_connect PROTO((UINT4));

UINT4           addr;

#define	IP_POOL_TIME	3 * 60 /* Need a response within this time period */

static void     pool_init PROTO((AATV *));
static int      pool_addr PROTO((AUTH_REQ *, int, char *));
static void     pool_cleanup PROTO((void));

static AATV     ip_alloc_aatv = DEF_AATV_FREPLY_FULL("IPPOOL", -1, pool_init,
							NULL, pool_addr,
							pool_cleanup, 0);
AATVPTR         rad_ip_pool_aatv = & ip_alloc_aatv;

/*************************************************************************
 *
 *	Function: pool_init
 *
 *	Purpose: Initialize the TCP connection to the RADIPAD daemon.
 *
 *************************************************************************/

static void
pool_init (unused)

AATV   *unused;

{
	UINT4           addr;
	static int      init_count = 0;
	static char    *func = "pool_init";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	init_count++;
	if (init_count <= 1)	/* Ensure that users file is read into core. */
	{
		return;
	}

	if (sockfd == -1)
	{
		sockfd = socket (AF_INET, SOCK_STREAM, 0); /* for TCP */
	}

	/* This assumes the users file has been read into memory.  See above. */
	if ((addr = parse_hosts ()) == INADDR_ANY)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: No server listed for RADIPAD", func);
		return;
	}

	return;
} /* end of pool_init () */

/*************************************************************************
 *
 *	Function: pool_addr
 *
 *	Purpose: This is an AATV action function for the
 *		 PW_ASCEND_RADIPA_ALLOCATE [50] and
 *		 PW_ASCEND_RADIPA_RELEASE [51] requests.
 *
 *************************************************************************/

static int
pool_addr (authreq, value, afpar)

AUTH_REQ       *authreq;
int             value;
char           *afpar;

{
	int             cmd;
	int             count;
	int             how_many;
	int             loop_count;
	int             result;
	ipaddr_t        addr;
	UINT4           daemon;
	UINT4           ip_address;
	UINT4           router_ip_address;
	UINT4           z = (UINT4) 0;
	ADDRESS_CHUNK   chunks_0[MAX_ADDRESS_CHUNKS];
	ADDRESS_CHUNK  *chunks = chunks_0;
	ADDRESS_CHUNK  *payload;
	VALUE_PAIR     *vp;
	char           *user_name;
	struct sockaddr_in sin;
	sigset_t        signals;
	struct sigaction action;
	struct sigaction a_save;
	static char    *func = "pool_addr";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Set up alarm and signal handler to bail us out if we take too long */
	alarm (IP_POOL_TIME); /* Make it pretty long */

	memset ((char *) &action, '\0', sizeof (action));
	sigemptyset (&signals);			/* Init signal enable mask */
	sigaddset (&signals, SIGALRM);  
	action.sa_handler = pool_time_out;	/* Set up our timeout handler */
	sigaction (SIGALRM, &action, &a_save);	/* Ready to handle it */
	sigprocmask (SIG_UNBLOCK, &signals, NULL);	/* Now allow alarm */

	/* If one was pending, we just took it. */
	timeout = (u_char) 0;	/* Initing now will ignore any pending alarm */

	if ((vp = get_vp (authreq->cur_request, PW_NAS_IP_ADDRESS)) == NULL_VP)
	{
		missing_attribute (authreq, func, PW_NAS_IP_ADDRESS, NULL);
		sigaction (SIGALRM, &a_save, NULL);	/* Restore state. */
		return EV_NAK;
	}
	router_ip_address = vp->lvalue;

	if (authreq->code == PW_ASCEND_RADIPA_ALLOCATE)
	{
		if ((vp = get_vp (authreq->cur_request,
					PW_USER_NAME)) == NULL_VP)
		{
			missing_attribute (authreq, func, PW_USER_NAME, NULL);
			sigaction (SIGALRM, &a_save, NULL); /* Restore state. */
			return EV_NAK;
		}
		user_name = vp->strvalue;	/* name of the router */

		count = parse_pool (user_name, &chunks);
		if (count == 0)
		{
			dprintf(2, (LOG_AUTH, LOG_DEBUG,
				"zero address chunks for %s", func, user_name));
			avpair_add (&authreq->cur_request,
						PW_FRAMED_IP_ADDRESS, &z, 0);
			sigaction (SIGALRM, &a_save, NULL); /* Restore state. */
			return EV_ACK;
		}
	}
	else /* was PW_ASCEND_RADIPA_RELEASE */
	{
		if ((vp = get_vp (authreq->cur_request,
					PW_FRAMED_IP_ADDRESS)) == NULL_VP)
		{
			missing_attribute (authreq, func,
						PW_FRAMED_IP_ADDRESS, NULL);
			sigaction (SIGALRM, &a_save, NULL); /* Restore state. */
			return EV_NAK;
		}
		ip_address = vp->lvalue;	/* address to release */
	}

	/* This assumes the users file has been read into memory. */
	if ((daemon = parse_hosts ()) == INADDR_ANY)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: No server listed for RADIPAD", func);
		sigaction (SIGALRM, &a_save, NULL);	/* Restore state. */
		return EV_NAK;
	}

	if (authreq->code == PW_ASCEND_RADIPA_ALLOCATE)
	{
		cmd = RADIPA_ALLOCATE;
		addr = INADDR_NONE;
		payload = chunks;
		how_many = count;
	}
	else /* was PW_ASCEND_RADIPA_RELEASE */
	{
		cmd = RADIPA_RELEASE;
		addr = ip_address;
		payload = (ADDRESS_CHUNK *) NULL;
		how_many = 0;
	} 

	loop_count = 0;
	do
	{
		if (ip_address_send (cmd, authreq, router_ip_address,
						addr, payload, how_many) < 0)
		{
			result = try_connect (daemon);
			switch (result)
			{
			    case 1:		/* Address was in use. */
				sleep (2);
				loop_count++;
				break;

			    case 0:
				loop_count = 3;	/* Indicate done with loop. */
				break;

			    case -1:		/* Some error occurred. */
				/* Restore signal state. */
				sigaction (SIGALRM, &a_save, NULL);
				return EV_NAK;
			}
			continue;
		} 
		break;
	} while (loop_count < 2);

	for (;;) /* Block for RADIPAD packet in this child process. */
	{
		if (ip_address_recv (daemon, authreq) < 0)
		{
			sigaction (SIGALRM, &a_save, NULL); /* Restore state. */
			return EV_NAK;
		}

		sigaction (SIGALRM, &a_save, NULL);	/* Restore state. */
		return EV_ACK;		/* It's a success! */
	}

} /* end of pool_addr () */

/*************************************************************************
 *
 *	Function: pool_cleanup
 *
 *	Purpose: Release the socket file descriptor.
 *
 *************************************************************************/

static void
pool_cleanup ()

{
	static char    *func = "pool_cleanup";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	close (sockfd);
	sockfd = -1;

	return;
} /* end of pool_cleanup () */

/*************************************************************************
 *
 *	Function: ip_address_send
 *
 *	Purpose: Sends IP address pool requests to RADIPAD.
 *
 *	Returns: number of bytes sent, or
 *		 -1, otherwise.
 *
 *************************************************************************/

static int
ip_address_send (code, handle, router_address, ipaddr, chunks, count)

int             code;
void           *handle;
UINT4           router_address;
ipaddr_t        ipaddr;
ADDRESS_CHUNK  *chunks;
int             count;

{
	int             bytes_sent;
	int             total_length;
	RADIPA_PACKET  *packet;
	static char    *func = "ip_address_send";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	packet = (RADIPA_PACKET *) send_buffer;
	packet->rp_code = (char) code; /* RADIPA_ALLOCATE or RADIPA_RELEASE */
	packet->rp_pad = ~0;
	packet->rp_count = htons(count);
	packet->rp_handle = handle;
	packet->rp_router_address = router_address;
	packet->rp_ip_address = htonl(ipaddr);

	if (code == RADIPA_ALLOCATE)
	{
		memcpy (packet->RP_CHUNKS, chunks,
			sizeof (ADDRESS_CHUNK) * count);
		total_length = SIZEOF_RADIPA_ALLOCATE (count);
	}
	else
	{
		total_length = SIZEOF_RADIPA;
	}

	bytes_sent = send (sockfd, (char *) send_buffer, total_length, 0);

	if (bytes_sent < 0)
	{
		if (timeout)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: timeout on connect to %s",
			 	func, ip_hostname (router_address));
		}
		else
		{
			logit (LOG_DAEMON, LOG_ERR,
			    "%s: Can't send release ipaddr request to %s: %s",
			    func, ip_hostname (router_address), get_errmsg ());
		}
		return (-1);
	}
	return bytes_sent;
} /* end of ip_address_send () */

/*************************************************************************
 *
 *	Function: ip_address_recv
 *
 *	Purpose: Reads RADIPAD replies from server.
 *
 *	Returns:  0, if successful,
 *		 -1, otherwise.
 *
 *************************************************************************/

static int
ip_address_recv (ipaddr, authreq)

UINT4           ipaddr;
AUTH_REQ       *authreq;

{
	int             total_length;
	UINT4           ip_address;
	RADIPA_PACKET  *packet;
	static char    *func = "ip_address_recv";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((total_length = recv (sockfd, (char *) recv_buffer, 
						sizeof (recv_buffer), 0)) < 0)
	{
		if (timeout)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: timeout on connect to %s",
		 		func, ip_hostname (ipaddr));
		}
		else
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: %s s=%s",
				func, get_errmsg (), ip_hostname (ipaddr));
		}
		return (-1);
	}

	if (total_length == 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: EOF on radipad socket", func, strerror (errno));
		return (-1);
	}

	if (authreq->code == PW_ASCEND_RADIPA_ALLOCATE)
	{
		packet = reorder_integers ((char *) recv_buffer);
		ip_address = packet->rp_ip_address;
		avpair_add (&authreq->cur_request, PW_FRAMED_IP_ADDRESS,
								&ip_address, 0);
	}
	return 0;

} /* end of ip_address_recv () */

/*************************************************************************
 *
 *	Function: parse_global
 *
 *	Purpose: Parse ASCEND_ASSIGN_IP_GLOBAL_POOL reply-item.
 *
 *	Returns: 0, if pool name not found,
 *		 number of ADDRESS_CHUNKs found otherwise.
 *
 *************************************************************************/

static int
parse_global (pool_name, chunk_0)

char           *pool_name;
ADDRESS_CHUNK  *chunk_0;

{
	long            count;
	char           *base_addr_string;
	char           *count_string;
	ADDRESS_CHUNK  *chunk;
	VALUE_PAIR     *reply_pairs;
	VALUE_PAIR     *vp;
	ipaddr_t        base_address;
	static char    *func = "parse_global";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	chunk = chunk_0;

	if (user_find (NULL, pool_name, 0, (VALUE_PAIR **) NULL,
			(VALUE_PAIR **) NULL, &reply_pairs, 0))
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: Pseudo-user '%s' not found", func, pool_name);
		return 0;
	}

	for (vp = reply_pairs; vp; vp = vp->next)
	{
		if (vp->attribute == ASCEND_IP_POOL_DEFINITION)
		{
			/* vp->strvalue[vp->lvalue] = '\0'; */
			base_addr_string = strtok (vp->strvalue, white_space);

			if (!strchr (base_addr_string, '.'))
			{
				base_addr_string = strtok (NULL, white_space);
			}

			count_string = strtok (NULL, white_space);
			base_address = inet_addr (base_addr_string);
			count = strtol (count_string, (char **) NULL, 0);

			if (base_address == INADDR_NONE)
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Bogus pool definition address: %s",
					func, base_addr_string);
			}
			else if (count == 0)
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Bogus pool definition count: %s",
					func, count_string);
			}
			else
			{
				chunk->base_address = base_address;
				chunk->count = htonl(count);
				chunk++;
			}
		}
	}
	avpair_free (reply_pairs);

	return (chunk - chunk_0);
} /* end of parse_global () */

/*************************************************************************
 *
 *	Function: parse_hosts
 *
 *	Purpose: Extract the "radipa-hosts" pseudo-user from the users
 *		 file and parse the IP addresses.  The server (if found)
 *		 is added to the clients list.  Any other NAS clients
 *		 are also added to the clients list, if not already there.
 *
 *	Returns: address of the RADIPAD server, or
 *		 INADDR_ANY, if RADIPA_HOSTS is not found.
 *
 *************************************************************************/

static UINT4
parse_hosts ()

{
	UINT4           ipaddr;
	VALUE_PAIR     *reply_pairs = 0;
	VALUE_PAIR     *vp;
	static char    *func = "parse_hosts";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (user_find (NULL, RADIPA_HOSTS, 0, (VALUE_PAIR **) NULL,
			(VALUE_PAIR **) NULL, &reply_pairs, 0))
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: Pseudo-user '%s' not found", func, RADIPA_HOSTS);
		return INADDR_ANY;
	}

	for (vp = reply_pairs; vp; vp = vp->next)
	{
		if (vp->attribute == ASCEND_ASSIGN_IP_SERVER)
		{
			ipaddr = vp->lvalue;
		}
	}
	avpair_free (reply_pairs);

	return ipaddr;
} /* end of parse_hosts () */

/*************************************************************************
 *
 *	Function: parse_pool
 *
 *	Purpose: Given a user name, look for reply-items of pool names to
 *		 parse.  Allocate memory for the "chunks" of IP addresses.
 *
 *	Returns: 0, if user name not found,
 *		 number of ADDRESS_CHUNKs found otherwise.
 *
 *************************************************************************/

static int
parse_pool (user_name, chunks_ptr)

char           *user_name;
ADDRESS_CHUNK **chunks_ptr;

{
	int             chunks_malloced = 0;
	int             count;
	ADDRESS_CHUNK  *chunks;
	VALUE_PAIR     *reply_pairs;
	VALUE_PAIR     *vp;
	char            pool_name[AUTH_STRING1_LEN + 1];
	static char    *func = "parse_pool";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	if (user_find (NULL, user_name, 0, (VALUE_PAIR **) NULL,
			(VALUE_PAIR **) NULL, &reply_pairs, 0))
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: User '%s' not found", func, user_name);
		return 0;
	}

	if (*chunks_ptr == 0)
	{
		*chunks_ptr = MALLOC(ADDRESS_CHUNK, MAX_ADDRESS_CHUNKS);
		chunks_malloced = 1;
	}
	chunks = *chunks_ptr;

	for (vp = reply_pairs; vp; vp = vp->next)
	{
		if (vp->attribute == ASCEND_ASSIGN_IP_GLOBAL_POOL)
		{
			strncpy (pool_name, vp->strvalue, vp->lvalue);
			pool_name[vp->lvalue] = '\0';
			chunks += parse_global (pool_name, chunks);
		}
	}
	avpair_free (reply_pairs);

	count = chunks - *chunks_ptr;
	if (chunks_malloced)
	{
		*chunks_ptr = REALLOC(*chunks_ptr, ADDRESS_CHUNK, count);
	}

	return count;
} /* end of parse_pool () */

/*************************************************************************
 *
 *	Function: pool_time_out
 *
 *	Purpose: Alarm signal handler to just set flag on alarm.
 *
 *************************************************************************/

static void
pool_time_out (signal)

int 	signal;

{
	/* Set flag.  EINTR will cause blocking routines to look */
	timeout = 1;	
} /* end of pool_time_out */

/*************************************************************************
 *
 *	Function: reorder_integers
 *
 *	Purpose: Convert a request block from network to host byte ordering.
 *
 *************************************************************************/

RADIPA_PACKET *
reorder_integers (buf)

char           *buf;

{
	RADIPA_PACKET  *request;
	static char    *func = "reorder_integers";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	request = (RADIPA_PACKET *) buf;
	request->rp_count = ntohs(request->rp_count);
	request->rp_ip_address = ntohl(request->rp_ip_address);
	request->rp_router_address = ntohl(request->rp_router_address);

	if (request->rp_code == RADIPA_ALLOCATE)
	{
		ADDRESS_CHUNK  *chunk = request->RP_CHUNKS;
		ADDRESS_CHUNK  *end = &chunk[request->rp_count];

		for ( ; chunk < end ; chunk++)
		{
			chunk->base_address = ntohl(chunk->base_address);
			chunk->count = ntohl(chunk->count);
		}
	}

	return request;
} /* end of reorder_integers () */

/*************************************************************************
 *
 *	Function: try_connect
 *
 *	Purpose: Attempt to connect to the TCP socket.
 *
 *	Returns: 1 if the TCP socket is in use,
 *		 0 if the session is connected successfully,
 *		-1 if there was an error or a timeout occurred.
 *
 *************************************************************************/

static int
try_connect (addr)

UINT4           addr;

{
	u_short         svc_port;
	int             loop_count;
	struct sockaddr_in sin;
	static char    *func = "try_connect";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	logit (LOG_DAEMON, LOG_INFO, "%s: Establishing connection to RADIPAD",
		func);
	svc_port = RADIPA_SERVICE_PORT;	/* Use default configured value */

	memset ((char *) &sin, '\0', sizeof (sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(addr);
	sin.sin_port = htons(svc_port);

	if (connect (sockfd, (struct sockaddr *) &sin, sizeof (sin)) < 0)
	{
		if (errno == EADDRINUSE) /* For possible other child. */
		{
			sleep (2);
			return 1;
		}

		if (timeout)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: timeout on connect to %s",
		 		func, ip_hostname (addr));
		}
		else
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: %s dest=%s",
		 		func, get_errmsg (), ip_hostname (addr));
		}
		return (-1);
	}

	return 0;
} /* end of try_connect */

#endif	/* IP_ADDR_POOL */
