#ifdef USR_CCA

/*
 *
 * Copyright (c) 1996 U.S. Robotics, Access Corp.
 * All rights reserved.
 *
 * Permission to copy, display, distribute and make derivative works
 * from this material in whole or in part for any purpose is granted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all copies.  THIS SOFTWARE IS PROVIDED "AS IS" AND
 * WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES INCLUDING, WITHOUT
 * LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * If providing code not subject to a copyright please indicate that the
 * code has been dedicated to the public.
 *
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

static char     rcsid[] = "$Id: res_qry.c,v 1.9 1998/07/23 20:27:21 rsc Exp $";

/****************************************************************************
 *
 *			res_qry.c
 *
 *	This file contains the AATV and the functions required for
 *	Resource Querying.
 *
 ***************************************************************************/

#include	<sys/param.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<sys/ioctl.h>
#include	<sys/file.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/wait.h>
#include	<sys/stat.h>

#if defined(sys5)
#include	<sys/sysmacros.h>
#endif	/* sys5 */

#include	<net/if.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<fcntl.h>
#include	<errno.h>
#include	<memory.h>
#include	<signal.h>
#include	<syslog.h>
#include	<string.h>

#include	"radius.h"

extern int           debug_flag;
extern AUTH_REQ_Q    global_auth_q;
extern UINT4         self_ip[];
extern u_short       auth_port;

/****************************************************************************
 *
 *	RES_QRY_RESP AATV definition
 *
 ***************************************************************************/

static int rad_rq_resp_action PROTO ((AUTH_REQ *, int, char *));
static int add_resources PROTO ((VALUE_PAIR *, char *, CLIENT_ENTRY *));
static ASSIGNED_IP *make_ip_node PROTO ((UINT4, UINT4, USER_ENTRY *, int));
static char nas_proc PROTO ((AUTH_REQ *, AUTH_REQ *));
static char proxy_proc PROTO ((AUTH_REQ *, AUTH_REQ *));
static int add_local_vpn PROTO ((AUTH_REQ *, char *));

static AATV     rq_resp_aatv = DEF_AATV_DIRECT("RES_QRY_RESP",
						rad_rq_resp_action);
AATVPTR         rad_rq_resp_aatv = &rq_resp_aatv;

/***************************************************************************
 *
 *	Function: rq_req_init
 *
 *	Purpose: Make up AUTH_REQ structures with code PW_RESOUECE_QUERY_REQ
 *		 for each Proxy client and insert them into the given queue.
 *
 ***************************************************************************/

void
rq_req_init ()

{
	CLIENT_ENTRY   *client_ent;
	CLIENT_ENTRY   *client_list;
	char           *func = "rq_req_init";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	client_list = get_client_list ();

	/* Process clients. */
	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_ent->next)
	{
		if ((((client_ent->client_type & CE_PROXY) != CE_PROXY) &&
			    ((client_ent->client_type & CE_NAS) != CE_NAS)) ||
			client_ent->version == VER1)
		{
			client_ent->state = GOT_RQ_RESP;
			continue;
		}


		send_rq_req (client_ent);
	}
	return;
} /* end of rq_req_init () */

/****************************************************************************
 *
 *	Function: rad_rq_resp_action
 *	Purpose: Action function of the RES_QRY_RESP AATV.  It handles the
 *		 incoming Resource Query Responses while in the Initialization
 *		 FSM.
 *
 ****************************************************************************/

static int
rad_rq_resp_action (authreq, value, afpar)

AUTH_REQ       *authreq;
int             value;
char           *afpar;

{
	AUTH_REQ       *auth;
	AUTH_REQ_Q     *aaq;
	AATV           *paatv;
	CLIENT_ENTRY   *ce;
	char           *func = "rad_rq_resp_action";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	authreq->fsmstatus = EV_ACK;

	aaq = queue_find (PW_RESOURCE_QUERY_REQ);

	for (auth = aaq->q; auth != (AUTH_REQ *) NULL;
		auth = auth->next)
	{
		if (auth->code != PW_RESOURCE_QUERY_REQ)
		{
			continue;
		}

		if (auth->ipaddr == authreq->ipaddr &&
			auth->rep_id == authreq->rep_id)
		{
			break; /* Found original Resource Query Request. */
		}
	}

	if (auth == (AUTH_REQ *) NULL)
	{
		/* Reply does not match any request.  Silently drop it */
		return EV_NAK;
	}

	if (auth->type == USR_PROXY || auth->type == (USR_PROXY | USR_CHILD))
	{
		authreq->ipaddr = auth->nas_ip;
		authreq->rep_id = auth->fwd_id;

		paatv = find_aatv ("REPLY");
		call_action (paatv, authreq, 0, "");
		free_authreq (auth);
		return EV_DONE;
	}

	if (get_vp (authreq->request, PW_NAS_IP_ADDRESS) == (VALUE_PAIR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Missing NAS IP Address", func);
		return EV_NAK;
	}

	if (get_vp_vend (authreq->request, PW_USR_RQ_INDEX, VC_USR) == NULL_VP)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Missing Resource Query Index", func);
		return EV_NAK;
	}

	if (find_client (authreq->ipaddr, &ce) != 0)
	{
		return EV_NAK;
	}

/* XXX Moved this code from find_client () - It didn't seem to get used XXX
	if ((ce->client_type & CE_DAS ) != CE_DAS)
	{
		logit (LOG_DAEMON, LOG_ERR,
 "%s: Received a Resource Query Request from a RADIUS server that is not a DAS",
			func);
		return (EV_NAK);
	}
*/

	/*
	 *	Call add_resources() only if there are a/v pairs
	 *	other than the NAS-IP-Address and the Index.
	 */
	if ((authreq->cur_count > 2) &&
		add_resources (authreq->request,
				authreq->client->file_pfx, ce) == EV_NAK)
	{
		return EV_NAK;
	}

	if ((ce->client_type & CE_NAS) == CE_NAS)
	{
		return nas_proc (auth, authreq);
	}
	else if ((ce->client_type & CE_PROXY) == CE_PROXY)
	{
		return proxy_proc (auth, authreq);
	}
	else
	{
		logit (LOG_DAEMON, LOG_ERR,
"%s: Received Resource Query Response from a host that is not a NAS or a Proxy",
		       func);
		return EV_NAK;
	}
} /* end of rad_rq_resp_action () */

/***************************************************************************
 *
 *	Function: add_resources
 *
 *	Purpose: Creates resource records and places them on the appropriate
 *		 resource queue
 *
 **************************************************************************/

static int
add_resources (vp_list, file_pfx, ce)

VALUE_PAIR     *vp_list;
char           *file_pfx;
CLIENT_ENTRY   *ce;

{
	char            default_pool;
	UINT4           ipaddr = 0;
	UINT4           nas;
	u_int		length;
	int             nas_port = 0;
	int             result = EV_ACK;
	char           *name = (char *) NULL;
	char           *packet;
	VALUE_PAIR     *list;
	VALUE_PAIR     *q;
	VALUE_PAIR     *vp;
	USER_ENTRY     *user = (USER_ENTRY *) NULL;
	ADDR_POOL      *apool = (ADDR_POOL *) NULL;
	ASSIGNED_IP   **a;
	ASSIGNED_IP   **rem;
	ASSIGNED_IP    *temp;
	FILE_LIST      *file_ent;
	struct in_addr  add;
	char           *func = "add_resources";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((q = get_vp (vp_list, PW_NAS_IP_ADDRESS)) == (VALUE_PAIR *) NULL)
	{
		return EV_NAK;
	}

	nas = q->lvalue;

	if ((file_ent = find_file_ent (file_pfx)) == (FILE_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: NULL file entry", func);
		return EV_NAK;
	}

	if (file_ent->user_list == (USER_ENTRY *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: No users defined in %s.users", func, file_pfx);
		/*
		 *	This is okay because if no users are defined,
		 *	then no users can be authenticated.
		 */
		return EV_ACK;
	}

	q = vp_list;

	for ( ; q != (VALUE_PAIR *) NULL ; q = q->next)
	{
		if (q->attribute != PW_USR_PACKET ||
			q->ap->vendor_id != VC_USR)
		{
			continue;
		}

		if ((packet = (char *) malloc (q->lvalue)) == (char *) NULL)
		{
			logit (LOG_DAEMON, LOG_INFO,
				"%s: Malloc Failed\n", func);
			return EV_NAK;
		}

		memcpy (packet, q->strvalue, q->lvalue);
		length = q->lvalue;
		list = gen_valpairs ((AUTH_HDR *) packet, length,
							ce->veps, GVP_DROP);

		if ((vp = get_vp (list, PW_USER_NAME)) != (VALUE_PAIR *) NULL)
		{
			if ((name = strdup (vp->strvalue)) == (char *) NULL)
			{
				free (packet);
				list_free (list);
				logit (LOG_DAEMON, LOG_INFO,
					"%s: Malloc Failed\n", func);
				return EV_NAK;
			}
		}
		else
		{
			free (packet);
			list_free (list);
			continue;
		}

		if ((vp = get_vp (list, PW_FRAMED_IP_ADDRESS))
							!= (VALUE_PAIR *) NULL)
		{
			ipaddr = vp->lvalue;
		}
		else
		{
			/*
			 *	Ok to not find a Framed IP Address.
			 *	We could be managing Sessions-Allowed.
			 */
		}

		if ((vp = get_vp (list, PW_NAS_PORT)) != (VALUE_PAIR *) NULL)
		{
			nas_port = vp->lvalue;
		}
		else
		{
			free (packet);
			list_free (list);
			continue;
		}

		free (packet);
		list_free (list);

		for (user = file_ent->user_list; user != (USER_ENTRY *) NULL;
				user = user->next)
		{
			if (strcmp (user->name, name) == 0)
			{
				break;
			}
		}

		if (user == (USER_ENTRY *) NULL)
		{
			logit (LOG_DAEMON, LOG_INFO,
			       "%s: User %s does not exist", func, name);
			free (name);
			continue;
		}

		free (name); /* Don't need this anymore */

		apool = file_ent->pool_list;
		default_pool = 0;
		if (strcmp (user->pool_name, DEF_POOL_NAME) != 0)
		{
			apool = apool->next;  /* First entry is Default pool */
			for (; apool != (ADDR_POOL *) NULL; apool = apool->next)
			{
				if ((ipaddr & apool->netmask) == apool->network)
				{
					break;
				}
			}
		}
		else
		{
			default_pool = 1;
		}

		if (apool == (ADDR_POOL *) NULL)
		{
			add.s_addr = ipaddr;
			logit (LOG_DAEMON, LOG_INFO,
			       "%s: Couldn't find pool for %s",
			       func, inet_ntoa (add));
			continue;
		}
		if (!default_pool && apool->count >= apool->range)
		{
			add.s_addr = apool->ip_address;
			logit (LOG_DAEMON, LOG_INFO,
		  "%s: Address Pool %s full or this resource already allocated",
				func, inet_ntoa (add));
			continue;
		}

		if (user->count >= user->sessions)
		{
			logit (LOG_DAEMON, LOG_INFO,
			    "%s: User %s cannot be assigned any more addresses",
			       func, user->name);
			continue;
		}
		user->count++;

		/* Find a place in the address pool to insert new node */
		a = &apool->user_q;
		rem = a;
		if (!default_pool && *a != (ASSIGNED_IP *) NULL)
		{
			for ( ; *a != (ASSIGNED_IP *) NULL ; a = &((*a)->next))
			{
				if ((*a)->ip_address == ipaddr)
				{
					add.s_addr = ipaddr;
					logit (LOG_DAEMON, LOG_INFO,
					      "%s: Address %s already assigned",
					       func, inet_ntoa (add));
					result = EV_NAK;
					break;
				}
				if ((*a)->ip_address < ipaddr)
				{
					rem = &(*a)->next;
				}
			}
		}

		if (result == EV_NAK)
		{
			result = EV_ACK;
			user->count--;
			continue;
		}
		temp = make_ip_node (ipaddr, nas, user, nas_port);
		temp->next = *rem;
		*rem = temp;
		if (!default_pool)
		{
			apool->count++;
		}
	}

	return EV_ACK;
} /* end of add_resources () */

/**************************************************************************
 *
 *	Function: make_ip_node
 *
 *	Purpose: Makes up an ASSIGNED_IP node, fills in the given IP address,
 *		 NAS IP address and the pointer to the user.
 *
 *************************************************************************/

static ASSIGNED_IP *
make_ip_node (ipaddr, nas, user, nas_port)

UINT4           ipaddr;
UINT4           nas;
USER_ENTRY     *user;
int             nas_port;

{
	ASSIGNED_IP    *ip;
	char           *func = "make_ip_node";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((ip = (ASSIGNED_IP *) malloc (sizeof (ASSIGNED_IP)))
							== (ASSIGNED_IP *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: FATAL out of memory", func);
		abort ();
	}

	ip->user_ent = user;
	ip->ip_address = ipaddr;
	ip->nas_ip = nas;
	ip->nas_port = nas_port;
	ip->next = (ASSIGNED_IP *) NULL;

	return ip;
} /* end of make_ip_node */

/*****************************************************************************
 *
 *	Function: nas_proc
 *
 *	Purpose: Process a resource query response from a NAS
 *
 ****************************************************************************/

static char
nas_proc (request, reply)

AUTH_REQ       *request;
AUTH_REQ       *reply;

{
	VALUE_PAIR     *old_index;
	VALUE_PAIR     *nas;
	UINT4           index;
	int             result;
	AATV           *paatv;
/* 	struct in_addr  addr; */
	char           *func = "nas_proc";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	old_index = get_vp_vend (reply->request, PW_USR_RQ_INDEX, VC_USR);

	nas = get_vp (reply->request, PW_NAS_IP_ADDRESS);

/*	if (old_index->lvalue < VENDOR_ID_LENGTH + SUB_ID_LENGTH + INT_LEN)
	{
		addr.s_addr = ntohl(nas->lvalue);
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Invalid Resource Query Index in reply from %s",
		       func, inet_ntoa (addr));
		return EV_NAK;
	}
	memcpy ((char *) &index,
		&old_index->strvalue[VENDOR_ID_LENGTH + SUB_ID_LENGTH],
		INT_LEN);
	index = ntohl(index);
*/
	index = old_index->lvalue;

	if (index == 0)
	{
		if (set_client (reply->ipaddr, GOT_RQ_RESP) == EV_NAK)
		{
			result = EV_NAK;
		}
		else
		{
			result = EV_DONE;
		}
	}
	else
	{
		index++;
		list_free (reply->request);
		list_free (reply->cur_request);
		reply->request = reply->cur_request = (VALUE_PAIR *) NULL;
		reply->cur_count = 2;
		index = htonl(index);	/* to prepare for avpair_add_vend() */
		avpair_add_vend (&reply->request, PW_USR_RQ_INDEX, &index,
				 INT_LEN, VC_USR);
		avpair_add (&reply->request, PW_NAS_IP_ADDRESS,
			    &reply->ipaddr, 0);
		list_copy (&reply->cur_request, reply->request);

		if (!add_local_vpn (reply, reply->client->file_pfx))
		{
			return EV_NAK;
		}
		reply->code = PW_RESOURCE_QUERY_REQ;
		reply->rep_id = new_id (global_auth_q.q);
		reply->ttl = RQUERY_TIMER;
		reply->retry_cnt = 0;
		paatv = find_aatv ("REPLY");
		call_action (paatv, reply, 0, "");
		result = EV_ACK;
	}
	free_authreq (request);
	return result;
} /* end of nas_proc () */

/******************************************************************************
 *
 *	Function: proxy_proc
 *
 *	Purpose: Process the resource query responses got from the proxy server
 *
 *****************************************************************************/

static char
proxy_proc (request, reply)

AUTH_REQ       *request;
AUTH_REQ       *reply;

{
	UINT4           index;
	UINT4           old_index;
	VALUE_PAIR     *index_vp;
	VALUE_PAIR     *nas;
	VALUE_PAIR     *old_index_vp;
	AUTH_REQ       *auth = (AUTH_REQ *) NULL;
	AUTH_REQ_Q     *aaq = queue_find (PW_RESOURCE_QUERY_REQ);


	AATV           *paatv;
/* 	struct in_addr  addr; */
	char           *func = "proxy_proc";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	index_vp = get_vp_vend (reply->request, PW_USR_RQ_INDEX, VC_USR);
/*	if (index_vp->lvalue < VENDOR_ID_LENGTH + SUB_ID_LENGTH + INT_LEN)
	{
		addr.s_addr = ntohl(nas->lvalue);
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Invalid Resource Query Index in reply from %s",
		       func, inet_ntoa (addr));
		return EV_NAK;
	}
	memcpy ((char *) &index,
		&index_vp->strvalue[VENDOR_ID_LENGTH + SUB_ID_LENGTH],
		INT_LEN);
	index = ntohl(index);
*/
	index = index_vp->lvalue;

	nas = get_vp (reply->request, PW_NAS_IP_ADDRESS);
	reply->nas_ip = nas->lvalue;
	old_index_vp = get_vp_vend (request->request, PW_USR_RQ_INDEX, VC_USR);
/*	memcpy ((char *) &old_index,
		&old_index_vp->strvalue[VENDOR_ID_LENGTH + SUB_ID_LENGTH],
		INT_LEN);
	old_index = ntohl(old_index);
*/
	old_index = old_index_vp->lvalue;

	if (old_index != 0)
	{
		for (auth = aaq->q; auth != (AUTH_REQ *) NULL;
			auth = auth->next)
		{
			if (auth->code != PW_RESOURCE_QUERY_REQ)
			{
				continue;
			}

			if (auth->ipaddr == reply->ipaddr && auth->nas_ip == 0)
			{
				break; /* Found first Resource Query Request. */
			}
		}
		if (auth == (AUTH_REQ *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
			      "%s: No matching original Resource Query Request",
			       func);
			return EV_NAK;
		}
	}
	else
	{
		auth = request;
		auth->qry_count++;
	}

	auth->ttl = RQUERY_TIMER;

	if (old_index != 0)
	{
		free_authreq (request);
	}

	if (index == 0)
	{
		if (set_client (reply->ipaddr, GOT_RQ_RESP) == EV_NAK)
		{
			return EV_NAK;
		}
		return EV_DONE;
	}

	index++;
	list_free (reply->request);
	list_free (reply->cur_request);
	reply->request = (VALUE_PAIR *) NULL;
	reply->cur_request = (VALUE_PAIR *) NULL;
	index = htonl(index);	/* to prepare for avpair_add_vend() call */
	avpair_add_vend (&reply->request, PW_USR_RQ_INDEX,
			 &index, INT_LEN, VC_USR);
	avpair_add (&reply->request, PW_NAS_IP_ADDRESS, &reply->nas_ip, 0);
	list_copy (&reply->cur_request, reply->request);
	reply->rep_id = new_id (aaq->q);
	reply->code = PW_RESOURCE_QUERY_REQ;
	reply->ttl = RQUERY_TIMER;
	reply->retry_cnt = 0;
	paatv = find_aatv ("REPLY");
	call_action (paatv, reply, 0, "");
	return EV_ACK;
} /* end of proxy_proc () */

/*****************************************************************************
 *
 *	Function: new_id
 *
 *	Purpose: ??? XXX ???
 *
 ****************************************************************************/

u_char
new_id (queue)

AUTH_REQ       *queue;

{
	u_char          auth_ident = 0;
	AUTH_REQ       *entry;

	do
	{
		entry = queue;
		auth_ident++;
		for (; entry != (AUTH_REQ *) NULL; entry = entry->next)
		{
			if (entry->rep_id == auth_ident)
			{
				break;
			}
		}
	} while (entry != (AUTH_REQ *) NULL);

	return auth_ident;
} /* end of new_id () */

/*****************************************************************************
 *
 *	Function: set_client
 *
 *	Purpose:  ??? XXX ???
 *
 ****************************************************************************/

int
set_client (ipaddr, flag)

UINT4           ipaddr;
int             flag;

{
	CLIENT_ENTRY   *client_ent;
	CLIENT_ENTRY   *client_list;
	IP_ADDRESS     *addr_ent;
	struct in_addr  addr;
	static char    *func = "set_client";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	client_list = get_client_list ();
	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_ent->next)
	{
		if (ipaddr == get_ipaddr (client_ent->hostname))
		{
			break;
		}

		for (addr_ent = client_ent->addrs;
			addr_ent != (IP_ADDRESS *) NULL;
			addr_ent = addr_ent->next)
		{
			if (ipaddr == addr_ent->ipaddr.s_addr)
			{
				break;
			}
		}

		if (addr_ent != (IP_ADDRESS *) NULL)
		{
			break;
		}
	}

	if (client_ent == (CLIENT_ENTRY *) NULL)
	{
		addr.s_addr = ntohl(ipaddr);
		logit (LOG_DAEMON, LOG_ERR, "%s: Unknown Client %s",
			func, inet_ntoa (addr));
		return EV_NAK;
	}
	if (flag == TEST_GOT)
	{
		if (client_ent->state != GOT_RQ_RESP)
		{
			client_ent->state = NO_RQ_RESP;
		}
	}
	else
	{
		client_ent->state = flag;
	}

	return EV_ACK;
} /* end of set_client () */

/*****************************************************************************
 *
 *	Function: send_rq_req
 *
 *	Purpose:  Makes up, and sends out Resource Query Requests
 *
 ****************************************************************************/

void
send_rq_req (client_ent)

CLIENT_ENTRY   *client_ent;

{
	UINT4           index = 0;
	AUTH_REQ       *auth_ent;
	AUTH_REQ_Q     *aaq;
	AATV           *paatv;
	char           *func = "send_rq_req";

	dprintf (2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	client_ent->state = QUERY_IN_PROGRESS;

	if ((auth_ent = (AUTH_REQ *) calloc (1, sizeof (AUTH_REQ)))
							== (AUTH_REQ *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT, "%s: FATAL out of memory", func);
		abort ();
	}
	if (client_ent->addrs != (IP_ADDRESS *) NULL)
	{
		auth_ent->ipaddr = client_ent->addrs->ipaddr.s_addr;
	}
	else
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Could not resolve IP Address for %s",
		       func, client_ent->hostname);
		return;
	}

	auth_ent->udp_port = auth_port;
	auth_ent->code = PW_RESOURCE_QUERY_REQ;
	auth_ent->client = client_ent;
	auth_ent->ttl = RQUERY_TIMER;
	auth_ent->type = USR_PARENT;
	auth_ent->state = ST_INIT;
	auth_ent->fsmstatus = EV_ACK;
	auth_ent->repstatus = -2;	/* Initial, invalid value */
	auth_ent->cur_count = 1;

	/* Add an Index value of Zero */
	if (avpair_add_vend (&auth_ent->cur_request, PW_USR_RQ_INDEX, &index,
			     INT_LEN, VC_USR) == NULL_VP)
	{
		logit (LOG_DAEMON, LOG_ERR,
		   "%s: Problem adding Index AV Pair to Resource Query Request",
			func);
		free_authreq (auth_ent);
		return;
	}

	if ((client_ent->client_type & CE_NAS) == CE_NAS)
	{
		if (!add_local_vpn (auth_ent, client_ent->file_pfx))
		{
			free_authreq (auth_ent);
			return;
		}
	}

	aaq = queue_find (auth_ent->code);
	if (enqueue_authreq (aaq, auth_ent) == (AUTH_REQ *) NULL)
	{
		free_authreq (auth_ent);
		return;
	}

	auth_ent->rep_id = new_id (aaq->q);
	auth_ent->fwd_id = auth_ent->rep_id;
	memset (auth_ent->repvec, '\0', AUTH_VECTOR_LEN);
	memset (auth_ent->fwdvec, '\0', AUTH_VECTOR_LEN);

	paatv = find_aatv ("REPLY");
	auth_ent->fsm_aatv = paatv;
	call_action (paatv, auth_ent, 0, "");
	return;
} /* end of send_rq_req () */

/*****************************************************************************
 *
 *	Function: add_local_vpn
 *
 *	Purpose:  ??? XXX ???
 *
 ****************************************************************************/

static int
add_local_vpn (authreq, file_pfx)

AUTH_REQ       *authreq;
char           *file_pfx;

{
	int             count = 0;
	UINT4           vpn_id = 0;
	FILE_LIST      *file_ent;
	VALUE_PAIR     *vpn_list = (VALUE_PAIR *) NULL;
	AUTH_ENTRY     *pauth;
	char           *func = "add_local_vpn";

	if ((file_ent = find_file_ent (file_pfx)) == (FILE_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Couldn't find authfile with prefix %s",
		       func, file_pfx);
		return 0;
	}

	/* Add VPN 0 */
	if (avpair_add_vend (&vpn_list, PW_USR_VPN_ID, &vpn_id, INT_LEN,
			     VC_USR) == NULL_VP)
	{
		logit (LOG_DAEMON, LOG_ERR,
		  "%s: Problem adding VPN ID AV Pair to Resource Query Request",
		       func);
		return 0;
	}
	count++;
	/* Add all locally defined VPNs */
	for (pauth = file_ent->auth_list;
			pauth != (AUTH_ENTRY *) NULL;
			pauth = pauth->next)
	{
		if (pauth->type != AA_LOCAL_VPN)
		{
			continue;
		}
		vpn_id = htonl(pauth->vpn->id); /* to prepare for call below */
		if (avpair_add_vend (&vpn_list, PW_USR_VPN_ID,
				     &vpn_id, INT_LEN, VC_USR) == NULL_VP)
		{
			logit (LOG_DAEMON, LOG_ERR,
		  "%s: Problem adding VPN ID AV Pair to Resource Query Request",
				func);
			return 0;
		}
		count++;
	}

	list_cat (&authreq->request, vpn_list);
	list_cat (&authreq->cur_request, vpn_list);
	return count;
} /* end of add_local_vpn () */

#endif	/* USR_CCA */
