#ifdef USR_CCA

/*
 *
 * Copyright (c) 1996 U.S. Robotics, Access Corp.
 * All rights reserved.
 *
 * Permission to copy, display, distribute and make derivative works
 * from this material in whole or in part for any purpose is granted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all copies.  THIS SOFTWARE IS PROVIDED "AS IS" AND
 * WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES INCLUDING, WITHOUT
 * LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * If providing code not subject to a copyright please indicate that the
 * code has been dedicated to the public.
 *
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

static char     rcsid[] = "$Id: resources.c,v 1.6 1998/07/06 17:43:33 web Exp $";

/******************************************************************************
 *
 *			resources.c
 *
 *	This file contains the functions required for managing resources
 *
 *****************************************************************************/

#include        <sys/param.h>
#include        <sys/types.h>
#include        <sys/socket.h>
#include        <netinet/in.h>
#include        <arpa/inet.h>
#include	<sys/ioctl.h>
#include	<sys/file.h>
#include        <sys/time.h>
#include        <sys/file.h>
#include        <sys/wait.h>
#include        <sys/stat.h>
#include        <net/if.h>

#include        <stdio.h>
#include        <stdlib.h>
#include        <netdb.h>
#include        <fcntl.h>
#include        <errno.h>
#include        <signal.h>
#include        <syslog.h>

#include        "radius.h"

extern int      debug_flag;

static void     put_in_pool PROTO((ADDR_POOL *, ADDR_POOL **));
static int      range_ok PROTO((UINT4, UINT4, int));
static char     pool_overlap PROTO((ADDR_POOL **, char *, UINT4));

/*****************************************************************************
 *
 *	Function: add_to_addr_pool
 *
 *	Purpose: Parse the given string and get the base IP address and range
 *		 for this address pool.  Then link this pool into the global
 *		 list of address pools by calling put_in_pool().
 *
 *****************************************************************************/

int
add_to_addr_pool (pool, string, line_nbr)

ADDR_POOL     **pool;
char           *string;
int             line_nbr;	/* Current users file line */

{
	char           *store;
	ADDR_POOL      *addrptr;
	char           *temp;
	UINT4           first_host;
	char            error = 0;
	char           *name;
	static char    *func = "add_to_addr_pool";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((name = strtok (string, " \t\n\r")) == NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		  "%s: users file, line %d - Invalid Assigned IP Address Pool.",
		       func, line_nbr);
		return (-1);
	}

	if ((store = strtok (NULL, " \t\n\r")) == NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
    "%s: users file, line %d - Assigned IP Address Pool - Missing base address",
			func, line_nbr);
		return (-1);
	}

	if (good_ipaddr (store) != 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
	"%s: users file, line %d - Assigned IP Address Pool - Bad base address",
			func, line_nbr);
		return (-1);
	}

	if (strcmp (store, "0.0.0.0") == 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
 "%s: users file, line %d - Assigned IP Address Pool - Base address is 0.0.0.0",
			func, line_nbr);
		return (-1);
	}

	/* Base address is okay. */
	if (pool_overlap (pool, name, ntohl(inet_addr (store))))
	{
		logit (LOG_DAEMON, LOG_ALERT,
"%s: users file, line %d - Address Pool - Name repeated, or base address overlaps with another pool",
		       func, line_nbr);
		return (-1);
	}

	if ((addrptr = (ADDR_POOL *) malloc (sizeof (ADDR_POOL))) == NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
		"%s: users file, line %d - Couldn't allocate ADDR_POOL storage",
			func, line_nbr);
		abort ();
		return (-1);
	}

	addrptr->name = add_string (name, ASIS);
	addrptr->ip_address = ntohl(inet_addr (store));
	addrptr->count = 0;
	addrptr->user_q = (ASSIGNED_IP *) NULL;
	addrptr->next = (ADDR_POOL *) NULL;

	if ((store = strtok (NULL, " \t\n\r")) == NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
"%s: users file, line %d - Missing Assigned IP Address Pool netmask and range.",
			func, line_nbr);
		free (addrptr);
		return (-1);
	}

	if (good_ipaddr (store) != 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
	  "%s: users file, line %d - Invalid Assigned IP Address Pool netmask.",
			func, line_nbr);
		free (addrptr);
		return (-1);
	}

	addrptr->netmask = ntohl(inet_addr (store));
	addrptr->network = addrptr->ip_address & addrptr->netmask;

	if ((store = strtok (NULL, " \t\n\r")) == NULL)
	{
		/*
		 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line %d -
		 * Missing Assigned IP Address Pool range.", func, line_nbr);
		 */
		free (addrptr);
		return (-1);
	}

	for (temp = store; *temp != '\0'; temp++)
	{
		if (!isdigit(*temp))
		{
			error = 1;
			break;
		}
	}

	if (error == 1 || ((addrptr->range = atoi (store)) <= 0))
	{

		/*
		 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line %d -
		 * Invalid Assigned IP Address Pool range.", func, line_nbr);
		 */
		free (addrptr);
		return (-1);
	}

/*	Now check the supplied address, netmask, range, etc. */
	if (addrptr->netmask == 0xffffffff)
	{
		if (IN_CLASSA (addrptr->ip_address) &&
		  (((first_host = addrptr->ip_address & IN_CLASSA_HOST) == 0) ||
			first_host == IN_CLASSA_HOST))
		{
			/*
			 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line
			 * %d - Class A address with host part all zeros or
			 * broadcast", func, line_nbr);
			 */
			free (addrptr);
			return (-1);
		}

		if (IN_CLASSB (addrptr->ip_address) &&
		  (((first_host = addrptr->ip_address & IN_CLASSB_HOST) == 0) ||
			first_host == IN_CLASSB_HOST))
		{
			/*
			 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line
			 * %d - Class B address with host part all zeros or
			 * broadcast", func, line_nbr);
			 */
			free (addrptr);
			return (-1);
		}

		if (IN_CLASSC (addrptr->ip_address) &&
		  (((first_host = addrptr->ip_address & IN_CLASSC_HOST) == 0) ||
			first_host == IN_CLASSC_HOST))
		{
			/*
			 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line
			 * %d - Class C address with host part all zeros or
			 * broadcast", func, line_nbr);
			 */
			free (addrptr);
			return (-1);
		}

		if (IN_CLASSD (addrptr->ip_address) ||
				IN_EXPERIMENTAL (addrptr->ip_address))
		{
			/*
			 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line
			 * %d - Class D or Class E address specified", func,
			 * line_nbr);
			 */
			free (addrptr);
			return (-1);
		}

	}

	/* Checking if host part is zero. */
	if (((addrptr->ip_address & (~addrptr->netmask)) == 0) &&
			(addrptr->netmask != 0xffffffff))
	{
		/*
		 * Host part of base address is zero and the base address
		 * is not host specific; set host part to one.
		 */
		addrptr->ip_address += 1;
	}

	if (!range_ok (addrptr->ip_address, addrptr->netmask, addrptr->range))
	{
		/*
		 * logit (LOG_DAEMON, LOG_ERR, "%s: users file, line %d -
		 * Range too large for given base IP address.", func,
		 * line_nbr);
		 */
		free (addrptr);
		return (-1);
	}

	put_in_pool (addrptr, pool);

	return 0;
} /* end of add_to_addr_pool () */

/*****************************************************************************
 *
 *	Function: put_in_pool
 *
 *	Purpose: Attach the given address pool to the global list of address
 *		 pools.
 *
 *****************************************************************************/

static void
put_in_pool (addrptr, pool)

ADDR_POOL      *addrptr;
ADDR_POOL     **pool;		/* pool to put this address pool in */

{

	ADDR_POOL      *aptr;
	static char    *func = "put_in_pool";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (*pool == (ADDR_POOL *) NULL)
	{
		*pool = addrptr;
	}
	else
	{
		for (aptr = *pool; aptr->next; aptr = aptr->next)
		{
			; /* continue */
		}

		aptr->next = addrptr;
	}
} /* end of put_in_pool () */

/*************************************************************************
 *
 *	Function: free_pool_ent
 *
 *	Purpose: Free all components of an ADDR_POOL structure.  Delete
 *		 all the ASSIGNED_IP structures hanging from the ADDR_POOL
 *		 structure.
 *
 ************************************************************************/

void
free_pool_ent (pool_ent)

ADDR_POOL      *pool_ent;

{
	ASSIGNED_IP    *assign_ent;

	if (pool_ent->user_q == (ASSIGNED_IP *) NULL)
	{
		free (pool_ent);
		return;
	}

	for (assign_ent = pool_ent->user_q; assign_ent;
			assign_ent = pool_ent->user_q)
	{
		pool_ent->user_q = assign_ent->next;
		free (assign_ent);
	}
	return;
} /* end of free_pool_ent () */

/*****************************************************************************
 *
 *	Function: range_ok
 *
 *	Purpose: This function checks whether the specified range is too large
 *		 for the given base IP address
 *
 *****************************************************************************/

static int
range_ok (ip_address, netmask, range)

UINT4           ip_address;
UINT4           netmask;
int             range;

{
	UINT4           first_host;
	UINT4           broadcast;
	static char    *func = "range_ok";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));
	if (netmask == 0xffffffff)
	{
		if (range != 1)
		{
			return (0);
		}
		return (1);
	}

	first_host = ip_address & (~netmask);
	broadcast = ~netmask;

	if ((first_host + range - 1) >= broadcast)
	{
		return (0);
	}

	return (1);
} /* end of range_ok () */

/******************************************************************************
 *
 *	Function: pool_overlap
 *
 *	Purpose: Checks if the given IP address lies within one of the pools
 *               defined in the given list of pools.
 *
 *****************************************************************************/

static char
pool_overlap (pool, name, ipaddr)

ADDR_POOL     **pool;
char           *name;
UINT4           ipaddr;

{
	ADDR_POOL      *pptr;

	for (pptr = *pool; pptr; pptr = pptr->next)
	{
		if (strcmp (name, pptr->name) == 0)
		{
			return (1);
		}

		if (ipaddr >= pptr->ip_address &&
			ipaddr < (pptr->ip_address + pptr->range))
		{
			return (1);
		}
	}

	return (0);
} /* end of pool_overlap () */

/******************************************************************************
 *
 *	Function: resource_mgmt
 *
 *	Purpose: This function assigns an IP address from one of the address
 *		 pools defined in the users file.  It also keeps track of all
 *		 the addresses that have been assigned and of all the users
 *		 who have more than one session allowed at a time.
 *
 *****************************************************************************/

int
resource_mgmt (authreq)

AUTH_REQ       *authreq;

{
	VALUE_PAIR     *user_name;
	VALUE_PAIR     *nas_addr;
	VALUE_PAIR     *nas_port;
	VALUE_PAIR     *framed_ip;
	USER_ENTRY     *user_ent;
	ADDR_POOL      *aptr;
	ASSIGNED_IP    *p;
	ASSIGNED_IP    *q;
	UINT4           new_address = 1;
	ASSIGNED_IP    *new_node;
	char            head = 0;
	ASSIGNED_IP   **ipptr;
	FILE_LIST      *file_ent;
	static char    *func = "resource_mgmt";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((user_name = get_vp (authreq->request, PW_USER_NAME))
			== (VALUE_PAIR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		  "%s: Missing User Name; resource management not possible",
		       func);
		return EV_NAK;
	}

	if ((nas_addr = get_vp (authreq->request, PW_NAS_IP_ADDRESS))
			== (VALUE_PAIR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		"%s: Missing NAS IP address, cannot assign %s an IP address",
		       func, user_name->strvalue);
		return EV_NAK;
	}

	if ((nas_port = get_vp (authreq->request, PW_NAS_PORT))
			== (VALUE_PAIR *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		     "%s: Missing NAS Port, cannot assign %s an IP address",
		       func, user_name->strvalue);
		return EV_NAK;
	}

	if ((file_ent = find_file_ent (authreq->client->file_pfx)) ==
			(FILE_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: NULL file_ent", func);
		return EV_NAK;
	}

	for (user_ent = file_ent->user_list;
			user_ent; user_ent = user_ent->next)
	{
		if (strcmp (user_ent->name, user_name->strvalue) == 0)
		{
			break; /* Found user. */
		}
	}

	if (user_ent == (USER_ENTRY *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
		       "%s: Invalid user %s", func, user_name);
		return EV_NAK;
	}

	if (user_ent->count >= user_ent->sessions)
	{
		logit (LOG_DAEMON, LOG_INFO,
		     "%s: User %s has maximum allowed sessions going", func,
		       user_name);
		return EV_NAK;
	}

	aptr = file_ent->pool_list;	/* The default pool */

/*	if (user_ent->pool_name[0] != '\0') */
	if (strcmp (user_ent->pool_name, DEF_POOL_NAME) != 0)
	{
		for (; aptr != (ADDR_POOL *) NULL; aptr = aptr->next)
		{
			if (strcmp (aptr->name, user_ent->pool_name) == 0)
			{
				break;
			}
		}

		if (aptr == (ADDR_POOL *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
			       "%s: Could not find an Address Pool for %s.",
			       func, user_name);
			return EV_NAK;
		}

		if (aptr->count >= aptr->range)
		{
			logit (LOG_DAEMON, LOG_ERR,
			 "%s: Address Pool is full; can't assign address to %s",
			       func, user_ent->name);
			return EV_NAK;
		}

		p = aptr->user_q;

		/* Processing address pool list. */
		if (p != (ASSIGNED_IP *) NULL)
		{
			if (p->ip_address != aptr->ip_address)
			{
/*				new_address = aptr->ip_address; */
				head = 1;
			}
			else
			{
				for (q = p->next; q; q = q->next)
				{
					if ((q->ip_address - p->ip_address) > 1)
					{
						break;
					}
					p = p->next;
				}
				new_address = p->ip_address + 1;
			}
		}
		else
		{
			head = 1;
/*			new_address = aptr->ip_address; */
		}

		new_node = (ASSIGNED_IP *) malloc (sizeof (ASSIGNED_IP));
		if (new_node == (ASSIGNED_IP *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT, "%s: OUT OF MEMORY",
			       func);
			return EV_NAK;
		}

		if (head == 1)
		{
			new_address = aptr->ip_address;
			new_node->next = aptr->user_q;
/*			new_node->next = (ASSIGNED_IP *) NULL; */
			aptr->user_q = new_node;
		}
		else
		{
			new_node->next = p->next;
			p->next = new_node;
		}
		new_node->ip_address = new_address;
		new_node->user_ent = user_ent;
		new_node->nas_ip = nas_addr->lvalue;
		new_node->nas_port = nas_port->lvalue;
		user_ent->count++;
		aptr->count++;

		if (avpair_add (&authreq->cur_request, PW_FRAMED_IP_ADDRESS,
				&new_address, 0) == (VALUE_PAIR *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
			 "%s: Problem adding Framed-IP-Address for user %s",
			       func, user_ent->name);
			return EV_NAK;
		}

		authreq->cur_count++;

		return EV_ACK;
	}
	else /* if (user_ent->sessions > 1) */

		/*
		 *	No Address Pool defined, but multiple sessions allowed.
		 *	Put session in the default pool.
		 */
	{
		for (ipptr = &aptr->user_q;
			*ipptr != (ASSIGNED_IP *) NULL;
			ipptr = &(*ipptr)->next)
		{
			; /* continue */
		}

		new_node = (ASSIGNED_IP *) malloc (sizeof (ASSIGNED_IP));
		if (new_node == (ASSIGNED_IP *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT, "%s: OUT OF MEMORY",
			       func);
			return EV_NAK;
		}

		new_node->user_ent = user_ent;
		if ((framed_ip = get_vp (authreq->cur_request,
				PW_FRAMED_IP_ADDRESS)) != (VALUE_PAIR *) NULL)
		{
			new_node->ip_address = framed_ip->lvalue;
		}
		else
		{
			new_node->ip_address = 0;
		}

		new_node->nas_ip = nas_addr->lvalue;
		new_node->nas_port = nas_port->lvalue;
		new_node->next = (ASSIGNED_IP *) NULL;
		*ipptr = new_node;
		user_ent->count++;
		aptr->count++;
		return EV_ACK;
	}
} /* end of resource_mgmt () */

/***************************************************************************
 *
 *	Function: free_resources
 *
 *	Purpose: Free the resource (given IP address) allocated to the
 *		 given user.
 *
 **************************************************************************/

int
free_resources (name, ipaddr, nas, nas_port)

char           *name;
UINT4           ipaddr;
UINT4           nas;
UINT4           nas_port;

{
	char            default_pool = FALSE;
	FILE_LIST      *file_ent;
	ADDR_POOL      *aptr;
	ASSIGNED_IP   **assign_prev;
	ASSIGNED_IP    *assign_cur;
	char           *func = "free_resources";

	dprintf(2, (LOG_DAEMON, LOG_DEBUG, "%s: entered", func));

	file_ent = get_default_file_entry ();

	if (nas == 0 || name == (char *) NULL || nas_port == 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
		    "%s: Insufficient information in resource free request",
		       func);
		return EV_NAK;
	}


	for (; file_ent != (FILE_LIST *) NULL; file_ent = file_ent->next)
	{
		for (aptr = file_ent->pool_list;
			aptr != (ADDR_POOL *) NULL;
			aptr = aptr->next)
		{
			if (strcmp (aptr->name, DEF_POOL_NAME) != 0)
			{
				if (aptr->count == 0)
				{
					continue;
				}

				if ((ipaddr & aptr->netmask) != aptr->network)
				{
					continue;
				}
			}
			else
			{
				default_pool = TRUE;
			}

			assign_prev = &aptr->user_q;
			assign_cur = *assign_prev;

			for ( ; assign_cur != (ASSIGNED_IP *) NULL;
				assign_cur = assign_cur->next)
			{
				if ((default_pool == TRUE ||
					    assign_cur->ip_address == ipaddr) &&
					assign_cur->nas_ip == nas &&
					assign_cur->nas_port == nas_port &&
					strcmp (name,
					       assign_cur->user_ent->name) == 0)
				{
					*assign_prev = assign_cur->next;
					(assign_cur->user_ent)->count--;
					aptr->count--;
					free (assign_cur);
					return EV_ACK;
				}
				assign_prev = &assign_cur->next;
			}
		}
	}
/*	logit (LOG_DAEMON, LOG_ERR, "%s: No such resource", func); */
	return EV_NAK;
} /* end of free_resources () */

#endif	/* USR_CCA */
