/*
 * RADIUS -- Remote Authentication Dial In User Service
 * 
 * 
 * Livingston Enterprises, Inc. 6920 Koll Center Parkway Pleasanton, CA   94566
 * 
 * Copyright 1992 Livingston Enterprises, Inc.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose and without fee is hereby granted, provided that this copyright
 * and permission notice appear on all copies and supporting documentation,
 * the name of Livingston Enterprises, Inc. not be used in advertising or
 * publicity pertaining to distribution of the program without specific
 * prior permission, and notice be given in supporting documentation that
 * copying and distribution is by permission of Livingston Enterprises, Inc.
 * 
 * Livingston Enterprises, Inc. makes no representations about the suitability
 * of this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 *
 *      Copyright (c) 1996 Ascend Communications, Inc.
 *      All rights reserved.
 *
 *      Permission to copy, display, distribute and make derivative works
 *      from this material in whole or in part for any purpose is granted
 *      provided that the above copyright notice and this paragraph are
 *      duplicated in all copies.  THIS SOFTWARE IS PROVIDED "AS IS" AND
 *      WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES INCLUDING, WITHOUT
 *      LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *      FOR A PARTICULAR PURPOSE.
 *
 */

/*
 * Copyright [C] The Regents of the University of Michigan and Merit Network,
 * Inc. 1992, 1993, 1994, 1995, 1996, 1997, 1998 All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation 
 * for any purpose and without fee is hereby granted, provided: 
 *
 * 1) that the above copyright notice and this permission notice appear in all
 *    copies of the software and derivative works or modified versions thereof, 
 *
 * 2) that both the copyright notice and this permission and disclaimer notice 
 *    appear in all supporting documentation, and 
 *
 * 3) that all derivative works made from this material are returned to the
 *    Regents of the University of Michigan and Merit Network, Inc. with
 *    permission to copy, to display, to distribute, and to make derivative
 *    works from the provided material in whole or in part for any purpose.
 *
 * Users of this code are requested to notify Merit Network, Inc. of such use
 * by sending email to aaa-admin@merit.edu
 *
 * Please also use aaa-admin@merit.edu to inform Merit Network, Inc of any
 * derivative works.
 *
 * Distribution of this software or derivative works or the associated
 * documentation is not allowed without an additional license.
 *
 * Licenses for other uses are available on an individually negotiated
 * basis.  Contact aaa-license@merit.edu for more information.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE.  The Regents of the
 * University of Michigan and Merit Network, Inc. shall not be liable for any
 * special, indirect, incidental or consequential damages with respect to any
 * claim by Licensee or any third party arising from use of the software.
 *
 * Merit AAA Server Support
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C.
 * Ann Arbor, Michigan, USA 48105-2785
 *
 * attn:  John Vollbrecht
 * voice: 734-764-9430
 * fax:   734-647-3185
 * email: aaa-admin@merit.edu
 *
 */

/*
 *
 * Public entry points in this file:
 * 
 * add_file_list
 * addr_is_us
 * change_pw
 * client_is_us
 * client_queue_size
 * config_files
 * config_fini
 * config_init
 * dns_recv
 * find_auth_type
 * find_auth_ent
 * find_client
 * find_client_by_name
 * find_file_ent [[if USR_CCA is defined]]
 * find_host_by_name
 * free_user_ent
 * get_client_list
 * get_default_file_entry
 * get_our_addr
 * host_is_us
 * ip_hostname
 * pair_parse
 * return_file_list
 * update_expire
 * user_find
 * user_gettime
 * user_update
 * 
 */

static char     sccsid[] =
		"@(#)users.c 1.3 Copyright 1992 Livingston Enterprises Inc";

static char     rcsid[] = "$Id: users.c,v 1.19 1998/07/07 16:52:14 web Exp $";

#include	<sys/types.h>
#include	<sys/param.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<memory.h>
#include	<unistd.h>
#include	<ctype.h>
#include	<dirent.h>
#include	<syslog.h>

#include	"radius.h"

static int         fieldcpy PROTO((char *, char **, int));

#ifdef USR_CCA
static char        check_user PROTO((USER_ENTRY *));
FILE_LIST         *find_file_ent PROTO((char *));
#else	/* USR_CCA */
static FILE_LIST  *find_file_ent PROTO((char *));
#endif	/* USR_CCA */

static void        free_clients PROTO((CLIENT_ENTRY *));
static void        free_file_lists PROTO((void));
static int         get_client_type PROTO((int, VENDOR_LIST **));

#ifdef USR_CCA
static char        parse_dns_nbns PROTO((char **, UINT4 *, UINT4 *));
#endif	/* USR_CCA */

static char       *parse_id PROTO((char *));
static int         read_auth PROTO((FILE_LIST *, int));
static int         read_clients PROTO((int));
static int         read_users PROTO((FILE_LIST *, int));

#if defined(ultrix) || defined(__hpux) || defined(__bsdi__) || defined(linux) || defined(SCO)
extern int         h_errno;
#endif	/* ultrix */

extern char        send_buffer[RAD_SEND_BUFFER_SIZE];
extern char        recv_buffer[RAD_RECV_BUFFER_SIZE];
extern char        ourhostname[MAXHOSTNAMELEN];
extern AATV       *authtype_tv[]; /* AATVs by authentication types */
extern int         debug_flag;
extern int         dumpcore;
extern char       *radius_dir;
extern FILE       *ddt;
extern int         authfile_cnt;
extern int         clients_cnt;
extern int         users_cnt;
extern int         file_logging; /* 0 => syslog, 1 => logfile, 2 => stderr */
extern AATV       *rad_ipc_aatv;
extern FILE       *msgfd;

int                default_reply_holdtime = CLEANUP_DELAY;
int                spawn_flag = 1; /* 0 => no spawning, 1 => spawning allowed */
int                dnspid = 0;     /* PID of current DNS resolver process */
UINT4              dns_address_aging = ADDRESS_AGING;
UINT4              dns_address_window = 60;	/* one minute */
int                doing_init = 1;	/* Always initing if not engine */
char               default_radius_server[128] = DEFAULT_RADIUS_SERVER;
char               default_tacacs_server[128] = DEFAULT_TACACS_SERVER;
char               authfile_id[128];
char               clients_id[128];
int                rad_ipc_port = 0;
MF_ENT             dns_addr_mf = { 0, 0 };	/* For DNS address structs. */
MF_ENT             dns_client_mf = { 0, 0 };	/* For DNS address structs. */
MF_ENT             dns_name_mf = { 0, 0 };	/* For DNS address structs. */
MF_ENT             vendor_mf = { 0, 0 };	/* For vendors. */
MF_ENT             vendor_list_mf = { 0, 0 };	/* For vendor lists. */

#ifdef USR_CCA
char               dns_done = FALSE;
UINT4              self_ip[SELF_IP_LEN]; /* Used with multi-homed servers */
#else	/* USR_CCA */
static UINT4       self_ip[11];		/* Used with multi-homed servers */
#endif	/* USR_CCA */

static FILE_LIST  *file_list = (FILE_LIST *) NULL;
static CLIENT_ENTRY *client_list = (CLIENT_ENTRY *) NULL;
static int         is_engine = 0;	/* rlmadmin(8) will not change this */
static CLIENT_ENTRY *old_clients;
static UINT4       last_client_ipaddr;
static CLIENT_ENTRY *last_client_found;
static char       *last_client_name;

/*************************************************************************
 *
 *	Function: add_file_list
 *
 *	Purpose: Find an existing FILE_LIST entry on file_list with the
 *		 specified prefix or add and init a new one if the
 *		 entry doesn't already exist.
 *
 *************************************************************************/

int
add_file_list (prefix)

char           *prefix;

{
	FILE_LIST      *file_ent;
	FILE_LIST     **fl_prev;

#ifdef USR_CCA
	ADDR_POOL      *apool;
#endif /* USR_CCA */

	static char    *func = "add_file_list";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	for (fl_prev = &file_list, file_ent = file_list;
		file_ent;
		fl_prev = &file_ent->next, file_ent = *fl_prev)
	{
		if (strcmp (file_ent->prefix, prefix) == 0)
		{
			return 0;
		}
	}

	if ((file_ent = (FILE_LIST *) calloc (1, sizeof (FILE_LIST)))
							== (FILE_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: Couldn't allocate FILE_ENTRY storage", func);
		return (-1);
	}
	file_ent->prefix = add_string (prefix, ASIS);
	file_ent->user_list = (USER_ENTRY *) NULL;
	file_ent->auth_list = (AUTH_ENTRY *) NULL;

#ifdef USR_CCA
	/*
	 *	Add an entry for the default pool. That is, the pool for
	 *	users who are statically assigned Framed IP addresses in
	 *	the users file.
	 */
	if ((file_ent->pool_list = (ADDR_POOL *) calloc (1, sizeof (ADDR_POOL)))
							== (ADDR_POOL *) NULL)
	{
		logit (LOG_DAEMON, LOG_ALERT,
			"%s: Couldn't allocate ADDR_POOL storage", func);
		return (-1);
	}
	apool = file_ent->pool_list;
	apool->name = DEF_POOL_NAME;
	apool->ip_address = 0;
	apool->netmask = 0;
	apool->network = 0;
	apool->range = 0;
	apool->count = 0;
	apool->user_q = (ASSIGNED_IP *) NULL;
	apool->next = (ADDR_POOL *) NULL;
#endif	/* USR_CCA */

	file_ent->next = (FILE_LIST *) NULL;
	*fl_prev = file_ent;

	return 0;
} /* end of add_file_list () */

/*************************************************************************
 *
 *	Function: addr_is_us
 *
 *	Purpose: Determine if we are the given host.
 *
 *	Returns: 1 if the given address is ours,
 *		 0 otherwise.
 *
 *************************************************************************/

int
addr_is_us (addr)

UINT4           addr;

{
	UINT4           *adptr;

	for (adptr = self_ip; *adptr > 0; adptr++)
	{
		if (*adptr == addr)
		{
			return 1;
		}
	}

	return 0;
} /* end of addr_is_us () */

/*************************************************************************
 *
 *	Function: change_pw
 *
 *	Purpose: Find the named user and change the password to the new value.
 *
 *	Remarks: This is called from passchange() only after the new
 *		 password has been validated.  The user was previously
 *		 found using user_find() so we know the user is in memory.
 *
 *	Returns: zero, if user's password has been changed in memory,
 *		 non-zero, otherwise.
 *
 *************************************************************************/

int
change_pw (name, protocol, curpass, newpass)

char           *name;
int             protocol;
VALUE_PAIR     *curpass;
char           *newpass;

{
	int             namelen;
	FILE_LIST      *file_ent;
	VALUE_PAIR     *dbpass;	
	VALUE_PAIR     *prot_ent;
	USER_ENTRY     *user_ent;
	USER_ENTRY     *dflt_ent;
	FILE           *debugout = stdout;
	static char    *func = "change_pw";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((file_ent = find_file_ent ((char *) NULL)) == (FILE_LIST *) NULL)
	{
		return (-1);
	}

	dflt_ent = file_list->user_list;

	/*
	 *	If there is no user file for this prefix, then use the
	 *	first entry in file_list as the default file_ent.
	 */
	if ((user_ent = file_ent->user_list) == (USER_ENTRY *) NULL)
	{
		if (dflt_ent == (USER_ENTRY *) NULL)
		{
			/* No file entry, no password */
			 return (-1);
		}

		user_ent = dflt_ent;
	}

	if ((namelen = strlen (name)) == 0)
	{
		return 1;	/* A null name would match every line. */
	}

	/*
	 *	First check what type of lookup to do.
	 *	See if user file(s) have been cached in memory.
	 */

	for (; user_ent; user_ent = user_ent->next)
	{
		/* Allow match for entry specifying framed protocol
		 * type specified in "protocol".  An entry with a 
		 * matching name but no framed-protocol type check 
		 * item matches unconditionally.  An entry with a 
		 * matching name and a framed-protocol type check 
		 * item must match value in "protocol".
		 */
		if (strcmp (user_ent->name, name) == 0)
		{
			if ((prot_ent = get_vp (user_ent->check,
						PW_FRAMED_PROTOCOL)) == NULL_VP)
			{
				break;
			}

			if (prot_ent->lvalue == protocol)
			{
				break;
			}
		}
	}

	if (!user_ent)	/* rc 1 => User not found */
	{
		return 1;
	}

	/* Find the password entry. */
	dbpass = get_vp_ci (user_ent->check, CI_USER_PASSWORD, 0);

	if ((dbpass == NULL_VP) || (dbpass->strvalue == (char *) NULL))
	{
		/* Missing our local copy of the password */
		logit (LOG_DAEMON, LOG_ERR,
			"%s: - Missing Local Password: %s", func);
		return 1;
	}

	/* Let's be paranoid about this */
	if (strcmp (dbpass->strvalue, curpass->strvalue))
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: - Password does not match cached password.", func);
		return 1;
	}

	/* Go ahead and change it */	
	avpair_string_mod (dbpass, newpass, -1);

	if (debug_flag >= 2)
	{
		if (ddt)
		{
			debugout = ddt;
		}
		fprintf (debugout, "Check items:\n");
		debug_list (debugout, user_ent->check);
		fprintf (debugout, "Reply items:\n");
		debug_list (debugout, user_ent->reply);
	}
	return (0);	/* rc 0 => User found */

} /* end of change_pw () */

#ifdef USR_CCA

/*************************************************************************
 *
 *	Function: check_user
 *
 *	Purpose: Processes Simultaneous-Use, Sessions-Allowed and
 *		 Address-Pool attributes, and checks for a valid
 *		 Termination-Action value.
 *
 ************************************************************************/

static char
check_user (user_ent)

USER_ENTRY      *user_ent;

{
	VALUE_PAIR      *vp;
	char            *func = "check_user";

	if ((vp = get_vp_ci (user_ent->reply, CI_SIMULTANEOUS_USE, 0))
								!= NULL_VP)
	{
		user_ent->sessions = vp->lvalue;
		avpair_del (&user_ent->reply, CI_SIMULTANEOUS_USE, 0);
	}

	if ((vp = get_vp_ci (user_ent->reply, CI_ADDRESS_POOL_NAME, 0))
								!= NULL_VP)
	{
		user_ent->pool_name = vp->strvalue;
		avpair_del (&user_ent->reply, CI_ADDRESS_POOL_NAME, 0);
		if ((vp = get_vp (user_ent->reply, PW_FRAMED_IP_ADDRESS))
								!= NULL_VP)
		{
			logit (LOG_DAEMON,LOG_ERR,
       "%s: User %s has a static Framed-IP-Address and an Address Pool defined",
				func, user_ent->name);
			return (-1);
		}
	}

	if (user_ent->sessions > 1 ||
		strcmp (user_ent->pool_name, DEF_POOL_NAME) != 0)
	{
		if (((vp = get_vp (user_ent->reply,
					PW_TERMINATION_ACTION)) == NULL_VP))
		{
			/*
			 *	This user's resources are to be managed.
			 *	A Termination-Action attribute is required.
			 */
			logit (LOG_DAEMON, LOG_ERR,
			   "%s: User %s-> Missing Termination Action attribute",
				func, user_ent->name);
			return (-1);
		}
		else if (vp->lvalue != MANAGE_RESOURCES)
		{
			logit (LOG_DAEMON, LOG_ERR,
			   "%s: User %s-> Invalid Termination Action Attribute",
				func, user_ent->name);
			return (-1);
		}
	}

	return (0);

} /* end of check_user () */

#endif /* USR_CCA */

/*************************************************************************
 *
 *	Function: client_is_us
 *
 *	Purpose: Determine if we are the given host.
 *
 *	Returns: 1, if the given hostname is the name of this host,
 *		 0, otherwise.
 *
 *************************************************************************/

int
client_is_us (ce)

CLIENT_ENTRY    *ce;

{
	UINT4            addr;
	UINT4           *adptr;
	IP_ADDRESS      *each_ip;

	if (ce != (CLIENT_ENTRY *) NULL)
	{
		for (each_ip = ce->addrs;
			each_ip != (IP_ADDRESS *) NULL;
			each_ip = each_ip->next)
		{
			addr = ntohl(each_ip->ipaddr.s_addr);

			for (adptr = self_ip; *adptr > 0; adptr++)
			{
				if (*adptr == addr)
				{
					return 1;
				}
			}
		}
	}

	return 0;
} /* end of client_is_us () */

/*************************************************************************
 *
 *	Function: client_queue_size
 *
 *	Purpose: Returns count of events queued up for the given client.
 *
 *	Returns: count of queued events,
 *		 0 otherwise.
 *
 *************************************************************************/

int
client_queue_size (cli)

CLIENT_ENTRY     *cli;

{
	int             i = 0;
	EVENT_ENT      *event;
	static char    *func = "client_queue_size";

	if (cli == (CLIENT_ENTRY *) NULL)
	{
		return 0;
	}
	
	for (event = cli->event_q;
		event != (EVENT_ENT *) NULL;
		event = event->client_next)
	{
		i++;	/* Count the event queued here. */
	}

	return i;
} /* end of client_queue_size () */

/*************************************************************************
 *
 *	Function: config_files
 *
 *	Purpose: Read database files into memory data structures.  Reads
 *		 "RADIUS_CLIENTS" and "RADIUS_AUTH" files unconditionally
 *		 and "RADIUS_USERS" file if users_flag is not zero.  Will
 * 		 read multiple users and authfiles if the "file_pfx" is
 *		 specified in a client entry (allowing use of different
 *		 files for different client NASes).
 *
 *		 If clear_flag is greater than zero, remove existing entries.
 *
 *		 A new CLIENT_ENTRY is added to the client_list for each
 *		 client appearing in the "RADIUS_CLIENTS" file.  A new
 *		 FILE_LIST entry is added for each unique file_pfx found
 *		 in the "RADIUS_CLIENTS" file.  Each FILE_LIST entry points
 *		 to a list of USER_ENTRYs containing the information read
 *		 from the "RADIUS_USERS" file with the corresponding
 *		 file_pfx.  Also each FILE_LIST entry contains a pointer to
 *		 the list of AUTH_ENTRYs containing realm information read
 *		 from the "RADIUS_AUTH" file with the corresponding file_pfx.
 *		 If either the "RADIUS_USERS" file or the "RADIUS_AUTH" file
 *		 with the given file_pfx did not exist, the default
 *		 (non-prefixed) file name entries are used instead.
 *
 *************************************************************************/

int
config_files (users_flag, clear_flag, dolog)

int             users_flag;
int             clear_flag;
int             dolog;

{
	int             i;
	int             result;
	FILE_LIST      *file_ent;
	struct hostent *hp;
	char          **paddr;
	static char    *func = "config_files";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (clear_flag > 0) /* Free current list, if already set up */
	{
		free_file_lists ();
	}

	authfile_id[0] = '\0';
	clients_id[0] = '\0';

	/*
	 * Add default file_list entry - the entry for the "users" and
	 * "authfile" with no optional prefix
	 */
	add_file_list ("");

	/*
	 *	Determine the IP address(es) of this machine
	 *
	 *	NOTE: This could hang for a while!
	 */
	if ((hp = gethostbyname (ourhostname)) == (struct hostent *) NULL)
	{
		logit (LOG_DAEMON, LOG_CRIT,
			"%s: Couldn't get our own IP address(es)", func);
		return (-1);
	}

	/*
	 *	First clear, then save our IP address(es)
	 *	leaving a zero entry at the end.
	 */
	memset ((char *) self_ip, '\0', sizeof (self_ip));

	if (hp->h_addr_list != (char **) NULL)
	{
		for (i = 0, paddr = hp->h_addr_list;
			(*paddr != (char *) NULL) &&
				(i < sizeof (self_ip) / 4 - 1);
			i++, paddr++)
		{
			memcpy ((char *) &self_ip[i], (char *) *paddr,
				hp->h_length);
			self_ip[i] = ntohl(self_ip[i]);
		}
	}

	/*
	 *	Release /etc/resolv.conf (hopefully).
	 */
	endhostent ();

	if ((result = read_clients (dolog)) < 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: FATAL error '%d' from read_clients()",
			func, result);
		return (-1);
	}

	/*
	 * Finally, go through all the file_list entries just added, reading
	 * in the "users" and "authfile" for each prefix found.
	 */
	for (file_ent = file_list; file_ent; file_ent = file_ent->next)
	{

#if !(defined(USE_DBM) || defined(USE_NDBM))
		if (users_flag)
			if (read_users (file_ent, dolog) != 0)
			{
				return (-1);
			}
#endif	/* USE_DBM || USE_NDBM */

		if (read_auth (file_ent, dolog) != 0)
		{
			return (-1);
		}

	}

	if (result < 0)
	{
		return (-1);
	}

	return 0;
} /* end of config_files () */

/*************************************************************************
 *
 *	Function: config_fini
 *
 *	Purpose: Cleanup environment after config_files() has run.
 *
 *************************************************************************/

void
config_fini ()

{
	free_clients (old_clients);
	old_clients = (CLIENT_ENTRY *) NULL;

	if ((file_logging == 1) && (msgfd != (FILE *) NULL))
	{
		fflush (msgfd);
	}

	if (dnspid == -1)
	{
		dnspid = 0;
	}

	update_clients ();       /* Start up the DNS resolver. */
	doing_init = 0;		/* Done with initialization */
	return;
} /* end of config_fini () */

/*************************************************************************
 *
 *	Function: config_init
 *
 *	Purpose: Setup environment for config_files() to run in.
 *
 *************************************************************************/

void
config_init ()

{
	is_engine = 1; /* flag set when engine calls us */

	/*
	 *	Set dnspid to defer the call to update_clients()
	 *	until the end of config_files().
	 */
	if (dnspid == 0)
	{
		dnspid = -1;
	}
	doing_init = 1;		/* Indicate we're doing initialization */

	/*
	 *	Save the old clients list so we can pick up
	 *	the DNS addresses for the new list.
	 */
	old_clients = client_list;
	client_list = (CLIENT_ENTRY *) NULL;
	return;
} /* end of config_init () */

/*************************************************************************
 *
 *	Function: dns_recv
 *
 *	Purpose: Process received DNS updates for clients database.
 *
 *************************************************************************/

void
dns_recv (sin, from_ipaddr, rcvlen)

struct sockaddr_in *sin;
UINT4               from_ipaddr;
int                 rcvlen;

{
	u_char           aliascnt;
	int              cnt;
	int              count;
	UINT4            temp;
	UINT4           *ourad;
	char            *ptr;
	IP_ADDRESS      *an_address;
	DNS_NAME        *a_name;
	CLIENT_ENTRY    *client_ent;
	static int       notify_count = 0;
	static char     *func = "dns_recv";

	notify_count++;

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered (%d)",
		func, notify_count));
 
	ddumpx(3, (LOG_DAEMON, LOG_DEBUG, recv_buffer, rcvlen, 0,
		"%s: Packet received from %s", func,
		ip_hostname (from_ipaddr)));

	/* Reset client_ent cache. */
	last_client_found = (CLIENT_ENTRY *) NULL;
	last_client_name = (char *) NULL;
	last_client_ipaddr = 0;

	ptr = recv_buffer + 1;

	for (ourad = self_ip;
		(*ourad != (UINT4) 0) && (*ourad != from_ipaddr);
		ourad++)
	{
		continue;
	}

	if (*ourad == (UINT4) 0)
	{
		logit (LOG_DAEMON, LOG_INFO, "%s: from %s - Security Breach",
			func, ip_hostname (from_ipaddr));
		return;
	}

	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL; 
		client_ent = client_ent->next)
	{
		if (strcmp (ptr, client_ent->hostname) == 0)
		{
			break;
		}
	}

	if (client_ent == (CLIENT_ENTRY *) NULL)
	{
		return;
	}		

	ptr += strlen (ptr) + 1;	/* Point over name */
	aliascnt = *ptr++;

	if (*ptr != '\0') /* If alias or IP address present, clear old ones */
	{
		/*
		 *	Reset expire_time with some randomness (0 - 60 minutes)
		 *	to avoid a burst of calls to gethostbyname().
		 *
		 *	[0, 1, 2, 3] * 1200 == [0, 1200, 2400, 3600]
		 */
		client_ent->expire_time = (time (0) + dns_address_aging +
						(rand () & 3) * 60 * 20) & ~3;

		dprintf(2, (LOG_AUTH, LOG_DEBUG,
			"%s: expire time = 0x%08X",
			func, client_ent->expire_time));

		for (an_address = client_ent->addrs;
			an_address != (IP_ADDRESS *) NULL;
			an_address = client_ent->addrs)
		{
			client_ent->addrs = an_address->next;
			free (an_address);
			dns_addr_mf.f++;
		}
 
		for (a_name = client_ent->names;
			a_name != (DNS_NAME *) NULL;
			a_name = client_ent->names)
		{
			client_ent->names = a_name->next;
			free (a_name);
			dns_name_mf.f++;
		}
	}
	else /* no alias or IP address present */
	{
		memcpy ((char *) &temp, ptr, sizeof (struct in_addr));
		if (temp == TRY_AGAIN) /* TRY_AGAIN is in netdb.h */
		{
			client_ent->expire_time = (time (0) + dns_address_aging)
									& ~3;
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: DNS timeout on client or host '%s'",
				func, client_ent->hostname);
			return;
		}
		else
		{
			/*
			 *	Name couldn't be resolved -- log it and retry
			 *	shortly.  Cleverly (or foolishly) use the low
			 *	two bits of expire_time to control the logging
			 *	frequency.
			 */
			if ((cnt = client_ent->expire_time & 3) == 0)
			{
				/* Log every fourth time (or once per hour) */
				logit (LOG_DAEMON, LOG_ALERT,
				     "%s: DNS couldn't resolve name '%s' (%ld)",
					func, client_ent->hostname, temp);
			}

			client_ent->expire_time =
				((time (0) + (dns_address_aging / 4)) & ~3) |
					(++cnt & 3);
			if (client_ent->addrs != (IP_ADDRESS *) NULL)
			{
				return;
			}

			/* Add invalid address to client_ent */
			memset (ptr, 255, sizeof (struct in_addr));
		}
	}

	/* Add alias names to client_ent structure */
	count = 0;
	while (aliascnt-- > 0)
	{
		count++;
		if (count > MAX_ALIAS)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: FATAL: Too many aliases for client '%s'",
				func, client_ent->hostname);
			abort ();
		}

		/* Note that DNS_NAME structure reserves one extra character. */
		a_name = (DNS_NAME *)
				get_memory (sizeof (DNS_NAME) + strlen (ptr),
						func, "DNS_NAME");
		dns_name_mf.m++;

		/* Note that type zero will always be the last one. */
		strcpy (a_name->name, ptr); 
		ptr += strlen (ptr) + 1;
		a_name->type =  (u_char) *ptr++;
		a_name->next = client_ent->names;
		client_ent->names = a_name;
	}

	/*
	 * For each address in the list, add the address to the client_ent.
	 */
	while (*ptr != '\0')
	{
		an_address = (IP_ADDRESS *) get_memory (sizeof (IP_ADDRESS),
							func, "IP_ADDRESS");
		dns_addr_mf.m++;

		an_address->next = client_ent->addrs;
		memcpy ((char *) &temp, ptr, sizeof (struct in_addr));
		an_address->ipaddr.s_addr = ntohl(temp);
		client_ent->addrs = an_address;
		ptr += sizeof (struct in_addr);
	}

	if ((notify_count % DNS_SLEEP) == 0)
	{
		logit (LOG_DAEMON, LOG_INFO, "%s: Notified of (%d) DNS changes",
			func, notify_count);
	}

	return;
} /* end of dns_recv () */

/*************************************************************************
 *
 *	Function: fieldcpy
 *
 *	Purpose: Copy a data field from the buffer to which "uptr" points.
 *		 Advance the buffer past the data field.
 *
 *	Returns: actual length of the data field,
 *		 or -1, if error.
 *
 *************************************************************************/

static int
fieldcpy (string, uptr, len)

char           *string;	/* Receiving buffer. */
char          **uptr;	/* Pointer to original data passed in. */
int             len;	/* Size of receiving buffer "string" */

{
	int             accum;
	int             count;
	char           *beg_string;
	char           *end;
	char           *end_tag;	/* end of tag in data string */
	char           *ptr;
	char            hex_buf[4];	/* 2 hex digits or 3 octal digits */
	static char    *func = "fieldcpy";

	ptr = *uptr;
	end = string + len - 1; /* Leave room for NULL character at end. */
	beg_string = string;

	/* Tagged attribute values are of the form :<tag>:<value> */
	if (*ptr == ':')
	{
		ptr++;	/* Skip leading colon character. */

		if ((end_tag = strchr (ptr, ':')) == (char *) NULL)
		{
			logit (LOG_DAEMON, LOG_INFO,
				"%s: No terminating colon for tag in ':%s'",
				func, ptr);
			return (-1);
		}

		*end_tag = '\0';
		accum = atoi (ptr);	/* fieldcpy() tag value */

		if ((accum < 0) || ( 255 < accum))
		{
			logit (LOG_DAEMON, LOG_INFO,
				"%s: Tag '%s' (%d) out of range (0-255)",
				func, ptr, accum);
			return (-1);
		}

		*string = accum;	/* Insert tag into buffer. */
		string++;
		ptr = end_tag + 1;	/* Position past tag. */
	}

	/* Quoted attribute strings are of the form "<value>" */
	if (*ptr == '"')
	{
		ptr++;	/* Skip leading quote character. */

		while (*ptr != '"' && *ptr != '\0' && *ptr != '\n')
		{
			if (*ptr == '\\')
			{
				accum = 0;
				count = 0;
				ptr++;
				/* octal conversion */
				while ((count < 3) &&
					(*ptr >= '0') &&
					(*ptr <= '7'))
				{
					count++;
					accum = (accum << 3) + (*ptr++ - '0');
				}

				if (count > 0)
				{
					*string++ = accum;
				}
				else
				{
					switch (*ptr)
					{
					    case 'a':	/* bell */
						*string++ = '\a';
						break;

					    case 'b':	/* backspace */
						*string++ = '\b';
						break;

					    case 'f':	/* formfeed */
						*string++ = '\f';
						break;

					    case 'n':	/* linefeed */
						*string++ = '\n';
						break;

					    case 'r':	/* return */
						*string++ = '\r';
						break;

					    case 't':	/* tab */
						*string++ = '\t';
						break;

					    case 'x':	/* hex */
					    case 'X':
						ptr++;
						strcpy (hex_buf, ptr);
						sscanf ("%x", hex_buf, &accum);
						ptr += strlen (hex_buf);
						*string++ = accum;
						break;

					    case '0':	/* octal or hex */
						ptr++;
						switch (*ptr)
						{
						    default:	/* \0 */
							*string++ = '\0';
							ptr--;	/* backup */
							break;

						    case 'x':	/* hex \0x## */
						    case 'X':	/* hex \0X## */
							ptr++;
							strcpy (hex_buf, ptr);
							sscanf ("%x", hex_buf,
							&accum);
							ptr += strlen (hex_buf);
							*string++ = accum;
							break;

						    case '1':	/* octal */
						    case '2':
						    case '3':
						    case '4':
						    case '5':
						    case '6':
						    case '7':
							while ((count < 3) &&
								(*ptr >= '0') &&
								(*ptr <= '7'))
							{
							    count++;
							    accum =
								(accum << 3) +
								(*ptr++ - '0');
							}
							*string++ = accum;
						} /* end of inner switch */
						break;

					    default:
						*string++ = *ptr;
					} /* end of switch */
					ptr++;
				} /* end of else count == 0 */
			}
			else /* was not escaped character */
			{
				*string++ = *ptr++;
			}

			if (string == end)
			{
				/* Done here for logit() call below. */
				*string = '\0';
				logit (LOG_DAEMON, LOG_INFO,
					"%s: Truncated string '%s'",
					func, string);
				break;
			}
		} /* end of while */
		*string = '\0';

		if (*ptr == '"')
		{
			ptr++;
		}
		*uptr = ptr;

		return (string - beg_string);
	} /* end of if quoted string */

	while (*ptr != ' ' && *ptr != '\t' && *ptr != '\0' && *ptr != '\n' &&
			*ptr != '=' && *ptr != ',')
	{
		*string++ = *ptr++;
		if (string == end)
		{
			*string = '\0'; /* Done here for logit() call below. */
			logit (LOG_DAEMON, LOG_INFO,
				"%s: Truncated value '%s'",
				func, string);
			break;
		}
	}
	*string = '\0';
	*uptr = ptr;

	return (string - beg_string);
} /* end of fieldcpy () */

/*************************************************************************
 *
 *	Function: find_auth_ent
 *
 *	Purpose: Gives access to the private AUTH_ENT for the given realm.
 *
 *	Returns: pointer to the AUTH_ENT for the given realm,
 *		 or, NULL, if error.
 *
 *************************************************************************/

AUTH_ENTRY *
find_auth_ent (u_realm, pfx)

char           *u_realm;
char           *pfx;

{
	int             head;
	int             pat_len;
	FILE_LIST      *file_ent;
	AUTH_ENTRY     *auth_ent;
	AUTH_ENTRY     *entry;
	char           *p;
	static char    *func = "find_auth_ent";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((file_ent = find_file_ent (pfx)) == (FILE_LIST *) NULL)
	{
		dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: NULL file_ent", func));
		return (AUTH_ENTRY *) NULL;
	}

	if ((auth_ent = file_ent->auth_list) == (AUTH_ENTRY *) NULL)
	{
		/* If no auth_list for this prefix */
		file_ent = file_list;

		/* Default file_ent is first in file_list */
		if ((auth_ent = file_ent->auth_list) == (AUTH_ENTRY *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: no default authfile data structure",
				func);
			return (AUTH_ENTRY *) NULL;
		}
	}

	/*
	 *	Match realm name (either exact match or substring match
	 *	based on *.realm syntax) with user supplied string.
	 */
	for ( ; auth_ent != (AUTH_ENTRY *) NULL; auth_ent = auth_ent->next)
	{
		if (auth_ent->parent == (AUTH_ENTRY *) NULL) /* parent realm */
		{
			entry = auth_ent;
		}
		else
		{
			entry = auth_ent->parent;
		}

		/* Look for name match. */
		if (entry->name[0] == '*') /* this is wildcard realm */
		{
			p = &entry->name[1];
			pat_len = strlen (p);
			if ((head = strlen (u_realm) - pat_len) >= 0 &&
				(strncmp ((char *) &u_realm[head],
					  (char *) &entry->name[1],
					  pat_len) == 0))
			{
				return entry;
			}
		}
		else /* not a wildcard realm */
		{
			if (strcasecmp (entry->name, u_realm) == 0)
			{
				return entry;
			}
		}
	}

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: realm not found", func));
	return (AUTH_ENTRY *) NULL;

} /* end of find_auth_ent () */

/*************************************************************************
 *
 *	Function: find_auth_type
 *
 *	Purpose: Find the proper AUTH_ENTRY to use for the given authentication
 *		 realm name from the FILE_LIST entry with the given file_pfx.
 *
 *	Returns: The authentication type, name of the authentication agent to
 *		 use, the primary realm name and any optional packet filter
 *		 to be applied are returned.
 *
 *	Returns:  0 = normal return,
 *		 -1 = error return
 *
 *************************************************************************/

int
find_auth_type (u_realm, prot, pfx, type, agent, realm, filter)

char           *u_realm;
int             prot;
char           *pfx;
int            *type;	/* receives resultant authentication type value */
char          **agent;	/* receives resultant authentication agent name */
char          **realm;	/* receives resultant primary realm name */
char          **filter;	/* receives resultant authentication filter name */

{
	int             head;
	int             pat_len;
	FILE_LIST      *file_ent;
	AUTH_ENTRY     *auth_ent;
	AUTH_ENTRY     *entry;
	char           *p;
	char           *realm_name;
	static char     temp[AUTH_ID_LEN + 1];
	static char    *func = "find_auth_type";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((file_ent = find_file_ent (pfx)) == (FILE_LIST *) NULL)
	{
		dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: NULL file_ent", func));
		return (-1);
	}

	if ((auth_ent = file_ent->auth_list) == (AUTH_ENTRY *) NULL)
	{
		/* If no auth_list for this prefix */
		file_ent = file_list;
		/* Default file_ent is first in file_list */
		if ((auth_ent = file_ent->auth_list) == (AUTH_ENTRY *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: no default authfile data structure",
				func);
			return (-1);
		}
	}

	/*
	 *	Match realm name (either exact match or substring match
	 *	based on *.realm syntax) with user supplied string.
	 */
	for ( ; auth_ent ; auth_ent = auth_ent->next )
	{
		realm_name = (char *) NULL;
		if (auth_ent->parent == (AUTH_ENTRY *) NULL) /* parent realm */
		{
			entry = auth_ent;
			/* Look for name match. */
			if (entry->name[0] == '*') /* this is wildcard realm */
			{
				p = &entry->name[1];
				pat_len = strlen (p);
				head = strlen (u_realm) - pat_len;
				if (strncmp ((char *) &u_realm[head],
					(char *) &entry->name[1], pat_len) == 0)
				{
					realm_name = u_realm;
				}
				else
				{
					realm_name = (char *) NULL;
				}
			}
			else /* not a wildcard realm */
			{
				if (strcasecmp (entry->name, u_realm) == 0)
				{
					realm_name = entry->name;
				}
			}
		}
		else /* this entry is an alias name for some real realm */
		{
			entry = auth_ent->parent;
			/* Look for name match. */
			if (entry->name[0] == '*') /* alias in wildcard realm */
			{
				p = &entry->name[1];
				pat_len = strlen (p);
				head = strlen (u_realm) - pat_len;
				if (strncmp ((char *) &u_realm[head],
					(char *) &entry->name[1], pat_len) == 0)
				{
					/* combine real prefix, parent suffix */
					strcpy (temp, u_realm);
					if (strtok (temp, ".") != (char *) NULL)
					{
						realm_name = strcat (temp,
							       &entry->name[1]);
					}
				}
				else
				{
					realm_name = (char *) NULL;
				}
			}
			else /* regular alias */
			{
				if (strcasecmp (auth_ent->name, u_realm) == 0)
				{
					realm_name = entry->name;
				}
			}
		}

		if (realm_name != (char *) NULL) /* then we have a name match */
		{
			if (!entry->prot || (entry->prot == prot))
			{
				break;
			}
		}
	}

	if (auth_ent == (AUTH_ENTRY *) NULL)
	{
		dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: realm not found", func));
		return (-1);
	}

	*type = entry->type;
	*agent = entry->host;
	*realm = realm_name;
	*filter = entry->filter;
	dprintf(2, (LOG_AUTH, LOG_DEBUG,
		"%s: type %d, agent '%s', realm '%s' and filter '%s'",
		func, entry->type,
		(entry->host == (char *) NULL) ? "?" : entry->host,
		(realm_name == (char *) NULL) ? "?" : realm_name,
		(entry->filter == (char *) NULL) ? "?" : entry->filter));
	return 0;
} /* end of find_auth_type () */

/*************************************************************************
 *
 *	Function: find_client
 *
 *	Purpose: Find the CLIENT_ENTRY in client_list for the client with
 *		 the given IP address.  If the entry is found, a pointer
 *		 to the found client structure is returned.
 *
 *	Returns: 0 = found client entry,
 *		-1 = client not found.
 *
 *************************************************************************/

int
find_client (ipaddr, client)

UINT4           ipaddr;
CLIENT_ENTRY  **client;         /* Pointer to Client entry */

{
	int             count;
	int             ud = 0;
	CLIENT_ENTRY   *client_ent;
	IP_ADDRESS     *an_address;
	time_t          cur_time;
	static char    *func = "find_client";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Check to see if the last match was us. */
	if ((last_client_ipaddr != 0) && (last_client_ipaddr == ipaddr))
	{
		client_ent = last_client_found;
	}
	else /* Reset cache */
	{
		last_client_found = (CLIENT_ENTRY *) NULL;
		last_client_ipaddr = 0;
		last_client_name = (char *) NULL;
		cur_time = time (0);
		for (client_ent = client_list;
			client_ent;
			client_ent = client_ent->next)
		{
			if (cur_time > client_ent->expire_time)
			{
				ud = 1;
			}

			count = 0;
			for (an_address = client_ent->addrs;
				an_address != (IP_ADDRESS *) NULL;
				an_address = an_address->next)
			{
				count++;
				if (count > MAX_ALIAS)
				{
					logit (LOG_AUTH, LOG_ALERT,
				"%s: FATAL: Too many addresses for client '%s'",
						func, client_ent->hostname);
					abort ();
				}

				if (an_address->ipaddr.s_addr == ipaddr)
				{
					break;
				}
			}

			if (an_address)
			{
				break;
			}
		}

		if (ud > 0)
		{
			update_clients ();
		}
	}

	/* Don't match host-only entries (those with a null secret) */
 	if (client_ent != (CLIENT_ENTRY *) NULL && *client_ent->secret == '\0')
	{
		client_ent = (CLIENT_ENTRY *) NULL;
	}

	if (client != (CLIENT_ENTRY **) NULL)
	{
		*client = client_ent;
	}

	if (client_ent != (CLIENT_ENTRY *) NULL)
	{
		last_client_found = client_ent;
		last_client_name = client_ent->hostname;
		last_client_ipaddr = ipaddr;
	}

	return client_ent ? 0 : -1;
} /* end of find_client () */

/*************************************************************************
 *
 *	Function: find_client_by_name
 *
 *	Purpose: Find the CLIENT_ENTRY in client_list for the client with
 *		 the given hostname.  If the entry is found, a pointer
 *		 to the found client structure is returned.
 *
 *	Returns: 0 = found client entry and resolved IP address,
 *		 1 = found client entry but no IP address,	
 *		 2 = found host entry but IP address not obtained 
 *		     (unresolvable DNS name),	
 *		-1 = client not found.
 *
 *************************************************************************/

int
find_client_by_name (hostname, ipaddr, client_entry)

char           *hostname;	/* Match this name */
UINT4          *ipaddr;		/* Receives resultant address, if found */
CLIENT_ENTRY  **client_entry;   /* Return pointer to structure */

{
	int             ud = 0;
	time_t          cur_time;
	CLIENT_ENTRY   *client_ent;
	DNS_NAME       *name_ent;
	static char    *func = "find_client_by_name";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	*client_entry = (CLIENT_ENTRY *) NULL;
	*ipaddr = 0;			/* Saftey. */

	if ((last_client_found != (CLIENT_ENTRY *) NULL) &&
		(last_client_name != (char *) NULL) &&
		(strcmp (last_client_name, hostname) == 0))
	{
		client_ent = last_client_found;
	}
	else if (good_ipaddr (hostname) == 0)
	{
		/* name == address --  Really just a find_client() call */
		*ipaddr = ntohl(inet_addr (hostname));
		return find_client (*ipaddr, client_entry);
	}
	else /* Reset cache. */
	{
		last_client_found = (CLIENT_ENTRY *) NULL;
		last_client_name = (char *) NULL;
		last_client_ipaddr = 0;

		if (strcmp (hostname, RADIUS_LOCALSERVER) == 0)
		{
			hostname = ourhostname;
		}

		cur_time = time (0);
		for (client_ent = client_list;
			client_ent != (CLIENT_ENTRY *) NULL;
			client_ent = client_ent->next)
		{
			if (cur_time > client_ent->expire_time)
			{
				ud = 1;
			}

			if (strcmp (client_ent->hostname, hostname) == 0)
			{
				last_client_name = client_ent->hostname;
				break;
			}

			for (name_ent = client_ent->names;
				name_ent != (DNS_NAME *) NULL;
				name_ent = name_ent->next)
			{
				if (strcmp (name_ent->name, hostname) == 0)
				{
					break;
				}
			}

			if (name_ent != (DNS_NAME *) NULL)
			{
				break;
			}
		}
	}

	if (ud > 0)
	{
		update_clients ();
	}

	/* Don't match host-only entries (those with a null secret) */
	if (client_ent == (CLIENT_ENTRY *) NULL || *client_ent->secret == '\0')
	{
		last_client_name = (char *) NULL;
		return (-1);
	}

	if (client_ent->addrs == (IP_ADDRESS *) NULL)
	{
		last_client_name = (char *) NULL;
		*ipaddr = 0;
		return (1);
	}

	if ((*ipaddr = client_ent->addrs->ipaddr.s_addr) == -1)
	{
		last_client_name = (char *) NULL;
		return (2);
	}

	last_client_ipaddr = *ipaddr;
	last_client_found = client_ent;
	if (last_client_name == (char *) NULL)
	{
		last_client_name = name_ent->name;
	}

	*client_entry = client_ent;
	return (0);
} /* end of find_client_by_name () */

/*************************************************************************
 *
 *	Function: find_file_ent
 *
 *	Purpose: Find a FILE_LIST entry on file_list with the specified
 *		 file_pfx.  The entry should be found as find_file_ent is
 *		 only called for file_pfx's that were found in the "clients"
 *		 file at initialization time.
 *
 *************************************************************************/

#ifdef USR_CCA
FILE_LIST *
#else
static FILE_LIST *
#endif /* USR_CCA */
find_file_ent (file_pfx)

char           *file_pfx;

{
	FILE_LIST      *file_ent;
	static char    *func = "find_file_ent";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((file_ent = file_list) == (FILE_LIST *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: No users/authfile data structure", func);
		return (FILE_LIST *) NULL;
	}
	if (file_pfx && file_pfx[0])
	{
		while (strcmp (file_ent->prefix, file_pfx) != 0)
		{
			if ((file_ent = file_ent->next) == (FILE_LIST *) NULL)
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Couldn't match %s in FILE_LIST",
					func, file_pfx);
				return (FILE_LIST *) NULL;
			}
		}
	}
	return file_ent;
} /* end of find_file_ent () */

/*************************************************************************
 *
 *	Function: find_host_by_name
 *
 *	Purpose: Resolve the host address by looking in the client list.
 *		 Non-clients (those with a null secret) in this list
 *		 are matched as well as normal clients.
 *
 *	Returns: 0 = found host entry and resolved IP address,
 *		 1 = found host entry but unresolved IP address,	
 *		 2 = found host entry but IP address not obtained 
 *		     (unresolvable DNS name - uses address 255.255.255.255),	
 *		-1 = host not found.
 *
 *************************************************************************/

int
find_host_by_name (ipaddr, hostname)

UINT4          *ipaddr;		/* receives resultant address if found */
char           *hostname;	/* Match this name */

{
	int             ud = 0;
	char           *p;
	char           *q;
	CLIENT_ENTRY   *client_ent;
	DNS_NAME       *name_ent;
	time_t          cur_time;
	static char    *func = "find_host_by_name";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (good_ipaddr (hostname) == 0)
	{
		*ipaddr = ntohl(inet_addr (hostname));
		return 0;
	}

	if (strcmp (hostname, RADIUS_LOCALSERVER) == 0)
	{
		*ipaddr = self_ip[0];
		return 0;
	}

	/* See if it's us.  Match full name or up to "." of our name */
	for (p = hostname, q = ourhostname; *p == *q; p++, q++)
	{
		if (*p == '\0')
		{
			break;
		}
	}

	if (*p == '\0' && (*q == '\0' || *q == '.'))
	{
		*ipaddr = self_ip[0];
		return 0;
	}

	cur_time = time (0);
	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_ent->next)
	{
		if (cur_time > client_ent->expire_time)
		{
			ud = 1;
		}

		if (strcmp (client_ent->hostname, hostname) == 0)
		{
			break;
		}

		for (name_ent = client_ent->names;
			name_ent != (DNS_NAME *) NULL;
			name_ent = name_ent->next)
		{
			if (strcmp (name_ent->name, hostname) == 0)
			{
				break;
			}
		}

		if (name_ent != (DNS_NAME *) NULL)
		{
			break;
		}
	}

	if (ud > 0)
	{
		update_clients ();
	}

	if (client_ent == (CLIENT_ENTRY *) NULL)
	{
		*ipaddr = 0;
		return (-1);
	}

	if (client_ent->addrs == (struct ip_address *) NULL)
	{
		*ipaddr = 0;
		return (1);
	}

	if ((*ipaddr = client_ent->addrs->ipaddr.s_addr) == -1)
	{
		return (2);
	}
	return (0);
} /* end of find_host_by_name () */

/*************************************************************************
 *
 *	Function: free_clients
 *
 *	Purpose: Toss client list entries and associated address structure.
 *
 *	Remark: Zap storage blocks to avoid leaving any secrets around.
 *
 *************************************************************************/

static void
free_clients (client_list)

CLIENT_ENTRY   *client_list;

{
	int                count;
	CLIENT_ENTRY      *client_ent;
	IP_ADDRESS        *an_address;	
	DNS_NAME          *a_name;
	static char       *func = "free_clients";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_list)
	{
		client_list = client_ent->next;

		count = 0;
		for (an_address = client_ent->addrs;
			an_address != (IP_ADDRESS *) NULL;
			an_address = client_ent->addrs)
		{
			count++;
			if (count > MAX_ALIAS)
			{
				logit (LOG_AUTH, LOG_ALERT,
				"%s: FATAL: Too many addresses for client '%s'",
					func, client_ent->hostname);
				abort ();
			}

			client_ent->addrs = an_address->next;
			free (an_address);
			dns_addr_mf.f++;
		}

		count = 0;
		for (a_name = client_ent->names;
			a_name != (DNS_NAME *) NULL;
			a_name = client_ent->names)
		{
			count++;
			if (count > MAX_ALIAS)
			{
				logit (LOG_AUTH, LOG_ALERT,
				  "%s: FATAL: Too many aliases for client '%s'",
					func, client_ent->hostname);
				abort ();
			}

			client_ent->names = a_name->next;
			free (a_name);
			dns_name_mf.f++;
		}

		(void) free_vendor_list (client_ent->veps);
		client_ent->veps = (VENDOR_LIST *) NULL;
		free (client_ent);
		dns_client_mf.f++;
	}
	return;
} /* end of free_clients () */

/*************************************************************************
 *
 *	Function: free_file_lists
 *
 *	Purpose: Free all the storage for the "users" and "authfile"
 *		 memory resident data structures allocated by calling
 *		 config_files().
 *
 *************************************************************************/

static void
free_file_lists ()

{
	FILE_LIST      *file_ent;
	USER_ENTRY     *user_ent;
	AUTH_ENTRY     *auth_ent;

#ifdef USR_CCA
	ADDR_POOL      *pool_ent;
	VPN_INFO       *vpn;
	IP_ADDRESS     *pdns;
#endif	/* USR_CCA */

	static char    *func = "free_file_lists";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	authfile_cnt = 0;
	authfile_id[0] = '\0';
	users_cnt = 0;

	for (file_ent = file_list; file_ent; file_ent = file_list)
	{
		for (user_ent = file_ent->user_list;
			user_ent;
			user_ent = file_ent->user_list)
		{
			file_ent->user_list = user_ent->next;
			free_user_ent (user_ent);
		}
		for (auth_ent = file_ent->auth_list;
			auth_ent;
			auth_ent = file_ent->auth_list)
		{
			file_ent->auth_list = auth_ent->next;

#ifdef USR_CCA
			if (auth_ent->parent == (AUTH_ENTRY *) NULL)
			{
				if (auth_ent->vpn != (VPN_INFO *) NULL)
				{
					vpn = auth_ent->vpn;
					free_vpn_rtr (&vpn->router);
					free (vpn);
				}

				if (auth_ent->dns_info != (IP_ADDRESS *) NULL)
				{
					pdns = auth_ent->dns_info;
					while (pdns != (IP_ADDRESS *) NULL)
					{
						auth_ent->dns_info = pdns->next;
						free (pdns);
						pdns = auth_ent->dns_info;
					}
				}

				if (auth_ent->nbns_info != (IP_ADDRESS *) NULL)
				{
					pdns = auth_ent->nbns_info;
					while (pdns != (IP_ADDRESS *) NULL)
					{
						auth_ent->nbns_info =
								pdns->next;
						free (pdns);
						pdns = auth_ent->nbns_info;
					}
				}
			}
#endif	/* USR_CCA */

			free (auth_ent);
		}
#ifdef USR_CCA
		for (pool_ent = file_ent->pool_list;
			pool_ent;
			pool_ent = file_ent->pool_list)
		{
			file_ent->pool_list = pool_ent->next;
			free_pool_ent (pool_ent);
		}
#endif
		file_list = file_ent->next;
		free (file_ent);
	}

	return;
} /* end of free_file_lists () */

/*************************************************************************
 *
 *	Function: free_user_ent
 *
 *	Purpose: Free all components of a USER_ENTRY structure.  Zap
 *		 the USER_ENTRY storage.
 *
 *************************************************************************/

void
free_user_ent (user_ent)

USER_ENTRY     *user_ent;

{
	list_free (user_ent->check);
	list_free (user_ent->deny);
	list_free (user_ent->reply);
	memset ((char *) user_ent, '\0', sizeof (USER_ENTRY));
	free (user_ent);
	return;
} /* end of free_user_ent () */

/*************************************************************************
 *
 *	Function: get_client_list
 *
 *	Purpose: This function returns a pointer to the client_list.
 *
 ************************************************************************/

CLIENT_ENTRY *
get_client_list ()

{
	return client_list;

} /* end of get_client_list () */

/****************************************************************************
 *
 *	Function: get_client_type
 *
 *	Purpose: Determines the host type for given entry in the cilents file.
 *
 ***************************************************************************/
 
static int
get_client_type (line_nbr, veps)
 
int            line_nbr;
VENDOR_LIST  **veps;
 
{
	typedef struct
	{
		char    *name;
		int      val;
	} type_map_t;

	int      cli_type;
	int      i;			/* for looping over typelist[] below */
	char    *p;
	char    *type;
	char    *each;
	char    *next;

	static type_map_t typelist[] =
	{
		{ "NAS",	CE_NAS       }, /* Entry is a NAS */
		{ "PROXY",	CE_PROXY     },	/* Entry is a RADIUS server */
		{ "DAS",	CE_DAS       },	/* USR */
		{ "FRGW",	CE_FRGW      },	/* USR */
		{ "NEIGHBOR",	CE_NEIGHBOR  },	/* USR */
		{ "RAD_RFC",	CE_RAD_RFC   },	/* Entry is RFC conformant */
		{ "ACCT_RFC",	CE_ACCT_RFC  },	/* Entry is accounting RFC */
		{ "DEBUG",	CE_DEBUG     },	/* Dump packets in and out */
		{ "APPEND",	CE_APPEND    },	/* Only add new attributes */
		{ "OLDCHAP",	CE_OLDCHAP   },	/* Pre-RFC CHAP semantics */
		{ "NOENCAPS",	CE_NOENCAPS  },	/* No VSA encapsulation */
		{ "HGAS1",	CE_HGAS1     },	/* HGAS flag #1 */
		{ "HGAS2",	CE_HGAS2     },	/* HGAS flag #2 */
		{ "HGAS3",	CE_HGAS3     },	/* HGAS flag #3 */
		{ "HGAS4",	CE_HGAS4     },	/* HGAS flag #4 */
		{ "LAS1",	CE_LAS1      },	/* LAS flag #1 */
		{ "LAS2",	CE_LAS2      },	/* LAS flag #2 */
		{ "LAS3",	CE_LAS3      },	/* LAS flag #3 */
		{ "LAS4",	CE_LAS4      },	/* LAS flag #4 */
		{ "CHECK_ALL",	CE_CHECK_ALL },	/* Duplicate checking flag */
		{ "NO_CHECK",	CE_NO_CHECK  },	/* Duplicate checking flag */
		{ NULL,		0            }	/* End-of-list */
	};
	char    *func = "get_client_type";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((p = strtok (NULL, " =\t\n")) == NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: missing host type on line %d",
			func, line_nbr);
		return (-1);
	}
 
	if ((type = parse_for_vendor_list (p, veps)) == (char *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: invalid vendor name '%s' on line %d",
			func, p, line_nbr);
		return (-1);
	}

	cli_type = 0;	/* Initial conditions. */
	
	/*
	 *	Allow for: "type=NAS", "type=NAS+ACCT_RFC", etc.
	 *	NOTE: Perhaps strpbrk (each, "+|") should be used instead.
	 */
	for (each = type, next = strchr (each, '+');
		each != NULL;
		each = next, next = strchr (each, '+'))
	{
		if (next != NULL)
		{
			*next = '\0';	/* NUL terminate string at '+' */
			next++;		/* Point past newly added NUL. */
		}

		for (i = 0; typelist[i].name != NULL; i++)
		{
			if (strcasecmp (each, typelist[i].name) == 0)
			{
				dprintf(2, (LOG_AUTH, LOG_DEBUG, 
					"%s: found type=%s, 0x%x", func,
					typelist[i].name, typelist[i].val));
				cli_type |= typelist[i].val;
				break;
			}
		}
	       
		/* Check for invalid types */
		if (typelist[i].name == NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: unknown host type '%s' on line %d",
				func, each, line_nbr);
			return (-1);
		}

		/* short circut test above. */
		if (next == NULL)
		{
			break;
		}
	}

	return cli_type;
} /* end of get_client_type () */

#ifdef USR_CCA

/*************************************************************************
 *
 *	Function: get_default_file_entry
 *
 *	Purpose: This function returns the first node from the file_list.
 *
 ************************************************************************/

FILE_LIST *
get_default_file_entry ()

{
	return file_list;	/* the first entry is the default */

} /* end of get_default_file_entry () */

#endif /* USR_CCA */

/*************************************************************************
 *
 *	Function: get_our_addr
 *
 *	Purpose: A global function to return a local variable (?)
 *
 *	Returns: (an) IP address of this machine.
 *
 *************************************************************************/

UINT4
get_our_addr ()

{
	return self_ip[0];
} /* end of get_our_addr () */

/*************************************************************************
 *
 *	Function: host_is_us
 *
 *	Purpose: Determine if we are the given host.
 *
 *	Returns: 1 if the given hostname is the name of this host,
 *		 0 otherwise.
 *
 *************************************************************************/

int
host_is_us (hostname)

char          *hostname;

{
	UINT4            addr;
	UINT4           *adptr;

	if (find_host_by_name (&addr, hostname) == 0)
	{
		for (adptr = self_ip; *adptr > 0; adptr++)
		{
			if (*adptr == addr)
			{
				return 1;
			}
		}
	}
	return 0;
} /* end of host_is_us () */

/*************************************************************************
 *
 *	Function: insert_client
 *
 *	Purpose: Inserts a CLIENT_ENTRY node into client_list for the
 *		 given hostname.
 *
 *	Returns: 0 - inserted ok
 *		-1 - bad news
 *
 *************************************************************************/

int
insert_client (hostname, secret, prefix, reply_hold, client_type, veps, rad_ver)

char           *hostname;
char           *secret;
char           *prefix;
int             reply_hold;
int             client_type;
VENDOR_LIST    *veps;
int             rad_ver;

{
	int             auth_port = 0;
	int             acct_port = 0;
	char           *col;
	char           *port;
	CLIENT_ENTRY   *client_ent = (CLIENT_ENTRY *) NULL;
	CLIENT_ENTRY   *oldent;
	CLIENT_ENTRY  **prev;
	IP_ADDRESS     *ip_address;
	struct in_addr  addr;
	static char    *func = "insert_client";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (reply_hold < 0)
	{
		reply_hold = CLEANUP_DELAY;
	}

	/* Allow <host>[:<auth_port>[:<acct_port>]] syntax on client names */

	if ((port = strchr (hostname, ':')) != (char *) NULL)
	{
		*port++ = '\0'; /* skip past the colon */
		/* atoi () ignores error conditions, like a ':' in number */
		auth_port = (u_short) atoi (port); /* parse clients file port */

		if((col = strchr (port, ':')) == (char *) NULL)
		{
			acct_port = auth_port + 1;
		}
		else
		{
			acct_port = (u_short) atoi (col + 1); /* clients file */
		}
	}

	/* Convert generic name for us to our real name */
	if (strcmp (hostname, RADIUS_LOCALSERVER) == 0)
	{
		hostname = ourhostname;
	}

	/* Look for entry from previous list (before HUP) */

	if (old_clients != (CLIENT_ENTRY *) NULL)
	{
		for (prev = &old_clients;
			(oldent = *prev) != (CLIENT_ENTRY *) NULL;
			prev = &oldent->next)
		{
			if (strcmp (hostname, oldent->hostname) == 0)
			{
				/* Matched - Remove from old list */
				*prev = oldent->next;
				client_ent = oldent;
				break;
			}
		}
	}

	if (client_ent == (CLIENT_ENTRY *) NULL)
	{
		client_ent = (CLIENT_ENTRY *) get_memory (sizeof (CLIENT_ENTRY),
							  func, "CLIENT_ENTRY");
		dns_client_mf.m++;
		client_ent->hostname = add_string (hostname, ASIS);
		client_ent->names = (DNS_NAME *) NULL;
		client_ent->addrs = (IP_ADDRESS *) NULL;
		client_ent->type = IP_DNS;

		/* Set constant addrs now so we don't have to wait for DNS */
		if (good_ipaddr (hostname) == 0)
		{
			client_ent->type = IP_NUMERIC;
			addr.s_addr = ntohl(inet_addr (hostname));
		}
		else
		{
			if (strcmp (hostname, ourhostname) == 0)
			{
				client_ent->type = IP_OURADDR;
				addr.s_addr = self_ip[0];
			}
		}

		if (client_ent->type != IP_DNS)
		{
			ip_address = (IP_ADDRESS *)
						get_memory (sizeof (IP_ADDRESS),
							    func, "IP_ADDRESS");
			dns_client_mf.m++;

			ip_address->ipaddr = addr;
			ip_address->next = (IP_ADDRESS *) NULL;
			client_ent->addrs = ip_address;
		}
	}
	else
	{
		(void) free_vendor_list (client_ent->veps);
	}

	client_ent->secret = add_string (secret, ASIS);
	client_ent->file_pfx = add_string (prefix, ASIS);
	client_ent->expire_time = (time_t) 0;
	client_ent->reply_holdtime = reply_hold;
	client_ent->client_type = client_type;
	client_ent->veps = veps;
	client_ent->event_q = (EVENT_ENT *) NULL;
	client_ent->version = rad_ver;
	client_ent->auth_port = auth_port;
	client_ent->acct_port = acct_port;
	client_ent->flags = 0;

#ifdef USR_CCA
	client_ent->state = NO_RQ_RESP;
#endif /* USR_CCA */

	/* find end of list */
	for (prev = &client_list; *prev; prev = &(*prev)->next)
	{
		continue;
	}
	client_ent->next = (CLIENT_ENTRY *) NULL;
	*prev = client_ent;

	/*
	 *	If the entry had an optional file prefix, add a new FILE_ENTRY
	 *	to the file_list to handle this prefix.  Add_file_list() will
	 *	not add duplicate entries.
	 */
	if (client_ent->file_pfx[0] != '\0')
	{
		add_file_list (client_ent->file_pfx);
	}
	return 0;
} /* end of insert_client () */

/*************************************************************************
 *
 *	Function: ip_hostname
 *
 *	Purpose: Return a printable host name (or IP address in dotted quad
 *		 notation) for the supplied IP address.
 *
 *************************************************************************/

char *
ip_hostname (h_ipaddr)

UINT4           h_ipaddr;

{
	UINT4          *ourad;
	CLIENT_ENTRY   *client_ent;
	IP_ADDRESS     *an_address;
	DNS_NAME       *a_name;
	struct hostent *hp;
	struct in_addr  inad;
	static char     hstname[MAXHOSTNAMELEN + 1];
	static char    *func = "ip_hostname";

	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_ent->next)
	{
		for (an_address = client_ent->addrs;
			an_address != (IP_ADDRESS *) NULL;
			an_address = an_address->next)
		{
			if (an_address->ipaddr.s_addr == h_ipaddr)
			{
				break;
			}
		}

		if (an_address != (IP_ADDRESS *) NULL)
		{
			break;
		}
	}

	if (client_ent != (CLIENT_ENTRY *) NULL)
	{
		if ((a_name = client_ent->names) == (DNS_NAME *) NULL ||
			(a_name->type != 0))
		{
			rad_strncpy (hstname, client_ent->hostname,
								MAXHOSTNAMELEN);
		}
		else /* return official name if it's not in the main entry */
		{
			rad_strncpy (hstname, a_name->name, MAXHOSTNAMELEN);
		}
	}
	else
	{
		for (ourad = self_ip;
			(*ourad > (UINT4) 0) && (*ourad != h_ipaddr);
			ourad++)
		{
			continue;
		}

		if (*ourad > (UINT4) 0)
		{
			rad_strncpy (hstname, ourhostname, MAXHOSTNAMELEN);
		}
		else
		{
			inad.s_addr = htonl(h_ipaddr);
			rad_strncpy (hstname, inet_ntoa (inad), MAXHOSTNAMELEN);

			/*
			 *	Special check for non-server use.
			 *	Note: a server always will have at
			 *	least one client.
			 */
			if (client_list == (CLIENT_ENTRY *) NULL)
			{
				if ((hp = gethostbyaddr ((char *) &inad.s_addr,
							sizeof (struct in_addr),
							AF_INET)) 
						!= (struct hostent *) NULL)
				{
					if (strlen (hp->h_name)
							      > MAXHOSTNAMELEN)
					{
						logit (LOG_AUTH, LOG_INFO,
				      "%s: '%s' truncated - hostname too long",
							func, inet_ntoa (inad));
					}

					rad_strncpy (hstname,
						(char *) hp->h_name,
						MAXHOSTNAMELEN);

					return (hstname);
				}
			}
		}
	}
	return (hstname);
} /* end of ip_hostname () */

#define PARSE_MODE_NAME		0
#define PARSE_MODE_EQUAL	1
#define PARSE_MODE_VALUE	2
#define PARSE_MODE_INVALID	3

/*************************************************************************
 *
 *	Function: pair_parse
 *
 *	Purpose: Parses the buffer to extract the attribute-value pairs.
 *
 *	Returns: 0 = successful parse of attribute-value pair,
 *		-1 = syntax (or other) error detected.
 *
 *************************************************************************/

int
pair_parse (buffer, eq_list, ne_list)

char           *buffer;
VALUE_PAIR    **eq_list;
VALUE_PAIR    **ne_list;

{
	u_char          tag = 0;
	int             len;
	int             mode;
	int             rc;
	UINT4		lvalue;
	char           *valptr;
	DICT_ATTR      *attr;
	DICT_VALUE     *dval;
	VALUE_PAIR     *pair;
	VALUE_PAIR    **to_list = (VALUE_PAIR **) NULL; /* eq_list or ne_list */
	struct tm      *tm;
	time_t          timeval;
	char            attrstr[AUTH_ID_LEN];
	char            buf[AUTH_STRING2_LEN];
	char            valstr[AUTH_STRING2_LEN];
	static char    *func = "pair_parse";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	mode = PARSE_MODE_NAME;
	while (*buffer != '\n' && *buffer != '\0')
	{
		if (*buffer == ' ' || *buffer == '\t' || *buffer == ',')
		{
			buffer++;
			continue;
		}

		switch (mode)
		{
		    case PARSE_MODE_NAME:		/* Attribute Name */
			fieldcpy (attrstr, &buffer, AUTH_ID_LEN);

			if ((attr = dict_attrfind (attrstr))
							== (DICT_ATTR *) NULL)
			{
				return (-1);
			}
			mode = PARSE_MODE_EQUAL;
			break;

		    case PARSE_MODE_EQUAL:		/* Equal sign */
			if ((buffer[0] == '=') &&
				(eq_list != (VALUE_PAIR **) NULL))
			{
				to_list = eq_list;	/* Assign to eq_list */
			}
			else
			{
				if (((buffer[0] == '!') &&
					(buffer[1] == '=')) &&
					(ne_list != (VALUE_PAIR **) NULL))
				{
					to_list = ne_list; /* Assign ne_list */
					buffer++;
				}
				else
				{
					return (-1);
				}
			}

			if (to_list == (VALUE_PAIR **) NULL)
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: no eq_list or ne_list", func);
				return (-2);
			}

			buffer++;			/* point past '=' */
			mode = PARSE_MODE_VALUE;
			break;

		    case PARSE_MODE_VALUE:		/* Value */
			len = fieldcpy (valstr, &buffer, AUTH_STRING2_LEN);
			valptr = valstr;	/* Start buffer correctly. */

			switch (attr->type)
			{
			    case PW_TYPE_TAG_STR:
				pair = avpair_add_vend_tag (to_list,
							attr->value, valstr,
							len, attr->vendor_id,
							-1);
				break;

#ifdef BINARY_FILTERS
			    case PW_TYPE_FILTER_BINARY:
				if ((rc = filter_binary (buf, valstr)) <= 0)
				{
					return (-1);
				}

				pair = avpair_add_vend (to_list, attr->value,
						buf, rc, attr->vendor_id);
				break;
#endif	/* BINARY_FILTERS */

			    case PW_TYPE_STRING:
				pair = avpair_add_vend (to_list, attr->value,
						valstr, len, attr->vendor_id);
				break;

			    case PW_TYPE_TAG_INT:
			        tag = (u_char) valstr[0];
				valptr++;
				len--;
				/***FALLTHROUGH***/

			    case PW_TYPE_OCTET:
			    case PW_TYPE_SHORT:
			    case PW_TYPE_INTEGER:
				if (isdigit(*valptr) || *valptr == '-')
				{
					lvalue = atoi (valptr); /* parse int */
				}
				else
				{
					if ((dval = dict_valfind (valptr,
							attr->name)) == NULLDV)
					{
						logit (LOG_DAEMON, LOG_ERR,
				 "%s: Attribute '%s' does not have value '%s'",
							func, attr->name,
							valptr);
						return (-1);
					}
					else
					{
						lvalue = dval->dv_value;
					}
				}

				pair = avpair_add_vend_tag (to_list,
							attr->value, &lvalue, 0,
							attr->vendor_id, tag);
				break;

			    case PW_TYPE_IPADDR:  /* Look for n.n.n.n or -n */
				if (isdigit(*valstr) || *valstr == '-')
				{
					if (good_ipaddr (valstr) == 0)
					{
						lvalue = 
						     ntohl(inet_addr (valstr));
					}
					else /* was not dotted quad notation */
					{
						lvalue = atol (valstr);
					}

					pair = avpair_add_vend (to_list,
							attr->value, &lvalue, 0,
							attr->vendor_id);
					break;
				}

				/* Now see if defined value (i.e., "ASSIGN") */
				if ((dval = dict_valfind (valstr, 
					    attr->name)) != (DICT_VALUE *) NULL)
				{
					lvalue = dval->dv_value;
					pair = avpair_add_vend (to_list,
							attr->value, &lvalue, 0,
							attr->vendor_id);
					break;
				}

				/* Neither so must be DNS name. */
				rc = find_host_by_name (&lvalue, valstr);
				if (rc == -1)   /* Name not in our list yet */
				{
					insert_client (valstr, "", "", 0, 0,
						      (VENDOR_LIST *) NULL, 0);
				}
				else
				{
					if (rc == 2) /* Unresolvable DNS name */
					{
						return (-1);
					}
				}

				pair = avpair_add_vend (to_list, attr->value,
						&lvalue, 0, attr->vendor_id);

				/*
				 *	If a DNS name was given, store it
				 *	in strvalue so we can re-resolve
				 *	these names.
				 */
				avpair_string_mod (pair, valstr, -1);
				break;

			    case PW_TYPE_DATE:
				timeval = time (0);
				tm = localtime (&timeval);
				tm->tm_hour = 0;
				tm->tm_min = 0;
				tm->tm_sec = 0;
				user_gettime (valstr, tm);

#ifdef TIMELOCAL
				lvalue = (UINT4) timelocal (tm);
#else	/* TIMELOCAL */
				lvalue = (UINT4) mktime (tm);
#endif	/* TIMELOCAL */

				pair = avpair_add_vend (to_list, attr->value,
						&lvalue, 0, attr->vendor_id);
				break;

			    default:
				return (-1);
			}

			mode = PARSE_MODE_NAME;
			to_list = (VALUE_PAIR **) NULL;
			break;

		    default:
			mode = PARSE_MODE_NAME;
			to_list = (VALUE_PAIR **) NULL;
			break;

		} /* end of main switch on mode */
	} /* end of while */

	return (0);
} /* end of pair_parse () */

#ifdef USR_CCA

/******************************************************************************
 *
 *	Function: parse_dns_nbns
 *
 *	Purpose:  Parse the Primary and Secondary DNS, NBNS IP addresses
 *
 *****************************************************************************/

static char
parse_dns_nbns (temp, pri, sec)

char       **temp;	/* Holds the pointer to the character from where
			 * the caller must resume parsing the current line. */
UINT4        *pri;
UINT4        *sec;

{
	char        *temp1;
	char        *temp2;
	int          len;
	char        *func = "parse_dns_nbns";

	temp1 = strtok (NULL, " \t\n");
	len = strlen (temp1);
	*temp = temp1 + len + 1;

	/* Detecting errors of type 'x.x.x.x,' */
	if (temp1[len - 1] == ',')
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: Error in DNS/NBNS information", func);
		return 1;
	}

	temp2 = strdup (temp1);
	temp1 = strtok (temp2, ",");

	if (isdigit(*temp1))
	{
		*pri = get_ipaddr (temp1);
		if (*pri == 0)
		{
			return 1;
		}
	}
	else
	{
		return 1;
	}

	if ((temp1 = strtok (NULL, "")) != NULL)
	{
		if (isdigit(*temp1))
		{
			*sec = get_ipaddr (temp1);
			if (*sec == 0)
			{
				return 1;
			}
		}
		else
		{
			return 1;
		}
	}

	return 0;
} /* end of parse_dns_nbns () */

#endif	/* USR_CCA */

/*************************************************************************
 *
 *	Function: parse_id
 *
 *	Purpose: Modify an RCS version string such as: $Revision: 1.19 $
 *
 *	Returns: just the version number.
 *
 *************************************************************************/

static char *
parse_id (id)

char           *id;

{
	char             *start;
	char             *ver;
	static char      *func = "parse_id";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Throw away leading whitespace. */
	for ( ; *id && isspace(*id) ; id++)
	{
		; /* continue */
	}

	start = id;

	/* RCS check */
	if (*id == '$')
	{
		for (id++ ; *id && !isspace(*id) ; id++)
		{
			; /* Look for the space after "Revision". */
		}

		/* Move to the version number digits. */
		for ( ; *id && isspace(*id) ; id++)
		{
			; /* continue */
		}

		if (*id == '\0')
		{
			return start;
		}

		ver = id;
		for ( ; *id && !isspace(*id) ; id++)
		{
			; /* Look for the space after the version number. */
		}
		*id == '\0';
		id = ver;
	}

	return id;
} /* end of parse_id () */

/*************************************************************************
 *
 *	Function: read_auth
 *
 *	Purpose: Reads in the realm information from the "authfile" with
 *		 file_pfx corresponding to the given file_ent.  The information
 *		 read is copied to a data structure that is linked to the
 *		 given file_ent.
 *
 *************************************************************************/

static int
read_auth (file_ent, dolog)

FILE_LIST      *file_ent;
int             dolog;

{
	int               count;
	int               error = 0;
	int		  len;
	int               line_no = 0;
	int               ent_cnt = 0;
	int               type_inx;
	int               prot_inx;
	FILE             *authfd;
	char             *name;
	char             *prot;
	char             *type;
	char             *host;
	char             *filter;
	char             *alias_list;
	AUTH_ENTRY       *auth_ent;
	AUTH_ENTRY      **end_ent;
	AUTH_ALIAS_ENTRY *alias_ent;
	DICT_VALUE       *type_val;
	static char      *auth_prots[] =
	{
		PW_PROTTYPES_DFLT,
		PW_PROTTYPES_CHAP,
		PW_PROTTYPES_PW,
		PW_PROTTYPES_HASDOT,
		NULL
	};
	char              buffer[256];
	char              fname[MAXPATHLEN];
	char              last_authfile_id[sizeof (authfile_id)];

#ifdef USR_CCA
	char              dns_nbns_err;
	int               router_type;
	UINT4             pri;
	UINT4             sec;
	UINT4             pri_dns;
	UINT4             sec_dns;
	UINT4             pri_nbns;
	UINT4             sec_nbns;
	UINT4             vpn_id;
	char             *temp;
	char             *vpn_name;
	VPN_INFO         *vpn_x;
	VPN_ROUTER       *router;
	IP_ADDRESS       *pdns;
	char             *temp2;
#endif /* USR_CCA */

	static char      *func = "read_auth";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	sprintf (fname, "%s/%s%s", radius_dir, file_ent->prefix, RADIUS_AUTH);
	if ((authfd = fopen (fname, "r")) == (FILE *) NULL)
	{
		/*
		 * It's okay if a non-prefixed RADIUS_AUTH file doesn't exist,
		 * as long as no CI_AUTHENTICATION_TYPE = REALM entries are
		 * configured in the RADIUS_USERS file.  If <pfx>RADIUS_AUTH
		 * doesn't exist, then the non-prefixed RADIUS_AUTH file will
		 * be used instead.
		 */
		return (0);
	}

	last_authfile_id[0] = '\0';	/* Terminate the id properly. */
	end_ent = &file_ent->auth_list;
	while (fgets (buffer, sizeof (buffer), authfd) != (char *) NULL)
	{
		line_no++;
		if (isspace(*buffer) || (*buffer == COMMENT))
		{
		    /*
		     *	Check for special directives.
		     */
		    if ((buffer[1] == '%') &&
			((name = strtok (buffer + 1, WHITESPACES))
							!= (char *) NULL))
		    {
			if (strcasecmp ("%AUTHFILEID", name) == 0)
			{
			    if ((name = strtok (NULL, "\r\n"))
							!= (char *) NULL)
			    {
				name = parse_id (name);
				strncpy (last_authfile_id, name,
						sizeof (last_authfile_id) - 1);
				last_authfile_id[ sizeof (last_authfile_id) - 1]
									= '\0';
			    }
			}
		    }

		    continue;
		}

		name = strtok (buffer, " ,\t\n\r");

		if (strcmp ("DEFAULT_RADIUS_SERVER", name) == 0)
		{
			if ((host = strtok (NULL, " \"\n\r")) != (char *) NULL)
			{
				strcpy (default_radius_server, host);
			}
			else
			{
				default_radius_server[0] = '\0';
			}		
			continue;
		}
		else
		{
			if (strcmp ("DEFAULT_TACACS_SERVER", name) == 0)
			{
				if ((host = strtok (NULL, " \"\n\r"))
							!= (char *) NULL)
				{
					strcpy (default_tacacs_server, host);
				}
				else
				{
					default_tacacs_server[0] = '\0';
				}		
				continue;
			}
		}

		/* Scan for optional alias list or protocol indicator */

		prot_inx = 0;
		alias_list = NULL;
		type = NULL;

		while (type == NULL && !error)
		{
			prot = strtok (NULL, " ,\t\n\r");
			if (prot == NULL)
			{
				error++;
				continue;
			}


			switch (*prot)
			{
			    case '(': 	/* "(<aliases>)" */
				alias_list = prot;
				if (prot[strlen (prot) - 1] != ')')
				{
					if (strtok (NULL, ")") == NULL)
					{
						error++;
					}
				/* We don't want to break up alias list yet */
					prot[strlen (prot)] = ' ';
				}
				break;

			    case ('-'):	/* "-<protocol>" */
				if (*++prot == '\0')
				{
					error++;
					break;
				}

				for (prot_inx = 0;
				     auth_prots[prot_inx] &&
				       strcmp (prot, auth_prots[prot_inx]) != 0;
				     prot_inx++)
				{
					; /* continue */
				}

				if (!auth_prots[prot_inx])
				{
					error++;
				}
				break;

			    default:
				type = prot;
				break;
			}
		}

		if (error)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Invalid entry in %s at line %d",
				func, fname, line_no);
			continue;
		}

		type_val = dict_valfind (type, "Authentication-Type");
		if (type_val == (DICT_VALUE *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Invalid TYPE '%s' in %s at line %d",
				func, type, fname, line_no);
			continue;
		}
		type_inx = (int) type_val->dv_value;

		host = strtok (NULL, " \t\n\r");
		filter = strtok (NULL, " \t\n\r");

		if ((type_inx == AA_UNIX) || (type_inx == AA_BSD))
		{
			filter = host;
			host = (char *) NULL;
		}
		else if (host == NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Invalid entry in %s at line %d",
				func, fname, line_no);
			continue;
		}

		/*
		 * Check for optional host
		 */
		if (host != (char *) NULL)
		{
			/*
			 * Check to make sure the host name doesn't overflow
			 * the EVENT_ENT.estring[AUTH_ID_LEN] array
			 */

			if ((len = strlen (host)) >= AUTH_ID_LEN)
			{
				logit (LOG_DAEMON, LOG_ERR,
		          "%s: Longer than %d, '%s', len=%d in %s at line %d",
				       func, AUTH_ID_LEN - 1, host, len,
				       fname, line_no);
				continue;
			}
		}

#ifdef USR_CCA
		router_type = 0;

		/*
		 *	Did the last call to strtok() return a NULL?
		 *	If not, was the last token returned the 'filter'
		 *	keyword?  If yes, the next call to strtok()
		 *	should return the filter name.
		 */

		if ((temp = filter) != NULL && strcasecmp (temp, FILTER) == 0)
		{
			if ((filter = strtok (NULL, "\t\n")) == NULL)
			{
				logit (LOG_DAEMON, LOG_ERR,
					"%s: Invalid entry in %s at line %d",
					func, fname, line_no);
				continue;
			}
			temp = strtok (NULL, " \t\n");
		}
		else
		{
			filter = (char *) NULL;
		}

		pri_dns = sec_dns = 0;
		pri_nbns = sec_nbns = 0;
		dns_nbns_err = FALSE;

		while (temp != NULL &&
			((strncmp (temp, "DNS", 3) == 0) ||
			(strncmp (temp, "NBNS", 4) == 0)))
		{
			pri = 0;
			sec = 0;
			if (parse_dns_nbns (&temp2, &pri, &sec) != 0)
			{
				logit (LOG_DAEMON, LOG_ERR,
				  "%s: Invalid entry in %s authfile at line %d",
					func, fname, line_no);
				dns_nbns_err = TRUE;
				break;
			}

			if (strncmp (temp, "DNS", 3) == 0)
			{
				pri_dns = pri;
				sec_dns = sec;
			}
			else
			{
				pri_nbns = pri;
				sec_nbns = sec;
			}
			temp = strtok (temp2, " \t\n");
		}

		if (dns_nbns_err == TRUE)
		{
			continue;
		}

		if (temp != NULL && (type_inx == AA_RAD ||
					type_inx == AA_LOCAL_VPN))
		{
			vpn_id = 0;
			vpn_name = NULL;
			router = (VPN_ROUTER *) NULL;
			if (parse_vpn (temp, &vpn_id, &router,
				&router_type, &vpn_name, type_inx) != 0)
			{
				logit (LOG_DAEMON, LOG_ERR,
				  "%s: Invalid entry in %s authfile at line %d",
					func, fname, line_no);
				/*vpn_id = INVALID_VPN;*/
				router_type = 0;
				continue;
			}
		}
#endif /* USR_CCA */

		if ((auth_ent = (AUTH_ENTRY *) calloc (1, sizeof (AUTH_ENTRY)))
							== (AUTH_ENTRY *) NULL)
		{
			logit (LOG_DAEMON, LOG_ALERT,
				"%s: No memory for auth_ent", func);
			fclose (authfd);
			return (-1);
		}
		auth_ent->parent = (AUTH_ENTRY *) NULL;
		auth_ent->type = type_inx;

		if (is_engine == 1) /* then we are being called by the engine */
		{
#ifdef USR_CCA
			if (type_inx != AA_LOCAL_VPN &&
				authtype_tv[type_inx] == (AATV *) NULL)
#else /* USR_CCA */
			if (authtype_tv[type_inx] == (AATV *) NULL)
#endif /* USR_CCA */
			{
				logit (LOG_DAEMON, LOG_ERR,
				  "%s: Missing AATV for entry on line %d of %s",
					func, line_no, fname);
				fclose (authfd);
				return (-1);
			}
		}

		auth_ent->prot = prot_inx;
		auth_ent->name = set_realm_name (name);
		auth_ent->host = add_string (host, ASIS);
		auth_ent->filter = add_string (filter, ASIS);

#ifdef USR_CCA
		if ((type_inx == AA_RAD || type_inx == AA_LOCAL_VPN) &&
			router_type != 0)
		{
			auth_ent->vpn = (VPN_INFO *)
						get_memory (sizeof (VPN_INFO),
							    func, "VPN_INFO");
			vpn_x = auth_ent->vpn;
			vpn_x->id = vpn_id;
			vpn_x->router = router;
			vpn_x->type = router_type;
			vpn_x->name = add_string (vpn_name, ASIS);
		}
		else
		{
			auth_ent->vpn = (VPN_INFO *) NULL;
		}

		if (pri_dns != 0)
		{
			auth_ent->dns_info = (IP_ADDRESS *)
						get_memory (sizeof (IP_ADDRESS),
							    func, "IP_ADDRESS");
			pdns = auth_ent->dns_info;
			pdns->ipaddr.s_addr = pri_dns;
			pdns->next = (IP_ADDRESS *) NULL;

			if (sec_dns != 0)
			{
				pdns->next = (IP_ADDRESS *)
						get_memory (sizeof (IP_ADDRESS),
							    func, "IP_ADDRESS");
				pdns->next->ipaddr.s_addr = sec_dns;
				pdns->next->next = (IP_ADDRESS *) NULL;
			}
		}
		else
		{
			auth_ent->dns_info = (IP_ADDRESS *) NULL;
		}

		if (pri_nbns != 0)
		{
			auth_ent->nbns_info = (IP_ADDRESS *)
						get_memory (sizeof (IP_ADDRESS),
							    func, "IP_ADDRESS");
			pdns = auth_ent->nbns_info;
			pdns->ipaddr.s_addr = pri_nbns;
			pdns->next = (IP_ADDRESS *) NULL;

			if (sec_nbns != 0)
			{
				pdns->next = (IP_ADDRESS *)
					get_memory (sizeof (IP_ADDRESS),
							func, "IP_ADDRESS");
				pdns->next->ipaddr.s_addr = sec_nbns;
				pdns->next->next = (IP_ADDRESS *) NULL;
			}
		}
		else
		{
			auth_ent->nbns_info = (IP_ADDRESS *) NULL;
		}
#endif	/* USR_CCA */

		auth_ent->next = (AUTH_ENTRY *) NULL;
		*end_ent = auth_ent;
		end_ent = &auth_ent->next;
		ent_cnt++;

		/* Add alias entries if aliases given */
		if (alias_list)
		{
			count = 0;
			alias_list++;
			name = strtok (alias_list, " ,)");
			while (name)
			{
				count++;
				if (count > MAX_AUTH_ALIAS)
				{
					logit (LOG_DAEMON, LOG_ALERT,
					     "%s: FATAL: Too many aliases '%s'",
						func, auth_ent->name);
					fclose (authfd);
					return (-1);
				}
				if ((alias_ent = 
					(AUTH_ALIAS_ENTRY *) calloc 
						(1, sizeof (AUTH_ALIAS_ENTRY)))
						   == (AUTH_ALIAS_ENTRY *) NULL)
				{
					logit (LOG_DAEMON, LOG_ALERT,
				     "%s: FATAL No memory for AUTH_ALIAS_ENTRY",
						func);
					fclose (authfd);
					return (-1);
				}
				alias_ent->name = add_string (name, ASIS);
				alias_ent->parent = auth_ent;
				alias_ent->next = (AUTH_ENTRY *) NULL;
				*end_ent = (AUTH_ENTRY *) alias_ent;
				end_ent = &alias_ent->next;
				name = strtok (NULL, " ,)");
			}
		}
	}
	fclose (authfd);

	if (dolog)
	{
		logit (LOG_DAEMON, LOG_INFO,
			"%s: %s (%s %u entries) read to memory",
			func, fname, last_authfile_id, ent_cnt);
	}

	if ((file_ent->prefix == (char *) NULL) || (*(file_ent->prefix)=='\0'))
	{
		strcpy (authfile_id, last_authfile_id);
	}

	authfile_cnt += ent_cnt;

	return 0;
} /* end of read_auth () */

/*************************************************************************
 *
 *	Function: read_clients
 *
 *	Purpose: Reads in the clients file information.
 *
 *		 Read in all clients file client entries, adding a new
 *		 "CLIENT_ENTRY" to the client_list for each valid client
 *		 entry found.  If the DNS reports there is more than one
 *		 IP address for a given client name, add an entry for each.
 *
 *	Returns: -2, FATAL exit: unable to process some client entry,
 *		 -1, FATAL exit: no file found at first or after many tries,
 *		  0, if no file found and using existing client entries,
 *		  n, count of client entries processed from file.
 *
 *************************************************************************/

static int
read_clients (dolog)

int             dolog;

{
	int             client_type;
	int             ent_cnt = 0;
	int             line_nbr = 0;
	int             rad_ver;
	int             reply_holdtime;
	int             result;
	char           *file_pfx;
	char           *host;
	char           *hostnm;
	char           *secret;
	FILE           *clientfd;
	VENDOR         *vep;
	VENDOR_LIST    *veps = (VENDOR_LIST *) NULL;
	char            buffer[128];
	char            fname[MAXPATHLEN];
	static int      clients_loop = 0;/* prevent infinite loops if no file */
	static char    *func = "read_clients";

	sprintf (fname, "%s/%s", radius_dir, RADIUS_CLIENTS);
	if ((clientfd = fopen (fname, "r")) == (FILE *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: Couldn't open %s for reading",
			func, fname);

		if (old_clients == (CLIENT_ENTRY *) NULL)
		{
			return (-1);	/* FATAL return -- no old clients */
		}

		clients_loop++;
		if (clients_loop >= CLIENT_LOOP) /* prevent infinite looping */
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Giving up on %s after %s tries",
				func, fname, clients_loop);
			return (-1);	/* FATAL return -- no file */
		}

		return (0);	/* quick exit -- use existing entries */
	}

	clients_cnt = 0;	/* Init global counter */
	clients_id[0] = '\0';	/* Init clients id */
	result = -1;		/* Init result code */

	while (fgets (buffer, sizeof (buffer), clientfd) != (char *) NULL)
	{
		line_nbr++;

		if (*buffer == COMMENT)
		{
			/*
			 *	Check for special directives.
			 */
			if ((buffer[1] == '%') &&
				((hostnm = strtok (buffer + 1, WHITESPACES))
							!= (char *) NULL))
			{
				if (strcasecmp ("%CLIENTSID", hostnm) == 0)
				{
					if ((hostnm = strtok (NULL, "\r\n"))
							!= (char *) NULL)
					{
					    hostnm = parse_id (hostnm);
					    strncpy (clients_id, hostnm,
						    sizeof (clients_id) - 1);
					    clients_id[sizeof (clients_id) - 1]
									= '\0';
					}
				}
			}

			continue;
		}

		/* first hostname */
		if ((hostnm = strtok (buffer, " \t\n\r")) == (char *) NULL)
		{
			continue;
		}

		/* and secret field */
		if ((secret = strtok (NULL, " \t\n\r")) == (char *) NULL)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: missing shared secret: %s line %d",
				func, fname, line_nbr);
			continue;
		}
		client_type = 0;
		reply_holdtime = default_reply_holdtime;

		/*
		 *	Set up a default vendor id,
		 *	in case no "type=" is specified.
		 */
		(void) free_vendor_list (veps);
		veps = (VENDOR_LIST *) NULL;

		if (DEFAULT_VENDOR_ID != 0)
		{
			if ((vep = find_vendor_by_id (DEFAULT_VENDOR_ID))
							== (VENDOR *) NULL)
			{
				logit (LOG_DAEMON, LOG_INFO,
					"%s: No DEFAULT_VENDOR_ID '%s'!",
					func, DEFAULT_VENDOR_ID);
			}
			else
			{
				if ((veps = (VENDOR_LIST *) calloc (1,
						sizeof (VENDOR_LIST)))
							!= (VENDOR_LIST *) NULL)
				{
					veps->vep = vep;
					veps->next = (VENDOR_LIST *) NULL;
					vendor_list_mf.m++;
				}
				else
				{
					logit (LOG_DAEMON, LOG_CRIT,
				    "%s: FATAL, calloc(1, %d) for VENDOR_LIST",
						func, sizeof (VENDOR_LIST));
					abort ();
				}
			}
		}

		/*
		 *	Parse NAS type, reply hold time and version.  Allow
		 *	hold time to be optional, before or after NAS type.
		 */
		if ((file_pfx = strtok (NULL, " =\t\n\r")) != (char *) NULL)
		{
			/* Check for "reply_holdtime = <value>" before "type" */
			if (strcasecmp (file_pfx, "reply_holdtime") == 0)
			{
				if ((file_pfx =
					strtok (NULL, " =\t")) != (char *) NULL)
				{
					if ((reply_holdtime =
							atoi (file_pfx)) < 0)
					{
						reply_holdtime = CLEANUP_DELAY;
					}
					file_pfx = strtok (NULL, WHITESPACES);
				}
			}

			/* Check for "type" */
			if (file_pfx && (strcasecmp (file_pfx, "type") == 0))
			{
				if ((client_type =
					get_client_type (line_nbr, &veps)) < 0)
				{
					(void) free_vendor_list (veps);
					veps = (VENDOR_LIST *) NULL;
					continue;
				}
				file_pfx = strtok (NULL, WHITESPACES);
			}

			/* Check for another reply_holdtime */
			if (file_pfx &&
				(strcasecmp (file_pfx, "reply_holdtime") == 0))
			{
				if ((file_pfx =
					strtok (NULL, " =\t")) != (char *) NULL)
				{
					if ((reply_holdtime =
							atoi (file_pfx)) < 0)
					{
						reply_holdtime = CLEANUP_DELAY;
					}
					file_pfx = strtok (NULL, WHITESPACES);
				}
			}
		}

		if (client_type == 0)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Missing required client 'type' : line %d",
				func, line_nbr);
			continue;
		}

		rad_ver = VER1;	/* Assume V1 */
		if (file_pfx || (file_pfx = strtok (NULL, " \t\n\r")) != NULL)
		{
			if (strcasecmp (file_pfx, RADIUS_V1) == 0)
			{
				file_pfx = strtok (NULL, " \t\n\r");
				rad_ver = VER1;
			}
			else if (strcasecmp (file_pfx, RADIUS_V2) == 0)
			{
				file_pfx = strtok (NULL, " \t\n\r");
				rad_ver = VER2;
			}
		}

		/*
		 * Look for "paired" <name1>/<name2> entries.  This type of
		 * entry allows core RADIUS servers to share a common "clients"
		 * file.  If we're name1, name2 is the target name; if we are
		 * name2, then name1 is the target.  If we're neither name,
		 * then this entry isn't of interest to us and we ignore it.
		 */
		host = hostnm;  /* Assume just one name */

		if (strchr (hostnm, '/'))	/* If <name1>/<name2> form */
		{
			strtok (hostnm, "/");
			host = strtok (NULL, " ");
			if (host_is_us (hostnm))
			{
				/* We are 1st name - client is 2nd */
			}
			else
			{
				if (host_is_us (host))
				{
					/* We are 2nd - client is 1st */
					host = hostnm;
				}
				else /* We are neither - no match */
				{
					continue;
				}
			}
		}

		if ((result = insert_client (host, secret, file_pfx,
						reply_holdtime, client_type,
						veps, rad_ver)) < 0)
		{
			break;
		}

		veps = (VENDOR_LIST *) NULL;
		ent_cnt++;
	} /* end of while () */
	fclose (clientfd);

	(void) free_vendor_list (veps);
	veps =(VENDOR_LIST *) NULL;

	if (dolog)
	{
		logit (LOG_DAEMON, LOG_INFO,
			"%s: %s (%s %u entries) read to memory",
			func, fname, clients_id, ent_cnt);
	}

	clients_loop = 0;

	clients_cnt = ent_cnt;

	if (result < 0)
	{
		return (-2);	/* FATAL return -- some client didn't process */
	}

	return ent_cnt;
} /* end of read_clients () */

#define FIND_MODE_NAME	0
#define FIND_MODE_REPLY	1
#define FIND_MODE_SKIP	2
#define FIND_MODE_FLUSH	3

/*************************************************************************
 *
 *	Function: read_users
 *
 *	Purpose: For each entry in a "users" file (with the file prefix
 *		 corresponding to the given file_ent), read all check_items
 *		 and reply_items into a data structure.  Each such
 *		 data structure is linked to the given file_ent.
 *
 *	Returns:  0 = normal (successful) return,
 *		 -1 = error return.
 *
 *************************************************************************/

static int
read_users (file_ent, dolog)

FILE_LIST      *file_ent;
int             dolog;

{
	FILE           *userfd;
	char           *ptr;
	int             mode;
	int             line_nbr = 0;
	int             count = 0;
	USER_ENTRY     *user_ent = (USER_ENTRY *) NULL;
	USER_ENTRY    **end_ent;
	char            buffer[1024];
	char            fname[MAXPATHLEN];

#ifdef USR_CCA
	ADDR_POOL      *apool;
#endif /* USR_CCA */

	static char    *func = "read_users";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/*
	 *	Open the user table
	 */
	sprintf (fname, "%s/%s%s", radius_dir, file_ent->prefix, RADIUS_USERS);
	if ((userfd = fopen (fname, "r")) == (FILE *) NULL)
	{
		/*
		 *  It's ok for prefixed user files to not exist, but
		 *  non-prefixed user file has to exist.
		 */
		if (file_ent->prefix[0] == 0)
		{
			logit (LOG_DAEMON, LOG_ERR,
				"%s: Couldn't open %s for reading",
				func, fname);
			return (-1);
		}
		return (0);
	}

	end_ent = &file_ent->user_list;

	mode = FIND_MODE_NAME;
	while (fgets (buffer, sizeof (buffer), userfd) != (char *) NULL)
	{
		line_nbr++;

		ptr = strchr (buffer, '\0');
		ptr--;
		if (*ptr != '\n') /* Look at character just before NULL */
		{
			logit (LOG_DAEMON, LOG_ERR,
	       "%s: No newline in 256 characters in line %d in users file '%s'",
				func, line_nbr, fname);
			return (-1);
		}

		if (*buffer == COMMENT)
		{
			continue;
		}

		for (;;)
		{
			if (mode == FIND_MODE_NAME)
			{
				if (isspace(*buffer))
				{
					break;  /* to read another line */
				}
				ptr = strtok (buffer, " \t\n\r");

#ifdef USR_CCA
				/* Check for address pools */
				if (strcmp (ptr, ADDRESS_POOL) == 0)
				{
					ptr += strlen (ptr) + 1;

					apool = file_ent->pool_list;

					/* First pool is the 'default' where all
					 * users without Address-Pools (but with
					 * multiple sessions allowed) are put.
					 */

					add_to_addr_pool (&(apool->next),
								ptr, line_nbr);
					break;
				}
#endif	/* USR_CCA */

				if ((user_ent = (USER_ENTRY *)
						calloc (1, sizeof (USER_ENTRY)))
							== (USER_ENTRY *) NULL)
				{
					logit (LOG_DAEMON, LOG_ALERT,
				     "%s: Couldn't allocate USER_ENTRY storage",
						func);
					return (-1);
				}
				user_ent->name = add_string (ptr, ASIS);
				user_ent->check = NULL_VP;
				user_ent->deny = NULL_VP;
				user_ent->reply = NULL_VP;
				user_ent->next = (USER_ENTRY *) NULL;

#ifdef USR_CCA
				user_ent->sessions = 1;
				user_ent->count = 0;
				user_ent->pool_name = DEF_POOL_NAME;
#endif	/* USR_CCA */

				/*
				 * Parse the check values
				 */
				ptr += strlen (ptr) + 1;
				if (pair_parse (ptr, &user_ent->check,
						&user_ent->deny) != 0)
				{
					logit (LOG_DAEMON, LOG_ERR,
				 "%s: %d: Parse error for user %s (check/deny)",
						func, line_nbr, user_ent->name);
					free_user_ent (user_ent);
					user_ent = (USER_ENTRY *) NULL;
				}
				else
				{
					mode = FIND_MODE_REPLY;
				}
				break;  /* to read another line */
			}

			/* Reading reply items */
			if (isspace(*buffer))
			{
				/* Parse the reply values */
				if (pair_parse (buffer, &user_ent->reply,
						(VALUE_PAIR **) NULL) != 0)
				{
					logit (LOG_DAEMON, LOG_ERR,
				      "%s: %d: Parse error for user %s (reply)",
						func, line_nbr, user_ent->name);
					free_user_ent (user_ent);
					user_ent = (USER_ENTRY *) NULL;
					mode = FIND_MODE_NAME;
				}
				break;
			}

#ifdef USR_CCA
			if (check_user (user_ent) != 0)
			{
				free_user_ent (user_ent);
				user_ent = (USER_ENTRY *)NULL;
				mode = FIND_MODE_NAME;
				continue;
			}
#endif	/* USR_CCA */

			/* Start of next entry */
			*end_ent = user_ent;
			end_ent = &user_ent->next;
			user_ent = (USER_ENTRY *) NULL;
			mode = FIND_MODE_NAME;
			count++;
			continue;       /* with same input */
		}
	}
	fclose (userfd);
	if (user_ent != (USER_ENTRY *) NULL)
	{

#ifdef USR_CCA
		if (check_user (user_ent) == 0)
		{
			*end_ent = user_ent;
			count++;
		}
#else	/* USR_CCA */
		*end_ent = user_ent;
		count++;
#endif	/* USR_CCA */

	}

	if (dolog)
	{
		logit (LOG_DAEMON, LOG_INFO,
			"%s: %s (%u entries) read to memory",
			func, fname, count);
	}
	users_cnt += count;

	return (0);
} /* end of read_users () */

/*************************************************************************
 *
 *	Function: return_file_list
 *
 *	Purpose: Returns the private file_list pointer for clients in need.
 *
 *************************************************************************/

FILE_LIST *
return_file_list ()

{
	return (file_list);
} /* end of return_file_list () */

/*************************************************************************
 *
 *	Function: update_clients
 *
 *	Purpose: Updates IP address(es) and expire_time in given
 *		 CLIENT_ENTRY node.
 *
 *************************************************************************/

int
update_clients ()

{
	int                len;
	int                notify_count = 0;
	struct hostent    *hp;
	struct in_addr   **addrlist;
	char              *aliascnt;
	char              *ptr;
	char             **name;
	CLIENT_ENTRY      *client_ent;
	struct sockaddr_in send_sin;
	time_t             cur_time;
	static char       *func = "update_clients";

	dprintf(4, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (dnspid != 0)	/* Already resolving addrs */
	{
		return 0;
	}

	if (spawn_flag > 0)
	{
		if ((dnspid = (int) rad_fork ("radiusd DNS update")) < 0)
		{
			dnspid = 0;
			logit (LOG_DAEMON, LOG_ALERT, "%s: fork: %s",
				func, get_errmsg ());
			return (-1);
		}

		if (dnspid != 0)	/* Parent */
		{
			return 0;
		}
	}
	
	/* ======= Child process code ======= */

	spawn_flag = 0;		/* Ensure that no children are produced. */

	memset ((char *) &send_sin, '\0', sizeof (send_sin));
	send_sin.sin_family = AF_INET;
	send_sin.sin_addr.s_addr = htonl(self_ip[0]);
	send_sin.sin_port = htons(rad_ipc_port);

	cur_time = time (0);
	for (client_ent = client_list;
		client_ent != (CLIENT_ENTRY *) NULL;
		client_ent = client_ent->next)
	{

		dprintf(2, (LOG_AUTH, LOG_DEBUG,
			"%s: expire time = 0x%08X",
			func, client_ent->expire_time));

		dprintf(2, (LOG_AUTH, LOG_DEBUG,
			"%s: current time = 0x%08X", func, time (0)));

		if ((client_ent->expire_time != 0) &&
		   (cur_time < (client_ent->expire_time - dns_address_window )))
		{
			continue;
		}

		hp = gethostbyname (client_ent->hostname);

		ptr = send_buffer;
		*ptr++ = '\0';		/* Only one code for now */
		strcpy (ptr, client_ent->hostname);
		ptr += strlen (ptr) + 1;
		*ptr = '\0';
		aliascnt = ptr++;	/* Save alias pointer for later. */

		if (hp != (struct hostent *) NULL)
		{
			if (hp->h_aliases != (char **) NULL)
			{
				for (name = hp->h_aliases;
					*name != (char *) NULL;
					name++)  
				{
					if (strcmp (client_ent->hostname,
						    *name) != 0)
					{
						(*aliascnt)++; 
						strcpy (ptr, *name);
						ptr += strlen (ptr) + 1; 

						/* Indicate just an alias */
						*ptr++ = 1;
					}
				}
			}

			/* Pass official name last */
			if (strcmp (client_ent->hostname, hp->h_name) != 0)
			{
				(*aliascnt)++; 
				strcpy (ptr, hp->h_name);
				ptr += strlen (ptr) + 1; 
				*ptr++ = 0;	/* Indicate official name */
			}

			if (hp->h_addr_list != (char **) NULL)
			{
				addrlist = (struct in_addr **) hp->h_addr_list;
				while (*addrlist)
				{
					memcpy (ptr, *(char **) addrlist,
						sizeof (struct in_addr));
					ptr += sizeof (struct in_addr);
					addrlist++;
				}
			}
			memset (ptr, '\0', sizeof (struct in_addr)); 
		}
		else /* Extra check for brain-dead gethostbyname() calls */
		{
			if (client_ent->type == IP_NUMERIC)
			{
				struct in_addr temp;

				temp.s_addr =
					ntohl(inet_addr (client_ent->hostname));
				memcpy (ptr, (char *) &temp,
					sizeof (struct in_addr));

				/* Terminate list correctly. */
				ptr += sizeof (struct in_addr);
				memset (ptr, '\0', sizeof (struct in_addr));
			}
			else
			{
				memset (ptr, '\0', sizeof (struct in_addr)); 
				/* Pass error code in packet */
				*(ptr + sizeof (struct in_addr) - 1) = h_errno;

				/* Terminate list correctly. */
				ptr += sizeof (struct in_addr);
				memset (ptr, '\0', sizeof (struct in_addr));
			}
		}

		len = ptr - send_buffer + sizeof (struct in_addr);
		notify_count++;

		dprintf(2, (LOG_DAEMON, LOG_DEBUG,
			"%s: Sendto call number (%d) for client '%s'",
			func, notify_count,
			(client_ent->hostname == (char *) NULL)
					? "?" : client_ent->hostname));

		if (notify_count % DNS_SLEEP == 0)
		{
			sleep (1);
		}

		/* Send it to main process */
		sendto (rad_ipc_aatv->sockfd, send_buffer, len, (int) 0,
			(struct sockaddr *) & send_sin, sizeof (send_sin));

	}

#ifdef USR_CCA
	ptr = send_buffer;
	*ptr = 2;
	len = 1;
	sendto (rad_ipc_aatv->sockfd, send_buffer, len, (int) 0,
		(struct sockaddr *) & send_sin, sizeof (send_sin));
#endif /* USR_CCA */

	if (radius_child == 1)	/* Then this code is in the child process. */
	{
		_exit (0);
	}

	return 0;
} /* end of update_clients () */

/*************************************************************************
 *
 *	Function: update_expire
 *
 *	Purpose: Find the named user and update any expiration
 *		 check-item to the new value.
 *
 *	Remarks: The user was previously found using user_find()
 *		 so we know the user is in memory.
 *
 *	Returns: zero, if the user's expiration has been changed in memory,
 *		 non-zero, otherwise.
 *
 *************************************************************************/

int
update_expire (name, protocol, check_items)

char           *name;
int		protocol;
VALUE_PAIR     *check_items;

{
	int             namelen;
	FILE_LIST      *file_ent;
	VALUE_PAIR     *ckexp;
	VALUE_PAIR     *dbexp;	
	USER_ENTRY     *user_ent;
	VALUE_PAIR     *prot_ent;
	USER_ENTRY     *dflt_ent;
	FILE           *debugout = stdout;
	static char    *func = "update_expire";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if ((file_ent = find_file_ent ((char *) NULL)) == (FILE_LIST *) NULL)
	{
		return (-1);
	}

	dflt_ent = file_list->user_list;

	/*
	 *	If there is no user file for this prefix, then use the
	 *	first entry in file_list as the default file_ent.
	 */
	if ((user_ent = file_ent->user_list) == (USER_ENTRY *) NULL)
	{
		if (dflt_ent == (USER_ENTRY *) NULL)
		{
			/* No file, no change */
			return (-1);
		}

		user_ent = dflt_ent;
	}

	if ((namelen = strlen (name)) == 0)
	{
		return 1;	/* A null name would match every line. */
	}

	for (; user_ent; user_ent = user_ent->next)
	{
		/* Allow match for entry specifying framed protocol
		 * type specified in "protocol".  An entry with a 
		 * matching name but no framed-protocol type check 
		 * item matches unconditionally.  An entry with a 
		 * matching name and a framed-protocol type check 
		 * item must match value in "protocol".
		 */
		if (strcmp (user_ent->name, name) == 0)
		{
			if ((prot_ent = get_vp (user_ent->check,
						PW_FRAMED_PROTOCOL)) == NULL_VP)
			{
				break;
			}

			if (prot_ent->lvalue == protocol)
			{
				break;
			}
		}
	}

	if (!user_ent)		/* rc 1 => User not found */
	{
		return 1;
	}

	/* Find expiration entry in check items. */
	ckexp = get_vp_ci (check_items, CI_EXPIRATION, 0);

	if (ckexp == NULL_VP)
	{
		return 0;	/* Nothing to update */
	}

	/* Find expiration entry in memory. */
	dbexp = get_vp_ci (user_ent->check, CI_EXPIRATION, 0);

	if (dbexp == NULL_VP)	/* Need to add it */
	{
		avpair_add (&user_ent->check, CI_EXPIRATION,
						&(ckexp->lvalue), 0);
	}
	else /* Go ahead and change it */	
	{
	        dbexp->lvalue = ckexp->lvalue;
	}

	if (debug_flag >= 2)
	{
		if (ddt)
		{
			debugout = ddt;
		}
		fprintf (debugout, "Check items:\n");
		debug_list (debugout, user_ent->check);
		fprintf (debugout, "Reply items:\n");
		debug_list (debugout, user_ent->reply);
	}
	return (0);	/* rc 0 => User found */

} /* end of update_expire () */

/*************************************************************************
 *
 *	Function: user_find
 *
 *	Purpose: Find the named user in the users database.  Create the
 *		 set of attribute-value pairs to check and reply with
 *		 for this user from the database.
 *
 *		 Note that the users database can be either in the file
 *		 "RADIUS_USERS" (-u option), in memory (default), or in
 *		 a dbm(3) database.
 *
 *************************************************************************/

int
user_find (pfx, name, prot, check_pairs, deny_pairs, reply_pairs, not_user_file)

char           *pfx;		/* Selects which "users" file to use */
char           *name;
int             prot;
VALUE_PAIR    **check_pairs;
VALUE_PAIR    **deny_pairs;
VALUE_PAIR    **reply_pairs;
int             not_user_file;	/* Look up user in a domain specific file */

{
	int             line_no = 0;
	int             mode;
	int             namelen;
	FILE           *userfd;
	VALUE_PAIR     *check_first = NULL_VP;
	VALUE_PAIR     *deny_first = NULL_VP;
	VALUE_PAIR     *reply_first = NULL_VP;
	VALUE_PAIR     *prot_ent;
	FILE_LIST      *file_ent;
	USER_ENTRY     *user_ent;
	USER_ENTRY     *dflt_ent;
	char           *ptr;
	FILE           *debugout = stdout;
	char            buffer[256];
	char            fname[MAXPATHLEN];
	static char    *func = "user_find";

#ifdef USE_NDBM
	datum           named;
	datum           contentd;
	DBM            *db;
#endif	/* USE_NDBM */

#ifdef USE_DBM
	datum           named;
	datum           contentd;
#endif	/* USE_DBM */

#ifdef USR_CCA
	VALUE_PAIR     *vp;
#endif /* USR_CCA */

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	if (check_pairs != (VALUE_PAIR **) NULL)
	{
		*check_pairs = NULL_VP;
	}

	if (deny_pairs != (VALUE_PAIR **) NULL)
	{
		*deny_pairs = NULL_VP;
	}

	if (reply_pairs != (VALUE_PAIR **) NULL)
	{
		*reply_pairs = NULL_VP;
	}

	if ((namelen = strlen (name)) == 0)
	{
		return 1;	/* A null name would match every line. */
	}

	/*
	 *	First check what type of lookup to do.
	 *	See if user file(s) have been cached in memory.
	 */

	if (not_user_file == 0 &&
		(dflt_ent = file_list->user_list) != (USER_ENTRY *) NULL)
	{
		if ((file_ent = find_file_ent (pfx)) == (FILE_LIST *) NULL)
		{
			return (-1);
		}
		if ((user_ent = file_ent->user_list) == (USER_ENTRY *) NULL)
		{
			/* If no user file for this prefix, ...  */
			/* ... then use first entry in file_list as ... */
			/* ... default file_ent. */
			user_ent = dflt_ent;
		}
		for (; user_ent; user_ent = user_ent->next)
		{
			/* Allow match for entry specifying framed protocol
			 * type specified in "prot".  An entry with a 
			 * matching name but no framed-protocol type check 
			 * item matches unconditionally.  An entry with a 
			 * matching name and a framed-protocol type check 
			 * item must match value in "prot".
			 */

			if (strcmp (user_ent->name, name) == 0)
			{
				if ((prot_ent =
					get_vp (user_ent->check,
						PW_FRAMED_PROTOCOL)) == NULL_VP)
				{
					break;
				}
				if (prot_ent->lvalue == prot)
				{
					break;
				}
			}
		}
		if (!user_ent)	/* rc 1 => User not found and no error */
		{
			return 1;
		}

/*#ifdef USR_CCA */
		/* If the user is given a fixed IP address (i.e. not assigned
		 * from an address pool) set the count field in the user
		 * structure to one.
		 */

/*                if ((vp = get_vp (user_ent->reply,
				PW_FRAMED_IP_ADDRESS)) != NULL_VP)
		{
			if (vp->lvalue != ASSIGN_IP_ADDR)
			{
				user_ent->count = 1;
			}
		}
#endif	* USR_CCA */

		if (debug_flag >= 2)
		{
			if (ddt)
			{
				debugout = ddt;
			}
		}

		/*
		 *	Return the check-items if present, and if requested.
		 */
		if ((check_pairs != (VALUE_PAIR **) NULL) &&
			(user_ent->check != NULL_VP))
		{
			list_copy (&check_first, user_ent->check);
			if (debug_flag >= 2)
			{
				fprintf (debugout, "Check items:\n");
				debug_list (debugout, user_ent->check);
			}
			*check_pairs = check_first;
		}
		else
		{
			list_free (check_first);
			check_first = NULL_VP;
		}

		/*
		 *	Return the deny-items if present, and if requested.
		 */
		if ((deny_pairs != (VALUE_PAIR **) NULL) &&
			(user_ent->deny != NULL_VP))
		{
			list_copy (&deny_first, user_ent->deny);
			if (debug_flag >= 2)
			{
				fprintf (debugout, "Deny items:\n");
				debug_list (debugout, user_ent->deny);
			}
			*deny_pairs = deny_first;
		}
		else
		{
			list_free (deny_first);
			deny_first = NULL_VP;
		}

		/*
		 *	Return the reply-items if present, and if requested.
		 */
		if ((reply_pairs != (VALUE_PAIR **) NULL) &&
			(user_ent->reply != NULL_VP))
		{
			list_copy (&reply_first, user_ent->reply);
			if (debug_flag >= 2)
			{
				fprintf (debugout, "Reply items:\n");
				debug_list (debugout, user_ent->reply);
			}
			*reply_pairs = reply_first;
		}
		else
		{
			list_free (reply_first);
			reply_first = NULL_VP;
		}

		return (0);	/* rc 0 => User found */

	} /* End of cached users file(s) lookup */

	/*
	 * Open the user table.
	 * If prefixed file doesn't exist, use default (non-prefixed) file.
	 */
	for (;;)
	{
		if ((pfx[0] == '/') ||
			(pfx[0] == '.' && pfx[1] == '/') ||
			(pfx[0] == '.' && pfx[1] == '.' && pfx[2] == '/'))
		{
			sprintf (fname, "%s%s", pfx, RADIUS_USERS);
		}
		else /* use compiled in directory */
		{
			sprintf (fname, "%s/%s%s",
				radius_dir, pfx, RADIUS_USERS);
		}

#if defined(USE_DBM) || defined(USE_NDBM)

#ifdef USE_NDBM
		if ((db = dbm_open (fname, O_RDONLY, 0)) == 0)
		{
#endif	/* USE_NDBM */

#ifdef USE_DBM
		if (dbminit (fname) != 0)
		{
#endif	/* USE_DBM */

#else	/* USE_DBM || USE_NDBM */
		if ((userfd = fopen (fname, "r")) == (FILE *) NULL)
		{
#endif	/* USE_DBM || USE_NDBM */

			if (not_user_file == 0 && pfx != NULL && pfx[0] != 0)
			{
				pfx = NULL;
				continue;
			}

			logit (LOG_DAEMON, LOG_ERR,
				"%s: Couldn't open %s for reading",
				func, fname);
			return (-1);
		}
		break;
	}

#if defined(USE_DBM) || defined(USE_NDBM)
/*
 * Note that the DBM feature currently does not support protocol specific user
 * entry matching.  It could be done by appending a protocol type to the
 * name when building the database and looking for that modified name here.
 * This seems like a non-trivial task and we don't use the DBM feature
 * here so I'll leave this to someone else to implement.
 */
	named.dptr = name;
	named.dsize = namelen;

#ifdef	USE_NDBM
	contentd = dbm_fetch (db, named);
#else	/* USE_NDBM */
	contentd = fetch (named);
#endif	/* USE_NDBM */

	if (contentd.dsize == 0)
	{
		named.dptr = "DEFAULT";
		named.dsize = strlen ("DEFAULT");

#ifdef	USE_NDBM
		contentd = dbm_fetch (db, named);
#else	/* USE_NDBM */
		contentd = fetch (named);
#endif	/* USE_NDBM */

		if (contentd.dsize == 0)
		{
#ifdef	USE_NDBM
			dbm_close (db);
#else	/* USE_NDBM */
			dbmclose ();
#endif	/* USE_NDBM */
			return (-1);
		}
	}

	/*
	 * Parse the check values
	 */
	ptr = contentd.dptr;
	contentd.dptr[contentd.dsize] = '\0';

	if (pair_parse (ptr, &check_first, &deny_first) != 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: Parse error for user %s in %s (DBM check)",
			func, name, fname);
		list_free (check_first);
		list_free (deny_first);

#ifdef	USE_NDBM
		dbm_close (db);
#else	/* USE_NDBM */
		dbmclose ();
#endif	/* USE_NDBM */
		return (-1);
	}
	while (*ptr != '\n' && *ptr != '\0')
	{
		ptr++;
	}
	if (*ptr != '\n')
	{
		list_free (check_first);
#ifdef	USE_NDBM
		dbm_close (db);
#else	/* USE_NDBM */
		dbmclose ();
#endif	/* USE_NDBM */
		return (-1);
	}
	ptr++;

	/*
	 * Parse the reply values
	 */
	if (pair_parse (ptr, &reply_first, (VALUE_PAIR **) NULL) != 0)
	{
		logit (LOG_DAEMON, LOG_ERR,
			"%s: Parse error for user %s in %s (DBM reply)",
			func, name, fname);
		list_free (check_first);
		list_free (deny_first);
		list_free (reply_first);

#ifdef	USE_NDBM
		dbm_close (db);
#else	/* USE_NDBM */
		dbmclose ();
#endif	/* USE_NDBM */
		return (-1);
	}
#ifdef	USE_NDBM
	dbm_close (db);
#else	/* USE_NDBM */
	dbmclose ();
#endif	/* USE_NDBM */

#else /* USE_DBM || USE_NBDM */

	mode = FIND_MODE_NAME;

	while (fgets (buffer, sizeof (buffer), userfd) != (char *) NULL)
	{
		line_no++;

		if (*buffer == COMMENT)
		{
			continue;
		}

		if (mode == FIND_MODE_NAME)
		{
			/* Find the entry starting with the users name. */
			if (strncmp (buffer, name, namelen) == 0 &&
			    (buffer[namelen] == ' ' || buffer[namelen] == '\t'))
			{
				ptr = &buffer[namelen];

				/* Parse the check values */
				if (pair_parse (ptr, &check_first, &deny_first)
									!= 0)
				{
					logit (LOG_DAEMON, LOG_ERR,
			 "%s: Parse error for user %s in %s at line %d (check)",
						func, name, fname, line_no);
					list_free (check_first);
					list_free (deny_first);
					fclose (userfd);
					return (-1);
				}

				/*
				 * Allow match for entry specifying framed
				 * protocol type specified in "prot".  An
				 * entry with matching name but no
				 * framed-protocol type check item matches
				 * unconditionally.  An entry with matching
				 * name and a framed-protocol type check item
				 * must match value in "prot".
				 */
				if ((prot_ent =
					get_vp (check_first,
						PW_FRAMED_PROTOCOL)) != NULL_VP)
				{
					if (prot_ent->lvalue != prot)
					{
						list_free (check_first);
						list_free (deny_first);
						continue;
					}
				}
				mode = FIND_MODE_REPLY;
			}
		}
		else
		{
			if (*buffer == ' ' || *buffer == '\t')
			{
				/*
				 * Parse the reply values
				 */
				if (pair_parse (buffer, &reply_first,
						(VALUE_PAIR **) NULL) != 0)
				{
					logit (LOG_DAEMON, LOG_ERR,
			 "%s: Parse error for user %s in %s at line %d (reply)",
						func, name, fname, line_no);
					list_free (check_first);
					list_free (deny_first);
					list_free (reply_first);
					fclose (userfd);
					return (-1);
				}
			}
			else /* We are done */
			{
				break;
			}
		}
	}
	fclose (userfd);

	/* Update the callers pointers */
	if (mode == FIND_MODE_NAME)
	{
		return 1;		/* No error and no match */
	}
#endif	/* USE_DBM || USE_NBDM */

	/*
	 *	Return the check items, deny items, and reply items.
	 */
	if (debug_flag >= 2)
	{
		if (ddt)
		{
			debugout = ddt;
		}

		fprintf (debugout, "Check items:\n");
		debug_list (debugout, check_first);
		fprintf (debugout, "Deny items:\n");
		debug_list (debugout, deny_first);
		fprintf (debugout, "Reply items:\n");
		debug_list (debugout, reply_first);
	}

	if (check_pairs != (VALUE_PAIR **) NULL)
	{
		*check_pairs = check_first;
	}
	else
	{
		list_free (check_first);
		check_first = NULL_VP;
	}

	if (deny_pairs != (VALUE_PAIR **) NULL)
	{
		*deny_pairs = deny_first;
	}
	else
	{
		list_free (deny_first);
		deny_first = NULL_VP;
	}

	if (reply_pairs != (VALUE_PAIR **) NULL)
	{
		*reply_pairs = reply_first;
	}
	else
	{
		list_free (reply_first);
		reply_first = NULL_VP;
	}

	return (0);

} /* end of user_find () */

static char * months[] =
		{
			"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
		};

/*************************************************************************
 *
 *	Function: user_gettime
 *
 *	Purpose: Turns printable string into correct tm struct entries.
 *
 *************************************************************************/

void
user_gettime (valstr, tm)

char           *valstr;
struct tm      *tm;

{
	int             i;
	static char    *func = "user_gettime";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	/* Get the month */
	for (i = 0; i < 12; i++)
	{
		if (strncmp (months[i], valstr, 3) == 0)
		{
			tm->tm_mon = i;
			i = 13;
		}
	}

	tm->tm_mday = atoi (&valstr[4]); /* Get the Day */

	tm->tm_year = atoi (&valstr[7]) - 1900; /* Now the year */

} /* end of user_gettime () */

/*************************************************************************
 *
 *	Function: user_update
 *
 *	Purpose: Updates a user in the database.  Replaces the original
 *		 entry with the name, the list of check items, and the
 *		 list of reply items which are supplied.
 *
 *************************************************************************/

int
user_update (name, user_check, user_deny, user_reply)

char           *name;
VALUE_PAIR     *user_check;
VALUE_PAIR     *user_deny;
VALUE_PAIR     *user_reply;

{
	FILE           *oldfd;
	FILE           *userfd;
	char            buffer[256];
	char            buffer1[256];
	int             namelen;
	int             mode;
	static char    *func = "user_update";

	dprintf(2, (LOG_AUTH, LOG_DEBUG, "%s: entered", func));

	sprintf (buffer, "%s/%s", radius_dir, RADIUS_USERS);
	sprintf (buffer1, "%s/%s", radius_dir, RADIUS_HOLD);

	/* Move the user table to a temporary location */
	if (rename (buffer, buffer1) != 0)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: Couldn't rename %s",
			func, buffer);
		return (-1);
	}

	/* Open the old user file (using the temporary name */
	if ((oldfd = fopen (buffer1, "r")) == (FILE *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: Couldn't open %s for reading",
			func, buffer1);
		exit (-9);
	}

	/* Open the new user file */
	if ((userfd = fopen (buffer, "w")) == (FILE *) NULL)
	{
		logit (LOG_DAEMON, LOG_ERR, "%s: Couldn't open %s for writing",
			func, buffer);
		exit (-9);
	}

	mode = FIND_MODE_NAME;
	namelen = strlen (name);

	/* Copy the old to the new, only recreating the changed user */
	while (fgets (buffer, sizeof (buffer), oldfd) != (char *) NULL)
	{
		if (mode == FIND_MODE_NAME)
		{
			if ((strncmp (buffer, name, namelen) == 0 &&
			   (buffer[namelen] == ' ' || buffer[namelen] == '\t')))
			{

				/* Write out new information */
				fprintf (userfd, "%s\t", name);

				/* Write out check items first. */
				while (user_check != NULL_VP)
				{
					fprint_attr_val (userfd, user_check);
					if ((user_check->next != NULL_VP) ||
						(user_deny != NULL_VP))
					{
						fprintf (userfd, ", ");
					}
					user_check = user_check->next;
				}

				/* Write out deny items on same line. */
				while (user_deny != NULL_VP)
				{
					fprint_attr_val_sws (userfd, user_deny,
								LF_NOT_EQUAL);
					if (user_deny->next != NULL_VP)
					{
						fprintf (userfd, ", ");
					}
					user_deny = user_deny->next;
				}

				/* Reply items go next -- one per line. */
				fprintf (userfd, "\n\t");
				while (user_reply != NULL_VP)
				{
					fprint_attr_val (userfd, user_reply);
					if (user_reply->next != NULL_VP)
					{
						fprintf (userfd, ",\n\t");
					}
					user_reply = user_reply->next;
				}

				fprintf (userfd, "\n");
				mode = FIND_MODE_SKIP;
			}
			else
			{
				fputs (buffer, userfd);
			}
		}
		else if (mode == FIND_MODE_SKIP)
		{
			if (*buffer != ' ' && *buffer != '\t')
			{
				fputs (buffer, userfd);
				mode = FIND_MODE_FLUSH;
			}
		}
		else
		{
			fputs (buffer, userfd);
		}
	}
	fclose (oldfd);
	fclose (userfd);
	return (0);
} /* end of user_update () */
