;;; lj-login.el --- login lj protocol mode support for ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(defcustom lj-update-friends-on-login-flag t
  "*When non-nil, ljupdate will update your cached friends list on login."
  :type '(choice (const :tag "Update friends on login" t)
                 (const :tag "Don't update friends on login" nil))
  :group 'ljupdate)

;;;###autoload
(defun lj-profile-update-friends (profile)
  "Update PROFILE's friends list."
  (interactive (list (lj--read-profile)))
  (let* ((system (lj-profile-system profile))
         (request (lj--make-request 'getfriends '()))
         (response (lj--send-request profile request)))
    (when (car response)
      (lj-profile-set-friends profile '())
      (let* ((friend-count-string
              (cdr (assoc "friend_count" (cdr response))))
             (friend-count
              (if (and friend-count-string
                       (stringp friend-count-string))
                  (string-to-number friend-count-string)
                0))
             (count 0)
             friend-user
             friend-name
             friend-fg
             friend-bg
             friend-type
             userobj)
        (while (< count friend-count)
          (setq count (+ count 1))
          (setq friend-user
                (cdr (assoc (concat "friend_" (int-to-string count) "_user")
                            (cdr response)))

                friend-type
                (cdr (assoc (concat "friend_" (int-to-string count) "_type")
                            (cdr response)))

                friend-fg
                (cdr (assoc (concat "friend_" (int-to-string count) "_fg")
                            (cdr response)))

                friend-bg
                (cdr (assoc (concat "friend_" (int-to-string count) "_bg")
                            (cdr response)))

                friend-name
                (cdr (assoc (concat "friend_" (int-to-string count) "_name")
                            (cdr response))))

          (setq userobj (lj-system-get-user-create system friend-user))
          (lj-user-set-real-name userobj friend-name)
          (lj-user-set-type userobj
                            (if friend-type
                                (intern friend-type)
                              'user))

          (lj-profile-add-friend
           profile
           (lj-friend-create friend-user
                             count
                             friend-fg
                             friend-bg)))
        (setq lj--startup-modified-p t)
        (lj--save-startup-file)))))

;;;###autoload
(defun lj-profile-login (profile)
  "Login to PROFILE."
  (interactive (list (lj--read-profile)))
  (lj--initialize)

  (lj--message 6 "Username to send to server is \"%s\"."
               (lj-profile-username profile))

  (let ((system (lj-profile-system profile)))
    (let ((login-request
           (lj--make-request 'login
             `((getpickws     . t)
               (clientversion . ,lj--client-version)
               (getmoods      . ,(apply
                                  'max
                                  (cons
                                   0
                                   (mapcar
                                    'cdr
                                    (lj-system-moods system))))))))
          login-response)

      (lj--message 9 "Login request is \"%s\"" login-request)

      (setq login-response (lj--send-request profile login-request))

      (put profile 'lj-logged-in-p (car login-response))

      (when (get profile 'lj-logged-in-p)
        ;; Get the list of valid pictures. This seems to break on
        ;; DeadJournal, so I'm conditionalizing the code on using
        ;; LiveJournal. It would be cool if this were somehow
        ;; smarter.
        (when (eq system 'livejournal)
          (let ((count 0)
                (maxcount (string-to-number
			   (or (cdr (assoc "pickw_count" login-response)) "0")))
                picklist)
            (while (< count maxcount)
              (setq count (+ count 1))
              (setq picklist (cons (cons (cdr (assoc
                                               (concat "pickw_"
                                                       (int-to-string count))
                                               login-response))
                                         count)
                                   picklist)))
            (setq lj--startup-modified-p t)
            (lj-profile-set-pictues profile picklist))

          (let* ((count 0)
                 (access_count (assoc "access_count" login-response))
                 (maxcount (if (consp access_count)
                               (string-to-number (cdr access_count))
                             nil))
                 commlist)
            (when maxcount
              (while (< count maxcount)
                (setq count (+ count 1))
                (setq commlist (cons (cons (cdr (assoc
                                                 (concat "access_"
                                                         (int-to-string count))
                                                 login-response))
                                           count)
                                     commlist)))
              (setq lj--startup-modified-p t)
              (lj-profile-set-communities profile commlist))))

        (let ((real-name (cdr (assoc "name" login-response))))
          (when real-name
            (setq lj--startup-modified-p t)
            (lj-profile-set-real-name profile real-name)))

	;; Get any new friend groups
	(let ((count 0)
	      (maxcount (string-to-number
			 (cdr (assoc "frgrp_maxnum" login-response))))
	      fglist
	      fgname)
	  (while (< count maxcount)
	    (setq count (+ count 1))
	    (setq fgname (cdr (assoc (concat "frgrp_"
					     (int-to-string count)
					     "_name")
				     login-response)))
	    (if fgname
		(setq fglist (cons (cons fgname count)
				   fglist))))
	  (setq lj--startup-modified-p t)
	  (lj-profile-set-friend-groups profile fglist))

        ;; Get any new moods
        (let* ((count 0)
               (mood_count (cdr (assoc "mood_count" login-response)))
               maxcount moodlist)
          (when mood_count
            (setq maxcount (string-to-number mood_count))
            (setq lj--startup-modified-p t)
            (while (< count maxcount)
              (setq count (+ count 1))
              (setq moodlist
                    (cons (cons
                           (cdr (assoc (concat "mood_"
                                               (int-to-string count)
                                               "_name")
                                       login-response))
                           (string-to-number
                            (cdr (assoc (concat "mood_"
                                                (int-to-string count)
                                                "_id")
                                        login-response))))
                          moodlist)))
            (lj-system-set-moods system
                                 moodlist)))

        ;; Check to see if we're allowed to use the fast servers
        (lj-profile-set-fast-p profile (assoc "fastserver" login-response))

        ;; Handle it when the livejournal server wants to tell the
        ;; user something.
        (let ((message (assoc "message" login-response)))
          (when (and (consp message)
                     (cdr message))
            (lj--message 2 "Server: %s" (cdr message)))))))

  (when lj-update-friends-on-login-flag
    (lj-profile-update-friends profile))

  (lj--save-startup-file))

(provide 'lj-login)
;;; lj-login.el ends here
