;;; lj-post.el --- postevent protocol support for ljupdate

;; Copyright (C) 2001, 2002 Edward O'Connor <ted@oconnor.cx>
;; Copyright (C) 2002 Theodore Ts'o <tytso@mit.edu>

;; Maintainer: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;;; History:

;;; Code:

(require 'ljcompat)

(defun lj--power-of-two (pow)
  "Calculates 2 ^ POW for integers between 0 and 30"
  (cond ((< pow 0)
         (signal 'args-out-of-range pow))
	((< pow 27)
	 (lsh 1 pow))
	((eq pow 27)
	 "134217728")
	((eq pow 28)
	 "268435456")
	((eq pow 29)
	 "536870912")
	((eq pow 30)
	 "1073741824")
	(t
         (signal 'args-out-of-range pow))))

(defun lj--post-1 (profile-string)
  "Post this update to the profile named by PROFILE-STRING."
  ;; Make sure that we've logged in before posting.
  (let* ((profile (intern profile-string))
	 (friendgroup
	  (cdr (assoc access
		      (lj-profile-friend-groups profile))))
         (system (lj-profile-system profile)))
    (progn
      (unless (get profile 'lj-logged-in-p)
        (lj-profile-login profile))
      (let* ((moodid (or (cdr (assoc
                               mood
                               (lj-system-moods system)))
                         -1))

             (fields '())
             (request nil))

        (setq fields `((year . ,(cdr (assoc 'year date-time-alist)))
                       (mon  . ,(cdr (assoc 'mon  date-time-alist)))
                       (day  . ,(cdr (assoc 'day  date-time-alist)))
                       (hour . ,(cdr (assoc 'hour date-time-alist)))
                       (min  . ,(cdr (assoc 'min  date-time-alist)))))

        (when (not (string-equal music ""))
          (add-to-list 'fields `(prop_current_music . ,music)))

        (cond ((not (= moodid -1))
               (add-to-list 'fields `(prop_current_moodid . ,moodid)))
              ((not (string-equal mood ""))
               (add-to-list 'fields `(prop_current_mood . ,mood))))

        (when (not (string-equal subject ""))
          (add-to-list 'fields `(subject . ,subject)))

        (when (not (string-equal community ""))
          (add-to-list 'fields `(usejournal . ,community)))

        (when (not (string-equal picture ""))
          (add-to-list 'fields `(prop_picture_keyword . ,picture)))

        (when (string-match "[Nn][Oo]" commentsp)
          (add-to-list 'fields '(prop_opt_nocomments . t)))

        (when (string-match "[Nn][Oo]" emailp)
          (add-to-list 'fields '(prop_opt_noemail . t)))

        (cond (friendgroup
               (add-to-list 'fields '(security . "usemask"))
               (add-to-list 'fields `(allowmask . (lj--power-of-two friendgroup))))
              ((string-match "private" access)
               (add-to-list 'fields '(security . "private")))
              ((string-match "public" access)
               (add-to-list 'fields '(security . "public")))
              ((string-match "friends" access)
               (add-to-list 'fields '(security . "usemask"))
               (add-to-list 'fields '(allowmask . 1)))
              (t
               (add-to-list 'fields `(security . ,lj-default-access-level))))

        (add-to-list 'fields
                     `(event . ,(ljc-string-as-unibyte
                                 (buffer-substring-no-properties
                                  (progn
                                    (goto-char (ljc-mail-header-end))
                                    (forward-line 1)
                                    (point))
                                  (point-max)))))

        (setq request (lj--make-request 'postevent fields))

        (lj--send-request profile request)))))


;;;###autoload
(defun lj-post ()
  "Post this LiveJournal update,."
  (interactive)
  (lj--initialize)

  (lj--message 6 "current buffer is %s" (current-buffer))

  (let
      ((retval
        (car
         (let ((post-data nil))
           (save-excursion
             (mark-whole-buffer)
             (setq post-data (buffer-substring (point) (mark))))

           (with-temp-buffer
             (insert post-data)
             (lj-update-mode)

             ;; Reformat the paragraphs so that they are on a
             ;; single line; LJ will convert newlines to <BR>, so
             ;; we want to get rid of them before that happens.
             (let ((fill-column 65536)
                   (sentence-end-double-space t))
               (fill-region
                (progn (goto-char (ljc-mail-header-end))
                       (forward-line 1)
                       (point))
                (point-max) 'left t))

             (let* ((subject (lj--fetch-header "Subject"))
                    (profiles
                     (split-string (lj--fetch-header "X-LJ-Profile") ", "))
                    (music (lj--fetch-header "X-LJ-Music"))
                    (mood (lj--fetch-header "X-LJ-Mood"))
                    (picture (lj--fetch-header "X-LJ-Picture"))
                    (community (lj--fetch-header "X-LJ-Community"))
                    (commentsp (lj--fetch-header "X-LJ-Allow-Comments"))
                    (emailp (lj--fetch-header "X-LJ-Receive-Mail-Notification"))
                    (access (lj--fetch-header "X-LJ-Access"))
		    (date-time-alist (lj--fetch-date-and-time-alist)))
               (ljc-mail-sendmail-undelimit-header)
               (mail-do-fcc (save-excursion
                              (ljc-mail-header-end)
                              (point-marker)))
               (mapcar 'lj--post-1 profiles)))))))
    (bury-buffer (current-buffer))
    retval))

;;;###autoload
(defun lj-post-and-exit ()
  "Post this update, then kill the buffer."
  (interactive)
  (lj--initialize)
  (when (car (lj-post))
    (kill-buffer (current-buffer))))

(provide 'lj-post)

;;; lj-post.el ends here
