;;; lj-write.el --- composition of journal entries support

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'sendmail)
(require 'thingatpt)

(condition-case nil
    (require 'overlay)
  (error nil))

(require 'ljcompat)
(require 'lj-url)

(defvar lj-update-mode-map nil
  "Keyboard mapping used by `lj-update-mode'.")

(defvar lj-update-mode-abbrev-table nil
  "Local abbrev table for lj-update mode.")

(define-abbrev-table 'lj-update-mode-abbrev-table ())

(defcustom lj-archive-file-name "~/ljarchive.mbox"
  "*Mailbox into which to archive LiveJournal updates.
Set to nil if you don't want this to happen."
  :type '(choice file (const :tag "None" nil))
  :group 'ljupdate)

(defcustom lj-update-mode-hook nil
  "Hooks to be executed when `lj-update-mode' is entered."
  :type 'hook
  :group 'ljupdate)

(unless lj-update-mode-map
  (setq lj-update-mode-map (make-sparse-keymap))
  (set-keymap-parent lj-update-mode-map text-mode-map)
  (define-key lj-update-mode-map "\C-i" 'lj--header-complete)
  (define-key lj-update-mode-map "\C-c?" 'describe-mode)
  (define-key lj-update-mode-map "\C-c\C-c" 'lj-post-and-exit)
  (define-key lj-update-mode-map "\C-c\C-f" 'lj-insert-lj-user)
  (define-key lj-update-mode-map "\C-c\C-s" 'lj-post))

(defun lj--read-lj-user ()
  (let* ((user-alist (lj--get-user-alist))
         (lj-user (completing-read "User: "
                                   (mapcar (lambda (cell)
                                             (cons (car cell)
                                                   (car cell)))
                                           user-alist)
                                   nil
                                   nil)))
    (let ((entry (assoc lj-user user-alist)))
      (if entry
          (cdr entry)
        (lj-user-create lj-user)))))

(defun lj--make-lj-user-jump-function (profile friend)
  (eval
   `(lambda ()
      (interactive)
      (let* ((profile ',profile)
             (system (lj-profile-system profile)))
        (browse-url
         (lj-url-friend-entries profile friend))))))

(defun lj-insert-lj-user (arg)
  (interactive "P")
  (let* ((curpoint (point))
         (profile
          (intern
           (car (split-string (lj--fetch-header "X-LJ-Profile") ", "))))
         (user (lj--read-lj-user))
         (friend (lj-profile-get-friend profile (lj-user-username user)))
         overlay
         (jump (lj--make-lj-user-jump-function profile user)))
    (goto-char curpoint)
    (when (and arg (lj-user-real-name user))
      (insert (lj-user-real-name user) " ("))
    (insert (format "<lj user=\"%s\">"
                    (lj-user-username user)))
    (when (and arg (lj-user-real-name friend))
      (insert ")"))
    (when (featurep 'overlay)
      (setq overlay (make-overlay curpoint
                                  (point)
                                  (current-buffer)
                                  t
                                  nil))
      (when lj-use-friend-faces-flag
        (overlay-put overlay
                     'face
                     (if (and friend (not (featurep 'xemacs)))
                         (list :background (lj-friend-bg friend)
                               :foreground (lj-friend-fg friend))
                       'font-lock-variable-name-face)))

      (overlay-put overlay 'evaporate t)

      (overlay-put overlay
                   'mouse-face 'highlight)

      (overlay-put overlay
                   'help-echo
                   (format "mouse-2: browse %s's journal"
                           (lj-user-username user)))
      (overlay-put overlay
                   'keymap
                   `(keymap (3 keymap (13 . ,jump))
                            (mouse-2 . ,jump))))))

;; This helper function goes through a list and if an element
;; contains a space, it will add surrounding double quotes.
(defun lj--space-quote-list (list)
  "Quote spaces appearing in elements of LIST."
  (lj--initialize)
  (mapcar (lambda (arg)
            (if (string-match " " arg)
                (concat "\"" arg "\"")
              arg))
          list))

;; Helper function which handles the generic completion
;; functionality This should really be changed to use the normal
;; completion facility.
(defun lj--do-complete (range alist)
  "Attempt to complete over RANGE from ALIST."
  (let* ((thing (if range (buffer-substring-no-properties (car range) (cdr range))))
	 (completion (try-completion (if thing thing "") alist)))
    (unless thing
      (setq thing ""))
    (when (and (car range) (cdr range))
      (delete-region (car range) (cdr range)))
    (cond ((equal completion t)
	   (insert thing))
	  ((stringp completion)
	   (insert completion)
	   (let ((completions (all-completions thing alist)))
	     (when (> (length completions) 1)
	       (lj--message 2 "Possible completions: %s"
                            (lj--space-quote-list completions)))))
	  (t (beep)
             (insert thing)
             (lj--message 2 "No completion for \"%s\"!" thing)))))

(defun lj--get-mood-alist ()
  "Return an alist of available moods."
  (save-excursion
    (let* ((profiles
            (split-string (lj--fetch-header "X-LJ-Profile") ", "))
           (systems (mapcar (lambda (profile-string)
                              (lj-profile-system
                               (intern profile-string)))
                            profiles))
           (allsame (car systems)))
      (mapc (lambda (system)
              (unless (eq system allsame)
                (setq allsame nil)))
            (cdr systems))
      (if allsame
          (lj-system-moods allsame)
        nil))))

(defun lj--get-picture-alist ()
  "Return an alist of available pictures."
  (save-excursion
    (let* ((profiles
            (mapcar (lambda (profile)
                      (intern profile))
                    (split-string (lj--fetch-header "X-LJ-Profile") ", ")))
           (allsame (car profiles)))
      (mapc (lambda (profile)
              (unless (eq profile allsame)
                (setq profile nil)))
            (cdr profiles))
      (if allsame
          (lj-profile-pictures allsame)
        nil))))

(defun lj--get-communities-alist ()
  "Return an alist of available communitites."
  (save-excursion
    (let* ((profiles
            (mapcar (lambda (profile)
                      (intern profile))
                    (split-string (lj--fetch-header "X-LJ-Profile") ", ")))
           (allsame (car profiles)))
      (mapc (lambda (profile)
              (unless (eq profile allsame)
                (setq profile nil)))
            (cdr profiles))
      (if allsame
          (lj-profile-communities allsame)
        nil))))

(defun lj--get-user-alist ()
  "Return an alist of users available for tab completion."
  (save-excursion
    (let* ((profiles
            (split-string (lj--fetch-header "X-LJ-Profile") ", "))
           (systems (mapcar (lambda (profile-string)
                              (lj-profile-system
                               (intern profile-string)))
                            profiles))
           (allsame (car systems)))
      (mapc (lambda (system)
              (unless (eq system allsame)
                (setq allsame nil)))
            (cdr systems))
      (if allsame
          (lj-system-users allsame)
        nil))))

(defun lj--get-friend-groups-alist ()
  "Return an alist of available friend groups."
  (save-excursion
    (let* ((profiles
            (mapcar (lambda (profile)
                      (intern profile))
                    (split-string (lj--fetch-header "X-LJ-Profile") ", ")))
           (allsame (car profiles)))
      (mapc (lambda (profile)
              (unless (eq profile allsame)
                (setq profile nil)))
            (cdr profiles))
      (if allsame
          (lj-profile-friend-groups allsame)
        nil))))

(defun lj--header-complete ()
  "Handle completion of the various headers."
  (interactive)
  (lj--initialize)
  (let* (range
	 thing
	 alist
	 complete
	 eoh
	 (completion-ignore-case t)
	 (the-point (point))
	 (eol-point (save-excursion
		      (end-of-line)
		      (if (re-search-backward "[^ 	]" the-point t)
			  (+ the-point 1)
			the-point))))
    (beginning-of-line)
    (cond ((re-search-forward "^X-LJ-Mood:" eol-point t)
           (setq eoh (point))
           (goto-char the-point)
           (setq range (bounds-of-thing-at-point 'word)
                 alist (lj--get-mood-alist)
                 complete t))
          ((re-search-forward "^X-LJ-Profile:" eol-point t)
           (setq eoh (point))
           (goto-char the-point)
           (setq range (bounds-of-thing-at-point 'symbol)
                 alist (lj--profile-alist)
                 complete t))
          ((re-search-forward "^X-LJ-Music:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^         ]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
               music (or (lj--get-music) "")
               alist (list (cons music music))
	       complete t))
          ((re-search-forward "^X-LJ-Allow-Comments:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^ 	]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
                 alist '(("yes" . "yes")
                         ("no" . "no"))
                 complete t))
          ((re-search-forward "^X-LJ-Receive-Mail-Notification:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^ 	]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
                 alist '(("yes" . "yes")
                         ("no" . "no"))
                 complete t))
          ((re-search-forward "^X-LJ-Access:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^ 	]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
                 alist (append (lj--get-friend-groups-alist)
			       '(("friends" . "friends")
				 ("private" . "private")
				 ("public" . "public")))
                 complete t))
          ((re-search-forward "^X-LJ-Community:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^ 	]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
                 alist (lj--get-communities-alist)
                 complete t))
          ((re-search-forward "^X-LJ-Picture:" eol-point t)
           (setq eoh (point))
           (if (re-search-forward "[^ 	]" (+ eol-point 1) t)
               (backward-char 1))
           (setq range (cons (point) eol-point)
                 alist (lj--get-picture-alist)
                 complete t)))
    (goto-char the-point)
    (if complete
	(if (or (< the-point eoh)
		(> the-point eol-point))
	    (beep)
	  (lj--do-complete range alist))
      ;; We used to insert a literal tab, but that's evil. This is
      ;; supposed to be a text-mode derivative, so let's at least
      ;; try to do something reasonably text-mode like.
      (indent-relative-maybe))))

(defun lj--insert-initial-buffer-contents (&optional defaults-alist profile)
  "Insert into the current buffer the base contents of a LiveJournal post.
If DEFAULTS-ALIST is non-nil, treat it as an alist containing
default values for the various header fields."
  (lj--initialize)
  (insert
   (concat
    "X-LJ-Profile: "
    (let ((profile (or profile lj--current-profile lj-default-profile)))
      (if (listp profile)
          (mapconcat (lambda (foo)
                       (symbol-name foo))
                     profile
                     ", ")
        (symbol-name profile))) "\n"
    "X-LJ-Community: \n"
    "Subject: " (or (cdr (assoc 'subject defaults-alist))
		    lj-default-subject) "\n"
    "X-LJ-Music: " (or (cdr (assoc 'music defaults-alist))
		       (lj--get-music)
		       "") "\n"
    "X-LJ-Mood: " (or (cdr (assoc 'mood defaults-alist))
		      lj-default-mood) "\n"
    "X-LJ-Picture: " (or (cdr (assoc 'picture defaults-alist)) "") "\n"
    "FCC: " (or lj-archive-file-name "") "\n"
    "X-LJ-Allow-Comments: " (or (cdr (assoc 'comments defaults-alist))
				lj-default-comments-flag
				"yes") "\n"
    "X-LJ-Receive-Mail-Notification: " (or (cdr (assoc 'mail-notification defaults-alist))
				lj-default-mail-notification-flag
				"yes") "\n"
    "X-LJ-Access: " (or (cdr (assoc 'access defaults-alist))
			lj-default-access-level
			"public") "\n"))

  (set (make-local-variable 'lj-composition) t)

  (set (make-local-variable 'header-end)
       (if (fboundp 'point-marker)
           (point-marker)
         (point)))

  (put-text-property (point)
                     (progn
                       (insert mail-header-separator "\n")
                       (1- (point)))
                     'category 'mail-header-separator))

(defun lj--fetch-header (header)
  "Grab the requested HEADER field."
  (lj--initialize)
  (goto-char (point-min))
  (if (re-search-forward (format "^%s:[ ]" header) (ljc-mail-header-end) t)
      (let ((begin (point)))
	(end-of-line)
	(if (re-search-backward "[^ 	]" begin t)
	    (buffer-substring begin (+ (point) 1))
	  ""))
    ""))

(defun lj--fetch-date-and-time-alist ()
  "Create a date and time alist from the current time, or from the
   X-LJ-Entry-Time entry if it is defined."
  (lj--initialize)

  (let ((et-header (lj--fetch-header "X-LJ-Entry-Time")))

    (if (string-equal et-header "") ; no X-LJ-Entry-Time entry defined
	(let ((time-list (split-string (current-time-string) "[ :]+"))
	      (month-list '("Jan" "Feb" "Mar" "Apr" "May" "Jun"
			    "Jul" "Aug" "Sep" "Oct" "Nov" "Dec"))
	      (dt-alist '()))

	  (add-to-list 'dt-alist (cons 'year
                                       (nth (- (length time-list) 1)
                                            time-list)))
	  (add-to-list 'dt-alist
		       (cons 'mon
			     (format "%d"
				     (1+ (- (length month-list)
					    (length (member (cadr time-list)
							    month-list)))))))
	  (add-to-list 'dt-alist (cons 'day (nth 2 time-list)))
	  (add-to-list 'dt-alist (cons 'hour (nth 3 time-list)))
	  (add-to-list 'dt-alist (cons 'min (nth 4 time-list)))
	  dt-alist)

      ; now the case where X-LJ-Entry-Time is defined
      (let ((time-list (split-string et-header "[-: ]+"))
	    (dt-alist '()))

	(add-to-list 'dt-alist (cons 'year (nth 0 time-list)))
	(add-to-list 'dt-alist (cons 'mon (nth 1 time-list)))
	(add-to-list 'dt-alist (cons 'day (nth 2 time-list)))
	(add-to-list 'dt-alist (cons 'hour (nth 3 time-list)))
	(add-to-list 'dt-alist (cons 'min (nth 4 time-list)))
	dt-alist))))

;;;###autoload
(defun lj-update-mode ()
  "Major mode for editing a LiveJournal update.
Similar to `mail-mode' and `text-mode', but with additional commands:
	\\[lj-post] `lj-post' (post the update)	\\[lj-post-and-exit] `lj-post-and-exit'
\\[lj--header-complete] can be used for completion of various headers.
\\[describe-mode] should tell you all sorts of neat things about this mode."
  (interactive)
  (lj--initialize)

  (kill-all-local-variables)
  (setq major-mode 'lj-update-mode)
  (setq mode-name "LiveJournal")

  ;; Since we're pretending to be like mail, why don't we let the mail
  ;; font locking do the work for us? :)
  (set (make-local-variable 'font-lock-defaults)
       '(mail-font-lock-keywords t))

  (use-local-map lj-update-mode-map)

  ;; Abbrev support
  (setq local-abbrev-table lj-update-mode-abbrev-table)

  ;; Make paragraph fill functions work correctly
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat (regexp-quote mail-header-separator)
				"$\\|\t*\\([-|#;>* ]\\|(?[0-9]+[.)]\\)+$"
				"\\|[ \t]*[a-z0-9A-Z]*>+[ \t]*$\\|[ \t]*$\\|"
				"-- $\\|---+$\\|"
				page-delimiter))
  (setq paragraph-separate paragraph-start)

  (when (buffer-file-name)
    (if (file-exists-p (buffer-file-name))
        ;; We're editing a journal that we've already been working
        ;; on, so we have to initialize header-end appropriately.
        (save-excursion
          (goto-char (point-min))
          (search-forward mail-header-separator nil t)
          (beginning-of-line)
          (set (make-local-variable 'header-end)
               (if (fboundp 'point-marker)
                   (point-marker)
                 (point))))
      (lj--insert-initial-buffer-contents nil)
      (set-buffer-modified-p nil)))

  ;; But if you don't like the above two lines, why not leave them
  ;; there and add a hook that will fix things for you? :) Some
  ;; suggested things to turn on in this hook: flyspell (we get spell
  ;; checking for free. yay.)
  (run-hooks 'lj-update-mode-hook))

;;;###autoload
(defun lj-compose (&optional profile)
  "Compose a LiveJournal update, using PROFILE."
  (interactive)
  (lj--initialize)
  (switch-to-buffer (get-buffer-create "*LiveJournal*"))
  (delete-region (point-min) (point-max))
  (or (eq major-mode 'lj-update-mode) (lj-update-mode))
  (lj--insert-initial-buffer-contents nil profile))

(provide 'lj-write)
;;; lj-write.el ends here
