;;; ljcheckf.el --- Poll LiveJournal for friend updates.
;; Copyright (C) 2001  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience
;; RCS Id: $Id: ljcheckf.el,v 1.1 2002/08/19 22:06:50 hober Exp $

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; To use ljcheckf, add something like the following to your .emacs
;; file:

;; (require 'ljcheckf)

;; Use the command `lj-checkfriends-mode' to enable or disable this
;; mode.

;; When it tells you your friends page has been updated, and after
;; you've looked at it, C-c ` resets the checking thingy.

;; I would like bug reports, feature requests, and anything else for
;; that matter. If you just want to complain about the ickier parts of
;; the code, that's cool.

;;; History:
;; This is here so checkdoc doesn't complain. :)

;;; Code:

(require 'ljcompat)

(or (boundp 'lj--curent-profile)
    (setq lj--current-profile lj-default-profile))

(defvar lj-checkfriends-mode nil
  "Whether or not lj-checkfriends-mode is on or off.
Setting this variable will not change this; try calling
the function `lj-checkfriends-mode'.")

(defvar lj-checkfriends-new-string    " [*LJ*]")
(defvar lj-checkfriends-no-new-string " [-LJ-]")
(defvar lj-checkfriends-maybe-string  " [?LJ?]")

(defvar lj-checkfriends-mode-map nil
  "Keyboard mapping used by `lj-checkfriends-mode'.")

(defvar lj-checkfriends-lastupdate nil
  "The lastupdate field value from LiveJournal.
Please don't mess with this.")

(defvar lj-checkfriends-interval nil
  "The interval field value from LiveJournal.
Please don't mess with this.")

(unless lj-checkfriends-mode-map
  (setq lj-checkfriends-mode-map (make-sparse-keymap))
  (define-key lj-checkfriends-mode-map "\C-c`" 'lj-checkfriends-reset))

(defvar lj-checkfriends-timer nil
  "The timer of `lj-checkfriends-timer-func'.
Do not mess with this.")

(lj--deferror lj-checkfriends-error (lj-error)
  "Unable to check friends, so I'm giving up")

(defun lj-checkfriends-timer-func ()
  "Function run via Emacs' timer.
Do not mess with this."

  (lj--message 6 "Running `lj-checkfriends-timer-func'.")

  ;; if there is new shit, cancel the timer.
  (cancel-timer lj-checkfriends-timer)

  ;; notification will be by frobbing the entry in the
  ;; minor-mode-alist.

  (let* ((fields `((lastupdate . ,lj-checkfriends-lastupdate)))
         (request (lj--make-request 'checkfriends fields))
         (server-response (lj--send-request lj--current-profile request))
         (worked (car server-response))
         (response-alist (cdr server-response))
         (new (if (equal "1" (cdr (assoc "new" response-alist)))
                  t
                nil)))

    (lj--message 6 "tstart")
    (mapcar (lambda (elt)
              (lj--message 6 "item: %S" elt))
            response-alist)
    (lj--message 6 "tend")

    (unless worked
      (lj-checkfriends-mode -1)
      (signal 'lj-checkfriends-error nil))

    (when worked

      ;; initialize variables.
      (setq lj-checkfriends-lastupdate
            (cdr (assoc "lastupdate" response-alist)))

      (setq lj-checkfriends-interval
            (string-to-int (cdr (assoc "interval" response-alist))))

      (when new
        ;; frob minor-mode-alist
        (setcdr (assoc 'lj-checkfriends-mode minor-mode-alist)
                (list lj-checkfriends-new-string)))

      (unless new
        ;; frob minor-mode-alist
        (setcdr (assoc 'lj-checkfriends-mode minor-mode-alist)
                (list lj-checkfriends-no-new-string))

        ;; install timer.
        (setq lj-checkfriends-timer
              ;; perhaps this should use run-with-idle-timer?
              (run-at-time
               (concat (int-to-string lj-checkfriends-interval) " sec")
               nil 'lj-checkfriends-timer-func))))))

(defun lj-checkfriends-reset ()
  "Reset friend-checking.
This should be called when you've looked at your friends page."
  (interactive)

  ;; notification will be by frobbing the entry in the
  ;; minor-mode-alist.

  (let* ((fields '())
         (request (lj--make-request 'checkfriends fields))
         (server-response (lj--send-request lj--current-profile request))

         (worked (car server-response))
         (response-alist (cdr server-response)))

    (when worked
      ;; initialize variables.
      (setq lj-checkfriends-lastupdate
            (assoc "lastupdate" response-alist))

      (lj--message 6 "start")
      (mapcar (lambda (elt) (lj--message 6 "item: %s" elt))
              response-alist)
      (lj--message 6 "end")

      (setq lj-checkfriends-interval
            (string-to-int (cdr (assoc "interval" response-alist))))

      ;; cancel an old timer if there is one.
      (when lj-checkfriends-timer
        (cancel-timer lj-checkfriends-timer))

      ;; frob minor-mode-alist
      (setcdr (assoc 'lj-checkfriends-mode minor-mode-alist)
              (list lj-checkfriends-maybe-string))

      ;; install timer.
      (setq lj-checkfriends-timer
            ;; perhaps this should use run-with-idle-timer?
            (run-at-time
             (concat (int-to-string lj-checkfriends-interval) " sec")
             nil 'lj-checkfriends-timer-func)))))

;;;###autoload
(defun lj-checkfriends-mode (arg)
  "Toggle LiveJournal Check Friends mode.
With ARG, turn the mode on if arg is positive, off otherwise."
  (interactive "P")

  (setq lj-checkfriends-mode
        (if (null arg)
            (not lj-checkfriends-mode)
          (> (prefix-numeric-value arg) 0)))

  (when lj-checkfriends-mode   ; wax on
    (unless (get lj--current-profile 'lj-logged-in-p)
      (lj-profile-login lj--current-profile))


    (add-to-list 'minor-mode-map-alist
             (cons 'lj-checkfriends-mode lj-checkfriends-mode-map))
    (add-to-list 'minor-mode-alist
                 '(lj-checkfriends-mode " maybe"))
    (lj-checkfriends-reset))

  (unless lj-checkfriends-mode ; wax off
    (setq minor-mode-map-alist
          (remove (cons 'lj-checkfriends-mode lj-checkfriends-mode-map)
                  minor-mode-map-alist))
    (setq minor-mode-alist
          (remove '(lj-checkfriends-mode lj-checkfriends-maybe-string)
                  minor-mode-alist))
    (setq minor-mode-alist
          (remove '(lj-checkfriends-mode lj-checkfriends-new-string)
                  minor-mode-alist))
    (setq minor-mode-alist
          (remove '(lj-checkfriends-mode lj-checkfriends-no-new-string)
                  minor-mode-alist)))

  (if (interactive-p)
      (if lj-checkfriends-mode
          (lj--message 2 "Check Friends mode enabled")
        (lj--message 2 "Check Friends mode disabled"))))

(provide 'ljcheckf)
;;; ljcheckf.el ends here
