#include "parser.h"
#include "printer.h"
#include "polynomial.h"
#include "wallideal.h"
#include "lp.h"
#include "polyhedralcone.h"
#include "gfanapplication.h"
#include "polyhedralfan.h"
#include "halfopencone.h"

class GroebnerConeApplication : public GFanApplication
{
  SimpleOption optionRestrict;
  SimpleOption optionPair;
  SimpleOption optionAsFan;
public:
  const char *helpText()
  {
    return "This program computes a Groebner cone. Three different cases are handled. The input may be a marked reduced Groebner basis in which case its Groebner cone is computed. The input may be just a marked minimal basis in which case the cone computed is not a Groebner cone in the usual sense but smaller. (These cones are described in [Fukuda, Jensen, Lauritzen, Thomas]). The third possible case is that the Groebner cone is possibly lower dimensional and given by a pair of Groebner bases as it is useful to do for tropical varieties, see option --pair. The facets of the cone can be read off in section FACETS and the equations in section IMPLIED_EQUATIONS.\n";
  }
  GroebnerConeApplication():
    optionRestrict("--restrict","Add an inequality for each coordinate, so that the the cone is restricted to the non-negative orthant."),
    optionPair("--pair","The Groebner cone is given by a pair of compatible Groebner bases. The first basis is for the initial ideal and the second for the ideal itself. See the tropical section of the manual."),
    optionAsFan("--asfan","Writes the cone as a polyhedral fan with all its faces instead. In this way the extreme rays of the cone are also computed.")
  {
    registerOptions();
  }

  char *name()
  {
    return "_groebnercone";
  }

  int main()
  {
    PolynomialSet m=FileParser(Stdin).parsePolynomialSetWithRing();
    PolynomialSet g(m.getRing());
    if(optionPair.getValue())
      {
	g=FileParser(Stdin).parsePolynomialSet(m.getRing());
      }
    else
      {
	g=m;
	m=g.markedTermIdeal();
      }
    int n=g.getRing().getNumberOfVariables();
	
    IntegerVectorList equalities=wallInequalities(m);
    IntegerVectorList normals=algebraicTest(wallInequalities(g),g);
    if(optionRestrict.getValue())
      {
	for(int i=0;i<n;i++)
	  normals.push_back(IntegerVector::standardVector(n,i));
      }

    // AsciiPrinter(Stderr).printVectorList(normals);

    PolyhedralCone c(normals,equalities,n);
    c.canonicalize();
    if(!optionAsFan.getValue())
      AsciiPrinter(Stdout).printPolyhedralCone(c);
    else
      {
	AsciiPrinter P(Stdout);
	IntegerVectorList empty;
	HalfOpenCone C(n,c.getEquations(),c.getHalfSpaces(),empty,true);
	//	PolyhedralFan F=faceComplexOfCone(C);
	PolyhedralFan F(n);F.insert(C.closure());
	F.printWithIndices(&P,false,0);
      }

    return 0;
  }
};

static GroebnerConeApplication theApplication;

