#include <stdlib.h>
#include <stdio.h>

#include "router.h"

ether_addr broadcast_ether_addr = {0xff,0xff,0xff,0xff,0xff,0xff};

static ipv4_packet *arp_wait_packet = NULL;
static u32 arp_wait_ip = 0;


void free_ether_packet (ether_packet *pkt)
{
  free_memblock_chain (pkt->data);
  free (pkt);
}

void free_ipv4_packet(ipv4_packet *pkt)
{
  free_ether_packet(pkt->data);
  free(pkt);
}

void send_arp_request(u8 interface, u32 ip)
{
  arp_packet sent;
  ether_packet *ether;

  sent.op=1;
  memcpy(sent.src_en_addr, router_ether_addr [interface],6);
  sent.src_ip_addr = router_ip_addr [interface];
  memcpy(sent.dest_en_addr, broadcast_ether_addr, 6);
  sent.dest_ip_addr=ip;
  
  ether=encode_arp_packet(&sent);
  send_packet(interface,ether);
  free_ether_packet(ether);
}

void send_ipv4_packet(u8 interface, ether_addr hw_addr, ipv4_packet *pkt)
{
  ether_packet *ether;

  ether=pack_ipv4_packet(pkt);
  memcpy(ether->dest_addr,hw_addr,6);
  pack_ether_header(ether);

  send_packet(interface, ether);

}


void handle_ipv4_packet (ether_packet *packet)
{
  ipv4_packet *received;
  rt_table_entry *rt_entry;
  arp_cache_entry *arp_entry;
  u32 tmpsum;

  printf("handling ipv4 packet\n");
  received = unpack_ipv4_packet(packet);
  
  /* Check if the packet is for us (either of our adreses or
     anything on the loopback net), if so drop */

  if (received->dest == router_ip_addr[0] ||
      received->dest == router_ip_addr[1] ||
      (received->dest & 0xff000000) == 0x7f000000) {
    fprintf(stderr, "ipv4 packet for us, no routing needed!\n");
    free_ipv4_packet (received);
    return;
  }

  /* handle the ttl/checksum change. Note, if ttl becomes zero, drop
     it. */
  
  /* From RFC 1141 */

  received->ttl--;                  /* decrement ttl */

  if (received->ttl==0) {
    fprintf(stderr, "dropping ipv packet, ttl=0\n");
    free_ipv4_packet (received);
    return;
  }
  tmpsum = received->cksum + 0x100;  /* increment checksum high byte*/
  received->cksum = (tmpsum + (tmpsum>>16)); /* add carry */
  
  /* Long version in case the above does not work: */
#ifdef 0
  
  u16 old;
  
  old = received->ttl;
  received->ttl -= 1;
  tmpsum = old + (~(received->ttl) & 0xffff);
  tmpsum += received->cksum;
  tmpsum = (tmpsum & 0xffff) + (tmpsum>>16);
  received->cksum = tmpsum + (tmpsum>>16);
#endif /* 0 */


  /* Look up the next hop IP and interface in the routing table. */

  rt_entry=rt_route(received->dest);

  /* if we don't know how to route it, drop it on the floor. */
  if (rt_entry==NULL) {
    fprintf(stderr, "no route entry for ip packet.  dropping\n");
    free_ipv4_packet (received);
    return;    
  }
  
  
  /* Look up the ip for the next hop in the ARP cache. */
  arp_entry = arp_cache_lookup_ip(rt_entry->next_hop_ip);

  if (arp_entry==NULL) {
    printf("next hop for ip not in arp cache, doing arp request\n");
    /* If not available, send an appropriate arp request and put the ip
       packet in the holding area. */
    send_arp_request (rt_entry->next_hop_if,rt_entry->next_hop_ip);
    arp_wait_packet=received;
    arp_wait_ip=rt_entry->next_hop_ip;
  } else {
    /* If available, change the destination ethernet adress, re-encode
       the packet, and send it to the next hop out the appropriate 
       interface using send_packet. */
    printf("sending ipv4 packet\n");
    send_ipv4_packet (rt_entry->next_hop_if, arp_entry->hw_addr, received);
    free_ipv4_packet (received);
  }
}



void handle_arp_packet(u8 interface, ether_packet *packet) 
{
  arp_packet *received;

  arp_packet sent;
  ether_packet *ether;
     
  received = decode_arp_packet (packet);

  /* Alwyas add to cache */
  arp_cache_add_entry(received->src_ip_addr, received->src_en_addr);
  
  printf("done adding arp entry, ifx = %d, op = %d, destip = %x, myip = %x\n",
	 interface, received->op, 
	 received->dest_ip_addr, router_ip_addr[interface]);
  /* If request was received, send reply. */

  if (received->op == 1 &&
      received->dest_ip_addr == router_ip_addr[interface]) {

      printf("recieved arp request, sending reply\n");
    sent.op = 2;
    sent.src_ip_addr = router_ip_addr[interface];
    memcpy(sent.src_en_addr, &router_ether_addr[interface], 6);
    sent.dest_ip_addr = received->src_ip_addr;
    memcpy(sent.dest_en_addr, received->src_en_addr, 6);

    ether = encode_arp_packet(&sent);
    
    send_packet(interface, ether);
    free_ether_packet(ether);
  } 

  /* If we're holding a packet and just got an ARP for its IP,
     send it out.*/
  if (arp_wait_packet !=NULL && received->src_ip_addr==arp_wait_ip) {
      printf("got arp packet for ip packet on wait\n");
    handle_ipv4_packet (arp_wait_packet->data);
    free(arp_wait_packet);
    arp_wait_packet=NULL;
  }

  free (received);
  free_ether_packet (packet);
}
