;;;                                                      George Madrid
;;;                                                      6.034
;;;                                                      Problem Set 1

(in-package 'user)

;;; Problem 5.  

;;; (a) Well, it's not the best choice since (by experience) you can
;;; wander around Central Square for quite a while and eventually find
;;; your destination.  But it's far from the best route, and the
;;; search path can get quite deep.  Specifically, depth-first doesn't
;;; take into account the geography of Central Square.  We have some
;;; knowledge here that we fail to take into account.  Using geography
;;; would probably speed this up.  It would make a fine hill climbing
;;; heuristic. 

;;; (b) The difference between breadth-first and depth-first is that
;;; the former checks the guys in the queue before continuing with the
;;; newly expanded paths.
;;;
;;; breadth-first fails because it find the route with the fewest
;;; junctions, and not the one with the least distance.

(defun breadth-first (start finish &optional
			    (queue (list (list start))))
  (cond ((endp queue) nil)                      ;Queue empty?
        ((equal finish (first (first queue)))   ;Finish found?
         (reverse (first queue)))               ;Return path.
        (t (breadth-first                       ;Try again.
             start
             finish
	     (append (rest queue)
		     (extend (first queue)))))))

;;; (c) This implementation of best-first uses a straight "as the crow
;;; flies" distance as it heuristic to determine what path to take.
;;;
;;; It fails to find the least cost path in the case given in the
;;; problem set because the first node to which it moves is actually
;;; "closer" to the destination than the next node on the optimal
;;; path.   
(defun best-first (start finish &optional 
			 (queue (list (list start))))
  (cond ((endp queue) nil)
	((equal finish (first (first queue)))
	 (reverse (first queue)))
	(t (best-first start finish
		       (sort (append (rest queue)
				     (extend (first queue)))
			     #'(lambda (a b)
				 (closerp a b finish)))))))

;;; (d) Okay, I tried it on a few examples.  It seems somewhat slow.
;;; Hmm, one reason that it is so slow, I think is that it ends up
;;; doing a sort of breadth-first search, since (in general) the paths
;;; with fewer junctions are going to be the shorter ones.
;;;
;;; Funny, though.  It seems to be slower than the fiasco that I
;;; created below.  I think this is due to the sort.  Sort is pretty
;;; expensive, I guess.  Specifically, path-length is expensive, and
;;; not-b-and-b doesn't call path-length (with a non-nil arg) until it
;;; can start removing paths from the queue, that is, until they can
;;; start doing some good.
;;;
;;; I found that this procedure took so long to figure the path from
;;; '(ericson-ct green st) to '(washington-st pine-st) that I didn't
;;; let it finish.  It seems to do smaller problems correctly, though.
;;; (But still slowly.)
(defun b-and-b (start finish &optional 
		      (queue (list (list start))))
  (cond ((endp queue) nil)
	((equal finish (first (first queue)))
	 (reverse (first queue)))
	(t (b-and-b start finish 
		    (sort (append (rest queue)
				  (extend (first queue)))
			  #'(lambda (a b)
			      (< (path-length a)
				 (path-length b))))))))

;;; As a sort of side note, here is something that I implemented
;;; first, thinking it was branch and bound.  It keeps track of a
;;; shortest path to the goal, and expands every path that is shorter
;;; than this path.  It doesn't sort the queue before recursing,
;;; though, which is why it's so slow, I guess.  (It was *really*
;;; slow.)  It was sort of a British Museum that would stop when a
;;; path became infeasible.  Oh, look, it was also slow because it was
;;; doing a breadth-first search, so it was taking forever to find a
;;; path to use for a bound.  Just thought I'd include this rather
;;; large mistake because I actually learned something explaining to
;;; you why it was so slow.  You can ignore this.  
;;;
;;; Oh, thinking about this some more (I have a lot of time to think
;;; while waiting for these big searches.), I realized why I made this
;;; mistake.  I was trying to implement the first paragraph of the
;;; section on branch-and-bound.  This is where they talk about the
;;; oracle telling me a solution and then eliminating anything longer
;;; than that solution.  That's what this does, but it's too slow.
(defun not-b-and-b (start finish &optional 
			  (queue (list (list start)))
			  (shortest nil))
  (cond ((endp queue) (reverse shortest))
	((equal finish (first (first queue)))
	 (not-b-and-b start finish (rest queue) (first queue)))
	(t 
	 (let ((short-length (path-length shortest)))
	   (not-b-and-b start finish 
		    (remove-if 
		     #'(lambda (path) 
			 (and shortest
			      (not (< (path-length path)
				      short-length))))
		     (append (rest queue)
			     (extend (first queue))))
		    shortest)))))
	       
;;; (e) Oh, this is *much* faster.  Polishing off the path that was
;;; too large for b-and-b in a matter of seconds.  By cutting the
;;; branching factor way down and eliminating excess work, it causes a
;;; major speedup.
;;;
;;; I can't find an example for which this finds a cheaper path than
;;; b-and-b.  I would hope not, since b-and-b yields an optimum path.
(defun dyna-search (start finish &optional
			  (queue (list (list start))))
  (cond ((endp queue) nil)
	((equal finish (first (first queue)))
	 (reverse (first queue)))
	(t (dyna-search start finish
			(sort (dynamic-prog (extend (first queue))
					    (rest queue))
			      #'(lambda (a b)
				  (< (path-length a)
				     (path-length b))))))))

(defun dynamic-prog (new-paths old-paths)
  "
   Apply the dynamic programming principle to the queue formed by
   adding the partial paths in <new-paths> to those in <old-paths>.
   Return the resulting queue.  For an explanation, see the comments below.
  "
  (let ((queue old-paths))                      ;Initialize queue
    (mapc #'(lambda (new-path)                  ;For each new path
	      ;Find an old path containing the newest node
	      (let* ((old-path (find (first new-path)
				     queue
				     :test #'member)))
		;Keep the shorter of old path and new path in the queue
		(cond ((not old-path) (push new-path queue))
		      ((< (path-length new-path)
			  (path-length (member (first new-path) old-path)))
		       (setf queue (substitute new-path old-path queue))))))
	  new-paths)
    queue))                                     ;Return the queue

;;; (f) This is about 1 sec. faster than dyna-search on that long path
;;; that b-and-b won't finish.
;;;
;;; My h-star is straight-line-distance from the last node on a path
;;; to the finish node.
;;;
;;; Yes, it is guaranteed to find an optimal solution, because h-star
;;; is guaranteed to provide an underestimate.  
(defun h-star (path finish)
  (straight-line-distance (first path) finish))

(defun a-star (start finish &optional
			  (queue (list (list start))))
  (cond ((endp queue) nil)
	((equal finish (first (first queue)))
	 (reverse (first queue)))
	(t (a-star start finish
			(sort (dynamic-prog (extend (first queue))
					    (rest queue))
			      #'(lambda (a b)
				  (< (+ (path-length a) 
					(h-star a finish))
				     (+ (path-length b)
					(h-star b finish)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;; Redefine do-search to check its parameters so that it won't
;;; recurse forever.  
(defun do-search (method &key
		       (origin (request-jct 'origin))
		       (dest (request-jct 'destination)))
  (if (or (null dest)
	  (null origin))
      (print "Null argument to search.")
    (progn
      (find-path (jct-or-from-names origin)
		 (jct-or-from-names dest)
		 method)
      (if *window-output* (draw-map))
      (print-path))))

