/** @file if_cf.c
  * @brief This file contains CF interface handling functions
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	09/29/05: Add Doxygen format comments
	
********************************************************/

//#define CF8381
#define CF8385
#define RF8015
#define HELPER_IMAGE

#include <pcmcia/cs_types.h>
#include	"include.h"
#include 	"if_cf.h"
#ifdef CF8381
  #include "cf8381.h"
#endif

#ifdef CF8389
  #include "cf8389.h"
#endif

#ifdef CF8385
  #ifdef RF8015
    #include "cf8385.h"
  #endif
  #ifdef RF8031
    #include "cf8385h.h"
  #endif
#endif

#ifdef CF8399
  #include "cf8399.h"
#endif
#ifdef CF8305
  #include  "cf8305.h"
#endif

#ifdef HELPER_IMAGE
#ifdef CF8381
  #include "helper.h"
#endif
#ifdef CF8385
  #include "helper2.h"
#endif
  #define init_image helper_image
#else
  #define init_image fmimage
#endif

/********************************************************
		Local Variables
********************************************************/
#define HP_RETRY      (10000)
#define FW_RETRY      (5000)
#define FW_BLOCK_SIZE (256)

#define CIS(X) \
  do { \
    tuple.DesiredTuple = X; \
    \
    if (pcmcia_get_first_tuple(handle, &tuple)) \
      goto error; \
    if (pcmcia_get_tuple_data(handle, &tuple)) \
      goto error; \
    \
    cisbuf[ofs++] = tuple.TupleCode; \
    cisbuf[ofs++] = tuple.TupleLink; \
    memcpy(cisbuf+ofs, tuple.TupleData, tuple.TupleLink); \
    ofs += tuple.TupleLink; \
  } while (0)

int   *register_cf_driver(cf_notifier_fn_add ,cf_notifier_fn_remove , void *);
void  unregister_cf_driver(void);

static struct pcmcia_device cisinfo;

/* event_flag is used to save if event happened. 
 * According to CF spec, Event bit(BIT4) in card status register is 
 * read clear. In order for sbi_get_int_status to get event happening 
 * correctly, Interrupt handler saves event happening to event_flag from 
 * int_cause register.*/
static u16 event_flag = 0;

/********************************************************
		Global Variables
********************************************************/

u8 usb_fw_ready = TRUE;

/********************************************************
		Local Functions
********************************************************/
static int disable_host_int_mask(wlan_private *priv,u8 int_mask);
static int enable_host_int_mask (wlan_private *priv , u8 int_mask);

/** 
 *  @brief  Sets register bits specified by mask
 *  @param priv         pointer to wlan_private
 *  @param regoff   	register offset 
 *  @param mask     	mask bit
 *  @return 	   	N/A
 */
static inline void cf_UshortRegSetBits(wlan_private *priv,
          u16 regOffset, u16 mask)
{
	/* Set the bit in host status to indicate cmd download is complete */
	if (((struct cf_card_rec *)(priv->wlan_dev.card))->do_read_modify_write == 0) {
	    	outw(mask, priv->wlan_dev.netdev->base_addr + regOffset);
  	}
	else {
		outw(inw(priv->wlan_dev.netdev->base_addr + regOffset) | mask,
			      priv->wlan_dev.netdev->base_addr + regOffset);
	}
}

/** 
 *  @brief  This function polls device to see if RxUpLdRdy bit is set
 *  @param priv         pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS if RxUpLdRdy; 
 *  			WLAN_STATUS_FAILURE otherwiaw
 */
static inline int cf_IsRxUploadReady(wlan_private *priv)
{
	u16 usval;

  	usval = inw(priv->wlan_dev.netdev->base_addr+CF_CCR_CARD_STATUS);
  	return (usval & CF_HCR_HIM_RxUpLdRdy) ? WLAN_STATUS_SUCCESS: WLAN_STATUS_FAILURE;
}

/** 
 *  @brief  This function polls cmd ready bit to see if CmdDnLdRdy is ready
 *  @param priv         pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS if CmdDnLdRdy; 
 *  			WLAN_STATUS_FAILURE otherwiaw
 */
static inline int cf_IsCmdReady(wlan_private *priv)
{
	u16 	usval;
	u16	loop = 0;

#define MAX_LOOP 	100
	
	do {
		usval = inw(priv->wlan_dev.netdev->base_addr+CF_CCR_CARD_STATUS);
		if (usval & CF_HCR_HIM_CmdDnLdRdy)
			break;

		if (++loop >= MAX_LOOP)
			break;

		mdelay(1);
	}while (1);

	if (loop)
		PRINTM(INFO, "cf_IsCmdReady: CFMACREG_CCR_CARD_STATUS = 0x%x, loop 0x%x\n",
				usval, loop);

	return (usval & CF_HCR_HIM_CmdDnLdRdy) ? WLAN_STATUS_SUCCESS : WLAN_STATUS_FAILURE;
}

/** 
 *  @brief  This function polls cmd ready bit to see if cmd response is ready
 *  @param priv         pointer to wlan_private
 *  @return 	   	WLAN_STATUS_SUCCESS if CmdUpLdRdy; 
 *  			WLAN_STATUS_FAILURE otherwiaw
 */
static inline int cf_IsRespReady(wlan_private *priv)
{
  	u16 usval;

  	usval = inw(priv->wlan_dev.netdev->base_addr+CF_CCR_CARD_STATUS);
 	return (usval & CF_HCR_HIM_CmdRspRdy) ? WLAN_STATUS_SUCCESS : -WLAN_STATUS_FAILURE;
}

/** 
 *  @brief  This function ends download of current packet and trigger FW
 *  @param priv         pointer to wlan_private
 *  @return 	   	N/A
 */
static inline void cf_RxEndPacket(wlan_private *priv)
{
  	static u32 rxPacketCnt = 0;

  	rxPacketCnt++;

  	cf_UshortRegSetBits(priv, CF_HCR_HOST_STATUS,
          			CF_HCR_HS_RxUpLdOvr);

  	/* This triggers an interrupt in the card FW. */
  	cf_UshortRegSetBits(priv, CF_HCR_CARD_INT_CAUSE,
          			CF_HCR_CIC_RxUpLdOvr);
}


/** 
 *  @brief This function waits till register value becoming to expected value.
 *  @param addr    	register address
 *  @param value   	expected register value 
 *  @param retry     	retry times
 *  @param usec		delay between polling register
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int poll_reg(uint addr, u8 value, u32 retry, u32 usec)
{
  	u32 count;
	
  	for (count = 0; count < retry; count ++) {
	    if (inb(addr) == value)
      		break;

    		udelay(usec);
  	}

  	return (count >= retry) ? WLAN_STATUS_FAILURE: WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function initializes CF register address.
 *  @param priv		a pointer to wlan_private structure
 *  @return 	   	n/a
 */
static void init_cf_addr(wlan_private *priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u32 base = priv->wlan_dev.netdev->base_addr;

 	reg->host.status            = CF_HCR_HOST_STATUS + base;
 	reg->host.int_cause         = CF_HCR_CARD_INT_CAUSE + base;
  	reg->host.int_mask          = CF_HCR_HOST_INT_MASK + base;
  	reg->host.raddr0            = CF_HCR_READ_BASE_LOW + base;
  	reg->host.raddr1            = CF_HCR_READ_BASE_HIGH + base;
  	reg->host.waddr0            = CF_HCR_WRITE_BASE_LOW + base;
  	reg->host.waddr1            = CF_HCR_WRITE_BASE_HIGH + base;
  	reg->host.io_data_read      = CF_HCR_IO_READ_PORT + base;
  	reg->host.io_cmd_read       = CF_HCR_IO_CMD_READ_PORT + base;
  	reg->host.io_data_write_len = CF_HCR_IO_WRITE_LEN + base;
  	reg->host.io_data_write     = CF_HCR_IO_WRITE_PORT + base;
  	reg->host.io_cmd_write_len  = CF_HCR_IO_CMD_WRITE_LEN + base;
  	reg->host.io_cmd_write      = CF_HCR_IO_CMD_WRITE_PORT + base;
  	reg->host.scratch           = CF_HCR_SCRATCH_PORT + base;
  	reg->host.cis_addr          = CF_HCR_CIS_ADDR_PORT + base;
  	reg->host.cis_data          = CF_HCR_CIS_DATA_PORT + base;
  	reg->host.gbus_read         = CF_HCR_IO_GBUS_REG_READ + base;
  	reg->host.gbus_write        = CF_HCR_IO_GBUS_REG_WRITE + base;

  	reg->card.product_id        = CF_CCR_PRODUCT_ID + base;
  	reg->card.status            = CF_CCR_CARD_STATUS + base;
  	reg->card.int_cause         = CF_CCR_HOST_INT_CAUSE + base;
  	reg->card.io_data_len       = CF_CCR_IO_READ_LEN + base;
  	reg->card.sq_raddr0         = CF_CCR_SQ_READ_BASE_LOW + base;
  	reg->card.sq_raddr1         = CF_CCR_SQ_READ_BASE_HIGH + base;
  	reg->card.sq_waddr0         = CF_CCR_SQ_WRITE_BASE_LOW + base;
  	reg->card.sq_waddr1         = CF_CCR_SQ_WRITE_BASE_HIGH + base;
  	reg->card.io_cmd_read_len   = CF_CCR_IO_CMD_READ_LEN + base;
  	reg->card.sq_cmd_raddr0     = CF_CCR_SQ_CMD_READ_BASE_LOW + base;
  	reg->card.sq_cmd_raddr1     = CF_CCR_SQ_CMD_READ_BASE_HIGH + base;
 	reg->card.sq_cmd_waddr0     = CF_CCR_SQ_CMD_WRITE_BASE_LOW + base;
  	reg->card.sq_cmd_waddr1     = CF_CCR_SQ_CMD_WRITE_BASE_HIGH + base;
  	reg->card.cfg_base          = CF_CCR_CFG_REG_BASE_ADR + base;
  	reg->card.card_cfg          = CF_CCR_CARD_CFG + base;
  	reg->card.scratch           = CF_CCR_SCRATCH_PORT + base;
  	reg->card.tx_seq            = CF_CCR_TX_FRAME_SEQ_NUM + base;
 	reg->card.tx_stat           = CF_CCR_TX_FRAME_STATUS + base;
  	reg->card.int_mask          = CF_CCR_CARD_INT_MASK + base;
  	reg->card.cis_addr          = CF_CCR_CIS_ADR_PORT + base;
  	reg->card.cis_data          = CF_CCR_CIS_DATA_PORT + base;
	enable_host_int_mask(priv,HIM_ENABLE);
}

/**
 *  @brief This function is interrupt handler.
 *  @param iqr		interrupt number
 *  @param dev_id	pointer to net_device structure
 *  @param regs		pointer to pt_regs structure
 *  @return 	   	n/a
 */
static int cf_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
  	struct net_device *dev = dev_id;
  	wlan_private *priv    = (wlan_private *) dev->priv;
  	wlan_adapter *Adapter = priv->adapter;

  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u16 cause;

  	ENTER();

  	cause = inw(reg->card.int_cause);
  	PRINTM(INFO, "(1) int_cause = 0x%02x @ %lu\n", cause, jiffies);
 
  	/*
	 * Remeber if event happening. event happening
	 * when bit4 in int_cause is 1
	 */
  	event_flag |= (cause & CF_CCR_CS_CardEvent);

#define CARD_REMOVED	0xffff
    	if (cause) { 
  		if (cause == CARD_REMOVED ) {
      			Adapter->SurpriseRemoved = TRUE;
		}
		else {
			cause &= CF_CCR_HIC_MASK;

			// Before writing int cause register to clear interrupt 
			// mask interrupt is required to avoid the interrupt missing issue.
    		disable_host_int_mask(priv, HIM_DISABLE);
			outw(cause, reg->card.int_cause);
		}

		PRINTM(INFO, "int_cause = 0x%X, HisRegCpy = 0x%x\n",
			       cause, Adapter->HisRegCpy);
    		wlan_interrupt(dev);
 	 }
  	else {
    		PRINTM(FATAL, "(1) no interrupt\n");
  	}

  	LEAVE();
	return IRQ_HANDLED;
}

/**
 *  @brief This function sends command to FW.
 *  @param priv		pointer to wlan_private structure
 *  @param buf		pointer to command data
 *  @param len		cmd data length
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int cf_send_cmd(wlan_private * priv, u16 * buf, u16 len)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  
  	ENTER();

  	if ( cf_IsCmdReady(priv) )
  	{
    		PRINTM(INFO, "device not ready!\n");
   		return WLAN_STATUS_FAILURE;
  	}

  	outw(len, reg->host.io_cmd_write_len);
  	outsw(reg->host.io_cmd_write, buf, len / 2);

  	if (len & 0x0001) 
    		outb(((u8 *)buf)[len - 1], reg->host.io_cmd_write);

  	cf_UshortRegSetBits(priv, CF_HCR_HOST_STATUS, CF_HCR_HS_CmdDnLdOvr);
          
  	wmb();

  	outw(CF_HCR_CIC_CmdDnLdOvr, reg->host.int_cause);
  	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function sends data to FW.
 *  @param priv		pointer to wlan_private structure
 *  @param buf		pointer to data
 *  @param len		data length
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int cf_tx_block(wlan_private *priv, u8 *buf, u16 len)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;

  	ENTER();

  	outw(len, reg->host.io_data_write_len);
  	outsw(reg->host.io_data_write, buf, len / 2);

  	if (len & 0x0001) 
    	outb(buf[len - 1], reg->host.io_data_write);

  	priv->adapter->SeqNum++;

  	cf_UshortRegSetBits(priv, CF_HCR_HOST_STATUS, CF_HCR_HS_TxDnLdOvr);
  	cf_UshortRegSetBits(priv, CF_HCR_CARD_INT_CAUSE, CF_HCR_CIC_TxDnLdOvr);

  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function enables the host interrupts mask
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param mask	   the interrupt mask
 *  @return 	   WLAN_STATUS_SUCCESS
 */
static int enable_host_int_mask (wlan_private *priv , u8 int_mask)
{
	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u16 mask = int_mask;

  	ENTER();

  	if (!mask) 
    		mask = CF_HCR_HIM_MASK;

  	mask &= CF_HCR_HIM_MASK;

  	PRINTM(INFO, "mask bit = 0x%X\n", mask);
 	PRINTM(INFO, "before masking host_int_mask = 0x%X\n", cardp.host_int_mask);

  	cardp.host_int_mask &= ~mask;

  	outw(cardp.host_int_mask, reg->host.int_mask);

	PRINTM(INFO, "after host_int_mask=0x%x\n", cardp.host_int_mask);

  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/**  @brief This function disables the host interrupts mask.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param mask	   the interrupt mask
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int disable_host_int_mask(wlan_private *priv,u8 int_mask)
{
	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u16 mask = (u16) int_mask;

  	ENTER();

  	if (!mask)
    		mask = CF_HCR_HIM_MASK;

  	mask &= CF_HCR_HIM_MASK;
  	cardp.host_int_mask |= mask;
  	PRINTM(INFO, "host in mask in init = 0x%X\n", cardp.host_int_mask);

  	outw(cardp.host_int_mask, reg->host.int_mask);

  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}
/********************************************************
		Global Functions
********************************************************/

/**  
 *  @brief This function registers CF driver.
 *  @param add		pointer to add_card callback function
 *  @param remove	pointer to remove card callback function
 *  @param arg		pointer to call back function parameter
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int *sbi_register(wlan_notifier_fn_add add, wlan_notifier_fn_remove remove, void *arg)
{
  	return  register_cf_driver( (cf_notifier_fn_add)add, 
      				(cf_notifier_fn_remove)remove, NULL );
}

/**  
 *  @brief removes CF driver.
 *  @return 	   	N/A
 */
void sbi_unregister(void)
{
  	unregister_cf_driver();
}

/** 
 *  @brief This function gets interrupt status.
 *  @param priv		pointer to wlan_private structure
 *  @param ireg		pointer to int status value
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_get_int_status(wlan_private * priv, u8 * ireg)
{
	wlan_adapter 	*Adapter = priv->adapter;
	cfreg 		*reg = (cfreg *)priv->wlan_dev.ioport;
	struct sk_buff	*skb;
	u8		*cmdBuf;
	u16 		cause;
	u8 cf_int_cause = 0;
	
	ENTER();

	if (Adapter->SurpriseRemoved)
		return WLAN_STATUS_FAILURE;

	/* read and clean cause */	
	cause = inw(reg->card.int_cause) & CF_CCR_HIC_MASK;
  	outw(cause, reg->card.int_cause);


	/* Read the reason code */
	cause |= inw(reg->card.status) & CF_CCR_HIC_MASK;

	/*Catch event happening, remebered in interrupt handler*/
	cause |= event_flag;

	enable_host_int_mask(priv, HIM_ENABLE);

	if (!cause) {
		PRINTM(FATAL, "(2) no interrupt\n");
		goto sbi_get_int_status_exit;
	}

	PRINTM(INFO, "(2) int_cause = 0x%02x @ %lu\n", cause, jiffies);

	if (cause & CF_CCR_CS_TxDnLdRdy) {
		priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
		cf_int_cause |= HIS_TxDnLdRdy;
	}

	if (cause & CF_CCR_CS_RxUpLdRdy)
		cf_int_cause |= HIS_RxUpLdRdy;
#ifdef THROUGHPUT_TEST
	else if (!Adapter->NumTransfersRx && 
				Adapter->ThruputTest & 0x02)
		cf_int_cause |= HIS_RxUpLdRdy;
#endif

	if (cause & CF_CCR_CS_CmdDnLdRdy)
		cf_int_cause |= HIS_CmdDnLdRdy;

	if (cause & CF_CCR_CS_CmdRspRdy)
		cf_int_cause |= HIS_CmdUpLdRdy;

	if (cause & CF_CCR_CS_CardEvent)
		cf_int_cause |= HIS_CardEvent;


sbi_get_int_status_exit:
	if (cf_int_cause & HIS_RxUpLdRdy) {
			
		if (!(skb = dev_alloc_skb(MRVDRV_ETH_RX_PACKET_BUFFER_SIZE))) {
			PRINTM(FATAL, "No free skb\n");
			priv->stats.rx_dropped++;
			cf_int_cause &= ~HIS_RxUpLdRdy;
			goto restore_int;
		}
		
		if (sbi_card_to_host(priv, MVMS_DAT, 
					&priv->wlan_dev.upld_len,
				skb->tail, WLAN_UPLD_SIZE) < 0) {
			/* skb->tail is passed as we are calling skb_put after 
			 * we are reading the data */
			PRINTM(FATAL, "ERROR: Data Transfer from device"
					" failed\n");
			cf_int_cause &= ~HIS_RxUpLdRdy;
			kfree_skb(skb);
			goto restore_int;
		}

		skb_put(skb, priv->wlan_dev.upld_len);
		list_add_tail((struct list_head *) skb,
			(struct list_head *) &Adapter->RxSkbQ);
	}

	if (cf_int_cause & HIS_CmdUpLdRdy) {

		if (!Adapter->CurCmd) {
			cmdBuf = priv->wlan_dev.upld_buf;
			cf_int_cause &= ~HIS_CmdUpLdRdy;
		} else {
			cmdBuf = Adapter->CurCmd->BufVirtualAddr;
		}

		if (sbi_card_to_host(priv, MVMS_CMD, 
			&priv->wlan_dev.upld_len,
			cmdBuf, WLAN_UPLD_SIZE) < 0) {
			/* Move this into ProcessRxedPacket */
			
			PRINTM(FATAL, "ERROR: Data Transfer from device"
					" failed\n");
			goto restore_int;
		}
	}

	if (cf_int_cause & HIS_TxDnLdRdy)
      		priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;

restore_int:	
	*ireg = cf_int_cause;
	cf_int_cause = 0;


	PRINTM(INFO, "cf_int_cause = 0x%x\n", *ireg);
	PRINTM(INFO, "adapter->HisRegCpy = 0x%x\n", Adapter->HisRegCpy);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function reads data from FW.
 *  @param priv		pointer to wlan_private structure
 *  @param type		type of data to read
 *  @param nb		number of data bytes
 *  @param payload	buffer for payload data
 *  @param npayload	number of bytes to read
 *  @return 	   	WLAN_STATUS_SUCCESS; WLAN_STATUS_FAILURE
 */
int sbi_card_to_host(wlan_private *priv, u32 type, u32 *nb, u8 *payload,
              u16 npayload)
{
	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;

	ENTER();

	if ((!payload) || (!npayload)) 
		return WLAN_STATUS_FAILURE;

	if (type == MVMS_DAT) {
		u16 len = inw(reg->card.io_data_len);

#define MIN_PACKET_LEN	30	
		if (len <= MIN_PACKET_LEN || len > MRVDRV_ETH_RX_PACKET_BUFFER_SIZE) {
			PRINTM(WARN, "invalid length %d - discarding\n", len);
			priv->stats.rx_dropped++;
			cf_RxEndPacket(priv);
			return WLAN_STATUS_FAILURE;
		}

		*nb = len;

		insw(reg->host.io_data_read, (u16 *)payload, len / sizeof(u16));

		if (len & 0x0001) 
			payload[len - 1] = inb(reg->host.io_data_read);

		cf_UshortRegSetBits(priv, CF_HCR_HOST_STATUS,     CF_HCR_HS_RxUpLdOvr);
		cf_UshortRegSetBits(priv, CF_HCR_CARD_INT_CAUSE,  CF_HCR_CIC_RxUpLdOvr);
	} else if (type == MVMS_CMD) {
		u16 len;

		if (cf_IsRespReady(priv)) {
			PRINTM(WARN, "device not ready\n");
			LEAVE();
			return WLAN_STATUS_FAILURE;
		}

		len = inw(reg->card.io_cmd_read_len);
		*nb = len;

		if ((len == 0) || (len > MRVDRV_SIZE_OF_CMD_BUFFER)) {
			PRINTM(INFO, "invalid length!\n");
			return WLAN_STATUS_FAILURE;
		}

		insw(reg->host.io_cmd_read, (u16 *)payload, len / sizeof(u16));

		if (len & 0x0001) 
			payload[len - 1] = inb(reg->host.io_cmd_read);

		PRINTM(INFO, "received %d bytes\n", len);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function reads event cause.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_read_event_cause(wlan_private * priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u16 usCardStatus;

  	ENTER();

  	usCardStatus = inw(reg->card.status);
  	usCardStatus &= CF_CCR_CS_STATUS_MASK;
  	usCardStatus = usCardStatus >> 5;
  	priv->adapter->EventCause = usCardStatus;
  
  	/* Clear Event flag and trigger FW to send next event*/  
  	event_flag = 0;
  	outw(CF_HCR_CIC_HstEvent, reg->host.int_cause);
  
  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function is stub function  
 *  @param priv		pointer to wlan_private structure
 *  @param bits		interrupt mask bit
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_reenable_host_interrupt(wlan_private * priv, u8 bits)
{
  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function disables specified interrupt.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_disable_host_int(wlan_private * priv)
{
    	disable_host_int_mask(priv, HIM_DISABLE);
	return  WLAN_STATUS_SUCCESS;
}

/**
 *  @brief This function enables specified interrupt.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_enable_host_int(wlan_private * priv)
{
	enable_host_int_mask(priv,HIM_ENABLE);
	return  WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function removes registered wlan device and free irq.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_unregister_dev(wlan_private * priv)
{
  	ENTER();
	
  	kfree((void *)priv->wlan_dev.ioport);
  	free_irq(priv->wlan_dev.netdev->irq, priv->wlan_dev.netdev);
	
  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function registers wlan device and request irq handler.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_register_dev(wlan_private * priv)
{
  	int ret = WLAN_STATUS_SUCCESS;

  	ENTER();

  	strncpy(priv->wlan_dev.name, cfio_dev_info,
		       	sizeof(priv->wlan_dev.name));

  	priv->wlan_dev.ioport   = (u32)kmalloc(sizeof(cfreg), GFP_KERNEL);
  	priv->wlan_dev.upld_rcv = 0;
  	priv->wlan_dev.upld_typ = 0;
  	priv->wlan_dev.upld_len = 0;

  	cardp.eth_dev = priv->wlan_dev.netdev;

  	PRINTM(INFO, "IRQ %d\n", cardp.irq);

  	ret = request_irq(cardp.irq, cf_interrupt, SA_SHIRQ, 
      				"cf_irq", priv->wlan_dev.netdev );
           
  	if (ret != 0)
  	{
    		PRINTM(INFO, "Call to request_irq failed\n");
    		priv->wlan_dev.card = NULL;
   		 LEAVE();
    		return ret;
  	}

  	priv->wlan_dev.card               = (void *)&cardp;
  	priv->wlan_dev.netdev->irq        = cardp.irq;
  	priv->wlan_dev.netdev->base_addr  = cardp.port;
  	priv->adapter->irq                = cardp.irq;

  	init_cf_addr(priv);
	
  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function downloads data to FW  
 *  @param priv		pointer to wlan_private structure
 *  @param type		type of data
 *  @param buf		pointer to data buffer
 *  @param len		number of bytes
 *  @return 	   	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int sbi_host_to_card(wlan_private *priv, u8 type, u8 *buf, u16 len)
{
  	int ret = WLAN_STATUS_FAILURE;

  	ENTER();

  	if (type == MVMS_DAT) {
    		priv->wlan_dev.dnld_sent = DNLD_DATA_SENT;
    		ret = cf_tx_block(priv, buf, (u16)len);
  	} 
  	else if (type == MVMS_CMD) {
   		ret = cf_send_cmd(priv, (u16 *)buf, len);
  	} 
  	else {
    		ret = WLAN_STATUS_FAILURE;
  	}

  	LEAVE();
  	return ret;
}

/** 
 *  @brief This function reads CIS table.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_get_cis_info(wlan_private *priv)
{
  	wlan_adapter *Adapter = priv->adapter;
  	client_handle_t handle = &cisinfo;
  	tuple_t tuple;
  	char buf[64], cisbuf[512];
  	int ofs=0, count=6;

  	ENTER();
  
  	memset(cisbuf, 0x0, sizeof(cisbuf));

  	tuple.Attributes = 0;
  	tuple.TupleData = buf;
  	tuple.TupleDataMax = sizeof(buf);
  	tuple.TupleOffset = 0;

  	CIS(CISTPL_CONFIG);
  	CIS(CISTPL_CFTABLE_ENTRY);
  	CIS(CISTPL_MANFID);
  	CIS(CISTPL_FUNCID);
  	CIS(CISTPL_FUNCE);

  	do 
  	{
    		if (pcmcia_get_next_tuple(handle, &tuple))
      			goto error;
    		if (pcmcia_get_tuple_data(handle, &tuple))
      			goto error;
  
    		cisbuf[ofs++] = tuple.TupleCode;
    		cisbuf[ofs++] = tuple.TupleLink;
    		memcpy(cisbuf+ofs, tuple.TupleData, tuple.TupleLink);
    		ofs += tuple.TupleLink;
  	} while (count--);
  
  	CIS(CISTPL_VERS_1);
 	CIS(CISTPL_DEVICE_A);
 	CIS(CISTPL_DEVICE);

  	memset(Adapter->CisInfoBuf, 0x0, sizeof(Adapter->CisInfoBuf));
  	memcpy(Adapter->CisInfoBuf, cisbuf, ofs);
  	Adapter->CisInfoLen = ofs;

error:

  	PRINTM(INFO, "card configuration failed...\n");

  	LEAVE();

  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function is stub function
 *  @param              pointer to card  
 *  @return 	   	WLAN_STATUS_SUCCESS 
 */
int sbi_probe_card(void *card_p)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function checks if FW download success.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_verify_fw_download(wlan_private * priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;
  	u8 rev = inb(reg->card.product_id);

      	priv->adapter->chip_rev = rev;

  	PRINTM(INFO, "chip rev is: %02x\n", rev);

  	cardp.do_read_modify_write      = 0;

#ifdef CF8305
  	cardp.do_read_modify_write      = 1;
#endif

#ifdef CF8381

#define CF8381_B3_REV 0x04
  	if (rev < CF8381_B3_REV) {
    		PRINTM(INFO, "sbi_register_dev: cf8381 older than rev. B3.\n");
    		PRINTM(INFO, "sbi_register_dev: do r-m-w and intr w-around.\n");
    		cardp.do_read_modify_write    = 1;
  	}
#endif

#ifdef CF8385
  
#define CF8385_B1_REV 0x12
  	if (rev < CF8385_B1_REV) {
    		PRINTM(INFO, "sbi_register_dev: cf8385 older than rev. B1.\n");
   		PRINTM(INFO, "sbi_register_dev: do r-m-w and intr w-around.\n");
    		cardp.do_read_modify_write    = 1;
  	}
#endif

  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function downloads helper.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_prog_firmware(wlan_private * priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;

  	u32 size_total;
  	u32 size_sent;
  	u32 size_block;
  	u8  buf[256];
  	u8  temp;
  	int r;

	usb_fw_ready = FALSE;

  	temp = inb(reg->card.scratch);

  	if (temp == CF_SCRATCH_FIRMWARE_READY)
    		return WLAN_STATUS_SUCCESS;
 
  	if (temp != CF_SCRATCH_HOST_BOOT_WAITING)
    		return WLAN_STATUS_FAILURE;

  	size_total  = sizeof(init_image);
  	size_sent   = 0;
  	size_block  = FW_BLOCK_SIZE;

  	while (size_sent < size_total) 
  	{
    		PRINTM(INFO, "loading %06d of %06d bytes\n", size_sent, size_total);

    		if (size_total - size_sent < size_block)
      			size_block = (size_total - size_sent);

    		memcpy(buf, &init_image[size_sent], size_block);

    		outw(FW_BLOCK_SIZE, reg->host.io_cmd_write_len);
    		outsw(reg->host.io_cmd_write, buf, FW_BLOCK_SIZE / 2);

    		outb(CF_HCR_HS_CmdDnLdOvr, reg->host.status);
    		
		wmb();
    		outw(CF_HCR_CIC_CmdDnLdOvr, reg->host.int_cause);
#define FW_POLL_DELAY_10US	10	/*10us delay*/
    		r = poll_reg( reg->card.status, CF_CCR_CS_CmdDnLdRdy, 
	        		FW_RETRY, FW_POLL_DELAY_10US);

    		if (r != WLAN_STATUS_SUCCESS)
      			return WLAN_STATUS_FAILURE;

    		size_sent += size_block;
  	}

  	outw(0x00, reg->host.io_cmd_write_len);
  	outb(CF_HCR_HS_CmdDnLdOvr, reg->host.status);
  	wmb();
  	outw(CF_HCR_CIC_CmdDnLdOvr, reg->host.int_cause);
  	
	usb_fw_ready = TRUE;
	return WLAN_STATUS_SUCCESS;
}

#ifdef HELPER_IMAGE
int sbi_prog_helper(wlan_private * priv)
{
  	return sbi_prog_firmware(priv);
}
#endif

#ifdef HELPER_IMAGE

/** 
 *  @brief This function downloads firmware.
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_prog_firmware_w_helper(wlan_private * priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;

  	u16 len;
  	int i;
  	int r;
  	int retry;

#define HELPER_DNLD_READY	0x10
#define HELPER_POLL_DELAY_1MS	1000	/*1ms delay*/

  	r = poll_reg(reg->card.sq_raddr0, HELPER_DNLD_READY, 
		  		HP_RETRY, HELPER_POLL_DELAY_1MS);

  	if (r != WLAN_STATUS_SUCCESS)
    		return WLAN_STATUS_FAILURE;

  	retry = 0;

  	for (i = 0; i < sizeof(fmimage); i += len)
  	{
    		len = inw(reg->card.sq_raddr0);

    		retry = (len & 0x0001) ? retry + 1: 0;

#define MAX_RETRY_TIMES 20
    		if (retry > MAX_RETRY_TIMES)
      			return WLAN_STATUS_FAILURE;

    		if (retry > 0) 
      		i -= len;

    		PRINTM(INFO, "loading %06d of %06d bytes\n", i, sizeof(fmimage));

    		outw( len, reg->host.io_cmd_write_len);
    		outsw(reg->host.io_cmd_write, &fmimage[i], (len+1) / 2);
    		outb(CF_HCR_HS_CmdDnLdOvr,  reg->host.status);
    		outw(CF_HCR_CIC_CmdDnLdOvr, reg->host.int_cause);

#define HELPER_POLL_NO_DELAY	0	
    		/*no delay between poll*/
   		 r = poll_reg( reg->card.status, CF_CCR_CS_CmdDnLdRdy, 
				FW_RETRY, HELPER_POLL_NO_DELAY );

    		if (r != WLAN_STATUS_SUCCESS) 
    		{
      			PRINTM(INFO, "failed  %06d bytes\n", i);
      			return WLAN_STATUS_FAILURE;
    		}
  	}

#define HELPER_POLL_DELAY_50US	50 /*delay 50 us*/
  	r = poll_reg(reg->card.scratch, CF_SCRATCH_FIRMWARE_READY, 
		  FW_RETRY, HELPER_POLL_DELAY_50US);

  	if (r != WLAN_STATUS_SUCCESS) {
    		PRINTM(INFO, "firmware failed!\n");
    		return WLAN_STATUS_FAILURE;
  	}

 	PRINTM(INFO, "firmware started!\n");
  	return WLAN_STATUS_SUCCESS;
}
#endif /* HELPER_IMAGE */

#ifdef DEEP_SLEEP
/** 
 *  @brief This function triggers FW to exit deepsleep .
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
inline int sbi_exit_deep_sleep(wlan_private* priv)
{
  	cfreg *reg = (cfreg *)priv->wlan_dev.ioport;

  	outb(CF_SCRATCH_DEEPSLEEP_EXIT, reg->card.scratch);
  	cfio_read_cfg_reg(priv);

  	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function is stub function
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
inline int sbi_reset_deepsleep_wakeup(wlan_private * priv)
{
 	return WLAN_STATUS_SUCCESS;
}
#endif  /* DEEP_SLEEP */

/** 
 *  @brief This function reads CF register
 *  @param priv		pointer to wlan_private structure
 *  @param offset	register offset
 *  @return 	   	register value
 */
int sbi_read_cfreg(wlan_private * priv, int offset)
{
 	return inw(priv->wlan_dev.netdev->base_addr + offset);
}

/** 
 *  @brief This function writes CF register
 *  @param priv		pointer to wlan_private structure
 *  @param offset	register offset
 *  @param value	register value
 *  return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_write_cfreg(wlan_private * priv, int offset, u16 value)
{
	outw(value, priv->wlan_dev.netdev->base_addr + offset);

 	return WLAN_STATUS_SUCCESS;
}

#ifdef ENABLE_PM
/** 
 *  @brief This function is stub function  
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_suspend(wlan_private * priv)
{
 	return WLAN_STATUS_SUCCESS;
}

/**  
 *  @brief This function is stub function  
 *  @param priv		pointer to wlan_private structure
 *  @return 	   	WLAN_STATUS_SUCCESS
 */
int sbi_resume(wlan_private * priv)
{
 	return WLAN_STATUS_SUCCESS;
}

#endif
/* End of file */
