/** @file wlan_rx.c
  * @brief This file contains the handling of RX in wlan
  * driver.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	12/09/05: ADD Sliding window SNR/NF Average Calculation support
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

#ifdef SLIDING_WIN_AVG
/** 
 *  @brief This function computes the AvgSNR .
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   AvgSNR
 */
static u8 wlan_getAvgSNR(wlan_private * priv)
{
	u8 i;
	u16 temp = 0;
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF == 0)
		return 0;
	for (i = 0; i <Adapter->numSNRNF; i++)
		temp += Adapter->rawSNR[i];
	return (u8)(temp/Adapter->numSNRNF);
	
}

/** 
 *  @brief This function computes the AvgNF
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   AvgNF
 */
static u8 wlan_getAvgNF(wlan_private * priv)
{
	u8 i;
	u16 temp = 0;
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF == 0)
		return 0;
	for (i = 0; i <Adapter->numSNRNF; i++)
		temp += Adapter->rawNF[i];
	return (u8)(temp/Adapter->numSNRNF);
	
}

/** 
 *  @brief This function save the raw SNR/NF to our internel buffer
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param pRxPD   A pointer to RxPD structure of received packet
 *  @return 	   n/a
 */
static void wlan_save_rawSNRNF(wlan_private * priv, RxPD * pRxPD)
{
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF < Adapter->data_avg_factor)
		Adapter->numSNRNF++;
	Adapter->rawSNR[Adapter->nextSNRNF] = pRxPD->SNR;
	Adapter->rawNF[Adapter->nextSNRNF] = pRxPD->NF;
	Adapter->nextSNRNF++;
	if(Adapter->nextSNRNF >= Adapter->data_avg_factor)
		Adapter->nextSNRNF = 0;
	return;
}
#endif //SLIDING_WIN_AVG


/** 
 *  @brief This function computes the RSSI in received packet.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param pRxPD   A pointer to RxPD structure of received packet
 *  @return 	   n/a
 */
static void wlan_compute_rssi(wlan_private *priv, RxPD *pRxPD)
{
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "RxPD: SNR = %d, NF = %d\n", pRxPD->SNR, pRxPD->NF);
	PRINTM(INFO, "Befroe computing SNR: SNR- avg = %d, NF-avg = %d\n",
		Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE, 
		Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);

	Adapter->SNR[TYPE_RXPD][TYPE_NOAVG] = pRxPD->SNR;
	Adapter->NF[TYPE_RXPD][TYPE_NOAVG] = pRxPD->NF;
#ifdef SLIDING_WIN_AVG
	wlan_save_rawSNRNF(priv , pRxPD);
#endif //SLIDING_WIN_AVG

	Adapter->RxPDSNRAge = os_time_get();
	
#ifdef SLIDING_WIN_AVG
	Adapter->SNR[TYPE_RXPD][TYPE_AVG] = wlan_getAvgSNR(priv) * AVG_SCALE;
	Adapter->NF[TYPE_RXPD][TYPE_AVG] = wlan_getAvgNF(priv ) * AVG_SCALE;
#else
	/* Average out the SNR from the received packet */
	Adapter->SNR[TYPE_RXPD][TYPE_AVG] =
		CAL_AVG_SNR_NF(Adapter->SNR[TYPE_RXPD][TYPE_AVG],
			       pRxPD->SNR, Adapter->data_avg_factor);

	/* Average out the NF value */
	Adapter->NF[TYPE_RXPD][TYPE_AVG] = 
		CAL_AVG_SNR_NF(Adapter->NF[TYPE_RXPD][TYPE_AVG],
			       pRxPD->NF, Adapter->data_avg_factor);
#endif //SLIDING_WIN_AVG
	PRINTM(INFO, "After computing SNR: SNR-avg = %d, NF-avg = %d\n",
		Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE, 
		Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);

	Adapter->RSSI[TYPE_RXPD][TYPE_NOAVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_NOAVG],
			Adapter->NF[TYPE_RXPD][TYPE_NOAVG]);

	Adapter->RSSI[TYPE_RXPD][TYPE_AVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE,
			Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);
	
	LEAVE();
}

/********************************************************
		Global functions
********************************************************/

/**
 *  @brief This function processes recevied packet and forwards it
 *  to kernel/upper layer
 *  
 *  @param priv    A pointer to wlan_private
 *  @param skb     A pointer to skb which includes the received packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int ProcessRxed_802_3_Packet(wlan_private *priv, struct sk_buff *skb)
{
    	wlan_adapter	*Adapter = priv->adapter;
	RxPD		*pRxPD = (RxPD *) skb->data;
    	u8  		*ptr = skb->data;
    	int             i, ret = WLAN_STATUS_SUCCESS;

    	ENTER();

 	HEXDUMP("Before chop RxPD", ptr, MIN(skb->len, 100));

	if (skb->len < (ETH_HLEN + 8 + sizeof(RxPD))) {
		PRINTM(INFO, "RX Error: FRAME RECEIVED WITH BAD LENGTH\n");
		priv->stats.rx_length_errors++;
		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}

#ifdef BIG_ENDIAN
	endian_convert_RxPD(pRxPD);
#endif

	/* 
	 * Check RxPD status and update 802.3 stat,
	 */
	if (!(pRxPD->Status & MRVDRV_RXPD_STATUS_OK)) {
		PRINTM(INFO, "RX Error: frame received with bad status\n");
		priv->stats.rx_errors++;
		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}

	ptr += sizeof(RxPD);

	HEXDUMP("After chop RxPD", ptr, skb->len-sizeof(RxPD));

	HEXDUMP("Dest   Mac", ptr, MRVDRV_ETH_ADDR_LEN);
	HEXDUMP("Source Mac", (ptr + MRVDRV_ETH_ADDR_LEN), MRVDRV_ETH_ADDR_LEN);

#ifndef FW_REMOVED_SNAP_HEADER
	// Shift everything after the DEST/SRC bytes forward...
	// to take care of the SNAP header that is somewhere here...
#define SHIFT_DEST_SRC_ADDRESS_LEN	11	// 6 + 6 -1
	for (i = SHIFT_DEST_SRC_ADDRESS_LEN; i >= 0; i--)
		ptr[MRVDRV_SNAP_HEADER_LEN + i] = ptr[i];

	ptr += MRVDRV_SNAP_HEADER_LEN;
#endif

	/* 
	 * Take the data rate from the RxPD structure 
	 * only if the rate is auto
	 */
	if (Adapter->Is_DataRate_Auto)
		Adapter->DataRate = index_to_data_rate(pRxPD->RxRate);

	wlan_compute_rssi(priv, pRxPD);

#ifndef FW_REMOVED_SNAP_HEADER
	/* Compute actual packet size */
	i = MRVDRV_SNAP_HEADER_LEN + sizeof(RxPD);
#else
	i = sizeof(RxPD);
#endif
	skb_pull(skb, i);
	
	PRINTM(INFO, "Size of actual 802.3 packet = %d\n", skb->len);

	if (os_upload_rx_packet(priv, skb)) {
		PRINTM(INFO, "RX Error: os_upload_rx_packet returns failure\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

 	priv->stats.rx_bytes += skb->len; 
	priv->stats.rx_packets++;
	
	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();

	return (ret);
}
