/*
 * Copyright (c) 2007 Paul Sokolovsky
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 */

//#define DEBUG

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/pm.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <linux/platform_device.h>
#include <linux/battery.h>
#include <linux/adc.h>
#include <linux/adc_battery.h>

#include <asm/irq.h>
#include <asm/apm.h>

#define PIN_NO_VOLT 0
#define PIN_NO_CURR 1
#define PIN_NO_TEMP 2

struct battery_adc_priv {
	struct battery batt_cdev;

	struct battery_adc_platform_data *pdata;

	struct adc_request req;
	struct adc_sense pins[3];
	struct adc_sense last_good_pins[3];

	struct workqueue_struct *wq;
	struct delayed_work work;
};

/*
 *  Battery classdev methods
 */

#define GENERATE_ACCESSOR(property) \
static int adc_battery_get_##property(struct battery *b) \
{ \
	struct battery_adc_priv* drvdata = (struct battery_adc_priv*)b; \
	return drvdata->pdata->battery_info.property; \
}

GENERATE_ACCESSOR(min_voltage)
GENERATE_ACCESSOR(max_voltage)
GENERATE_ACCESSOR(min_current)
GENERATE_ACCESSOR(max_current)
GENERATE_ACCESSOR(min_capacity)
GENERATE_ACCESSOR(max_capacity)

static int adc_battery_get_status(struct battery *b)
{
	struct battery_adc_priv* drvdata = (struct battery_adc_priv*)b;
	return drvdata->pdata->charge_status;
}
static int adc_battery_get_voltage(struct battery *b)
{
	struct battery_adc_priv* drvdata = (struct battery_adc_priv*)b;
	return drvdata->last_good_pins[PIN_NO_VOLT].value * drvdata->pdata->voltage_mult / 1000;
}
static int adc_battery_get_current(struct battery *b)
{
	struct battery_adc_priv* drvdata = (struct battery_adc_priv*)b;
	return drvdata->last_good_pins[PIN_NO_CURR].value * drvdata->pdata->current_mult / 1000;
}
static int adc_battery_get_temp(struct battery *b)
{
	struct battery_adc_priv* drvdata = (struct battery_adc_priv*)b;
	return drvdata->last_good_pins[PIN_NO_TEMP].value * drvdata->pdata->temperature_mult / 1000;
}
static int adc_battery_get_capacity(struct battery *b)
{
	return ((adc_battery_get_voltage(b) - adc_battery_get_min_voltage(b)) * 
	     (adc_battery_get_max_capacity(b) - adc_battery_get_min_capacity(b))) /
	     (adc_battery_get_max_voltage(b) - adc_battery_get_min_voltage(b));
}
/*
 *  Driver body
 */

static void adc_battery_query(struct battery_adc_priv *drvdata)
{
	struct battery_adc_platform_data *pdata = drvdata->pdata;
	int powered, charging;

	adc_request_sample(&drvdata->req);

	powered = battery_is_external_power_supplied(&drvdata->batt_cdev);
	charging = pdata->is_charging ? pdata->is_charging() : -1;

	if (powered && charging)
		pdata->charge_status = BATTERY_STATUS_CHARGING;
	else if (powered && !charging && charging != -1)
		pdata->charge_status = BATTERY_STATUS_FULL;
	else
		pdata->charge_status = BATTERY_STATUS_DISCHARGING;

	/* Throw away invalid samples, this may happen soon after resume for example. */
	if (drvdata->pins[PIN_NO_VOLT].value > 0) {
	        memcpy(drvdata->last_good_pins, drvdata->pins, sizeof(drvdata->pins));
#ifdef DEBUG
		printk("%d %d %d\n", drvdata->pins[PIN_NO_VOLT].value, 
				     drvdata->pins[PIN_NO_CURR].value, 
				     drvdata->pins[PIN_NO_TEMP].value);
#endif
	}
}

static void adc_battery_charge_power_changed(struct battery *bat)
{
	struct battery_adc_priv *drvdata = (struct battery_adc_priv*)bat;
	cancel_delayed_work(&drvdata->work);
	queue_delayed_work(drvdata->wq, &drvdata->work, 0);
}

static void adc_battery_work_func(struct work_struct *work)
{
	struct delayed_work *delayed_work = container_of(work, struct delayed_work, work);
	struct battery_adc_priv *drvdata = container_of(delayed_work, struct battery_adc_priv, work);

	adc_battery_query(drvdata);
	battery_status_changed(&drvdata->batt_cdev);

	queue_delayed_work(drvdata->wq, &drvdata->work, (5000 * HZ) / 1000);
}

static int adc_battery_probe(struct platform_device *pdev)
{
        int retval;
	struct battery_adc_platform_data *pdata = pdev->dev.platform_data;
	struct battery_adc_priv *drvdata;

	// Initialize ts data structure.
	drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
	if (!drvdata)
		return -ENOMEM;

	drvdata->batt_cdev.name             = pdata->battery_info.name;
	drvdata->batt_cdev.main_battery     = pdata->battery_info.main_battery;
	drvdata->batt_cdev.get_voltage      = adc_battery_get_voltage;
	drvdata->batt_cdev.get_min_voltage  = adc_battery_get_min_voltage;
	drvdata->batt_cdev.get_max_voltage  = adc_battery_get_max_voltage;
	drvdata->batt_cdev.get_current      = adc_battery_get_current;
	drvdata->batt_cdev.get_min_current  = adc_battery_get_min_current;
	drvdata->batt_cdev.get_max_current  = adc_battery_get_max_current;
	drvdata->batt_cdev.get_capacity     = adc_battery_get_capacity;
	drvdata->batt_cdev.get_min_capacity = adc_battery_get_min_capacity;
	drvdata->batt_cdev.get_max_capacity = adc_battery_get_max_capacity;
	drvdata->batt_cdev.get_temp         = adc_battery_get_temp;
	drvdata->batt_cdev.get_status       = adc_battery_get_status;
	if (!pdata->voltage_pin)
		drvdata->batt_cdev.get_voltage = NULL;
	if (!pdata->current_pin)
		drvdata->batt_cdev.get_current = NULL;
	if (!pdata->temperature_pin)
		drvdata->batt_cdev.get_temp = NULL;

	drvdata->batt_cdev.external_power_changed =
	                          adc_battery_charge_power_changed;

	retval = battery_register(&pdev->dev, &drvdata->batt_cdev);
	if (retval) {
		printk("adc-battery: Error registering battery classdev");
		return retval;
	}

	drvdata->req.senses = drvdata->pins;
	drvdata->req.num_senses = ARRAY_SIZE(drvdata->pins);
	drvdata->pins[PIN_NO_VOLT].name = pdata->voltage_pin;
	drvdata->pins[PIN_NO_CURR].name = pdata->current_pin;
	drvdata->pins[PIN_NO_TEMP].name = pdata->temperature_pin;

	adc_request_register(&drvdata->req);

	/* Set default multipliers */
	if (!pdata->voltage_mult)
		pdata->voltage_mult = 1000;
	if (!pdata->current_mult)
		pdata->current_mult = 1000;
	if (!pdata->temperature_mult)
		pdata->temperature_mult = 1000;

	drvdata->pdata = pdata;
	pdata->drvdata = drvdata; /* Seems ugly, we need better solution */

	platform_set_drvdata(pdev, drvdata);

        // Load initial values ASAP
	adc_battery_query(drvdata);

	// Still schedule next sampling soon
	INIT_DELAYED_WORK(&drvdata->work, adc_battery_work_func);
	drvdata->wq = create_workqueue(pdev->dev.bus_id);
	if (!drvdata->wq)
		return -ESRCH;

	queue_delayed_work(drvdata->wq, &drvdata->work, (5000 * HZ) / 1000);

	return retval;
}

static int adc_battery_remove(struct platform_device *pdev)
{
	struct battery_adc_priv *drvdata = platform_get_drvdata(pdev);
	cancel_delayed_work(&drvdata->work);
	destroy_workqueue(drvdata->wq);
	battery_unregister(&drvdata->batt_cdev);
	adc_request_unregister(&drvdata->req);
	return 0;
}

static struct platform_driver adc_battery_driver = {
	.driver		= {
		.name	= "adc-battery",
	},
	.probe		= adc_battery_probe,
	.remove		= adc_battery_remove,
};

static int __init adc_battery_init(void)
{
	return platform_driver_register(&adc_battery_driver);
}

static void __exit adc_battery_exit(void)
{
	platform_driver_unregister(&adc_battery_driver);
}

module_init(adc_battery_init)
module_exit(adc_battery_exit)

MODULE_AUTHOR("Paul Sokolovsky");
MODULE_DESCRIPTION("Battery driver for ADC device");
MODULE_LICENSE("GPL");
