/*
 * Common power driver for PDAs and phones with one or two external
 * power supplies (AC/USB) connected to main and backup batteries,
 * and optional builtin charger.
 *
 * Copyright 2007 Anton Vorontsov <cbou@mail.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/external_power.h>
#include <linux/pda_power.h>
#include <linux/timer.h>

static unsigned int get_irq_flags(struct resource *res)
{
	unsigned int flags = IRQF_DISABLED;

	flags |= res->flags & IRQF_TRIGGER_MASK;

	if (res->flags & IORESOURCE_IRQ_SHAREABLE)
		flags |= IRQF_SHARED;

	return flags;
}

static struct resource *ac_irq, *usb_irq;
static struct pda_power_pdata *pdata;
static struct timer_list isr_timer;

static int pda_power_is_ac_online(struct power_supply *psy)
{
	return pdata->is_ac_online ? pdata->is_ac_online() : 0;
}

static int pda_power_is_usb_online(struct power_supply *psy)
{
	return pdata->is_usb_online ? pdata->is_usb_online() : 0;
}

static char *pda_power_supplied_to[] = {
	"main-battery",
	"backup-battery",
};

static struct power_supply pda_power_supplies[] = {
	{
		.name = "ac",
		.type = "ac",
		.supplied_to = pda_power_supplied_to,
		.num_supplicants = ARRAY_SIZE(pda_power_supplied_to),
		.is_online = pda_power_is_ac_online,
	},
	{
		.name = "usb",
		.type = "dc",
		.supplied_to = pda_power_supplied_to,
		.num_supplicants = ARRAY_SIZE(pda_power_supplied_to),
		.is_online = pda_power_is_usb_online,
	},
};

static void update_charger(void)
{
	if (!pdata->set_charge)
		return;

	if (pdata->is_ac_online && pdata->is_ac_online()) {
		pr_debug("pda_power: charger on (AC)\n");
		pdata->set_charge(PDA_POWER_CHARGE_AC);
	}
	else if (pdata->is_usb_online && pdata->is_usb_online()) {
		pr_debug("pda_power: charger on (USB)\n");
		pdata->set_charge(PDA_POWER_CHARGE_USB);
	}
	else {
		pr_debug("pda_power: charger off\n");
		pdata->set_charge(0);
	}

	return;
}

static void isr_timer_func(unsigned long irq)
{
	if (ac_irq && irq == ac_irq->start) {
		power_supply_changed(&pda_power_supplies[0]);
	}
	else if (usb_irq && irq == usb_irq->start) {
		power_supply_changed(&pda_power_supplies[1]);
	}
	else {
		printk(KERN_WARNING "pda_power: spurious irq %li", irq);
		return;
	}

	update_charger();

	return;
}

static irqreturn_t power_changed_isr(int irq, void *unused)
{
	isr_timer.data = irq;
	mod_timer(&isr_timer, jiffies + HZ);
	return IRQ_HANDLED;
}

static int pda_power_probe(struct platform_device *pdev)
{
	int ret = 0;

	if (pdev->id != -1) {
		printk(KERN_WARNING "pda_power: it's meaningless to "
		       "register several pda_powers, use id = -1\n");
		ret = -EINVAL;
		goto wrongid;
	}

	pdata = pdev->dev.platform_data;

	setup_timer(&isr_timer, isr_timer_func, 0);

	ac_irq = platform_get_resource_byname(pdev, IORESOURCE_IRQ, "ac");
	usb_irq = platform_get_resource_byname(pdev, IORESOURCE_IRQ, "usb");
	if (!ac_irq && !usb_irq) {
		printk(KERN_ERR "pda_power: no ac/usb irq specified\n");
		ret = -ENODEV;
		goto noirqs;
	}

	ret = power_supply_register(&pdev->dev, &pda_power_supplies[0]);
	if (ret) {
		printk(KERN_ERR "pda_power: failed to register %s power "
		       "supply\n", pda_power_supplies[0].name);
		goto supply0_failed;
	}

	ret = power_supply_register(&pdev->dev, &pda_power_supplies[1]);
	if (ret) {
		printk(KERN_ERR "pda_power: failed to register %s power "
		       "supply\n", pda_power_supplies[1].name);
		goto supply1_failed;
	}

	if (ac_irq) {
		ret = request_irq(ac_irq->start, power_changed_isr,
		                  get_irq_flags(ac_irq), ac_irq->name,
		                  &pda_power_supplies[0]);
		if (ret) {
			printk(KERN_ERR "pda_power: request ac irq failed\n");
			goto ac_irq_failed;
		}
	}

	if (usb_irq) {
		ret = request_irq(usb_irq->start, power_changed_isr,
		                  get_irq_flags(usb_irq), usb_irq->name,
		                  &pda_power_supplies[1]);
		if (ret) {
			printk(KERN_ERR "pda_power: request usb irq failed\n");
			goto usb_irq_failed;
		}
	}

	update_charger();

	goto success;

usb_irq_failed:
	if (ac_irq)
		free_irq(ac_irq->start, &pda_power_supplies[0]);
ac_irq_failed:
	power_supply_unregister(&pda_power_supplies[1]);
supply1_failed:
	power_supply_unregister(&pda_power_supplies[0]);
supply0_failed:
noirqs:
wrongid:
success:
	return ret;
}

static int pda_power_remove(struct platform_device *pdev)
{
	if (usb_irq)
		free_irq(usb_irq->start, &pda_power_supplies[1]);
	if (ac_irq)
		free_irq(ac_irq->start, &pda_power_supplies[0]);
	del_timer_sync(&isr_timer);
	power_supply_unregister(&pda_power_supplies[1]);
	power_supply_unregister(&pda_power_supplies[0]);
	return 0;
}

static struct platform_driver pda_power_pdrv = {
	.driver = {
		.name = "pda-power",
	},
	.probe = pda_power_probe,
	.remove = pda_power_remove,
};

static int __init pda_power_init(void)
{
	return platform_driver_register(&pda_power_pdrv);
}

static void __exit pda_power_exit(void)
{
	platform_driver_unregister(&pda_power_pdrv);
	return;
}

module_init(pda_power_init);
module_exit(pda_power_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Anton Vorontsov <cbou@mail.ru>");
