/** @file wlan_cmdresp.c
  * @brief This file contains the handling of command
  * responses as well as events generated by firmware.
  * 
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	10/10/05: Add Doxygen format comments
	11/11/05: Add support for WMM Status change event
	12/13/05: Add Proprietary periodic sleep support
	12/23/05: Fix bug in adhoc start where the current index was
	          not properly being assigned before it was used.
	01/05/06: Add kernel 2.6.x support	
	01/11/06: Conditionalize new scan/join structures.
	          Update assoc response handling; entire IEEE response returned
	04/06/06: Add TSPEC, queue metrics, and MSDU expiry support
	04/10/06: Add hostcmd generic API
	04/18/06: Remove old Subscrive Event and add new Subscribe Event
		  implementation through generic hostcmd API
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function handles disconnect event. it
 *  reports disconnect to upper layer, clean tx/rx packets,
 *  reset link state etc.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void MacEventDisconnected(wlan_private * priv)
{
	wlan_adapter	*Adapter = priv->adapter;
	union iwreq_data wrqu;

	ENTER();

	if (Adapter->MediaConnectStatus != WlanMediaStateConnected)
		return;
	
	PRINTM(INFO, "Handles disconnect event.\n");

	memset(wrqu.ap_addr.sa_data, 0x00, ETH_ALEN);
	wrqu.ap_addr.sa_family = ARPHRD_ETHER;

	/* 
	 * Cisco AP sends EAP failure and de-auth in less than 0.5 ms.
	 * It causes problem in the Supplicant
	 */

	msleep_interruptible(1000);
	wireless_send_event(priv->wlan_dev.netdev,
		       		SIOCGIWAP, &wrqu, NULL);

	/* Free Tx and Rx packets */
	os_free_tx_packet(priv);
	wlan_send_rxskbQ(priv);

	/* report disconnect to upper layer */
	os_stop_queue(priv);
	os_carrier_off(priv);

	/* reset SNR/NF/RSSI values */
	memset(Adapter->SNR, 0x00, sizeof(Adapter->SNR));
	memset(Adapter->NF, 0x00, sizeof(Adapter->NF));
	memset(Adapter->RSSI, 0x00, sizeof(Adapter->RSSI));
	memset(Adapter->rawSNR, 0x00, sizeof(Adapter->rawSNR));
	memset(Adapter->rawNF, 0x00, sizeof(Adapter->rawNF));
	Adapter->nextSNRNF = 0;
	Adapter->numSNRNF = 0;
	Adapter->RxPDRate = 0;	
	PRINTM(INFO, "Current SSID=%s, Ssid Length=%u\n",
			Adapter->CurBssParams.ssid.Ssid, 
			Adapter->CurBssParams.ssid.SsidLength);
	PRINTM(INFO, "Previous SSID=%s, Ssid Length=%u\n",
			Adapter->PreviousSSID.Ssid, 
			Adapter->PreviousSSID.SsidLength);

	/* reset internal flags */
	Adapter->AdHocCreated = FALSE;

	cleanup_txqueues(priv);
		
	Adapter->SecInfo.WPAEnabled = FALSE;
	Adapter->SecInfo.WPA2Enabled = FALSE;
	Adapter->Wpa_ie_len = 0;
	Adapter->SecInfo.Auth1XAlg = WLAN_1X_AUTH_ALG_NONE;
	Adapter->SecInfo.EncryptionMode = CIPHER_NONE;


	Adapter->MediaConnectStatus = WlanMediaStateDisconnected;
	Adapter->LinkSpeed = MRVDRV_LINK_SPEED_1mbps;


	/* 
	 * memorize the previous SSID and BSSID
	 * it could be used for re-assoc
	 */
	memcpy(&Adapter->PreviousSSID, 
           &Adapter->CurBssParams.ssid,
           sizeof(struct WLAN_802_11_SSID));
	memcpy(Adapter->PreviousBSSID,
           Adapter->CurBssParams.bssid,
           MRVDRV_ETH_ADDR_LEN);

	/* need to erase the current SSID and BSSID info */
	Adapter->pAttemptedBSSDesc = 0;
	memset(&Adapter->CurBssParams, 0, sizeof(Adapter->CurBssParams));


	if (Adapter->PSState != PS_STATE_FULL_POWER) {
		/* make firmware to exit PS mode */
		PRINTM(INFO, "Disconnected, so exit PS mode.\n");
		PSWakeup(priv, 0);
	}

	LEAVE();
}

/** 
 *  @brief This function handles link lost, deauth and
 *  disassoc events.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void HandleDisconnectEvent(wlan_private * priv)
{
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		MacEventDisconnected(priv);
#ifdef REASSOCIATION
		if (Adapter->Reassoc_on == TRUE ) {
			PRINTM(INFO, "RE-ASSOC: trigger the timer\n");
			Adapter->TimerIsSet = TRUE;
			ModTimer(&Adapter->MrvDrvTimer, 0);
		}
#endif /* REASSOCIATION */
	}
}

/** 
 *  @brief This function handles MIC failure event.
 *
 *  @param priv    A pointer to wlan_private structure
 *  @para  event   the event id
 *  @return 	   n/a
 */
static void HandleMICFailureEvent(wlan_private *priv, u32 event)
{
	u8		buf[50];
	
	ENTER();

	memset(buf, 0, sizeof(buf));
	
	sprintf(buf, "%s", "MLME-MICHAELMICFAILURE.indication ");
	
	if (event == MACREG_INT_CODE_MIC_ERR_UNICAST) {
		strcat(buf, "unicast ");
	} else {
		strcat(buf, "multicast ");
	}
	
	send_iwevcustom_event(priv, buf);

	LEAVE();
}

/** 
 *  @brief This function handles the command response of reg_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param type	   the type of reg access (MAC, BBP or RF)
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_reg_access(wlan_private * priv,
	       			u16 type, 
			        struct HostCmd_DS_COMMAND * resp)
{
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

    	switch (type) {
	    	case HostCmd_RET_MAC_REG_ACCESS:
			{
				struct HostCmd_DS_MAC_REG_ACCESS *reg;

				reg = (struct HostCmd_DS_MAC_REG_ACCESS *) & 
				resp->params.macreg;

				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
				break;
			}
	
	    	case HostCmd_RET_BBP_REG_ACCESS:
			{
				struct HostCmd_DS_BBP_REG_ACCESS *reg;
				reg = (struct HostCmd_DS_BBP_REG_ACCESS *) & 
					resp->params.bbpreg;

				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
				break;
			}

	    	case HostCmd_RET_RF_REG_ACCESS:
			{
				struct HostCmd_DS_RF_REG_ACCESS *reg;
				reg = (struct HostCmd_DS_RF_REG_ACCESS *) & 
					resp->params.rfreg;
				
				Adapter->OffsetValue.offset = reg->Offset;
				Adapter->OffsetValue.value = reg->Value;
	    			break;
			}

		default:
			LEAVE();
			return WLAN_STATUS_FAILURE;
	}

    	LEAVE();
    	return WLAN_STATUS_SUCCESS;
}
 
/** 
 *  @brief This function handles the command response of get_hw_spec
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_get_hw_spec(wlan_private * priv, 
				struct HostCmd_DS_COMMAND * resp)
{
	u32			i;
	struct HostCmd_DS_GET_HW_SPEC	*hwspec = &resp->params.hwspec;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;

	ENTER();

	Adapter->HardwareStatus = WlanHardwareStatusReady;
	Adapter->fwCapInfo = wlan_le32_to_cpu(hwspec->fwCapInfo);


	/* permanent address should only be set once at start up */
	if (Adapter->PermanentAddr[0] == 0xff) {
		/* permanent address has not been set yet, set it */
		memcpy(Adapter->PermanentAddr, hwspec->PermanentAddr,
							MRVDRV_ETH_ADDR_LEN);
	}

	memcpy(priv->wlan_dev.netdev->dev_addr, hwspec->PermanentAddr,
								ETH_ALEN);
	Adapter->FWReleaseNumber = hwspec->FWReleaseNumber;

	PRINTM(INFO, "GET_HW_SPEC: FWReleaseVersion- 0x%X\n", Adapter->FWReleaseNumber);
	PRINTM(INFO, "GET_HW_SPEC: Permanent addr- %2x:%2x:%2x:%2x:%2x:%2x\n",
			hwspec->PermanentAddr[0], hwspec->PermanentAddr[1],
			hwspec->PermanentAddr[2], hwspec->PermanentAddr[3],
			hwspec->PermanentAddr[4], hwspec->PermanentAddr[5]);
	PRINTM(INFO, "GET_HW_SPEC: HWIfVersion=0x%X  Version=0x%X\n",hwspec->HWIfVersion,
							hwspec->Version);

	Adapter->RegionCode = wlan_le16_to_cpu(hwspec->RegionCode);

	for (i = 0; i < MRVDRV_MAX_REGION_CODE; i++) {
		/* use the region code to search for the index */
		if (Adapter->RegionCode == RegionCodeToIndex[i]) {
			Adapter->RegionTableIndex = (u16) i;
			break;
		}
	}

//	/* if it's unidentified region code, use the default (USA) */
//	if (i >= MRVDRV_MAX_REGION_CODE) {
		Adapter->RegionCode = 0x10;
		Adapter->RegionTableIndex = 0;
		PRINTM(WARN, "unidentified region code, use the default (USA)\n");
//	}

	if (Adapter->CurrentAddr[0] == 0xff) {
		memmove(Adapter->CurrentAddr, hwspec->PermanentAddr,
				MRVDRV_ETH_ADDR_LEN);
	}

	if (wlan_set_regiontable(priv, Adapter->RegionCode, 0)) {
			ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	if (wlan_set_universaltable(priv, 0)) {
			ret = WLAN_STATUS_FAILURE;
		goto done;
	}


done:
	LEAVE();
	return ret;
}


/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_sleep_params(wlan_private * priv,
			      		struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_SLEEP_PARAMS 	*sp = &resp->params.sleep_params;
	wlan_adapter   			*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "error=%x offset=%x stabletime=%x calcontrol=%x\n" 
			" extsleepclk=%x\n", sp->Error, sp->Offset, 
			sp->StableTime, sp->CalControl, sp->ExternalSleepClk);
	Adapter->sp.sp_error = wlan_le16_to_cpu(sp->Error);
	Adapter->sp.sp_offset = wlan_le16_to_cpu(sp->Offset);
	Adapter->sp.sp_stabletime = wlan_le16_to_cpu(sp->StableTime);
	Adapter->sp.sp_calcontrol = wlan_le16_to_cpu(sp->CalControl);
	Adapter->sp.sp_extsleepclk = wlan_le16_to_cpu(sp->ExternalSleepClk);
	Adapter->sp.sp_reserved = wlan_le16_to_cpu(sp->Reserved);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_sleep_period(wlan_private * priv,
			      		struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_SLEEP_PERIOD	*sp_period = &resp->params.ps_sleeppd;
	wlan_adapter   			*Adapter = priv->adapter;

	ENTER();

	Adapter->sleep_period.period = wlan_le16_to_cpu(sp_period->Period);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function handles the command response of mac_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_mac_control(wlan_private * priv,
	       			struct HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function handles the command response of set_wep
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_set_wep(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function handles the command response of reset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_reset(wlan_private * priv,
	       			struct HostCmd_DS_COMMAND * resp)
{
    	ENTER();
    	PRINTM(INFO, "HWAC - Reset command successful\n");

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief This function handles the command response of statistics
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_stat(wlan_private * priv, struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_GET_STAT	*p11Stat = &resp->params.gstat;
	wlan_adapter			*Adapter = priv->adapter;
	
	/* TODO Convert it to Big endian befor copy */
	memcpy(&Adapter->wlan802_11Stat,
		       p11Stat, 
		       sizeof(struct HostCmd_DS_802_11_GET_STAT));
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of snmp_mib
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_snmp_mib(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_SNMP_MIB	*smib = &resp->params.smib;
	u16				OID = wlan_le16_to_cpu(smib->OID);
	u16				QueryType = wlan_le16_to_cpu(smib->QueryType);

	ENTER();

	PRINTM(INFO, "SNMP_RESP: value of the OID = %x, QueryType=%x\n", OID,QueryType);
	PRINTM(INFO, "SNMP_RESP: Buf size  = %x\n", wlan_le16_to_cpu(smib->BufSize));
	
	if(QueryType == HostCmd_ACT_GEN_GET){
		switch(OID){
			case FragThresh_i:
				priv->adapter->FragThsd =  wlan_le16_to_cpu(*((unsigned short *) (smib->Value)));
				PRINTM(INFO, "SNMP_RESP: FragThsd =%u\n", priv->adapter->FragThsd);
				break;
			case RtsThresh_i:			
				priv->adapter->RTSThsd = wlan_le16_to_cpu(*((unsigned short *) (smib->Value)));
				PRINTM(INFO, "SNMP_RESP: RTSThsd =%u\n", priv->adapter->RTSThsd);
				break;
			case ShortRetryLim_i:	
				priv->adapter->TxRetryCount = wlan_le16_to_cpu(*((unsigned short *) (smib->Value)));
				PRINTM(INFO, "SNMP_RESP: TxRetryCount =%u\n", priv->adapter->RTSThsd);
				break;
			default:
				break;			
		}		
	}
		
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of radio_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_radio_control(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	ENTER();

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of key_material
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_key_material(wlan_private * priv,
					struct HostCmd_DS_COMMAND * resp) 
{
	struct HostCmd_DS_802_11_KEY_MATERIAL *pKey = &resp->params.keymaterial;
	wlan_adapter			*Adapter = priv->adapter;

	ENTER();

	if (pKey->Action == HostCmd_ACT_SET)
		Adapter->IsGTK_SET = TRUE;

	memcpy(Adapter->aeskey.KeyParamSet.Key, pKey->KeyParamSet.Key,
				sizeof(pKey->KeyParamSet.Key));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of mac_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_mac_address(wlan_private * priv,
					struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_MAC_ADDRESS	*MacAdd = &resp->params.macadd;
	wlan_adapter			*Adapter = priv->adapter;
	
	ENTER();
	
	memcpy(Adapter->CurrentAddr, MacAdd->MacAdd, ETH_ALEN);
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_tx_power
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_tx_power(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_RF_TX_POWER	*rtp = &resp->params.txp;
	wlan_adapter			*Adapter = priv->adapter;

	ENTER();

	Adapter->TxPowerLevel = wlan_le16_to_cpu(rtp->CurrentLevel); 
 
	PRINTM(INFO, "Current TxPower Level = %d\n", Adapter->TxPowerLevel);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_antenna
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_antenna(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_RF_ANTENNA* pAntenna = &resp->params.rant;
	wlan_adapter			*Adapter = priv->adapter;
	u16				Action = wlan_le16_to_cpu(pAntenna->Action);

	if (Action == HostCmd_ACT_GET_RX)
		Adapter->RxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

	if (Action == HostCmd_ACT_GET_TX)
		Adapter->TxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

	PRINTM(INFO, "RF_ANT_RESP: Action = 0x%x, Mode = 0x%04x\n",
			Action, wlan_le16_to_cpu(pAntenna->AntennaMode));

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of multicast_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_mac_multicast_adr(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rate_adapt_rateset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rate_adapt_rateset(wlan_private *priv,
						struct HostCmd_DS_COMMAND *resp)
{
	struct HostCmd_DS_802_11_RATE_ADAPT_RATESET	*rates = 
						&resp->params.rateset;
	wlan_adapter				*Adapter = priv->adapter;

	ENTER();

	if (rates->Action == HostCmd_ACT_GET) {
		Adapter->EnableHwAuto = rates->EnableHwAuto;
		Adapter->RateBitmap = rates->Bitmap;
	}

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of data_rate
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_data_rate(wlan_private * priv, 
					struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_DATA_RATE	*pDataRate = &resp->params.drate;
	wlan_adapter			*Adapter = priv->adapter;
	u8				Dot11DataRate;

	ENTER();

	HEXDUMP("DATA_RATE_RESP: data_rate- ",
			(u8 *)pDataRate,sizeof(struct HostCmd_DS_802_11_DATA_RATE));

	Dot11DataRate = pDataRate->DataRate[0];
	if (pDataRate->Action == HostCmd_ACT_GET_TX_RATE) 
	{
		memcpy(Adapter->SupportedRates, pDataRate->DataRate,
				sizeof(Adapter->SupportedRates));
	}
	Adapter->DataRate = index_to_data_rate(Dot11DataRate);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_channel
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rf_channel(wlan_private * priv,
				      struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_RF_CHANNEL	*rfchannel = &resp->params.rfchannel;
	wlan_adapter					*Adapter   = priv->adapter;
	u16 Action	   = wlan_le16_to_cpu(rfchannel->Action);
	u16 newChannel = wlan_le16_to_cpu(rfchannel->CurrentChannel);

	ENTER();

	if (Action == HostCmd_OPT_802_11_RF_CHANNEL_GET
		&& Adapter->CurBssParams.channel != newChannel)
	{
		PRINTM(INFO, "Channel Switch: %d to %d\n", 
			   Adapter->CurBssParams.channel, newChannel);


		/* Update the channel again */
		Adapter->CurBssParams.channel = newChannel;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rssi
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_802_11_rssi(wlan_private * priv, 
				struct HostCmd_DS_COMMAND * resp)
{
	struct HostCmd_DS_802_11_RSSI_RSP *rssirsp = &resp->params.rssirsp;
	wlan_adapter		*Adapter = priv->adapter;

	/* store the non average value */
	Adapter->SNR[TYPE_BEACON][TYPE_NOAVG]	= wlan_le16_to_cpu(rssirsp->SNR);
	Adapter->NF[TYPE_BEACON][TYPE_NOAVG]	= wlan_le16_to_cpu(rssirsp->NoiseFloor);
	    
	Adapter->SNR[TYPE_BEACON][TYPE_AVG] = wlan_le16_to_cpu(rssirsp->AvgSNR);
	Adapter->NF[TYPE_BEACON][TYPE_AVG] = wlan_le16_to_cpu(rssirsp->AvgNoiseFloor);

	Adapter->RSSI[TYPE_BEACON][TYPE_NOAVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
				Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);

	Adapter->RSSI[TYPE_BEACON][TYPE_AVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_AVG] / AVG_SCALE,
			Adapter->NF[TYPE_BEACON][TYPE_AVG] / AVG_SCALE);
    
	PRINTM(INFO, "Beacon RSSI value = 0x%x\n", 
		Adapter->RSSI[TYPE_BEACON][TYPE_AVG]);

	return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief This function handles the command response of eeprom_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_eeprom_access(wlan_private *priv,
	       				struct HostCmd_DS_COMMAND *resp)	
{
	wlan_adapter    *Adapter = priv->adapter;
	wlan_ioctl_regrdwr	*pBuf = (wlan_ioctl_regrdwr*)Adapter -> pRdeeprom;
	PRINTM(INFO,"eeprom read len=%x\n",wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount));
	if(pBuf->NOB < wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount)){
		pBuf->NOB = 0;
		PRINTM(INFO, "eeprom read return length is too big\n");
		return WLAN_STATUS_FAILURE;	
	}
	pBuf->NOB =  wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount);
	if(pBuf->NOB > 0){
		
		memcpy(&pBuf->Value, (u8 *) &resp -> params.rdeeprom.Value, 
				wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount));
		HEXDUMP("Adapter", (char *)&pBuf->Value, wlan_le16_to_cpu(resp -> params.rdeeprom.ByteCount));
	}
	return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief This function handles the command response of get_log
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_ret_get_log(wlan_private *priv,
	       			struct HostCmd_DS_COMMAND *resp)
{
	struct HostCmd_DS_802_11_GET_LOG  *LogMessage = 
			(struct HostCmd_DS_802_11_GET_LOG *)&resp->params.glog;
	wlan_adapter 	*Adapter = priv->adapter;

	ENTER();

	/* TODO Convert it to Big Endian before copy */
	memcpy(&Adapter->LogMsg, LogMessage, 
					sizeof(struct HostCmd_DS_802_11_GET_LOG));

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function handles the command response
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_process_rx_command(wlan_private * priv)
{
	u16			RespCmd;
	struct HostCmd_DS_COMMAND	*resp;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;
	ulong			flags;
	u16			Result;

	ENTER();

	PRINTM(INFO, "CMD_RESP: @ %lu\n", jiffies);

	/* Now we got response from FW, cancel the command timer */
	if (Adapter->CommandTimerIsSet) {
		CancelTimer(&Adapter->MrvDrvCommandTimer);
		Adapter->CommandTimerIsSet = FALSE;
	}

	if (!Adapter->CurCmd) {
		PRINTM(INFO, "CMD_RESP: NULL CurCmd=%p\n", Adapter->CurCmd);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	resp = (struct HostCmd_DS_COMMAND *) (Adapter->CurCmd->BufVirtualAddr);

	HEXDUMP("CMD_RESP:", Adapter->CurCmd->BufVirtualAddr,
						priv->wlan_dev.upld_len);

	RespCmd = wlan_le16_to_cpu(resp->Command);

	Result = wlan_le16_to_cpu(resp->Result);

	PRINTM(INFO, "CMD_RESP: %x Result: %d Length: %d\n", RespCmd,
			Result, priv->wlan_dev.upld_len);
	
	if (!(RespCmd & 0x8000)) {
		PRINTM(INFO, "Invalid response to command!");
		Adapter->CurCmdRetCode = WLAN_STATUS_FAILURE;
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
		
		ret = WLAN_STATUS_FAILURE;
		goto done;		
	}

	/* Store the response code to CurCmdRetCode. */
	Adapter->CurCmdRetCode = wlan_le16_to_cpu(resp->Result);

	if (RespCmd == HostCmd_RET_802_11_PS_MODE) {
		struct HostCmd_DS_802_11_PS_MODE *psmode;

		psmode = &resp->params.psmode;
		PRINTM(INFO, "CMD_RESP: PS_MODE cmd reply result=%#x action=0x%X\n",
						resp->Result, psmode->Action);
		psmode->Action = wlan_cpu_to_le16(psmode->Action);

		if (Result) {
			PRINTM(INFO, "CMD_RESP: PS command failed- %#x \n", resp->Result);
			if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
				/* 
				 * We should not re-try enter-ps command in 
				 * ad-hoc mode. It takes place in 
				 * ExecuteNextCommand().
				 */
				if (psmode->Action == HostCmd_SubCmd_Enter_PS)
					Adapter->PSMode = 
						Wlan802_11PowerModeCAM;
			}
		} else if (psmode->Action == HostCmd_SubCmd_Enter_PS) {
			Adapter->NeedToWakeup = FALSE;
			Adapter->PSState = PS_STATE_AWAKE;
			PRINTM(INFO, "CMD_RESP: Enter_PS command response\n");
			if (Adapter->MediaConnectStatus != WlanMediaStateConnected){
				/*
				 * When Deauth Event received before Enter_PS command
				 * response, We need to wake up the firmware.
				 */
				PRINTM(INFO, "Disconnected, Going to invoke PSWakeup\n");
				PSWakeup(priv, 0);
			}
		} else if (psmode->Action == HostCmd_SubCmd_Exit_PS) {
			Adapter->NeedToWakeup = FALSE;
			Adapter->PSState = PS_STATE_FULL_POWER;
			PRINTM(INFO, "CMD_RESP: Exit_PS command response\n");
		} else {
			PRINTM(INFO, "CMD_RESP: PS- Action=0x%X\n",psmode->Action);
		}

		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}

	if (Adapter->CurCmd->CmdFlags & CMD_F_HOSTCMD) {
		/* Copy the response back to response buffer */
		memcpy(Adapter->CurCmd->pdata_buf,
				resp, resp->Size);

		Adapter->CurCmd->CmdFlags &= ~CMD_F_HOSTCMD;
	}
    
	/* If the command is not successful, cleanup and return failure */
	if ((Result != HostCmd_RESULT_OK || !(RespCmd & 0x8000))) {
		PRINTM(INFO, "CMD_RESP: command reply %#x result=%#x\n",
				resp->Command, resp->Result);
		/*
		 * Handling errors here
		 */
		switch (RespCmd) {
		case HostCmd_RET_HW_SPEC_INFO:
		case HostCmd_RET_802_11_RESET:
			PRINTM(INFO, "CMD_RESP: Reset command Failed\n");
			Adapter->HardwareStatus = WlanHardwareStatusNotReady;
			break;

		}
		
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

		return WLAN_STATUS_FAILURE;
	}

	switch (RespCmd) {
	case HostCmd_RET_MAC_REG_ACCESS:
	case HostCmd_RET_BBP_REG_ACCESS:
	case HostCmd_RET_RF_REG_ACCESS:
		ret = wlan_ret_reg_access(priv, RespCmd, resp);
		break;
	    
	case HostCmd_RET_HW_SPEC_INFO:
		ret = wlan_ret_get_hw_spec(priv, resp);
		break;

	case HostCmd_RET_802_11_SCAN:
		ret = wlan_ret_802_11_scan(priv, resp);
        break;

	case HostCmd_RET_MAC_CONTROL:
		ret = wlan_ret_mac_control(priv, resp);
		break;
			
	case HostCmd_RET_802_11_GET_LOG:
		ret = wlan_ret_get_log(priv, resp);		
		break;

	case HostCmd_RET_802_11_ASSOCIATE:
	case HostCmd_RET_802_11_REASSOCIATE:
		ret = wlan_ret_802_11_associate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_DISASSOCIATE:
	case HostCmd_RET_802_11_DEAUTHENTICATE:
		ret = wlan_ret_802_11_disassociate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_SET_WEP:
		ret = wlan_ret_802_11_set_wep(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_AD_HOC_START:
	case HostCmd_RET_802_11_AD_HOC_JOIN:
		ret = wlan_ret_802_11_ad_hoc_start(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RESET:
		ret = wlan_ret_802_11_reset(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_AUTHENTICATE:
		ret = wlan_ret_802_11_authenticate(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_STAT:
		ret = wlan_ret_802_11_stat(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_SNMP_MIB:
		ret = wlan_ret_802_11_snmp_mib(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RF_TX_POWER:
		ret = wlan_ret_802_11_rf_tx_power(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_RADIO_CONTROL:
		ret = wlan_ret_802_11_radio_control(priv, resp);
		break;



	case HostCmd_RET_802_11_SET_AFC:
	case HostCmd_RET_802_11_GET_AFC:
		memmove(Adapter->CurCmd->pdata_buf, 
			&resp->params.afc,
			sizeof(struct HostCmd_DS_802_11_AFC));

		break;
	case HostCmd_RET_802_11_RF_ANTENNA:
		ret = wlan_ret_802_11_rf_antenna(priv, resp);
		break;
	    
	case HostCmd_RET_MAC_MULTICAST_ADR:
		ret = wlan_ret_mac_multicast_adr(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_DATA_RATE:
		ret = wlan_ret_802_11_data_rate(priv, resp);
		break;
	case HostCmd_RET_802_11_RATE_ADAPT_RATESET:
	    	ret = wlan_ret_802_11_rate_adapt_rateset(priv, resp);
		break;
	case HostCmd_RET_802_11_RF_CHANNEL:
		ret = wlan_ret_802_11_rf_channel(priv, resp);
		break;
	    		
	case HostCmd_RET_802_11_RSSI:
		ret = wlan_ret_802_11_rssi(priv, resp);
		break;

	case HostCmd_RET_802_11_MAC_ADDRESS:
		ret = wlan_ret_802_11_mac_address(priv, resp);
		break;

	case HostCmd_RET_802_11_AD_HOC_STOP:
		ret = wlan_ret_802_11_ad_hoc_stop(priv, resp);
		break;
	    
	case HostCmd_RET_802_11_BEACON_STOP:
		break;
			
	case HostCmd_RET_802_11_ENABLE_RSN:
		break;
	case HostCmd_RET_802_11_KEY_MATERIAL:
		PRINTM(INFO, "CMD_RESP: KEY_MATERIAL command response\n"); 
		ret = wlan_ret_802_11_key_material(priv, resp); 
		break;
			
	case HostCmd_RET_802_11_EEPROM_ACCESS:
		ret = wlan_ret_802_11_eeprom_access(priv, resp);
		break;

				

	case HostCmd_RET_802_11D_DOMAIN_INFO:
		ret = wlan_ret_802_11d_domain_info(priv, resp);
		break;

	case HostCmd_RET_802_11_SLEEP_PARAMS:		
		ret = wlan_ret_802_11_sleep_params(priv, resp);
		break;
	case HostCmd_RET_802_11_INACTIVITY_TIMEOUT:
		*((u16 *)Adapter->CurCmd->pdata_buf) =
			wlan_le16_to_cpu(resp->params.inactivity_timeout.Timeout);
		break;

		
	case HostCmd_RET_802_11_SLEEP_PERIOD:
		ret = wlan_ret_802_11_sleep_period(priv, resp);
		break;
	case HostCmd_RET_802_11_TPC_CFG:
		memmove(Adapter->CurCmd->pdata_buf, 
		        &resp->params.tpccfg,
		        sizeof(struct HostCmd_DS_802_11_TPC_CFG));
		break;
	case HostCmd_RET_802_11_LED_GPIO_CTRL:
		memmove(Adapter->CurCmd->pdata_buf,
		        &resp->params.ledgpio,
		        sizeof(struct HostCmd_DS_802_11_LED_CTRL));
		break;	
	case HostCmd_RET_802_11_PWR_CFG:
		memmove(Adapter->CurCmd->pdata_buf,
		        &resp->params.pwrcfg,
		        sizeof(struct HostCmd_DS_802_11_PWR_CFG));

		break;

	case HostCmd_RET_GET_TSF:		
	    memcpy(priv->adapter->CurCmd->pdata_buf,
               &resp->params.gettsf.TsfValue,
               sizeof(u64));
	    break;
	case  HostCmd_RTE_802_11_TX_RATE_QUERY:
	    priv->adapter->TxRate = resp->params.txrate.TxRate;
	    break;
	default:
		PRINTM(INFO, "CMD_RESP: Unknown command response %#x\n",
				resp->Command);
		break;
	}

	if (Adapter->CurCmd) {
		/* Clean up and Put current command back to CmdFreeQ */
		CleanupAndInsertCmd(priv, Adapter->CurCmd);
		spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
		Adapter->CurCmd = NULL;
		spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
	}
        
done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function handles events generated by firmware
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_process_event(wlan_private * priv)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "EVENT Cause %x\n", Adapter->EventCause);

	switch (Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT) {
		case MACREG_INT_CODE_LINK_SENSED:
			PRINTM(INFO, "EVENT: MACREG_INT_CODE_LINK_SENSED\n");
			break;

		case MACREG_INT_CODE_DEAUTHENTICATED:
			PRINTM(INFO, "EVENT: Deauthenticated\n");
			HandleDisconnectEvent(priv);
			break;

		case MACREG_INT_CODE_DISASSOCIATED:
			PRINTM(INFO, "EVENT: Disassociated\n");
			HandleDisconnectEvent(priv);
			break;

		case MACREG_INT_CODE_LINK_LOSE_NO_SCAN:
			PRINTM(INFO, "EVENT: Link lost\n");
			HandleDisconnectEvent(priv);
			break;

		case MACREG_INT_CODE_PS_SLEEP:
			PRINTM(INFO, "EVENT: SLEEP\n");
			PRINTM(INFO, "_");

			/* handle unexpected PS SLEEP event */
			if (Adapter->PSState == PS_STATE_FULL_POWER) {
				PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS SLEEP\n");
				break;
			}
			Adapter->PSState = PS_STATE_PRE_SLEEP;

			PSConfirmSleep(priv, (u16) Adapter->PSMode);

			break;

		case MACREG_INT_CODE_PS_AWAKE:
			PRINTM(INFO, "EVENT: AWAKE \n");
			PRINTM(INFO, "|");

			/* handle unexpected PS AWAKE event */
			if (Adapter->PSState == PS_STATE_FULL_POWER) {
				PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS AWAKE\n");
				break;
			}

			Adapter->TxLockFlag = 0;
			if(TRUE == CheckLastPacketIndication(priv)){
				if(Adapter->gen_null_pkg)
				{
					SendNullPacket(priv,    MRVDRV_TxPD_POWER_MGMT_NULL_PACKET | 
							MRVDRV_TxPD_POWER_MGMT_LAST_PACKET);	
					Adapter->TxLockFlag = 1;
				}					
			}	
				
			Adapter->PSState = PS_STATE_AWAKE;

			if (Adapter->NeedToWakeup == TRUE) {
				/*
				 * wait for the command processing to finish
				 * before resuming sending 
				 * Adapter->NeedToWakeup will be set to FALSE 
				 * in PSWakup()
				 */
				PRINTM(INFO, "Waking up...\n");
				PSWakeup(priv, 0);
			}
			break;



		case MACREG_INT_CODE_MIC_ERR_UNICAST:
			PRINTM(INFO, "EVENT: UNICAST MIC ERROR\n");
			HandleMICFailureEvent(priv, 
					MACREG_INT_CODE_MIC_ERR_UNICAST);
			break;
		
		case MACREG_INT_CODE_MIC_ERR_MULTICAST:
			PRINTM(INFO, "EVENT: MULTICAST MIC ERROR\n");
			HandleMICFailureEvent(priv, 
					MACREG_INT_CODE_MIC_ERR_MULTICAST);
			break;
		case MACREG_INT_CODE_MIB_CHANGED:
		case MACREG_INT_CODE_INIT_DONE:
			break;

		case MACREG_INT_CODE_ADHOC_BCN_LOST:
			PRINTM(INFO, "EVENT: HWAC - ADHOC BCN LOST\n");		
			break;

	case MACREG_INT_CODE_RSSI_LOW:
		PRINTM(MSG, "EVENT: RSSI_LOW\n");		
		break;
	case MACREG_INT_CODE_SNR_LOW:
		PRINTM(MSG, "EVENT: SNR_LOW\n");		
		break;
	case MACREG_INT_CODE_MAX_FAIL:
		PRINTM(MSG, "EVENT: MAX_FAIL\n");		
		break;
	case MACREG_INT_CODE_RSSI_HIGH:
		PRINTM(MSG, "EVENT: RSSI_HIGH\n");		
		break;
	case MACREG_INT_CODE_SNR_HIGH:
		PRINTM(MSG, "EVENT: SNR_HIGH\n");		
		break;
		
	default:
		PRINTM(INFO, "EVENT: unknown event id: %#x\n",
			   Adapter->EventCause >> SBI_EVENT_CAUSE_SHIFT);
		break;
	}

	Adapter->EventCause = 0;
	LEAVE();
	return ret;
}

