/*
 * Copyright (c) 1995, 1996, 1997 by Internet Software Consortium
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* ev_connects.c - implement asynch connect/accept for the eventlib
 * vix 16sep96 [initial]
 */

#if !defined(LINT) && !defined(CODECENTER)
static const char rcsid[] = "$Id: ev_connects.c,v 8.12 1997/12/03 22:32:53 halley Exp $";
#endif

/* Import. */

#include "port_before.h"
#include "fd_setsize.h"

#include <sys/types.h>
#include <sys/socket.h>

#include <assert.h>
#include <unistd.h>

#include <isc/eventlib.h>
#include "eventlib_p.h"

#include "port_after.h"

/* Forward. */

static void	listener(evContext ctx, void *uap, int fd, int evmask);
static void	connector(evContext ctx, void *uap, int fd, int evmask);

/* Public. */

int
evListen(evContext opaqueCtx, int fd, int maxconn,
	 evConnFunc func, void *uap, evConnID *id)
{
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *new;
	int mode;

	OKNEW(new);
	new->flags = EV_CONN_LISTEN;
	OK(mode = fcntl(fd, F_GETFL, NULL));	/* side effect: validate fd. */
	/*
	 * Remember the nonblocking status.  We assume that either evSelectFD
	 * has not been done to this fd, or that if it has then the caller
	 * will evCancelConn before they evDeselectFD.  If our assumptions
	 * are not met, then we might restore the old nonblocking status
	 * incorrectly.
	 */
	if ((mode & O_NONBLOCK) == 0) {
		OK(fcntl(fd, F_SETFL, mode | O_NONBLOCK));
		new->flags |= EV_CONN_BLOCK;
	}
	OK(listen(fd, maxconn));
	if (evSelectFD(opaqueCtx, fd, EV_READ, listener, new, &new->file) < 0){
		int save = errno;

		FREE(new);
		errno = save;
		return (-1);
	}
	new->flags |= EV_CONN_SELECTED;
	new->func = func;
	new->uap = uap;
	new->fd = fd;
	if (ctx->conns != NULL)
		ctx->conns->prev = new;
	new->prev = NULL;
	new->next = ctx->conns;
	ctx->conns = new;
	new->prevDone = NULL;
	new->nextDone = NULL;
	if (id)
		id->opaque = new;
	return (0);
}

int
evConnect(evContext opaqueCtx, int fd, void *ra, int ralen,
	  evConnFunc func, void *uap, evConnID *id)
{
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *new;

	OKNEW(new);
	new->flags = 0;
	/* Do the select() first to get the socket into nonblocking mode. */
	if (evSelectFD(opaqueCtx, fd, EV_MASK_ALL,
		       connector, new, &new->file) < 0) {
		int save = errno;

		FREE(new);
		errno = save;
		return (-1);
	}
	new->flags |= EV_CONN_SELECTED;
	if (connect(fd, ra, ralen) < 0 &&
	    errno != EWOULDBLOCK &&
	    errno != EAGAIN &&
	    errno != EINPROGRESS) {
		int save = errno;

		(void) evDeselectFD(opaqueCtx, new->file);
		FREE(new);
		errno = save;
		return (-1);
	}
	/* No error, or EWOULDBLOCK.  select() tells when it's ready. */
	new->func = func;
	new->uap = uap;
	new->fd = fd;
	if (ctx->conns != NULL)
		ctx->conns->prev = new;
	new->prev = NULL;
	new->next = ctx->conns;
	ctx->conns = new;
	new->prevDone = NULL;
	new->nextDone = NULL;
	if (id)
		id->opaque = new;
	return (0);
}

int
evCancelConn(evContext opaqueCtx, evConnID id) {
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *this = id.opaque;
	int mode;

	if ((this->flags & EV_CONN_SELECTED) != 0)
		(void) evDeselectFD(opaqueCtx, this->file);
	if ((this->flags & EV_CONN_BLOCK) != 0) {
		mode = fcntl(this->fd, F_GETFL, NULL);
		if (mode == -1) {
			if (errno != EBADF)
				return (-1);
		} else
			OK(fcntl(this->fd, F_SETFL, mode | O_NONBLOCK));
	}
	
	/*
	 * The connections list is doubly threaded.  First, there's ctx->conns
	 * that's used by evDestroy() to find and cancel all connections.
	 * Second, there's ctx->connDone (head) and ctx->connLast (tail) which
	 * threads through the potentially smaller number of completed
	 * connections and is used to allow connections found with
	 * evTryAccept() to be dispatched cleanly from the main event loop.
	 */

	/* Unlink from ctx->conns. */
	if (this->prev != NULL)
		this->prev->next = this->next;
	else
		ctx->conns = this->next;
	if (this->next != NULL)
		this->next->prev = this->prev;

	/*
	 * If 'this' is on the ctx->connDone list, remove it.  Update
	 * ctx->connLast if necessary.
	 */
	if (this->prevDone == NULL && this->nextDone == NULL) {
		/*
		 * Either 'this' is the only item on the done list, or it's
		 * not on the done list.  If the former, then we unlink it
		 * from the list.  If the latter, we leave the list alone.
		 */
		if (ctx->connDone == this) {
			ctx->connDone = NULL;
			ctx->connLast = NULL;
		}
	} else {
		if (this->prevDone != NULL)
			this->prevDone->nextDone = this->nextDone;
		else
			ctx->connDone = this->nextDone;
		if (this->nextDone != NULL)
			this->nextDone->prevDone = this->prevDone;
		else
			ctx->connLast = this->prevDone;
	}

	FREE(this);
	return (0);
}

int evHold(evContext opaqueCtx, evConnID id) {
	evConn *this = id.opaque;

	if ((this->flags & EV_CONN_LISTEN) == 0) {
		errno = EINVAL;
		return (-1);
	}
	if ((this->flags & EV_CONN_SELECTED) == 0)
		return (0);
	this->flags &= ~EV_CONN_SELECTED;
	return (evDeselectFD(opaqueCtx, this->file));
}

int evUnhold(evContext opaqueCtx, evConnID id) {
	evConn *this = id.opaque;
	int ret;

	if ((this->flags & EV_CONN_LISTEN) == 0) {
		errno = EINVAL;
		return (-1);
	}
	if ((this->flags & EV_CONN_SELECTED) != 0)
		return (0);
	ret = evSelectFD(opaqueCtx, this->fd, EV_READ, listener, this,
			 &this->file);
	if (ret == 0)
		this->flags |= EV_CONN_SELECTED;
	return (ret);
}

int
evTryAccept(evContext opaqueCtx, evConnID id) {
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *conn = id.opaque;
	int new;

	if ((conn->flags & EV_CONN_LISTEN) == 0 ||
	    (conn->flags & EV_CONN_SELECTED) != 0) {
		errno = EINVAL;
		return (-1);
	}
	conn->ralen = sizeof conn->ra;
	conn->newfd = accept(conn->fd, &conn->ra, &conn->ralen);
	if (conn->newfd >= 0) {
		conn->lalen = sizeof conn->la;
		if (getsockname(conn->newfd, &conn->la, &conn->lalen) < 0) {
			conn->ioErrno = errno;
			(void) close(conn->newfd);
			conn->newfd = -1;
		} else
			conn->ioErrno = 0;
	} else {
		conn->ioErrno = errno;
		if (errno = EAGAIN || errno == EWOULDBLOCK)
			return (-1);
	}
	if (ctx->connLast != NULL)
		ctx->connLast->next = conn;
	else
		ctx->connDone = conn;
	conn->prevDone = ctx->connLast;
	conn->nextDone = NULL;
	ctx->connLast = conn;
	return (0);
}

/* Private. */

static void
listener(evContext opaqueCtx, void *uap, int fd, int evmask) {
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *conn = uap;
	struct sockaddr la, ra;
	int new, lalen, ralen;

	assert((evmask & EV_READ) != 0);
	ralen = sizeof ra;
	new = accept(fd, &ra, &ralen);
	if (new >= 0) {
		lalen = sizeof la;
		if (getsockname(new, &la, &lalen) < 0) {
			int save = errno;

			(void) close(new);
			errno = save;
			new = -1;
		}
	}
	(*conn->func)(opaqueCtx, conn->uap, new, &la, lalen, &ra, ralen);
}

static void
connector(evContext opaqueCtx, void *uap, int fd, int evmask) {
	evContext_p *ctx = opaqueCtx.opaque;
	evConn *conn = uap;
	struct sockaddr la, ra;
	int lalen, ralen;
	char buf[1];
	void *conn_uap;
	evConnFunc conn_func;
	evConnID id;

	lalen = sizeof la;
	ralen = sizeof ra;
	conn_uap = conn->uap;
	conn_func = conn->func;
	id.opaque = conn;
	if (evCancelConn(opaqueCtx, id) < 0 ||
	    getsockname(fd, &la, &lalen) < 0 ||
	    getpeername(fd, &ra, &ralen) < 0 ||
#ifdef NETREAD_BROKEN
	    0
#else
	    read(fd, buf, 0) < 0
#endif
	    ) {
		int save = errno;

		(void) close(fd);	/* XXX closing caller's fd */
		errno = save;
		fd = -1;
	}
	(*conn_func)(opaqueCtx, conn_uap, fd, &la, lalen, &ra, ralen);
}
