/**
 ** cmdlineo.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 ** cmdlineo.c
 ** v3.0 Sat Aug 10 19:51:54 1991  Randy Sargent
 **                           put globals into the Cmdline data structure so
 **                           you can have more than one command line
 ** v3.1 Thu Oct 17 11:57:12 1991  Randy Sargent
 **                           added backward compatibility functions
 **                           so old programs can use this module
 **/

/* implement tsch/emacs - like command line editing */

/* todo:
   meta-b,f
   parameterize meta-b-d-f, etc
   parameterize paren highlighting
   yank buffer
   incremental search
   */


#ifdef UNIX
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#endif

#include CONFIG
#include "util.h"
#include "iob.h"
#include "cmdlineo.h"

#include <ctype.h>

#ifdef PC
Int cmd_ansi= 0;
#else
Int cmd_ansi= 1;
#endif
Int cmd_no_reverse_wrap= 0;

Cmdline cmdline_global;  /* single global commandline for backwards
			    compatibility mode.  Yuk. */

Int cmd_string_len(Cmdline *cmdline, char *s);

/* 0 if successful */
Int cmdline_init(Cmdline *c, IOStream *in, FILE *out)
{
    memset(c, 0, sizeof (Cmdline));
    c->in= in;
    c->out= out;
    c->look_for_hangup= 0;
    c->carrier_detected= 0;
    return 0;
}

Int cmdline_init_default(Cmdline *c)
{
    return cmdline_init(c, io_open_stdin(), stdout);
}

void cmdline_term(Cmdline *c)
{
    Int i;
    for (i= 0; i< MAXHISTORY; i++) {
	if (c->history[i]) {
	    free(c->history[i]);
	    c->history[i]= 0;
	}
    }
}

#define cmd_char_len(c) (((c) & 127) < 32 ? 2 : 1)

void cmdline_ansi_enable(Int enable)
{
    cmd_ansi= enable;
}

void cmd_putc_repeat(Cmdline *cmdline, Int c, Int n)
{
    while(n-- > 0) {
	if (c == '\b') {
	    if (cmd_no_reverse_wrap && !(cmdline->char_pos % 80) && cmd_ansi) {
		Int i;
		/* oh shit */
		/* up line */
		io_putchar(27, cmdline->in); io_putchar('[', cmdline->in); io_putchar('A', cmdline->in);
		/* end of line */
		for (i= 0; i< 79; i++) {
		    io_putchar(27, cmdline->in); io_putchar('[', cmdline->in); io_putchar('C', cmdline->in);
		}
	    } else {
		io_putchar(c, cmdline->in);
	    }
	    cmdline->char_pos--;
	} else {
	    cmdline->char_pos++;
	    io_putchar(c, cmdline->in);
	}
    }
}

void cmd_put_char(Cmdline *cmdline, Int c)
{
#ifndef MAC
    if (cmd_ansi && (c & 128)) io_printf(cmdline->in, "\033[4m");
#endif    
    if ((c & 127) < 32) {
	cmdline->char_pos += 1;
	io_putchar('^', cmdline->in);
	io_putchar((c & 127) +'@', cmdline->in);
    } else {
#ifdef MAC    
    	io_putchar(c, cmdline->in);
#else    	
    	io_putchar(c & 127, cmdline->in);
#endif    	
    }
    cmdline->char_pos += 1;
#ifndef MAC    
    if (cmd_ansi && (c & 128)) io_printf(cmdline->in, "\033[m");
#endif    
}

void add_cursor(Cmdline *cmdline)
{
#ifdef MAC
    strcat(cmdline->line, " ");
    cmdline->line[cmdline->pos] |= 0x80;
#endif    
}

void remove_cursor(Cmdline *cmdline)
{
#ifdef MAC
    char *line= cmdline->line;
    Int len;
    len= strlen(line);
    if (len && (line[len-1] & 0x7f) == ' ') {
    	line[len-1]= 0;
    }
#endif    
}

void cmd_set_old(Cmdline *cmdline)
{
    add_cursor(cmdline);
    cmdline->old_pos= cmdline->pos;
    strcpy(cmdline->old_line, cmdline->line);
    remove_cursor(cmdline);
}

void cmd_update_line(Cmdline *cmdline, Int match_parens)
{
    Int  i, x, clear_len, level;
    char *s;

    for (i= 0; cmdline->line[i]; i++) cmdline->line[i] &= 127;
    if (match_parens) {
	if (cmdline->pos && strchr(")]}", cmdline->line[cmdline->pos-1])) {
	    for (level= 0, i= cmdline->pos-1; i >= 0; i--) {
		if (strchr(")]}", cmdline->line[i])) level++;
		if (strchr("([{", cmdline->line[i])) level--;
		if (!level) {
		    if (cmd_ansi) {
			cmdline->line[cmdline->pos-1] |= 128;
			cmdline->line[i] |= 128;
		    }
		    break;
		}
	    }
	}
    }
    if (match_parens) add_cursor(cmdline);
    for (i= 0; cmdline->old_line[i] == cmdline->line[i] && cmdline->line[i]; i++);

    if (cmdline->old_line[i] != cmdline->line[i]) {
	for (x= cmdline->old_pos; x< i; x++)
	  cmd_put_char(cmdline, cmdline->line[x]);
	for (x= i; x< cmdline->old_pos; x++)
	  cmd_putc_repeat(cmdline, '\b', cmd_char_len(cmdline->old_line[x]));
	for (s= &cmdline->line[i]; *s; s++) cmd_put_char(cmdline, *s);
	clear_len= cmd_string_len(cmdline, &cmdline->old_line[i]) - cmd_string_len(cmdline, &cmdline->line[i]);
	cmd_putc_repeat(cmdline, ' ',  clear_len);
	cmd_putc_repeat(cmdline, '\b', clear_len);
	cmd_putc_repeat(cmdline, '\b', cmd_string_len(cmdline, &cmdline->line[cmdline->pos]));
    } else {
	for (x= cmdline->old_pos; x< cmdline->pos; x++)
	  cmd_put_char(cmdline, cmdline->line[x]);
	for (x= cmdline->pos; x< cmdline->old_pos; x++)
	  cmd_putc_repeat(cmdline, '\b', cmd_char_len(cmdline->old_line[x]));
    }
    io_flush(cmdline->in);
    if (match_parens) remove_cursor(cmdline);
}

Int cmd_string_len(Cmdline *cmdline, char *s)
{
    Int ret= 0;
    UNUSED(cmdline);
    while (*s) ret += cmd_char_len(*s++);
    return ret;
}

void cmd_beep(Cmdline *cmdline)
{
    io_putchar(7, cmdline->in);
}

void cmdline_start_get_line(Cmdline *cmdline, char *prompt, char *line)
{
   /*printf(" start cmdline_start_get_line(%s)\n", line);*/
   cmdline->prompt= prompt;
   cmdline->line= line;
   cmdline->cmd_new= cmdline->current;
   
   io_printf(cmdline->in, "%s", prompt);
   
   cmdline->prompt= prompt;
   cmdline->char_pos= strlen(cmdline->prompt);
   
   cmdline->pos= 0;  cmdline->line= "";
   cmd_set_old(cmdline);
   cmdline->old_line[0]= 0;
   cmdline->line= line; cmdline->pos= strlen(line);
   cmd_update_line(cmdline, 1);
   cmdline->repeat= 1;
   cmdline->meta= 0;
   cmdline->cursor= 0;
   cmdline->quote= 0;
   /*printf(" end cmdline_start_get_line: >%s<(%d)\n",
	  cmdline->line, cmdline->pos);*/
}

/* For backwards compatibility with cmdline.c */

void cmd_get_line(char *prompt, IOStream *in, FILE *out, char *line)
{
    static initted= 0;
    if (!initted) {
	cmdline_init(&cmdline_global, in, out);
	initted= 1;
    }
    cmdline_global.in= in;
    cmdline_global.out= out;
    cmdline_get_line(&cmdline_global, prompt, line);
}

void cmdline_get_line(Cmdline *cmdline, char *prompt, char *line)
{
    Int c;
    cmdline_start_get_line(cmdline, prompt, line);
    while (1) {
	c= io_getchar(cmdline->in, -1);
	if (cmdline_process_key(cmdline, c) == 1) break;
    }
}

Int cmdline_getchar(Cmdline *cmdline, Int timeout)
{
    Int c;
    if (cmdline->look_for_hangup) {
	do {
	    Int this_timeout= timeout;
	    if (this_timeout > 10000 || this_timeout < 0)
	      this_timeout= 10000;
	    if (cmdline_poll_hangup(cmdline) == CMDLINE_HANGUP)
	      return CMDLINE_HANGUP;
	    c= io_getchar(cmdline->in, this_timeout);
	    if (c != EOF) return c;
	    if (timeout > 0) timeout -= this_timeout;
	} while (timeout);
	return EOF;
    }
    else {
	return io_getchar(cmdline->in, timeout);
    }
}

Int cmdline_poll_hangup(Cmdline *cmdline)
{
    Int detect= io_carrier_detect(cmdline->in);

    if (!detect && cmdline->carrier_detected) return CMDLINE_HANGUP;
    if (detect) cmdline->carrier_detected= 1;
    return 0;
}
	
Int cmdline_get_line_timeout(Cmdline *cmdline, char *prompt, char *line, Int timeout)
{
    Int c;
    cmdline_start_get_line(cmdline, prompt, line);
    while (1) {
	    
	    
	c= cmdline_getchar(cmdline, timeout);
	if (c == EOF || c == CMDLINE_HANGUP) return c;
	if (cmdline_process_key(cmdline, c) == 1) break;
    }
    return 0;
}

Int cmdline_get_key(Cmdline *cmdline)
{
    Int c;

    c= io_getchar(cmdline->in, 0);
    if (c == EOF) return -1;

    return cmdline_process_key(cmdline, c);
}
    
    
Int cmdline_process_key(Cmdline *cmdline, Int c)
{
    Int iter, oldc, newc, i, did_command;
    Int ctrl_c_pressed= 0;
    char *l;

    /*printf("before: cmd:>%s<(%d)\n", cmdline->line, cmdline->pos);*/
    if (cmdline->meta) {
	c |= 128;
	cmdline->meta= 0;
    }

    if (cmdline->cursor) {
	switch(c) {
	  case 'A': c= 16; break;  /* up arrow */
	  case 'B': c= 14; break;  /* down arrow */
	  case 'C': c=  6; break;  /* right arrow */
	  case 'D': c=  2; break;  /* left arrow */
	}
	cmdline->cursor= 0;
    }
    
    cmd_set_old(cmdline);

    did_command= 1;
    for (iter= 0; iter < cmdline->repeat; iter ++) {
	if (cmdline->quote) goto normal_char;
	switch(c) {
	  case 21:
	    cmdline->repeat *= 4;
	    if (cmdline->repeat > 256) cmdline->repeat= 256;
	    iter= cmdline->repeat;
	    did_command= 0;
	    break;
	  case 27:   /* esc */
	    cmdline->meta= 1;
	    did_command= 0;
	    break;
	  case 'Q'-64:
	    cmdline->quote= 1;
	    did_command= 0;
	    break;
	  case '['+128:
	    cmdline->cursor= 1;
	    did_command= 0;
	    break;
	  case 'd'+128: case 'D'+128:
	    for (i= cmdline->pos; cmdline->line[i] && !isalnum(cmdline->line[i]); i++);
	    for (; cmdline->line[i] && isalnum(cmdline->line[i]); i++);
	    strcpy(&cmdline->line[cmdline->pos], &cmdline->line[i]);
	    break;
	  case 8  +128: case 127+128:
	    for (i= cmdline->pos-1; i >=0 && !isalnum(cmdline->line[i]); i--);
	    for (; i >= 0 && isalnum(cmdline->line[i]); i--);
	    strcpy(&cmdline->line[i+1], &cmdline->line[cmdline->pos]);
	    cmdline->pos= i+1;
	    break;
	  case '\n': /* return */
	  case '\r': /* return */
	    /*printf("after: CR cmd:>%s<(%d)\n", cmdline->line, cmdline->pos);*/
	    cmdline->pos= strlen(cmdline->line);
	    cmd_update_line(cmdline, 0);
	    if (cmdline->pos) {
		if (cmdline->history[cmdline->current]) free(cmdline->history[cmdline->current]);
		cmdline->history[cmdline->current]= malloc(strlen(cmdline->line)+1);
		strcpy(cmdline->history[cmdline->current], cmdline->line);
		cmdline->current= (cmdline->current + 1) % MAXHISTORY;
	    }
	    return 1;
	  case '!':
	    if (cmdline->pos == 1 && cmdline->line[0] == '!' && !cmdline->line[1]) c= 16;
	    else goto normal_char;
          case IO_UPARROW: c= 16; goto change_line;
          case IO_DOWNARROW: c=14; goto change_line;
          change_line:
	  case 14:   /* ^N */
	  case 16:   /* ^P */
	    cmdline->cmd_new= (cmdline->cmd_new + MAXHISTORY + 15 - c) % MAXHISTORY;
	    if (!cmdline->history[cmdline->cmd_new]) {
		cmdline->cmd_new = (cmdline->cmd_new - 15 + c) % MAXHISTORY;
		goto error;
	    }
	  set_line:
	    strcpy(cmdline->line, cmdline->history[cmdline->cmd_new]);
	    /* FALL THROUGH TO END OF LINE */
	  case 5: case IO_END:   /* ^E */
	    cmdline->pos= strlen(cmdline->line);
	    break;
	  case 1: case IO_HOME:   /* ^A */
	    cmdline->pos= 0;
	    break;
	  case 6: case IO_RIGHTARROW:   /* ^F */
	    if (cmdline->line[cmdline->pos]) cmdline->pos++;
	    else goto error;
	    break;
	  case 11:   /* ^K */
	    cmdline->line[cmdline->pos]= 0;
	    break;
	  case 2: case IO_LEFTARROW:    /* ^B */
	    if (cmdline->pos) cmdline->pos--; else goto error;
	    break;
	  case 8:    /* ^H */
	  case 127:  /* del */
	    if (cmdline->pos) {
		cmdline->pos--;
		strcpy(&cmdline->line[cmdline->pos], &cmdline->line[cmdline->pos+1]);
	    } else goto error;
	    break;
	  case 4: case IO_DELETE:   /* ^D */
	    if (cmdline->line[cmdline->pos]) {
		strcpy(&cmdline->line[cmdline->pos], &cmdline->line[cmdline->pos+1]);
	    } else goto error;
	    break;
	  case 'C'-'@':    /* ^C */
	    cmdline->pos= 0;
	    cmdline->line[0]= 0;
	    ctrl_c_pressed= 1;
	    break;
	  case ' ':
	    if (cmdline->pos &&
		cmdline->line[0]=='!' &&
		!cmdline->line[cmdline->pos] &&
		!strchr(cmdline->line, ' ')) {
		Int i;
		for (i= (cmdline->current-1) % MAXHISTORY;
		     cmdline->history[i] && i != cmdline->current;
		     i = (i + MAXHISTORY - 1) % MAXHISTORY) {
		    if (!strncmp(cmdline->history[i], &cmdline->line[1], cmdline->pos-1)) {
			cmdline->cmd_new = i;
			goto set_line;
		    }
		}
	    }
	    /* fall through to character */
	  default:
	  normal_char:
	    if (c < 32 || c > 127) goto error; 
	    for(newc= c, l= &cmdline->line[cmdline->pos]; (oldc= newc); l++) {
		newc= *l; *l= (char) oldc;
	    }
	    *l= 0;
	    cmdline->pos++;
	    break;
	  error:
	    cmd_beep(cmdline);
	    iter= cmdline->repeat;
	    break;
	}
    }
    if (did_command) {
	cmdline->repeat= 1;
	cmdline->meta= 0;
	cmdline->cursor= 0;
	cmdline->quote= 0;
    }
    cmd_update_line(cmdline, 1);
    /*printf("after: cmd:>%s<(%d)\n", cmdline->line, cmdline->pos);*/
    return ctrl_c_pressed ? -1 : 0;
}

#ifdef STAND_ALONE
#if 0
NOPRPTO void main(Int argc, char **argv) /*__ no proto */
{
    Cmdline c;
    char foo[200];
    io_make_stdin_unbuffered();
    cmdline_init_default(&c);
    strcpy(foo, "hi there");
    while (1) {
	cmdline_get_line(&c, "foo> ", foo);
	printf("\n");
	foo[0]= 0;
    }
}
#endif

NOPROTO void main(Int argc, char **argv)
{
    IOStream *keyb= io_open_stdin();
    char foo[200];

    io_make_stdin_unbuffered();
    strcpy(foo, "hi there");
    while (1) {
	cmd_get_line("foo> ", keyb, stdout, foo);
	printf("\n");
	foo[0]= 0;
    }
}
#endif
