/**
 ** iob_mac.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

#include CONFIG
#include "iob.h"
#include "queue.h"
#include "stringlb.h"
#include "util.h"
#include <console.h>
#include <Serial.h>
#include <Devices.h>
#include <Files.h>

#define check_error(exp) do_check_error(exp, #exp, __FILE__, __LINE__)

Int do_check_error(Int err, char *exp, char *file, Int line)
{
	if (err) {
		printf("%s:%d Error %d (expression %s)\n", file, line, err, exp);
	} else if (io_debug) {
		printf("%s returns %d\n", exp, err);
	}
	return err;
}


Int io_carrier_detect(IOStream *s)
{
	/* Not implemented */
	return 1;
}

void io_make_stdin_unbuffered(void)
{
	csetmode(C_RAW, stdout);
}

void io_sigint_interrupt(void)
{
	exit(0);
}

/*******************/
/* SERIAL ROUTINES */
/*******************/

short driver1, driver2;

void close_mac_serial_drivers(void)
{
	if (driver1) KillIO(driver1);
	if (driver2) KillIO(driver2);
	if (driver1) CloseDriver(driver1);
	if (driver2) {
		CloseDriver(driver2);
		printf("Closing serial drivers\n");
		msleep(1000);
	}
	driver1= 0;
	driver2= 0;
}

IOStream *io_open_serial(char *name)
{
    Int e1, e2;
    IOStream *ret= malloc(sizeof(IOStream));
    ret->fileistream= NULL;
    ret->fileostream= NULL;
    close_mac_serial_drivers();
    if (!stricmp(name, "printer")) {
    	e1= OpenDriver("\005.BOut", &ret->ostream);
    	e2= OpenDriver("\004.BIn", &ret->istream);
    }
    else if (!stricmp(name, "modem")) {
    	e1= OpenDriver("\005.AOut", &ret->ostream);
    	e2= OpenDriver("\004.AIn", &ret->istream);
    }
    else {
    	printf("Illegal port name %s (should be printer or modem)\n", name);
    }
    
    if (!ret->istream) {
    	printf("e1: %d, e2: %d\n", e1, e2);
    	exit(1);
    }
    
    driver1= ret->ostream;
    driver2= ret->istream;
    
    atexit(close_mac_serial_drivers);
    
	check_error(SerSetBuf(ret->istream, 0, 0));
    
	{
		SerShk handshake;
		memset(&handshake, 0, sizeof(handshake));
		check_error(Control(ret->ostream, 14, &handshake));
	}
	
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    queue_init(&ret->inbuf, 1024, 1);
    queue_init(&ret->outbuf, 1024, 1);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;

    if (io_debug) printf("driver # %d,%d\n", ret->istream, ret->ostream);
    return ret;
}

IOStream *io_open_stdin(void)
{
    IOStream *ret= malloc(sizeof(IOStream));
    
    ret->istream= ret->ostream= 0;
    ret->fileistream= stdin;
    ret->fileostream= stdout;
    
    cinverse(1, stdin);
    
    queue_init(&ret->inbuf, 1024, 1);
    queue_init(&ret->outbuf, 1024, 1);

    return ret;
}

Int io_mac_getbaud(long baud)
{
    switch (baud) {
      case 300:	return 380;
      case 600:   return 189;
      case 1200:  return 94;
      case 2400:  return 46;
      case 4800:  return 22;
      case 9600:  return 10;
      case 19200: return 4;
      case 38400: return 0;
      default:
	printf("Illegal baud rate %ld in getbaud\n", baud);
	exit(1);
    }
}

void io_serial_init(long baud, IOStream *s)
{
    short init= io_mac_getbaud(baud) + 0 + 3072 + 16384;
    if (io_debug) printf("SerReset(%d, %d)\n", s->istream, init);
    if (io_debug) printf("SerReset(%d, %d)\n", s->ostream, init);

    check_error(SerReset(s->ostream, init));
	check_error(SerReset(s->istream, init));

    if (io_debug) printf("serial init finished\n");
    io_discard_input(s);
}

Int io_write(IOStream *s, char *buf, long len)
{
	if (s->fileostream) {
		Int i;
		for (i= 0; i< len; i++) {
			putc(buf[i], s->fileostream);
		}
	} else if (s->ostream) {
		long to_write= len;
     	if (io_debug) printf("FSWrite(%d, %ld, [0x%lx])\n", 
			     s->ostream, to_write, (long) buf);
    	FSWrite(s->ostream, &to_write, buf);
    	if (to_write != len) {
    		printf("Wrote only %ld of %ld bytes in io_write\n",
    			   len-to_write, len);
	    	die(("Error on write in io_putchar"));
    	}
    }
    else {
    	die(("Illegal output stream in io_putchar\n"));
    }
    return len;
}

void msleep(Int msec)
{
    unsigned long first_tick_count= clock();
    unsigned long last_tick_count= first_tick_count + 1 + msec / 17L;
    if (io_debug) printf("msleep till %ld\n", last_tick_count);
    while (clock() <= last_tick_count) ;
	if (io_debug) printf("msleep done\n");
}

Int io_getchar_multiple_inputs_n(IOStream **inputs, Int *which, Int timeout, Int n)
{
    Int i, j= 0;
    Int put_cursor;
    Int c, nread;
    char buf[100];
    unsigned long first_tick= clock();
    unsigned long last_tick= first_tick + 1 + timeout/17;
    if (io_debug) printf("io_getc_mult_inp_n %d streams, dest time= %ld\n", n, last_tick);
#if 0    
    put_cursor= (n == 1 && timeout < 0 && inputs[0]->fileistream);
    if (put_cursor) {
		io_printf("%c%c 
#endif		
    for (i= 0; i<n; i++) {
		if (!QUEUE_EMPTY(&inputs[i]->inbuf)) {
	   		*which= i;
	   	 	return *(unsigned char*)queue_remove_head(&inputs[i]->inbuf);
		}
    }

    do {
		for (j= 0; j< n; j++) {
	    	nread= io_read_nowait(inputs[j], buf, 100);
	    	if (nread > 0) {
	    		if (io_debug) printf("io_getc_mult_n got %ld chars\n", nread);
	    		for (i= 0; i< nread; i++) {
		    		queue_add_tail(&inputs[j]->inbuf, &buf[i]);
		    	}
		    	if (io_debug) printf("io_getc_mult_n stuffed chars\n");
				*which= j;
	   		 	return *(unsigned char*)queue_remove_head(&inputs[j]->inbuf);
	    	}
	    }
	    if (io_debug) printf("io_getc_mult_inp_n at time %ld\n", clock());
    } while (timeout < 0 || clock() <= last_tick);
    return EOF;
}

Int io_read_nowait(IOStream *s, char *buf, long len)
{
	if (s->istream) return io_read_serial_nowait(s, buf, len);
	else if (s->fileistream) return io_read_console_nowait(s, buf, len);
	else {
		die(("Illegal IOStream in io_read_nowait\n"));
	}
}

Int io_read_console_nowait(IOStream *s, char *buf, long len)
{
	long i;
	for (i= 0; i< len; i++) {
		Int c;
		c= getc(s->fileistream);
		if (c == EOF) break;
		buf[i]= c;
	}
	if (i == 0) return -1;
	return i;
}

Int io_read_serial_nowait(IOStream *s, char *buf, long len)
{
	long to_read;
    if (io_debug) printf("entering io_read_serial_nowait\n");
	if (io_debug) printf("SerGetBuf(%d,&pending)\n", s->istream);
	SerGetBuf(s->istream, &to_read);
	if (io_debug) printf(" -> to_read = %ld\n", to_read);
	if (to_read > 0) {
		if (len > to_read) len= to_read;
		to_read= len;
	    FSRead(s->istream, &to_read, buf);
	    if (io_debug) printf("read %ld of %ld chars in read_serial_nowait\n",
	    					 to_read, len);
	    if (to_read <= 0) {
	  		return EOF;
	   	} else {
	   		return to_read;
	    }
	} else {
	    if (io_debug) printf("no char, returning EOF\n");
	    return EOF;
	}
}

