/**
 ** queue.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 **  queue.c
 **
 ** v1.0  Thu May  9 08:46:43 1991  Randy Sargent.  created
 ** v1.1  Fri May 31 22:27:25 1991  Randy Sargent.  fixed bugs
 **/

#include CONFIG

#include "queue.h"

Int queue_debug= 0;

/* COMPUTE_TAIL may return an invalid location if the queue is empty! */
#define COMPUTE_TAIL(q, tail) { tail= q->head + q->fullness - q->elem_size; \
				if (tail >= q->max_size) tail -= q->max_size; }
void queue_display_stats(Queue *q)
{
    printf("Queue %lx: fullness %ld, elems %lx\n",
	   (long)q, q->fullness, (long)q->elems); 
}

Int queue_init(Queue *q, Int max_elems, Int elem_size)
{
    q->fullness= 0;
    q->max_size= max_elems * elem_size;
    q->elem_size= elem_size;
    q->head= 0;
    q->elems= malloc(max_elems * elem_size);
    if (!q->elems) return 1;
    if (queue_debug) queue_display_stats(q);
    return 0;
}

void queue_term(Queue *q)
{
    if (queue_debug) queue_display_stats(q);
    free(q->elems);
    q->elems= 0;
}

void *queue_head(Queue *q)
{
    if (queue_debug) queue_display_stats(q);
    if (QUEUE_EMPTY(q)) return 0;
    return q->elems + q->head;
}

void *queue_tail(Queue *q)
{
    Int tail;
    if (queue_debug) queue_display_stats(q);
    if (QUEUE_EMPTY(q)) return 0;
    COMPUTE_TAIL(q, tail);
    return q->elems + tail;
}

void *queue_remove_head(Queue *q)
{
    void *ret= queue_head(q);
    if (queue_debug) queue_display_stats(q);
    if (ret) {
	q->fullness -= q->elem_size;
	q->head += q->elem_size;
	if (q->head >= q->max_size) q->head -= q->max_size;
    }
    if (queue_debug) printf("QUEUE REMOVE HEAD %lx (%lx)\n", 
        *(Int*)ret, (long)ret);
    return ret;
}

void *queue_remove_tail(Queue *q)
{
    void *ret= queue_tail(q);
    if (queue_debug) queue_display_stats(q);
    if (ret) {
	q->fullness -= q->elem_size;
    }
    if (queue_debug) printf("QUEUE REMOVE TAIL %lx (%lx)\n", 
        *(Int*)ret, (long) ret);
    return ret;
}

Int queue_add_head(Queue *q, void *elem)
{
    if (queue_debug) queue_display_stats(q);
    if (QUEUE_FULL(q)) return 1;
    q->fullness += q->elem_size;
    q->head -= q->elem_size;
    if (q->head < 0) q->head += q->max_size;
    memcpy(q->elems + q->head, elem, q->elem_size);
    if (queue_debug) printf("QUEUE ADD HEAD %lx (%lx)\n", 
        *(Int*)elem, (long) q->elems + q->head);
    return 0;
}

/* 0 if successful */
Int queue_add_tail(Queue *q, void *elem)
{
    Int tail;
    if (queue_debug) queue_display_stats(q);
    if (QUEUE_FULL(q)) return 1;
    q->fullness += q->elem_size;
    COMPUTE_TAIL(q, tail);
    memcpy(q->elems + tail, elem, q->elem_size);
    if (queue_debug) printf("QUEUE ADD TAIL %lx (%lx)\n", 
        *(Int*)elem, (long) q->elems + tail);
    return 0;
}

Int queue_length(Queue *q)
{
    return q->fullness / q->elem_size;
}

Int queue_space(Queue *q)
{
    return (q->max_size - q->fullness) / q->elem_size;
}
