/**
 ** mon.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

#include CONFIG

#include <iob.h>
#include <util.h>

#include <binrec.h>
#include <board.h>
#include <bs6811.h>
#include <cmdlineo.h>
#include <confexec.h>
#include <dl6811.h>
#include <parsargs.h>
#include <queue.h>
#include <radix.h>
#include <s19.h>
#include <scan.h>
#include <stream.h>
#include <filelib.h>

Cmdline cl;

Int test_ram_debug= 0;


#include "mon.h"

#ifdef MAC
#include <mac-args.h>
#endif

#ifndef IC_LIB_DIRECTORY
#define IC_LIB_DIRECTORY LIB_DIRECTORY "ic/"
#endif

char lib_directory_config[]=
  MAKE_CONFIGURATION(IC_LIB_DIRECTORY,
		     "Directory where libraries reside");
#define lib_directory GET_CONFIGURATION(lib_directory_config)

char port_config[]=MAKE_CONFIGURATION(DEFAULT_SERIAL_PORT, "Default serial device name (in the form /dev/*)");
#define port GET_CONFIGURATION(port_config)

Int noinit= 0;
char bootstrapfile[200];
Int config= 0;
Int single_chip= 0;

void mon_rev1_5(void) {
    dl_ignore= 1;
    board_loopback= 1;
}

void mon_ignore_input(void) {
    board_no_return_comm= 1;
    dl_ignore= 1;
}

Arg_descriptor ad[] = {
    {"port", "s", port, "Serial device name (in the form /dev/*)"},
    {"noinit", "b", &noinit, "Do not download bootstrap communications to board"},
    {"bootstrap", "s", bootstrapfile, "Filename (.s19 format) to use instead of standard bootstrap"},
    {"libdir", "s", lib_directory, "Library directory"},
    {"config", "b", &config, "Reconfigure executable (executable must be in current directory)"},
    {"bs_ignore", "b", &dl_ignore, "Ignore serial in during bootstrap download (rev 1.5)"},
    {"loopback", "b", &board_loopback, "Expect board to echo characters (rev 1.5)"},
    {"ignore_input", "p", (void*) mon_ignore_input, "Don't pay attention to returning serial"},
    {"1.5", "p", (void*) mon_rev1_5, "Set -bs_ignore, -loopback"},
    {"board_debug", "b", &board_debug, "Turn on board module debugging"},
    {"io_debug", "b", &io_debug, "Turn on io module debugging"},
    {"single_chip", "b", &single_chip, "Turn on single chip mode"},
    LAST_ARG_DESCRIPTOR
};

IOStream *serial_iostream, *keyboard_iostream;

void mon_help(char **argv)
{
	printf("Arguments are:\n");
	printf("-port printer    or    -port modem   Set which serial port to use\n");
	printf("\n");
	printf("Possibly useful for boards other than the 6.270 board or rug warrior:\n");
	printf("-bootstrap filename.s19      Use filename.s19 as the bootstrap\n");
	printf("-config_reg value            Set config register to value (in hex).\n");
	printf("-bs_ignore                   Ignore serial responses in bootstrap sequence\n");
	printf("-loopback                    Expect hardware serial echoes from board\n");
	printf("-1.5                         Set -bs_ignore, -loopback, -eeprom, config_reg 0xff\n");
	printf("-ignore_input                Completely ignore returning serial\n");
	printf("-noinit                      Don't download bootstrap first\n");
	printf("\n");
	printf("Example: -port modem \n");
}

void download_bootstrap(void)
{
    Binrec *rec;
    if (bootstrapfile[0]) {
	FILE *bsfile= fopen_here_or_there_with_format
	  (lib_directory, bootstrapfile, "r", "Using bootstrap file %s\n");
	if (!bsfile) exit(1);
	rec= read_contiguous_s19_file(bsfile);
    }
    else {
	rec= read_contiguous_s19_string(bootstrapstring);
    }
    if (!rec) {
	die(("Error in parsing .s19 bootstrap\n"));
    }
    
    if (single_chip) {
	long i;
	for (i= 0; i< rec->len; i++) {
	    if (rec->data[i] == 0xf5) {
		rec->data[i]= 0xd5;
		break;
	    }
	}
    }
    
    dl_6811_user_friendly(serial_iostream, rec->data, rec->len, keyboard_iostream);
    io_serial_init(9600L, serial_iostream);
}

char logfile[1000];

#ifdef MAC
void main()
#else
void main(Int argc, char **argv)
#endif
{ 

    UNUSED(argc);
    
    #ifdef MAC
    mac_get_args("mon", "-port modem", mon_help);
    #endif

    if (*parse_args(ad, argv+1)) {
	usage(ad, argv, "");
	exit(1);
    }

    keyboard_iostream= io_open_stdin();
    io_make_stdin_unbuffered();
    printf("mon  6811 monitor  Version 2.861  %s   by Randy Sargent\n", __DATE__);
    printf("\n");

    if (config) {
	configure_executable(argv[0], keyboard_iostream, stdout);
	exit(0);
    }

    serial_iostream= io_open_serial(port);

    if (!serial_iostream) {
       printf("Couldn't open serial port %s\n", port);
       io_sigint_interrupt();
       exit(0);
    }
    
    
    board_set_streams(serial_iostream, keyboard_iostream);

    if (!noinit) {
	download_bootstrap();
    }

    do_monitor(serial_iostream, keyboard_iostream);
}

void help(char *help_type)
{
    if (help_type && !strnicmp(help_type, "config", 6)) {
	help_config();
    }
    else help_generic();
}

void help_generic(void)
{
    printf("mon help:\n");
    printf("help                          show this list\n");
    printf("help config                   help for the config register\n");
    printf("read <address>                reads address\n");
    printf("write <address> <data>        writes address\n");
    printf("write_eeprom <address> <data> programs eeprom address\n");
    printf("erase_eeprom <address>        erases eeprom byte to FF\n");
    printf("config <data>                 sets config register to <data>\n");
    printf("erase_config                  erases the config register\n");
    printf("testmem                       Tests the memory\n");
    printf("test21                        Tests the Rev 2.1 board\n");
    printf("testpb                        Tests the programmable brick\n");
    printf("testlcd                       Tests LCD display (6.270 board)\n");
    printf("analogs                       Print analog values\n");
    printf("pb_analogs                    Print programmable brick analogs\n");
    printf("download                      Download .s19 file\n");
    printf("bs                            Download bootstrap again\n");
    printf("logfile <filename>            Set logfile for jsr interaction\n");
    printf("upload <filename> <start-addr> <end-addr> Upload file\n");
    printf("Note that address and data are in hex.  (If you'd rather use decimal,\n");
    printf("you can type 0d######, or binary is 0b#######).\n");
}

void help_config(void)
{
    printf("THE CONFIG REGISTER:\n");
    printf("\n");
    printf("The config register tells the 6811 things like whether to have its ROM\n");
    printf("and EEPROM enabled.  If it is configured incorrectly, your 6811 won't\n");
    printf("really get programmed when you download pcode to it.\n");
    printf("\n");
    printf("How do you tell if your config is messed up?  Look at location 103f:\n");
    printf("\n");
    printf("mon> read 103f\n");
    printf("103F: 0C\n");
    printf("\n");
    printf("103f is where the config register lives.  0C is the right value when\n");
    printf("you are using IC.  If it's not 0C, you'll need to change it.  You\n");
    printf("can't just change it by using \"write\" or \"write_eeprom\" -- you have to\n");
    printf("use the special \"config\" command:\n");
    printf("\n");
    printf("mon> config 0c\n");
    printf("\n");
    printf("(Note that read 103f will -not- return the new value you stuck in; the\n");
    printf("old value stays until you press reset).\n");
    printf("\n");
    printf("Now power-cycle your board and try re-downloading.\n");
}
IOStream *brd_stream;

void do_monitor(IOStream *board, IOStream *keyboard)
{
    Int i;
    char buf[2000];
    Int nargs;
    long arg_i[20];
    char *arg_s[20];
    char *cmd;
    Scan s;

    brd_stream= board;
    cmdline_init(&cl, keyboard, stdout);
    while (1) {
	buf[0]= 0;
	cmdline_get_line(&cl, "mon> ", buf);
	printf("\n");
	scan_init(&s, buf, strlen(buf));
	cmd= scan_get_string(&s, " ");
	if (!cmd) continue;
	for (nargs= 0; nargs < 20; nargs++) {
	    arg_s[nargs]= scan_get_string(&s, " ,");
	    if (!arg_s[nargs]) break;
	    arg_i[nargs]= radix_string_to_integer(arg_s[nargs], 'x');
	}

	if (!stricmp(cmd, "quit") ||
	    !stricmp(cmd, "exit")) {
	    exit(0);
	}

	else if (!stricmp(cmd, "help")) {
	    help(arg_s[0]);
	}

	else if (!stricmp(cmd, "upload")) {
	    upload(arg_s[0], arg_i[1], arg_i[2]);
	}
	
	else if (!stricmp(cmd, "download")) {
	    download_s19(arg_s[0], 0);
	}

	else if (!stricmp(cmd, "logfile")) {
	    strcpy(logfile, arg_s[0]);
	}

	else if (!stricmp(cmd, "bs")) {
	    download_bootstrap();
	}

	else if (!stricmp(cmd, "jsr")) {
	    jsr(arg_i[0]);
	}

	else if (!stricmp(cmd, "version")) {
	    Int major, minor;
	    board_get_version(&major, &minor);
	    printf("Version %ld.%ld\n", major, minor);
	}

	else if (!stricmp(cmd, "read")) {
	    long low, high;
	    if (nargs != 1 && nargs != 2) {
		help(NULL);
	    }
	    else {
		long i;
		low= high= arg_i[0];
	    
		if (nargs == 2) high= arg_i[1];
		
		printf("%04lX:", (Int) low);
		for (i= low; i<= high; i++) {
		    printf(" %02lX", board_read_mem(i));
		}
		printf("\n");
	    }
	}

	else if (!stricmp(cmd, "write")) {
	    if (nargs != 2) {
		help(NULL);
	    }
	    else {
		printf("%04lX: %02lX written\n",
		       (Int) arg_i[0], (Int) arg_i[1]);
		board_write_ram(arg_i[0], arg_i[1]);
	    }
	}

	else if (!stricmp(cmd, "write_eeprom")) {
	    if (nargs != 2) {
		help(NULL);
	    }
	    else {
		printf("%04lX: %02lX written into eeprom\n",
		       (Int) arg_i[0], (Int) arg_i[1]);
		board_erase_eeprom(arg_i[0]);
		board_write_eeprom(arg_i[0], arg_i[1]);
	    }
	}

	else if (!stricmp(cmd, "erase_eeprom")) {
	    if (nargs != 1) {
		help(NULL);
	    }
	    else {
		printf("%04lX: %02lX written into eeprom\n",
		       (Int) arg_i[0], (Int) arg_i[1]);
		board_erase_eeprom(arg_i[0]);
	    }
	}

	else if (!stricmp(cmd, "erase_config")) {
	    if (nargs != 0) {
		help(NULL);
	    }
	    else {
		board_erase_config();
	    }
	}

	else if (!stricmp(cmd, "config")) {
	    if (nargs != 1) {
		help(NULL);
	    }
	    board_erase_config();
	    board_write_eeprom(0x103f, arg_i[0]);
	}

	else if (!stricmp(cmd, "analogs")) {
	    print_analogs();
	}
	else if (!stricmp(cmd, "pb_analogs")) {
	    print_pb_analogs();
	}
	else if (!stricmp(cmd, "pb_all_analogs")) {
	    print_pb_all_analogs();
	}
	else if (!stricmp(cmd, "test21")) {
	    test21();
	}

	else if (!stricmp(cmd, "testpb")) {
	    testpb();
	}

	else if (!stricmp(cmd, "testmem")) {
	    test_memory();
	}

	else if (!stricmp(cmd, "testlcd")) {
	    test_lcd();
	}
	
	else help(NULL);

	for (i= 0; i< nargs; i++) {
	    free(arg_s[i]);
	    arg_s[i]= 0;
	}
	free(cmd);
	cmd= 0;
    }
}

void jsr(long pc)
{
    Int j= 0;
    FILE *logfile_stream= NULL;
    if (logfile[0]) {
	logfile_stream= fopen(logfile, "w");
	if (!logfile_stream) {
	    fprintf(stderr, "Couldn't open %s for output\n", logfile);
	} else {
	    fprintf(stderr, "Logging to %s\n", logfile);
	}
    }
    printf("JSR to 0x%4lx\n", pc);
    board_jsr_new(pc, pc);
    while (1) {
	Int i;
	/* flush after channel idle for 100ms */
	i= io_getchar(brd_stream, 100);
	j++;
	if ((j % 100) == 0) fflush(stdout);
	if (i < 0) {
	    fflush(stdout);
	    j= 0;
	    i= io_getchar(brd_stream, -1);
	}
	if (i != '>') {
	    if (logfile_stream) putc(i, logfile_stream);
	    putchar(i);
	}
	else {
	    if (logfile_stream) {
		putc('\n', logfile_stream);
		fclose(logfile_stream);
	    }
	    return;
	}
    }
}

void start_analog(void)
{
    /* turn on AD power up in OPTION reg */
    board_write_ram(0x1039, board_read_mem(0x1039) | 0x80);
}
    
Int analog(Int n)
{
    /* read gnd several times */
    board_write_ram(0x1030, 0x1d);
    /* read our sensor once */
    board_write_ram(0x1030, n);
    return board_read_mem(0x1031);
}

void print_analogs(void)
{
    Int i;
    start_analog();
    for (i= 0; i< 8; i++) {
	printf("%3ld ", analog(i));
    }
    printf("\n");
}

void print_pb_analogs(void)
{
    start_analog();
    board_write_ram(0x6800, 0x0d);
    printf("%3ld ", analog(0));
    board_write_ram(0x6800, 0x2d);
    printf("%3ld ", analog(0));
    board_write_ram(0x6800, 0x29);
    printf("%3ld ", analog(0));
    board_write_ram(0x6800, 0x09);
    printf("%3ld ", analog(0));
    
    board_write_ram(0x6800, 0x0c);
    printf(" knob: %3ld ", analog(0));
    board_write_ram(0x6800, 0x2c);
    printf(" Buttons top: %3ld ", analog(0));
    board_write_ram(0x6800, 0x08);
    printf("bottom: %3ld", analog(0));
    printf("\n");

    board_write_ram(0x6800, 0x29);
    printf("%3ld ", analog(4));
    board_write_ram(0x6800, 0x0d);
    printf("%3ld ", analog(4));
    printf("\n");
}

void print_pb_all_analogs(void)
{
    board_write_ram(0x6800, 0x08);
    print_analogs();
    board_write_ram(0x6800, 0x09);
    print_analogs();
    board_write_ram(0x6800, 0x0C);
    print_analogs();
    board_write_ram(0x6800, 0x0D);
    print_analogs();
    board_write_ram(0x6800, 0x28);
    print_analogs();
    board_write_ram(0x6800, 0x29);
    print_analogs();
    board_write_ram(0x6800, 0x2C);
    print_analogs();
    board_write_ram(0x6800, 0x2D);
    print_analogs();
}

enum {
    test_begin,
    test_end,
    test_ok,
    test_not_valid
  };

void start_lcd(void)
{
    board_write_ram(0x103c, 0xd5);  /* single chip mode */
    board_write_ram(0x1004, 0x0);
    board_write_ram(0x1007, 0xff); /* write */
    board_write_ram(0x1003, 0x38); /* command! */
    board_write_ram(0x1000, 0x10); /* enable high */
    board_write_ram(0x1000, 0x0);  /* enable low */
    board_write_ram(0x1003, 0x0f); /* command! */
    board_write_ram(0x1000, 0x10);
    board_write_ram(0x1000, 0x0);
    board_write_ram(0x1003, 0x1);  /* command! */
    board_write_ram(0x1000, 0x10);
    board_write_ram(0x1000, 0x0);
}
void stop_lcd(void)
{
    board_write_ram(0x103c, 0xf5); /* expanded mode */
}

void lcd_putchar(Int c)
{
    board_write_ram(0x1004, 0x2);   /* set mode line */
    board_write_ram(0x1007, 0xff);  /* write mode! */
    board_write_ram(0x1003, c);     /* data */
    board_write_ram(0x1000, 0x10);  /* enable high */
    board_write_ram(0x1000, 0x0);   /* enable low */
}

void lcd_puts(char *msg)
{
    while (*msg) lcd_putchar(*msg++);
}

void test_lcd(void)
{
    start_lcd();
    lcd_puts("ABCDEFGabcdefghijklmnopqrstuvwxyzABCDEFGabcdefghijklmnopqrstuvwxyzABCDEFGabcdefghijklmnopqrstuvwxyz");
    stop_lcd();
}
				       


void 	    test21_motors     (void);
void 	    test	      (Int (*func) (Int i));
Int 	    test21_motors_exec (Int i);


void test21(void)
{
    test_memory();
    test21_motors();
    printf("All tests done\n");
}

void testpb(void)
{
    test_pb_memory();
}


void test21_motors(void)
{
    printf("Testing motors\n");
    test(test21_motors_exec);
}

unsigned char *test_ram_image= NULL;
unsigned char *test_ram_used= NULL;

void test_init_ram(void)
{
    if (!test_ram_image) test_ram_image= malloc(32768);
    if (!test_ram_used) test_ram_used= malloc(32768);
    if (!test_ram_image || !test_ram_used) {
	printf("Cannot allocate memory for memory test\n");
	exit(1);
    }
    memset(test_ram_used, 0, 32768);
}

void test_init_ram_pb(void)
{
    if (test_ram_image) free(test_ram_image);
    test_ram_image= malloc(32768*4);
    if (test_ram_used) free(test_ram_used);
    test_ram_used= malloc(32768*4);

    if (!test_ram_image || !test_ram_used) {
	printf("Cannot allocate memory for memory test\n");
	exit(1);
    }
    
    memset(test_ram_used, 0, 32768*4);
}

int test_ram_verbose= 1;

void test_write_ram(long addr, Int data)
{
    if (addr < 32768 || addr > 65535) {
	printf("illegal address in write ram\n");
	exit(1);
    }
    test_ram_used[addr-32768]= 1;
    test_ram_image[addr-32768]= data;
    if (test_ram_verbose) {
       printf("wrote %04lx:%02lx\n", addr, data);
    }
    board_write_ram(addr, data);
}

void test_write_ram_pb(long addr, Int data)
{
    if (addr < 0 || addr >= 32768 + 32768*4) {
	printf("illegal address in write ram\n");
	exit(1);
    }
    test_ram_used[addr]= 1;
    test_ram_image[addr]= data;
    board_write_ram_pb(addr, data);
    if (test_ram_debug) {
	printf("Write %ld:%04lX, %02lX\n",
	       addr>>15, (addr & 0x7fff) + 0x8000,
	       data);
    }
}

void board_write_ram_pb(long addr, Int data)
{
    board_bank_select_pb(addr);
    board_write_ram((addr & 0x7fff) + 0x8000, data);
}

Int board_read_mem_pb(long addr)
{
    board_bank_select_pb(addr);
    return board_read_mem((addr & 0x7fff) + 0x8000);
}


void board_bank_select_pb(long addr)
{
    static bank= -1;
    Int newbank= (addr >> 15);
    if (newbank != bank) {
	bank= newbank;
	board_write_ram(0x5000,
			((bank & 1) ? 0x02 : 0) |
			((bank & 2) ? 0x20 : 0));
    }
}

void test_read_ram_pb(void)
{
    long i;
    for (i= 0; i< 131072; i++) {
	if (test_ram_used[i]) {
	    Int a;
	    
	    a= board_read_mem_pb(i);
	    if (a != test_ram_image[i]) {
		printf("Error: Address %ld:%04lX, wrote %02X, read %02lX\n",
		       i>>15, (i & 0x7fff) + 0x8000,
		       test_ram_image[i], a);
	    }
	    else {
		if (test_ram_debug) {
		    printf("Read %ld:%04lX, %02lX\n",
			   i>>15, (i & 0x7fff) + 0x8000,
			   a);
		} else {
		    printf(".");
		    fflush(stdout);
		}
	    }
	}
    }
}

void test_read_ram(void)
{
    long i;
    for (i= 0; i< 32768; i++) {
	if (test_ram_used[i]) {
	    Int a;
	    a= board_read_mem(i+32768);
	    if (a != test_ram_image[i]) {
		printf("Error: Address %04lX, wrote %02X, read %02lX\n",
		       i+32768,
		       test_ram_image[i], a);
	    }
	    else { printf("."); fflush(stdout); }
	}
    }
}

void test_pb_memory(void)
{
    Int a,i;

    
    test_init_ram_pb();

    test_ram_debug= 1;

    /* test address bits */
    printf("Testing address bits\n");
    for (a= 1; a < 131072; a <<= 1) {
	test_write_ram_pb(a, rand() & 255);
    }
    test_write_ram_pb(0, rand() & 255);
    test_read_ram_pb();

    test_ram_debug= 0;
    
    printf("Address bits tested\n");

    printf("Testing data bits\n");
    test_init_ram_pb();

    /* test data bits */
    for (a= 1; a < 256; a <<= 1) {
	test_write_ram_pb(rand() & 131071, a);
    }
    test_write_ram_pb(rand() & 131071, 0);

    test_read_ram_pb();
    printf("Data bits tested\n");
    test_init_ram_pb();

    printf("Testing random accesses\n");

    for (i= 0; i< 400; i++) {
	test_write_ram_pb(rand() & 131071, rand() & 255);
    }
    test_read_ram_pb();
    printf("Random accesses tested\n");
}


void test_memory(void)
{
    Int a,i;
    
    test_init_ram();
    /* test address bits */
    printf("Testing address bits\n");
    for (a= 1; a < 32768; a <<= 1) {
	test_write_ram(32768 + a, rand() & 255);
    }
    test_write_ram(32768, rand() & 255);
    test_read_ram();
    printf("Address bits tested\n");

    printf("Testing data bits\n");
    test_init_ram();

    /* test data bits */
    for (a= 1; a < 256; a <<= 1) {
	test_write_ram(32768 + (rand() & 32767), a);
    }
    test_write_ram(32768 + (rand() & 32767), 0);

    test_read_ram();
    printf("Data bits tested\n");
    test_init_ram();

    printf("Testing random accesses\n");

    for (i= 0; i< 200; i++) {
	test_write_ram(32768 + (rand() & 32767), rand() & 255);
    }
    test_read_ram();
    printf("Random accesses tested\n");
}


void test(Int (*func) (Int i))
{
    Int i= 1;
    Int delta= 1;
    char buf[100];

    buf[0]= 0;
    
    while (1) {
	switch ((*func)(i)) {
	  case test_begin:
	    delta= 1;
	  case test_not_valid:
	    i += delta;
	    break;
	  case test_ok:
	    buf[0]= 0;
	    cmdline_get_line(&cl,
			     "Return to continue, 'back' to go back  ", buf);
	    printf("\n");
	    if (!stricmp(buf, "back")) delta= -1; else delta= 1;
	    if (!stricmp(buf, "quit")) return;
	    i += delta;
	    break;
	  case test_end:
	    buf[0]= 0;
	    cmdline_get_line(&cl,
			     "No more tests.  "
			     "Return to continue, 'back' to go back  ", buf);
	    printf("\n");
	    if (stricmp(buf, "back")) return;
	    delta= -1;
	    break;
	  default:
	    printf("error in internal function test\n");
	    break;
	}
    }
}

Int test21_motors_exec(Int i)
{
    switch(i) {
      case 0:
	return test_begin;
      case 5:
	printf("all green lights\n");
	board_write_ram(0x7000, 0xfa);
	break;
      case 10:
	printf("all red lights\n");
	board_write_ram(0x7000, 0xf5);
	break;
      case 20:
	printf("no lights\n");
	board_write_ram(0x7000, 0x05);
	break;
      case 30:
	printf("green 0 (top)\n");
	board_write_ram(0x7000, 0x10);
	break;
      case 40:
	printf("red 0 (top)\n");
	board_write_ram(0x7000, 0x11);
    	break;
      case 50:
	printf("green 1 (top)\n");
	board_write_ram(0x7000, 0x22);
    	break;
      case 60:
	printf("red 1 (top)\n");
	board_write_ram(0x7000, 0x20);
    	break;
      case 70:
	printf("green 2 (top)\n");
	board_write_ram(0x7000, 0x40);
    	break;
      case 80:
	printf("red 2 (top)\n");
	board_write_ram(0x7000, 0x44);
    	break;
      case 90:
	printf("green 3 (bottom)\n");
	board_write_ram(0x7000, 0x88);
    	break;
      case 100:
	printf("red 3 (bottom)\n");
	board_write_ram(0x7000, 0x80);
    	break;
      case 110:
	printf("exp board bottom red\n");
	board_write_ram(0x7000, 0x05);
	board_write_ram(0x4000, 0x01);
	break;
      case 120:
	printf("exp board top red\n");
	board_write_ram(0x4000, 0x02);
	break;
      case 130:
	printf("exp board left red led\n");
	board_write_ram(0x4000, 0xec);
	break;
      case 140:
	printf("exp board left green led\n");
	board_write_ram(0x4000, 0xdc);
	break;
      case 150:
	printf("exp board right red led\n");
	board_write_ram(0x4000, 0xf8);
	break;
      case 160:
	printf("exp board right green led\n");
	board_write_ram(0x4000, 0xf4);
	break;
      case 170:
	return test_end;
      default:
	return test_not_valid;
    }
    return test_ok;
}

void download_s19(char *filename, Int dl_eeprom)
{
    FILE *stream;
    Stream st;

    stream= fopen(filename, "r");
    if (!stream) {
	fprintf(stderr, "Couldn't open file %s for input\n", filename);
	return;
    }
	
    stream_init_from_file(&st, stream);
    
    printf("\nDownloading %s\n",filename);

    while (1) {
	Binrec *rec;
	rec= read_s19_record_from_stream(&st);
	if (!rec) break;
	if (dl_eeprom) {
	    long i;
	    for (i= 0; i< rec->len; i++) {
		board_erase_eeprom(rec->addr + i);
		board_write_eeprom(rec->addr + i, rec->data[i]);
	    }
	}
	else {
	    board_download_block(rec->addr, rec->len, rec->data);
	}
	printf(".");
	fflush(stdout);
	free(rec);
    }
    printf("Done\n");
}

void upload(char *filename, Int from, Int to)
{
    Int i;
    FILE *f= fopen(filename, "w");
    if (!f) {
	fprintf(stderr, "cannot open %s for output\n", filename);
    }
    for (i= from; i<= to; i++) {
	if (!((i-from) % 32)) {
	    fprintf(f, "\n");
	    fprintf(f, "%04lX: ", i);
	    printf("."); fflush(stdout);
	}
	fprintf(f, "%02lX", board_read_mem(i));
    }
    fprintf(f, "\n");
    fclose(f);
}

    
    
