// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;
import java.io.*;
import java.util.*;
import javax.xml.*;
import javax.xml.parsers.*;
import javax.xml.transform.dom.*;
import javax.xml.validation.*;
import javax.xml.xpath.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.*;
import org.xml.sax.*;

/**
 * The <tt>GameWriter</tt> writes ChessBoard objects out to XML files.
 * @specfield board : Board     // The Board as of when the game was saved
 * @specfield whiteTimer   : Timer // White's Timer as of when the game was saved
 * @specfield blackTimer   : Timer // Black's Timer as of when the game was saved
 **/
public class GameWriter
{
	private ChessBoard board;
	private GameTimer whiteTimer;
	private GameTimer blackTimer;
	private Document document;

	/**
	 * @effects Construct a new GameWriter with the given Board, and no timers
	 * @requires board != null
	 */
	public GameWriter(ChessBoard board)
		throws IOException {
		this(board, null, null);
	}

	/**
	 * Construct a new GameWriter with the given Board, and the given timers
	 * 	 * @requires board != null && ((whiteTimer == null && blackTimer == null)
	 *	|| (whiteTimer != null && blackTimer != null))
	 */
	public GameWriter(ChessBoard board, GameTimer whiteTimer, GameTimer blackTimer)
		throws IOException {
		this.board = board;
		this.whiteTimer = whiteTimer;
		this.blackTimer = blackTimer;
		writeDocument();
	}
	
	/**
	 * @effects Write the given board and timer state out to an XML file on disk
	 * @throws IOException if an IO error occurs writing the file
	 */
	public void writeGame(File file)
		throws IOException {
		writeDocument(new StreamResult(file));
	}

	/**
	 * @returns a String containing the XML representation of the
	 * game
	 */
	public String writeGame()
		throws IOException {
		StringWriter writer = new StringWriter();
		writeDocument(new StreamResult(writer));
		return writer.toString();
	}

	/**
	 * @effect Write the given game into a DOM tree and transform it
	 * through the given Result
	 */
	private void writeDocument(Result out)
		throws IOException {
		try {
			TransformerFactory tFactory = TransformerFactory.newInstance();
			Transformer transformer = tFactory.newTransformer();
			DOMSource source = new DOMSource(document);
			transformer.transform(source, out);
		} catch(Exception e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * @effect Populate this.document with an DOM XML document
	 * encoding the given game
	 */
	private void writeDocument()
		throws IOException {
		try {
			document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
		} catch(javax.xml.parsers.ParserConfigurationException e) {
			throw new IOException("Can't create XML DOM: " + e.getMessage());
		}
		Element root = document.createElement("game");
		document.appendChild(root);
		root.setAttribute("ruleset", "6170-spring-2007");
		Element time = document.createElement("time");
		root.appendChild(time);
		if(whiteTimer != null) {
			time.setAttribute("timed", "true");
			time.setAttribute("initWhite", Long.toString(whiteTimer.getStartTime()));
			time.setAttribute("initBlack", Long.toString(blackTimer.getStartTime()));
			time.setAttribute("currentWhite", Long.toString(whiteTimer.getTime()));
			time.setAttribute("currentBlack", Long.toString(blackTimer.getTime()));
		} else {
			time.setAttribute("timed", "false");
		}

		root.appendChild(writeMoveHistory());
		root.appendChild(writePieces());
		Element gameOver = writeGameOver();
		if(gameOver != null)
			root.appendChild(gameOver);
	}

	/**
	 * @return a &lt;moveHistory&gt; XML Element representing the game's
	 * move history
	 */
	private Element writeMoveHistory() {
		Element moveHistory = document.createElement("moveHistory");
		MoveHistory<ChessMove> history = board.getMoveHistory();
		while(history != null) {
			ChessMove m = history.getLastMove();
			Element move = document.createElement("move");
			if(moveHistory.getFirstChild() != null)
				moveHistory.insertBefore(move, moveHistory.getFirstChild());
			else
				moveHistory.appendChild(move);
			move.setAttribute("side",
							  m.getPlayer().toString().toLowerCase());
			move.setAttribute("value", moveToString(m));
			if(!history.getLastMoveTime().equals(""))
				move.setAttribute("time", history.getLastMoveTime());
			history = history.removeLastMove();
		}
		return moveHistory;
	}

	/**
	 * @return a String representation of the given ChessMove
	 */
	private String moveToString(ChessMove m) {
		return posToString(m.getPiece().getRow(),
						   m.getPiece().getColumn())
			+ "-" + posToString(m.getRow(),
								m.getColumn());
	}

	/**
	 * @return a string in chess notation for the given (row, col)
	 * coordinates.
	 */
	private String posToString(int row, int col) {
		return "abcdefgh".substring(col,col+1) + "12345678".substring(row, row+1);
	}

	/**
	 * @return a &lt;pieces&gt; XML Element representing the game's
	 * current state
	 */
	private Element writePieces() {
		Element pieces = document.createElement("pieces");
		for(Piece piece : board.getPieces()) {
			Element square = document.createElement("square");
			pieces.appendChild(square);
			square.setAttribute("side", piece.getPlayer().toString().toLowerCase());
			if(piece.getRow() < 0 || piece.getColumn() < 0)
				System.err.println(piece);
			square.setAttribute("id", posToString(piece.getRow(),
												  piece.getColumn()));
			square.setAttribute("piece", piece.getType().getName().toLowerCase());
		}
		return pieces;
	}

	/**
	 * @return a &lt;gameOver&gt; XML Element representing how the
	 * game ended, or <tt>null</tt> if the game isn't over.
	 */
	private Element writeGameOver() {
		if(!board.isGameOver())
			return null;
		Element gameOver = document.createElement("gameOver");
		gameOver.setAttribute("winner", board.getWinner().toString().toLowerCase());
		gameOver.setAttribute("description", gameOverDescription());
		return gameOver;
	}

	/**
	 * @return a textual description of how the game ended
	 */
	private String gameOverDescription() {
		switch(board.getGameOverReason()) {
		case Board.OUTOFTIME:
			return "timeExpired";
		case ChessBoard.STALEMATE:
			return "stalemate";
		case ChessBoard.CHECKMATE:
			return "checkmate";
		case AntichessBoard.OUTOFPIECES:
			return "piecesLost";
		default:
			return "unknown";
		}
	}
}
