#include <stdio.h>
#include <ctype.h>
#include "S.h"
#include "options.h"
#include "y.tab.h"
#include "graphics.h"
#include "device.h"
#include "eval.h"

#define CTM gr_state
#define RASTERS(x,y,rx,ry) (rx = x*am(37)+am(36),\
			    ry = y*am(39)+am(38))
#define WORLD_R(rx,ry,x,y) (x = ((float)rx-am(36))/am(37),\
			    y = ((float)ry-am(38))/am(39))

int in_graphics = 0;	/* for interrupts while executing graphics code */

static vector *marks(), *text(), *lines(), *segments(), *polygon();
static vector *dev_menu(), *hook(), *start_pars(), *delta_pars();
static vector *del1_par(), *do_set(), *alc_pic_fun(), *do_gr_dist();
static void set_last();
static void do_gr_disp(), gr_reset(), get_transform(), set_transform();
static void set_clip(), set_rclip(), get_clip(), get_box(), set_box();
static void check_pic(), append_pic(), to_device(), to_world();
static float *xpand(), pt_dist(), line_dist();
static long max();
static int pic_fun_code();

static gr_par pars[] = {
	"transformation",6,FALSE,150,6,0,
	"clip",7,FALSE,156,4,6,
	"box",8,FALSE,160,4,17,
	"rotation",3,TRUE, 14, 1,10,
	"char.rotation",4,TRUE, 48, 1,11,
	"color",0,TRUE, 10, 1,12,
	"line.type",0,TRUE, 8, 1,13,
	"line.width",0,TRUE, 9, 1,14,
	"char.size",1,TRUE, 18, 1,15,
	"font",0,TRUE,79,1,16,
	NULL,0,0,0,0,0,0 };
#define PARSIZE 20

/*
 * the graphics state is kept in two ways: a state that is set in displaying
 * graphics objects and reset afterwards -- this is the Postscript model,
 * and also a state that is that of the last (in a time sense)
 * graphics object -- this is the sense used in old-S graphics code & must
 * be available to maintain the semantics of old-S functions
 * The arrays gr_state and gr_last respectively hold these two states
 */
static float gr_last[PARSIZE]; static int gr_last_set = FALSE;
static float gr_state[PARSIZE];
static float gr_deflt[PARSIZE] = {
	1,0,0,1,0,0,	0,0,1,1,	0,0,	0,
	0,	1,	1,	1,	1,	1,	0
};

vector *S_gr_things(ent, arglist)
vector *ent, *arglist;
{
	vector *value;
	int which = sys_index;
	if( cur_device==NULL_DEVICE || !cur_device->active){
		if(which!=DEV_wrap)
			Recover("No active graphics device",ent);
		return(S_void);
	}
	if(UNSCALED)F77_SUB(zscalz) ();
	if(which<0 || which>=NPRIMITIVES)
		Recover("Invalid index for graphics  primitive",ent);
	switch(which ) {
	case DEV_marks: value = marks(ent,arglist); break;
	case DEV_text: value = text(ent,arglist); break;
	case DEV_lines: value = lines(ent,arglist); break;
	case DEV_segments: value = segments(ent,arglist); break;
	case DEV_polygon: value = polygon(ent,arglist); break;
	case DEV_hook: value = hook(ent,arglist); break;
	case DEV_menu: value = dev_menu(ent,arglist); break;
	default: (*(cur_device->routines[which])) (); value = S_void;
	}
	return(value);
}

vector *S_gr_disp(ent, arglist)
vector *ent, *arglist;
{
	long n; vector **args, *arg;
	if(cur_device==NULL_DEVICE || !cur_device->active) {
		Recover("No active graphics device",ent);
		return(S_void);
	}
	delta_pars(); /* make sure graphics state is up to date */
	n = arglist->length; args = arglist->value.tree;
	while(n--) {
		arg = *args++;
		do_gr_disp(arg);
	}
	return(S_void);
}

static void do_gr_disp(picture)
vector *picture;
{
	vector **elements, *el, *par;
	long n; int code; float *old_state;
	static vector *arglist = NULL;
if(check) {
	sanity(picture,"picture object to be displayed");
}
	if(VOID(picture)||picture->length==0)return;
	if(picture->mode!=GRAPHICS)
		Recover("Cannot display non-graphics data",picture);
	old_state = (float *)S_alloc((long)PARSIZE,sizeof(float)); /* gsave */
	MEMCPY(old_state, gr_state, PARSIZE);
	elements = picture->value.tree; n = picture->length;
	while(n--){
		el = *elements++;
		switch(el->mode){
		case FUN_CALL: /* either a graphics primitive or evaluate it */
			if(arglist==NULL){
				long prev = set_alloc(PERM_FRAME);
				arglist = New_vector();
				set_alloc(prev);
			}
			*arglist = *el; /* create an arglist; all the args are data already */
			arglist->value.tree++; arglist->length--;
			par = *(el->value.tree);
			if(par->mode != NAME)
				Recover("Invalid mode for picture function name",el);
			code = pic_fun_code(par->value.name);
			if(code<0)eval(el);
			else switch(code ) {
			case DEV_marks: marks(el,arglist); break;
			case DEV_text: text(el,arglist); break;
			case DEV_lines: lines(el,arglist); break;
			case DEV_segments: segments(el,arglist); break;
			case DEV_polygon: polygon(el,arglist); break;
			case DEV_hook: hook(el,arglist); break;
			case DEV_menu: Recover("Menu interaction meaningless from graphics structure",NULL_ENTRY);
			default: (*(cur_device->routines[code])) ();
			}
			break;
		case GRAPHICS: /*sub_picture */
			do_gr_disp(el);
			break;
		default: /* must be a parameter setting */
			do_set(el,FALSE);
		}
	}
	gr_reset(old_state);	/* grestore */
}

static void set_last()
{
	MEMCPY(gr_last, gr_state, PARSIZE);
	gr_last_set = TRUE;
}

static int pic_fun_code(name)
char *name;
{
	if(*name++ != '.')
		return(-1);
	switch(*name++) { /* which device function ? */
	case 'P':
		if(name_eq(name,"olygon"))return(DEV_polygon); break;
	case 'L':
		if(name_eq(name,"ines"))return(DEV_lines); break;
	case 'S':
		if(name_eq(name,"egments"))return(DEV_segments);
		else if(name_eq(name,"ignalled"))return(DEV_signalled);
		break;
	case 'T':
		if(name_eq(name,"ext"))return(DEV_text); break;
	case 'M':
		if(name_eq(name,"arks"))return(DEV_marks);
		else if(name_eq(name,"enu"))return(DEV_menu);
		break;
	case 'C': /* which should this be: eject or clear??*/
		if(name_eq(name,"lear"))return(DEV_clear); break;
	case 'E':
		if(name_eq(name,"ject"))return(DEV_clear); break;
	case 'I':
		if(name_eq(name,"nput"))return(DEV_input);
		else if(name_eq(name,"nitial"))return(DEV_initial);
		break;
	case 'W':
		if(name_eq(name,"rap"))return(DEV_wrap); break;
	case 'H':
		if(name_eq(name,"ook"))return(DEV_hook); break;
	default:
		return(-1); break;
	}
#ifdef lint
	return(-1);
#endif
}

vector *S_gr_set(ent, arglist)
vector *ent, *arglist;
{
	vector **args = arglist->value.tree, *arg, **values, *res;
	long narg = arglist->length;
	if(cur_device==NULL_DEVICE || !cur_device->active) {
		Recover("No active graphics device", ent);
		return(S_void);
	}
	if(sys_index==3) {
		MEMCPY(gr_state, gr_deflt, PARSIZE);
		MEMCPY(gr_last, gr_deflt, PARSIZE);
		gr_last_set = TRUE;
		return(S_void);
	}
	delta_pars(); /* make sure graphics state is up to date */
	if(narg==1)  res = do_set(args[0],sys_index);
	else if(narg==0 && sys_index==1) { /* dump all par's */
		gr_par *par = pars; res = alcvec(LIST,0L);
		while(par->name) {
			arg = alcvec(REAL, (long)par->length);
			MEMCPY(arg->value.Float, gr_state+par->position, par->length);
			arg->name = par->name;
			append_el(res,(long)NOARG,arg);
			par++;
		}
	}
	else {
		res = alcvec(LIST, narg);
		values = res->value.tree;
		while(narg--)
			*values++ = do_set(*args++, sys_index);
	}
	return(res);
}

static vector *do_set(arg, query)
vector *arg; int query;
{
	vector *value,**values, **children;
	long  lcopy, nres, ll;
	float *from, *to;
	char *pname, *name = arg->name;
	gr_par *par;

	if(query && IS_NULL_STRING(name)) name=string_value(arg);
	if(!(query || atomic_type(arg->mode))) {
		switch(arg->mode) {
		case LIST:
		case GRAPHICS:
			nres = arg->length;
			value = alcvec(GRAPHICS,nres);
			children = arg->value.tree;
			values = value->value.tree;
			while(nres--) *values++ = do_set(*children++,query);
			return(value);
		default: if(IS_NULL_STRING(name)) {
				Warning("Un-named graphics parameter ignored",arg);
				return(S_void);
			}
			arg = coevec(arg,REAL,TRUE,PRECIOUS(arg));
		}
	}
	/* following 2 lines could be replaced by hashing */
	par = pars;
	while((pname=par->name) != NULL && !name_eq(pname,name)){par++;}
	if(pname==NULL)return(S_void);
	if((!query) && arg->mode != MISSING){
		arg = coevec(arg,REAL,TRUE,CHECK_IT);
		lcopy = par->length<arg->length ? par->length : arg->length;
	}
	else lcopy = par->length;
	ll=lcopy;
	value = alcvec(REAL,lcopy); /* copy out old values */
	from = gr_state+par->position; to = value->value.Float;
	while(ll--) *to++ = *from++;
	value->name = name;
	/* either stuff in the new values or take a special action */
	if(!query) {
		ll = lcopy; to = gr_state+par->position; from = arg->value.Float;
		while(ll--) *to++ = *from++; /* copy in new values */
		switch(par->action) { /*do what is needed for old-S code*/
		case 0:
			ll=lcopy;
			to = amodes(par->am_start); from = arg->value.Float;
			while(ll--) *to++ = *from++;
			break;
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
			F77_SUB(zgrspz) (&(par->action), arg->value.Float);
			break;
		case 6: set_transform(arg->value.Float); break;
		case 7: set_clip(arg->value.Float); break;
		case 8: set_box(arg->value.Float); break;
		}
	}
	return(value);
}

static vector *delta_pars()
{
	vector *value, *p; int n;
	value = alcvec(LIST,0L);
	for(n=0; (pars+n)->name; n++)
		if(p=del1_par(n, (float *)NULL))append_el(value,(long)NOARG,p);
	switch((int)value->length) {
	case 0: return(NULL_ENTRY);
	case 1: return(value->value.tree[0]);
	default: return(value);
	}
}

/* reset the parameters to default */
static void gr_reset(new_pars)
float *new_pars;
{
	gr_par *par; vector *value, *p; int n;
	if(new_pars == (float *)0)
		new_pars = gr_deflt;
	value = alcvec(LIST, 0L);
	for(n=0; (par=(pars+n))->name; n++)
		if(p=del1_par(n, new_pars+par->position))
			append_el(value,(long)NOARG,p);
	switch((int)value->length) {
	case 0: value = NULL_ENTRY; break;
	case 1: value = value->value.tree[0]; break;
	}
	if(value)do_set(value,FALSE);
}

static vector *start_pars()
{
	float *from = gr_deflt, *to = gr_state;
	long n = PARSIZE;
	while(n--) *to++ = *from++;
	return(delta_pars());	
}

static vector *del1_par(i, new)
int i; float *new;
{
	gr_par *par = pars + i;
	long n;
	float *p, *p0, *old, *to,  pp[6];
	vector *value;

	if(new) p0 = new;
	else switch(i) {
	case 0: p0 = pp; get_transform(pp); break;
	case 1: p0 = pp; get_clip(pp); break;
	case 2: p0 = pp; get_box(pp); break;
	default: p0 = amodes(par->am_start);
	}
	n = par->length; p = p0; old = gr_state + par->position;
	while(n--)if(*p++ != *old++)break;
	if(++n) { /* a change */
		value = alcvec(REAL,(long)par->length); to = value->value.Float;
		n = par->length; p = p0; old = gr_state + par->position;
		while(n--)*to++ = *old++ = *p++;
		value->name = par->name;
		return(value);
	}
	else return(NULL_ENTRY);
}


static void get_transform(z)
float z[];
{
	z[0] = 72*am(28)*am(37);	z[1] = 0;
	z[2] = 0;			z[3] = 72*am(29)*am(39);
	z[4] = 72*am(28)*am(36);	z[5] = 72*am(29)*am(38);
}

static void set_transform(z)
float *z;
{
	/* needs to be generalized to Postscript model of transform matrix? */
	am(37) = z[0]/(72*am(28)); am(39) = z[3]/(72*am(29));
	am(36) = z[4]/(72*am(28)); am(38) = z[5]/(72*am(29));
	set_rclip();
}

static void set_clip(z)
float *z;
{
	am(61) = *z++; am(63) = *z++;
	am(62) = *z++; am(64) = *z;
	am(65) = 0.;
	set_rclip();
}

static void set_rclip()
{
	am(32) = am(61)*am(37)+am(36);
	am(33) = am(62)*am(37)+am(36);
	am(34) = am(63)*am(39)+am(38);
	am(35) = am(64)*am(39)+am(38);
	am(76) = am(28)*(am(33)-am(32))/(am(62)-am(61)); /* user c's in in's*/
	am(77) = am(29)*(am(35)-am(34))/(am(64)-am(63));
}

static void get_clip(z)
float z[];
{
	float fx, fy;

	if(!am(65)) {
		*z++ = am(61); *z++ = am(63);
		*z++ = am(62); *z   = am(64);
	} else {
		fx = (am(62)-am(61))/(am(41)-am(40));
		fy = (am(64)-am(63))/(am(43)-am(42));
		*z++ = am(61) - fx*am(40);
		*z++ = am(63) - fy*am(42);
		*z++ = am(61) + fx*(1-am(40));
		*z   = am(63) + fy*(1-am(42));
	}
}

static void get_box(z)
float z[];
{
	z[0] = z[1] = 0;
	z[2] = am(98)*72;
	z[3] = am(99)*72;
}

static void set_box(z)
float z[];
{
	UNUSED(z);
	/* currently does nothing */
}

static char obdsmsg[60];

/* UNUSED ?
void out_of_bounds(what, x, y)
char *what; float x, y;
{
	char *encs2();
	sprintf(obdsmsg,"(%g, %g)",x,y);
	Warning(encs2("%s out of bounds at %s",what,obdsmsg),NULL_ENTRY);
}
*/

#define Coerce(x,m)	if(x->mode!=(m))x=coevec(LANGUAGE_TYPE(x->mode)?eval(x):x,(m),FALSE,CHECK_IT)

static vector *marks(ent, arglist)
vector *ent, *arglist;
{
	vector *x, *y, *m, **args;
	float *xx, *yy;
	long *mm, nx, ny, nm, ix, iy, im, n, one = 1, cur_mark;

	if(arglist->length != 3) {
		Recover("invalid low-level mark structure", ent);
		return(S_void);
	}
	args = arglist->value.tree; x = args[0]; y = args[1]; m = args[2]; 
	Coerce(x, REAL); xx = x->value.Float; ix = nx = x->length;
	Coerce(y, REAL); yy = y->value.Float; iy = ny = y->length;
	Coerce(m, INT); mm = m->value.Long; im = nm = m->length;
	cur_mark = (long)am(15);
	if(nm==1 && nx == ny) { /* the fast &common case */
		am(15) = (float)mm[0];
		(*(cur_device->routines[DEV_marks])) (xx,yy,&nx);
		am(15) = (float)cur_mark;
		return(S_void);
	}
	n = nx > ny ? nx : ny;
	while(n--) {
		am(15) = (float)mm[0];
		(*(cur_device->routines[DEV_marks])) (xx,yy,&one);
		xx++; yy++; mm++;
		if(--ix <= 0) {ix = nx; xx -= nx;}
		if(--iy <= 0) {iy = ny; yy -= ny;}
		if(--im <= 0) {im = nm; mm -= nm;}
	}
	am(15) = (float)cur_mark;
	return(S_void);
}

static vector *lines(ent, arglist)
vector *ent, *arglist;
{
	vector *x, *y, **args;
	float *xx, *yy;
	long nx, ny, n;

	if(arglist->length != 2) {
		Recover("invalid low-level lines structure", ent);
		return(S_void);
	}
	args = arglist->value.tree; x = args[0]; y = args[1];
	Coerce(x, REAL); xx = x->value.Float; nx = x->length;
	Coerce(y, REAL); yy = y->value.Float; ny = y->length;
	n = nx > ny ? nx : ny;
	if(nx < n) xx = xpand(xx, nx, n);
	if(ny < n) yy = xpand(yy, ny, n);
	(*(cur_device->routines[DEV_lines])) (xx,yy,&n);
	return(S_void);
}

static vector *polygon(ent, arglist)
vector *ent, *arglist;
{
	vector *x, *y, **args;
	float *xx, *yy;
	long nx, ny, n;

	if(arglist->length != 2) {
		Recover("invalid low-level polygon structure", ent);
		return(S_void);
	}
	args = arglist->value.tree; x = args[0]; y = args[1];
	Coerce(x, REAL); xx = x->value.Float; nx = x->length;
	Coerce(y, REAL); yy = y->value.Float; ny = y->length;
	n = nx > ny ? nx : ny;
	if(nx < n) xx = xpand(xx, nx, n);
	if(ny < n) yy = xpand(yy, ny, n);
	(*(cur_device->routines[DEV_polygon])) (xx,yy,&n);
	return(S_void);
}

static vector *text(ent, arglist)
vector *ent, *arglist;
{
	vector *x, *y, *s, *p, **args;
	float *xx, *yy, pos;
	long nx, ny, ns, ix, iy, is, n, slen;
	char **ss, *string;

	if(arglist->length != 4) {
		Recover("invalid low-level text structure", ent);
		return(S_void);
	}
	args = arglist->value.tree; x = args[0]; y = args[1]; s = args[2]; p = args[3];
	Coerce(x, REAL); xx = x->value.Float; ix = nx = x->length;
	Coerce(y, REAL); yy = y->value.Float; iy = ny = y->length;
	Coerce(s, CHAR); ss = s->value.Char; is = ns = s->length;
	Coerce(p, REAL); pos = p->value.Float[0];
	n = nx > ny ? nx : ny;
	while(n--) {
		string = *ss;
		while(isspace(*string))
			string++;
		slen = strlen(string);
#ifdef F_CHARSTRUCT
		{F_CHARSTRUCT tmp;F_CHARP(&tmp)=string;F_CHARLEN(&tmp)=slen;
		(*(cur_device->routines[DEV_text])) (xx,yy,&tmp,&slen,&pos);}
#else
		(*(cur_device->routines[DEV_text])) (xx,yy,string,&slen,&pos);
#endif
		xx++; yy++; ss++;
		if(--ix <= 0) {ix = nx; xx -= nx;}
		if(--iy <= 0) {iy = ny; yy -= ny;}
		if(--is <= 0) {is = ns; ss -= ns;}
	}
	return(S_void);
}

static vector *segments(ent, arglist)
vector *ent, *arglist;
{
	vector *x, *y, **args;
	float *x1, *y1, *x2, *y2;
	long nx1, ny1, nx2, ny2, n;

	if(arglist->length != 4) {
		Recover("invalid low-level segments structure", ent);
		return(S_void);
	}
	args = arglist->value.tree;
	x = args[0]; Coerce(x, REAL); x1 = x->value.Float; nx1 = x->length;
	y = args[1]; Coerce(y, REAL); y1 = y->value.Float; ny1 = y->length;
	x = args[2]; Coerce(x, REAL); x2 = x->value.Float; nx2 = x->length;
	y = args[3]; Coerce(y, REAL); y2 = y->value.Float; ny2 = y->length;
	n = nx1 > nx2 ? nx1 : nx2; if(ny1 > n) n = ny1; if(ny2 > n) n = ny2;
	if(nx1 < n) x1 = xpand(x1, nx1, n);
	if(ny1 < n) y1 = xpand(y1, ny1, n);
	if(nx2 < n) x2 = xpand(x2, nx2, n);
	if(ny2 < n) y2 = xpand(y2, ny2, n);
	(*(cur_device->routines[DEV_segments])) (x1,y1,x2,y2,&n);
	return(S_void);
}

static vector *hook(ent, arglist)
vector *ent, *arglist;
{
	vector *x, **args = arglist->value.tree, *type = args[0];
	long n, m, t;
	float *x1, *x2;
	UNUSED(ent);

	Coerce(type, INT); t = type->value.Long[0];
	x = args[1]; Coerce(x, REAL);
	n = x->length; x1 = x->value.Float;
	if(arglist->length <= 2) {
		x = blt_in_NULL;
		m = 0; x2 = (float *)0;
	} else {
		x = args[2]; Coerce(x, REAL);
		m = x->length; x2 = x->value.Float;
	}
	(*(cur_device->routines[DEV_hook])) (&t, x1, &n, x2, &m);
	return(x);
}

static vector *dev_menu(ent, arglist)
vector *ent, *arglist;
{
	vector *s, **args, *choice;
	char **ss;
	long ns;

	if(arglist->length != 1) {
		Recover("invalid low-level menu structure", ent);
		return(S_void);
	}
	args = arglist->value.tree; s = args[0]; 
	Coerce(s, CHAR); ss = s->value.Char; ns = s->length;
	if(cur_device->routines[DEV_menu]!=NULL)
		choice = (*(cur_device->routines[DEV_menu])) (ss,&ns);
	else{
		choice = alcvec(INT, 1L);
		*(choice->value.Long) = menu(ss, ns);
	}
	return(choice);
}

static float *xpand(x, old, new)
float *x;
long old, new;
{
	long n = old;
	float *xp, *newx, *newxp;

	newx = (float *)S_alloc(new, sizeof(float));
	xp = x; newxp = newx;
	while(new--) {
		*newx++ = *xp++;
		if(--n <= 0) {
			n = old;
			xp = x;
		}
	}
	return(newxp);
}

static vector *cur_pic = NULL;	/* the current picture object */
static long cur_pic_frame = 0;	/* where the picture is */

#define DEV_CHECK(w) if(cur_device==NULL_DEVICE || !cur_device->active || !cur_device->routines[w]) Recover("Inactive device or unimplemented device primitive",NULL_ENTRY)
#define invoke(w) (*(cur_device->routines[w]))

/*sys_index==0:  return the current picture object & clear it */
/*sys_index==1:  set the frame */
vector *S_cur_pic(ent,arglist)
vector *ent, *arglist;
{
	vector *value;
	UNUSED(ent);

	switch(sys_index) {
	case 1: /* .Begin.pic() */
		if(gr_show)return(S_void);
		cur_pic_frame = parent_frame[cur_frame]; cur_pic = NULL;
		if(gr_last_set)gr_reset(gr_last);
		return(S_void);
	case 0: /* .Cur.pic(picture) */
		if(gr_show)return(arglist->value.tree[0]);
		if(!cur_pic_frame)Recover(".Begin.pic() should have been called earlier",NULL_ENTRY);
		delta_pars(); /* make sure graphics state is up to date */
		set_last(); /* save the last graphics state */
		value = cur_pic;
		cur_pic_frame = 0; cur_pic = NULL;
		return(value ? value : S_void);
	}
#ifdef lint
	return(0);
#endif
}

static long prev_frame = 0;

static void check_pic()
{
	vector *del, **args; long n;
	if(prev_frame)Recover("Internal problem in frame assignment for picture",NULL_ENTRY);
	prev_frame = set_alloc(cur_pic_frame);	
	if(!cur_pic){
		cur_pic = alcvec(GRAPHICS,0L);
		del = start_pars(); /* diff against deflt's */
	}
	else del = delta_pars();
	if(del == 0)
		return;
	/* append the par changes */
	if(del->mode == LIST) {
		n = del->length;
		args = del->value.tree;
		while(n--)
			append_el(cur_pic, (long)NOARG, *args++);
	} else
		append_el(cur_pic, (long)NOARG, del);
}

static void append_pic(el)
vector *el;
{
	long n; vector **args;
	/* append new piece to picture object */
	if(el && (n=el->length)>0)
		if(el->mode == LIST) {
			args = el->value.tree;
			while(n--)
				append_el(cur_pic, (long)NOARG, *args++);
		} else
			append_el(cur_pic, (long)NOARG, el);
	set_alloc(prev_frame);
	prev_frame = 0;
}

static vector *alc_pic_fun(what,narg,x,y,n)
char *what; long narg, n; float *x, *y;
{
	vector *call, **args, *xy; long nn; float *to;
	call = alcvec(FUN_CALL,narg+1); args = call->value.tree;
	args[0] = alc_name(what);
	if(narg>0){
		args[1] = xy = alcvec(REAL,n);
		for(nn=n, to = xy->value.Float; nn; nn--,to++, x++)*to = *x;
	}
	if(narg>1) {
		args[2] = xy = alcvec(REAL,n);
		for(nn=n, to = xy->value.Float; nn; nn--,to++, y++)*to = *y;
	}
	return(call);
}

F77_SUB(ztextz)(x,y,text,n,pos)
float *x,*y,*pos; F_CHARTYPE text; long *n;
{
	vector *value, **args, *arg;
	if(gr_show) {
		(*(cur_device->routines[DEV_text])) (x,y,text,n,pos);
		return;
	}
	check_pic();
	value = alc_pic_fun(".Text",4L,x,y,1L);
	args = value->value.tree;
	args[3] = arg = alcvec(CHAR,1L);
	arg->value.Char[0] = c_s_cpy(F_CHARP(text));
	args[4] = arg = alcvec(REAL,1L); arg->value.Float[0] = *pos;
	append_pic(value);
}
	
F77_SUB(zlinsz)(x,y,n)
float *x,*y; long *n;
{
	if(gr_show) {
		(*(cur_device->routines[DEV_lines])) (x,y,n);
		return;
	}
	check_pic();
	append_pic(alc_pic_fun(".Lines",2L,x,y,*n));
}
	
F77_SUB(zpntsz)(x,y,n)
float *x,*y; long *n;
{
	vector *value, *mark;
	if(gr_show) {
		(*(cur_device->routines[DEV_marks])) (x,y,n);
		return;
	}
	check_pic();
	value = alc_pic_fun(".Marks",3L,x,y,*n);
	mark = alcvec(INT,1L); mark->value.Long[0] = (long)am(15);
	value->value.tree[3] = mark;
	append_pic(value);
}
	
F77_SUB(zpolyz)(x,y,n)
float *x,*y; long *n;
{
	if(gr_show) {
		(*(cur_device->routines[DEV_polygon])) (x,y,n);
		return;
	}
	check_pic();
	append_pic(alc_pic_fun(".Polygon",2L,x,y,*n));
}
	
F77_SUB(zsegsz)(x,y,x2,y2,n)
float *x,*x2,*y,*y2; long *n;
{
	vector *value, **args, *xy; long nn; float *to;
	if(gr_show) {
		(*(cur_device->routines[DEV_segments])) (x,y,x2,y2,n);
		return;
	}
	check_pic();
	value = alc_pic_fun(".Segments",4L,x,y,*n);
	args = value->value.tree;
	args[3] = xy = alcvec(REAL,*n);
	for(nn = *n, to = xy->value.Float; nn; nn--,to++, x2++)*to = *x2;
	args[4] = xy = alcvec(REAL,*n);
	for(nn = *n, to = xy->value.Float; nn; nn--,to++, y2++)*to = *y2;
	append_pic(value);
}
	
F77_SUB(zejecz)()
{
	vector *value;
	if(gr_show) {
		(*(cur_device->routines[DEV_clear])) ();
		return;
	}
	check_pic();
	value = alcvec(FUN_CALL,1L);
	value->value.tree[0] = alc_name(".Eject");
	append_pic(value);
}
	
F77_SUB(zwrapz)()
{
	if(gr_show)
		(*(cur_device->routines[DEV_wrap])) ();
	else cur_pic = NULL;
}
		
F77_SUB(zinitz)()
{
	if(cur_device==NULL_DEVICE)
		Recover("Device not active",NULL_ENTRY);
}
	
F77_SUB(zhookz)(type, x, n, y, m)
float *x, *y;
long *n, *m, *type;
{
	vector *value, *arg;

	if(gr_show) {
		(*(cur_device->routines[DEV_hook]))(type, x, n, y, m);
		return;
	}
	check_pic();
	value = alc_pic_fun(".Hook", 1L, x, y, *n);
	arg = alcvec(INT, 1L);
	arg->value.Long[0] = *type;
	append_el(value, 1L, arg);
	append_pic(value);
}


F77_SUB(zrdpnz)(x,y,n,nmax)
float *x,*y; long *n,*nmax;
{
	DEV_CHECK(DEV_input); in_graphics = 1;
	invoke(DEV_input) (x,y,n,nmax); in_graphics = 0;
}

F77_SUB(zquxyz)(x,y,ind)
float *x,*y; long *ind;
{
	DEV_CHECK(DEV_query); in_graphics = 1;
	invoke(DEV_query) (x,y,ind); in_graphics = 0;
}
	
	
F77_SUB(zparmz)(x,n)
float *x; long *n;
{
	UNUSED(x); UNUSED(n);
	if(cur_device==NULL_DEVICE)
		Recover("Device not active",NULL_ENTRY);
}
	
F77_SUB(zlengz)(text,inches)
F_CHARTYPE text; float *inches;
{
	DEV_CHECK(DEV_length); in_graphics = 1;
	invoke(DEV_length) (text,inches); in_graphics = 0;
}
	
F77_SUB(zlinez)(ix,iy)
long *ix,*iy;
{
	DEV_CHECK(DEV_line); in_graphics = 1;
	invoke(DEV_line) (ix,iy); in_graphics = 0;
}
	
F77_SUB(zptchz)(ich,crot)
F_CHARTYPE ich; float *crot;
{
	DEV_CHECK(DEV_ptchar); in_graphics = 1;
	invoke(DEV_ptchar) (ich,crot); in_graphics = 0;
}
	
F77_SUB(zseekz)(ix,iy)
long *ix,*iy;
{
	DEV_CHECK(DEV_seek); in_graphics = 1;
	invoke(DEV_seek) (ix,iy); in_graphics = 0;
}
	
F77_SUB(zflshz)()
{
	DEV_CHECK(DEV_flush); in_graphics = 1;
	invoke(DEV_flush) (); in_graphics = 0;
}
	
F77_SUB(zintrz)()
{
	gr_signalled();
}

gr_signalled()
{
	if(!in_graphics)return;
	DEV_CHECK(DEV_signalled); in_graphics = 0;
		/* do not set in_graphics, since interrupt while doing
		signalled code should not re-execute the signal handler */
	invoke(DEV_signalled) ();
	gr_reset((float *)NULL); /* reset the graphics state */
	cur_pic = NULL; /* throw away the current picture object */
	prev_frame = 0; /* reset for check_pic() */
}


void gr_transform(x,y,nnx,nny,which)
float *x, *y; long *nnx, *nny, *which;
{
	long nx = *nnx, ny = *nny, n; double det;
	n = max(nx,ny);
	if(n <= 0)return;
	switch((int)*which) {
	case 0: 
		while(n--){
			to_device(x, y);
			if(!nx--){ nx = *nnx; x -= nx;}
			if(!ny--){ ny = *nny; y -= ny;}
			x++; y++;
		}
		break;
	case 1: 
		det = CTM[0]*CTM[3]-CTM[1]*CTM[2];
		if(det==0.)Recover("Invalid (i.e., singular) transformation matrix",NULL_ENTRY);
		while(n--){
			to_world(det,x,y);
			if(!nx--){ nx = *nnx; x -= nx;}
			if(!ny--){ ny = *nny; y -= ny;}
			x++; y++;
		}
		break;
	}
}

static void to_device(x,y)
float *x, *y;
{
	float X= *x, Y = *y;
/* this ought to work but doesn't -- no one ever seems to set CTM (?)
	*x = X*CTM[0]+Y*CTM[2]+CTM[4];
	*y = X*CTM[1]+Y*CTM[3]+CTM[5];
*/
	*x = X*am(37) + am(36);
	*y = Y*am(39) + am(38);
}

static void to_world(det,x,y)
double det; float *x, *y;
{
/*
	float X = *x-CTM[4], Y = *y-CTM[5];
	*x = (X*CTM[3]-Y*CTM[2])/det;
	*y = (Y*CTM[0]-X*CTM[1])/det;
*/
	float X = *x, Y = *y; UNUSED(det);
	*x = (X - am(36))/am(37);
	*y = (Y - am(38))/am(39);
}

#define DEVICE(x,y,rx,ry) (rx = x*CTM[0]+y*CTM[2]+CTM[4],ry = x*CTM[1]+y*CTM[3]+CTM[5])

/* distance to a simple graphics object */
vector *
gr_distance(ent, arglist)
vector *ent, *arglist;
{
	vector *value, *object, *el, **els, **args;
	int found, code, error = 0, precious;
	long n;
	float *xy;
	UNUSED(ent);

	object = arglist->value.tree[0]; precious = PRECIOUS(object);
	if(object->mode!=GRAPHICS)Recover("Argument should be a graphics object",NULL_ENTRY);
	el = arglist->value.tree[1]; /* the position */
	if(PRECIOUS(el))el = copy_data(el,NULL_ENTRY);
	el = coevec(el,REAL,TRUE,FALSE);
	xy = el->value.Float;
	els = object->value.tree; n = object->length; found = FALSE;
	value = alcvec(LIST,0L);
	while(n--){
		el = *els++;
		switch(el->mode) {
		case GRAPHICS: error=1; break;
		case FUN_CALL:
			args = el->value.tree;
			if((args[0])->mode != NAME)
				Recover("Bad function in graphics object",el);
			code = pic_fun_code((args[0])->value.name);
			if(!code){error=2;break;}/* non-basic function */
			if(precious)el = copy_data(el,NULL_ENTRY);
			args = el->value.tree+1;
			if(!do_gr_dist(code,args,xy,value))
				found--; /* inocuous, e.g. .Eject */
			if(found++)error=3;
			break;
		default: break; /* presumably parameters */
		}
		if(error) {
			switch(error) {
			case 3: Warning("Extra graphics information ignored",el);					break;
			case 1: Warning("Subpictures ignored",NULL_ENTRY);
				break;
			case 2: Warning("Non-graphics function ignored",NULL_ENTRY);
				break;
			}
			error=0;
		}
	}
	return(value);
}

static vector *do_gr_dist(code,args,xy,value)
int code; vector **args, *value; float *xy;
{
	float *x1, *y1, d, dd, xm1, ym1;
	long n, nx1, ny1, nxx1, nyy1, im[4], zero = 0;
	vector *xyfound = NULL, *dist, *arg, *which = NULL;

	to_device(xy,xy+1);
	switch(code) {
	case DEV_marks:
		dist = alcvec(REAL,1L); dist->name = "distance";
		arg = args[0]; x1 = arg->value.Float; nx1 = arg->length;
		arg = args[1]; y1 = arg->value.Float; ny1 = arg->length;
		gr_transform(x1,y1,&nx1,&ny1,&zero);
		n = max(nx1,ny1); d = SINGLE_XMAX; nxx1 = nx1; nyy1 = ny1;
#ifdef lint
		xm1 = ym1 = SINGLE_XMAX;
#endif
		while(n--){
			if( (dd = pt_dist(*x1,*y1,xy)) < d){
				xm1 = *x1; ym1 = *y1; d = dd;
				im[0] = nx1-nxx1+1; im[1] = ny1-nyy1+1;
			}
			if(!nxx1--){nxx1 = nx1; x1 -= nx1;}
			if(!nyy1--){nyy1 = ny1; y1 -= ny1;}
			x1++; y1++;
		}
		if(d<SINGLE_XMAX) {
			xyfound = alcvec(REAL,2L); xyfound->name = "xy";
			dist->value.Float[0] = d;
			xyfound->value.Float[0] = xm1;
			xyfound->value.Float[1] = ym1;
			if(im[0] == im[1]) {
				which = alcvec(INT, 1L);
				which->value.Long[0] = im[0];
			}
			else {
				which = alcvec(INT, 1L);
				MEMCPY(which->value.Long, im, 2);
			}
			which->name = "which";
		}
		else na_set(dist->value.Float);
		break;
	case DEV_lines:
	case DEV_segments:
	case DEV_polygon:
	case DEV_text:
		Recover("Not implemented yet",NULL_ENTRY);
	default:
		return(NULL);
	}
	append_el(value,(long)NOARG,xyfound);
	if(dist)append_el(value,(long)NOARG,dist);
	if(which)append_el(value,(long)NOARG,which);
	return(value);
}

static float pt_dist(x,y,xy)
double x, y;
float *xy;
{
	double tt,ss;
	tt = x-xy[0];
	ss = tt*tt;
	tt = y-xy[1];
	return(ss + tt * tt);
}

static float line_dist(x,y,xy)
double x, y;
float *xy;
{
	UNUSED(x); UNUSED(y); UNUSED(xy);
	return(SINGLE_XMAX);
}

static long
max(a,b)
long a,b;
{
	return a>b?a:b;
}

#define N		0
#define H		1
#define L		2
#define S		3
#define BLACK		-1
#define WHITE		-2
#define GRAY		-3
#define CMISSING	-1234567
#define NULLSTR		((char *)0)
#define NULLCHAR	('\0')

struct Hue {
	char hname[10];
	double hvalue;
};

static struct Hue hue_names[] = {
	"blue",		0,
	"magenta",	60,
	"red",		120,
	"orange",	150,
	"yellow",	180,
	"green",	240,
	"cyan",		300,
	"black",	BLACK,
	"white",	WHITE,
	"gray",		GRAY,
	"",		CMISSING
};

static int sign, l_default, s_default, betw();
static char *getrange(), *gettok(), *getname(), *getint();
static double from[4], to[4], by[4], range[4], increm[4];

int color_spec(), color_fill();

/*
 * Parse the string `spec' to produce a set of color definitions
 * in (h,l,s) space.  Syntax of specification is one of:
 *	nrange
 *	nrange,hrange
 *	nrange,hrange,lrange
 *	nrange,hrange,lrange,srange
 * where each range has the generic form
 *	from:to:by
 * where each of from, to and by are nonnegative integers.  Default
 * to is same as from and default by is 1.  By is ignored except in
 * nrange.  The argument tabsize says how big the color table is; any
 * color numbers not in the range [1,tabsize] are ignored.
 *
 * nrange: Range of color numbers to define.
 *	   If no nrange is given, color 1 is defined.  If no `to' is given,
 *	   only color `from' is defined.  Each number implied by nrange
 *	   is used as an index into the input hue, lightness and satura-
 *	   tion vectors.
 * hrange: Range of hues as angles in degrees.  Names can be given (see
 *	   definition of hue_names[]).  If no hrange is given, 0:360 is
 *	   is used.  If from==to then from:(from+360) is used.
 * lrange: Range of lightnesses.  If no lrange is given, 50 is used, unless
 *	   `gray', `white' or `black' were specified in hrange, in which case
 *	   lrange is 0:100, 100 or 0; in all three cases default saturation
 *	   is set to 0.
 * srange: Range of saturations.  If no srange is given, 100 is used, with the
 *	   exception noted under lrange.
 */

color_spec(spec, tabsize)
char **spec;
long *tabsize;
{
	char *str, *s, *t, *malloc();
	int i;

	/* make a copy of spec, delete white space and terminate with a comma */
	if(*spec == NULLSTR)
		return;
	str = malloc((unsigned)2+strlen(*spec));
	if(str == NULLSTR) {
		fprintf(stderr, "color_spec: no memory\n");
		return;
	}
	for(s = *spec, t = str; *s; s++)
		if(!isspace(*s))
			*t++ = *s;
	*t++ = ',';
	*t = NULLCHAR;

	/* parse up to 4 comma-separated ranges from str */
	s_default = 100; l_default = 50;
	s = str;
	for(i = 0; i < 4; i++) {
		s = getrange(s, &from[i], &to[i], &by[i]);
		if(s == NULLSTR) {
			fprintf(stderr, "color_spec: bad specification: %s\n", *spec);
			return;
		}
	}

	/* fill in information missing from the spec, using default rules */
	if(from[N] == CMISSING)
		from[N] = to[N] = 1;
	if(by[N] <= 0)
		by[N] = 1;
	from[N] = (int)from[N];
	to[N] = (int)to[N];
	by[N] = (int)by[N];
	if(from[N] < 1 || to[N] < 1) {
		fprintf(stderr, "color_spec: color numbers must be positive\n");
		return;
	}
	from[N] = from[N] - 1;
	to[N] = to[N] - 1;
	if(from[H] == CMISSING) {
		from[H] = 0;
		to[H] = 360;
	}
	if(from[H] == to[H] && by[H] != 0)
		to[H] += 360;
	if(from[L] == CMISSING)
		if(s_default == 0)
			if(from[N] == to[N])
				from[L] = to[L] = 50;
			else {
				from[L] = 0;
				to[L] = 100;
			}
		else
			from[L] = to[L] = l_default;
	if(from[S] == CMISSING)
		from[S] = to[S] = s_default;

	/* set up ranges and increments */
	for(i = 0; i < 4; i++)
		range[i] = to[i] - from[i];
	sign = 1;
	if(range[N] < 0) {
		sign = -1;
		range[N] = -range[N];
	}
	range[N] = (int)(range[N] / by[N]);
	increm[N] = range[N] ? range[N] : 1;
	for(i = 1; i < 4; i++)
		increm[i] = range[i] / increm[N];

	/* cleanup; return value is largest color number, 1-origin */
	free(str);
	*tabsize = 1 + to[N];
}

color_fill(hue, lightness, saturation)
double hue[], lightness[], saturation[];
{
	int which, i;

	for(i = 0; i <= range[N]; i++) {
		which = from[N] + sign*i*by[N];
		hue[which] = from[H] + i*increm[H];
		lightness[which] = from[L] + i*increm[L];
		saturation[which] = from[S] + i*increm[S];
	}
}

/*
 * Parse a range from the string p.
 * Recognizes:
 *	case 1: NULLCHAR or ,	from <- CMISSING
 *	case 2: tok1,		from <- tok1, to <- from, by <- 0
 *	case 3: tok1:tok2,	from <- tok1, to <- tok2, by <- 1
 *	case 4: tok1:tok2:tok3,	from <- tok1, to <- tok2, by <- tok3
 * The return value is a pointer to the null byte or the byte
 * after the comma, or NULLSTR for a syntax error.
 */
static char *
getrange(p, from, to, by)
char *p;
double *from, *to, *by;
{
	/* case 1 */
	*from = CMISSING;
	if(*p == NULLCHAR)
		return(p);
	if(*p == ',')
		return(++p);

	/* case 2 */
	if((p = gettok(p, from)) == NULLSTR)
		return(NULLSTR);
	if(*p == ',') {
		*to = *from;
		*by = 0;
		return(++p);
	} else if(*p++ != ':')
		return(NULLSTR);

	/* case 3 */
	if((p = gettok(p, to)) == NULLSTR)
		return(NULLSTR);
	if(*p == ',') {
		*by = 1;
		return(++p);
	} else if(*p++ != ':')
		return(NULLSTR);

	/* case 4 */
	if((p = gettok(p, by)) == NULLSTR)
		return(NULLSTR);
	if(*p == ',')
		return(++p);
	else
		return(NULLSTR);
}

static char *
gettok(p, n)
char *p;
double *n;
{
	if(isdigit(*p))
		return(getint(p, n));
	else if(isalpha(*p))
		return(getname(p, n));
	else
		return(NULLSTR);
}

static char *
getname(p, val)
char *p;
double *val;
{
	struct Hue *h;
	char buf[20], *bp;

	*val = 0;
	for(bp = buf; isalpha(*p) && bp - buf < 19; bp++, p++)
		*bp = isupper(*p) ? tolower(*p) : *p;
	*bp = NULLCHAR;
	for(h = hue_names; *h->hname; h++)
		if(strcmp(buf, h->hname) == 0)
			break;
	switch((int)h->hvalue) {
	case CMISSING:
		fprintf(stderr, "colstr: unknown color name: %s\n", buf);
		return(NULLSTR);
	case BLACK:
		l_default = 0;
		break;
	case WHITE:
		l_default = 100;
		break;
	case GRAY:
		s_default = 0;
		break;
	default:
		*val = h->hvalue;
		break;
	}
	return(p);
}

static char *
getint(p, val)
char *p;
double *val;
{
	*val = 0;
	while(isdigit(*p))
		*val = *val * 10 + *p++ - '0';
	*val = (int)(*val * 100) / 100.0;
	return(p);
}

static int
betw(low, i, high)
int low, i, high;
{
        return(low <= i && i <= high);
}

/*
 * hlsrgb(): Map a color definition from (h,l,s)
 * space into (r,g,b) space.
 * Domain is [0,360] x [0,100] x [0,100].
 * Range is [0,1] x [0,1] x [0,1].
 * Hue is reduced modulo 360; lightness and
 * saturation are truncated at 0 and 100.
 * Source: Computer Graphics, Vol 13, No 3, August 1979, page III-38
 */
hlsrgb(h, l, s, r, g, b)
double h, l, s;
float *r, *g, *b;
{
	float M, m, bound(), gun();
	double floor();

	h = h - floor(h/360) * 360;
	l = bound(0.0, l/100, 1.0);
	s = bound(0.0, s/100, 1.0);
	M = l + s*(l > 0.5 ? 1.0 - l : l);
	m = 2*l - M;
	*r = gun(m, M, h);
	*g = gun(m, M, h-120);
	*b = gun(m, M, h-240);
}

/*
 * rgbhls(): Map a color definition from (r,g,b)
 * space into (h,l,s) space.
 * Domain is [0,1] x [0,1] x [0,1].
 * Range is [0,360] x [0,100] x [0,100].
 * Source: Computer Graphics, Vol 13, No 3, August 1979, page III-37
 */
void
rgbhls(r, g, b, h, l, s)
double r, g, b;
float *h, *l, *s;
{
	float M, m, sum, diff, fmax(), fmin(), bound();
	double floor();

	r = bound(0.0, r, 1.0);
	g = bound(0.0, g, 1.0);
	b = bound(0.0, b, 1.0);
	M = fmax(r, fmax(g, b));
	m = fmin(r, fmin(g, b));
	sum = M + m;
	diff = M - m;
	*h = *s = 0;
	*l = sum / 2;
	if(diff > 0) {
		*s = diff / (*l < 0.5 ? sum : 2 - sum);
		if(M == r)
			*h = 2 + (g-b) / diff;
		else if(M == g)
			*h = 4 + (b-r) / diff;
		else
			*h = 6 + (r-g) / diff;
		*h *= 60;
	}
	*h -= floor(*h/360) * 360;
	*l *= 100;
	*s *= 100;
}

float
gun(m, M, h)
double m, M, h;
{
	if(h < 0)
		h += 360;
	if(h < 60)
		return(m + h*(M-m)/60);
	if(h < 180)
		return(M);
	if(h < 240)
		return(m + (240-h)*(M-m)/60);
	return(m);
}

float
bound(low, x, high)
double low, x, high;
{
	if(x < low)
		return(low);
	if(x > high)
		return(high);
	return(x);
}

float
fmax(a, b)
double a, b;
{
	return(a > b ? a : b);
}

float
fmin(a, b)
double a, b;
{
	return(a < b ? a : b);
}
