#ifndef MALLOC
/* dummy versions */
int
pop_brk()
{
}

int
lesscore()
{
}

#else

/* routines that update the minimum position to which a brk() can be
/* done to reclaim space after completing a user expression */
/* these are currently rather primitive -- cleverer would be to try
/* to find where the buffer for the opened files comes (portably??) */
#include <stdio.h>
#include "S.h"

perm_open(file)
FILE *file;
{
	UNUSED(file);
	min_brk = sbrk(0);
}


perm_close(file)
FILE *file;
{
	UNUSED(file);
}

char *
perm_alloc(n,size)
unsigned int n, size;
{
	char *p, *pend;
	p = calloc(n,size);
	if(p && (pend = p + n*size-1)>min_brk)min_brk = pend;
	return(p);
}

set_brk(new_brk)
char *new_brk;
{
	if(new_brk>min_brk) min_brk = new_brk;
}

char *
perm_realloc(ptr,size)
char *ptr; unsigned int size;
{
	char *p, *pend;
	p = realloc(ptr,size);
	if(p && (pend = p + size-1)>min_brk)min_brk = pend;
	return(p);
}


perm_free(ptr)
char *ptr;
{
/* if there were a chain of perm_alloc's, this could do something: */
/* for example, find the largest outstanding perm_alloc or break set */
/* from perm_open.  But there isn't currently, so it doesn't */
	free(ptr);
}


/**********************************************************************
	Memory management functions:

	char *malloc(unsigned int size);
		To request a block at least as big as 'size'.
	char *realloc(char *old, unsigned int size);
		To reset the size of 'old' to 'size'.
	void free(char *old);
		To free the 'old' block.
	void lesscore(); 
		To reset the program 'brk' point to the lowest
		possible address without destroying data.

	The following #-parameters may be redefined:
	GETCORE: a function to get more core memory. If not SEGMENTED,
		GETCORE(0) is assumed to return the next available
		address. Default is 'sbrk'.
	ERRCORE: the error code as returned by GETCORE.
		Default is ((char*)(-1)).
	CORESIZE: a desired unit (measured in bytes) to be used
		with GETCORE. Default is (1024*WORDSIZE).
	SEGMENTED: if defined, memory requests are assumed to be
		non-contiguous across calls of GETCORE's.

	This package is based on a best fit algorithm with lists of
	free elts maintained in a self-adjusting binary tree ordered
	by element size. Each list contains all elts of the same size.
	For results on self-adjusting trees, see the paper:
		Self-Adjusting Binary Trees,
		DD Sleator & RE Tarjan, JACM 1985.

	The header of a block contains the size of the data part in bytes.
	Since the size of a block is 0%4, the low two bits of the header
	are free and used as follows:

		BIT0:	1 for busy (block is in use), 0 for free.
		BIT1:	if the block is busy, this bit is 1 if the
			preceding block in contiguous memory is free.
			Otherwise, it is always 0.

	Kiem-Phong Vo, ulysses!kpv,
	AT&T Bell Laboratories, MH3C536A, 201-582-4869.
**********************************************************************/

pop_brk()
{
	extern char *min_brk;
if(check) { /* check any allocated arenas */
	if(check_arenas(min_brk)){
		fflush(stderr);
		PROBLEM "Can't pop break location" RECOVER(NULL_ENTRY);
	}
	if(sbrk(0)-min_brk>Compact) lesscore();
	min_brk = sbrk(0);
}

/* system call to get more core */
#ifndef GETCORE
#define GETCORE		sbrk
#endif
#ifndef ERRCORE
#define ERRCORE		((char *)(-1))
#endif
#ifndef CORESIZE
#define CORESIZE	(1024*WORDSIZE)
#endif

extern char	*GETCORE(), *malloc(), *realloc();

/* for conveniences */
#define reg		register
#define uint		unsigned int
#define NULL		(0L)
#define NIL(p)		((p)(NULL))
#define ROUND(x,y)	((((x)+((y)-1))/(y))*(y))

/* debugging macros */
#ifdef	DEBUG
#define	ASSERT(p)	if(!(p))abort()
#define COUNT(n)	n++
static int		nmalloc, nrealloc, nfree;
#else
#define	ASSERT(p)
#define COUNT(n)
#endif /*DEBUG*/

/* function to copy data from one area to another */
/* S version */
#define memcopy(to,fr,n)	memcpy(to,fr,n)

/* alignment so pointers can be stored and the size is 0%4 */
#define WORDSIZE	ROUND(sizeof(char*),4)

/* the proto-word */
typedef union _w_
{
	unsigned int	w_i;		/* an int */
	struct _t_	*w_p;		/* a pointer */
	char		w_a[WORDSIZE];	/* to force alignment */
} WORD;

/* structure of a node in the free tree */
typedef struct _t_
{
	WORD	t_s;	/* size of this element */
	WORD	t_n;	/* next in link list */
	WORD	t_p;	/* parent node or previous in link list */
	WORD	t_l;	/* left child */
	WORD	t_r;	/* right child */
	WORD	t_d;	/* dummy to reserve space for self-pointer */
} TREE;

/* usable # of bytes in the block */
#define SIZE(b)		(((b)->t_s).w_i)

/* free tree pointers */
#define PARENT(b)	(((b)->t_p).w_p)
#define LEFT(b)		(((b)->t_l).w_p)
#define RIGHT(b)	(((b)->t_r).w_p)

/* forward and backward links for lists in the tree */
#define LINK(b)		(((b)->t_n).w_p)
#define BACK(b)		(((b)->t_p).w_p)

/* set/test indicator if a block is in the tree or in a list */
#define SETNOTREE(b)	(LEFT(b) = (TREE*)(-1))
#define ISNOTREE(b)	(LEFT(b) == (TREE*)(-1))

/* functions to get information on a block */
#define DATA(b)		(((char*) (b)) + WORDSIZE)
#define BLOCK(d)	((TREE*) ((d) - WORDSIZE))
#define SELFP(b)	((TREE**) (((char*) (b)) + SIZE(b)))
#define LAST(b)		(*((TREE**) (((char*) (b)) - WORDSIZE)))
#define NEXT(b)		((TREE*) (((char*) (b)) + SIZE(b) + WORDSIZE))
#define BOTTOM(b)	((DATA(b)+SIZE(b)+WORDSIZE) == Baddr)

/* functions to set and test the lowest two bits of a word */
#define	BIT0		(01)	/* ....01 */
#define BIT1		(02)	/* ...010 */
#define BITS01		(03)	/* ...011 */
#define ISBIT0(w)	((w) & BIT0)
#define ISBIT1(w)	((w) & BIT1)
#define	SETBIT0(w)	((w) |= BIT0)
#define SETBIT1(w)	((w) |= BIT1)
#define CLRBIT0(w)	((w) &= ~BIT0)
#define CLRBIT1(w)	((w) &= ~BIT1)
#define SETBITS01(w)	((w) |= BITS01)
#define CLRBITS01(w)	((w) &= ~BITS01)

static TREE	*Root,		/* root of the free tree */
		*Bottom,	/* the last free chunk in the arena */
		*morecore();	/* function to get more core */

static char	*Baddr,		/* current high address of the arena */
		*Free;		/* the previous freed block */

/*
**	Release lower part of core
*/
lesscore()
{
#ifndef SEGMENTED
	reg int		n;
	reg TREE	*tp;
	reg char	*addr;

	if(!(tp = Bottom) || SIZE(tp) < 2*CORESIZE || GETCORE(0) != Baddr)
		return;
	n = ((SIZE(tp) - CORESIZE)/CORESIZE)*CORESIZE;
	if((addr = GETCORE(-n)) != ERRCORE)
	{
		SIZE(tp) = (addr - ((char*) tp)) - 2*WORDSIZE;
		SIZE(NEXT(tp)) = BIT0;
		Baddr = addr;
	}
#endif
}

/*
**	Allocation of small blocks
*/
#define MINSIZE	(sizeof(TREE)-sizeof(WORD))
#define LGET	64
static TREE	*List[(MINSIZE-WORDSIZE)/WORDSIZE]; /* lists of small blocks */

static char	*smalloc(size)
reg uint	size;
{
	reg TREE	*tp;
	reg int		i, n;

	if(size == 0)
		return NIL(char*);

	/* list to use */
	i = size/WORDSIZE - 1;
	if(List[i] == NIL(TREE*))
	{
		if(!(tp = (TREE *) malloc((size+WORDSIZE)*LGET)))
			return NIL(char*);
		List[i] = tp;

		/* make them into a link list */
		for(n = LGET; n > 0;)
		{
			SIZE(tp) = size;
			tp = LINK(tp) = --n ? NEXT(tp) : NIL(TREE*);
		}
	}

	/* allocate from the head of the queue */
	tp = List[i];
	List[i] = LINK(tp);
	return DATA(tp);
}

/*
**	malloc().
*/
char	*malloc(size)
reg uint	size;
{
	reg int		n;
	reg TREE	*tp, *sp;

	/**/ COUNT(nmalloc);

	/* free the delayed free block */
	if(Free)
		free(Free);

	/* make sure that size is 0 mod WORDSIZE */
	size = ROUND(size,WORDSIZE);

	/* small blocks */
	if(size < MINSIZE)
		return smalloc(size);

	/* search for an elt of the right size */
	sp = NIL(TREE*);
	if(Root)
	{
		tp = Root;
		while(1)
		{	/* branch left */
			if(SIZE(tp) >= size)
			{
				sp = tp;
				if(LEFT(tp))
					tp = LEFT(tp);
				else	break;
			}
			else
			{ /* branch right */
				if(RIGHT(tp))
					tp = RIGHT(tp);
				else	break;
			}
		}

		if(sp)
		{	/* found one */
			if(tp = LINK(sp))	/* assignment == */
				sp = tp;
			t_delete(sp);
		}
		else if(tp != Root)
		{	/* half the search path */
			t_splay(tp);
			Root = tp;
		}
	}

	if(!sp)
	{	/* none in the tree fits */
		if(!(sp = Bottom) || size > SIZE(sp))
			sp = morecore(size);
		if(!sp)
			return NIL(char*);
	}

	/* tell the forward neighbor that we're busy */
	CLRBIT1(SIZE(NEXT(sp)));	/**/ ASSERT(ISBIT0(SIZE(NEXT(sp))));

	/* if the leftover is enough for a new free piece */
	if((n = (SIZE(sp) - size) - WORDSIZE) >= MINSIZE)
	{
		SIZE(sp) = size;
		tp = NEXT(sp);
		SIZE(tp) = n;
		free(DATA(tp));
	}
	else if(BOTTOM(sp))
		Bottom = NIL(TREE*);

	/* return the allocated space */
	SETBIT0(SIZE(sp));
	return DATA(sp);
}

/*
**	realloc().
**	If the block size is increasing, we try forward merging first.
**	This is not best-fit but it avoids some data recopying.
*/
char	*realloc(old,size)
char		*old;
reg uint	size;
{
	reg TREE	*tp, *np;
	reg int		n, ts;
	reg char	*new;

	/**/ COUNT(nrealloc);

	/* free the last delayed free block */
	if(Free)
		free(Free);

	/* make sure that size is 0 mod WORDSIZE */
	size = ROUND(size,WORDSIZE);

	/* pointer to the block */
	tp = BLOCK(old);
	ts = SIZE(tp);

	/* nothing to do */
	CLRBITS01(SIZE(tp));
	if(size == SIZE(tp))
	{
		SIZE(tp) = ts;
		return old;
	}

	/* special cases involving small blocks */
	if(size < MINSIZE || SIZE(tp) < MINSIZE)
		goto call_malloc;

	/* block is increasing in size, try merging the next block */
	if(size > SIZE(tp))
	{
		np = NEXT(tp);
		if(!ISBIT0(SIZE(np)))
		{
			/**/ ASSERT(SIZE(np) >= MINSIZE);
			/**/ ASSERT(!ISBIT1(SIZE(np)));
			SIZE(tp) += SIZE(np)+WORDSIZE;
			if(np != Bottom)
				t_delete(np);
			else	Bottom = NIL(TREE*);
			CLRBIT1(SIZE(NEXT(np)));
		}

#ifndef SEGMENTED
		/* not enough & at TRUE end of memory, try extending core */
		if(size > SIZE(tp) && BOTTOM(tp) && GETCORE(0) == Baddr)
		{
			Bottom = tp;
			morecore(size);
		}
#endif /*!SEGMENTED*/
	}

	/* got enough space to use */
	if(size <= SIZE(tp))
	{
		if((n = (SIZE(tp) - size) - WORDSIZE) >= MINSIZE)
		{
			SIZE(tp) = size;
			np = NEXT(tp);
			SIZE(np) = n;
			free(DATA(np));
		}
		else if(BOTTOM(tp))
			Bottom = NIL(TREE*);

		/* the previous block may be free */
		if(ISBIT1(ts))
			SETBIT1(SIZE(tp));
		SETBIT0(SIZE(tp));
		return old;
	}

	/* call malloc to get a new block */
call_malloc:
	SETBIT0(SIZE(tp));
	if(ISBIT1(ts))
		SETBIT1(SIZE(tp));
	if((new = malloc(size)))
	{
		ts = SIZE(tp);
		CLRBITS01(ts);
		memcopy(new,old,ts < size ? ts : size);
	}
	free(old);
	return new;
}

/*
**	free().
*/
free(old)
char	*old;
{
	reg TREE	*tp, *sp, *np;
	reg int		ts, size;

	/**/ COUNT(nfree);

	if(!Free)
	{	/* delay one cycle to save dumb programs that access freed data */
		Free = old;
		return;
	}

	/* pointer to the block to be freed */
	tp = BLOCK(Free);
	Free = Free == old ? NIL(char*) : old;
	ts = SIZE(tp);
	CLRBITS01(SIZE(tp));

	/* small block, put it in the right linked list */
	if(SIZE(tp) < MINSIZE)
	{
		ts = SIZE(tp)/WORDSIZE - 1;
		LINK(tp) = List[ts];
		List[ts] = tp;
		return 0;
	}

	/* see if coalescing with next block is warranted */
	np = NEXT(tp);
	if(!ISBIT0(SIZE(np)))
	{
		if(np != Bottom)
			t_delete(np);
		SIZE(tp) += SIZE(np)+WORDSIZE;
	}

	/* the same with the preceding block */
	if(ISBIT1(ts))
	{
		np = LAST(tp);		/**/ ASSERT(!ISBIT0(SIZE(np)));
					/**/ ASSERT(np != Bottom);
		t_delete(np);
		SIZE(np) += SIZE(tp)+WORDSIZE;
		tp = np;
	}

	/* initialize tree info */
	LINK(tp) = PARENT(tp) = LEFT(tp) = RIGHT(tp) = NIL(TREE*);

	/* the last word of the block contains self's address */
	*(SELFP(tp)) = tp;

	/* set bottom block, or insert in the free tree */
	if(BOTTOM(tp))
		Bottom = tp;
	else if(!Root)
		Root = tp;
	else
	{	/* do a leaf or list insertion */
		size = SIZE(tp);
		np = Root;
		while(1)
		{
			if(SIZE(np) > size)
			{
				if(LEFT(np))
					np = LEFT(np);
				else
				{
					LEFT(np) = tp;
					PARENT(tp) = np;
					break;
				}
			}
			else if(SIZE(np) < size)
			{
				if(RIGHT(np))
					np = RIGHT(np);
				else
				{
					RIGHT(np) = tp;
					PARENT(tp) = np;
					break;
				}
			}
			else /* SIZE(np) == size */
			{
				if(sp = LINK(np))	/* assignment = */
				{
					LINK(tp) = sp;
					BACK(sp) = tp;
				}
				LINK(np) = tp;
				BACK(tp) = np;
				SETNOTREE(tp);
				break;
			}
		}
	}

	/* tell next block that this one is free */
	SETBIT1(SIZE(NEXT(tp)));	/**/ ASSERT(ISBIT0(SIZE(NEXT(tp))));

	return 0;
}

/*
**	Get more core. Gaps in memory are noted as busy blocks.
*/
static TREE	*morecore(size)
reg int		size;
{
	reg TREE	*tp;
	reg char	*addr;

	/* get a multiple of CORESIZE */
	size = ROUND((size + 2*WORDSIZE),CORESIZE);
	if((addr = GETCORE(size)) == ERRCORE)
		return NIL(TREE*);

	/* contiguous memory */
	if(addr == Baddr)
	{
		if(Bottom)
		{
			addr = ((char *)Bottom);
			size += SIZE(Bottom) + 2*WORDSIZE;
		}
		else
		{
			addr = Baddr-WORDSIZE;
			size += WORDSIZE;
		}
	}

	/* new bottom address */
	Baddr = addr + size;

	/* new bottom block */
	tp = ((TREE *) addr);
	SIZE(tp) = size - 2*WORDSIZE;	/**/ASSERT((SIZE(tp)%WORDSIZE) == 0);

	/* reserved the last word to head any noncontiguous memory */
	SIZE(NEXT(tp)) = BIT0;

	/* non-contiguous memory, free old bottom block */
	if(Bottom && Bottom != tp)
		free(DATA(Bottom));

	return tp;
}

/*
**	Tree rotation functions (BU: bottom-up, TD: top-down)
*/

#define LEFT1(x,y)	if((RIGHT(x) = LEFT(y))) PARENT(RIGHT(x)) = x;\
			if((PARENT(y) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(y)) = y;\
				else RIGHT(PARENT(y)) = y;\
			LEFT(y) = x; PARENT(x) = y

#define RIGHT1(x,y)	if((LEFT(x) = RIGHT(y))) PARENT(LEFT(x)) = x;\
			if((PARENT(y) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(y)) = y;\
				else RIGHT(PARENT(y)) = y;\
			RIGHT(y) = x; PARENT(x) = y

#define BULEFT2(x,y,z)	if((RIGHT(x) = LEFT(y))) PARENT(RIGHT(x)) = x;\
			if((RIGHT(y) = LEFT(z))) PARENT(RIGHT(y)) = y;\
			if((PARENT(z) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(z)) = z;\
				else RIGHT(PARENT(z)) = z;\
			LEFT(z) = y; PARENT(y) = z; LEFT(y) = x; PARENT(x) = y

#define BURIGHT2(x,y,z)	if((LEFT(x) = RIGHT(y))) PARENT(LEFT(x)) = x;\
			if((LEFT(y) = RIGHT(z))) PARENT(LEFT(y)) = y;\
			if((PARENT(z) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(z)) = z;\
				else RIGHT(PARENT(z)) = z;\
			RIGHT(z) = y; PARENT(y) = z; RIGHT(y) = x; PARENT(x) = y

#define TDLEFT2(x,y,z)	if((RIGHT(y) = LEFT(z))) PARENT(RIGHT(y)) = y;\
			if((PARENT(z) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(z)) = z;\
				else RIGHT(PARENT(z)) = z;\
			PARENT(x) = z; LEFT(z) = x;

#define TDRIGHT2(x,y,z)	if((LEFT(y) = RIGHT(z))) PARENT(LEFT(y)) = y;\
			if((PARENT(z) = PARENT(x)))\
				if(LEFT(PARENT(x)) == x) LEFT(PARENT(z)) = z;\
				else RIGHT(PARENT(z)) = z;\
			PARENT(x) = z; RIGHT(z) = x;

/*
**	Delete a tree element
*/
static	t_delete(op)
reg TREE	*op;
{
	reg TREE	*tp, *sp, *gp;

	/* if this is a non-tree node */
	if(ISNOTREE(op))
	{
		tp = BACK(op);
		if(sp = LINK(op))	/* assignment = */
			BACK(sp) = tp;
		LINK(tp) = sp;
		return;
	}

	/* make op the root of the tree */
	if(PARENT(op))
		t_splay(op);

	/* if this is the start of a list */
	if(tp = LINK(op))	/* assignment = */
	{
		PARENT(tp) = NIL(TREE*);
		if(sp = LEFT(op))	/* assignment = */
			PARENT(sp) = tp;
		LEFT(tp) = sp;

		if(sp = RIGHT(op))	/* assignment = */
			PARENT(sp) = tp;
		RIGHT(tp) = sp;

		Root = tp;
		return;
	}

	/* if op has a non-null left subtree */
	if(tp = LEFT(op))	/* assignment = */
	{
		PARENT(tp) = NIL(TREE*);

		if(RIGHT(op))
		{
			/* make the right-end of the left subtree its root */
			for(sp = RIGHT(tp); sp != NIL(TREE*); sp = RIGHT(tp))
			{
				if(gp = RIGHT(sp))	/* assignment = */
				{
					TDLEFT2(tp,sp,gp);
					tp = gp;
				}
				else
				{
					LEFT1(tp,sp);
					tp = sp;
				}
			}

			/* hook the right subtree of op to the above elt */
			RIGHT(tp) = RIGHT(op);
			PARENT(RIGHT(tp)) = tp;
		}
	}

	/* no left subtree */
	else if(tp = RIGHT(op))		/* assignment = */
		PARENT(tp) = NIL(TREE*);

	Root = tp;
}

/*
**	Bottom up splaying (simple version).
*/
static	t_splay(tp)
reg TREE	*tp;
{
	reg TREE	*pp, *gp;

	/* iterate until tp is the root */
	for(pp = PARENT(tp); pp != NIL(TREE*); pp = PARENT(tp))
	{	/* grandparent of tp */
		gp = PARENT(pp);

		/* x is a left child */
		if(LEFT(pp) == tp)	
		{
			if(gp && LEFT(gp) == pp)
			{
				BURIGHT2(gp,pp,tp);
			}
			else
			{
				RIGHT1(pp,tp);
			}
		}
		else
		{		/**/ ASSERT(RIGHT(pp) == tp);
			if(gp && RIGHT(gp) == pp)
			{
				BULEFT2(gp,pp,tp);
			}
			else
			{
				LEFT1(pp,tp);
			}
		}
	}
} 
#endif

