/* 
 * Kodkod -- Copyright (c) 2005-2011, Emina Torlak
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kodkod.engine.ucore;

import kodkod.engine.fol2sat.TranslationLog;
import kodkod.engine.fol2sat.Translator;
import kodkod.engine.satlab.ReductionStrategy;
import kodkod.engine.satlab.ResolutionTrace;
import kodkod.util.ints.IntCollection;
import kodkod.util.ints.IntIterator;
import kodkod.util.ints.IntSet;
import kodkod.util.ints.Ints;
/**
 * Adaptive Recycling Core Extraction is a strategy for generating unsat cores that are minimal at the logic level.  
 * Specifically, let C be a core that is minimal according to this strategy, 
 * and let F(C) be the top-level logic constraints
 * corresponding to C.  Then, this strategy guarantees that there is no clause
 * c in C such that F(C - c) is a strict subset of F(C). Furthermore, it also
 * guarantees that for all f in F(C), F(C) - f is satisfiable.  This is a stronger
 * guarantee than that of {@linkplain HybridStrategy}.  In general, using this strategy
 * is more expensive, timewise, than using {@linkplain HybridStrategy}.
 * 
 * <p>Unlike RCE, ARCE is parameterized with 3 values that control the amount of recycling.  The
 * first is the <tt>noRecycleRatio</tt>, which completely disables recycling if it is greater than
 * the ratio of the size of the core passed to {@linkplain #next(ResolutionTrace)} and the number of axioms in the 
 * trace.  The default value is .03; if the core makes up only 3 percent of the axioms, no recycling 
 * will happen.  The remaining two parameters are the <tt>recycleLimit</tt> and the <tt>hardnessCutOff</tt>.
 * If the hardness of the proof passed to {@linkplain #next(ResolutionTrace)} is greater than <tt>hardnessCutOff</tt>,
 * then the number of relevant axioms, |A_r|, plus the number of recycled resolvents is no greater than 
 * |A_r|*<tt>recycleLimit</tt>.  Otherwise, all valid
 * resolvents are recycled (i.e. added to the relevant axioms).  
 * Proof hardness is the ratio of the trace size to the number of axioms in the trace.
 * Default value for <tt>hardnessCutOff</tt> is 2.0, and  default value for <tt>recycleLimit</tt> is 1.2. 
 * 
 * <p>This implementation of ARCE will work properly only on CNFs generated by the kodkod {@linkplain Translator}. </p>
 * 
 * @specfield noRecycleRatio: double
 * @specfield hardnessCutOff: double
 * @specfield recycleLimit: double
 * @invariant noRecycleRatio in [0..1]
 * @invariant recycleLimit >= 1
 * @invariant hardnessCutOff >= 1
 * @author Emina Torlak
 * @see HybridStrategy
 */
public final class AdaptiveRCEStrategy implements ReductionStrategy {
	private final IntCollection varsToTry;
	private final double noRecycleRatio, recycleLimit, hardnessCutOff;
	private static final boolean DBG = true;

	/**
	 * Constructs an ARCE strategy that will use the given translation
	 * log to relate the cnf clauses back to the logic constraints from 
	 * which they were generated. 
	 * @ensures this.hardnessCutOff' = 2 and this.recycleLimit' = 1.2 and this.noRecycleRatio' = .03
	 */
	public AdaptiveRCEStrategy(final TranslationLog log) {
		this(log, .03, 2.0, 1.2);
	}
	
	/**
	 * Constructs an ARCE strategy that will use the given translation
	 * log to relate the cnf clauses back to the logic constraints from 
	 * which they were generated. 
	 * @ensures this.hardnessCutOff' = hardnessCutOff and this.recycleLimit' = recycleLimit and 
	 * this.noRecycleRatio' = noRecycleRatio
	 */
	public AdaptiveRCEStrategy(final TranslationLog log, double noRecycleRatio, double hardnessCutOff, double recycleLimit) {
		varsToTry = StrategyUtils.rootVars(log); 
		if (noRecycleRatio<0 || noRecycleRatio>1) 
			throw new IllegalArgumentException("noRecycleRatio must be in [0..1]: " + noRecycleRatio);
		if (hardnessCutOff < 1)
			throw new IllegalArgumentException("hardnessCutOff must be >=1: " + hardnessCutOff);
		if (recycleLimit < 1)
			throw new IllegalArgumentException("recycleLimit must be >=1: " + recycleLimit);
		this.noRecycleRatio = noRecycleRatio;
		this.hardnessCutOff = hardnessCutOff;
		this.recycleLimit = recycleLimit;
	}

	/**
	 * {@inheritDoc}
	 * @see kodkod.engine.satlab.ReductionStrategy#next(kodkod.engine.satlab.ResolutionTrace)
	 */
	public IntSet next(ResolutionTrace trace) {
		if (varsToTry.isEmpty()) return Ints.EMPTY_SET; // tried everything

		final IntSet relevantVars = StrategyUtils.coreTailUnits(trace);

		
		for(IntIterator varItr = varsToTry.iterator( ); varItr.hasNext();) {
			final int var = varItr.next();
			varItr.remove();
			if (relevantVars.remove(var)) {	// remove maxVar from the set of relevant variables
				if (relevantVars.isEmpty()) break; // there was only one root formula left
				// get all axioms and resolvents corresponding to the clauses that
				// form the translations of formulas identified by relevant vars
				final IntSet relevantClauses = clausesFor(trace, relevantVars); 
				assert !relevantClauses.isEmpty() && !relevantClauses.contains(trace.size()-1);
				
				if (DBG) System.out.println("relevant clauses: " + relevantClauses.size() + ", removed " + var);
				
				return relevantClauses;
			}
		}
		
		varsToTry.clear();		
		return Ints.EMPTY_SET;
	}
	

	/**
	 * Returns the indices of all axioms and resolvents
	 * in the given trace that form the translations of the formulas
	 * identified by the given variables.  This method assumes that
	 * the axioms in the given trace were generated by the Kodkod
	 * {@linkplain Translator}.
	 * @return 
	 * let C = { c: trace.prover.clauses | c.maxVariable() in relevantVars },
	 *     T = { c1, c2: C | c2.maxVariable() in abs(c1.literals) },
	 *     A = C.*T | 
	 *     trace.backwardReachable(A) - trace.backwardReachable(trace.axioms() - A)
	 */
	private IntSet clausesFor(ResolutionTrace trace, IntSet relevantVars) { 
		final double hardness = (double) trace.size() / (double) trace.axioms().size();
		final double coreRatio = ((double) trace.core().size() / (double) trace.axioms().size());
		
		if (DBG) System.out.println("\ntrace size: " + trace.size() + ", axioms: " + trace.axioms().size() + ", core: " + trace.core().size() + ", resolvents: " + trace.resolvents().size());
		if (DBG) System.out.println("hardness: " + hardness + ", coreRatio: " + coreRatio);
		
		final IntSet relevantAxioms = StrategyUtils.clausesFor(trace, relevantVars);
		if (DBG) System.out.println("relevant axioms:  " + relevantAxioms.size());
		
		if (coreRatio < noRecycleRatio) {
			return relevantAxioms;
		} else if (hardness < hardnessCutOff) { 
			return trace.learnable(relevantAxioms);
		} else {
			IntSet current = relevantAxioms, last;
			final int maxRelevant = (int) Math.rint(relevantAxioms.size()*recycleLimit);
			do {
				last = current;
				current = trace.directlyLearnable(current);
			} while (last.size() < current.size() && current.size() < maxRelevant);
			
			if (DBG) System.out.println("last: " + last.size() +", current: " + current.size() + ", maxRelevant: " + maxRelevant);
			
			return current.size() < maxRelevant ? current : last;
		}
				
	}

}
