/* this file is a part of gau software, (C) Hwang chi-deok 1997, 1998, 1999 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifndef USE_XIM
#define USE_XIM
#endif
#include <gdk/gdkprivate.h>
#include "ansiterm.h"
#include "gau.h"

extern int gau_status_draw(XIC ic, gpointer user_data, XIMStatusDrawCallbackStruct *st);


enum {TERM_GROUND, TERM_ESC, TERM_CSI, TERM_IYAGI,
	TERM_SCS0, TERM_IGNORE, TERM_DEC, TERM_OSC};
enum {
  ARG_0,
  ARG_HADJUSTMENT,
  ARG_VADJUSTMENT
};
#define LINE_CHANGING 
#define NOW_LINE_CHANGING {gtk_signal_emit(GTK_OBJECT(term), term_signals[LINE_CHANGED]);}

#define DRAW_TIMEOUT 100

static char tmp[20];

#if HANGUL
static int iyagi_color_table[] = {
	0, 4, 2, 6, 1, 5, 3, 7,
	0, 4, 2, 6, 1, 5, 3, 7};
#endif

#define BS_CLEARS FALSE

#define V_CURRENT_BYTE(term) (*CURRENT_BYTE(term))
#define V_CURRENT_COLOR(term) (*CURRENT_COLOR(term))
#define V_CURRENT_ATTR(term) (*CURRENT_ATTR(term))
#define V_COLOR(fg, bg) ((fg << 4) | bg)

static void ansi_term_clear(AnsiTerm *term);
static void ansi_term_scroll_up(AnsiTerm *term);
static void ansi_term_scroll_down(AnsiTerm *term);
static int ansi_term_receive_string(AnsiTerm *term, unsigned char *buf, int bytes);
static void ansi_gdk_draw_text(GdkWindow *win, GdkGC *gc, int x, int y, 
	guchar *buf, int len, AnsiTerm *term);
static void ansi_term_rcv_byte(AnsiTerm *term, unsigned char c);
static void ansi_term_clear_to_end_of_line(AnsiTerm *term);
static void ansi_term_clear_to_bol(AnsiTerm *term);
static void ansi_term_clear_line(AnsiTerm *);
static void ansi_term_clear_to_end_of_screen(AnsiTerm *term);
static void ansi_term_clear_to_begin_of_screen(AnsiTerm *term);
static void ansi_term_line_clean(AnsiTerm *, int i);
static void ansi_term_line_copy(AnsiTerm *term, int des_line, int src_line);

static int ansi_term_expose(GtkWidget *w, GdkEventExpose *ev);
static void ansi_term_destroy(GtkObject *obj);
static void ansi_term_realize(GtkWidget *w);
static gint ansi_term_focus_in(GtkWidget *w, GdkEventFocus *ev);
static gint ansi_term_focus_out(GtkWidget *w, GdkEventFocus *ev);
static void ansi_term_unrealize (GtkWidget *widget);
static void ansi_term_size_allocate(GtkWidget *w, GtkAllocation *allocation); 
static void ansi_term_size_request (GtkWidget *w, GtkRequisition *requisition);
gint ansi_term_im_init(AnsiTerm *term);

static void ansi_term_insert_spaces(AnsiTerm *, int n);
static void ansi_term_overwrite_spaces(AnsiTerm *, int n);
static void ansi_term_delete_chars(AnsiTerm *, int n);
static void ansi_term_insert_empty_lines(AnsiTerm *, int n);
static void ansi_term_delete_lines(AnsiTerm *, int n);
static void translate_position(AnsiTerm *term, GdkRectangle *area, int *y1,
	int *y2, int *x1, int *x2);
static void ansi_term_restore_cursor(AnsiTerm *term);
static void ansi_term_save_cursor(AnsiTerm *term);
static void ansi_term_complete_draw(AnsiTerm *term);
static void ansi_term_expose_line_nopixmap(AnsiTerm *term, int y, int x1, int x2);

static void	state_ground(AnsiTerm *, unsigned char c);
static void	state_esc(AnsiTerm *, unsigned char c);
static void	state_csi(AnsiTerm *, unsigned char c);
#if HANGUL
static void	state_iyagi(AnsiTerm *, unsigned char c);
#endif
static void state_scs0(AnsiTerm *term, unsigned char c);
static void state_dec(AnsiTerm *term, unsigned char c);
static void state_osc(AnsiTerm *term, unsigned char c);

static void cursor_up(AnsiTerm *);
static void cursor_down(AnsiTerm *);
static void ansi_term_tmp_buffer(AnsiTerm *term, char *buf, int byte);
static char tran_graphic_mode(char c);
static void ansi_term_new_line(AnsiTerm *term);
static void ansi_term_set_attr(AnsiTerm *term);
static void ansi_term_line_erase(AnsiTerm *term);
static void ansi_term_screen_erase(AnsiTerm *term);
static void ansi_term_report(AnsiTerm *term);
static void ansi_term_goto(AnsiTerm *term);

static gint ansi_term_timer(gpointer data);
static void ansi_term_selection_received(GtkWidget *widget, GtkSelectionData *selection_data, guint time);
static void ansi_term_finalize(GtkObject *object);
//static void ansi_term_draw (GtkWidget *widget, GdkRectangle *area);
static void ansi_term_set_adjustments (AnsiTerm *term, GtkAdjustment *hadj, GtkAdjustment *vadj);
static GdkColor ** ansi_term_color_alloc(GdkColormap *cmap);

static void ansi_term_init(AnsiTerm *term);
static void ansi_term_class_init(AnsiTermClass *class);

static GtkWidgetClass *parent_class = NULL;
enum {
	SIZE_CHANGED,
	LINE_CHANGED,
	RECEIVED,
	LAST_SIGNAL
};
static gint term_signals[LAST_SIGNAL] = { 0 };

GtkType
ansi_term_get_type(void)
{
    static GtkType term_type = 0;
    if (!term_type) {
	GtkTypeInfo ansi_info =
	{
	    "AnsiTerm",
	    sizeof(AnsiTerm),
	    sizeof(AnsiTermClass),
	    (GtkClassInitFunc) ansi_term_class_init,
	    (GtkObjectInitFunc) ansi_term_init,
	    /* reserved_1 */ NULL,
	    /* reserved_2 */ NULL,
	    (GtkClassInitFunc) NULL
	};
	term_type = gtk_type_unique(gtk_widget_get_type(), &ansi_info);
    }
    return term_type;
}

static void
ansi_term_class_init(AnsiTermClass *klass)
{
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass *)klass;
    widget_class = (GtkWidgetClass*) klass;
    parent_class = gtk_type_class (gtk_widget_get_type());

    term_signals[SIZE_CHANGED] = gtk_signal_new("size_changed",
	    GTK_RUN_LAST,
	    object_class->type,
	    GTK_SIGNAL_OFFSET (AnsiTermClass, size_changed),
	    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);
    term_signals[LINE_CHANGED] = gtk_signal_new("line_changed",
	    GTK_RUN_LAST,
	    object_class->type,
	    GTK_SIGNAL_OFFSET (AnsiTermClass, line_changed),
	    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);
    term_signals[RECEIVED] = gtk_signal_new("received",
	    GTK_RUN_LAST,
	    object_class->type,
	    GTK_SIGNAL_OFFSET (AnsiTermClass, received),
	    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);
    gtk_object_class_add_signals (object_class, term_signals, LAST_SIGNAL);

    widget_class->expose_event = ansi_term_expose;
    widget_class->key_press_event = ansi_term_key_press;
    widget_class->size_allocate = ansi_term_size_allocate;
    widget_class->size_request = ansi_term_size_request;
    widget_class->focus_in_event = ansi_term_focus_in;
    widget_class->focus_out_event = ansi_term_focus_out;
    /*
    widget_class->selection_clear_event = ansi_term_selection_clear;
    */
    widget_class->selection_received = ansi_term_selection_received;
    widget_class->realize = ansi_term_realize;
    widget_class->unrealize = ansi_term_unrealize;
    widget_class->set_scroll_adjustments_signal =
    	gtk_signal_new ("set_scroll_adjustments", GTK_RUN_LAST,
		object_class->type,
		GTK_SIGNAL_OFFSET (AnsiTermClass, set_scroll_adjustments),
		gtk_marshal_NONE__POINTER_POINTER,
		GTK_TYPE_NONE, 2, GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);
//    widget_class->draw = ansi_term_draw;
    object_class->finalize = ansi_term_finalize;
    object_class->destroy = ansi_term_destroy;
    klass->received = NULL;
    klass->size_changed = NULL;
    klass->line_changed = NULL;
    klass->set_scroll_adjustments = ansi_term_set_adjustments;
}

static void
ansi_term_init(AnsiTerm *term)
{
    GTK_WIDGET_SET_FLAGS (term, GTK_CAN_FOCUS);
    GTK_WIDGET_UNSET_FLAGS (term, GTK_NO_WINDOW);
    term->keymap_list = NULL;
    term->draw_flag = TRUE;
    term->state = TERM_GROUND;
    term->real_visible_line_begin = 0;
    term->visible_line_begin = 0;
    term->cu_x = term->cu_y = 0;
    term->cu_attr = 0;
    term->charset = 'B';
    term->mode = 0;
    term->tmp_str = g_new(char, 256);
    term->tmp_alloc = 256;
    term->tmp_len = 0;
    term->timer = 0;

    term->normal_fg = TERM_COLOR_BLACK;
    term->normal_bg = TERM_COLOR_WHITE;
    term->use_bold = TRUE;
    term->use_under = TRUE;
    term->need_slide_update = FALSE;
    term->can_block = TRUE;
    term->auto_linefeed = FALSE;
    term->silent = FALSE;
    term->draw_flag = TRUE;
}

GtkWidget *
ansi_term_new(char *font_name, int col, int row, int saved_lines, 
	ToTermFunc to_term, FromTermFunc from_term)
{
    AnsiTerm*term;
    GtkWidget *w;
    
    term = gtk_type_new(ansi_term_get_type());
    w = GTK_WIDGET(term);

    term->font = gdk_fontset_load(font_name);
    if (term->font == NULL) {
	g_print("Can't load terminal font %s\n", font_name);
#if HANGUL
	g_print("Instead, will use -schumacher-clean-medium-r-normal--16-160-*-irv*,*-daewoo-*\n");
	term->font = gdk_fontset_load("-schumacher-clean-medium-r-normal--16-160-*-irv*,*-daewoo-*");
#else
	g_print("Instead, will use fixed font\n");
	term->font = gdk_font_load("fixed");
#endif
    }

    g_assert(term->font != NULL);
    term->font_width = gdk_string_width(term->font," ");
#if HANGUL
    g_assert (term->font_width*2 == gdk_string_width(term->font, ""));
#endif
    term->font_ascent = term->font->ascent;
    term->font_height = term->font->descent + term->font->ascent;

    term->col = col;
    term->row = row;
    term->top = 1; term->bot = row;
    if (saved_lines < row) saved_lines = row;
    if (saved_lines > 2000) {
	    g_warning("too huge saved_lines is requested");
	    g_warning("re-adjusted to 2000");
	    saved_lines = 2000;
    }
    term->adjustment = NULL;

    term->saved_lines = saved_lines;
    term->buf = g_new(guchar, col*saved_lines);
    term->color = g_new(guchar, col*saved_lines);
    term->attr = g_new(guchar, col*saved_lines);
    if (to_term == NULL) term->to_term = (ToTermFunc)ansi_term_receive_string;
    else term->to_term = to_term;
    term->from_term = from_term;
    return w;
}

static void
ansi_term_selection_received(GtkWidget *widget, 
			     GtkSelectionData *selection_data,
			     guint time)
{
    AnsiTerm *term;
    term = ANSI_TERM(widget);
    term->from_term(selection_data->data, selection_data->length);
}

static void
ansi_term_line_clean(AnsiTerm *term, int line)
{
    char *s;
    int offset = linenumber(line) * term->col;
    guchar default_color = V_COLOR(term->normal_fg, term->normal_bg);
    /* char */
    s = term->buf + offset;
    memset(s, ' ', term->col);
    /* color */
    s = term->color + offset;
    memset(s, default_color, term->col);
    /* attr */
    s = term->attr + offset;
    memset(s, 0, term->col);
}

static void
ansi_term_line_copy(AnsiTerm *term, int des_line, int src_line)
{
    int des_real_line = linenumber(des_line)*term->col;
    int src_real_line = linenumber(src_line)*term->col;
    char *des, *src;
    /* char */
    des = term->buf + des_real_line;
    src = term->buf + src_real_line;
    memcpy(des, src, term->col);
    /* color */
    des = term->color + des_real_line;
    src = term->color + src_real_line;
    memcpy(des, src, term->col);
    /* attr */
    des = term->attr + des_real_line;
    src = term->attr + src_real_line;
    memcpy(des, src, term->col);
}

static void 
ansi_term_clear(AnsiTerm *term)
{ 
    int i;
    if (term->max_reached == 0 && term->cu_x == 0) return;

    gdk_window_clear_area(GTK_WIDGET(term)->window, 0, 0,
		term->col * term->font_width,
		(term->max_reached + 1) * term->font_height);

    term->real_visible_line_begin += term->max_reached + 1;
    term->real_visible_line_begin %= term->saved_lines;
    if (term->real_visible_line_begin > term->adjustment->value 
	&& term->real_visible_line_begin <= (term->saved_lines - term->row)) {
	term->adjustment->value = term->real_visible_line_begin;
	term->adjustment->upper = term->real_visible_line_begin + term->row;
	if (term->timer) {
	    term->need_slide_update = TRUE;
	} else {
	    gtk_signal_emit_by_name (GTK_OBJECT(term->adjustment), "changed", NULL);
	}
    }
    term->visible_line_begin = term->real_visible_line_begin;
    for (i = term->row - term->max_reached - 1; i < term->row; i++) {
	ansi_term_line_clean(term, i);
    }
    term->cu_fg = term->normal_fg;
    term->cu_bg = term->normal_bg;
    term->cu_attr = 0;
    term->cu_x = term->cu_y = 0;
    term->max_reached = 0;
}

static void 
ansi_term_scroll_up(AnsiTerm *term)
{
    int y;
    int real_bot = MIN(term->bot, term->max_reached + 1);
    if (term->top == 1) {
	/* This is global scroll */
	/* shift scrolled lines */
	term->real_visible_line_begin += 1;
	term->real_visible_line_begin %= term->saved_lines;
	if (term->real_visible_line_begin > term->adjustment->value 
		&& term->adjustment->value < (term->saved_lines - term->row)) {
	    term->adjustment->value = term->real_visible_line_begin;
	    term->adjustment->upper = term->real_visible_line_begin + term->row;
	    if (term->timer) {
		term->need_slide_update = TRUE;
	    } else {
		gtk_signal_emit_by_name (GTK_OBJECT(term->adjustment), "changed", NULL);
	    }
	}
	term->visible_line_begin = term->real_visible_line_begin;
	if (term->bot != term->row)
		term->max_reached = term->row;
	if (term->bot <= term->max_reached + 1) {
	    for (y = term->max_reached; y >= term->bot; y--) 
		ansi_term_line_copy(term, y, y - 1);
	}
    } else {
	/* local scroll */
	/* do not save */
	for (y = term->top; y < real_bot; y++) {
	    ansi_term_line_copy(term, y-1, y);
	}
    }
    ansi_term_line_clean(term, real_bot - 1);
    if (!term->draw_flag) return;
    for (y = term->top - 1; y < real_bot; y++) 
    	ansi_term_expose_line(term, y, 0, term->col -1);
}

static void 
ansi_term_scroll_down(AnsiTerm *term)
{
    int y;
    for (y = term->bot - 1; y >= term->top; y--) {
	ansi_term_line_copy(term, y, y -1);
    }
    ansi_term_line_clean(term, term->top -1);
    if (!term->draw_flag)  return;
    for (y = term->top - 1; y < term->bot; y++) 
    	ansi_term_expose_line(term, y, 0, term->col -1);
}

void 
ansi_term_hide_cursor(AnsiTerm *term)
{
    int n;

    if (term->timer) return;

    if (term->cu_x >= term->col) return;
    if (term->visible_line_begin != term->real_visible_line_begin) return;
    n = 0;
    ansi_term_expose_line(term, term->cu_y, term->cu_x, term->cu_x + n);
}

void
ansi_term_show_cursor(AnsiTerm *term)
{
    int n;
    GtkWidget *w;

    if (term->timer || !GTK_WIDGET_VISIBLE(term)) return;

    w = GTK_WIDGET(term);

    /* test whether screen is virtual or not. */
    if (term->visible_line_begin != term->real_visible_line_begin) return;
    if (term->cu_x >= term->col) return;
#if HANGUL
    if ((V_CURRENT_BYTE(term)) & 0x80) n = 2;
    else 
#endif
    n = 1;
    gdk_gc_set_foreground(term->cursor_gc, term->color_table[TERM_COLOR_CURSOR_BG]);
    gdk_draw_rectangle(w->window, term->cursor_gc, TRUE,
	    term->cu_x *term->font_width, term->cu_y * term->font_height,
	    term->font_width*n, term->font_height);
    gdk_gc_set_foreground(term->cursor_gc, term->color_table[TERM_COLOR_CURSOR_FG]);
    ansi_gdk_draw_text(w->window, term->cursor_gc, 
	    term->cu_x * term->font_width, 
	    term->cu_y * term->font_height + term->font_ascent,
	    CURRENT_BYTE(term), n, term);
    if (GTK_WIDGET_HAS_FOCUS(w) && gdk_im_ready() && term->ic && 
	  (gdk_ic_get_style (term->ic) & GDK_IM_PREEDIT_POSITION)) {
	int mask = 0;
	if (term->ic_attr->spot_location.x != term->cu_x * term->font_width || term->ic_attr->spot_location.y != term->cu_y * term->font_height + term->font_ascent) {
	    mask = GDK_IC_SPOT_LOCATION;
	    term->ic_attr->spot_location.x = term->cu_x * term->font_width;
	    term->ic_attr->spot_location.y = term->cu_y * term->font_height 
		    + term->font_ascent;
	}
	if (term->ic_attr->preedit_foreground.pixel != term->color_table[term->cu_fg]->pixel)
	        term->ic_attr->preedit_foreground.pixel = term->color_table[term->cu_fg]->pixel;
	if (term->cu_bg == term->normal_bg) {
	    if (term->ic_attr->preedit_background.pixel != w->style->bg[GTK_STATE_NORMAL].pixel) {
		mask |= GDK_IC_PREEDIT_BACKGROUND;
		term->ic_attr->preedit_background.pixel = 
		    	w->style->bg[GTK_STATE_NORMAL].pixel;
	    }
	} else if (term->ic_attr->preedit_background.pixel != term->color_table[term->cu_bg]->pixel) {
	    mask |= GDK_IC_PREEDIT_BACKGROUND;
	    term->ic_attr->preedit_background.pixel = 
			term->color_table[term->cu_bg]->pixel;
	}
	if (mask) gdk_ic_set_attr (term->ic, term->ic_attr, mask);
    }
}

void
ansi_term_flush(AnsiTerm *term)
{
    if (term->tmp_len != 0) {
	ansi_term_receive_string(term, term->tmp_str, term->tmp_len);
	term->tmp_len = 0;
    }
}

int
ansi_term_receive_string(AnsiTerm*term, unsigned char *buf, int bytes)
{
    int i;
    if (bytes == 0) return FALSE;

    if (term->visible_line_begin != term->real_visible_line_begin) {
	if (term->can_block) {
	    ansi_term_tmp_buffer(term, buf, bytes);
	    return FALSE;
	}
	term->adjustment->value = term->adjustment->upper - term->row;
	gtk_signal_emit_by_name(GTK_OBJECT(term->adjustment), "value_changed");
    }
    ansi_term_hide_cursor(term);
    
    while (bytes > 0) {
	if (term->state == TERM_GROUND) {
	    for (i = 0; (i< term->col - term->cu_x) && (bytes > 0); i++, bytes--) {
		if (buf[i] < 0x20) break;
	    }
	    if (i) {
		int real_line_offset = term->col *linenumber(term->cu_y);
		if (term->charset == '0') {
		    int j;
		    for (j = 0; j < i; j++) {
			buf[j] = tran_graphic_mode(buf[j]);
		    }
		} else {
		    if (term->charset == 'C') {
			int j;
			for (j = 0; j < i; j++) {
			    if (buf[i] >= 0x21 && buf[i] <= 0x7e) buf[i] |= 0x80;
			}
		    }
		}
		memcpy(term->buf + real_line_offset + term->cu_x,
			buf, i);
		memset(term->color + real_line_offset + term->cu_x,
			V_COLOR(term->cu_fg, term->cu_bg), i);
		memset(term->attr + real_line_offset + term->cu_x,
			term->cu_attr, i);
		ansi_term_expose_line(term, term->cu_y, term->cu_x, term->cu_x + i - 1);
		term->cu_x += i;
		buf += i;
	    }
	} 
	if (bytes > 0) {
	    ansi_term_rcv_byte(term, *buf++);
	    bytes--;
	}
    }
    if (!term->draw_flag) {
	term->draw_flag = TRUE;
	ansi_term_queue_draw(term);
    }
    gtk_signal_emit((GtkObject*)term, term_signals[RECEIVED]);
    ansi_term_show_cursor(term);
    return FALSE;
}

void 
ansi_term_rcv_byte(AnsiTerm *term, unsigned char c)
{ 
    switch (term->state) {
	case TERM_GROUND:
	    state_ground(term, c);
	    break;
	case TERM_ESC: /* ESC */
	    state_esc(term, c);
	    break;
	case TERM_CSI: /* ESC [ */
	    state_csi(term, c);
	    break;
#if HANGUL
	case TERM_IYAGI:
	    state_iyagi(term, c);
	    break;
#endif
	case TERM_IGNORE:
	    term->state = TERM_GROUND;
	    break;
	case TERM_SCS0:
	    state_scs0(term, c);
	    break;
	case TERM_DEC:
	    state_dec(term, c);
	    break;
	case TERM_OSC:
	    state_osc(term, c);
	    break;
    }
}

static void 
state_ground(AnsiTerm *term, unsigned char c)
{
	switch (c) {
        case 128+ESC : /* ESC+128 == ESC [ */
			break;
			term->state = TERM_CSI;
        case ESC : /* ESC */
			term->state = TERM_ESC;
			break;
        case '\b' :
			term->cu_x = MAX(0,term->cu_x-1);
			if (BS_CLEARS) V_CURRENT_BYTE(term) = ' ';
			break;
		case '\n' : 
			if (term->cu_x != 0) NOW_LINE_CHANGING;
			ansi_term_new_line(term);
			break;
        case '\r' : 
			NOW_LINE_CHANGING;
			term->cu_x = 0;
			break;
		case 7: /* bell */
			if (!term->silent) gdk_beep();
			break;
		case 14: 
			/*
			 ^N
			 FIXME
			 enter alternate charset ==> g1
			*/
			term->charset = '0';
			break;
		case 15: /* ^O */
		    /* enter default charset ==> g0
			// below is the dirty hack.
			// but it seems to work in most cases.
			*/
		    term->charset = 'B';
			/* FIXME:: */
			break;
		case 12: /* ^L */
			term->argv[0] = 0;
			cursor_down(term);
			break;
		case 9: /* horizontal tab */
			term->cu_x += 8; /* FIXME  */
			term->cu_x = (term->cu_x/8)*8;
			if (term->cu_x >= term->col)
			{
				term->cu_x = 0;
				term->cu_y++;
				if (term->cu_y >= term->row) {
					term->draw_flag = FALSE;
					ansi_term_scroll_up(term);
					term->cu_y = term->row -1;
				}
				term->max_reached = MAX(term->max_reached, term->cu_y);
			}
			break;
		case 0:
			/* FIXME:: */
			break;
		default:
			/* FIXME
			// term->insert check.
			*/
			if (term->cu_x >= term->col)
			{
				LINE_CHANGING;
				term->cu_x = 0;
				term->cu_y++;
				if (term->cu_y >= term->row) {
					term->draw_flag = FALSE;
					ansi_term_scroll_up(term);
					term->cu_y = term->row -1;
				}
				term->max_reached = MAX(term->max_reached, term->cu_y);
			}
			if (term->charset == '0') {
				V_CURRENT_BYTE(term) = tran_graphic_mode(c);
			} else {
				/* FIXME */
				if (term->charset == 'C') c |= 0x80;
				V_CURRENT_BYTE(term) = c;
			}
			V_CURRENT_COLOR(term) = V_COLOR(term->cu_fg, term->cu_bg);
			ansi_term_expose_line(term, term->cu_y, term->cu_x, term->cu_x);
			term->cu_x += 1;
			break;
      };
}

static void
state_esc(AnsiTerm *term, unsigned char c)
{
    switch (c) {
	case 7:
	    if (!term->silent) gdk_beep();
	    term->state = TERM_GROUND;
	    break;
        case 'Z': /* Report terminal type */
/*
	    if (vt_type == VT100)
		v_termout("\033[?1;0c", 0);
	    else	
*/
		sprintf(tmp, "\033[?c");
            term->from_term(tmp, 4);
			term->state = TERM_GROUND;
			break;	
	case '[' : /* ESC [ */
	    term->argc  = 0;
	    term->argv[0] = 0;
	    term->state = TERM_CSI;
	    break;
	case '(': /* choose charset */
	    term->state = TERM_SCS0;
	    break;
	case 'M':
	    LINE_CHANGING;
	    ansi_term_scroll_down(term);
	    term->state = TERM_GROUND;
	    break;
	case 'D':
	    LINE_CHANGING;
	    if (term->cu_y >= (term->bot - 1)) {
		term->draw_flag = FALSE;
		ansi_term_scroll_up(term);
	    } else
		term->cu_y++;
	    term->state = TERM_GROUND;
	    break;
	case '7':
	    ansi_term_save_cursor(term);
	    term->state = TERM_GROUND;
	    break;
	case '8': /* restore the old position of cursor */
	    ansi_term_restore_cursor(term);
	    term->state = TERM_GROUND;
	    break;
	case '=':
	    /* enter keypad application mode
	    // FIXME
		*/
	    term->state = TERM_GROUND;
	    break;
	case '>':
	    term->state = TERM_GROUND;
	    /* leave keypad application mode
	    // FIXME
		*/
	    break;
	case ')':
	    /* FIXME */
	    term->state = TERM_IGNORE;
	    break;
	case ']':
	    term->argv[0] = 0;
	    term->state = TERM_OSC;
	    break;
	case 'c':
	    /* reset: FIXME */
	    ansi_term_reset(term);
	    term->state = TERM_GROUND;
	    break;
	case 14:
	    term->charset = '0';
	    term->state = TERM_GROUND;
	    break;
	case 15:
	    term->charset = 'B';
	    term->state = TERM_GROUND;
	    break;
        default: 
	    term->state = TERM_GROUND;
	    break;
      };
}

void
cursor_up(AnsiTerm *term)
{
    term->argv[0] = MAX(1,term->argv[0]);
    term->cu_y = MIN(term->row-1,MAX(0,term->cu_y-term->argv[0]));
    term->max_reached = MAX(term->cu_y, term->max_reached);
}

void
cursor_down(AnsiTerm *term)
{
    term->argv[0] = MAX(1,term->argv[0]);
    term->cu_y = MIN(term->row - 1, term->cu_y + term->argv[0]);
    term->max_reached = MAX(term->max_reached, term->cu_y);
}

static void
state_scs0(AnsiTerm *term, unsigned char c)
{
	/* FIXME */
    switch (c) {
	case '0':
	case 'B':
	case 'C':
	term->charset = c;
	break;
    }
    term->state = TERM_GROUND;
}

static void
state_dec(AnsiTerm *term, unsigned char c)
{
    switch(c) {
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	    term->argv[term->argc] = 10*(term->argv[term->argc]) + c - '0';
	    break;
	case 'h': 
	    /* set */
	    term->state = TERM_GROUND;
	    switch(term->argv[term->argc]) {
		case 1:
		    term->mode |= CUR_APL;
		    break;
		case 1000: case 1001:
		    term->mode |= EDITOR_BUTTON;
		    break;
	    }
	    break;
	case 'l':
	    /* reset */
	    term->state = TERM_GROUND;
	    switch(term->argv[term->argc]) {
		case 1:
		    term->mode &= ~CUR_APL;
		    break;
		case 1000: case 1001:
		    term->mode &= ~EDITOR_BUTTON;
		    break;
	    }
	    break;
	default:
	    term->state = TERM_GROUND;
    }
}

static void
state_osc(AnsiTerm *term, unsigned char c)
{
	static int init = TRUE;
	char title[150];
	static char *s;
	GtkWidget *top;
	if (init && isdigit(c)) {
		term->argv[0] = 10*(term->argv[0]) + c - '0';
	} else if (init && !isdigit(c)) {
		init = FALSE;
		s = title;
	} else if (isprint(c) || c & 0x80) {
		*s++ = c;
	} else {
		*s = 0;
		if (term->argv[0] == 0 || term->argv[0] == 2) {
			top = gtk_widget_get_toplevel(GTK_WIDGET(term));
			gtk_window_set_title(GTK_WINDOW(top), title);
		}
		term->state = TERM_GROUND;
		init = TRUE;
	}
}

static void
state_csi(AnsiTerm *term, unsigned char c)
{
	switch (c) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			term->argv[term->argc] = 10*(term->argv[term->argc]) + c - '0';
			break;
        case ';':
			term->argc = MIN(term->argc+1,MAX_ARGS-1);
			term->argv[term->argc] = 0;
			break;
		case 'A' : /* ESC [ Pn A               Cursor Up                */
			LINE_CHANGING;
			cursor_up(term);
			term->state = TERM_GROUND;
			break;
        case 'B' : /* ESC [ Pn B               Cursor Down              */
			LINE_CHANGING;
			cursor_down(term);
			term->state = TERM_GROUND;
			break;
        case 'C' : /* ESC [ Pn C               Cursor Right             */
			term->argv[0] = MAX(1,term->argv[0]);
			term->cu_x = MIN(term->col-1,term->cu_x+term->argv[0]);
			term->state = TERM_GROUND;
			break;
        case 'D' : /* ESC [ Pn D               Cursor Left              */
			term->argv[0] = MAX(1,term->argv[0]);
			term->cu_x = MAX(0,term->cu_x-term->argv[0]);
			term->state = TERM_GROUND;
			break;
        case 'm' : /* ESC [ Ps ; ... ; Ps m    Select Graphic Rendition */
			ansi_term_set_attr(term);
			break;
        case 'f' : /* ESC [ Py ; Px f    Direct Cursor Addressing */
        case 'H' : /* ESC [ Py ; Px H    Direct Cursor Addressing */
			LINE_CHANGING;
			ansi_term_goto(term);
			break;
        case 'K' : /* ESC [ Pn K         Line Erasing             */
			ansi_term_line_erase(term);
			break;
        case 'J' : /* ESC [ Pn J         Screen Erasing           */
			ansi_term_screen_erase(term);
			break;
        case 'n' : /* ESC [ Pn n          Report */
			ansi_term_report(term);
			break;
		case 'r':
			term->top = term->argv[0];
			if (term->argc >= 1) term->bot = term->argv[1];
			else term->bot = term->row;
			if (term->top < 1) term->top = 1;
			if (term->bot > term->row || term->bot <= 0) term->bot = term->row;
			term->state = TERM_GROUND;
			break;
		case 's': /* save position of cursor */
			ansi_term_save_cursor(term);
			term->state = TERM_GROUND;
			ansi_term_restore_cursor(term);
			break;
		case 'u': /* restore the old position of cursor */
			ansi_term_restore_cursor(term);
			term->state = TERM_GROUND;
			break;
#if HANGUL
		case '=': /* iyagi ansi code */
			term->state = TERM_IYAGI;
			break;
		case 'O':
			/* line draw iyagi ansi */
			term->state = TERM_GROUND;
			break;
#endif
		case 'X':
			/* overwrite spaces at current positoin */
			ansi_term_overwrite_spaces(term, term->argv[0]);
			term->state = TERM_GROUND;
			break;
		case '@':
			/* insert spaces at current position */
			if (term->argv[0] < 1) term->argv[0] = 1;
			ansi_term_insert_spaces(term, term->argv[0]);
			term->state = TERM_GROUND;
			break;
		case 'P':
			if (term->argv[0] < 1) term->argv[0] = 1;
			ansi_term_delete_chars(term, term->argv[0]);
			term->state = TERM_GROUND;
			break;
		case 'L':
			if (term->argv[0] < 1) term->argv[0] = 1;
			ansi_term_insert_empty_lines(term, term->argv[0]);
			term->state = TERM_GROUND;
			break;
		case 'M':
			if (term->argv[0] < 1) term->argv[0] = 1;
			ansi_term_delete_lines(term, term->argv[0]);
			term->state = TERM_GROUND;
			break;
		case '?':
			/* enter dec state; */
			term->argc = 0;
			term->argv[0] = 0;
			term->state = TERM_DEC;
			/* FIXME */
			break;
		case 'l':
			/* FIXME */
			if (term->argv[0] == 4) {
				/* enter insert mode
				// term->insert = FALSE;
				*/
			}
			term->state = TERM_GROUND;
			break;
		case 'h':
			/* FIXME */
			if (term->argv[0] == 4) {
				/* enter insert mode
				// term->insert = TRUE;
				*/
			}
			term->state = TERM_GROUND;
			break;
		case 14:
			term->charset = '0';
			term->state = TERM_GROUND;
			break;
		case 15:
			term->charset = 'B';
			term->state = TERM_GROUND;
			break;
		case ESC:
			/*  Ѽ Ƚ ΰ Ķ */
			term->state = TERM_ESC;
			term->argc = 0;
			term->argv[0] = 0;
			break;
        default  :
			/*g_print("char %d(%c) in state %d\n", 
			//	c,(c&0x7f)>31?c:'?',term->state); 
			*/
			term->state = TERM_GROUND;
			break;
      }
}

#if HANGUL
static void
state_iyagi(AnsiTerm *term, unsigned char c)
{
	switch(c) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			term->argv[term->argc] = 10*(term->argv[term->argc]) + c - '0';
			break;
        case ';':
			term->argc = MIN(term->argc+1,MAX_ARGS-1);
			term->argv[term->argc] = 0;
			break;
		case 'F':
			/* foreground color */
			if (term->argv[0] > 15) {
				g_warning("invalid foreground color %d", term->argv[0]);
			} else
				term->cu_fg = 
				     iyagi_color_table[term->argv[0]];
			term->state = TERM_GROUND;
			break;
		case 'G':
			/* background color */
			if (term->argv[0] > 15) {
				g_warning("invalid background color %d", term->argv[0]);
			} else
				term->cu_bg = iyagi_color_table[term->argv[0]];
			term->state = TERM_GROUND;
			break;
		default:
			term->state = TERM_GROUND;
			break;
	}
}
#endif

void 
ansi_term_home(AnsiTerm *term)
{
	term->cu_x = 0;
	term->cu_y = 0;
}

void 
ansi_term_reset(AnsiTerm *term)
{
	ansi_term_clear(term);
	ansi_term_sane(term);
}

void
ansi_term_sane(AnsiTerm *term)
{
    term->state = TERM_GROUND;
    term->mode = 0;
    term->charset = 'B';

    term->top = 1;
    term->bot = term->row;

    term->cu_attr = 0;
    term->cu_fg = term->normal_fg;
    term->cu_bg = term->normal_bg;
}

static void
ansi_term_save_cursor(AnsiTerm *term)
{
     term->old_x = term->cu_x;
     term->old_y = term->cu_y;
     term->old_fg = term->cu_fg;
     term->old_bg = term->cu_bg;
     term->old_attr = term->cu_attr;
}

static void
ansi_term_restore_cursor(AnsiTerm *term)
{
    term->cu_x = term->old_x;
    term->cu_y = term->old_y;
    term->cu_fg = term->old_fg;
    term->cu_bg = term->old_bg;
    term->cu_attr = term->old_attr;
    if (term->max_reached < term->cu_y) term->max_reached = term->cu_y;
}

static gint
nearest_pow (gint num)
{
  gint n = 1;

  while (n < num)
    n <<= 1;

  return n;
}

static void
ansi_term_tmp_buffer(AnsiTerm *term, char *buf, int byte)
{
    if (term->tmp_alloc < byte + term->tmp_len) {
	term->tmp_alloc = nearest_pow(byte + term->tmp_len);
	term->tmp_str = g_realloc(term->tmp_str, term->tmp_alloc);
    }
    memcpy(term->tmp_str + term->tmp_len, buf, byte);
    term->tmp_len += byte;
}

static void
ansi_term_complete_draw(AnsiTerm *term)
{
    int y;

    for (y = 0; y < term->row; y++) {
	ansi_term_expose_line(term, y, 0, term->col - 1);
    }
    ansi_term_show_cursor(term);
}

static int
ansi_term_expose(GtkWidget *w, GdkEventExpose *ev)
{
    int y, y1, y2, x1, x2;
    AnsiTerm *term;

    if (!GTK_WIDGET_VISIBLE(w)) return TRUE;

    term = ANSI_TERM(w);

    translate_position(term, &ev->area, &y1, &y2, &x1, &x2);
    for (y = y1; y <= y2; y++) {
	ansi_term_expose_line(term, y, x1, x2);
    }
    x1--; x2++;
    if (y1 <= term->cu_y && y2 >= term->cu_y && x1 <= term->cu_x && x2 >= term->cu_x) {
	ansi_term_show_cursor(term);
    }
    return TRUE;
}

static void
translate_position(AnsiTerm *term, GdkRectangle *area, int *y1,
	int *y2, int *x1, int *x2)
{
    *y1 = area->y/term->font_height;
    *y2 = (area->y + area->height - 1)/term->font_height;
    if (*y2 >= term->row) *y2 = term->row - 1;
    *x1 = area->x/term->font_width;
    *x2 = (area->x + area->width - 1)/term->font_width;
    if (*x2 >= term->col) *x2 = term->col - 1;
}

void
ansi_term_expose_line(AnsiTerm *term, int y, int x1, int x2)
{
    int real_line_offset;
    char *buf;
    char *color, *attr;
    static int old_fg_color = -1;
    static int old_bg_color = -1;
    int x;
    int bg_color, fg_color;
    char old_attr;
    char old_color;
    if (!term->have_bgpixmap) {
	ansi_term_expose_line_nopixmap(term, y, x1, x2);
	return;
    }
    if (!term->draw_flag || term->timer) return;
    if (!GTK_WIDGET_VISIBLE(term)) return;
    if (x2 < x1) return;
    if (x1 < 0 || x2 >= term->col) return;

    real_line_offset = linenumber(y) * term->col;
    buf = term->buf + real_line_offset;

    if (is_second_han(buf, x1)) x1--;
    if (is_first_han(buf, x2)) x2++;

    if (x1 < 0) x1= 0;
    if (x2 >= term->col) x2 = term->col - 1;

    color = term->color + real_line_offset + x1;
    attr = term->attr + real_line_offset + x1;
    y = y * term->font_height;
    x = x1;
    while (x1 <= x2) {
	old_attr = *attr;
	old_color = *color;
	do {
	    x++;
	    attr++;
	    color++;
	} while ((x<= x2) && (*attr == old_attr) && (*color == old_color));
	if (old_attr & ATTR_SELECTION) {
		bg_color = (old_color>>4) & 15;
	} else if (old_attr & ATTR_REVERSE) {
		bg_color = (old_color>>4) & 15;
	} else 
	    bg_color = old_color & 15;
	if (bg_color != term->normal_bg && old_bg_color != bg_color) {
		if (bg_color < 0 || bg_color > 7) {
			/*g_warning("invalid bg_color %d\n", bg_color); */
		} else {
			gdk_gc_set_foreground(term->bg_gc, term->color_table[bg_color]);
			old_bg_color = bg_color;
		}
	}

	if (bg_color == term->normal_bg) {
		gdk_window_clear_area(GTK_WIDGET(term)->window, x1*term->font_width,
			y, term->font_width*(x-x1), term->font_height);
	} else gdk_draw_rectangle(GTK_WIDGET(term)->window, term->bg_gc, TRUE,
		x1*term->font_width,
		y, term->font_width*(x - x1), term->font_height);
	if (old_attr & ATTR_SELECTION) {
		fg_color = old_color & 15;
	} else if (old_attr & ATTR_REVERSE) {
		fg_color = old_color & 15;
	} else
		fg_color = (old_color>>4) & 15;
	if (fg_color == term->normal_bg && bg_color == term->normal_bg) {
		fg_color = term->normal_fg;
	} else if (fg_color == bg_color) {
		if (fg_color != term->normal_fg) fg_color = term->normal_fg;
		else {
			if (term->normal_fg == 7) fg_color = 0;
			else fg_color = 7;
		}
	}
	if (fg_color != old_fg_color) {
	    if (fg_color < 0 || fg_color > 7) {
		/*g_warning("invalid fg_color %d\n", fg_color); */
	    } else {
		old_fg_color = fg_color;
		gdk_gc_set_foreground(term->gc, term->color_table[fg_color]);
	    }
	}
	ansi_gdk_draw_text(GTK_WIDGET(term)->window, term->gc, x1*term->font_width,
	    y + term->font_ascent, buf + x1, x - x1, term);
	if (term->use_bold && (old_attr & ATTR_BOLD)) {
	    ansi_gdk_draw_text(GTK_WIDGET(term)->window, term->gc, 
		x1*term->font_width + 1, 
		y + term->font_ascent, 
		buf + x1 , x - x1, term);
	}
	if (term->use_under && (old_attr & ATTR_UNDER)) {
	    gdk_draw_line(GTK_WIDGET(term)->window, term->gc,
			x1*term->font_width, y + term->font_height - 1,
			x*term->font_width, y + term->font_height - 1);
	}
	x1 = x;
    }
}

static void
ansi_term_expose_line_nopixmap(AnsiTerm *term, int y, int x1, int x2)
{
    int real_line_offset;
    char *buf;
    char *color, *attr;
    static int old_fg_color = -1;
    static int old_bg_color = -1;
    static GdkPixmap *pixmap = NULL;
    static int pixmap_size = 0;
    int startx;
    int x;
    int bg_color, fg_color;
    char old_attr;
    char old_color;
    if (!term->draw_flag || term->timer) return;
    if (!GTK_WIDGET_VISIBLE(term)) return;
    if (x2 < x1) return;
    if (x1 < 0 || x2 >= term->col) return;

    real_line_offset = linenumber(y) * term->col;
    buf = term->buf + real_line_offset;

    if (is_second_han(buf, x1)) x1--;
    if (is_first_han(buf, x2)) x2++;

    if (x1 < 0) x1= 0;
    if (x2 >= term->col) x2 = term->col - 1;

    startx = x1;
    if (old_bg_color != term->normal_bg) {
	gdk_gc_set_foreground (term->bg_gc, &GTK_WIDGET(term)->style->bg[GTK_STATE_NORMAL]);
	old_bg_color = term->normal_bg;
    }
    if (x2 - x1 + 1 > pixmap_size) {
	if(pixmap) gdk_pixmap_unref (pixmap);
	pixmap = gdk_pixmap_new (GTK_WIDGET(term)->window, 
		    term->font_width*term->col,
		    term->font_height, -1);
	pixmap_size = term->col;
    }
    gdk_draw_rectangle(pixmap, term->bg_gc, TRUE,
		       0, 0,
		       term->font_width*(x2 - x1 + 1), term->font_height);
    color = term->color + real_line_offset + x1;
    attr = term->attr + real_line_offset + x1;
    y = y * term->font_height;
    x = x1;
    while (x1 <= x2) {
	old_attr = *attr;
	old_color = *color;
	do {
	    x++;
	    attr++;
	    color++;
	} while ((x<= x2) && (*attr == old_attr) && (*color == old_color));
	if (old_attr & ATTR_SELECTION) {
		bg_color = (old_color>>4) & 15;
	} else if (old_attr & ATTR_REVERSE) {
		bg_color = (old_color>>4) & 15;
	} else 
	    bg_color = old_color & 15;
	if (bg_color != term->normal_bg && old_bg_color != bg_color) {
		if (bg_color < 0 || bg_color > 7) {
			/*g_warning("invalid bg_color %d\n", bg_color); */
		} else {
			gdk_gc_set_foreground(term->bg_gc, term->color_table[bg_color]);
			old_bg_color = bg_color;
		}
	}

	if (bg_color != term->normal_bg) {
	    gdk_draw_rectangle(pixmap, term->bg_gc, TRUE,
		(x1 - startx)*term->font_width, 0,
		term->font_width*(x - x1), term->font_height);
	}
	if (old_attr & ATTR_SELECTION) {
		fg_color = old_color & 15;
	} else if (old_attr & ATTR_REVERSE) {
		fg_color = old_color & 15;
	} else
		fg_color = (old_color>>4) & 15;
	if (fg_color == term->normal_bg && bg_color == term->normal_bg) {
		fg_color = term->normal_fg;
	} else if (fg_color == bg_color) {
		if (fg_color != term->normal_fg) fg_color = term->normal_fg;
		else {
			if (term->normal_fg == 7) fg_color = 0;
			else fg_color = 7;
		}
	}
	if (fg_color != old_fg_color) {
	    if (fg_color < 0 || fg_color > 7) {
		/*g_warning("invalid fg_color %d\n", fg_color); */
	    } else {
		old_fg_color = fg_color;
		gdk_gc_set_foreground(term->gc, term->color_table[fg_color]);
	    }
	}
	ansi_gdk_draw_text(pixmap, term->gc, (x1-startx)*term->font_width,
	    term->font_ascent, buf + x1, x - x1, term);
	if (term->use_bold && (old_attr & ATTR_BOLD)) {
	    ansi_gdk_draw_text(pixmap, term->gc, 
		(x1-startx)*term->font_width + 1, 
		term->font_ascent, 
		buf + x1 , x - x1, term);
	}
	if (term->use_under && (old_attr & ATTR_UNDER)) {
	    gdk_draw_line(pixmap, term->gc,
			(x1-startx)*term->font_width, term->font_height - 1,
			(x-startx)*term->font_width, term->font_height - 1);
	}
	x1 = x;
    }
    gdk_draw_pixmap (GTK_WIDGET(term)->window, term->gc,
    	pixmap, 
	0, 0, 
	startx*term->font_width,
	y,
	(x2-startx+1)*term->font_width, term->font_height);
}

#define Clear_Color V_COLOR(term->normal_fg, term->normal_bg)
static void
ansi_term_clear_to_end_of_line(AnsiTerm *term)
{
    int n = term->col - term->cu_x;
    int offset = linenumber(term->cu_y)*term->col + term->cu_x;
    memset(term->buf + offset, ' ', n);
    memset(term->color + offset, Clear_Color, n);
    memset(term->attr + offset, 0, n);
    ansi_term_expose_line(term, term->cu_y, term->cu_x, term->col - 1);
}

static void
ansi_term_clear_to_bol(AnsiTerm *term)
{
    int n = term->cu_x + 1;
    int offset = linenumber(term->cu_y) * term->col;
    memset(term->buf + offset, ' ', n);
    memset(term->color + offset, Clear_Color, n);
    memset(term->attr + offset, 0, n);
    ansi_term_expose_line(term, term->cu_y, 0, term->cu_x);
}

static void
ansi_term_clear_line(AnsiTerm *term)
{
    int offset = linenumber(term->cu_y) * term->col;
    memset(term->buf + offset, ' ', term->col);
    memset(term->color + offset, Clear_Color, term->col);
    memset(term->attr + offset, 0, term->col);
    ansi_term_expose_line(term, term->cu_y, 0, term->col - 1);
}

static void
ansi_term_clear_to_end_of_screen(AnsiTerm *term)
{
    int y;
    if (term->cu_y == 0 && term->cu_x == 0) {
	ansi_term_clear(term);
	return;
    }
    ansi_term_clear_to_end_of_line(term);
    y = term->cu_y;
    for (term->cu_y++; term->cu_y < term->row; term->cu_y++)
	ansi_term_clear_line(term);
    term->cu_y = y;
}

static void
ansi_term_clear_to_begin_of_screen(AnsiTerm *term)
{
    int y;
    if (term->cu_y == (term->row -1)) {
	ansi_term_clear(term);
	return;
    }
    y = term->cu_y;
    ansi_term_clear_to_bol(term);
    while (--term->cu_y >= 0) ansi_term_clear_line(term);
    term->cu_y = y;
}

static void
ansi_term_insert_spaces(AnsiTerm *term, int n)
{
    char *buf, *color, *attr;
    int i;
    if(!n) return;
    buf = term->buf + linenumber(term->cu_y)*term->col;
    color = term->color + linenumber(term->cu_y)*term->col;
    attr = term->attr + linenumber(term->cu_y)*term->col;
    for (i = term->cu_x; (i + n) < term->col; i++) {
	buf[i+n] = buf[i];
	color[i+n] = color[i];
	attr[i+n] = attr[i];
    }
    for (i = 0; i < n; i++) {
	buf[term->cu_x + i] = ' ';
	color[term->cu_x + i] = Clear_Color;
	attr[term->cu_x + i] = 0;
    }
    ansi_term_expose_line(term, term->cu_y, term->cu_x, term->col - 1);
    term->cu_x += n;
}

static void
ansi_term_overwrite_spaces(AnsiTerm *term, int n)
{
    char *buf, *color, *attr;
    int i;
    if (!n) return;
    buf = term->buf + linenumber(term->cu_y)*term->col;
    color = term->color + linenumber(term->cu_y)*term->col;
    attr = term->attr + linenumber(term->cu_y)*term->col;
    for (i = 0; i < n; i++) {
	buf[term->cu_x + i] = ' ';
	color[term->cu_x + i] = V_COLOR(term->cu_fg, term->cu_bg);
	attr[term->cu_x + i] = term->cu_attr;
    }
    ansi_term_expose_line(term, term->cu_y, term->cu_x, term->cu_x + n);
    term->cu_x += n;
    /*g_print("cu_x: %d\n", term->cu_x); */
}

static void
ansi_term_delete_chars(AnsiTerm *term, int n)
{
    char *buf, *color, *attr;
    int i;
    if (!n) return;
    buf = term->buf + linenumber(term->cu_y)*term->col;
    color = term->color + linenumber(term->cu_y)*term->col;
    attr = term->attr + linenumber(term->cu_y)*term->col;
    for (i = term->cu_x + 1; (i + n) < term->col; i++) {
	buf[i] = buf[i + n];
	color[i] = color[i + n];
	attr[i] = attr[i + n];
    }
    for (i = 1; i <= n; i++) {
	buf[term->col - i] = ' ';
	color[term->col - i] = Clear_Color;
	attr[term->col - i] = 0;
    }
    ansi_term_expose_line(term, term->cu_y, term->cu_x + 1, term->col - 1);
}

static void
ansi_term_insert_empty_lines(AnsiTerm *term, int n)
{
    int old_top, old_bot;
    if (!n) return;
    old_top = term->top;
    old_bot = term->bot;
    term->top = term->cu_y + 1;
    term->bot = term->row;
    while (n--) ansi_term_scroll_down(term);
    term->top = old_top;
    term->bot = old_bot;
}

static void
ansi_term_delete_lines(AnsiTerm *term, int n)
{
    int old_top, old_bot;
    if (!n) return;
    old_top = term->top;
    old_bot = term->bot;
    term->top = term->cu_y + 1;
    term->bot = term->row;
    while (n--) ansi_term_scroll_up(term);
    term->top = old_top;
    term->bot = old_bot;
}

static void
ansi_term_new_line(AnsiTerm *term)
{

    if (term->auto_linefeed) term->cu_x  = 0;
    term->cu_y += 1;
    if (term->cu_y >= term->bot) {
	term->draw_flag = FALSE;
	ansi_term_scroll_up(term);
	term->cu_y = term->bot-1;
    }  
    term->max_reached = MAX(term->max_reached, term->cu_y);
}

static void
ansi_term_set_attr(AnsiTerm *term)
{
    int i;
    for (i = 0; i <= term->argc; i++) {
	int arg = term->argv[i];
	if (arg >= 30 && arg <= 37) {
		term->cu_fg = arg - 30;
	} else if (arg >= 40 && arg <= 47) {
		term->cu_bg = arg - 40;
	} else 
	switch(arg) {
	    case 0: case -1: case 100:
		term->cu_fg = term->normal_fg;
		term->cu_bg = term->normal_bg;
		term->cu_attr = 0;
		break;
	    case 1: /* bold */
		term->cu_attr |= ATTR_BOLD;
		break;
	    case 4: /* underline */
		term->cu_attr |= ATTR_UNDER;
		break;
	    case 5: /* blink */
		term->cu_attr |= ATTR_BLINK;
		break;
	    case 7:  /* start reverse */
		term->cu_attr |= ATTR_REVERSE;
		break;
	    case 10: case 11:
		break;
	    case 24: /* turn off underline */
		term->cu_attr &= ~ATTR_UNDER;
		break;
	    case 25:
		term->cu_attr &= ~ATTR_BLINK;
		break;
	    case 27: /* end reverse */
		term->cu_attr &= ~ATTR_REVERSE;
		break;
	    case 39:
		term->cu_fg = term->normal_fg;
		break;
	    case 48: /* lower index ?? */
		term->cu_bg = term->normal_bg;
		break;
	    case 49: /* upper index ?? */
		term->cu_bg = term->normal_bg;
		break;
	    default :
		/*
		//g_print("--------> attribute %d\n",arg); 
		*/
		break;
	}
    }
    term->state = TERM_GROUND;
}

static void
ansi_term_line_erase(AnsiTerm *term)
{
    switch (term->argv[0]) {
	case 0: /* ESC [ 0 K         Clear to end of line     */
	    ansi_term_clear_to_end_of_line(term);
	    break;
	case 1: /* ESC [ 1 K         Clear to begin of line   */
	    ansi_term_clear_to_bol(term);
	    break;
	case 2: /* ESC [ 2 K         Clear entire      line   */
	    ansi_term_clear_line(term);
	    break;
    }
    term->state = TERM_GROUND;
}

static void
ansi_term_screen_erase(AnsiTerm *term)
{
    switch (term->argv[0]) {
	case 0: /* ESC [ 0 J         Clear to end of screen   */
	    ansi_term_clear_to_end_of_screen(term);
	    break;
	case 1: /* ESC [ 1 J         Clear to begin of screen */
	    ansi_term_clear_to_begin_of_screen(term);
	    break;
	case 2: /* ESC [ 2 J      Clear entire screen (+ cu home) */
	    ansi_term_clear(term);
	    break;
	default:
	    break;
    }
    term->state = TERM_GROUND;
}

static void
ansi_term_report(AnsiTerm *term)
{
    switch(term->argv[0]) { 
	case 5: /* ESC [ 5 n   Status */
	    sprintf(tmp, "\033[0n");
	    term->from_term(tmp, 4);
	    break;
	case 6:	/* ESC [ 6 n   Cursor Position */
	    sprintf(tmp,"\033[%d;%dR",term->cu_y+1,term->cu_x+1);
	    term->from_term( tmp, strlen(tmp));
	    break;
    }
    term->state = TERM_GROUND;
}

static void
ansi_term_goto(AnsiTerm *term)
{
    int x,y; 
    y = MAX(1,term->argv[0]);
    x = term->argc < 1 ? 1 : MAX(1,term->argv[1]);
    term->cu_x = MIN(term->col-1,MAX(0,x-1));
    term->cu_y = MIN(term->row-1,  MAX(0,y-1));
    term->max_reached = MAX(term->cu_y, term->max_reached);
    term->state = TERM_GROUND;
}

#define SET_COLOR(col, redv, greenv, bluev) { \
     co[col].red = redv; \
	 co[col].green = greenv; \
	 co[col].blue = bluev;}

static char
tran_graphic_mode(char c)
{
    if (c == 'q') return 1;
    if (c == 'k') return 2;
    if (c == 'l') return 3;
    if (c == 'x') return 4;
    if (c == 'j') return 5;
    if (c == 'm') return 6;
    if (c == 't') return 7;
    if (c == 'u') return 8;
    if (c == 'w') return 9;
    if (c == 'v') return 10;
    return c;
}

void
ansi_term_queue_draw(AnsiTerm *term)
{
    if (term->timer) return;
    term->timer = gtk_timeout_add(DRAW_TIMEOUT, ansi_term_timer, term);
}

static gint
ansi_term_timer(gpointer data)
{
    AnsiTerm *term;

    term = ANSI_TERM(data);
    term->timer = 0;
    ansi_term_complete_draw(term);
    if (term->need_slide_update) {
	gtk_signal_emit_by_name (GTK_OBJECT(term->adjustment), "changed", NULL);
	term->need_slide_update = FALSE;
    }
    return FALSE;
}

void
ansi_gdk_draw_text(GdkWindow *win, GdkGC *gc, int x, int y, guchar *buf, int len, AnsiTerm *term)
{
    int i;
    int line_center = y - term->font_ascent + term->font_height/2;
    while (len> 0) {
	i = 0;
	while (i < len && buf[i] > 10) i++;
	if (i) {
	    gdk_draw_text(win, term->font, gc, x, y, buf, i);
	    len -= i;
	    if (len <= 0) break;
	    x += term->font_width*i;
	    buf += i;
	}
	switch(buf[0]) {
	    case 1:
		/* horizontal line */
		gdk_draw_line(win, gc, x, line_center, 
			    x + term->font_width, line_center);
		break;
	    case 2:
		/* upper right */
		gdk_draw_line(win, gc, x, line_center,
			    x + term->font_width/2, line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center + term->font_height);

		break;
	    case 3:
		/* upper left */
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width, line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center + term->font_height);
		break;
	    case 4:
		/* vertical */
		gdk_draw_line(win, gc, x + term->font_width/2, 
			    line_center - term->font_height/2,
			    x + term->font_width/2, 
			    line_center + term->font_height/2);
		    break;
	    case 5:
		/* lower right */
		gdk_draw_line(win, gc, x, line_center, x + term->font_width/2, 
			    line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center - term->font_height/2);
		break;
	    case 6:
		/* lower left */
		gdk_draw_line(win, gc, x + term->font_width/2, line_center, 
			    x + term->font_width, line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center - term->font_height/2);
		break;
	    case 7:
		/* vertical and c-r horizontal */
		gdk_draw_line(win, gc, x + term->font_width/2, 
			    line_center - term->font_height/2,
			    x + term->font_width/2, line_center + term->font_height/2);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center, 
			    x + term->font_width, line_center);
		break;
	    case 8:
		/* vertical and l-c horizontal */
		gdk_draw_line(win, gc, x, line_center, x + term->font_width/2, 
			    line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, 
			    line_center - term->font_height/2,
			    x + term->font_width/2, line_center + term->font_height/2);
		break;
	    case 9:
		/* horizontal and c-l vertical  */
		gdk_draw_line(win, gc, x, line_center, 
			    x + term->font_width, line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center + term->font_height/2);
		break;
	    case 10:
		/* horizontal and u-c vertical */
		gdk_draw_line(win, gc, x, line_center, 
			    x + term->font_width, line_center);
		gdk_draw_line(win, gc, x + term->font_width/2, line_center,
			    x + term->font_width/2, line_center - term->font_height/2);
	    default:
		break;
	}
	buf++;
	len--;
	x += term->font_width;
    }
}

static void
ansi_term_set_adjustments (AnsiTerm *term, GtkAdjustment *hadj, GtkAdjustment *vadj)
{
    if (!vadj) 
    	vadj = GTK_ADJUSTMENT (gtk_adjustment_new(0, 0, 0, 1, term->row, term->row));  
    if (term->adjustment && term->adjustment != vadj) {
	gtk_signal_disconnect_by_data (GTK_OBJECT(term->adjustment), term);
	gtk_object_unref (GTK_OBJECT (term->adjustment));
    }
    if (term->adjustment != vadj) {
	term->adjustment = vadj;
	gtk_object_ref (GTK_OBJECT (vadj));
	gtk_object_sink (GTK_OBJECT (vadj));
	gtk_signal_connect (GTK_OBJECT (vadj), "value_changed",
		(GtkSignalFunc) ansi_term_adjustment_changed, term);
	vadj->upper = term->row;
	vadj->page_size = term->row;
	vadj->value = vadj->upper - term->row;
	vadj->step_increment = 1;
	vadj->page_increment = term->row-2;

    }
}

int
ansi_term_adjustment_changed(GtkAdjustment *adj, AnsiTerm *term)
{
    int new_visible_line_begin;
    
    if (term->adjustment->upper <= 0) return TRUE;

    new_visible_line_begin =
	((int)term->adjustment->value + term->saved_lines + 
	term->real_visible_line_begin - 
	(int)term->adjustment->upper + term->row)%term->saved_lines;
    if (term->visible_line_begin == new_visible_line_begin) return TRUE;
    term->visible_line_begin = new_visible_line_begin;
    if (term->real_visible_line_begin == term->visible_line_begin &&
	term->tmp_len) {
	ansi_term_flush(term);
    }
    ansi_term_queue_draw(term);
    return TRUE;
}

void
ansi_term_set_size(AnsiTerm *term, int new_col, int new_row)
{
    char *buf, *color, *attr;
    GtkAdjustment *adj;
    char *src, *des;
    int i, saved_lines;
    int min_col;
    unsigned char v_color = V_COLOR(term->normal_fg, term->normal_bg);
    //g_print("%s: %d %d\n", __FUNCTION__, new_col, new_row);

    adj = term->adjustment;
    saved_lines = term->saved_lines;
    buf = g_new(char, new_col*saved_lines);
    color = g_new(char, new_col*saved_lines);
    attr = g_new(char, new_col*saved_lines);
    min_col = MIN(new_col,term->col);
    for (i = 0; i < saved_lines; i++) {
	src = term->buf + term->col*i;
	des = buf + new_col*i;
	memcpy(des, src, min_col);
	if (min_col < new_col)
	    memset(des + min_col, ' ', new_col - min_col);
	src = term->color + term->col*i;
	des = color + new_col*i;
	memcpy(des, src, min_col);
	if (min_col < new_col)
	    memset(des + min_col, v_color, new_col - min_col);
	src = term->attr + term->col*i;
	des = attr + new_col*i;
	memcpy(des, src, min_col);
	if (min_col < new_col)
	    memset(des + min_col, 0, new_col - min_col);
    }


    g_free(term->buf);
    g_free(term->color);
    g_free(term->attr);
    term->buf = buf;
    term->color = color;
    term->attr = attr;

    term->col = new_col;
    if (term->row != new_row) {
	if (term->row < new_row) {
	    for (i = term->row; i < new_row; i++) ansi_term_line_clean(term, i);
	}
	term->row = new_row;
	if (term->cu_y >= new_row) {
	    term->real_visible_line_begin += term->cu_y - new_row + 1;
	    if (term->real_visible_line_begin >= saved_lines) 
		    term->real_visible_line_begin -= saved_lines;
	    term->cu_y = new_row - 1;
	}
	adj->page_size = new_row;
	adj->value = adj->upper - new_row;
	if (adj->value < 0) adj->value = 0;
    }
    term->visible_line_begin = term->real_visible_line_begin;
    if (term->cu_x > new_col) {
	term->cu_x = new_col - 1;
    }
    term->bot = term->row;
    gtk_signal_emit_by_name (GTK_OBJECT(term->adjustment), "changed", NULL);
    gtk_signal_emit(GTK_OBJECT(term), term_signals[SIZE_CHANGED]);
}

static void
ansi_term_size_request(GtkWidget *w, GtkRequisition *requisition)
{
    AnsiTerm *term;
    g_return_if_fail (w != NULL);

    term = ANSI_TERM(w);
    requisition->width = term->col * term->font_width;
    requisition->height = term->row * term->font_height;
    //g_print("%d %d\n", requisition->width, requisition->height);
}

static void
ansi_term_size_allocate(GtkWidget *w, GtkAllocation *allocation)
{
    int new_col, new_row;
    AnsiTerm *term;

    g_return_if_fail (w != NULL);
    term = ANSI_TERM(w);
    new_col = (allocation->width) / term->font_width;
    new_row = (allocation->height) / term->font_height;
    //g_print("new_col=%d, alloc:%d %d\n", new_col, allocation->width, allocation->height);
    ansi_term_set_size(term, new_col, new_row);
    w->allocation = *allocation;

    if (GTK_WIDGET_REALIZED (w)) {
	gdk_window_move_resize (w->window,
		    allocation->x, allocation->y,
		    allocation->width, allocation->height);
    }
    if (term->ic && (gdk_ic_get_style (term->ic) & GDK_IM_PREEDIT_POSITION)) {
	term->ic_attr->preedit_area.width = allocation->width;
	term->ic_attr->preedit_area.height = allocation->height;
	gdk_ic_set_attr (term->ic, term->ic_attr,
	      		   GDK_IC_PREEDIT_AREA);
    }
}

static void
ansi_term_realize (GtkWidget *w)
{
    AnsiTerm *term;
    GdkWindowAttr attributes;
    gint attributes_mask;
    int i;

    g_return_if_fail (w != NULL);

    term = ANSI_TERM(w);
    GTK_WIDGET_SET_FLAGS(w, GTK_REALIZED);

    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.x = w->allocation.x;
    attributes.y = w->allocation.y;
    attributes.width = w->allocation.width;
    attributes.height = w->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.visual = gtk_widget_get_visual (w);
    attributes.colormap = gtk_widget_get_colormap (w);
    attributes.event_mask = gtk_widget_get_events (w);
    attributes.event_mask |= (GDK_EXPOSURE_MASK 
	    | GDK_KEY_PRESS_MASK
	    | GDK_BUTTON_PRESS_MASK 
	    | GDK_BUTTON_RELEASE_MASK
	    | GDK_BUTTON1_MOTION_MASK
	    | GDK_BUTTON3_MOTION_MASK);

    attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

    w->window = gdk_window_new (gtk_widget_get_parent_window(w), &attributes, attributes_mask);
    gdk_window_set_user_data (w->window, term);
    w->style = gtk_style_attach (w->style, w->window);
    gtk_style_set_background (w->style, w->window, GTK_STATE_NORMAL);

    term->gc = gdk_gc_new(w->window);
    term->bg_gc = gdk_gc_new(w->window);
    term->white_gc = gdk_gc_new(w->window);
    term->cursor_gc = gdk_gc_new(w->window);
    term->black_gc = GTK_WIDGET(term)->style->black_gc;
    term->i_sel_gc = gdk_gc_new(w->window);
    term->color_table = ansi_term_color_alloc(attributes.colormap);

    /*
    gdk_gc_set_font(term->gc, term->font);
    gdk_gc_set_font(term->cursor_gc, term->font);
    gdk_gc_set_font(term->white_gc, term->font);
    */
    gdk_gc_set_foreground(term->white_gc, term->color_table[TERM_COLOR_WHITE]);

    gdk_gc_set_foreground(term->i_sel_gc, term->color_table[TERM_COLOR_I_SELECT]);
    /*
    gdk_gc_set_font(term->i_sel_gc, term->font);
    */
    term->cu_fg = term->normal_fg;
    term->cu_bg = term->normal_bg;

    for (i = 0; i < term->row; i++) {
	ansi_term_line_clean(term, i);
    }
    if(!w->style->bg_pixmap[GTK_STATE_NORMAL]) term->have_bgpixmap = FALSE;
    else term->have_bgpixmap = TRUE;
    gtk_signal_emit_by_name(GTK_OBJECT(term->adjustment), "value_changed", NULL);
    ansi_term_im_init(term);
}
/*
static void
ansi_term_draw (GtkWidget *widget, GdkRectangle *area)
{
    return;
}
*/

static void
ansi_term_finalize(GtkObject *object)
{
    AnsiTerm *term;

    term = (AnsiTerm *)object;
    g_free(term->buf);
    g_free(term->color);
    g_free(term->attr);
    g_free(term->tmp_str);
    if (GTK_OBJECT_CLASS(parent_class)->finalize)
	    (*GTK_OBJECT_CLASS(parent_class)->finalize) (object);
}

static void
ansi_term_unrealize (GtkWidget *widget)
{
    AnsiTerm *term;
    term = ANSI_TERM(widget);
    gdk_gc_destroy(term->gc);
    gdk_gc_destroy(term->bg_gc);
    gdk_gc_destroy(term->white_gc);
    gdk_gc_destroy(term->cursor_gc);
    if (term->ic) {
      gdk_ic_destroy (term->ic);
      term->ic = NULL;
    }
    if (term->ic_attr) {
      gdk_ic_attr_destroy (term->ic_attr);
      term->ic_attr = NULL;
    }
    gtk_object_destroy(GTK_OBJECT(term->adjustment));
}

static void
ansi_term_destroy(GtkObject *object)
{
    if (GTK_OBJECT_CLASS(parent_class)->destroy)
	    (*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}

static GdkColor **
ansi_term_color_alloc(GdkColormap *cmap)
{
    GdkColor **color_table;
    int i;

    color_table = g_new(GdkColor *, TERM_COLOR_END);
    color_table[TERM_COLOR_BLACK] = black;
    color_table[TERM_COLOR_RED] = red;
    color_table[TERM_COLOR_GREEN] = green;
    color_table[TERM_COLOR_YELLOW] = yellow;
    color_table[TERM_COLOR_BLUE] = blue;
    color_table[TERM_COLOR_MAGENTA] = magenta;
    color_table[TERM_COLOR_CYAN] = cyan;
    color_table[TERM_COLOR_WHITE] = white;

    color_table[TERM_COLOR_CURSOR_FG] = cursor_fg;
    color_table[TERM_COLOR_CURSOR_BG] = cursor_bg;

    color_table[TERM_COLOR_I_SELECT] = g_new(GdkColor, 1);
    color_table[TERM_COLOR_I_SELECT]->red = 0.9 * 65535;
    color_table[TERM_COLOR_I_SELECT]->green = 0.5 * 65535;
    color_table[TERM_COLOR_I_SELECT]->blue = 0.7 * 65535;

    for (i = 0; i < TERM_COLOR_END; i++)
	gdk_color_alloc(cmap, color_table[i]);

    return color_table;
}

static gint
ansi_term_focus_in (GtkWidget     *widget,
		    GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);

  if (ANSI_TERM(widget)->ic)
    gdk_im_begin (ANSI_TERM(widget)->ic, widget->window);

  return FALSE;
}

static gint
ansi_term_focus_out (GtkWidget     *widget,
		     GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);

  gdk_im_end ();

  return FALSE;
}

gint
ansi_term_im_init(AnsiTerm *term)
{
    GtkWidget *widget;
    widget = GTK_WIDGET(term);
    if (gdk_im_ready () && (term->ic_attr = gdk_ic_attr_new ()) != NULL) {
        GdkEventMask mask;
        GdkColormap *colormap;
        GdkICAttr *attr = term->ic_attr;
        GdkICAttributesType attrmask = GDK_IC_ALL_REQ;
        GdkIMStyle style;
        GdkIMStyle supported_style = GDK_IM_PREEDIT_NONE |
				   GDK_IM_PREEDIT_NOTHING |
			           GDK_IM_PREEDIT_POSITION |
			           GDK_IM_STATUS_NONE |
				   GDK_IM_STATUS_NOTHING |
			           GDK_IM_STATUS_CALLBACKS;

        attr->style = style = gdk_im_decide_style (supported_style);
        attr->client_window = gtk_widget_get_toplevel(widget)->window;

        if ((colormap = gtk_widget_get_colormap (widget)) !=
	    gtk_widget_get_default_colormap ()) {
	    attrmask |= GDK_IC_PREEDIT_COLORMAP;
	    attr->preedit_colormap = colormap;
	}
	/*
        attrmask |= GDK_IC_PREEDIT_FOREGROUND;
        attrmask |= GDK_IC_PREEDIT_BACKGROUND;
	*/
        attr->preedit_foreground.pixel = 22222; 
        attr->preedit_background.pixel = 22222; 
	if ((style & GDK_IM_PREEDIT_MASK) == GDK_IM_PREEDIT_POSITION) {
	    attrmask |= GDK_IC_PREEDIT_POSITION_REQ;
	    attr->spot_location.x = 0;
	    attr->spot_location.y = term->font_ascent;
	    attr->preedit_area.x = 0;
	    attr->preedit_area.y = 0;
	    attr->preedit_area.width = widget->allocation.width;
	    attr->preedit_area.height = widget->allocation.height;
	    attr->preedit_fontset = term->font;
	}
        term->ic = gdk_ic_new (attr, attrmask);
     
        if (term->ic == NULL)
	    g_warning ("Can't create input context.");
        else {
	    mask = gdk_window_get_events (widget->window);
	    mask |= gdk_ic_get_events (term->ic);
	    gdk_window_set_events (widget->window, mask);
	    if ((style & GDK_IM_STATUS_MASK) == GDK_IM_STATUS_CALLBACKS) {
		typedef struct { gchar *name; gpointer value; } GdkImArg;
		GdkImArg arg[2] = {{NULL, NULL}, {NULL, NULL}};
		XIMCallback status_draw_cb;
		XIMCallback status_done_cb;
		XIMCallback status_start_cb;
		status_draw_cb.client_data = NULL;
		status_draw_cb.callback = (XIMProc)gau_status_draw;
		arg->name = XNStatusDrawCallback;
		arg->value = (gpointer) &status_draw_cb;
		XSetICValues(((GdkICPrivate *)term->ic)->xic, XNStatusAttributes, arg, NULL);

		status_start_cb.client_data = NULL;
		status_start_cb.callback = (XIMProc)gtk_true;
		arg->name = XNStatusStartCallback;
		arg->value = (gpointer) &status_start_cb;
		XSetICValues(((GdkICPrivate *)term->ic)->xic, XNStatusAttributes, arg, NULL);

		status_done_cb.client_data = NULL;
		status_done_cb.callback = (XIMProc)gtk_true;
		arg->name = XNStatusDoneCallback;
		arg->value = (gpointer) &status_done_cb;
		XSetICValues(((GdkICPrivate *)term->ic)->xic, XNStatusAttributes, arg, NULL);
	    } else {
		g_warning("Your Input Method does not support Status Callback");
	    }
	    if (GTK_WIDGET_HAS_FOCUS(widget))
		gdk_im_begin (term->ic, widget->window);
	}
    }
    return TRUE;
}

/* 
FIXME!!!!
̹  ۽ 
1. ķ Ȯ
  - HOST->Terminal (ķ Ȯ ڵ)
    0x1B 0x26 0x51
  - Terminal->HOST (ȭ Ѵٴ ڵ)
    ̾߱ - KMODEM  
    0x1B 0x26 0x41 0x31 0x32 0x33 0x34 0x35 0x0D

    Ÿ   Ÿ ſķ - ZMODEM
    0x1B 0x26 0x33 0x0D
*/
/* vt100 graphic mode
   q: horizontal line => 1
   k: upper right => 2
   l: upper left => 3 
   x: vertical line => 4
   j: lower right => 5
   m: lower left => 6
   t: vertical and c-r horizontal => 7
   u: vertical and l-c horizontal => 8
   w: horizontal and c-l vertical => 9
   v: horizontal and u-c vertical => 10
	if(c == 'w') return 9;
	if(c == 'v') return 10;
*/
