/* this file is a part of gau software, (C) Hwang chi-deok 1997-1999        */

#include <string.h>		/* memmove */
#include <ctype.h>		/* isspace isalnum */
#include <gdk/gdkkeysyms.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkvscrollbar.h>
#include <gtk/gtkdrawingarea.h>
#include <gtk/gtkmain.h>
#include "gau.h"
#include "gauchat.h"
#include <gdk/gdkprivate.h>

#define LINE_BUFFER(y) (chat->buf[y])
#define CURRENT_LINE_BUFFER LINE_BUFFER(chat->current_y)
#define CURRENT_CHAR_BUFFER   ((CURRENT_LINE_BUFFER + chat->current_x))

enum {
    SEND,
    LINE_CHANGED,
    LAST_SIGNAL
};

typedef void (*GauChatFunction) (GauChat * chat);

#define CHAT_MASK ( GDK_EXPOSURE_MASK | \
					GDK_BUTTON_PRESS_MASK | \
					GDK_KEY_PRESS_MASK)
gint gau_chat_im_init(GauChat *chat);

static void gau_chat_class_init(GauChatClass * class);
static void gau_chat_init(GauChat * chat);
static void gau_chat_destroy(GtkObject * object);
static void gau_chat_finalize (GtkObject * object);

static gint gau_chat_expose(GtkWidget * w, GdkEventExpose * ev);
static gint gau_chat_button_press(GtkWidget * w, GdkEventButton * ev);
static gint gau_chat_key_press(GtkWidget * w, GdkEventKey * ev);
static void gau_chat_realize(GtkWidget * w);
static void gau_chat_unrealize(GtkWidget * w);
static gint gau_chat_focus_in(GtkWidget *w, GdkEventFocus *ev);
static gint gau_chat_focus_out(GtkWidget *w, GdkEventFocus *ev);
static void gau_chat_size_request(GtkWidget *w, GtkRequisition *requisition);
static void gau_chat_size_allocate (GtkWidget *w, GtkAllocation *allocation);
static void gau_chat_set_adjustments (GauChat *chat, GtkAdjustment *hadj, GtkAdjustment *vadj);
static int gau_chat_adjustment_changed(GtkAdjustment *adj, GauChat *chat);

static void gau_chat_delete_forward(GauChat * chat);
static void gau_chat_delete_backward(GauChat * chat);
static void gau_chat_delete_forward_word(GauChat * chat);
static void gau_chat_delete_backward_word(GauChat * chat);
static void gau_chat_delete_eol(GauChat * chat);
static void gau_chat_delete_line(GauChat * chat);
static void gau_chat_expose_eol(GauChat * chat);
static void gau_chat_clear_line(GauChat * chat, int y, int x1, int x2);
static void gau_chat_show_cursor(GauChat * chat);
static void gau_chat_hide_cursor(GauChat * chat);
static void gau_chat_draw_line(GauChat * chat, int y, int x1, int x2);
static void gau_chat_move_forward(GauChat * chat);
static void gau_chat_move_backward(GauChat * chat);
static void gau_chat_move_upward(GauChat * chat);
static void gau_chat_move_downward(GauChat * chat);
static void gau_chat_move_eol(GauChat * chat);
static void gau_chat_move_bol(GauChat * chat);
static void gau_chat_move_forward_word(GauChat * chat);
static void gau_chat_move_backward_word(GauChat * chat);
static void gau_chat_move_scroll(GauChat * chat);
static gint gau_chat_check_offset(GauChat * chat);

static GtkDrawingAreaClass *parent_class = NULL;
static gint chat_signals[LAST_SIGNAL] =
{0};
static GauChatFunction control_keys[26] =
{
    gau_chat_move_bol,		/* a */
    gau_chat_move_backward,	/* b */
    NULL,			/* c */
    gau_chat_delete_forward,	/* d */
    gau_chat_move_eol,		/* e */
    gau_chat_move_forward,	/* f */
    NULL,			/* g */
    NULL,			/* h */
    NULL,			/* i */
    NULL,			/* j */
    gau_chat_delete_eol,	/* k */
    NULL,			/* l */
    NULL,			/* m */
    gau_chat_move_downward,	/* n */
    NULL,			/* o */
    gau_chat_move_upward,	/* p */
    NULL,			/* q */
    NULL,			/* r */
    NULL,			/* s */
    NULL,			/* t */
    gau_chat_delete_line,	/* u */
    NULL,			/* v */
    gau_chat_delete_backward_word,	/* w */
    NULL,			/* x */
    NULL,			/* y */
    NULL,			/* z */
};

static GauChatFunction alt_keys[26] =
{
    NULL,			/* a */
    gau_chat_move_backward_word,	/* b */
    NULL,			/* c */
    gau_chat_delete_forward_word,	/* d */
    NULL,			/* e */
    gau_chat_move_forward_word,	/* f */
    NULL,			/* g */
    NULL,			/* h */
    NULL,			/* i */
    NULL,			/* j */
    NULL,			/* k */
    NULL,			/* l */
    NULL,			/* m */
    NULL,			/* n */
    NULL,			/* o */
    NULL,			/* p */
    NULL,			/* q */
    NULL,			/* r */
    NULL,			/* s */
    NULL,			/* t */
    NULL,			/* u */
    NULL,			/* v */
    NULL,			/* w */
    NULL,			/* x */
    NULL,			/* y */
    NULL,			/* z */
};

guint
gau_chat_get_type(void)
{
    static guint chat_type = 0;
    if (!chat_type) {
	GtkTypeInfo chat_info =
	{
	    "GauChat",
	    sizeof(GauChat),
	    sizeof(GauChatClass),
	    (GtkClassInitFunc) gau_chat_class_init,
	    (GtkObjectInitFunc) gau_chat_init,
	    /* reserved_1 */ NULL,
	    /* reserved_2 */ NULL,
	    (GtkClassInitFunc) NULL
	};
	chat_type = gtk_type_unique(gtk_drawing_area_get_type(), &chat_info);
    }
    return chat_type;
}

static void 
gau_chat_class_init(GauChatClass * class)
{
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    parent_class = gtk_type_class(gtk_drawing_area_get_type());

    chat_signals[0] = gtk_signal_new("send",
				     GTK_RUN_LAST,
				     object_class->type,
				   GTK_SIGNAL_OFFSET(GauChatClass, send),
			     gtk_marshal_NONE__POINTER_INT, GTK_TYPE_NONE, 2,
				     GTK_TYPE_POINTER, GTK_TYPE_INT);
    chat_signals[1] = gtk_signal_new("line_changed",
				     GTK_RUN_LAST,
				     object_class->type,
			   GTK_SIGNAL_OFFSET(GauChatClass, line_changed),
			gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);
    gtk_object_class_add_signals(object_class, chat_signals, LAST_SIGNAL);
    widget_class->expose_event = gau_chat_expose;
    widget_class->set_scroll_adjustments_signal =
    	gtk_signal_new ("set_scroll_adjustments", GTK_RUN_LAST,
		object_class->type,
		GTK_SIGNAL_OFFSET (GauChatClass, set_scroll_adjustments),
		gtk_marshal_NONE__POINTER_POINTER,
		GTK_TYPE_NONE, 2, GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);
    widget_class->key_press_event = gau_chat_key_press;
    widget_class->button_press_event = gau_chat_button_press;
    widget_class->focus_in_event = gau_chat_focus_in;
    widget_class->focus_out_event = gau_chat_focus_out;
    widget_class->realize = gau_chat_realize;
    widget_class->unrealize = gau_chat_unrealize;
    widget_class->size_request = gau_chat_size_request;
    widget_class->size_allocate = gau_chat_size_allocate;

    object_class->destroy = gau_chat_destroy;
    object_class->finalize = gau_chat_finalize;

    class->set_scroll_adjustments = gau_chat_set_adjustments;
}

static void 
gau_chat_init(GauChat * chat)
{
    int i;
    chat->offset = 0;
    chat->add = -1;
    chat->prefix1 = g_strdup("");
    chat->prefix2 = g_strdup("");
    chat->suffix = g_strdup("");
    chat->current_x = chat->current_y = 0;
    chat->adj = NULL;
    for (i = 0; i < 20; i++)
	chat->fbinding[i] = NULL;
}

GtkWidget *
gau_chat_new(GdkFont * font, int font_height, int font_ascent, int col, int row, int vrow)
{
    GauChat *chat;
    gint mask;
    GtkWidget *w;
    int i;
	//g_print("gc: col: %d\n", col);

    chat = gtk_type_new(gau_chat_get_type());
    w = GTK_WIDGET(chat);
    chat->buf = g_new0(char *, row);
    chat->buf_alloc = g_new(int, row);
    for (i = 0; i < row; i++) {
	chat->buf_alloc[i] = 50;
	chat->buf[i] = g_new(char, chat->buf_alloc[i]);
	chat->buf[i][0] = 0;
    }
    chat->col = col;
    chat->row = row;
    chat->vrow = vrow;
    GTK_WIDGET_SET_FLAGS(chat, GTK_CAN_FOCUS);
    mask = gtk_widget_get_events(w);
    gtk_widget_set_events(w, mask | CHAT_MASK | GDK_STRUCTURE_MASK);
    chat->font_width = gdk_string_width(font, " ");
    if (font_height <= 0)
	chat->font_height = font->ascent + font->descent;
    else
	chat->font_height = font_height;
    if (font_ascent <= 0)
	chat->font_ascent = font->ascent;
    else
	chat->font_ascent = font_ascent;
    chat->font = font;
    return GTK_WIDGET(chat);
}

void 
gau_chat_binding_set(GauChat * chat, int index, char *buf)
{
    chat->fbinding[index] = buf;
}

void 
gau_chat_insert_text(GauChat * chat, char *buf, int len)
{
    int x, y;
    int buf_x;
    char *s;
    gau_chat_hide_cursor(chat);
    s = CURRENT_CHAR_BUFFER;
    buf_x = strlen(s);
    x = chat->current_x;
    y = chat->current_y;
    if ((x+buf_x+len) >= chat->buf_alloc[y]) {
	chat->buf_alloc[y] += 50;
	chat->buf[y] = g_realloc(chat->buf[y], chat->buf_alloc[y]);
	s = CURRENT_CHAR_BUFFER;
    }
    gau_chat_clear_line(chat, y, x, x+buf_x);
    memmove(s + len, s, buf_x + 1);
    memmove(s, buf, len);
    gau_chat_expose_eol(chat);
    chat->current_x += len;
    gau_chat_show_cursor(chat);
}

void 
gau_chat_insert_char(GauChat * chat, char c)
{
    char *s;
    static char *text = NULL;
    int len;
    /*
       g_print("%s: %d %d\n", __FUNCTION__, chat->current_x, chat->current_y);
     */
    if (c == '\r') {
	if (text)
	    g_free(text);
	if (chat->add > 0 && *CURRENT_LINE_BUFFER != '/') {
	    if (chat->add == 1)
		s = chat->prefix1;
	    else
		s = chat->prefix2;
	    text = g_strconcat (s, CURRENT_LINE_BUFFER, chat->suffix, NULL);
	} else {
	    text = g_strdup(CURRENT_LINE_BUFFER);
	}
	if (chat->current_y == chat->row - 1)
	    gau_chat_move_scroll(chat);
	else {
	    gau_chat_hide_cursor(chat);
	    chat->current_x = 0;
	    gau_chat_move_downward(chat);
	}
	gtk_signal_emit(GTK_OBJECT(chat), chat_signals[SEND],
			text, strlen(text));
	return;
    }
    s = CURRENT_CHAR_BUFFER;
    len = strlen(s);
    if ((len + chat->current_x) >= (chat->buf_alloc[chat->current_y] - 1)) {
	chat->buf_alloc[chat->current_y] += 50;
	chat->buf[chat->current_y] =
	    g_realloc(chat->buf[chat->current_y],
		      chat->buf_alloc[chat->current_y]);
	s = CURRENT_CHAR_BUFFER;
    }
    gau_chat_clear_line(chat, chat->current_y, chat->current_x, chat->current_x + len + 1);
    memmove(s + 1, s, len + 1);
    *s = c;
    /*
       g_print("%s\n", s);
     */
    gau_chat_expose_eol(chat);
    chat->current_x++;
    /*
       gau_chat_check_offset(chat);
     */
    gau_chat_show_cursor(chat);
}

void 
gau_chat_add_set(GauChat * chat, gint flag)
{
    chat->add = flag;
}

void 
gau_chat_prefix_set(GauChat * chat, char *s, gint flag)
{
    g_return_if_fail(s != NULL);
    g_return_if_fail(chat != NULL);
    g_return_if_fail(GAU_IS_CHAT(chat));

    if (flag == 1) {
	g_free(chat->prefix1);
	chat->prefix1 = g_strdup(s);
    } else {
	g_free(chat->prefix2);
	chat->prefix2 = g_strdup(s);
    }
}

void 
gau_chat_suffix_set(GauChat * chat, char *s)
{
    g_return_if_fail(s != NULL);
    g_return_if_fail(chat != NULL);
    g_return_if_fail(GAU_IS_CHAT(chat));

    g_free(chat->suffix);
    chat->suffix = g_strdup(s);
}


static int 
length_of_line(GauChat * chat, int y)
{
    //g_print("%s: %d %s\n", __FUNCTION__, y, chat->buf[y]);
    return strlen(LINE_BUFFER(y));
}

static void 
gau_chat_show_cursor(GauChat * chat)
{
    int x, y;
    GdkGC *gc;
    /*
       g_print("%d %d %d\n", chat->current_x, chat->current_y, chat->offset);
     */
    if (gau_chat_check_offset(chat))
	return;
    x = chat->font_width * (chat->current_x - chat->offset);
    y = chat->font_height * (chat->current_y - chat->adj->value);
    if (GTK_WIDGET_HAS_FOCUS(chat)) {
	gc = GTK_WIDGET(chat)->style->fg_gc[GTK_STATE_NORMAL];
    } else
	gc = GTK_WIDGET(chat)->style->black_gc;
    gdk_draw_line(GTK_WIDGET(chat)->window, gc,
		  x, y, x, y + chat->font_height - 1);
    if (GTK_WIDGET_HAS_FOCUS(chat) && gdk_im_ready() && chat->ic && 
	  (gdk_ic_get_style (chat->ic) & GDK_IM_PREEDIT_POSITION)) {
	chat->ic_attr->spot_location.x = x;
	chat->ic_attr->spot_location.y = y + chat->font_ascent;
	gdk_ic_set_attr (chat->ic, chat->ic_attr, GDK_IC_SPOT_LOCATION);
    }
}

static void 
gau_chat_hide_cursor(GauChat * chat)
{
    int mul;
    /*
       g_print("%s %d %d\n", __FUNCTION__, chat->current_x, chat->current_y);
     */
    if (*CURRENT_CHAR_BUFFER & 0x80)
	mul = 2;
    else
	mul = 1;
    gdk_window_clear_area(GTK_WIDGET(chat)->window,
		     (chat->current_x - chat->offset) * chat->font_width,
			  (chat->current_y - chat->adj->value)* chat->font_height,
			  mul * chat->font_width, chat->font_height);
    if (*CURRENT_CHAR_BUFFER)
	gau_chat_draw_line(chat, chat->current_y,
			   chat->current_x, chat->current_x + mul);

}

static void 
gau_chat_move_bol(GauChat * chat)
{
    if (chat->current_x == 0)
	return;
    gau_chat_hide_cursor(chat);
    chat->current_x = 0;
    gau_chat_show_cursor(chat);
}

static void 
gau_chat_move_eol(GauChat * chat)
{
    gau_chat_hide_cursor(chat);
    chat->current_x = length_of_line(chat, chat->current_y);
    gau_chat_show_cursor(chat);
}


static void 
gau_chat_delete_forward(GauChat * chat)
{
    char *s = CURRENT_CHAR_BUFFER;
    int delta = 1;
    gau_chat_clear_line(chat, chat->current_y, chat->current_x,
			chat->current_x + strlen(s));
    if (*s & 0x80)
	delta = 2;
    memmove(s, s + delta, strlen(s + delta) + 1);
    gau_chat_expose_eol(chat);
    gau_chat_show_cursor(chat);
}

static void gau_chat_delete_backward(GauChat * chat)
{
    if (chat->current_x == 0) {
	/* already beginning of line */
	return;
    }
    gau_chat_move_backward(chat);
    gau_chat_delete_forward(chat);
}

static void gau_chat_delete_forward_word(GauChat * chat)
{
    guchar *s = CURRENT_CHAR_BUFFER;
    guchar *e = s;
    gau_chat_clear_line(chat, chat->current_y, chat->current_x,
			chat->current_x + strlen(s));
    if (isspace(*e)) {
	while (isspace(*e))
	    e++;
    }
    while (!isspace(*e) && *e)
	e++;
    memmove(s, e, strlen(e) + 1);
    gau_chat_expose_eol(chat);
    gau_chat_show_cursor(chat);
}

static void gau_chat_delete_backward_word(GauChat * chat)
{
    guchar *s = CURRENT_CHAR_BUFFER;
    guchar *e = s;
    if (chat->current_x == 0)
	return;
    gau_chat_clear_line(chat, chat->current_y, chat->current_x,
			chat->current_x + strlen(s));
    if (*s == 0) {
	s--;
	chat->current_x--;
    }
    while (isspace(*s) && chat->current_x >= 0) {
	chat->current_x--;
	s--;
    }
    while (!isspace(*s) && chat->current_x >= 0) {
	chat->current_x--;
	s--;
    }
    s++;
    chat->current_x++;
    gau_chat_clear_line(chat, chat->current_y, chat->current_x,
			chat->current_x + s - e);
    memmove(s, e, strlen(e) + 1);
    gau_chat_expose_eol(chat);
    gau_chat_show_cursor(chat);
}

static void gau_chat_delete_line(GauChat * chat)
{
    gau_chat_clear_line(chat, chat->current_y, 0, strlen(CURRENT_LINE_BUFFER));
    chat->current_x = 0;
    *CURRENT_LINE_BUFFER = 0;
    gau_chat_show_cursor(chat);
}


static void gau_chat_delete_eol(GauChat * chat)
{
    gau_chat_clear_line(chat, chat->current_y, chat->current_x,
			strlen(CURRENT_LINE_BUFFER + chat->current_x));
    *CURRENT_CHAR_BUFFER = 0;
    gau_chat_show_cursor(chat);
}


/* right key action */
static void gau_chat_move_forward(GauChat * chat)
{
    if (chat->current_x == length_of_line(chat, chat->current_y)) {
	gau_chat_move_bol(chat);
	gau_chat_move_downward(chat);
	return;
    }
    gau_chat_hide_cursor(chat);
    chat->current_x++;
    if (is_second_han(CURRENT_LINE_BUFFER, chat->current_x))
	chat->current_x++;
    gau_chat_show_cursor(chat);
}

static void gau_chat_move_forward_word(GauChat * chat)
{
    guchar *s;
    if (chat->current_x == length_of_line(chat, chat->current_y)) {
	gau_chat_move_forward(chat);
	return;
    }
    s = CURRENT_CHAR_BUFFER;
    gau_chat_hide_cursor(chat);
    while (*s & 0x80 || isalnum(*s) || *s == '_') {
	s++;
	chat->current_x++;
    }
    if (isspace(*s)) {
	do {
	    s++;
	    chat->current_x++;
	} while (isspace(*s));
    }
    gau_chat_show_cursor(chat);
}


/* left key action */
static void gau_chat_move_backward(GauChat * chat)
{
    if (chat->current_x == 0) {
	if (chat->current_y == 0) {
	    gdk_beep();
	    return;
	}
	gau_chat_move_upward(chat);
	gau_chat_move_eol(chat);
	return;
    }
    gau_chat_hide_cursor(chat);
    chat->current_x--;
    if (is_second_han(CURRENT_LINE_BUFFER, chat->current_x))
	chat->current_x--;
    gau_chat_show_cursor(chat);
}

static void gau_chat_move_backward_word(GauChat * chat)
{
    guchar *s = CURRENT_CHAR_BUFFER;
    /*
       g_print("%s\n", __FUNCTION__);
     */
    if (chat->current_x == 0) {
	gau_chat_move_backward(chat);
	return;
    }
    /*
       g_print("%d \n", chat->current_x);
     */
    gau_chat_hide_cursor(chat);
    s--;
    chat->current_x--;
    if (isspace(*s))
	do {
	    s--;
	    chat->current_x--;
	} while (isspace(*s) && chat->current_x >= 0);
    while (isalnum(*s) || *s == '_' || (*s & 0x80)) {
	if (chat->current_x == 0)
	    break;
	s--;
	chat->current_x--;
    };
    if (chat->current_x != 0)
	chat->current_x++;
    /*
       g_print("%d \n", chat->current_x);
     */
    gau_chat_show_cursor(chat);
}


/* up key action */
static void gau_chat_move_upward(GauChat * chat)
{
    int len;
    if (chat->current_y == 0) {	/* first line: */
	return;
    } else {
	gau_chat_hide_cursor(chat);
	chat->current_y--;
	gtk_signal_emit(GTK_OBJECT(chat), chat_signals[LINE_CHANGED]);
    }
    len = length_of_line(chat, chat->current_y);
    if (len < chat->current_x)
	chat->current_x = len;
    if (is_second_han(CURRENT_LINE_BUFFER, chat->current_x))
	chat->current_x++;
    /*
       gau_chat_check_offset(chat);
     */
    gau_chat_show_cursor(chat);
}

/* down key action */
static void gau_chat_move_downward(GauChat * chat)
{
    int len;
    //g_print("%s:%d %d\n", __FUNCTION__, chat->current_y, chat->row);
    if (chat->current_y == chat->row - 1) {	/* last line: . */
	return;
    } else {
	gau_chat_hide_cursor(chat);
	chat->current_y++;
	gtk_signal_emit(GTK_OBJECT(chat), chat_signals[LINE_CHANGED]);
    }
    len = length_of_line(chat, chat->current_y);
    if (len < chat->current_x)
	chat->current_x = len;
    if (is_second_han(CURRENT_LINE_BUFFER, chat->current_x))
	chat->current_x++;
    /*
       gau_chat_check_offset(chat);
     */
    gau_chat_show_cursor(chat);
}

static void gau_chat_move_scroll(GauChat * chat)
{
    int y;
    char *buf0 = chat->buf[0];
    int buf_alloc0 = chat->buf_alloc[0];
    gdk_window_clear(GTK_WIDGET(chat)->window);

    for (y = 0; y < chat->row - 1; y++) {
	chat->buf[y] = chat->buf[y + 1];
	chat->buf_alloc[y] = chat->buf_alloc[y + 1];
    }
    chat->buf[chat->row - 1] = buf0;
    chat->buf_alloc[chat->row - 1] = buf_alloc0;
    buf0[0] = 0;
    chat->current_x = 0;
    gtk_widget_draw(GTK_WIDGET(chat), NULL);
    gau_chat_show_cursor(chat);
}



static void gau_chat_expose_eol(GauChat * chat)
{
    int x1 = chat->current_x;
    int x2;
    char *buf = CURRENT_CHAR_BUFFER;
    if (*buf == 0)
	return;
    x2 = x1 + strlen(buf);
    gau_chat_draw_line(chat, chat->current_y, x1, x2);
}

static void gau_chat_draw_line(GauChat * chat, int y, int x1, int x2)
{
    int yoffset = chat->adj->value;
    char *buf = LINE_BUFFER(y);
    if (x1 < chat->offset)
	x1 = chat->offset;
    if (x2 <= x1)
	return;
    if (is_second_han(buf, x1))
	x1--;
    if (is_first_han(buf, x2 - 1))
	x2++;
    //g_print("draw:%d %d %d %s\n", y, x1, x2, buf + x1);
    //g_print("    :%d %d \n", (x1-chat->offset)*chat->font_width, (y-yoffset)*chat->font_height + chat->font_ascent);
//    if (strlen(buf+x1) < x2 - x1) x2 = strlen(buf+x1) + x1;
    gdk_draw_text(GTK_WIDGET(chat)->window, chat->font, chat->gc,
		  (x1 - chat->offset) * chat->font_width,
		  (y - yoffset) * chat->font_height + chat->font_ascent,
		  buf + x1, x2 - x1);
}

static void gau_chat_clear_line(GauChat * chat, int y, int x1, int x2)
{
    if (x1 < chat->offset)
	x1 = chat->offset;
    if (x2 <= x1)
	return;
	//g_print("clear:%d %d %d\n", y, x1, x2);
    gdk_window_clear_area(GTK_WIDGET(chat)->window,
			  (x1 - chat->offset) * chat->font_width,
			  (y - chat->adj->value) * chat->font_height,
			  (x2 - x1) * chat->font_width,
			  chat->font_height);
}


static gint
gau_chat_key_press(GtkWidget * w, GdkEventKey * ev)
{
    gint key = ev->keyval;
    GauChat *chat = GAU_CHAT(w);
    if (key >= GDK_Shift_L && key <= GDK_Alt_R)
	return FALSE;
    if (key >= GDK_F1 && key <= GDK_F12) {
	key = key - GDK_F1;
	if (ev->state & GDK_SHIFT_MASK && key < 10) {
	    key += 10;
	}
	if (chat->fbinding[key]) {
	    gau_chat_insert_text(chat, chat->fbinding[key],
				 strlen(chat->fbinding[key]));
	    return TRUE;
	}
	return FALSE;
    }
    if (ev->state & GDK_CONTROL_MASK) {
	if ((key >= 'A') && (key <= 'Z'))
	    key -= 'A' - 'a';
	if ((key >= 'a') && (key <= 'z') && control_keys[key - 'a']) {
	    (*control_keys[key - 'a']) (chat);
	    return TRUE;
	}
    } else if (ev->state & GDK_MOD1_MASK) {
	if ((key >= 'A') && (key <= 'Z'))
	    key -= 'A' - 'a';
	if ((key >= 'a') && (key <= 'z') && alt_keys[key - 'a']) {
	    (*alt_keys[key - 'a']) (chat);
	    return TRUE;
	}
    } else {
	switch (key) {
	case GDK_Return:
	case GDK_KP_Enter:
	    gau_chat_insert_char(chat, '\r');
	    return TRUE;
	case GDK_Left:
	case GDK_KP_Left:
	    gau_chat_move_backward(chat);
	    return TRUE;
	case GDK_Right:
	case GDK_KP_Right:
	    gau_chat_move_forward(chat);
	    return TRUE;
	case GDK_Up:
	case GDK_KP_Up:
	    gau_chat_move_upward(chat);
	    return TRUE;
	case GDK_Down:
	case GDK_KP_Down:
	    gau_chat_move_downward(chat);
	    return TRUE;
	case GDK_Delete:
	case GDK_KP_Delete:
	    gau_chat_delete_forward(chat);
	    return TRUE;
	case GDK_BackSpace:
	    gau_chat_delete_backward(chat);
	    return TRUE;
	}
	if (ev->length > 0) {
	    /* normal */
	    gau_chat_insert_text(chat, ev->string, ev->length);
	    return TRUE;
	}
    }
    return FALSE;
}

static gint
gau_chat_expose(GtkWidget * w, GdkEventExpose * ev)
{
    int x1, x2;
    int y1, y2;
    int i1, i2;
    int y;
    int len;
    GauChat *chat;
    g_return_val_if_fail (w != NULL, FALSE);
    chat = GAU_CHAT(w);
    /* */
    //g_print("expose:adj->value = %d\n", (int)chat->adj->value);
    ev->area.y += chat->adj->value*chat->font_height;
    x1 = ev->area.x / chat->font_width;
    x2 = (ev->area.x + ev->area.width) / chat->font_width;
    y1 = ev->area.y / chat->font_height;
    y2 = (ev->area.y + ev->area.height - 1) / chat->font_height;
    //g_print("%d %d %d %d %d\n", x1, x2, y1, y2, chat->offset);
    if (y2 >= chat->row) y2 = chat->row - 1;
    for (y = y1; y <= y2; y++) {
	len = length_of_line(chat, y);
	//g_print("%d %d\n", y, len);
	if (x1 + chat->offset >= len)
	    continue;
	if (is_second_han(LINE_BUFFER(y), x1 + chat->offset))
	    i1 = x1 - 1;
	else
	    i1 = x1;
	if (x2 + chat->offset >= len)
	    i2 = len - chat->offset - 1;
	else {
	    if (is_first_han(LINE_BUFFER(y), x2 + chat->offset))
		i2 = x2 + 1;
	    else
		i2 = x2;
	}
	gau_chat_draw_line(chat, y, i1 + chat->offset, i2 + chat->offset + 1);
	if (chat->current_y == y) {
	    if (chat->current_x <= i2 + chat->offset + 1
		&& chat->current_x >= i1 + chat->offset)
		gau_chat_show_cursor(chat);
	}
    }
    return TRUE;
}

static gint
gau_chat_button_press(GtkWidget * w, GdkEventButton * ev)
{
    int len;
    GauChat *chat;
    int x, y;
    if (!GTK_WIDGET_HAS_FOCUS(w))
	gtk_widget_grab_focus(w);
    chat = GAU_CHAT(w);
    /* FIXME */
    x = ev->x / chat->font_width + chat->offset;
    y = ev->y / chat->font_height + chat->adj->value;
    gau_chat_hide_cursor(chat);
    if (x > (len = length_of_line(chat, y)))
	x = len;
    else if (is_second_han(LINE_BUFFER(y), x))
	x++;
    chat->current_x = x;
    chat->current_y = y;
    gau_chat_show_cursor(chat);
    return FALSE;
}

static gint
gau_chat_check_offset(GauChat * chat)
{
    if (chat->current_x - chat->offset >= chat->col) {
	do {
	    chat->offset += chat->col / 2;
	} while (chat->current_x - chat->offset >= chat->col);
	//g_print("check_offset-1\n");
	gdk_window_clear(GTK_WIDGET(chat)->window);
	gtk_widget_draw(GTK_WIDGET(chat), NULL);
	return TRUE;
    } else if (chat->current_x < chat->offset) {
	do {
	    chat->offset -= chat->col / 2;
	} while (chat->current_x < chat->offset);
	//g_print("check_offset-2\n");
	gdk_window_clear(GTK_WIDGET(chat)->window);
	gtk_widget_draw(GTK_WIDGET(chat), NULL);
	return TRUE;
    }
    return FALSE;
}

static void 
gau_chat_realize(GtkWidget * w)
{
    GauChat *chat = GAU_CHAT(w);
    GTK_WIDGET_CLASS(parent_class)->realize (w);
    gdk_window_set_background(w->window, &w->style->white);
    chat->gc = gdk_gc_new(w->window);
    gdk_gc_set_foreground(chat->gc, &w->style->black);
    gau_chat_im_init(chat);
}

static void 
gau_chat_unrealize(GtkWidget * w)
{
    GauChat *chat = GAU_CHAT(w);
    gdk_gc_destroy (chat->gc);
    if (GTK_WIDGET_CLASS(parent_class)->unrealize)
	(*GTK_WIDGET_CLASS(parent_class)->unrealize) (w);
}

static void 
gau_chat_finalize (GtkObject * object)
{
    GauChat *chat = GAU_CHAT(object);
    g_free(chat->buf);
    g_free(chat->prefix1);
    g_free(chat->prefix2);
    g_free(chat->suffix);
    if (GTK_OBJECT_CLASS(parent_class)->finalize)
	(*GTK_OBJECT_CLASS(parent_class)->finalize) (object);
}

static void 
gau_chat_destroy(GtkObject * object)
{
    if (GTK_OBJECT_CLASS(parent_class)->destroy)
	(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}

static void
gau_chat_size_allocate (GtkWidget *w, GtkAllocation *allocation)
{
    int new_col, new_row;
    GauChat *chat;
    /* FIXME */

    g_return_if_fail (w != NULL);
    chat = GAU_CHAT(w);
    new_col = chat->col = (allocation->width) / chat->font_width;
    new_row = chat->vrow = (allocation->height) / chat->font_height;
    w->allocation = *allocation;

    if (GTK_WIDGET_REALIZED (w)) {
	    gdk_window_move_resize (w->window,
		    allocation->x, allocation->y,
		    allocation->width, allocation->height);
    }
    if (chat->ic && (gdk_ic_get_style (chat->ic) & GDK_IM_PREEDIT_POSITION)) {
	chat->ic_attr->preedit_area.width = allocation->width;
	chat->ic_attr->preedit_area.height = allocation->height;
	chat->ic_attr->preedit_area.x = 0;
	chat->ic_attr->preedit_area.y = 0;
	//g_print("allocation: x=%d y=%d\n", allocation->x, allocation->y);
	gdk_ic_set_attr (chat->ic, chat->ic_attr,
	      		   GDK_IC_PREEDIT_AREA);
	//g_print("allocation: width=%d height=%d\n", allocation->width, allocation->height);
    }
}

static void
gau_chat_size_request(GtkWidget *w, GtkRequisition *requisition)
{
    GauChat *chat;
    g_return_if_fail (w != NULL);

    chat = GAU_CHAT(w);
    requisition->width = chat->col * chat->font_width;
    requisition->height = chat->vrow * chat->font_height;
    //g_print("%s:%d %d\n", __FUNCTION__, requisition->width, requisition->height);
}


static void
gau_chat_set_adjustments (GauChat *chat, GtkAdjustment *hadj, GtkAdjustment *vadj)
{
    if (!vadj) 
    	vadj = GTK_ADJUSTMENT (gtk_adjustment_new(0, 0, 0, 1, chat->row, chat->row));  
    if (chat->adj && chat->adj != vadj) {
	gtk_signal_disconnect_by_data (GTK_OBJECT(chat->adj), chat);
	gtk_object_unref (GTK_OBJECT (chat->adj));
    }
    if (chat->adj != vadj) {
	chat->adj = vadj;
	gtk_object_ref (GTK_OBJECT (vadj));
	gtk_object_sink (GTK_OBJECT (vadj));
	gtk_signal_connect (GTK_OBJECT (vadj), "value_changed",
		(GtkSignalFunc) gau_chat_adjustment_changed, chat);
	vadj->value = 0;
	vadj->page_size = chat->vrow;
	vadj->page_increment = chat->vrow / 2;
	vadj->step_increment = 10;
	vadj->upper = chat->row;

    }
}

static int
gau_chat_adjustment_changed(GtkAdjustment *adj, GauChat *chat)
{
    //g_print("%s\n", __FUNCTION__);
    gdk_window_clear(GTK_WIDGET(chat)->window);
    gtk_widget_draw(GTK_WIDGET(chat), NULL);
    return TRUE;
}

static gint
gau_chat_focus_in (GtkWidget     *widget,
		    GdkEventFocus *event)
{
    GauChat *chat;
    g_return_val_if_fail (widget != NULL, FALSE);
    g_return_val_if_fail (event != NULL, FALSE);
    chat = GAU_CHAT(widget);

    GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);

    if (chat->ic)
        gdk_im_begin (chat->ic, widget->window);

    return FALSE;
}

static gint
gau_chat_focus_out (GtkWidget     *widget,
		     GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);

  gdk_im_end ();
  return FALSE;
}

void
gau_chat_preedit_draw(XIC ic, GauChat *chat, XIMPreeditDrawCallbackStruct *cbs)
{
}

void
gau_chat_preedit_start(XIC ic, GauChat *chat)
{
}

void
gau_chat_preedit_done(XIC ic, GauChat *chat)
{
}

static void
gau_chat_temp_draw(GauChat *chat)
{
#if 0
    if (gau_chat_check_offset(chat))
	return;
    x = chat->font_width * (chat->current_x - chat->offset);
    y = chat->font_height * (chat->current_y - chat->adj->value);
    if (GTK_WIDGET_HAS_FOCUS(chat)) {
	gc = GTK_WIDGET(chat)->style->fg_gc[GTK_STATE_NORMAL];
    } else
	gc = GTK_WIDGET(chat)->style->black_gc;
    gdk_draw_line(GTK_WIDGET(chat)->window, gc,
		  x, y, x, y + chat->font_height - 1);
    if (GTK_WIDGET_HAS_FOCUS(chat) && gdk_im_ready() && chat->ic && 
	  (gdk_ic_get_style (chat->ic) & GDK_IM_PREEDIT_POSITION)) {
	chat->ic_attr->spot_location.x = x;
	chat->ic_attr->spot_location.y = y + chat->font_ascent;
	gdk_ic_set_attr (chat->ic, chat->ic_attr, GDK_IC_SPOT_LOCATION);
    }
#endif
}

void
gau_chat_preedit_caret(XIC ic, GauChat *chat,XIMPreeditCaretCallbackStruct *cbs)
{
    int old_pos = chat->ebuf->caret;
    PreeditBuf *ebuf = chat->ebuf;
    switch(cbs->direction) {
	case XIMAbsolutePosition:
	    ebuf->caret = cbs->position;
	    if (ebuf->caret < 0) ebuf->caret = 0;
	    else if (ebuf->caret > ebuf->len) ebuf->caret = ebuf->len;
	    break;
	case XIMForwardChar:
	    ebuf->caret += cbs->position; 
	    if (ebuf->caret > ebuf->len) ebuf->caret = ebuf->len;
	    break;
	case XIMBackwardChar:
	    ebuf->caret += cbs->position; 
	    if (ebuf->caret < 0) ebuf->caret = 0;
	    break;
	default:
	    /* not supported */
	    break;
    }
    if (ebuf->caret != old_pos) {
	gau_chat_temp_draw(chat);
    }
}

gint
gau_chat_im_init(GauChat *chat)
{
    GtkWidget *widget;

    widget = GTK_WIDGET(chat);
    if (gdk_im_ready () && (chat->ic_attr = gdk_ic_attr_new ()) != NULL) {
	/* XIM was created by brain-damaged persons */
	/* We will never win with this crumsy protocol */
        GdkEventMask mask;
        GdkColormap *colormap;
        GdkICAttr *attr = chat->ic_attr;
        GdkICAttributesType attrmask = GDK_IC_ALL_REQ;
        GdkIMStyle style;
        GdkIMStyle supported_style = GDK_IM_PREEDIT_NONE |
				   GDK_IM_PREEDIT_NOTHING |
			           GDK_IM_PREEDIT_POSITION |
			           GDK_IM_STATUS_NONE |
				   GDK_IM_STATUS_NOTHING |
				   GDK_IM_STATUS_CALLBACKS;

        attr->style = style = gdk_im_decide_style (supported_style);
        attr->client_window = gtk_widget_get_toplevel(widget)->window;

        if ((colormap = gtk_widget_get_colormap (widget)) !=
	    gtk_widget_get_default_colormap ()) {
	  attrmask |= GDK_IC_PREEDIT_COLORMAP;
	  attr->preedit_colormap = colormap;
	}
	if (style & GDK_IM_PREEDIT_POSITION) {
	    attrmask |= GDK_IC_PREEDIT_FOREGROUND;
	    attrmask |= GDK_IC_PREEDIT_BACKGROUND;
	    attr->preedit_foreground = widget->style->black;
	    attr->preedit_background = widget->style->white;
	    attrmask |= GDK_IC_PREEDIT_POSITION_REQ;
	    attrmask |= GDK_IC_PREEDIT_AREA_REQ;
	    attr->spot_location.x = 0;
	    attr->spot_location.y = chat->font_ascent;
	    attr->preedit_area.x = 0;
	    attr->preedit_area.y = 0;
	    attr->preedit_area.width = widget->allocation.width;
	    attr->preedit_area.height = widget->allocation.height;
	    //g_print("width=%d height=%d\n", attr->preedit_area.width, widget->allocation.height);
	    attr->preedit_fontset = chat->font;
	}
        chat->ic = gdk_ic_new (attr, attrmask);
     
        if (chat->ic == NULL)
	    g_warning ("Can't create input context.");
        else {
	    mask = gdk_window_get_events (widget->window);
	    mask |= gdk_ic_get_events (chat->ic);
	    gdk_window_set_events (widget->window, mask);
	    if ((style & GDK_IM_STATUS_MASK) == GDK_IM_STATUS_CALLBACKS) {
		extern int gau_status_draw(XIC ic, gpointer user_data, XIMStatusDrawCallbackStruct *st);
		typedef struct { gchar *name; gpointer value; } GdkImArg;
		GdkImArg arg[2] = {{NULL, NULL}, {NULL, NULL}};
		XIMCallback status_draw_cb;
		XIMCallback status_done_cb;
		XIMCallback status_start_cb;
		status_draw_cb.client_data = NULL;
		status_draw_cb.callback = (XIMProc)gau_status_draw;
		arg->name = XNStatusDrawCallback;
		arg->value = (gpointer) &status_draw_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);

		status_start_cb.client_data = NULL;
		status_start_cb.callback = (XIMProc)gtk_true;
		arg->name = XNStatusStartCallback;
		arg->value = (gpointer) &status_start_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);

		status_done_cb.client_data = NULL;
		status_done_cb.callback = (XIMProc)gtk_true;
		arg->name = XNStatusDoneCallback;
		arg->value = (gpointer) &status_done_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);
	    }
	    if ((style & GDK_IM_PREEDIT_MASK) == GDK_IM_PREEDIT_CALLBACKS) {
		typedef struct { gchar *name; gpointer value; } GdkImArg;
		GdkImArg arg[2] = {{NULL, NULL}, {NULL, NULL}};
		XIMCallback preedit_draw_cb;
		XIMCallback preedit_done_cb;
		XIMCallback preedit_start_cb;
		XIMCallback preedit_caret_cb;
		preedit_draw_cb.client_data = NULL;
		preedit_draw_cb.callback = (XIMProc)gau_chat_preedit_draw;
		arg->name = XNPreeditDrawCallback;
		arg->value = (gpointer) &preedit_draw_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);

		preedit_start_cb.client_data = NULL;
		preedit_start_cb.callback = (XIMProc)gau_chat_preedit_start;
		arg->name = XNPreeditStartCallback;
		arg->value = (gpointer) &preedit_start_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);

		preedit_done_cb.client_data = NULL;
		preedit_done_cb.callback = (XIMProc)gau_chat_preedit_done;
		arg->name = XNPreeditDoneCallback;
		arg->value = (gpointer) &preedit_done_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);

		preedit_caret_cb.client_data = NULL;
		preedit_caret_cb.callback = (XIMProc)gau_chat_preedit_caret;
		arg->name = XNPreeditCaretCallback;
		arg->value = (gpointer) &preedit_caret_cb;
		XSetICValues(((GdkICPrivate *)chat->ic)->xic, XNStatusAttributes, arg, NULL);
	    }
	    if (GTK_WIDGET_HAS_FOCUS(widget))
		gdk_im_begin (chat->ic, widget->window);
	}
    }
    return TRUE;
}
