#
# $Id: tkinspect.tcl,v 1.1 1993/03/28 02:15:15 sls Exp $
#
if {$argc != 1} {
    puts stderr "Ooops! tkinspect has not been installed correctly."
    exit 1
}
set tkinspect_library $argv
source $tkinspect_library/class.tcl
source $tkinspect_library/bindings.tcl
source $tkinspect_library/filechsr.tcl

wm title . "Tk Browser: "
wm iconname . "Tk Browser"

#
# global variables
#
set target none

#
# Create the main menu.
#
class MainMenu {
    member w {}
    member filechooser {}
    method create w {
	setmember w $w
	setmember filechooser [new FileChooser]
	frame $w -relief raised -bd 2
	menubutton $w.file -text "File" -menu $w.file.m -underline 0
	set m [menu $w.file.m]
	$m add cascade -label "Select Application" \
	    -command "MainMenu:fillApps $this" \
	    -menu $m.apps -underline 7 -accelerator =>
	$m add command -label "Save Value..." -underline 0 \
	    -command "MainMenu:saveValue $this"
	$m add command -label "Load Value..." -underline 0 \
	    -command "MainMenu:loadValue $this"
	$m add command -label "Update Lists" -underline 0 \
	    -command updateLists
	$m add separator
	$m add cascade -label "Delete Interpreter..." -underline 0 \
	    -command "MainMenu:fillDelInterps $this" -menu $m.delapps \
	    -accelerator =>
	$m add separator
	$m add command -label "Quit" -command "destroy ." -underline 0
	menu $m.apps
	menu $m.delapps
	pack append $w $w.file left
	tk_menuBar $w $w.file
	return $w
    }
    method fillDelInterps {} {
	set w [getmember w]
	catch {$w.file.m.delapps delete 0 last}
	foreach i [lsort [winfo interps]] {
	    $w.file.m.delapps add command -label $i \
		-command [list delInterp $i]
	}
    }
    method fillApps {} {
	set w [getmember w]
	catch {$w.file.m.apps delete 0 last}
	foreach i [lsort [winfo interps]] {
	    $w.file.m.apps add command -label $i \
		-command [list setTarget $i]
	}
    }
    method saveValue {} {
	set fc [getmember filechooser]
	if {[set file [FileChooser:run $fc "Save value to:" 0 1]] != {}} {
	    set f [open $file w]
	    puts $f [.value.t get 1.0 end]
	    close $f
	}
    }
    method loadValue {} {
	set fc [getmember filechooser]
 	if {[set file [FileChooser:run $fc "Load value from:" 1 0]] != {}} {
	    .value.t delete 1.0 end
	    set f [open $file r]
	    .value.t insert end [read $f]
	    close $f
	}	
    }
}
pack append . [in [new MainMenu] { create .main }] "top fillx"

#
# center a toplevel window
#
proc centerWindow {win} {
    wm withdraw $win
    update idletasks
    set w [winfo reqwidth $win]
    set h [winfo reqheight $win]
    set sh [winfo screenheight $win]
    set sw [winfo screenwidth $win]
    wm geometry $win +[expr {($sw-$w)/2}]+[expr {($sh-$h)/2}]
    wm deiconify $win
}

#
# yes no box
#
class YesNo {
    member w .yesno
    member yes 0
    method run {question} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm transient $w .
	frame $w.top
	message $w.top.msg -text $question -justify center \
	    -font -Adobe-helvetica-medium-r-normal--*-240* -aspect 200
	pack append $w.top $w.top.msg "fill padx 5 pady 5 expand"
	frame $w.bot
	button $w.yes -text "Yes" -command "YesNo:yes $this"
	pack append $w.bot $w.yes "left expand padx 20"
	frame $w.bot.no -relief sunken -bd 1
	pack append $w.bot $w.bot.no "left expand padx 20 pady 20"
	button $w.no -text "No" -command "YesNo:no $this"
	pack append $w.bot.no $w.no "expand padx 12 pady 12"
	pack append $w $w.top "top fill expand" $w.bot "bottom fill expand"
	bind $w.top <Enter> "$w.no activate"
	bind $w.top.msg <Enter> "$w.no activate"
	bind $w.bot <Enter> "$w.no activate"
	bind $w.top <Leave> "$w.no deactivate"
	bind $w.top.msg <Leave> "$w.no deactivate"
	bind $w.bot <Leave> "$w.no deactivate"
	centerWindow $w
	focus $w
	bind $w <Return> "YesNo:no $this"
	grab set $w
	tkwait window $w
	return [getmember yes]
    }
    method yes {} {
	setmember yes 1
	destroy [getmember w]
    }
    method no {} {
	setmember yes 0
	destroy [getmember w]
    }
}

#
# Ask if you really want to delete the interp.
#
set yesno [new YesNo]
proc delInterp {interp} {
    global yesno tkinspect_library
    set txt "Really delete interpreter \"$interp\"?"
    if {[catch {send $interp info tclversion}] == 0} {
	append txt "\n(It is still responding.)"
    }
    if [YesNo:run $yesno $txt] {
	exec $tkinspect_library/delinterp $interp
    }
}

#
# set the current target.
#
proc setTarget {t} {
    global target
    set target $t
    wm title . "Tk Browser: $t"
    updateLists
}

#
# CommandEntry handles sending commands to the remote interpreter.  It
# also provides a history using ^n or Down and ^p or Up to scan through
# the history.
#
class CommandEntry {
    member w {}
    member history {}
    member ndx 0
    member n 0
    method create {w} {
	setmember w $w
	frame $w -borderwidth 2
	entry $w.e -relief sunken
	bindEntry $w.e
	bind $w.e <Return> "+CommandEntry:do $this"
	bind $w.e <Control-p> "CommandEntry:previous $this"
	bind $w.e <Up> "CommandEntry:previous $this"
	bind $w.e <Down> "CommandEntry:next $this"
	label $w.l -text "Command:"
	button $w.s -text "Send Command" -command "+CommandEntry:do $this"
	pack append $w \
	    $w.l left \
	    $w.e "left fillx expand" \
	    $w.s "right padx .2c"
	return $w
    }
    method do {} {
	global target
	if {$target == ""} {
	    statusMessage "no interpreter selected"
	    return
	}
	set cmd [[getmember w].e get]
	catch [list send $target $cmd] stuff
	insertValue $stuff
	lappendmember history $cmd
	set n [getmember n]
	if {$n == 50} {
	    setmember history [lrange [getmember history] 1 end]
	} else {
	    setmember n [expr $n+1]
	}
	set ndx $n
    }
    method previous {} {
	set ndx [getmember ndx]
	if {$ndx == 0} return
	set w [getmember w]
	incr ndx -1
	setmember ndx $ndx
	$w.e delete 0 end
	$w.e insert insert [lindex [getmember history] $ndx]
	tk_entrySeeCaret $w.e
    }
    method next {} {
	set ndx [getmember ndx]
	if {$ndx == [getmember n]} return
	set w [getmember w]
	incr ndx
	setmember ndx $ndx
	$w.e delete 0 end
	$w.e insert insert [lindex [getmember history] $ndx]
	tk_entrySeeCaret $w.e
    }
}
pack append . [CommandEntry:create [new CommandEntry] .cmd] "fillx expand top"

#
# PatternEditor sets up a dialog that edits include/exclude patterns.
#
class PatternEditor {
#|
# w is the name of the toplevel window.
# ok is 1 if ok was clicked, 0 if cancel was clicked.
# patterns is the list of patterns if ok was clicked.
#|
    member w .patedit
    member ok 0
    member patterns {}
#|
# run starts the dialog and waits until ok or cancel is clicked.  It
# returns 1 if ok, 0 if cancel.
#|
    method run {title patterns} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm title $w $title
	wm iconname $w $title
	frame $w.top
	label $w.l -text "Pattern:"
	entry $w.e -width 40 -relief sunken
	bindEntry $w.e
	pack append $w.top $w.l "left" $w.e "left fillx expand"
	pack append $w $w.top "top fillx pady .25c"
	frame $w.buttons -bd 3
	button $w.ok -text "Ok" -command "PatternEditor:ok $this"
	button $w.cancel -text "Cancel" -command "PatternEditor:cancel $this"
	button $w.add -text "Add Pattern" \
	    -command "PatternEditor:add $this"
	button $w.del -text "Delete Pattern(s)" \
	    -command "PatternEditor:delete $this"
	radiobutton $w.inc -variable patternType -value include \
	    -relief flat -text "Include Patterns"
	radiobutton $w.exc -variable patternType -value exclude \
	    -relief flat -text "Exclude Patterns"
	pack append $w.buttons \
	    $w.inc "top fillx pady .1c frame w" \
	    $w.exc "top fillx pady .1c frame w" \
	    $w.add "top fillx pady .1c" \
	    $w.del "top fillx pady .1c" \
	    $w.cancel "bottom fillx pady .1c" \
	    $w.ok "bottom fillx pady .1c"
	pack append $w $w.buttons "left filly"
	frame $w.lframe
	scrollbar $w.scroll -command "$w.list yview"
	listbox $w.list -yscroll "$w.scroll set" -relief raised \
	    -geometry 40x10 -exportselection false
	foreach pat $patterns {
	    $w.list insert end $pat
	}
	setmember patterns $patterns
	pack append $w.lframe \
	    $w.scroll "right filly" \
	    $w.list "left expand fill"
	pack append $w $w.lframe "right fill expand"
	grab set $w
	tkwait window $w
	return [getmember ok]
    }
    method getpatterns {} {
	getmember patterns
    }
    method cancel {} {
	setmember ok 0
	destroy [getmember w]
    }
    method ok {} {
	setmember ok 1
	setmember patterns {}
	set w [getmember w]
	for {set i 0} {$i < [$w.list size]} {incr i} {
	    lappendmember patterns [$w.list get $i]
	}
	destroy [getmember w]
    }
#|
# add adds a pattern.
#|
    method add {} {
	set w [getmember w]
	set thing [string trim [$w.e get]]
	if {$thing == {}} return
	$w.list insert end $thing
    }
#|
# delete deletes a pattern.
#|
    method delete {} {
	set w [getmember w]
	while {[set s [$w.list curselection]] != {}} {
	    $w.list delete [lindex $s 0]
	}
    }
}

#
# SelectiveList is a listbox that only displays things that either match
# or don't match a set of patterns.
#
class SelectiveList {
#|
# w is the name of the window
# command is the command to be invoked when a new selection is made.
# patternEditor holds the pattern editor.
# excludes is a list of patterns to exclude.
# contents stores all the contents.
#|
    member w {}
    member command {}
    member patternEditor {}
    member patternType exclude
    member patterns {}
    member contents {}
#|
# create creates the window.
#|
    method create {w title {command {}}} {
	setmember w $w
	setmember command $command
	frame $w -bd 2 -relief raised
	button $w.pats -text "Edit Patterns..." \
	    -command "SelectiveList:editPatterns $this"
	scrollbar $w.scroll -command "$w.list yview"
	listbox $w.list -yscroll "$w.scroll set" -relief raised \
	    -geometry 30x10 -exportselection false
	tk_listboxSingleSelect $w.list
	frame $w.lframe -bd 2 -relief sunken
	pack append $w.lframe $w.scroll "right filly" \
	    $w.list "left expand fill"
	label $w.title -text $title -anchor w
	pack append $w \
	    $w.title "top fillx" \
	    $w.lframe "top fill expand" \
	    $w.pats "padx 4 pady 4 bottom frame w"
	bind $w.list <1> [format {
	    %%W select from [%%W nearest %%y]
	    SelectiveList:select %s %%y
	} $this]
	setmember patternEditor [new PatternEditor]
    }
#|
# setPatterns sets the list of patterns.
#|
    method setPatterns {pats} {
	setmember patterns $pats
    }
#|
# clear clears all the entries.
#|
    method clear {} {
	setmember contents {}
	updateList
    }
#|
# updateList goes through the current contents only inserting those
# items that should be included.
#|
    method updateList {} {
	set w [getmember w]
	$w.list delete 0 end
	foreach thing [getmember contents] {
	    if [include? $thing] {
		$w.list insert end $thing
	    }
	}
    }
#|
# include? checks to see if a thing should be included.
#|
    method include? {thing} {
	foreach re [getmember patterns] {
	    if [regexp $re $thing] {
		if {[getmember patternType] == "include"} {
		    return 1
		} else {
		    return 0
		}
	    }
	}
	if {[getmember patternType] == "include"} {
	    return 0
	} else {
	    return 1
	}
    }
#|
# add adds some stuff.
#|
    method add {stuff} {
	foreach thing $stuff {
	    lappendmember contents $thing
	    if [include? $thing] {
		[getmember w].list insert end $thing
	    }
	}
    }
#|
# editPatterns brings up the pattern editor.
#|
    method editPatterns {} {
	set p [getmember patternEditor]
	global patternType
	set patternType [getmember patternType]
	if [PatternEditor:run $p "Pattern Editor" [getmember patterns]] {
	    setmember patterns [PatternEditor:getpatterns $p]
	    setmember patternType $patternType
	}
	updateList
    }
#|
# select invokes command if there is one.
#|
    method select {y} {
	set command [getmember command]
	if {$command != {}} {
	    set w [getmember w]
	    append command " [$w.list get [$w.list nearest $y]]"
	    eval $command
	}
    }
}

#
# Create the lists that hold the interpreter names, the procs, and the
# global variables.
#
frame .lists -borderwidth 2
in [set procList [new SelectiveList]] {
    create .lists.procs "Procs:" selectProc
    setPatterns {auto_.* tk_.*}
}
in [set globalList [new SelectiveList]] {
    create .lists.globals "Globals:" selectGlobal
    setPatterns {auto_.* tk_.*}
}
in [set winList [new SelectiveList]] {
    create .lists.windows "Windows:" selectWindow
}
pack append .lists \
    .lists.procs "left fill expand" \
    .lists.globals "left fill expand" \
    .lists.windows "left fill expand"
pack append . .lists "fill expand"

#
# Create the text widget that holds the value.
#
frame .value -bd 2 -relief raised
label .value.l -text "Value:" -anchor w
frame .value.tframe -bd 2 -relief sunken
scrollbar .value.sb -command ".value.t yview"
text .value.t -relief raised -bd 2 -yscroll ".value.sb set" -width 80
bindText .value.t
pack append .value.tframe \
    .value.t "left fill expand" \
    .value.sb "right filly"
button .value.s -text "Send Value" -command sendValue
pack append .value \
    .value.l "top fillx" \
    .value.tframe "top fill expand" \
    .value.s "padx 4 pady 4 bottom frame w"
pack append . .value "fill expand"

#
# Create a status line.
#
frame .status -bd 2
label .status.l -relief sunken -anchor w
pack append .status .status.l "fill expand"
pack append . .status "fill expand"

#
# statusMessage sets the contents of the status line.
#
proc statusMessage msg {
    .status.l config -text $msg
}

#
# insertValue inserts a line into the value widget.
#
proc insertValue args {
    .value.t delete 1.0 end
    eval .value.t insert end $args
}

#
# sendValue sends the currentValue back to the interpreter.
#
proc sendValue {} {
    global target
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return
    }
    if {[catch {send $target [.value.t get 1.0 end]} stuff] != 0} {
	insertValue $stuff
    }
}

#
# updateLists updates the lists of globals, procs, and windows.
#
proc updateLists {} {
    global target
    if {[catch {send $target info tclversion} result] != 0} {
	statusMessage $result
	return
    }
    global procList
    in $procList {
	clear
	add [lsort [send $target info procs]]
    }
    global globalList
    in $globalList {
	clear
	add [lsort [send $target info globals]]
    }
    global winList
    SelectiveList:clear $winList
    getWindows $target [send $target winfo children .]
    statusMessage "now sending to $target"
}

#
# getWindows recursively adds windows to the list of windows.
#
proc getWindows {target wins} {
    global winList
    SelectiveList:add $winList $wins
    foreach win $wins {
	getWindows $target [send $target winfo children $win]
    }
}

#
# selectProc handles getting the defn. of a proc in the remote interpreter.
#
proc selectProc {thing} {
    global target
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return
    }
    set args {}
    foreach arg [send $target info args $thing] {
	if [send $target info default $thing $arg __tkinspectDefaultArg__] {
	    lappend args [list $arg [send $target set __tkinspectDefaultArg__]]
	} else {
	    lappend args $arg
	}
    }
    catch {send $target unset __tkinspectDefaultArg__}
    insertValue [list proc $thing $args [send $target info body $thing]]
}

#
# selectGlobal handles getting the value of a global in the remote
# interpreter.
#
proc selectGlobal {thing} {
    global target
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return
    }
    if {[catch {send $target "set $thing"} result] == 0} {
	set stuff [list set $thing $result]
    } else {
	set stuff ""
	foreach name [lsort [send $target [list array names $thing]]] {
	    append stuff [list set [format %s(%s) $thing $name] \
			  [send $target [list set [format %s(%s) $thing \
						   $name]]]] \
		"\n"
	}
    }
    insertValue $stuff
}

#
# selectWindow handles getting the configuration options for a window.
#
set flashColor #ffaeb9
proc selectWindow {win} {
    global target flashColor
    if {$target == "none"} {
	statusMessage "no interpreter selected"
	return
    }
    set class [send $target winfo class $win]
    set stuff "# Class: $class\n$win config"
    foreach item [send $target $win config] {
	if {[llength $item] != 5} continue
	append stuff " \\\n\t"
	lappend stuff [lindex $item 0] [lindex $item 4]
    }
    insertValue $stuff
    set oldbg [lindex [send $target $win config -bg] 4]
    send $target $win config -bg $flashColor
    send $target after 100 [list $win config -bg $oldbg]
}

statusMessage "ready"
